#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#ifdef linux
#include <net/if_arp.h>
#else
#include <netinet/in_systm.h>
#include <sys/socket.h>
#include <net/if.h>
#ifndef	if_mtu
#include <net/if_var.h>
#endif
#include <netinet/in.h>
#include <netinet/if_ether.h>
#endif

#ifndef	ETHERTYPE_IP
#define	ETHERTYPE_IP	0x0800
#endif

#ifndef	IPPORT_RESERVED
#define	IPPORT_RESERVED	1024
#endif

#include <config.h>
#include <support.h>
#include <osdep.h>

#define	EXPIRE_INTERVAL	5			/* 5sec */
#define	EXPIRE_TCP	(300/EXPIRE_INTERVAL)	/* 5min */
#define	EXPIRE_UDP	(60/EXPIRE_INTERVAL)	/* 1min */
#define	EXPIRE_ICMP	(10/EXPIRE_INTERVAL)	/* 10sec */

#include "log.h"
#include "xcio.h"
/*#include "command.h"*/
#include "option.h"
#include "timer.h"
#include "frame.h"
#include "ipsupport.h"
#include "mbuf.h"
#include "slcompress.h"
#include "ipcp.h"
#include "console.h"
#include "filter.h"
#include "ipap.h"
#include "ipap.list"

/**********************************************************************
  IP Filter
 **********************************************************************/

struct ipfilter_s {
    struct ipfilter_s *next;
    struct ipaddrset_s {
	struct in_addr addr;		/* address */
	struct in_addr mask;		/* address mask */
	u_int16_t *hbo_port;		/* ports */
    } src, dst;
    f_act_t act;			/* action */
    struct in_addr ad_addr;		/* new address */
    struct modinfo_s *ad_mod;		/* modifier */
    time_t ad_keep;			/* keep time */
    u_int16_t *ad_hbo_bport;		/* 'burrow' ports */
    int ad_rsd;				/* rresvport() */
    int id;				/* filter ID */
    u_int16_t ad_nbo_port;		/* new port */
    u_int16_t ad_icmpid;		/* icmp echo/reply ID */
    priority_t ad_pri;			/* priority */
    u_int8_t tcp_flag, tcp_mask;
    u_int8_t ad_bproto;			/* 'burrow' protocol */
    u_int8_t proto;
    u_int8_t expire;
    u_int8_t flag;
};

struct modinfo_s {
    struct modinfo_s *next;
    int (*input)();
    int (*output)();
    u_int32_t d_seq, d_ack;
    u_int8_t xused;
    u_int8_t cused;
} *modHead;

static struct ipfilter_s *flHead[F_TYPE_NUM];

static struct slcompress slCs;
static bool_t slInitialized;
static struct timer_s expireTimer;

int
IpRoute(int argc, char **argv)
{
    extern char *ifName;
    char *dev;
    u_int32_t dest, gway, mask;
    int ret=0, cmd;

    /*
     * 0       1       2          3          4
     * iproute addhost remoteaddr localaddr  (dev)
     * iproute addnet  netaddr    netmask    (dev)
     * iproute addnet  default    remoteaddr (dev)
     * iproute delhost remoteaddr (dev)
     * iproute del(net)  netaddr    (dev)
     * iproute del(net)  default    (dev)
     */

    if (argc < 3) return(ret);
    if (!strncmp(argv[1], "add", 3))
	cmd = (!strcmp(argv[1], "addnet")) ? IPRT_ADD_NET:IPRT_ADD_HOST;
    else if (!strncmp(argv[1], "del", 3))
	cmd = (!strcmp(argv[1], "delnet")) ? IPRT_DEL_NET:IPRT_DEL_HOST;
    else return(ret);
    gway = 0;
    mask = (cmd & IPRT_ADD) ? 0xffffffff: 0;
    if (!strcmp(argv[2], "default")) {
	dest = 0;
	mask = 0;
	cmd &= ~IPRT_HOST;
    } else dest = inet_addr(argv[2]);
    if (argc > 3) {
	if (cmd == IPRT_ADD_NET && dest)
	    mask = inet_addr(argv[3]);
	else
	    gway = inet_addr(argv[3]);
    }
    dev = argv[argc - 1];
    if (cmd & IPRT_ADD) {
	if (argc < 5) dev = ifName;
	ret = SysIpRoute(cmd, dest, gway, mask, dev);
    } else {
	if (argc < 4) dev = ifName;
	ret = SysIpRoute(cmd, dest, gway, mask, dev);
    }
    return(ret);
}

int
IpConfig(int argc, char **argv)
{
    extern char *ifName;
    char *dev;
    int ret;
    /*
     * 0        1         2          3       4   5
     * ipconfig localaddr remoteaddr netmask mtu (dev)
     * ipconfig down      (dev)
     */

    if (argc < 2) return(-1);
    dev = argv[argc - 1];
    if (!strcasecmp(argv[1], "down")) {
	ret = SysIpConfig(0, 0, 0, MTU_DOWNFLAG, argc < 3 ? ifName: dev);
    } else {
	if (argc < 5) return(-1);
	ret = SysIpConfig(inet_addr(argv[1]), inet_addr(argv[2]),
			  inet_addr(argv[3]), atoi(argv[4]),
			  argc < 6 ? ifName: dev);
    }
    return(ret);
}

/*
 * initialize/setup
 */

static void
IpFilterFree(struct ipfilter_s *ifp)
{
    if (ifp->act == F_ACT_TRANSLATE) {
	if (ifp->ad_mod && ifp->ad_mod->xused) ifp->ad_mod->xused --;
	if (ifp->ad_rsd >= 0) close(ifp->ad_rsd);
    }
    if (ifp->act == F_ACT_MAPPER) {
	if (ifp->ad_hbo_bport) Free(ifp->ad_hbo_bport);
    }
    if (ifp->src.hbo_port) Free(ifp->src.hbo_port);
    if (ifp->dst.hbo_port) Free(ifp->dst.hbo_port);
    free(ifp);
}

static void
IpFilterExpire(struct ipfilter_s **head, u_int8_t mask)
{
    struct ipfilter_s *ifp, *ifp0=NULL;

    ifp = *head;
    while (ifp) {
	if (ifp->flag & mask) {
	    struct ipfilter_s *ifp1=ifp->next;

	    if (ifp0) ifp0->next = ifp1;
	    else *head = ifp1;
	    IpFilterFree(ifp);
	    ifp = ifp1;
	} else {
	    if (ifp->flag & IPFF_WILLEXPIRE) {
		ifp->flag &= ~IPFF_WILLEXPIRE;
		ifp->flag |= IPFF_EXPIRED;
	    } else if (ifp->expire) {
		ifp->expire --;
		if (!ifp->expire) {
		    if (ifp->ad_mod && ifp->ad_mod->cused) {
			/*
			 * ifp is a control connection !!
			 */
			ifp->expire = EXPIRE_TCP;
		    }
		    if (!ifp->expire) ifp->flag |= IPFF_EXPIRED;
		}
	    }
	    ifp0 = ifp;
	    ifp = ifp->next;
	}
    }
}

static void
IpModExpire()
{
    struct modinfo_s *mod, *mod0=NULL;
    mod = modHead;
    while (mod) {
	if (mod->xused) {
	    mod0 = mod;
	    mod = mod->next;
	} else {
	    struct modinfo_s *mod1=mod->next;

	    if (mod0) mod0->next = mod1;
	    else modHead = mod1;
	    free(mod);
	    mod = mod1;
	}
    }
}

static void
ExpireTimerHandler()
{
    expireTimer.rest = EXPIRE_INTERVAL;
    IpFilterExpire(&flHead[F_TYPE_INPUT], IPFF_EXPIRED);
    IpFilterExpire(&flHead[F_TYPE_OUTPUT], IPFF_EXPIRED);
    IpModExpire();
}

void
IpInit()
{
    SysIpInit();
    expireTimer.rest = EXPIRE_INTERVAL;
    TimerAdd(&expireTimer);
    IpFilterExpire(&flHead[F_TYPE_INPUT], IPFF_AUTO);
    IpFilterExpire(&flHead[F_TYPE_OUTPUT], IPFF_AUTO);
    IpModExpire();
}

void
VjInit(bool_t initialize)
{
    if (initialize) {
	sl_compress_init(&slCs, ipcprReg.cproto.opt.vj.max_slot_id);
	slInitialized = TRUE;
    } else slInitialized = FALSE;
}

void
IpSetup()
{
    int n=0;
    struct servent *sp;

    for (n = 0; modList[n].name; n ++) {
	if ((modList[n].proto == IPPROTO_TCP
	     || modList[n].proto == IPPROTO_UDP)
	    && !modList[n].nbo_port) {
	    sp = getservbyname(modList[n].name,
			       (modList[n].proto == IPPROTO_TCP)
			       ? "tcp": "udp");
	    if (sp) {
		modList[n].nbo_port = sp->s_port;
	    }
	}
    }
    expireTimer.name = "IP";
    expireTimer.f_interval = 1;
    expireTimer.callback = ExpireTimerHandler;
#ifdef	DEBUG_IP
    DebugOpen();
#endif
}

/**********************************************************************
  IP filter functions
 **********************************************************************/

#define	PROXYPORT_BASE	40000

u_int16_t
ProxyNboPort(bool_t next)
{
    static u_int16_t port;

    if (next) port ++;
    return(htons(port|PROXYPORT_BASE));
}

/*
 * IpTranslateAddress
 *	rewrite IP address and recalculate IP/TCP/UDP checksums
 *	ff: filter type
 *		F_TYPE_INPUT          -> rewrite destination IP address
 *		other (F_TYPE_OUTPUT) -> rewrite source IP address
 *	iphp: pointer to IP packet
 *	tcphdr: pointer to TCP/UDP packet
 *	len: IP packet length
 *	ifp: pointer to the filter
 */

static struct ipfilter_s *
IpXFilterAdd(struct ipfilter_s **headp,
	     u_int32_t src, u_int32_t smsk,
	     u_int32_t dst, u_int32_t dmsk, u_int32_t addr,
	     u_int16_t nbo_sp, u_int16_t nbo_dp, u_int16_t port,
	     u_int8_t proto, u_int8_t flag)
{
    struct ipfilter_s *ifp;

    ifp = *headp;
    while (ifp) {
	if (ifp->act == F_ACT_TRANSLATE
	    && ifp->proto == proto
	    && ifp->src.addr.s_addr == src
	    && ifp->dst.addr.s_addr == dst) {
	    if (ifp->proto == IPPROTO_ICMP) {
		if ((ifp->flag & IPFF_ICMPID) && ifp->ad_icmpid == port)
		    return(ifp);
	    } else {
		if (ifp->src.hbo_port && ifp->dst.hbo_port
		    && (*ifp->src.hbo_port == ntohs(nbo_sp))
		    && (*ifp->dst.hbo_port == ntohs(nbo_dp)))
		    return(ifp);
	    }
	}
	ifp = ifp->next;
    }

    ifp = TCALLOC(struct ipfilter_s);
    ifp->id = *headp ? (*headp)->id + 1: 1;
    ifp->next = *headp;
    ifp->act = F_ACT_TRANSLATE;
    ifp->proto = proto;
    ifp->flag = flag;
    ifp->ad_rsd = -1;
    *headp = ifp;

    /*
      if ((ifp->src.addr.s_addr = src) != 0)
      ifp->src.mask.s_addr = (u_int32_t)-1;
      if ((ifp->dst.addr.s_addr = dst) != 0)
      ifp->dst.mask.s_addr = (u_int32_t)-1;
     */

    ifp->src.addr.s_addr = src;
    ifp->dst.addr.s_addr = dst;
    ifp->src.mask.s_addr = smsk;
    ifp->dst.mask.s_addr = dmsk;
    ifp->ad_addr.s_addr = addr;

    if (nbo_sp) {
	ifp->src.hbo_port = Malloc(sizeof(u_int16_t) * 3);
	ifp->src.hbo_port[0] = ntohs(nbo_sp);
	ifp->src.hbo_port[1] = ntohs(nbo_sp);
	ifp->src.hbo_port[2] = 0;
    }
    if (nbo_dp) {
	ifp->dst.hbo_port = Malloc(sizeof(u_int16_t) * 3);
	ifp->dst.hbo_port[0] = ntohs(nbo_dp);
	ifp->dst.hbo_port[1] = ntohs(nbo_dp);
	ifp->dst.hbo_port[2] = 0;
    }
    printf("IpXFilterAdd\n");
    return(ifp);
}

struct ipfilter_s *
IpXFilterPairAdd(struct ipfilter_s *cifp,
		 u_int32_t src, u_int32_t dst, u_int32_t addr,
		 u_int16_t nbo_sp, u_int16_t nbo_dp, u_int16_t port,
		 u_int8_t proto, u_int8_t flag)
{
    struct ipfilter_s *ifp, *ofp;

    ifp = IpXFilterAdd(&flHead[F_TYPE_INPUT],
		       dst, INADDR_NONE,
		       addr, INADDR_NONE,
#if 1
		       src ? src: ipcpOpt.l_addr.s_addr,
#else
		       src ? src: ipcplReg.addr.s_addr,
#endif
		       nbo_dp, nbo_sp, port, proto, IPFF_AUTO);
    ofp = IpXFilterAdd(&flHead[F_TYPE_OUTPUT],
		       src, src ? INADDR_NONE: 0,
		       dst, INADDR_NONE,
		       addr, nbo_sp, nbo_dp, port, proto, IPFF_AUTO);
    if (proto == IPPROTO_ICMP) {
	ifp->ad_icmpid = ofp->ad_icmpid = port;
	ifp->flag |= IPFF_ICMPID;
	ofp->flag |= IPFF_ICMPID;
	ifp->expire = ofp->expire = EXPIRE_ICMP;
    } else {
	int n = 0;
	struct modinfo_s *mod=NULL;

	while (modList[n].proto) {
	    if (modList[n].proto == proto
		&& (modList[n].nbo_port == nbo_sp
		    || modList[n].nbo_port == nbo_dp)) {
		mod = TCALLOC(struct modinfo_s);
		mod->input = modList[n].input;
		mod->output = modList[n].output;
		mod->xused = 2;
		if (modHead) mod->next = modHead;
		modHead = mod;
		break;
	    }
	    n ++;
	}
	if (!mod && port) {
	    int hbo_port;

	    hbo_port = ntohs(nbo_sp);
	    if (0 < hbo_port && hbo_port < IPPORT_RESERVED) {
		SuperPrivilege(TRUE);
		if ((ifp->ad_rsd = rresvport(&hbo_port)) >= 0)
		    port = htons(hbo_port);
		SuperPrivilege(FALSE);
	    }
	}
	ifp->ad_mod = ofp->ad_mod = mod;
	ifp->expire = ofp->expire =
	    (proto == IPPROTO_UDP) ? EXPIRE_UDP: EXPIRE_TCP;
	if (port) {
	    ofp->ad_nbo_port = port;
	    ifp->ad_nbo_port = nbo_sp;
	    ifp->dst.hbo_port[0] = ifp->dst.hbo_port[1] = ntohs(port);
	}
	if (cifp && cifp->ad_mod) {
	    cifp->ad_mod->cused ++;
#if 0
	    ifp->cifp = ofp->cofp = cifp;
#endif
	}
    }
    return(ofp);
}

static void
IpTranslateAddress(struct ipfilter_s *ifp, f_type_t ff,
		   ipv4_hdr *iphp, tcp_hdr *tcphp, int *lenp)
{
    u_int16_t sum;

    if ((sum = ip_calc_csum(iphp)) != 0) {
	Logf(LOG_ERROR, "IP: bad checksum(%#x)\n", iphp->ip_sum);
    }

    if (ff == F_TYPE_INPUT) iphp->ip_dst.s_addr = ifp->ad_addr.s_addr;
    else iphp->ip_src.s_addr = ifp->ad_addr.s_addr;

    if (iphp->ip_p == IPPROTO_TCP) {
	if (ifp->ad_mod) {
	    int dif=0;

	    if (ff == F_TYPE_INPUT) {
		if (ifp->ad_mod->input) {
		    dif = ifp->ad_mod->input();
		}
		tcphp->th_ack = htonl(ntohl(tcphp->th_ack)
				      - ifp->ad_mod->d_seq);
	    } else {
		tcphp->th_seq = htonl(ntohl(tcphp->th_seq)
				      + ifp->ad_mod->d_seq);
		if (ifp->ad_mod->output) {
		    dif = ifp->ad_mod->output(ifp, iphp, *lenp);
		    iphp->ip_len = htons(ntohs(iphp->ip_len) + dif);
		    *lenp += dif;
		}
		ifp->ad_mod->d_seq += dif;
	    }
	}
	if (ifp->ad_nbo_port) {
	    if (ff == F_TYPE_INPUT) tcphp->th_dport = ifp->ad_nbo_port;
	    else tcphp->th_sport = ifp->ad_nbo_port;
	}
	tcphp->th_sum = 0;
	tcphp->th_sum = tcpudp_check_sum(iphp, IPPROTO_TCP,
					 *lenp - (iphp->ip_hl << 2));
    } else if (iphp->ip_p == IPPROTO_UDP) {
	udp_hdr *udphp = (udp_hdr *)tcphp;
	if (ifp->ad_nbo_port) {
	    if (ff == F_TYPE_INPUT) udphp->uh_dport = ifp->ad_nbo_port;
	    else udphp->uh_sport = ifp->ad_nbo_port;
	}
	udphp->uh_sum = 0;
	udphp->uh_sum = tcpudp_check_sum(iphp, IPPROTO_UDP,
					 *lenp - (iphp->ip_hl << 2));
    }
    if (!sum) ip_check_sum(iphp);
}

void
IpXFilterRemap()
{
    struct ipfilter_s *ifp;

    ifp = flHead[F_TYPE_INPUT];
    while (ifp) {
	if ((ifp->flag & IPFF_AUTO) && (ifp->act == F_ACT_TRANSLATE)
	    && (ifp->dst.addr.s_addr == ipcpOpt.l_addr.s_addr))
	    ifp->dst.addr.s_addr = ipcplReg.addr.s_addr;
	ifp = ifp->next;
    }

    ifp = flHead[F_TYPE_OUTPUT];
    while (ifp) {
	if ((ifp->flag & IPFF_AUTO) && (ifp->act == F_ACT_TRANSLATE)
	    && (ifp->ad_addr.s_addr == ipcpOpt.l_addr.s_addr))
	    ifp->ad_addr.s_addr = ipcplReg.addr.s_addr;
	ifp = ifp->next;
    }
}

static void
IpBFilterMapper(struct ipfilter_s *mfp, f_type_t ff,
		ipv4_hdr *iphp, tcp_hdr *dp, int *lenp)
{
    if (ff == F_TYPE_OUTPUT) {
	u_int32_t new=mfp->ad_addr.s_addr ?
	    mfp->ad_addr.s_addr: ipcplReg.addr.s_addr;
	struct ipfilter_s *ifp, *ofp;
	int n=1;
	u_int16_t *hbo_bp;

	mfp->flag |= IPFF_USED;
	ifp = IpXFilterAdd(&flHead[F_TYPE_INPUT],
			   iphp->ip_dst.s_addr, INADDR_NONE,
			   new, INADDR_NONE,
			   iphp->ip_src.s_addr,
			   0, 0, 0, mfp->ad_bproto, IPFF_AUTO);
	ofp = IpXFilterAdd(&flHead[F_TYPE_OUTPUT],
			   iphp->ip_src.s_addr, INADDR_NONE,
			   iphp->ip_dst.s_addr, INADDR_NONE,
			   new, 0, 0, 0, mfp->ad_bproto, IPFF_AUTO);

	ifp->expire = ofp->expire = 0;
	hbo_bp = mfp->ad_hbo_bport;
	while (*hbo_bp) {
	    n ++;
	    hbo_bp ++;
	}
	if (n > 1) {
	    int i;

	    ifp->dst.hbo_port = Malloc(sizeof(u_int16_t) * n);
	    ofp->src.hbo_port = Malloc(sizeof(u_int16_t) * n);
	    for (i = 0; i < n; i ++) {
		ifp->dst.hbo_port[i] = ofp->src.hbo_port[i]
		    =  mfp->ad_hbo_bport[i];
	    }
	}
    }
}

static void
IpXFilterMapper(struct ipfilter_s *ifp, f_type_t ff,
		ipv4_hdr *iphp, tcp_hdr *dp, int *lenp)
{
    if (ff == F_TYPE_OUTPUT) {
	u_int32_t new=ifp->ad_addr.s_addr ?
	    ifp->ad_addr.s_addr: ipcplReg.addr.s_addr;
	switch(iphp->ip_p) {
	case IPPROTO_TCP:
	    ifp = IpXFilterPairAdd(NULL, iphp->ip_src.s_addr,
				   iphp->ip_dst.s_addr,
				   new,
				   ((tcp_hdr *)dp)->th_sport,
				   ((tcp_hdr *)dp)->th_dport,
				   ProxyNboPort(TRUE),
				   IPPROTO_TCP, IPFF_AUTO);
	    break;
	case IPPROTO_UDP:
	    ifp = IpXFilterPairAdd(NULL, iphp->ip_src.s_addr,
				   iphp->ip_dst.s_addr,
				   new,
				   ((udp_hdr *)dp)->uh_sport,
				   ((udp_hdr *)dp)->uh_dport,
				   ProxyNboPort(TRUE),
				   IPPROTO_UDP, IPFF_AUTO);
	    break;
	case IPPROTO_ICMP:
	    if (((icmp_hdr *)dp)->icmp_type == ICMP_ECHO) {
		ifp = IpXFilterPairAdd(NULL, iphp->ip_src.s_addr,
				       iphp->ip_dst.s_addr,
				       new,
				       0, 0, ((icmp_hdr *)dp)->icmp_id,
				       IPPROTO_ICMP, IPFF_AUTO);
	    }
	    break;
	}
    }
    IpTranslateAddress(ifp, ff, iphp, dp, lenp);
}

static int
addrbyname(char *name, struct in_addr *addr)
{
    struct hostent *hp;

    if ((hp = gethostbyname(name)) != NULL) {
        addr->s_addr = *(u_int32_t *)hp->h_addr;
        return 1;
    } else {
	return(inet_aton(name, addr));
    }
}

static u_int16_t
portbyname(char *name, u_int8_t proto)
{
    struct servent *sp;
    char *pname=(proto == IPPROTO_TCP) ? "tcp": "udp";

    if ((sp = getservbyname(name, pname)) != NULL)
	return(ntohs(sp->s_port));
    else
	return(atoi(name));
}

static int
next_option(char *narg)
{
    if (narg[0] == '-') {
        switch (narg[1]) {
        case 'S':
        case 'D':
        case 'X':
        case 'F':
        case 'K':
        case 'P':
        case 'B':
            return narg[1];
        }
    }
    return 0;
}

static int
addrset(struct ipaddrset_s *isp, char *str)
{
    char *p;

    if ((p = strchr(str, '/')) != NULL) {
	*p = '\0';
	p ++;
    }
    if (!addrbyname(str, &isp->addr)) {
        return -1;
    }
    if (p) {
	if (strchr(p, '.')) {
            if (!addrbyname(p, &isp->mask)) {
                return -1;
            }
        } else {
            if (atoi(p)>=0 && atoi(p)<=32) {
                isp->mask.s_addr =
		    htonl((u_int32_t) 0xffffffff << (32 - atoi(p)));
            } else {
                return-1;
            }
        }
    } else if (isp->addr.s_addr) {
        isp->mask.s_addr = (u_int32_t) 0xffffffff;
    } else {
        isp->mask.s_addr = 0;
    }

    return 1;
}

static int
portset(u_int16_t **hbo_pp, u_int8_t proto, int argc, char **argv)
{
    int i, n;
    char *p;
    u_int16_t port, from, til, *hbo_port;

    for(n=0; n<argc && !next_option(argv[n]); n++);
    if (n <= 0) return(0);

    *hbo_pp = hbo_port = Malloc(sizeof(u_int16_t) * (n*2+1));
    hbo_port[n*2]=0;

    for (i=0; i<n; i++) {
        if ((p = strchr(argv[i], ':')) !=  NULL) {
            *p = '\0';
            p ++;
            from = portbyname(argv[i], proto);
            til  = portbyname(p, proto);
            if (!from) from = 1;
            if (!til)  til  = 0xffff;
            if (from<=til) {
                hbo_port[i*2]   = from;
                hbo_port[i*2+1] = til;
            } else {
                return -1;
            }
        } else {
	    port = portbyname(argv[i], proto);
            if (port) {
                hbo_port[i*2]   = port;
                hbo_port[i*2+1] = port;
            } else {
                return -1;
            }
	}
    }

    return(n);
}

static int
flagset(u_int8_t *flag, u_int8_t *flag_mask, u_int8_t proto,
        int argc, char **argv)
{
    int i;

    for (i=0; i< argc && !next_option(argv[i]); i++) {
        if (!strcmp(argv[i], "syn"))       *flag |= TH_SYN;
        else if (!strcmp(argv[i], "fin"))  *flag |= TH_FIN;
        else if (!strcmp(argv[i], "rst"))  *flag |= TH_RST;
        else if (!strcmp(argv[i], "ack"))  *flag |= TH_ACK;
        else if (!strcmp(argv[i], "!syn")) *flag_mask |= TH_SYN;
        else if (!strcmp(argv[i], "!fin")) *flag_mask |= TH_FIN;
        else if (!strcmp(argv[i], "!rst")) *flag_mask |= TH_RST;
        else if (!strcmp(argv[i], "!ack")) *flag_mask |= TH_ACK;
        else return -1;
    }
    if (*flag & *flag_mask) {
        return -1;
    }

    return i;
}

int
IpFilterAdd(int proto, f_type_t ff, int id, f_act_t act,
	    time_t keeptime, priority_t pri, int argc, char **argv)
{
    struct ipfilter_s *ifp, **headp;
    char buf[256];
    int a;
    int rst;

    if (argc < 0) return(-1);

    /*
     * -F <flag> -S <src>/<smsk> <sp> -D <dst>/<dmsk> <dp> -X <new>
     * -F <flag> -S <src>/<smsk> <sp> -D <dst>/<dmsk> <dp> -X <new> -K <sec>
     * -F <flag> -S <src>/<smsk> <sp> -D <dst>/<dmsk> <dp> -P <priority>
     */

    headp = &flHead[ff];
    ifp = TCALLOC(struct ipfilter_s);
    ifp->id = *headp ? (*headp)->id + 1: 1;
    ifp->src.addr.s_addr = 0;
    ifp->src.mask.s_addr = 0;
    ifp->src.hbo_port = NULL;
    ifp->ad_rsd = -1;

    ifp->dst.addr.s_addr = 0;
    ifp->dst.mask.s_addr = 0;
    ifp->dst.hbo_port = NULL;

    switch (act) {
    case F_ACT_KEEPTIME:
	ifp->ad_keep = keeptime;
	break;
    case F_ACT_PRIORITY:
	ifp->ad_pri = pri;
	break;
    }

    a=0;
    while (a < argc) {
	switch (next_option(argv[a++])) {
	case 'S':
	    if (ifp->src.addr.s_addr 
		|| ifp->src.mask.s_addr 
		|| ifp->src.hbo_port) {
		sprintf(buf, "Error: multiple -S option\n");
		ConsoleOutf(buf);
		goto badform;
	    }
	    rst = addrset(&ifp->src, argv[a]);
	    if (rst < 0) {
		sprintf(buf, "Error: wrong source addr: %-.16s\n", argv[a]);
		ConsoleOutf(buf);
		goto badform;
	    } else a += rst;
	    rst = portset(&ifp->src.hbo_port, proto, argc - a, &argv[a]);
	    if (rst > 0 && proto != IPPROTO_TCP && proto != IPPROTO_UDP) {
		sprintf(buf, "Error: port without TCP nor UDP\n");
		ConsoleOutf(buf);
		goto badform;
	    } else if (rst < 0) {
		sprintf(buf, "Error: wrong source port\n");
		ConsoleOutf(buf);
		goto badform;
	    } else a+= rst;
	    break;
	case 'D':
	    if (ifp->dst.addr.s_addr 
		|| ifp->dst.mask.s_addr 
		|| ifp->dst.hbo_port) {
		sprintf(buf, "Error: multiple -D option\n");
		ConsoleOutf(buf);
		goto badform;
	    }
	    rst = addrset(&ifp->dst, argv[a]);
	    if (rst < 0) {
		sprintf(buf,"Error: wrong destination addr: %-.16s\n",argv[a]);
		ConsoleOutf(buf);
		goto badform;
	    } else a += rst;
	    rst = portset(&ifp->dst.hbo_port, proto, argc - a, &argv[a]);
	    if (rst > 0 && proto != IPPROTO_TCP && proto != IPPROTO_UDP) {
		sprintf(buf, "Error: port without TCP nor UDP\n");
		ConsoleOutf(buf);
		goto badform;
	    } else if (rst < 0) {
		sprintf(buf, "Error: wrong destination port\n");
		ConsoleOutf(buf);
		goto badform;
	    } else a+= rst;
	    break;
	case 'X':
	    if (ifp->ad_addr.s_addr) {
		sprintf(buf, "Error: multiple -X option\n");
		ConsoleOutf(buf);
		goto badform;
	    }
	    if (act != F_ACT_MAPPER) {
		sprintf(buf, "Error: Wrong use of -X option\n");
		ConsoleOutf(buf);
		goto badform;
	    }
	    rst = inet_aton(argv[a], &ifp->ad_addr);
	    if (rst < 0) {
		sprintf(buf, "Error: Wrong translate addr: %-.16s\n", argv[a]);
		ConsoleOutf(buf);
		goto badform; 
	    } else a++;
	    break;
        case 'F':
	    if (proto != IPPROTO_TCP) {
		sprintf(buf, "Error: Flag without TCP.\n");
		ConsoleOutf(buf);
		goto badform;
	    }
	    rst = flagset(&ifp->tcp_flag, &ifp->tcp_mask, 
			  proto, argc -a, &argv[a]);
	    if (rst <= 0) {
		sprintf(buf, "Error: Wrong flags\n");
		ConsoleOutf(buf);
		goto badform;
	    } else a+= rst;
	    break;
	case 'K':
	    if (act != F_ACT_KEEPTIME) {
		sprintf(buf, "Error: Wrong use of -K option\n");
		ConsoleOutf(buf);
		goto badform;
	    }
	    ifp->ad_keep = atol(argv[a++]);
	    break;
	case 'P':
	    if (ifp->ad_pri) {
		sprintf(buf, "Error: multiple -P option\n");
		ConsoleOutf(buf);
		goto badform;
	    }
	    if (act != F_ACT_PRIORITY) {
		sprintf(buf, "Error: Wrong use of -P option\n");
		ConsoleOutf(buf);
		goto badform;
	    }
	    ifp->ad_pri = atoi(argv[a]);
	    if (ifp->ad_pri >= PRI_NPMAX) {
		sprintf(buf, "Error: Wrong priority: %s\n", argv[a]);
		ConsoleOutf(buf);
		goto badform; 
	    } else a++;
	    break;
	case 'B':
	    if (ifp->ad_hbo_bport) {
		sprintf(buf, "Error: multiple -B option\n");
		ConsoleOutf(buf);
		goto badform;
	    }
	    if (!strcasecmp(argv[a], "tcp"))
		ifp->ad_bproto = IPPROTO_TCP;
	    else if (!strcasecmp(argv[a], "udp"))
		ifp->ad_bproto = IPPROTO_UDP;
	    if (ifp->ad_bproto) a ++;
	    else goto badform;
	    rst = portset(&ifp->ad_hbo_bport, proto, argc - a, &argv[a]);
	    if (rst > 0 && proto != IPPROTO_TCP && proto != IPPROTO_UDP) {
		sprintf(buf, "Error: port without TCP nor UDP\n");
		ConsoleOutf(buf);
		goto badform;
	    } else if (rst < 0) {
		sprintf(buf, "Error: wrong source port\n");
		ConsoleOutf(buf);
		goto badform;
	    } else a+= rst;
	    break;
	default:
	    sprintf(buf, "Error: Unknown Option %-.16s\n", argv[--a]);
	    ConsoleOutf(buf);
	    goto badform;
	}
    }

    ifp->next = *headp;
    ifp->act = act;
    ifp->proto = proto;
    *headp = ifp;
    printf("IpFilterAdd\n");
    return(0);
 badform:
    if (ifp->src.hbo_port) Free(ifp->src.hbo_port);
    if (ifp->dst.hbo_port) Free(ifp->dst.hbo_port);
    Free(ifp);
    return(-1);
}

void
IpFilterDel(int proto, f_type_t ff, int id)
{
    struct ipfilter_s **headp, *ifp, *next, *prev=NULL;

    headp = &flHead[ff];
    ifp = *headp;
    while (ifp) {
	if (id < 0 || ifp->id == id) {
	    next = ifp->next;
	    if (ifp->src.hbo_port) Free(ifp->src.hbo_port);
	    if (ifp->dst.hbo_port) Free(ifp->dst.hbo_port);
	    Free(ifp);
	    if (id >= 0) {
		if (prev) prev->next = next;
		else *headp = next;
		break;
	    }
	    ifp = next;
	} else {
	    prev = ifp;
	    ifp = ifp->next;
	}
    }
    if (id < 0) *headp = NULL;
    printf("IpFilterDel\n");
}

void
IpFilterShow(int proto, f_type_t ff)
{
    char buf[1024], pstr[15];
    u_int16_t *pp;
    int i=1, b, k;
    struct ipfilter_s *ifp;

    ifp = flHead[ff];
    switch (ff) {
    case F_TYPE_TRIGGER:
	ff = 'T';
	break;
    case F_TYPE_INPUT:
	ff = 'I';
	break;
    case F_TYPE_OUTPUT:
	ff = 'O';
	break;
    }
    while (ifp) {
	if (!proto || proto == ifp->proto) {
	    sprintf(buf, "%3d: %4s %c%c%c", ifp->id,
		    (ifp->proto == IPPROTO_TCP) ? "TCP":
		    (ifp->proto == IPPROTO_UDP) ? "UDP":
		    (ifp->proto == IPPROTO_ICMP) ? "ICMP": "IP",
		    (ifp->flag & IPFF_EXPIRED) ? '-':
		    ((ifp->flag & IPFF_USED) ? '!': '+'),
		    (ifp->act == F_ACT_ACCEPT) ? 'a':
		    (ifp->act == F_ACT_REJECT) ? 'r':
		    (ifp->act == F_ACT_MAPPER) ? 'm':
		    (ifp->act == F_ACT_TRANSLATE) ? 'x':
		    (ifp->act == F_ACT_PRIORITY) ? 'p': 'k', ff);

	    if (ifp->src.addr.s_addr != 0
		|| ifp->src.addr.s_addr != 0
		|| ifp->src.hbo_port != 0) {
		strcat(buf, " -S ");
		strcat(buf, inet_ntoa(ifp->src.addr));

		strcat(buf, "/");
		for (b = k = 0; k < 32; k ++)
		    if ((ifp->src.mask.s_addr >> k) & 1) b ++;
		sprintf(pstr, "%d", b);
		strcat(buf, pstr);

		pp = ifp->src.hbo_port;
		if (pp) {
		    while (*pp) {
			if (pp[0] == pp[1]) {
			    sprintf(pstr, " %d", pp[0]);
			} else {
			    sprintf(pstr, " %d:%d", pp[0], pp[1]);
			}
			strcat(buf, pstr);
			pp +=2;
		    }
		}
	    }

	    if (ifp->dst.addr.s_addr != 0
		|| ifp->dst.addr.s_addr != 0
		|| ifp->dst.hbo_port != 0) {
		strcat(buf, " -D ");
		strcat(buf, inet_ntoa(ifp->dst.addr));
	    
		strcat(buf, "/");
		for (b = k = 0; k < 32; k ++)
		    if ((ifp->dst.mask.s_addr >> k) & 1) b ++;
		sprintf(pstr, "%d", b);
		strcat(buf, pstr);

		pp = ifp->dst.hbo_port;
		if (pp) {
		    while (*pp) {
			if (pp[0] == pp[1]) {
			    sprintf(pstr, " %d", pp[0]);
			} else {
			    sprintf(pstr, " %d:%d", pp[0], pp[1]);
			}
			strcat(buf, pstr);
			pp += 2;
		    }
		}
	    }
	    if (ifp->act == F_ACT_MAPPER && ifp->ad_hbo_bport) {
		strcat(buf, " -B ");

		if (ifp->ad_bproto == IPPROTO_TCP)
		    strcat(buf, "TCP");
		else if (ifp->ad_bproto == IPPROTO_UDP)
		    strcat(buf, "UDP");
		pp = ifp->ad_hbo_bport;
		if (pp) {
		    while (*pp) {
			if (pp[0] == pp[1]) {
			    sprintf(pstr, " %d", pp[0]);
			} else {
			    sprintf(pstr, " %d:%d", pp[0], pp[1]);
			}
			strcat(buf, pstr);
			pp += 2;
		    }
		}
	    }
	    if (ifp->flag & IPFF_ICMPID) {
		sprintf(pstr, " 0x%x", ifp->ad_icmpid);
		strcat(buf, pstr);
	    }

	    if (ifp->ad_addr.s_addr && (ifp->act == F_ACT_TRANSLATE
					|| ifp->act == F_ACT_MAPPER)) {
		if (ifp->ad_mod) strcat(buf, " -X ");
		else strcat(buf, " -x ");
		strcat(buf, inet_ntoa(ifp->ad_addr));
		if (ifp->ad_nbo_port) {
		    sprintf(pstr, " %d", ntohs(ifp->ad_nbo_port));
		    strcat(buf, pstr);
		}
	    }
	    if ((ifp->proto == IPPROTO_TCP)
		&& (ifp->tcp_flag || ifp->tcp_mask)) {
		strcat(buf, " -F");
		if (ifp->tcp_flag & TH_SYN) strcat(buf, " syn");
		if (ifp->tcp_flag & TH_FIN) strcat(buf, " fin");
		if (ifp->tcp_flag & TH_ACK) strcat(buf, " ack");
		if (ifp->tcp_flag & TH_RST) strcat(buf, " rst");
                
		if (ifp->tcp_mask & TH_SYN) strcat(buf, " !syn");
		if (ifp->tcp_mask & TH_FIN) strcat(buf, " !fin");
		if (ifp->tcp_mask & TH_ACK) strcat(buf, " !ack");
		if (ifp->tcp_mask & TH_RST) strcat(buf, " !rst");

	    }
	    if (ifp->act == F_ACT_KEEPTIME) {
		sprintf(pstr, " -K %ld", ifp->ad_keep);
		strcat(buf, pstr);
	    }
	    if (ifp->act == F_ACT_PRIORITY) {
		sprintf(pstr, " -P %d", ifp->ad_pri);
		strcat(buf, pstr);
	    }
	    if (ifp->flag & IPFF_AUTO) {
		if (ifp->expire) {
		    sprintf(pstr, " (%d)",
			    ifp->expire * EXPIRE_INTERVAL);
		    strcat(buf, pstr);
		} else strcat(buf, " (auto)");
	    }
	    strcat(buf, "\n");
	    ConsoleOutf(buf);
	}
	ifp = ifp->next;
    }
}

f_act_t
IpFilterCheck(f_type_t ff, u_char *buf, int *lenp, time_t *idlep,
	      priority_t *prip)
{
    ipv4_hdr *iphp=(ipv4_hdr *)buf;
    struct ipfilter_s *ifp;
    tcp_hdr *tcphp=(tcp_hdr *)(buf + (iphp->ip_hl << 2));
    u_int16_t *pp, port;
    bool_t transed=FALSE;

    expireTimer.st_expired = 1;
    ifp = flHead[ff];
    do {
	if (ifp->proto != IPPROTO_IP && ifp->proto != iphp->ip_p)
	    continue;

	/* check source/destination address */
	if ((iphp->ip_src.s_addr & ifp->src.mask.s_addr)
	    != ifp->src.addr.s_addr
	    || (iphp->ip_dst.s_addr & ifp->dst.mask.s_addr)
	    != ifp->dst.addr.s_addr)
	    continue;

	if (ifp->proto != IPPROTO_TCP && ifp->proto != IPPROTO_UDP) {
	    if (ifp->proto == IPPROTO_ICMP) {
		if ((ifp->flag & IPFF_ICMPID)
		    && (((icmp_hdr *)tcphp)->icmp_id != ifp->ad_icmpid))
		    continue;
		if (ifp->flag & IPFF_AUTO) ifp->expire = EXPIRE_ICMP;
	    }
	    goto packet_matched;
	}

	/* for tcp/udp */
	if ((pp = ifp->src.hbo_port)) {
	    /* check source port */
	    if (*pp) {
		port = ntohs(tcphp->th_sport);
		while (*pp) {
		    if (pp[0] <= port && port <= pp[1]) break;
		    pp += 2;
		}
		if (!*pp) continue;
	    }
	}
	if ((pp = ifp->dst.hbo_port)) {
	    /* check destination port */
	    if (*pp) {
		port = ntohs(tcphp->th_dport);
		while (*pp) {
		    if (pp[0] <= port && port <= pp[1]) break;
		    pp += 2;
		}
		if (!*pp) continue;
	    }
	}
	if (ifp->proto == IPPROTO_TCP) {
	    if (ifp->tcp_flag) {
		if (!(tcphp->th_flags & ifp->tcp_flag)) continue;
	    }
	    if (ifp->tcp_mask) {
		if (!(~(tcphp->th_flags) & ifp->tcp_mask)) continue;
	    }
	    if (ifp->flag & IPFF_AUTO) {
		if (tcphp->th_flags & (TH_FIN|TH_RST))
		    ifp->flag |= IPFF_WILLEXPIRE;
		else ifp->expire = EXPIRE_TCP;
	    }
	} else if (ifp->flag & IPFF_AUTO) ifp->expire = EXPIRE_UDP;
    packet_matched:
	switch (ifp->act) {
	case F_ACT_TRANSLATE:
	    IpTranslateAddress(ifp, ff, iphp, tcphp, lenp);
	    transed = TRUE;
	    break;
	case F_ACT_PRIORITY:
	    if (prip) {
		*prip = ifp->ad_pri;
		prip = NULL;
	    }
	    break;
	case F_ACT_KEEPTIME:
	    if (idlep) {
		*idlep = ifp->ad_keep;
		idlep = NULL;
	    }
	    break;
	case F_ACT_MAPPER:
	    if (!transed) {
		if (ifp->ad_hbo_bport) {
		    if (!(ifp->flag & IPFF_USED))
			IpBFilterMapper(ifp, ff, iphp, tcphp, lenp);
		} else {
		    IpXFilterMapper(ifp, ff, iphp, tcphp, lenp);
		    transed = TRUE;
		}
	    }
	    break;
	default:
	    expireTimer.st_expired = 0;
	    return(ifp->act);
	}
    } while ((ifp = ifp->next));
    expireTimer.st_expired = 0;
    return(F_ACT_ACCEPT);
}

/**********************************************************************
  IP packet send/receive
 **********************************************************************/

void
IpMapperAdd()
{
    struct ipfilter_s *ifp, *ofp, **hpp;

    ofp = IpXFilterAdd(&flHead[F_TYPE_OUTPUT],
		       ipcpOpt.l_addr.s_addr, INADDR_NONE,
		       0, 0,
		       ipcplReg.addr.s_addr,
		       0, 0, 0, IPPROTO_IP, IPFF_AUTO);
    ofp->act = F_ACT_MAPPER;

    /* filter ip +rI -D ppp-localaddress/32 */
    ifp = TCALLOC(struct ipfilter_s);
    ifp->id = flHead[F_TYPE_INPUT] ? flHead[F_TYPE_INPUT]->id + 1: 1;
    ifp->ad_rsd = -1;
    ifp->dst.addr.s_addr = ipcplReg.addr.s_addr;
    ifp->dst.mask.s_addr = INADDR_NONE;
    ifp->proto = IPPROTO_IP;
    ifp->act = F_ACT_REJECT;
    ifp->flag |= IPFF_AUTO;
    ifp->next = flHead[F_TYPE_INPUT];
    flHead[F_TYPE_INPUT] = ifp;
#if 0
    ifp = IpXFilterAdd(&flHead[F_TYPE_INPUT], 0, 0,
		       ipcplReg.addr.s_addr, INADDR_NONE,
		       ipcpOpt.l_addr.s_addr,
		       0, 0, 0, IPPROTO_IP, IPFF_AUTO);
    ifp->act = F_ACT_MAPPER;
#endif
}

/*
 * IpRebuild
 *	rebuild IP packet in queue and register translation filter
 *	bp: pointer to IP packet
 *	len: IP packet len
 */

void
IpRebuild(u_char *bp, int len)
{
    ipv4_hdr *iphp;

    iphp = (ipv4_hdr *)bp;
    /* this packet uses old IP address ? */
    if (iphp->ip_src.s_addr
	&& iphp->ip_src.s_addr != ipcpOpt.l_addr.s_addr) return;
    /*
       called from RebuildQueue: need to rebuild
       check connectionless or not!!
    */
    if (iphp->ip_p == IPPROTO_TCP
	|| iphp->ip_p == IPPROTO_UDP) {
	struct ipfilter_s *ifp;
	tcp_hdr *tcphp;

	tcphp = (tcp_hdr *)(bp + (iphp->ip_hl << 2));
	ifp = IpXFilterPairAdd(NULL, (ipAtFlag == IPAT_ALL)
			       ? iphp->ip_src.s_addr: 0,
			       iphp->ip_dst.s_addr,
			       ipcplReg.addr.s_addr,
			       tcphp->th_sport, tcphp->th_dport, 0,
			       iphp->ip_p, IPFF_AUTO);
	IpTranslateAddress(ifp, F_TYPE_OUTPUT, iphp, tcphp, &len);
	return;
    }
    iphp->ip_src.s_addr = ipcplReg.addr.s_addr;
    ip_check_sum(iphp);
}

/*
 * IpTrigger
 *	an outgoing IP packet trigger off a connection?
 *	buf: pointer to IP packet
 *	len: IP packet len
 *	RETURNS:
 *		TRUE  <- trigger off
 *		FALSE <- ignore
 */

bool_t
IpTrigger(u_char *buf, int n)
{
    bool_t ret=TRUE;

    if (flHead[F_TYPE_TRIGGER]
	&& IpFilterCheck(F_TYPE_TRIGGER, buf, &n, NULL, NULL))
	    ret = FALSE;
    else {
	if (ISLOG(LOG_PRIVATE)) {
	    ipv4_hdr *iphp = (ipv4_hdr *)buf;
	    tcp_hdr *tcphp=NULL;

	    if (iphp->ip_p == IPPROTO_TCP || iphp->ip_p == IPPROTO_UDP)
		tcphp = (tcp_hdr *)(buf + (iphp->ip_hl << 2));
	    Logf(LOG_SYS, "TRIGGER: IP(%d) %s", iphp->ip_p,
		 inet_ntoa(iphp->ip_src));
	    if (tcphp) Logf(LOG_SYS, ":%d", ntohs(tcphp->th_sport));
	    Logf(LOG_SYS, " -> %s", inet_ntoa(iphp->ip_dst));
	    if (tcphp) Logf(LOG_SYS, ":%d", ntohs(tcphp->th_dport));
	}
	Logf(LOG_SYS, "\n");
    }
    return(ret);
}

time_t
IpEnqueue(u_char *buf, int n)
{
    priority_t pri=0;
    time_t idle=pppOpt.i_to;

#ifdef DEBUG_IP
    ExplainIp("O", FALSE, (ipv4_hdr *)buf, n);
#endif
    if (flHead[F_TYPE_OUTPUT] &&
	IpFilterCheck(F_TYPE_OUTPUT, buf, &n, &idle, &pri)) return(0);

#ifdef DEBUG_XIP
    if (flHead[F_TYPE_OUTPUT]) ExplainIp("O", TRUE, (ipv4_hdr *)buf, n);
#endif
    if (slInitialized) {
	u_int16_t nbo_proto=NBO_PROTO_IP;
	struct mbuf mb;
	u_char type;

	mb.m_off = buf;
	mb.m_len = n;
	type = sl_compress_tcp(&mb, (ipv4_hdr *)buf, &slCs,
			       ipcprReg.cproto.opt.vj.cmp_slot_id);
	switch (type) {
	case TYPE_IP:
	    nbo_proto = NBO_PROTO_IP;
	    break;
	case TYPE_UNCOMPRESSED_TCP:
	    nbo_proto = NBO_PROTO_VJUC;
	    break;
	case TYPE_COMPRESSED_TCP:
	    nbo_proto = NBO_PROTO_VJC;
	    break;
	}
	if (ISLOG(LOG_DUMP))
	    Logf(LOG_DUMP, "compress %s: %d->%d\n",
		 (nbo_proto == NBO_PROTO_VJUC) ? "VJUC":
		 (nbo_proto == NBO_PROTO_VJC) ? "VJC": "IP",
		 n, mb.m_len);
	FrameEnqueue(mb.m_off, mb.m_len, nbo_proto, pri);
    } else FrameEnqueue(buf, n, NBO_PROTO_IP, pri);
    return(idle);
}

time_t
IpInput(u_char *buf, int len, long nbo_proto)
{
    time_t idle=pppOpt.i_to;

#ifdef DEBUG_IP
    ExplainIp("I", FALSE, (ipv4_hdr *)buf, len);
#endif
    if (flHead[F_TYPE_INPUT] &&
	IpFilterCheck(F_TYPE_INPUT, buf, &len, &idle, NULL)) return(0);
#ifdef DEBUG_XIP
    if (flHead[F_TYPE_INPUT]) ExplainIp("I", TRUE, (ipv4_hdr *)buf, len);
#endif
    IfWrite(buf, len, ETHERTYPE_IP);
    return(idle);
}

time_t
VjInput(u_char *buf, int len, long nbo_proto)
{
    int uclen;

    /*
       We expect that the size of buf (== frameReg.rbuf[MAX_FRAMESIZ])
       is enough to store uncompressed packet.
    */
    uclen = sl_uncompress_tcp(&buf, len, (nbo_proto == NBO_PROTO_VJUC) ?
			      TYPE_UNCOMPRESSED_TCP:
			      TYPE_COMPRESSED_TCP, &slCs);
    if (ISLOG(LOG_DUMP)) Logf(LOG_DUMP, "uncompress %s: %d->%d\n",
			      (nbo_proto == NBO_PROTO_VJUC) ? "VJUC":
			      (nbo_proto == NBO_PROTO_VJC) ? "VJC": "IP",
			      len, uclen);
    return(IpInput(buf, uclen, 0));
}
