/*
 * This program is in the public domain and may be used freely by anyone
 * who wants to.
 *
 * Please send bug fixes/bug reports to: Peter Eriksson <pen@lysator.liu.se>
 *
 * $Id: openbsd.c,v 1.3 1999/12/12 20:08:57 odin Exp $
 */

#include <config.h>
#include <oidentd.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/socketvar.h>
#include <sys/user.h>
#define _KERNEL
#include <sys/file.h>
#undef _KERNEL
#include <sys/sysctl.h>

#include <limits.h>
#include <nlist.h>
#include <kvm.h>

#include <net/if.h>
#include <net/route.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/in_pcb.h>
#include <netinet/tcp.h>
#include <netinet/ip_var.h>
#include <netinet/tcp_timer.h>
#include <netinet/tcp_var.h>

extern u_int flags;

struct nlist nl[] = {
#define N_TCBTABLE 0
	{"_tcbtable"},
	{""}
};

static kvm_t *kd;
static struct inpcbtable tcbtable;

int k_open(void) {
	char errbuf[_POSIX2_LINE_MAX];

	/*
	** Open the kernel memory device
	*/
	if (!(kd = kvm_openfiles(NULL, "/dev/mem", NULL, O_RDONLY, errbuf))) {
		if (flags & DEBUG)
			syslog(DPRI, "kvm_openfiles: %m");
		return (-1);
	}

	/*
	** Extract offsets to the needed variables in the kernel
	*/
	if (kvm_nlist(kd, nl) < 0) {
		if (flags & DEBUG)
			syslog(DPRI, "kvm_nlist: %m");
		return (-1);
	}

	return 0;
}

/*
 * Get a piece of kernel memory with error handling.
 * Returns 1 if call succeeded, else 0 (zero).
 */
static int getbuf(long addr, char *buf, int len, char *what) {
	if (kvm_read(kd, addr, buf, len) < 0) {
		if (flags & DEBUG)
			syslog(DPRI, "getbuf: kvm_read(%08lx, %d) - %s : %m",
				addr, len, what);

		return (0);
	}

	return (1);
}

/*
 * Traverse the inpcb list until a match is found.
 * Returns NULL if no match.
 */
static struct socket *getlist(struct inpcbtable *tcbtablep,
	struct inpcbtable *ktcbtablep, const struct in_addr *faddr, int fport,
	const struct in_addr *laddr, int lport) {

	struct inpcb *kpcbp, pcb;

	if (!tcbtablep)
		return (NULL);

	for (kpcbp = tcbtablep->inpt_queue.cqh_first;
	    kpcbp != (struct inpcb *) ktcbtablep;
	    kpcbp = pcb.inp_queue.cqe_next) {
		if (!getbuf((long) kpcbp, (char *) &pcb, sizeof(struct inpcb), "tcb"))
			break;
		if (pcb.inp_faddr.s_addr == faddr->s_addr &&
		    pcb.inp_laddr.s_addr == laddr->s_addr &&
		    pcb.inp_fport == fport && pcb.inp_lport == lport)
			return (pcb.inp_socket);
	}
	return (NULL);
}

/*
 * Return the user number for the connection owner
 */

int get_user(int lport, int fport, const struct in_addr *laddr,
			 const struct in_addr *faddr) {
	struct socket *sockp, sock;

	if (!getbuf(nl[N_TCBTABLE].n_value, (char *) &tcbtable, sizeof(tcbtable), "tcbtable"))
		return (-1);

	sockp = getlist(&tcbtable, (struct inpcbtable *) nl[N_TCBTABLE].n_value,
		faddr, fport, laddr, lport);

	if (!sockp)
		return (-1);
	if (!getbuf((long) sockp, (char *) &sock, sizeof sock, "socket"))
		return (-1);
	if (!(sock.so_state & SS_CONNECTOUT))
		return (-1);

	return (sock.so_ruid);
}
