/*  Protocol compatible masqdialer server written in C
    Copyright (C) 1998 Charles P. Wright 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifndef MSERVER_H
#define MSERVER_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <pwd.h>
#include <sys/types.h>
#include <time.h>

typedef int bool;
#define false 0
#define true 1

#define MAXCOMMAND 100
#define MAXDATA 1024
#define MAXINTERFACES 32 /* Maximum number of network interfaces mserver will listen on */

/* structure representing a single parameter entry, with a key and its value */
struct param_t
{
    char* key;
    char* val;
};

/* structure containing a database of all parameter entries */
struct param_dbase_t
{
    int num_param;
    int space_for_param;
    struct param_t *param;
};

#define MAXCLIENTS 25
#define MAXCLIENTNAMES 2048

struct shared_mem_t {
	int clients;	
	char clientnames[MAXCLIENTNAMES];
	pid_t now_dialing;
	char cname[1024];
	int cspeed;
	time_t ctime;
	int lock_dial;
	int lock_kill;
	bool pending_kill;
	bool captured;
	char capname[1024];
	pid_t pppdpid;
	unsigned long start_in_bytes, start_in_packets;
	unsigned long start_out_bytes, start_out_packets;
};

struct msrv_client {
	pid_t pid;
	time_t start;
	char *name;
};

/* Configuration file routines */
bool config_init (const char *name);
void config_close_save (void);
void config_clear (void);
void config_refresh (int);

/* General Configuration routines */
void config_getvalue (const char *key, char *value, int maxchar);
void config_setvalue (const char *key, const char *value);
bool find_cname(const char *);

/* Special variable type config routines */
void config_getvalue_default (const char *key, char *value, char *defval, int maxchar);
void config_getvalue_cname (const char *key, char *value, int maxchar);
void config_getvalue_cname_specified (const char *key, char *value, const char *cname, int maxchar);
int config_getvalue_int (const char *key, int);
bool config_getvalue_bool (const char *key, bool defval);

/* Character processing functions */
int gettoken(char *);
void ungetch(int);
int getch(void);

/* Parameter Database functions */
void init_param_dbase(struct param_dbase_t* dbase);
void param_dbase_setval(struct param_dbase_t* dbase, const char* key, const char *val);
char* param_dbase_getval(const struct param_dbase_t* dbase, const char* key);
void clear_param_dbase(struct param_dbase_t* dbase);

/* General server functions */
void mserver_serve(void);
void mserver_cleanup(void);
int getstat(void);

/* Functions that Actually do stuff for the client */
void mserver_list(void);
void mserver_kill(void);
void mserver_kill_silent(void);
void mserver_dial(const char *, const char *);
void mserver_cinfo(const char *);
void mserver_time(void);
void mserver_ctime(void);
void mserver_stat(void);
void mserver_info_var (const char *);
void mserver_info_file (const char *);
void mserver_netload(void);
void mserver_lock(const char *);
void mserver_unlock(const char *);
void mserver_linfo(void);
void mserver_capture (const char *);
void mserver_release (void);

void auto_dial(const char *);

/* Functions to turn the server off politely */
void mserver_halt(int);
void mserver_halt_all(void);

/* Child functions */
void child_handler(int);
void child_write_count(int);
void add_client(pid_t, int);
void client_connect_script(pid_t, int);
int child_count(void);
bool child_alive(pid_t);

/* Connection file functions */
bool cfile_write(const char *);
void cfile_unlink();
void cfile_getcname(char *);
int cfile_getcspeed();
int cfile_gettime();

/* Parsing functions */
bool parse(const char *);
bool parse_init(const char *);
int parse_listenon(char *, char [MAXINTERFACES][], char [MAXINTERFACES][]);

/* Utility functions */
int util_system (const char *);
int util_system_wait (const char *);
int util_unlink (const char *);
int util_fflush (FILE *);
int util_shutdown(int, int);
unsigned int util_sleep (int);
time_t util_time (time_t *);
void *util_malloc (size_t);
int util_close(int);
bool util_validip(const char *listenon);
int util_select(int, fd_set *);

/* Authorization functions */
#define AUTH_NONE 	0
#define AUTH_DISCONNECT 1
#define AUTH_CONNECT 	2
#define AUTH_CONFIG 	4

bool auth_check(const char *, int);
bool auth_checkip(const char *, int);
bool auth_checkip_backend(const char *, const char *);
bool auth_checkuser(const char *, int);
void auth_downscript(void);
void auth_setpass (const char*);
void auth_setuser (const char*);
void auth_setverified (bool);
bool auth_getverified (void);
struct passwd *auth_getpwent(const char*);

/* TCP/IP and Socket functions */
int sock_host(int fd, char addr_string[], int limit);
void netload_init(void);

/* IPC functions */
void shmem_free(void);
void shmem_init(const char *);
void shmem_set_clients(int);
void shmem_set_dialing(pid_t);
void shmem_set_who(const char *);
void shmem_set_cspeed(int);
void shmem_set_ctime(time_t);
void shmem_set_cname(const char *);
int shmem_get_clients(void);
pid_t shmem_get_dialing(void);
void shmem_get_who(char *);
int shmem_get_cspeed(void);
time_t shmem_get_ctime(void);
void shmem_get_cname(char *);
void shmem_set_dial_lock(int);
int shmem_get_dial_lock(void);
void shmem_set_kill_lock(int);
int shmem_get_kill_lock(void);
void shmem_set_pending_kill(bool);
bool shmem_get_pending_kill(void);
void shmem_set_start_netload(unsigned long, unsigned long, unsigned long, unsigned long);
void shmem_get_start_netload(unsigned long *, unsigned long *, unsigned long *, unsigned long *);
void shmem_set_pppdpid(pid_t);
pid_t shmem_get_pppdpid(void);
void shmem_set_captured(bool);
bool shmem_get_captured(void);
void shmem_set_capname(const char *);
void shmem_get_capname(char *);

const char *get_exe_name(const char *);

#define VERSION_STR "C Masqdialer Server %s (Charles P. Wright)"
#define AUTHOR "Charles P. Wright"

#endif
