/*  General Configuration module written in C 
    Copyright (C) 1998  Charles P. Wright

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    You may link this program with the Qt Free Edition.
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <syslog.h>
#include <ctype.h>

#include "mserver.h"

#ifdef USE_HASH
extern struct htab *hashtab[HASHELEMENTS];
#else
struct param_dbase_t dbase;
#endif

FILE *cfile;
char filename[1024];
char **saveval;
int savenum;

bool config_init (const char *name)
{
	char s[1024];
	char *retval;
	char key[1024];
	char data[1024];
	char temp[1024];
	int i, n;
	size_t len;
	
	init_param_dbase(&dbase);

	strncpy (filename, name, 1024);
	cfile = fopen (filename, "r");

	if (cfile != NULL)
	{

		while ((retval = fgets(s, 1023, cfile)) != NULL)
		{
			char *comment = strchr(s, '#');
			if (comment)
				*comment = '\0';

			len = strlen(s);

			if (len != 0)
			{
	
				for (i = 0, n = 0; i < len && s[i] != '='; i++)
				{
					if (!isspace(s[i]))
					{
						key[n++] = s[i];
					}
				}
				key[n] = '\0';

				while (isspace(s[++i]));

				if (s[i] == '"')
				{
					for (i++, n = 0; i < len && s[i] != '"'; i++)
					{
						data[n++] = s[i];
					}
					data[n] = '\0';
				}
				else
				{
					for (n = 0; i < len; i++)
					{
						if (!isspace(s[i]))
						{
							data[n++] = s[i];
						}
					}
					data[n] = '\0';
				}

				config_setvalue(key, data);
			}
		}
		fclose (cfile);

		snprintf(temp, 1024, VERSION_STR, VERSION);
		config_setvalue ("version", temp);
		config_setvalue ("author", AUTHOR);

		/* If a client exits while there is a Connection Pending message
  		   no one will be allowed to do anything to the conneciton, this
                   is just a hack to make it so anyone can maniuplate the
                   connection instead of noone. */
		config_setvalue ("Connection Pending_ipallow", "*.*.*.*");
		config_setvalue ("Connection Pending_userallow", "*");

		return true;
	}
	else
	{
		return false;
	}

}

void config_refresh (int sig)
{
	syslog (LOG_NOTICE, "Recieved signal: %d", sig);

	config_clear();
	config_init(filename);

	syslog (LOG_INFO, "Reloaded configuration file: %s", filename);
	syslog (LOG_DEBUG, "Already existing children or siblings will not reread configuration file, unless you specifically send them the HUP signal");
	syslog (LOG_DEBUG, "Note not all values with take effect until you restart the masqdialer server. (port, slimit, ...)");
}

/* This no longer works because I never reimplemented it with the param_dbase
   code, it should close the configuration file and save all of the current
   values in the configuration table to it.  This was much more useful for the
   qt-mclient which uses this as like the Windows '95/NT registry */
void config_close_save (void)
{

	cfile = fopen (filename, "w+");
	fclose(cfile);
}

void config_clear (void)
{
	clear_param_dbase(&dbase);	
}

void config_getvalue(const char *key, char *value, int maxchar)
{
	char *temp;

	temp = param_dbase_getval(&dbase, key);

	if (temp == NULL)
	{
		value[0] = '\0';	
	}
	else
	{
		strncpy(value, temp, maxchar);
		value[maxchar - 1] = 0;
	}
}

int config_getvalue_int (const char *key, int defval)
{
	int retint;
	char strval[1024];

	config_getvalue(key, strval, 1024);
	
	if (!strncmp(strval, "", 1024))
	{
		retint = defval;
	}
	else
	{
		retint = atoi(strval);	
	}
	
	return retint;
}

void config_getvalue_default (const char *key, char *value, char *defval, int maxchar)
{
	char *retstr;

	retstr = (char *) util_malloc(sizeof(char) * maxchar);	
	
	if (retstr == NULL)
	{
		value[0] = '\0';
		return;
	}
	
	config_getvalue(key, retstr, maxchar);

	if (!strncmp(retstr, "", maxchar))
	{
		strncpy(value, defval, maxchar);
	}
	else
	{
		strncpy(value, retstr , maxchar);
	}
	free(retstr);
}

bool config_getvalue_bool (const char *key, bool defval)
{
	char value[1024];
	int maxchar = 1024;

	char *retstr;
	
	retstr = (char *) param_dbase_getval(&dbase, key);

	if (retstr != NULL)
	{
		strncpy(value, retstr, maxchar);
	}
	else
	{
		value[0] = '\0';
	}

	if (!strncmp(value, "true", maxchar))
	{
		return true;
	}
	else if (!strncmp(value, "false", maxchar))
	{
		return false;
	}
	else if (!strncmp(value, "yes", maxchar))
	{
		return true;
	}
	else if (!strncmp(value, "no", maxchar))
	{
		return false;
	}

	return defval;
}
	
void config_setvalue (const char *key, const char *value)
{
	param_dbase_setval(&dbase, key, value);
}

#ifndef UTIL
void config_getvalue_cname(const char *key, char *value, int maxchar)
{
	char cname[1024];

	cfile_getcname(cname);

	config_getvalue_cname_specified(key, value, cname, maxchar);
}
#endif

void config_getvalue_cname_specified(const char *key, char *value, const char *cname, int maxchar)
{
	char temp[1024];

	snprintf(temp, 1024, "%s_%s", cname, key);	
	
	config_getvalue(temp, value, maxchar);

	if (!strncmp(value, "", maxchar))
	{
		config_getvalue(key, value, maxchar);
	}
}
