;;; liece-ctcp.el --- CTCP handlers and commands.
;; Copyright (C) 1998, 1999 Daiki Ueno

;; Author: Daiki Ueno <ueno@ueda.info.waseda.ac.jp>
;; Created: 1998-09-28
;; Revised: 1998-11-25
;; Keywords: IRC, liece, CTCP

;; This file is part of Liece.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Code:

(eval-when-compile 
  (require 'liece-inlines)
  (require 'liece-commands)
  (require 'liece-misc)
  (require 'liece-intl)
  (require 'broken) 
  (require 'pccl))

(require 'llunf)

(eval-and-compile
  (if-broken ccl-usable
      (require 'liece-q-el)
    (require 'liece-q-ccl)))

(require 'liece-x-face)

(autoload 'liece-ctcp-dcc-message "liece-dcc")

(eval-and-compile
  (defconst liece-ctcp-supported-syms
    '(VERSION USERINFO CLIENTINFO PING TIME FILE X-FACE COMMENT HELP)))

(eval-and-compile
  (dolist (sym liece-ctcp-supported-syms)
    (let ((sym (downcase (symbol-name sym))))
      (fset (intern (concat "liece-menu-callback-ctcp-" sym))
	    `(lambda () 
	       (interactive) 
	       (dolist (nick liece-nick-region-nicks)
		 (funcall (symbol-function 
			   (intern (concat "liece-command-client-" ,sym)))
			  nick))))
      )))
	     
(defconst liece-ctcp-supported-messages
  (eval-when-compile
    (mapconcat #'symbol-name liece-ctcp-supported-syms " ")
    ))

(defvar liece-client-message 
  (concat liece-client-prefix "%s(%s) = %s")
  "Message in which info of other clients is displayed.")

(defvar liece-ctcp-buffer (append liece-D-buffer liece-O-buffer))

(defvar liece-ctcp-ping-time '(0 0 0))

(defvar liece-query-client-lastcommand nil
  "Place to keep last entered command")

(defvar liece-query-client-alist
  '(("VERSION") ("PING") ("CLIENTINFO") ("ACTION") ("USERINFO") 
    ("TIME") ("AWAY") ("ERRMSG") ("PLAY") ("DCC") 
    ("STREAM") ("X-FACE") ("HELP") ("FINGER")))

(defvar liece-query-client-nick nil
  "Place to keep last queried nick")

(defconst liece-query-client-insert-to-generic "")

(defconst liece-query-client-version
  (concat "\001VERSION\001" liece-query-client-insert-to-generic))

(defconst liece-query-client-userinfo
  (concat "\001USERINFO\001" liece-query-client-insert-to-generic))

(defconst liece-query-client-help
  (concat "\001HELP\001" liece-query-client-insert-to-generic))

(defconst liece-query-client-clientinfo
  (concat "\001CLIENTINFO\001" liece-query-client-insert-to-generic))

(defconst liece-query-client-ping
  (concat "\001PING\001" liece-query-client-insert-to-generic))

(defconst liece-query-client-time
  (concat "\001TIME\001" liece-query-client-insert-to-generic))

(defconst liece-query-client-x-face
  (concat "\001X-FACE\001" liece-query-client-insert-to-generic))

(defconst liece-client-error-message "Unrecognized command: '%s'"
  "Error message given to anyone asking wrong CLIENT data.")


(defun liece-query-client-nick-maybe-change (prefix rest) 
  (and 
   (stringp liece-query-client-nick)
   (string-equal prefix liece-query-client-nick)
   (setq liece-query-client-nick rest))
  nil)

(defun liece-query-client-nick-maybe-reset (prefix rest) 
  (and 
   (stringp liece-query-client-nick)
   (string-equal prefix liece-query-client-nick)
   (setq liece-query-client-nick nil)))

(add-hook 'liece-nick-hook 'liece-query-client-nick-maybe-change t)
(add-hook 'liece-quit-hook 'liece-query-client-nick-maybe-reset)
  
(defcustom liece-ctcp-file-save-directory "~/.liece"
  "Directory to save received files"
  :type 'directory
  :group 'liece-ctcp)
  
(defmacro liece-client-message (&rest msg)
  `(message "CLIENT %s" (format ,@msg)))

(defmacro liece-client-file-message (&rest msg)
  `(message "CLIENT FILE %s" (format ,@msg)))


(llunf-define-backend "ctcp")

(let ((messages `(,@liece-ctcp-supported-syms ACTION DCC ERRMSG)) name)
  (dolist (message messages)
    (setq name (downcase (symbol-name message)))
    (llunf-define-function name '(from chnl rest-of-line "ctcp")
			   (intern (format "liece-ctcp-%s-message"
					   name)))))

(defun* liece-ctcp-message (from chnl rest)
  (let (hook after-hook left now right message rest-of-line)
    (or (string-match "^\\([^\001]*\\)\001\\([^\001]*\\)\001" rest)
	(return-from liece-ctcp-message))
    (setq left (match-string 1 rest)
	  now (match-string 2 rest)
	  right (substring rest (match-end 0))
	  rest (concat left right))
    (cond 
     ((string-match "^\\([^ ]*\\) +:?" now)
      (setq message (downcase (match-string 1 now))
	    rest-of-line (substring now (match-end 0))))
     ((string-match "^\\([^ ]*\\)" now)
      (setq message (downcase (match-string 1 now))
	    rest-of-line nil))
     (t
      (setq message "errmsg"
	    rest-of-line (_ "Couldn't figure out what was said."))))
    ;; If we have a hook, and it returns T, do nothing more
    (setq hook (intern-soft (concat "liece-ctcp-" message "-hook"))
	  after-hook (intern-soft (concat "liece-after-ctcp-" 
					  message "-hook")))
	  
    (if (condition-case nil
	    (run-hook-with-args-until-success hook from chnl rest-of-line)
	  (error nil))
	(return-from liece-ctcp-message rest))
    
    ;; else call the handler
    (let ((func (llunf-find-function message '(from chnl rest-of-line) 
				     "ctcp")))
      (if func
	  (funcall func from chnl rest-of-line)
	(liece-ctcp-messages message from chnl rest-of-line)))
    
    (ignore-errors (run-hook-with-args after-hook from chnl rest-of-line))

    rest))

(defun liece-ctcp-messages (message from chnl rest-of-line)
  (liece-send "NOTICE %s :\001ERRMSG %s :%s\001"
	       from (upcase message) 
	       (format liece-client-error-message
		       (upcase message)))
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert (upcase message) from chnl rest-of-line))

(defun liece-ctcp-action-message (from chnl rest)
  "CTCP ACTION handler"
  (let ((liece-message-target (liece-channel-virtual chnl))
	(liece-message-speaker from)
	(liece-message-type 'action))
    (liece-display-message rest)))

(defun liece-client-insert (msg from &optional chnl rest)
  (if (or (null chnl) (liece-nick-equal chnl liece-real-nickname))
      (liece-client-message "%s query from %s." msg from)
    (liece-client-message "%s query from %s (%s)." msg from chnl)
    (liece-insert-client (liece-pick-buffer chnl)
			  (concat msg " query from " from 
				  (if rest (concat ":" rest) "")
				  "\n"))))

(defun liece-ctcp-version-message (from chnl rest)
  "CTCP VERSION handler"
  (liece-send 
   "NOTICE %s :\001VERSION %s :\001" 
   from liece-user-agent-value)
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert "VERSION" from chnl rest))

(defun liece-ctcp-userinfo-message (from chnl rest)
  "CTCP USERINFO handler"
  (liece-send 
   "NOTICE %s :\001USERINFO %s\001"
   from liece-client-userinfo)
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert "USERINFO" from chnl))

(defun liece-ctcp-clientinfo-message (from chnl rest)
  "CTCP CLIENTINFO handler"
  (liece-send 
   "NOTICE %s :\001CLIENTINFO %s\001" from liece-ctcp-supported-messages)
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert "CLIENTINFO" from chnl))

(defun liece-ctcp-help-message (from chnl rest)
  "CTCP HELP handler"
  (liece-send 
   "NOTICE %s :\001HELP VERSION gives version of this client\001" 
   from)
  (liece-send
   "NOTICE %s :\001HELP USERINFO gives user supplied info (if any)\001" 
   from)
  (liece-send 
   "NOTICE %s :\001HELP CLIENTINFO gives commands this client knows\001" 
   from)
  (liece-send 
   "NOTICE %s :\001HELP X-FACE gives you user supplied X-Face (if exists)\001" 
   from)
  (liece-send 
   "NOTICE %s :\001HELP HELP gives this help message\001" 
   from)
  (liece-send 
   "NOTICE %s :\001HELP ERRMSG tells you your command was not valid\001" 
   from)
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert "HELP" from chnl))

(defun liece-ctcp-comment-message (from chnl rest)  
  "CTCP COMMENT handler"
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert "COMMENT" from chnl))

(defun liece-ctcp-xyzzy-message (from chnl rest)
  "CTCP XYZZY handler"
  (liece-send "NOTICE %s :Nothing happens.(xyzzy inactive)" from)
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert "XYZZY" from chnl))

(defun liece-ctcp-ping-message (from chnl rest)
  "CTCP PING handler"
  (unless rest
    (setq rest ""))
  (liece-send "NOTICE %s :\001PING %s\001" from rest)
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert "PING" from chnl))

(defun liece-ctcp-time-message (from chnl rest)
  "CTCP TIME handler"
  (liece-send "NOTICE %s :\001TIME %s\001" 
	       from (funcall liece-format-time-function (current-time)))
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert "TIME" from chnl))

(defun liece-ctcp-x-face-message (from chnl rest)
  "CTCP X-FACE handler"
  (liece-send 
   "NOTICE %s :\001X-FACE %s\001"
   from liece-client-x-face)
  (setq chnl (liece-channel-virtual chnl))
  (liece-client-insert "X-FACE" from chnl))


(llunf-define-backend "ctcp-notice")

(let ((messages `(,@liece-ctcp-supported-syms DCC ERRMSG)) name)
  (dolist (message messages)
    (setq name (downcase (symbol-name message)))
    (llunf-define-function name '(prefix rest-of-line "ctcp-notice")
			   (intern (format "liece-client-%s-notice"
					   name)))))

(defun* liece-ctcp-notice (prefix rest) 
  (let (hook after-hook left now right message rest-of-line)
    (or (string-match "^\\([^\001]*\\)\001\\([^\001]*\\)\001" rest)
	(return-from liece-ctcp-notice))
    (setq left (match-string 1 rest)
	  now (match-string 2 rest)
	  right (substring rest (match-end 0))
	  rest (concat left right))
    (cond 
     ((string-match "^\\([^ ]*\\) +:?" now)
      (setq message (downcase (match-string 1 now))
	    rest-of-line (substring now (match-end 0))))
     ((string-match "^\\([^ ]*\\)" now)
      (setq message (downcase (match-string 1 now))
	    rest-of-line nil))
     (t
      (setq message "errmsg"
	    rest-of-line (_ "Couldn't figure out what was said."))))

    ;; If we have a hook, and it returns T, do nothing more
    (setq hook (intern-soft (concat "liece-ctcp-" message "-notice-hook"))
	  after-hook (intern-soft (concat "liece-after-ctcp-" 
					  message "-notice-hook")))
    
    (if (condition-case nil
	    (run-hook-with-args-until-success hook prefix rest-of-line)
	  (error nil))
	(return-from liece-ctcp-notice rest))
    
    ;; else call the handler
    (let ((func (llunf-find-function message '(prefix rest-of-line) 
				     "ctcp-notice")))
      (if func
	  (funcall func prefix rest-of-line) 
	(liece-ctcp-notices message prefix rest-of-line)))
    
    (ignore-errors (run-hook-with-args after-hook prefix rest-of-line))
    
    rest))

(defun liece-ctcp-notices (message prefix rest-of-line)
  (liece-message (_ "Unknown ctcp notice \":%s %s %s\"")
		  prefix (upcase message) rest-of-line))

(llunf-define-backend "ctcp-file")

(defun* liece-ctcp-file-notice (prefix rest)
  (if (not liece-file-accept)
      (liece-client-file-message 
       (_ "%s sending, set liece-file-accept and ask to resend")
       prefix)
    (let (hook after-hook message file-name rest-of-line fun)
      (string-match "^\\([^ ]*\\) \\([^ ]*\\) :\\(.*\\)" rest)
      (setq message (downcase (match-string 1 rest))
	    file-name (match-string 2 rest)
	    rest-of-line (match-string 3 rest))
      ;; If we have a hook, and it returns T, do nothing more
      (setq hook (intern-soft (concat "liece-file-" message "-hook"))
	    after-hook (intern-soft (concat "liece-after-file-" 
					    message "-hook")))
      
      (if (condition-case nil
	      (run-hook-with-args-until-success hook prefix file-name)
	    (error nil))
	  (return-from liece-ctcp-file-notice))
      ;; else call the handler
      (let ((func (llunf-find-function message 
				       '(prefix file-name rest-of-line)
				       'ctcp-file)))
	(if func
	    (funcall func prefix file-name rest-of-line)
	  (liece-file-notices message prefix file-name rest-of-line)))
      
      (ignore-errors (run-hook-with-args after-hook prefix file-name)))
    ))

(defun liece-file-notices (message prefix file-name rest-of-line)
  (liece-message (_ "Unknown FILE message \":%s %s %s %s\"")
		  prefix (upcase message) file-name rest-of-line))

(defun liece-file-start (prefix name data)
  "CTCP FILE start handler"
  (save-excursion
    (set-buffer (liece-get-buffer-create (format "*IRC R_FILE_%s*" name)))
    (erase-buffer)
    (insert data)))

(defun liece-file-cont (prefix name data)
  "CTCP FILE cont handler"
  (save-excursion
    (set-buffer (liece-get-buffer-create (format "*IRC R_FILE_%s*" name)))
    (goto-char (point-max))
    (insert data)))

(defun liece-file-end (prefix name data)
  "CTCP FILE cont handler"
  (save-excursion
    (set-buffer 
     (liece-get-buffer-create (format "*IRC R_FILE_%s*" name)))
    (goto-char (point-max))
    (insert data)
    (let (str)
      (setq str (buffer-string))
      (erase-buffer)
      (insert (liece-quote-decode-string str)))
    (goto-char (point-min))
    (if (and liece-file-confirm-save
	     (not (y-or-n-p (_ "Save file?"))))
	nil
      (unless (file-exists-p liece-ctcp-file-save-directory)
	(condition-case ()
	    (make-directory 
	     (expand-file-name liece-ctcp-file-save-directory))
	  (error nil)))
      (if (file-directory-p liece-ctcp-file-save-directory)
	  (write-region 
	   (point-min) (point-max) 
	   (expand-file-name 
	    (file-name-nondirectory
	     (format "%s(%s)" name prefix))
	    liece-ctcp-file-save-directory)))
      (kill-buffer (liece-get-buffer-create 
		    (format "*IRC R_FILE_%s*" name))))))

(defun liece-client-version-insert 
  (buffer prefix client-name &optional client-version environment)
  (if (not (listp buffer)) (setq buffer (list buffer)))
  (liece-insert buffer (concat (format liece-client-message 
					"VERSION" prefix "")
				client-name "\n"))
  (when client-version
    (liece-insert buffer (concat (format liece-client-message 
					"VERSION" prefix "")
				  "\t" client-version
				  (if environment
				      (concat " " environment)
				    "")
				  "\n"))))

(defun liece-client-version-notice (prefix rest)  
  "CTCP VERSION reply handler"
  (if (not rest)
      (liece-message (_ "Empty CLIENT version notice from \"%s\".") prefix)
    (let ((buffer liece-ctcp-buffer))
      (cond
       ((string-match "^\\([^:]*\\):\\([^:]+\\):?\\([^:]*\\)" rest)
	(liece-client-version-insert 
	 buffer prefix
	 (match-string 1 rest) (match-string 2 rest) (match-string 3 rest)))
       ((string-match "^\\([^:]*\\):\\(.*\\)" rest)
	(liece-client-version-insert
	 buffer prefix (match-string 1 rest)))
       (t 
	(liece-client-version-insert
	 buffer prefix rest))))))

(defun liece-client-clientinfo-notice (prefix rest)  
  "CTCP CLIENTINFO reply handler"
  (liece-insert liece-ctcp-buffer 
		 (format (concat liece-client-message "\n")
			 "CLIENTINFO" prefix rest)))

(defun liece-client-userinfo-notice (prefix rest)  
  "CTCP USERINFO reply handler"
  (liece-insert liece-ctcp-buffer 
		 (format (concat liece-client-message "\n")
			 "USERINFO" prefix rest)))

(defun liece-client-help-notice (prefix rest)  
  "CTCP HELP reply handler"
  (liece-insert liece-ctcp-buffer 
		 (format (concat liece-client-message "\n")
			 "HELP" prefix rest)))

(defun liece-client-x-face-notice (prefix rest)
  "CTCP X-FACE reply handler"
  (let ((buffer liece-ctcp-buffer))
    (liece-insert buffer
		   (format liece-client-message 
			   "X-FACE" prefix ""))
    (if (and liece-use-x-face
	     (string-match "[^ \t]" rest))
	(liece-x-face-insert 
	 buffer (replace-in-string rest "[ \t\r\n]+" "") prefix)
      (liece-insert buffer rest))
    (let (liece-display-time)
      (liece-insert buffer "\n"))))

(defun liece-client-errmsg-notice (prefix rest)  
  "CTCP ERRMSG reply handler"
  (liece-insert liece-ctcp-buffer 
		 (format (concat liece-client-message "\n")
			 "ERRMSG" prefix rest)))

(defun liece-client-comment-notice (from rest)  
  "CTCP COMMENT reply handler"
  (liece-insert liece-ctcp-buffer 
		 (format (concat liece-client-message "\n")
			 "COMMENT" from rest))
  (liece-client-message (_ "COMMENT query from %s.") from))

(defmacro liece-ctcp-prepare-ping-seconds (timenow)
  `(format (_ "%d sec")
	   (+ (* 65536 (- (car ,timenow) (car liece-ctcp-ping-time)))
	      (- (cadr ,timenow) (cadr liece-ctcp-ping-time)))))

(defun liece-client-ping-notice (from rest)  
  "CTCP PING reply handler"
  (let ((timenow (current-time)))
    (liece-insert liece-ctcp-buffer 
		   (format (concat liece-client-message "\n")
			   "PING" from 
			   (liece-ctcp-prepare-ping-seconds timenow)))))

(defun liece-client-time-notice (from rest)  
  "CTCP TIME reply handler"
  (liece-insert liece-ctcp-buffer 
		 (format (concat liece-client-message "\n")
			 "TIME" from rest)))


(defmacro liece-complete-client ()
  '(let ((completion-ignore-case t) (nick liece-query-client-nick))
     (liece-minibuffer-completing-default-read 
      (_ "Whose client: ") liece-nick-alist nil nil 
      (if nick (liece-channel-virtual nick)))))

(defun liece-minibuffer-complete-client-query ()
  (let* ((candidate (liece-minibuffer-prepare-candidate))
	 (completion (try-completion candidate liece-query-client-alist))
	 (all (all-completions candidate liece-query-client-alist)))
    (liece-minibuffer-finalize-completion completion candidate all)))

(defmacro liece-complete-query ()
  '(let ((completion-ignore-case t) 
	 (liece-minibuffer-complete-function
	  (function liece-minibuffer-complete-client-query)))
     (read-from-minibuffer 
      (_ "Which query: ") liece-query-client-lastcommand
      liece-minibuffer-map)))

;;;###liece-autoload
(defun liece-command-client-version (client-version-nick-var)
  "Ask about someones client version."
  (interactive (list (liece-complete-client)))
  (setq client-version-nick-var (liece-channel-real client-version-nick-var)
	liece-query-client-nick client-version-nick-var)
  (liece-send "PRIVMSG %s :%s" 
	       client-version-nick-var liece-query-client-version))

;;;###liece-autoload
(defun liece-command-client-userinfo (client-userinfo-nick-var)
  "Ask about someones client userinfo."
  (interactive (list (liece-complete-client)))
  (setq client-userinfo-nick-var (liece-channel-real client-userinfo-nick-var)
	liece-query-client-nick client-userinfo-nick-var)
  (liece-send "PRIVMSG %s :%s" 
	       client-userinfo-nick-var liece-query-client-userinfo))

;;;###liece-autoload
(defun liece-command-client-help (client-help-nick-var)
  "Ask about someones client help."
  (interactive (list (liece-complete-client)))
  (setq client-help-nick-var (liece-channel-real client-help-nick-var)
	liece-query-client-nick client-help-nick-var)
  (liece-send "PRIVMSG %s :%s"
	       client-help-nick-var liece-query-client-help))

;;;###liece-autoload
(defun liece-command-client-clientinfo (client-clientinfo-nick-var)
  "Ask about someones client clientinfo."
  (interactive (list (liece-complete-client)))
  (setq client-clientinfo-nick-var 
	(liece-channel-real client-clientinfo-nick-var)
	liece-query-client-nick client-clientinfo-nick-var)
  (liece-send "PRIVMSG %s :%s" 
	       client-clientinfo-nick-var liece-query-client-clientinfo))

;;;###liece-autoload
(defun liece-command-client-action (&optional private)
  "Send action ctcp - if on empty line, ask for the message"
  (interactive
   (if current-prefix-arg
       (list current-prefix-arg)))
  (let ((completion-ignore-case t) message
	(liece-message-type 'action))
    (if private
	(setq liece-privmsg-partner 
	      (liece-channel-virtual
	       (liece-minibuffer-completing-default-read 
		(_ "To whom: ")
		(append liece-nick-alist liece-channel-alist)
		nil nil liece-privmsg-partner))))
    (beginning-of-line)
    (setq message (buffer-substring (point)(progn (end-of-line)(point))))
    (if (string= message "")
	(setq message (read-string "Action: "))
      (liece-next-line 1))
    (liece-send "PRIVMSG %s :\001ACTION %s\001"
		 (if private
		     liece-privmsg-partner
		   (liece-channel-real liece-current-channel))
		 message)
    (if private
	(liece-own-private-message message)
      (liece-own-channel-message message))))

(define-obsolete-function-alias 'liece-command-send-action
  'liece-command-client-action)

;;;###liece-autoload
(defun liece-command-client-x-face (client-x-face-nick-var)
  "Ask about someones client x-face."
  (interactive (list (liece-complete-client)))
  (setq client-x-face-nick-var (liece-channel-real client-x-face-nick-var)
	liece-query-client-nick client-x-face-nick-var)
  (liece-send "PRIVMSG %s :%s"
	       client-x-face-nick-var liece-query-client-x-face))

;;;###liece-autoload
(defun liece-command-client-generic 
  (client-generic-nick-var liece-query-client-command)
  "Ask about someones client clientinfo."
  (interactive (list (liece-complete-client) (liece-complete-query)))
  (setq client-generic-nick-var (liece-channel-real client-generic-nick-var)
	liece-query-client-nick client-generic-nick-var
	liece-query-client-lastcommand liece-query-client-command)
  (if (string-equal-ignore-case liece-query-client-lastcommand "ping")
      (setq liece-ctcp-ping-time (current-time)))
  (liece-send "PRIVMSG %s :\001%s\001%s" 
	       client-generic-nick-var liece-query-client-lastcommand 
	       liece-query-client-insert-to-generic))

;;;###liece-autoload
(defun liece-command-client-userinfo-from-minibuffer ()
  "Ask about someones client clientinfo."
  (interactive)
  (setq liece-client-userinfo 
	(read-from-minibuffer 
	 (_ "New userinfo: ") liece-client-userinfo)))

;;;###liece-autoload
(defun liece-command-client-userinfo-from-commandbuffer ()
  "Ask about someones client clientinfo."
  (interactive)
  (beginning-of-line)
  (setq liece-client-userinfo 
	(buffer-substring (point)(progn (end-of-line)(point))))
  (liece-next-line 1))

;;;###liece-autoload
(defun liece-command-client-x-face-from-xbm-file (file)
  (interactive "fXBM File: ")
  (let (data)
    (and (file-exists-p file) (file-readable-p file)
	 (setq data (liece-x-face-encode file))
	 (setq liece-client-x-face
	       (replace-in-string (cadr (nth 3 data)) "[ \t\n]" "")))))

;;;###liece-autoload
(defun liece-command-client-x-face-from-minibuffer ()
  "Ask about someones client clientinfo."
  (interactive)
  (setq liece-client-x-face 
	(read-from-minibuffer 
	 (_ "New X-Face: ") liece-client-x-face)))

;;;###liece-autoload
(defun liece-command-client-x-face-from-commandbuffer ()
  "Ask about someones client clientinfo."
  (interactive)
  (beginning-of-line)
  (setq liece-client-x-face 
	(buffer-substring (point)(progn (end-of-line)(point))))
  (liece-next-line 1))

;;;###liece-autoload
(defun liece-command-client-ping (client-ping-nick-var)
  "Ask about someones client ping."
  (interactive (list (liece-complete-client)))
  (setq client-ping-nick-var (liece-channel-real client-ping-nick-var)
	liece-query-client-nick client-ping-nick-var
	liece-ctcp-ping-time (current-time))
  (liece-send "PRIVMSG %s :%s" 
	       client-ping-nick-var liece-query-client-ping))

;;;###liece-autoload
(defun liece-command-client-time (client-time-nick-var)
  "Ask about someones client time."
  (interactive (list (liece-complete-client)))
  (setq client-time-nick-var (liece-channel-real client-time-nick-var)
	liece-query-client-nick client-time-nick-var)
  (liece-send "PRIVMSG %s :%s"
	       client-time-nick-var liece-query-client-time))

;;;###liece-autoload
(defun liece-command-send-file (name to-who)
  "Send a file to given  user"
  (interactive "fFile name: \nsTo who: ")
  (save-excursion 
    (set-buffer (liece-get-buffer-create (format "*IRC S_FILE_%s*" name)))
    (erase-buffer)
    (save-restriction
      (narrow-to-region (point-max) (point-max))
      (goto-char (point-max))
      (insert-file-contents-as-binary name)
      (liece-quote-encode-region (point-min) (point-max))
      (goto-char (point-min))
      (let ((bound (min (point-max) (+ 80 (point)))))
	(liece-send "NOTICE %s :\001FILE START %s :%s\001" 
		     to-who name (buffer-substring (point) bound))
	(goto-char bound)
	(while (not (eobp))
	  (if (= 1 (mod (point) 800)) (sleep-for 1))
	  (setq bound (min (point-max) (+ 80 (point))))
	  (liece-send "NOTICE %s :\001FILE CONT %s :%s\001"
		       to-who name (buffer-substring (point) bound))
	  (goto-char bound))))
    (liece-send "NOTICE %s :\001FILE END %s : \001" to-who name)
    (kill-buffer (current-buffer))))

(provide 'liece-ctcp)

;;; liece-ctcp.el ends here
