;;; liece-compat.el --- Provide compatibility for various emacsen.
;; Copyright (C) 1998, 1999 Daiki Ueno

;; Author: Daiki Ueno <ueno@ueda.info.waseda.ac.jp>
;; Created: 1998-09-28
;; Revised: 1999-12-19
;; Keywords: IRC, liece, APEL

;; This file is part of Liece.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Code:

(require 'cl)
(require 'emu)
(require 'alist)
(require 'broken)
(require 'pcustom)

(broken-facility colon-keyword-usable
  (condition-case nil
      (prog1 t
	:symbol-for-testing-whether-colon-keyword-is-available-or-not)
    (void-variable nil)))

(eval-when-compile 
  (require 'wid-edit))

(eval-and-compile
  (autoload 'widget-convert-button "wid-edit"))

(defalias 'liece-widget-convert-button 'widget-convert-button)
(defalias 'liece-widget-button-click 'widget-button-click)

(defun-maybe turn-on-font-lock ()
  "Turn on Font Lock mode conditionally.
Turn on only if the terminal can display it."
  (interactive)
  (font-lock-mode 1))
  
(static-if (fboundp 'set-keymap-parents)
    (defalias 'liece-set-keymap-parents 'set-keymap-parents)
  (defmacro liece-set-keymap-parents (keymap parents)
    (if (fboundp 'set-keymap-parent)
	`(set-keymap-parent ,keymap (car ,parents))
      `(setq ,keymap (copy-keymap (car ,parents))))))
    
(defun-maybe event-buffer (event)
  "Return the buffer of the window over which mouse event EVENT occurred.
Return nil unless both (mouse-event-p EVENT) and
(event-over-text-area-p EVENT) are non-nil.
\[XEmacs emulating function]"
  (let ((window (posn-window (event-start event))))
    (and (windowp window) (window-buffer window))))

(defun-maybe event-point (event)
  "Return the character position of the mouse event EVENT.
If the event did not occur over a window, or did not occur over text,
then this returns nil.  Otherwise, it returns a position in the buffer
visible in the event's window.
\[XEmacs emulating function]"
  (posn-point (event-start event)))

(defun-maybe region-active-p ()
  "Return non-nil if the region is active.
If `zmacs-regions' is true, this is equivalent to `region-exists-p'.
Otherwise, this function always returns false.
\[XEmacs emulating function]"
  mark-active)

(defalias-maybe 'buffer-disable-undo 'buffer-flush-undo)

(defalias-maybe 'move-to-column-strictly 'move-to-column)

(defalias-maybe 'truncate-string-to-width 'truncate-string)

(eval-and-compile
  (if running-emacs-19
      (defmacro liece-get-buffer-window (buffer)
	`(get-buffer-window ,buffer t))
    (defmacro liece-get-buffer-window (buffer)
      `(get-buffer-window ,buffer))))

(defalias 'liece-make-overlay 'make-overlay)
(defalias 'liece-delete-overlay 'delete-overlay)
(defalias 'liece-overlay-put 'overlay-put)
(defalias 'liece-move-overlay 'move-overlay)
(defalias 'liece-overlay-end 'overlay-end)
(defalias 'liece-overlay-get 'overlay-get)
(defalias 'liece-overlays-at 'overlays-at)
(defalias 'liece-put-text-property 'put-text-property)
(defalias 'liece-add-text-property 'add-text-property)

(defun liece-kill-all-overlays ()
  "Delete all overlays in the current buffer."
  (let* ((overlayss (overlay-lists))
	 (buffer-read-only nil)
	 (overlays (delq nil (nconc (car overlayss) (cdr overlayss)))))
    (while overlays
      (delete-overlay (pop overlays)))))

(defalias 'liece-run-at-time 'run-at-time)
(defalias 'liece-cancel-timer 'cancel-timer)

(eval-and-compile
  (unless (fboundp 'with-timeout)
    (defun with-timeout-handler (tag)
      (throw tag 'timeout))
  
    (defmacro with-timeout (list &rest body)
      (let ((seconds (car list))
	    (timeout-forms (cdr list)))
	`(let ((with-timeout-tag (cons nil nil))
	       with-timeout-value with-timeout-timer)
	   (if (catch with-timeout-tag
		 (progn
		   (setq with-timeout-timer
			 (liece-run-at-time ,seconds nil
					     'with-timeout-handler
					     with-timeout-tag))
		   (setq with-timeout-value (progn . ,body))
		   nil))
	       (progn . ,timeout-forms)
	     (liece-cancel-timer with-timeout-timer)
	     with-timeout-value))))
    ))

(eval-and-compile
  (if (fboundp 'get-buffer-window-list)
      (defmacro liece-get-buffer-window-list (buf)
	`(get-buffer-window-list ,buf nil t))
    (defmacro liece-get-buffer-window-list (buf)
      `(list (get-buffer-window ,buf))))
  
  (if (fboundp 'window-displayed-height)
      (defalias 'liece-window-height 'window-displayed-height)
    (defalias 'liece-window-height 'window-height))
  )

(defalias 'liece-mode-line-buffer-identification 'identity)

(defun-maybe remassoc (key lst)
  "Delete by side effect any elements of LIST whose car is `equal' to KEY.
The modified LIST is returned.  If the first member of LIST has a car
that is `equal' to KEY, there is no way to remove it by side effect;
therefore, write `(setq foo (remassoc key foo))' to be sure of changing
the value of `foo'. [XEmacs emulating function]"
  (let ((i (length lst)) r)
    (while (> i 0)
      (setq i (1- i))
      (let ((current (nth i lst)))
	(if (not (equal (car current) key))
	    (setq r (cons current r)))))
    r))

(defun-maybe remassq (key lst)
  "Delete by side effect any elements of LIST whose car is `eq' to KEY.
The modified LIST is returned.  If the first member of LIST has a car
that is `eq' to KEY, there is no way to remove it by side effect;
therefore, write `(setq foo (remassq key foo))' to be sure of changing
the value of `foo'. [XEmacs emulating function]"
  (let ((i (length lst)) r)
    (while (> i 0)
      (setq i (1- i))
      (let ((current (nth i lst)))
	(if (not (eq (car current) key))
	    (setq r (cons current r)))))
    r))

(defun-maybe valid-plist-p (plist)
  "Given a plist, return non-nil if its format is correct.
If it returns nil, `check-valid-plist' will signal an error when given
the plist; that means it's a malformed or circular plist or has non-symbols
as keywords. [XEmacs emulating function]"
  (and (listp plist) (evenp (length plist))))
  
(defun-maybe plist-get (plist prop)
  "Extract a value from a property list.
PLIST is a property list, which is a list of the form
(PROP1 VALUE1 PROP2 VALUE2...).  This function returns the value
corresponding to the given PROP, or DEFAULT if PROP is not
one of the properties on the list."
  (let ((plist plist) value)
    (while plist
      (if (eq (car plist) prop)
	  (setq value (cadr plist)
		plist nil))
      (setq plist (cddr plist)))
    value))

(defvar-maybe current-language-environment "ASCII")
(defvar-maybe frame-title-format "")
(defvar-maybe enable-multibyte-characters nil)
(defvar-maybe completion-display-completion-list-function 
  'display-completion-list)
  
(defalias-maybe 'easy-menu-add-item 'ignore)
  
;; from XEmacs's help.el
(defmacro-maybe function-documentation (function &optional strip-arglist)
  "Return a string giving the documentation for FUNCTION, if any.
If the optional argument STRIP-ARGLIST is non-nil, remove the arglist
part of the documentation of internal subroutines."
  `(let ((doc (condition-case nil
		  (or (documentation ,function)
		      (_ "not documented"))
		(void-function ""))))
     (if (and ,strip-arglist
	      (string-match "[\n\t ]*\narguments: ?(\\(.*\\))\n?\\'" doc))
	 (setq doc (substring doc 0 (match-beginning 0)))
       doc)))
  
;; from XEmacs's minibuf.el
(defun-maybe temp-minibuffer-message (m)
  (let ((savemax (point-max)))
    (save-excursion
      (goto-char (point-max))
      (message nil)
      (insert m))
    (let ((inhibit-quit t))
      (sit-for 2)
      (delete-region savemax (point-max)))))

;; from XEmacs's subr.el
(defun-maybe replace-in-string (str regexp newtext &optional literal)
  "Replace all matches in STR for REGEXP with NEWTEXT string,
 and returns the new string.
Optional LITERAL non-nil means do a literal replacement.
Otherwise treat \\ in NEWTEXT string as special:
  \\& means substitute original matched text,
  \\N means substitute match for \(...\) number N,
  \\\\ means insert one \\."
  (let ((rtn-str "")
	(start 0)
	(special)
	match prev-start)
    (while (setq match (string-match regexp str start))
      (setq prev-start start
	    start (match-end 0)
	    rtn-str
	    (concat
	     rtn-str
	     (substring str prev-start match)
	     (cond (literal newtext)
		   (t (mapconcat
		       (lambda (c)
			 (if special
			     (progn
			       (setq special nil)
			       (cond ((eq c ?\\) "\\")
				     ((eq c ?&)
				      (substring str
						 (match-beginning 0)
						 (match-end 0)))
				     ((and (>= c ?0) (<= c ?9))
				      (if (> c (+ ?0 (length
						      (match-data))))
					  ;; Invalid match num
					  (error "Invalid match num: %c" c)
					(setq c (- c ?0))
					(substring str
						   (match-beginning c)
						   (match-end c))))
				     (t (char-to-string c))))
			   (if (eq c ?\\) (progn (setq special t) nil)
			     (char-to-string c))))
		       newtext ""))))))
    (concat rtn-str (substring str start))))
  
(defvar-maybe passwd-echo nil)

(defun-maybe read-passwd (prompt &optional confirm default)
  "Read a single line of text from user without echoing, and return it."
  (let ((ans "")
	(c 0)
	(echo-keystrokes 0)
	(cursor-in-echo-area t)
	(log-message-max-size 0)
	done msg truncate)
    (while (not done)
      (if (or (not passwd-echo) (string-equal "" ans))
	  (setq msg prompt)
	(setq msg (concat prompt (make-string (length ans) ?*)))
	(setq truncate
	      (1+ (- (length msg) (window-width (minibuffer-window)))))
	(and (> truncate 0)
	     (setq msg (concat "$" (substring msg (1+ truncate))))))
      (message msg)
      (setq c (read-char-exclusive))
      (cond ((= c ?\C-g)
	     (setq quit-flag t
		   done t))
	    ((or (= c ?\r) (= c ?\n) (= c ?\e))
	     (setq done t))
	    ((= c ?\C-u)
	     (setq ans ""))
	    ((and (/= c ?\b) (/= c ?\177))
	     (setq ans (concat ans (char-to-string c))))
	    ((> (length ans) 0)
	     (setq ans (substring ans 0 -1)))))
    (if quit-flag
	(prog1
	    (setq quit-flag nil)
	  (message "Quit")
	  (beep t))
      (message "")
      ans)))

(provide 'liece-compat)

;;; liece-compat.el ends here
