/*
 notifylist.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

#define ISON_EVENT "event 303"

typedef struct
{
    gchar *nick;
    gboolean hostok;
}
ISON_REC;

static gint notify_tag;

static ISON_REC *create_ison(SERVER_REC *server, gchar *nick, gboolean hostok)
{
    ISON_REC *rec;

    rec = g_new(ISON_REC, 1);
    server->ison_users = g_list_append(server->ison_users, rec);

    rec->nick = g_strdup(nick);
    rec->hostok = hostok;

    return rec;
}

static ISON_REC *find_ison(SERVER_REC *server, gchar *nick)
{
    GList *tmp;

    for (tmp = g_list_first(server->ison_users); tmp != NULL; tmp = tmp->next)
    {
        ISON_REC *rec = tmp->data;

        if (g_strcasecmp(rec->nick, nick) == 0)
            return rec;
    }

    return NULL;
}

LIST_REC *notifylist_add(gchar *nick, gchar *ircnet)
{
    LIST_REC *rec;

    rec = list_add(&notifies, nick, ircnet);
    signal_emit("notifylist new", 1, rec);
    return rec;
}

void notifylist_remove(LIST_REC *rec)
{
    signal_emit("notifylist remove", 1, rec);
    list_remove(&notifies, rec);
}

/* timeout function: send /ISON commands to server to check if someone in
   notify list is in IRC */
static void notifylist_timeout_server(SERVER_REC *server)
{
    GString *str;
    gchar *ptr;
    GList *tmp;

    g_return_if_fail(server != NULL);

    if (notifies == NULL) return; /* no-one in notify list */

    for (tmp = server_redirect_getqueue(server, ISON_EVENT, NULL); tmp != NULL; tmp = tmp->next)
    {
        REDIRECT_REC *rec = tmp->data;

        if (strcmp(rec->name, "notifylist event") == 0)
        {
            /* still not received all replies to previous /ISON commands.. */
            return;
        }
    }

    str = g_string_new("ISON :");
    for (tmp = g_list_first(notifies); tmp != NULL; tmp = tmp->next)
    {
        LIST_REC *rec = tmp->data;
        gint len;

        if (*rec->value != '\0' && (server->ircnet == NULL || !find_substr(rec->value, server->ircnet)))
            continue; /* wrong server */

        len = strlen(rec->key);

        if (str->len+len+1 > 510)
        {
            irc_send_cmd(server, str->str);
            server_redirect_event(server, NULL, 1, ISON_EVENT, "notifylist event", -1, NULL);
            g_string_assign(str, "ISON :");
        }

        if (str->str[str->len-1] != ':')
            g_string_append_c(str, ' ');

        /* stop at ! character */
        for (ptr = rec->key; *ptr != '\0' && *ptr != '!'; ptr++)
            g_string_append_c(str, *ptr);
    }

    irc_send_cmd(server, str->str);
    server_redirect_event(server, NULL, 1, ISON_EVENT, "notifylist event", -1, NULL);
    g_string_free(str, TRUE);
}

static gint notifylist_timeout_func(void)
{
    g_list_foreach(servers, (GFunc) notifylist_timeout_server, NULL);
    return 1;
}

static gboolean event_ison(gchar *data, SERVER_REC *server)
{
    GList *tmp, *next;
    gchar *params, *online;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &online);

    /* we're trying to find out who of the notify list members are in IRC */
    while (online != NULL && *online != '\0')
    {
        gchar *ptr;

        ptr = strchr(online, ' ');
        if (ptr != NULL) *ptr++ = '\0';
        server->ison_tempusers =
            g_list_append(server->ison_tempusers, g_strdup(online));
        online = ptr;
    }

    for (tmp = server_redirect_getqueue(server, ISON_EVENT, NULL); tmp != NULL; tmp = tmp->next)
    {
        REDIRECT_REC *rec = tmp->data;

        if (strcmp(rec->name, "notifylist event") == 0)
        {
            /* still /ison's left.. */
            g_free(params);
            return TRUE;
        }
    }

    /* all /ISON requests got - scan through notify list to check who's
     came and who's left */

    /* first check who's come to irc */
    for (tmp = g_list_first(server->ison_tempusers); tmp != NULL; tmp = next)
    {
        next = tmp->next;

        if (find_ison(server, tmp->data) == NULL)
        {
            /* not found from list, user's just joined to irc */
            gchar *str;

            create_ison(server, tmp->data, FALSE);
            str = g_strdup_printf("WHOIS %s", (gchar *) tmp->data);
            irc_send_cmd(server, str);
	    g_free(str);

	    server_redirect_event(server, tmp->data, 2,
				  "event 318", "event empty", 1,
				  "event 401", "event empty", 1,
				  "event 311", "notifylist event whois", 1,
				  "event 301", "event empty", 1,
				  "event 312", "event empty", 1,
				  "event 313", "event empty", 1,
				  "event 317", "event empty", 1,
				  "event 319", "event empty", 1, NULL);
	}
    }

    /* and then check who's left irc */
    for (tmp = g_list_first(server->ison_users); tmp != NULL; tmp = next)
    {
        ISON_REC *rec = tmp->data;
        next = tmp->next;

        if (glist_find_icase_string(server->ison_tempusers, rec->nick) == NULL)
        {
            /* not found from list, user's left irc */
            server->ison_users = g_list_remove(server->ison_users, rec);

            if (rec->hostok)
                signal_emit("notifylist left", 2, server, rec->nick);

            g_free(rec->nick);
            g_free(rec);
        }
    }

    /* free memory used by temp list */
    g_list_foreach(server->ison_tempusers, (GFunc) g_free, NULL);
    g_list_free(server->ison_tempusers);
    server->ison_tempusers = NULL;

    g_free(params);
    return TRUE;
}

gboolean notifylist_ison_server(SERVER_REC *server, gchar *nick)
{
    ISON_REC *rec;

    g_return_val_if_fail(nick != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    rec = find_ison(server, nick);
    return rec != NULL && rec->hostok;
}

static SERVER_REC *notifylist_ison_serverlist(gchar *nick, gchar *taglist)
{
    SERVER_REC *server;
    GList *list, *tmp;

    server = NULL;
    for (tmp = list = str2list(taglist, ' '); tmp != NULL; tmp = tmp->next)
    {
        server = server_find_ircnet(tmp->data);
        if (server != NULL && notifylist_ison_server(server, nick))
            break;
    }
    if (list != NULL)
    {
	g_free(list->data);
	g_list_free(list);
    }

    return tmp == NULL ? NULL : server;
}

SERVER_REC *notifylist_ison(gchar *nick, gchar *serverlist)
{
    GList *tmp;

    g_return_val_if_fail(nick != NULL, FALSE);
    g_return_val_if_fail(serverlist != NULL, FALSE);

    if (*serverlist != '\0')
        return notifylist_ison_serverlist(nick, serverlist);

    /* any server.. */
    for (tmp = g_list_first(servers); tmp != NULL; tmp = tmp->next)
        if (notifylist_ison_server(tmp->data, nick)) return tmp->data;

    return NULL;
}

static gboolean notifylist_init_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server_redirect_init(server, "command ison", 1, ISON_EVENT, NULL);
    return TRUE;
}

static gboolean notifylist_deinit_server(SERVER_REC *server)
{
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    for (tmp = g_list_first(server->ison_users); tmp != NULL; tmp = tmp->next)
    {
        ISON_REC *rec = tmp->data;

        g_free(rec->nick);
        g_free(rec);
    }
    g_list_free(server->ison_users);

    return TRUE;
}

LIST_REC *notifylist_find(gchar *nick)
{
    GList *tmp;
    gint len;

    len = strlen(nick);

    for (tmp = g_list_first(notifies); tmp != NULL; tmp = tmp->next)
    {
        LIST_REC *rec = tmp->data;

        if (g_strncasecmp(rec->key, nick, len) == 0 &&
            (rec->key[len] == '\0' || rec->key[len] == '!')) return rec;
    }

    return NULL;
}

static gboolean event_whois(gchar *data, SERVER_REC *server)
{
    LIST_REC *rec;
    ISON_REC *ison;
    gchar *params, *nick, *user, *host, *realname;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(server != NULL, FALSE);

    params = event_get_params(data, 6, NULL, &nick, &user, &host, NULL, &realname);

    rec = notifylist_find(nick);
    if (rec != NULL)
    {
        if (!irc_mask_match(rec->key, nick, user, host))
        {
            /* user or host didn't match */
            g_free(params);
            return TRUE;
        }
    }

    ison = find_ison(server, nick);
    if (ison != NULL)
    {
        ison->hostok = TRUE;
        signal_emit("notifylist joined", 5, server, nick, user, host, realname);
    }
    g_free(params);
    return TRUE;
}

void notifylist_init(void)
{
    notify_tag = gui_timeout_add(NOTIFY_TIMECHECK, (GUITimeoutFunction) notifylist_timeout_func, NULL);
    signal_add("server connected", (SIGNAL_FUNC) notifylist_init_server);
    signal_add("server disconnected", (SIGNAL_FUNC) notifylist_deinit_server);
    signal_add("notifylist event", (SIGNAL_FUNC) event_ison);
    signal_add("notifylist event whois", (SIGNAL_FUNC) event_whois);
}

void notifylist_deinit(void)
{
    gui_timeout_remove(notify_tag);
    signal_remove("server connected", (SIGNAL_FUNC) notifylist_init_server);
    signal_remove("server disconnected", (SIGNAL_FUNC) notifylist_deinit_server);
    signal_remove("notifylist event", (SIGNAL_FUNC) event_ison);
    signal_remove("notifylist event whois", (SIGNAL_FUNC) event_whois);
}
