/*
 signals.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

#define SIGNAL_LISTS 3

static GHashTable *signals;
static gboolean global_signals; /* "signal"s have been bound */

static void signal_add_to(gint pos, gchar *signal, SIGNAL_FUNC func)
{
    GSList **siglist;

    g_return_if_fail(signal != NULL);
    g_return_if_fail(func != NULL);

    if (strcmp(signal, "signal") == 0)
	global_signals = TRUE;

    siglist = g_hash_table_lookup(signals, signal);
    if (siglist == NULL)
    {
	siglist = g_new0(GSList*, SIGNAL_LISTS);
#ifdef MEM_DEBUG
	ig_set_data(signal);
#endif
	g_hash_table_insert(signals, g_strdup(signal), siglist);
#ifdef MEM_DEBUG
	ig_set_data("");
#endif
    }

    siglist[pos] = g_slist_prepend(siglist[pos], func);
}

void signal_add(gchar *signal, SIGNAL_FUNC func)
{
    signal_add_to(1, signal, func);
}

void signal_add_first(gchar *signal, SIGNAL_FUNC func)
{
    signal_add_to(0, signal, func);
}

void signal_add_last(gchar *signal, SIGNAL_FUNC func)
{
    signal_add_to(2, signal, func);
}

void signal_remove(gchar *signal, SIGNAL_FUNC func)
{
    GSList **siglist, *link;
    gchar *origkey;
    gint n;

    if (g_hash_table_lookup_extended(signals, signal, (gpointer *) &origkey, (gpointer *) &siglist))
    {
	for (n = 0; n < SIGNAL_LISTS; n++)
	{
	    link = g_slist_find(siglist[n], func);
	    if (link != NULL)
	    {
		/* remove function from emit list */
		siglist[n] = g_slist_remove_link(siglist[n], link);
		g_slist_free_1(link);

		if (siglist[0] == NULL && siglist[1] == NULL && siglist[2] == NULL)
		{
		    /* no functions left, destroy the whole signal */
		    if (strcmp(signal, "signal") == 0)
			global_signals = FALSE;

		    g_hash_table_remove(signals, origkey);
		    g_free(origkey);
		    g_free(siglist);
		}
		return;
	    }
	}
    }

    g_error("signal_remove() : signal \"%s\" isn't grabbed for %p", signal, func);
}

gboolean signal_emit(gchar *signal, gint params, ...)
{
    GSList **siglist, *emitlist, *next;
    gpointer arglist[7];
    va_list va;
    gint n;

    g_return_val_if_fail(signal != NULL, FALSE);
    g_return_val_if_fail(params >= 0 && params <= sizeof(arglist)/sizeof(arglist[0]), FALSE);

    /* get arguments */
    va_start(va, params);
    for (n = 0; n < 7; n++)
	arglist[n] = n >= params ? NULL : va_arg(va, gpointer);
    va_end(va);

    /* send "signal" */
    if (global_signals && strcmp(signal, "signal") != 0)
	signal_emit("signal", 7, signal, arglist[0], arglist[1], arglist[2], arglist[3], arglist[4], arglist[5]);

    /* find the emit lists */
    siglist = g_hash_table_lookup(signals, signal);
    if (siglist == NULL)
	return FALSE;

    for (n = 0; n < SIGNAL_LISTS; n++)
    {
	/* run signals in emit lists */
	emitlist = siglist[n];

	for (; emitlist != NULL; emitlist = next)
        {
            SIGNAL_FUNC func = emitlist->data;

            next = emitlist->next;
            if (!func(arglist[0], arglist[1], arglist[2], arglist[3], arglist[4], arglist[5], arglist[6]))
		return TRUE;
        }
    }

    return TRUE;
}

void signals_init(void)
{
    signals = g_hash_table_new((GHashFunc) g_str_hash, (GCompareFunc) g_str_equal);
    global_signals = FALSE;
}

static void signal_free(gpointer key, GSList **siglist)
{
    gint n;

    for (n = 0; n < SIGNAL_LISTS; n++)
	g_slist_free(siglist[n]);

    g_free(key);
    g_free(siglist);
}

void signals_deinit(void)
{
    g_hash_table_foreach(signals, (GHFunc) signal_free, NULL);
    g_hash_table_destroy(signals);
}
