/*
 channels.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

GList *channels; /* List of all channels */
CHANNEL_REC *cur_channel;

CHANNEL_REC *channel_create(SERVER_REC *server, gchar *channel, gint type)
{
    CHANNEL_REC *rec;

    g_return_val_if_fail(channel != NULL, NULL);

    rec = g_new0(CHANNEL_REC, 1);
    channels = g_list_append(channels, rec);

    rec->name = g_strdup(channel);
    rec->type = type;
    rec->server = server;
    rec->createtime = time(NULL);

    signal_emit("channel created", 1, rec);

    return rec;
}

void channel_destroy(CHANNEL_REC *channel)
{
    g_return_if_fail(channel != NULL);

    channels = g_list_remove(channels, channel);
    signal_emit("channel destroyed", 1, channel);

    if (channel->type == CHANNEL_TYPE_CHANNEL &&
        channel->server != NULL && !channel->left && !channel->kicked)
    {
        /* destroying channel record without actually left the channel yet */
        gchar *str;

        str = g_strconcat("PART ", channel->name, NULL);
        irc_send_cmd(channel->server, str);
        g_free(str);
    }

    if (channel->topic != NULL) g_free(channel->topic);
    if (channel->key != NULL) g_free(channel->key);
    g_free(channel->name);
    g_free(channel);
}

void channel_change_name(CHANNEL_REC *channel, gchar *name)
{
    g_return_if_fail(channel != NULL);
    g_return_if_fail(name != NULL);

    g_free(channel->name);
    channel->name = g_strdup(name);

    signal_emit("channel name changed", 1, channel);
}

CHANNEL_REC *channel_find(SERVER_REC *server, gchar *channel)
{
    GList *tmp;

    g_return_val_if_fail(channel != NULL, NULL);

    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

        if ((server == NULL || server == rec->server) &&
            ((*channel == '!' && *rec->name == '!' && g_strcasecmp(channel+1, rec->name+6) == 0) ||
             g_strcasecmp(channel, rec->name) == 0)) return rec;
    }

    return NULL;
}

CHANNEL_REC *channel_find_any(gchar *channel)
{
    GList *tmp;

    g_return_val_if_fail(channel != NULL, NULL);

    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

        if ((*channel == '!' && *rec->name == '!' && g_strcasecmp(channel+1, rec->name+6) == 0) ||
            g_strcasecmp(channel, rec->name) == 0) return rec;
    }

    return NULL;
}

CHANNEL_REC *channel_find_level(gint level)
{
    GList *tmp;

    /* ignore hilighting.. unless we really want only hilighting .. hm.. */
    if (level != MSGLEVEL_HILIGHT)
	level &= ~(MSGLEVEL_HILIGHT | MSGLEVEL_NOHILIGHT);

    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

        if (rec->level & level)
            return rec;
    }

    return NULL;
}

CHANNEL_REC *channel_find_closest(SERVER_REC *server, gchar *channel, gint level)
{
    CHANNEL_REC *chanrec;

    /* first try to find channel/query */
    chanrec = channel == NULL ? NULL : channel_find(server, channel);

    /* then try finding some window which level matches */
    if (chanrec == NULL)
        chanrec = channel_find_level(level);

    /* default to currently focused window */
    if (chanrec == NULL) chanrec = cur_channel;

    return chanrec;
}

static gboolean event_cannot_join(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);

    if (channel[0] == '!' && channel[1] == '!')
	channel++; /* server didn't understand !channels */

    chanrec = channel_find(server, channel);

    if (chanrec != NULL && !chanrec->names_got && chanrec->type == CHANNEL_TYPE_CHANNEL)
    {
        chanrec->left = TRUE;
        channel_destroy(chanrec);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_target_unavailable(gchar *data, SERVER_REC *server)
{
    gchar *params, *channel;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, NULL, &channel);
    if (ischannel(*channel))
    {
        /* channel is unavailable. */
        event_cannot_join(data, server);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_topic_get(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel, *topic;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 3, NULL, &channel, &topic);
    chanrec = channel_find(server, channel);
    if (chanrec != NULL)
    {
        if (chanrec->topic != NULL) g_free(chanrec->topic);
        chanrec->topic = *topic == '\0' ? NULL : g_strdup(topic);
        signal_emit("channel topic changed", 1, chanrec);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_topic(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel, *topic;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, &channel, &topic);
    chanrec = channel_find(server, channel);
    if (chanrec != NULL)
    {
        if (chanrec->topic != NULL) g_free(chanrec->topic);
        chanrec->topic = *topic == '\0' ? NULL : g_strdup(topic);
        signal_emit("channel topic changed", 1, chanrec);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_join(gchar *data, SERVER_REC *server, gchar *nick, gchar *address)
{
    gchar *params, *channel, *tmp;
    CHANNEL_REC *chanrec;

    g_return_val_if_fail(data != NULL, FALSE);

    if (g_strcasecmp(nick, server->nick) != 0)
    {
        /* someone else joined channel, no need to do anything */
        return TRUE;
    }

    params = event_get_params(data, 1, &channel);
    tmp = strchr(channel, 7); /* ^G does something weird.. */
    if (tmp != NULL) *tmp = '\0';

    if (*channel == '!')
    {
        /* !channels have 5 chars long identification string before it's name,
           it's not known when /join is called so rename !channel here to
           !ABCDEchannel */
        gchar *shortchan;

        shortchan = g_strdup(channel);
        sprintf(shortchan, "!%s", channel+6);
        chanrec = channel_find(server, shortchan);
        if (chanrec != NULL)
        {
            g_free(chanrec->name);
            chanrec->name = g_strdup(channel);
        }

        g_free(shortchan);
    }

    chanrec = channel_find(server, channel);
    if (chanrec == NULL)
    {
        /* didn't get here through /join command.. well, create the channel
           anyway. */
        chanrec = channel_create(server, channel, CHANNEL_TYPE_CHANNEL);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_part(gchar *data, SERVER_REC *server, gchar *nick)
{
    gchar *params, *channel, *reason;
    CHANNEL_REC *chanrec;

    g_return_val_if_fail(data != NULL, FALSE);

    if (g_strcasecmp(nick, server->nick) != 0)
    {
	/* someone else part, no need to do anything */
	return TRUE;
    }

    params = event_get_params(data, 2, &channel, &reason);

    chanrec = channel_find(server, channel);
    if (chanrec != NULL)
    {
	chanrec->left = TRUE;
	channel_destroy(chanrec);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_kick(gchar *data, SERVER_REC *server)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel, *nick, *reason;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 3, &channel, &nick, &reason);

    if (g_strcasecmp(nick, server->nick) != 0)
    {
	/* someone else was kicked, no need to do anything */
	g_free(params);
	return TRUE;
    }

    chanrec = channel_find(server, channel);
    if (chanrec != NULL)
    {
	chanrec->kicked = TRUE;
	channel_destroy(chanrec);
    }

    g_free(params);
    return TRUE;
}

gchar *channel_get_mode(CHANNEL_REC *channel)
{
    GString *mode;
    gchar *ret;

    g_return_val_if_fail(channel != NULL, NULL);

    mode = g_string_new(NULL);

    if (channel->mode_secret) g_string_append_c(mode, 's');
    if (channel->mode_private) g_string_append_c(mode, 'p');
    if (channel->mode_moderate) g_string_append_c(mode, 'm');
    if (channel->mode_invite) g_string_append_c(mode, 'i');
    if (channel->mode_nomsgs) g_string_append_c(mode, 'n');
    if (channel->mode_optopic) g_string_append_c(mode, 't');
    if (channel->mode_key) g_string_append_c(mode, 'k');
    if (channel->limit > 0) g_string_append_c(mode, 'l');

    if (channel->mode_key) g_string_sprintfa(mode, " %s", channel->key);
    if (channel->limit > 0) g_string_sprintfa(mode, " %d", channel->limit);

    ret = mode->str;
    g_string_free(mode, FALSE);
    return ret;
}

void channels_init(void)
{
    signal_add("event 403", (SIGNAL_FUNC) event_cannot_join); /* no such channel */
    signal_add("event 405", (SIGNAL_FUNC) event_cannot_join); /* too many channels */
    signal_add("event 407", (SIGNAL_FUNC) event_cannot_join); /* duplicate channel */
    signal_add("event 471", (SIGNAL_FUNC) event_cannot_join); /* channel is full */
    signal_add("event 473", (SIGNAL_FUNC) event_cannot_join); /* invite only */
    signal_add("event 474", (SIGNAL_FUNC) event_cannot_join); /* banned */
    signal_add("event 475", (SIGNAL_FUNC) event_cannot_join); /* bad channel key */
    signal_add("event 476", (SIGNAL_FUNC) event_cannot_join); /* bad channel mask */

    signal_add("event topic", (SIGNAL_FUNC) event_topic);
    signal_add("event join", (SIGNAL_FUNC) event_join);
    signal_add("event part", (SIGNAL_FUNC) event_part);
    signal_add("event kick", (SIGNAL_FUNC) event_kick);
    signal_add("event 332", (SIGNAL_FUNC) event_topic_get);
    signal_add("event 437", (SIGNAL_FUNC) event_target_unavailable); /* channel/nick unavailable */
}


void channels_deinit(void)
{
    signal_remove("event 403", (SIGNAL_FUNC) event_cannot_join); /* no such channel */
    signal_remove("event 405", (SIGNAL_FUNC) event_cannot_join); /* too many channels */
    signal_remove("event 407", (SIGNAL_FUNC) event_cannot_join); /* duplicate channel */
    signal_remove("event 471", (SIGNAL_FUNC) event_cannot_join); /* channel is full */
    signal_remove("event 473", (SIGNAL_FUNC) event_cannot_join); /* invite only */
    signal_remove("event 474", (SIGNAL_FUNC) event_cannot_join); /* banned */
    signal_remove("event 475", (SIGNAL_FUNC) event_cannot_join); /* bad channel key */
    signal_remove("event 476", (SIGNAL_FUNC) event_cannot_join); /* bad channel mask */

    signal_remove("event topic", (SIGNAL_FUNC) event_topic);
    signal_remove("event join", (SIGNAL_FUNC) event_join);
    signal_remove("event part", (SIGNAL_FUNC) event_part);
    signal_remove("event kick", (SIGNAL_FUNC) event_kick);
    signal_remove("event 332", (SIGNAL_FUNC) event_topic_get);
    signal_remove("event 437", (SIGNAL_FUNC) event_target_unavailable); /* channel/nick unavailable */
}
