/*
 dialog-dcc.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "gui-dcc.h"

static GtkWidget *clist = NULL;

static void clist_append_dcc(GtkWidget *clist, DCC_REC *dcc)
{
    gchar *titles[7];
    time_t going;

    g_return_if_fail(clist != NULL);
    g_return_if_fail(dcc != NULL);

    if (dcc->type != DCC_TYPE_SEND && dcc->type != DCC_TYPE_GET)
        return;

    going = time(NULL) - dcc->starttime;
    if (going == 0) going = 1; /* no division by zeros :) */

    titles[0] = dcc->nick;
    titles[1] = dcc_type2str(dcc->type);
    titles[2] = g_strdup_printf("%ld", dcc->size);
    titles[3] = g_strdup_printf("%ld", dcc->transfd);
    titles[4] = dcc->size == 0 ? g_strdup("??") :
        g_strdup_printf("%d%%", (gint) (100*dcc->transfd/dcc->size));
    titles[5] = g_strdup_printf("%0.2f", (gdouble) (dcc->transfd-dcc->skipped)/going/1024);
    titles[6] = dcc->arg;

    gtk_clist_set_row_data(GTK_CLIST(clist), gtk_clist_append(GTK_CLIST(clist), titles), dcc);

    g_free(titles[2]); g_free(titles[3]);
    g_free(titles[4]); g_free(titles[5]);
}

static void sig_create_dialog(GtkWidget *clist)
{
    DCC_REC *dcc;
    gint selection;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1) return;

    dcc = gtk_clist_get_row_data(GTK_CLIST(clist), selection);
    if (dcc != NULL) gui_dcc_dialog(dcc);
}

static void sig_select_row(GtkWidget *clist, gint row)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(row));
}

static void sig_unselect_row(GtkWidget *clist)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
}

static void sig_destroy(GtkWidget **dialog)
{
    g_return_if_fail(dialog != NULL);
    g_return_if_fail(*dialog != NULL);

    gtk_widget_destroy(*dialog);
    *dialog = NULL;
    clist = NULL;
}

static gboolean redraw_dcc_list(void)
{
    GList *tmp;

    if (clist != NULL)
    {
        gtk_clist_freeze(GTK_CLIST(clist));
        gtk_clist_clear(GTK_CLIST(clist));
        for (tmp = g_list_first(dcc_conns); tmp != NULL; tmp = tmp->next)
            clist_append_dcc(clist, tmp->data);
	gtk_clist_thaw(GTK_CLIST(clist));
	if (dcc_conns == NULL)
	{
	    /* last dcc connection closed.. */
	    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
	}
    }
    return TRUE;
}

static gboolean dcc_transfer_update(DCC_REC *dcc)
{
    static gint lastsecs = 0;
    glong secs;
    gchar *str;
    gint row;

    g_return_val_if_fail(dcc != NULL, FALSE);
    if (clist == NULL) return TRUE;

    secs = time(NULL)-dcc->starttime;
    if (secs == lastsecs) return TRUE;

    lastsecs = secs;
    if (secs <= 0) secs = 1;

    row = gtk_clist_find_row_from_data(GTK_CLIST(clist), dcc);

    str = g_strdup_printf("%ld", dcc->transfd);
    gtk_clist_set_text(GTK_CLIST(clist), row, 3, str); g_free(str);
    str = g_strdup_printf("%d%%", (gint) (100*dcc->transfd/dcc->size));
    gtk_clist_set_text(GTK_CLIST(clist), row, 4, str); g_free(str);
    str = g_strdup_printf("%0.2f", (gdouble) (dcc->transfd-dcc->skipped)/secs/1024);
    gtk_clist_set_text(GTK_CLIST(clist), row, 5, str); g_free(str);

    return TRUE;
}

static void clist_click_column(GtkCList *clist, gint column)
{
    if (column != clist->sort_column)
	gtk_clist_set_sort_column(clist, column);
    else
    {
	if (clist->sort_type == GTK_SORT_ASCENDING)
	    clist->sort_type = GTK_SORT_DESCENDING;
	else
	    clist->sort_type = GTK_SORT_ASCENDING;
    }

    gtk_clist_sort(clist);
}

void dialog_dcc(void)
{
    static GtkWidget *dialog = NULL;
    GtkWidget *scrollbox;
    gchar *titles[7];

    if (dialog != NULL)
    {
        /* dialog already open */
        gdk_window_raise(dialog->window);
        return;
    }

    titles[0] = _("Nick"); titles[1] = _("Type"); titles[2] = _("File Size");
    titles[3] = _("Transferred"); titles[4] = _("% done");
    titles[5] = _("kB/sec"); titles[6] = _("File name");

    dialog = gnome_dialog_new(PACKAGE, _("Create dialog"), GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_widget_set_usize(dialog, 400, 250);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "destroy",
                              GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &dialog);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    /* Create dcc clist widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrollbox, TRUE, TRUE, 0);

    clist = gtk_clist_new_with_titles(7, titles);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 4, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 5, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 6, TRUE);
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);
    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
                       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "click_column",
		       GTK_SIGNAL_FUNC(clist_click_column), NULL);

    redraw_dcc_list();
    gtk_clist_select_row(GTK_CLIST(clist), 0, 0);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_create_dialog), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &dialog);

    /* Make columns resizeable */
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 2, TRUE);
    gtk_widget_show_all(dialog);
}

void dialog_dcc_init(void)
{
    signal_add("dcc created", (SIGNAL_FUNC) redraw_dcc_list);
    signal_add("dcc destroyed", (SIGNAL_FUNC) redraw_dcc_list);
    signal_add("dcc transfer update", (SIGNAL_FUNC) dcc_transfer_update);
}

void dialog_dcc_deinit(void)
{
    signal_remove("dcc created", (SIGNAL_FUNC) redraw_dcc_list);
    signal_remove("dcc destroyed", (SIGNAL_FUNC) redraw_dcc_list);
    signal_remove("dcc transfer update", (SIGNAL_FUNC) dcc_transfer_update);
}
