/*****************************************************************************/
/*  rfc959.c - General purpose routines for the FTP protocol (RFC 959)       */
/*  Copyright (C) 1999 Brian Masney <masneyb@seul.org>                       */
/*                                                                           */
/*  This library is free software; you can redistribute it and/or            */
/*  modify it under the terms of the GNU Library General Public              */
/*  License as published by the Free Software Foundation; either             */
/*  version 2 of the License, or (at your option) any later version.         */
/*                                                                           */
/*  This library is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU        */
/*  Library General Public License for more details.                         */
/*                                                                           */
/*  You should have received a copy of the GNU Library General Public        */
/*  License along with this library; if not, write to the Free Software      */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "protocols.h"

static int rfc959_connect (gftp_request *request);
static void rfc959_disconnect (gftp_request *request);
static long rfc959_get_file (gftp_request *request, const char *filename, long startsize);
static int rfc959_put_file (gftp_request *request, const char *filename, long startsize);
static int rfc959_end_transfer (gftp_request *request);
static int rfc959_list_files (gftp_request *request);
static int rfc959_get_next_file (gftp_request *request, gftp_file *fle, FILE *fd);
static int rfc959_set_data_type (gftp_request *request, int data_type);
static long rfc959_get_file_size (gftp_request *request, const char *filename);

static int rfc959_parse_ls (const char *lsoutput, gftp_file *fle);
static int rfc959_data_connection_new (gftp_request *request);
static int rfc959_accept_active_connection (gftp_request *request);
static time_t parse_time (char **str);
static int rfc959_parse_ls_eplf (char *str, gftp_file *fle);
static int rfc959_parse_ls_unix (char *str, int cols, gftp_file *fle);
static int rfc959_parse_ls_nt (char *str, gftp_file *fle);
static int rfc959_parse_ls_novell (char *str, gftp_file *fle);
static char *goto_next_token (char *pos);
static char *copy_token (char **dest, char *source);
static int rfc959_send_command (gftp_request *request, const char *command);
static int rfc959_read_response (gftp_request *request);
static int rfc959_chdir (gftp_request *request, const char *directory);
static int rfc959_rmdir (gftp_request *request, const char *directory);
static int rfc959_rmfile (gftp_request *request, const char *file);
static int rfc959_mkdir (gftp_request *request, const char *directory);
static int rfc959_rename (gftp_request *request, const char *oldname, const char *newname);
static int rfc959_chmod (gftp_request *request, const char *file, int mode);
static int rfc959_site (gftp_request *request, const char *command);
static char *parse_ftp_proxy_string (gftp_request *request);

void rfc959_init (gftp_request *request) {
   g_return_if_fail (request != NULL);
   
   request->init = rfc959_init;
   request->connect = rfc959_connect;
   request->disconnect = rfc959_disconnect;
   request->get_file = rfc959_get_file;
   request->put_file = rfc959_put_file;
   request->end_transfer = rfc959_end_transfer;
   request->list_files = rfc959_list_files;
   request->get_next_file = rfc959_get_next_file;
   request->set_data_type = rfc959_set_data_type;
   request->get_file_size = rfc959_get_file_size;
   request->chdir = rfc959_chdir;
   request->rmdir = rfc959_rmdir;
   request->rmfile = rfc959_rmfile;
   request->mkdir = rfc959_mkdir;
   request->rename = rfc959_rename;
   request->chmod = rfc959_chmod;
   request->site = rfc959_site;
   request->url_prefix = "ftp";
   request->protocol_name = "FTP";
   request->supports_resume = 1;
   request->unsafe_symlinks = !request->resolve_symlinks;
   request->add_onek = 0;
}
/*****************************************************************************/
static int rfc959_connect (gftp_request *request) {
   char tempchar, resp, *startpos, *endpos, *tempstr, *connect_host, *dir;
   struct sockaddr_in remote_address;
   struct servent *service;
   unsigned int port;
   int curhost, sock;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->hostname != NULL, -2);
   g_return_val_if_fail (request->username != NULL, -2);
   
   if (request->sockfd != NULL) return (0);
   if ((request->use_proxy = gftp_need_proxy (request)) < 0) {
      return (-1);
   }
   else if (request->use_proxy == 1) request->host = NULL;
   
   if ((sock = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
      if (request->logging) {
         request->logging_function (gftp_logging_error, request->user_data, _("Failed to create a socket: %s\n"),
         	g_strerror (errno));
      }
      return (-1);
   }
   memset (&remote_address, 0, sizeof (remote_address));
   remote_address.sin_family = AF_INET;

   port = htons (request->use_proxy ? request->proxy_port : request->port);
   connect_host = request->use_proxy ? request->proxy_hostname : request->hostname;

   if (ntohs (port) == 0) {
      if ((service = getservbyname ("ftp", "tcp")) == NULL) {
         port = htons (21);
         request->port = 21;
      }
      else {
         port = service->s_port;
         request->port = ntohs (service->s_port);
      }
   }
   remote_address.sin_port = port;

   if (request->host == NULL) {
      if (request->logging) {
         request->logging_function (gftp_logging_misc, request->user_data, _("Looking up %s\n"),
         	connect_host);
      }
      if (!(request->host = gethostbyname (connect_host))) {
         if (request->logging) {
            request->logging_function (gftp_logging_error, request->user_data, _("Cannot look up hostname %s: %s\n"),
         	connect_host, g_strerror (errno));
         }
         close (sock);
         return (-1);
      }
   }

   for (curhost = 0; request->host->h_addr_list[curhost] != NULL; curhost++) {
      memcpy (&remote_address.sin_addr, request->host->h_addr_list[curhost], request->host->h_length);
      if (request->logging) {
         request->logging_function (gftp_logging_misc, request->user_data, _("Trying %s:%d\n"),
         	request->host->h_name, ntohs (port));
      }
      if (connect (sock, (struct sockaddr *) &remote_address, sizeof (remote_address)) == -1) {
         if (request->logging) {
            request->logging_function (gftp_logging_error, request->user_data, _("Cannot connect to %s: %s\n"),
         	connect_host, g_strerror (errno));
         }
      }
      else break;
   }

   if (request->host->h_addr_list[curhost] == NULL) {
      close (sock);
      return (-1);
   }

   if (!request->use_proxy) {
      g_free (request->hostname);
      request->hostname = g_malloc (strlen (request->host->h_name) + 1);
      strcpy (request->hostname, request->host->h_name);
      connect_host = request->hostname;
   }
   

   if (request->logging) {
      request->logging_function (gftp_logging_misc, request->user_data, _("Connected to %s:%d\n"),
      	connect_host, ntohs (port));
   }
   
   request->sockfd_unbuffered = sock;
   request->sockfd = fdopen (sock, "r+");

   /* Get the banner */
   while (rfc959_read_response (request) == -2);
   
   /* Login the proxy server if available */
   if (request->use_proxy) {
      resp = '3';
      startpos = endpos = tempstr = parse_ftp_proxy_string (request);
      while ((resp == '3' || resp == '2') && *startpos != '\0') {
         if (*endpos == '\n' || *endpos == '\0') {
            tempchar = *(endpos + 1);
            if (*endpos != '\0') *(endpos + 1) = '\0';
            resp = rfc959_send_command (request, startpos);
            if (*endpos != '\0') *(endpos + 1) = tempchar;
            else break;
            startpos = endpos + 1;
         }
         endpos++;
      }
      g_free (tempstr);
   }
   else {
      tempstr = g_strconcat ("USER ", request->username, "\r\n", NULL);
      resp = rfc959_send_command (request, tempstr);
      g_free (tempstr);
      if (resp == '3') {
         tempstr = g_strconcat ("PASS ", request->password, "\r\n", NULL);
         resp = rfc959_send_command (request, tempstr);
         g_free (tempstr);
      }
      if (resp == '3' && request->account) {
         tempstr = g_strconcat ("ACCT ", request->account, "\r\n", NULL);
         resp = rfc959_send_command (request, tempstr);
         g_free (tempstr);
      }
   }
   
   if (resp != '2') {
      fclose (request->sockfd);
      request->sockfd = NULL;
      return (-2);
   }
   tempstr = g_malloc (9);
   if (request->data_type == GFTP_TYPE_BINARY) {
      strcpy (tempstr, "TYPE I\r\n");
   }
   else {
      strcpy (tempstr, "TYPE A\r\n");
   }
   rfc959_send_command (request, tempstr);
   g_free (tempstr);

   if (request->directory != NULL && *request->directory != '\0') {
      rfc959_chdir (request, request->directory);
   }
   else {
      if (rfc959_send_command (request, "PWD\r\n") != '2') return (0);
      
      if ((tempstr = strchr (request->last_ftp_response, '"')) == NULL) {
         return (0);
      }
      dir = tempstr + 1;
      
      if ((tempstr = strchr (dir, '"')) == NULL) {
         return (0);
      }
      if (tempstr != NULL) *tempstr = '\0';
      
      request->directory = g_malloc (strlen (dir) + 1);
      strcpy (request->directory, dir);
   }
   return (0);
}
/*****************************************************************************/
static void rfc959_disconnect (gftp_request *request) {
   g_return_if_fail (request != NULL);
   
   if (request->sockfd != NULL) {
      if (request->logging) {
         request->logging_function (gftp_logging_misc, request->user_data, 
         	_("Disconnecting from site %s\n"), request->hostname);
      }
      fclose (request->sockfd);
      request->sockfd = NULL;
      if (request->datafd) {
         fclose (request->datafd);
         request->datafd = NULL;
      }
   }
   request->host = NULL;
}
/*****************************************************************************/
static long rfc959_get_file (gftp_request *request, const char *filename, long startsize) {
   char *command, *tempstr, resp;
   int ret;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (filename != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);

   if (request->datafd == NULL && (ret = rfc959_data_connection_new (request)) < 0) return (ret);
   if (startsize > 0) {
      command = g_strdup_printf ("REST %ld\r\n", startsize);
      resp = rfc959_send_command (request, command);
      g_free (command);
      if (resp != '3') {
         fclose (request->datafd);
         request->datafd = NULL;
         return (-2);
      }
   }
   tempstr = g_strconcat ("RETR ", filename, "\r\n", NULL);
   ret = rfc959_send_command (request, tempstr);
   g_free (tempstr);
   if (ret != '1') {
      fclose (request->datafd);
      request->datafd = NULL;
      return (-2);
   }
   if (request->transfer_type == gftp_transfer_active) {
      if ((ret = rfc959_accept_active_connection (request)) < 0) {
         return (ret);
      }
   }
   
   if ((tempstr = strrchr (request->last_ftp_response, '(')) == NULL) {
      tempstr = request->last_ftp_response + 4;
      while (!isdigit (*tempstr) && *tempstr != '\0') tempstr++;
   }
   else tempstr++;
   return (strtol (tempstr, NULL, 10));
}
/*****************************************************************************/
static int rfc959_put_file (gftp_request *request, const char *filename, long startsize) {
   char *command, *tempstr, resp;
   int ret;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (filename != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);

   if (request->datafd == NULL && (ret = rfc959_data_connection_new (request)) < 0) return (ret);
   if (startsize > 0) {
      command = g_strdup_printf ("REST %ld\r\n", startsize);
      resp = rfc959_send_command (request, command);
      g_free (command);
      if (resp != '3') {
         fclose (request->datafd);
         request->datafd = NULL;
         return (-2);
      }
   }
   tempstr = g_strconcat ("STOR ", filename, "\r\n", NULL);
   ret = rfc959_send_command (request, tempstr);
   g_free (tempstr);
   if (ret != '1') {
      fclose (request->datafd);
      request->datafd = NULL;
      return (-2);
   }
   if (request->transfer_type == gftp_transfer_active) {
      if ((ret = rfc959_accept_active_connection (request)) < 0) {
         return (ret);
      }
   }
   return (0);
}
/*****************************************************************************/
static int rfc959_end_transfer (gftp_request *request) {
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);
   g_return_val_if_fail (request->datafd != NULL, -2);
   
   fclose (request->datafd);
   request->datafd = NULL;
   return (rfc959_read_response (request) == '2' ? 0 : -2);
}
/*****************************************************************************/
static int rfc959_list_files (gftp_request *request) {
   char ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);
   
   if ((ret = rfc959_data_connection_new (request)) < 0) return (ret);
   if (rfc959_send_command (request, request->resolve_symlinks ? "LIST -L\r\n" : "LIST\r\n") != '1') {
      fclose (request->datafd);
      request->datafd = NULL;
      return (-2);
   }

   if (request->transfer_type == gftp_transfer_active) {
      if ((ret = rfc959_accept_active_connection (request)) < 0) {
         return (ret);
      }
   }
   return (0);
}
/*****************************************************************************/
static int rfc959_get_next_file (gftp_request *request, gftp_file *fle, FILE *fd) {
   char tempstr[255];
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (fle != NULL, -2);

   if (request->last_dir_entry) {
      g_free (request->last_dir_entry);
      request->last_dir_entry = NULL;
   }
   do {
      if (!fgets (tempstr, sizeof (tempstr), fd)) {
         memset (fle, 0, sizeof (gftp_file));
         return (-2);
      }
      if (rfc959_parse_ls (tempstr, fle) != 0) {
         if (request->logging && strncmp (tempstr, "total", 5) != 0) {
            request->logging_function (gftp_logging_misc, request->user_data, 
            	_("Warning: Cannot parse listing %s\n"), tempstr);
         }
         gftp_file_destroy (fle);
         continue;
      }
      else break;
   } while (1);
      
   request->last_dir_entry = g_malloc (strlen (tempstr) + 1);
   strcpy (request->last_dir_entry, tempstr);
   return (strlen (request->last_dir_entry));
}
/*****************************************************************************/
static int rfc959_set_data_type (gftp_request *request, int data_type) {
   char tempstr[9];

   g_return_val_if_fail (request != NULL, -2);

   if (request->sockfd != NULL && request->data_type != data_type) {
      if (data_type == GFTP_TYPE_BINARY) {
         strcpy (tempstr, "TYPE I\r\n");
      }
      else {
         strcpy (tempstr, "TYPE A\r\n");
      }
      if (rfc959_send_command (request, tempstr) != '2') return (-2);
   }
   request->data_type = data_type;
   return (0);
}
/*****************************************************************************/
static long rfc959_get_file_size (gftp_request *request, const char *filename) {
   char *tempstr;
   
   g_return_val_if_fail (request != NULL, 0);
   g_return_val_if_fail (filename != NULL, 0);
   g_return_val_if_fail (request->sockfd != NULL, 0);
   
   tempstr = g_strconcat ("SIZE ", filename, "\r\n", NULL);
   rfc959_send_command (request, tempstr);
   g_free (tempstr);

   if (*request->last_ftp_response != '2') return (0);
   return (strtol (request->last_ftp_response + 4, NULL, 10));
}
/*****************************************************************************/
static int rfc959_parse_ls (const char *lsoutput, gftp_file *fle) {
   int result, cols;
   char *str, *pos;

   g_return_val_if_fail (lsoutput != NULL, -2);
   g_return_val_if_fail (fle != NULL, -2);
   
   str = g_malloc (strlen (lsoutput) + 1);
   strcpy (str, lsoutput);
   memset (fle, 0, sizeof (gftp_file));

   if (str[strlen (str) - 1] == '\n') str[strlen (str) - 1] = '\0';
   if (str[strlen (str) - 1] == '\r') str[strlen (str) - 1] = '\0';
   if (*lsoutput == '+') {
      /* EPLF format */
      result = rfc959_parse_ls_eplf (str, fle);
   }
   else if (isdigit (str[0]) && str[2] == '-') {
      /* WinNT format */
      result = rfc959_parse_ls_nt (str, fle);
   }
   else if (str[1] == ' ' && str[2] == '[') {
      /* Novell format */
      result = rfc959_parse_ls_novell (str, fle);
   }
   else {
     /* UNIX/MacOS format */
      
     /* If there is no space between the attribs and links field, just make one */
     if (strlen (str) > 10) str[10] = ' ';
      
     /* Determine the number of columns */
     cols = 0;
     pos = str;
     while (*pos != '\0') {
        while (*pos != '\0' && *pos != ' ' && *pos != '\t') {
           if (*pos == ':') break;
           pos++;
        }
        cols++;
        if (*pos == ':') {
           cols++;
           break;
        }
        while (*pos != '\0' && (*pos == ' ' || *pos == '\t')) pos++;
     }
     if (cols > 6) result = rfc959_parse_ls_unix (str, cols, fle);
     else result = -2;
   }
   g_free (str);
   return (result);
}
/*****************************************************************************/
static int rfc959_data_connection_new (gftp_request *request) {
   char *pos, *pos1, resp, *command;
   struct sockaddr_in data_addr;
   size_t data_addr_len;
   unsigned int temp[6];
   unsigned char ad[6];
   int i, sock;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);
   
   if ((sock = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
      if (request->logging) {
         request->logging_function (gftp_logging_error, request->user_data, _("Failed to create a socket: %s\n"),
         	g_strerror (errno));
      }
      return (-1);
   }

   data_addr_len = sizeof (data_addr);
   memset (&data_addr, 0, data_addr_len);
   data_addr.sin_family = AF_INET;

   if (request->transfer_type == gftp_transfer_passive) {
      if ((resp = rfc959_send_command(request, "PASV\r\n")) != '2') {
         if(resp != '5') {
            close (sock);
            return (-2);
         }
         else {
            request->transfer_type = gftp_transfer_active;
            return (rfc959_data_connection_new (request));
         }
      }
      else {
         pos = request->last_ftp_response + 4;
         while (!isdigit (*pos) && *pos != '\0') pos++;
         if(*pos == '\0') {
            close (sock);
            return (-2);
         }
         if (sscanf (pos, "%d,%d,%d,%d,%d,%d", &temp[0], &temp[1], &temp[2], 
         	&temp[3], &temp[4], &temp[5]) != 6) {
            close (sock);
            return (-2);
         }
         for (i=0; i<6; i++) ad[i] = (unsigned char) (temp[i] & 0xff);

         memcpy (&data_addr.sin_addr, &ad[0], 4);
         memcpy (&data_addr.sin_port, &ad[4], 2);
         if (connect (sock, (struct sockaddr *) &data_addr, data_addr_len) == -1) {
            if (request->logging) {
               request->logging_function (gftp_logging_error, request->user_data, _("Cannot create a data connection: %s\n"),
               		g_strerror (errno));
            }
            close (sock);
            return (-1);
         }
      }
   }
   else {
      getsockname (request->sockfd_unbuffered, (struct sockaddr *) &data_addr, &data_addr_len);
      data_addr.sin_port = 0;
      if (bind (sock, (struct sockaddr *) &data_addr, data_addr_len) == -1) {
         if (request->logging) {
            request->logging_function (gftp_logging_error, request->user_data, 
            	_("Cannot bind a port: %s\n"), g_strerror (errno));
         }
         close (sock);
         return (-1);
      }
      getsockname (sock, (struct sockaddr *) &data_addr, &data_addr_len);
      if (listen (sock, 1) == -1) {
         if (request->logging) {
            request->logging_function (gftp_logging_error, request->user_data, 
            	_("Cannot listen on port %d: %s\n"), ntohs (data_addr.sin_port), g_strerror (errno));
         }
         close (sock);
         return (-1);
      }
      pos = (char *) &data_addr.sin_addr;
      pos1 = (char *) &data_addr.sin_port;
      command = g_strdup_printf ("PORT %d,%d,%d,%d,%d,%d\r\n",
      		pos[0] & 0xff, pos[1] & 0xff, pos[2] & 0xff, pos[3] & 0xff,
      		pos1[0] & 0xff, pos1[1] & 0xff);
      resp = rfc959_send_command (request, command);
      g_free (command);
      if (resp != '2') {
         close (sock);
         return(-2);
      }
   }
   request->datafd = fdopen (sock, "r+");
   request->datafd_unbuffered = sock;
   return(0);
}
/*****************************************************************************/
static int rfc959_accept_active_connection (gftp_request *request) {
   struct sockaddr_in cli_addr;
   size_t cli_addr_len;
   int infd;
  
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->datafd != NULL, -2);
   g_return_val_if_fail (request->transfer_type == gftp_transfer_active, -2);
   
   cli_addr_len = sizeof (cli_addr);
   if ((infd = accept (request->datafd_unbuffered, (struct sockaddr *) &cli_addr, &cli_addr_len)) == -1) {
      fclose (request->datafd);
      request->datafd = NULL;
      return (-1);
   }
   fclose (request->datafd);
   request->datafd = fdopen (infd, "r+");
   request->datafd_unbuffered = infd;
   return (0);
}
/*****************************************************************************/
static time_t parse_time (char **str) {
   const char *months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", 
   			   "Aug", "Sep", "Oct", "Nov", "Dec"};
   char *startpos, *endpos, *datepos;
   struct tm curtime, tt;
   time_t t;
   int i;

   startpos = *str;
   memset (&tt, 0, sizeof (tt));
   if (isdigit (startpos[0]) && startpos[2] == '-') {
      /* This is how NT will return the date/time */
      /* 07-06-99  12:57PM */
      if((endpos = strchr (startpos, '-')) == NULL) {
         g_free (str);
         return (0);
      }
      tt.tm_mon = strtol (startpos, NULL, 10) - 1;

      startpos = endpos + 1;
      if((endpos = strchr (startpos, '-')) == NULL) {
         g_free (str);
         return (0);
      }
      tt.tm_mday = strtol (startpos, NULL, 10);

      startpos = endpos + 1;
      if((endpos = strchr (startpos, ' ')) == NULL) {
         g_free (str);
         return (0);
      }
      tt.tm_year = strtol (startpos, NULL, 10);

      while (*endpos == ' ') endpos++;

      startpos = endpos + 1;
      if ((endpos = strchr (startpos, ':')) == NULL) {
         g_free (str);
         return (0);
      }
      tt.tm_hour = strtol (startpos, NULL, 10);

      startpos = endpos + 1;
      while ((*endpos != 'A') && (*endpos != 'P')) endpos++;
      if (*endpos == 'P') {
         if(tt.tm_hour == 12) tt.tm_hour = 0;
         else tt.tm_hour += 12;
      }
      tt.tm_min = strtol (startpos, NULL, 10);
   }
   else {
      /* This is how most UNIX, Novell, and MacOS ftp servers send their time */
      /* Jul 06 12:57 */
      t = time (NULL);
      curtime = *localtime (&t);
   
      /* Get the month */
      if ((endpos = strchr (startpos, ' ')) == NULL) {
         return (0);
      }
      for (i=0; i<12; i++) {
         if (strncmp (months[i], startpos, 3) == 0) {
            tt.tm_mon = i;
            break;
         }
      }
   
      /* Skip the blanks till we get to the next entry */
      startpos = endpos + 1;
      while (*startpos == ' ') startpos++;

      /* Get the day */
      if ((endpos = strchr (startpos, ' ')) == NULL) {
         return (0);
      }
      tt.tm_mday = strtol (startpos, NULL, 10);

      /* Skip the blanks till we get to the next entry */
      startpos = endpos + 1;
      while (*startpos == ' ') startpos++;

      if ((datepos = strchr (startpos, ':')) != NULL) {
         /* No year was specified. We will use the current year */
         tt.tm_year = curtime.tm_year;

         /* If the date is in the future, than the year is from last year */
         if ((tt.tm_mon > curtime.tm_mon) || 
      		((tt.tm_mon == curtime.tm_mon) && tt.tm_mday > curtime.tm_mday)) {
           tt.tm_year--;
         }

         /* Get the hours and the minutes */
         tt.tm_hour = strtol (startpos, NULL, 10);
         tt.tm_min = strtol (datepos + 1, NULL, 10);
      }
      else {
         /* Get the year. The hours and minutes will be assumed to be 0 */
         tt.tm_year = strtol (startpos, NULL, 10) - 1900;
      }
   }
   *str = startpos;
   return (mktime (&tt));
}
/*****************************************************************************/
static int rfc959_parse_ls_eplf (char *str, gftp_file *fle) {
   char *startpos;
 
   startpos = str;
   fle->attribs = g_malloc (11);
   strcpy (fle->attribs, "----------");
   while (startpos) {
      startpos++;
      switch (*startpos) {
         case '/' :
            *fle->attribs = 'd';
            break;
         case 's' :
            fle->size = strtol (startpos+1, NULL, 10);
            break;
         case 'm' :
            fle->datetime = strtol (startpos+1, NULL, 10);
            break;
      }
      startpos = strchr (startpos, ',');
   }
   if ((startpos = strchr (str, 9)) == NULL) {
      return (-2);
   }
   fle->file = g_malloc (strlen (startpos));
   strcpy (fle->file, startpos+1);
   fle->user = g_malloc (8);
   strcpy (fle->user, _("unknown"));
   fle->group = g_malloc (8);
   strcpy (fle->group, _("unknown"));
   return (0);
}
/*****************************************************************************/
static int rfc959_parse_ls_unix (char *str, int cols, gftp_file *fle) {
   char *endpos, *startpos;

   startpos = str;
   /* Copy file attributes */
   if ((startpos = copy_token (&fle->attribs, startpos)) == NULL) {
      return (-2);
   }

   if (cols >= 9) {
      /* Skip the number of links */
      startpos = goto_next_token (startpos);

      /* Copy the user that owns this file */
      if ((startpos = copy_token (&fle->user, startpos)) == NULL) {
         return (-2);
      }

      /* Copy the group that owns this file */
      if ((startpos = copy_token (&fle->group, startpos)) == NULL) {
         return (-2);
      }
   }
   else {
      fle->group = g_malloc (8);
      strcpy (fle->group, _("unknown"));
      if (cols == 8) {
         if ((startpos = copy_token (&fle->user, startpos)) == NULL) {
            return (-2);
         }
      }
      else {
         fle->user = g_malloc (8);
         strcpy (fle->user, _("unknown"));
      }
      startpos = goto_next_token (startpos);
   }
   

   if(fle->attribs[0] == 'b' || fle->attribs[0] == 'u' || fle->attribs[0] == 'c') {
      /* This is a block or character device. We will store the major number
         in the high word and the minor number in the low word */

      /* Get the major number */
      if ((endpos = strchr (startpos, ',')) == NULL) {
         return (-2);
      }
      fle->size = strtol (startpos, NULL, 10) << 16;

      startpos = endpos + 1;
      while (*startpos == ' ') startpos++;

      /* Get the minor number */
      if ((endpos = strchr(startpos, ' ')) == NULL) {
         return (-2);
      }
      fle->size |= strtol (startpos, NULL, 10);
   }
   else {
      /* This is a regular file  */
      if ((endpos = strchr (startpos, ' ')) == NULL) {
         return (-2);
      }
      fle->size = strtol (startpos, NULL, 10);
   }

   /* Skip the blanks till we get to the next entry */
   startpos = endpos + 1;
   while (*startpos == ' ') startpos++;

   if ((fle->datetime = parse_time (&startpos)) == 0) {
      return (-2);
   }
   
   /* Skip the blanks till we get to the next entry */
   startpos = goto_next_token (startpos);

   /* Parse the filename. If this file is a symbolic link, remove the -> part */
   if (fle->attribs[0] == 'l' && ((endpos = strstr (startpos, "->")) != NULL)) {
      *(endpos-1) = '\0';
   }
   fle->file = g_malloc (strlen (startpos) + 1);
   strcpy (fle->file, startpos);   
   return (0);
}
/*****************************************************************************/
static int rfc959_parse_ls_nt (char *str, gftp_file *fle) {
   char *startpos;
 
   startpos = str;
   if ((fle->datetime = parse_time (&startpos)) == 0) {
      return (-2);
   };

   /* No such thing on Windoze.. */
   fle->user = g_malloc (8);
   strcpy (fle->user, _("unknown"));
   fle->group = g_malloc (8);
   strcpy (fle->group, _("unknown"));

   startpos = goto_next_token (startpos);
   fle->attribs = g_malloc (11);
   if(startpos[0] == '<') {
      strcpy (fle->attribs, "drwxrwxrwx");
   }
   else {
      strcpy (fle->attribs, "-rw-rw-rw-");
      fle->size = strtol (startpos, NULL, 10);
   }
   startpos = goto_next_token (startpos);
   fle->file = g_malloc (strlen (startpos) + 1);
   strcpy (fle->file, startpos);
   return (0);
}
/*****************************************************************************/
static int rfc959_parse_ls_novell (char *str, gftp_file *fle) {
   char *startpos;
   
   if (str[12] != ' ') return (-2);
   str[12] = '\0';
   fle->attribs = g_malloc (13);
   strcpy (fle->attribs, str);
   startpos = str + 13;

   while ((*startpos == ' ' || *startpos == '\t') && *startpos != '\0') startpos++;
   if ((startpos = copy_token (&fle->user, startpos)) == NULL) {
      return (-2);
   }

   fle->group = g_malloc (8);
   strcpy (fle->group, _("unknown"));
   
   fle->size = strtol (startpos, NULL, 10);

   startpos = goto_next_token (startpos);
   if ((fle->datetime = parse_time (&startpos)) == 0) {
      return (-2);
   }

   startpos = goto_next_token (startpos);
   fle->file = g_malloc (strlen (startpos) + 1);
   strcpy (fle->file, startpos);
   return (0);
}
/*****************************************************************************/
static char *goto_next_token (char *pos) {
   while (*pos != ' ' && *pos != '\t' && *pos != '\0') pos++;
   if (pos == '\0') return (pos);
   while ((*pos == ' ' || *pos == '\t') && *pos != '\0') pos++;
   return (pos);
}
/*****************************************************************************/
static char *copy_token (char **dest, char *source) {
   /* This function is used internally by gftp_parse_ls () */
   char *endpos;

   endpos = source;
   while (*endpos != ' ' && *endpos != '\t' && *endpos != '\0') endpos++;
   if (*endpos == '\0') {
      return (NULL);
   }
   *endpos = '\0';
   *dest = g_malloc (endpos - source + 1);
   strcpy (*dest, source);
   
   /* Skip the blanks till we get to the next entry */
   source = endpos + 1;
   while ((*source == ' ' || *source == '\t') && *source != '\0') source++;
   return (source);
}
/*****************************************************************************/
static int rfc959_send_command (gftp_request *request, const char *command) {
   int ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (command != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);
   
   if (request->logging) { 
      request->logging_function (gftp_logging_send, request->user_data, command);
   }

   do {
      fwrite (command, 1, strlen (command), request->sockfd);
      ret = ferror (request->sockfd);
      if (ret != 0 && errno != EINTR) {
         return (-1);
      }
   } while (ret != 0 && errno == EINTR);
   
   return (rfc959_read_response (request));
}
/*****************************************************************************/
static int rfc959_read_response (gftp_request *request) {
   char tempstr[255], code[4];
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);

   *code = '\0';
   if (request->last_ftp_response) {
      g_free (request->last_ftp_response);
      request->last_ftp_response = NULL;
   }
   
   do {
      if (!fgets (tempstr, sizeof (tempstr), request->sockfd)) {
         break;
      }
      tempstr[strlen (tempstr) - 1] = '\0';
      if (tempstr[strlen (tempstr) - 1] == '\r') tempstr[strlen (tempstr) - 1] = '\0';
      if (isdigit (*tempstr) && isdigit (*(tempstr + 1)) && isdigit (*(tempstr + 2))) {
         strncpy (code, tempstr, 3);
         code[3] = ' ';
      }
      if (request->logging) { 
         request->logging_function (gftp_logging_recv, request->user_data, "%s\n", tempstr);
      }
   } while (strncmp (code, tempstr, 4) != 0);
   
   request->last_ftp_response = g_malloc (strlen (tempstr) + 1);
   strcpy (request->last_ftp_response, tempstr);
   
   return (request->last_ftp_response ? *request->last_ftp_response : -2);
}
/*****************************************************************************/
static int rfc959_chdir (gftp_request *request, const char *directory) {
   char ret, *tempstr, *dir;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (directory != NULL, -2);

   if (strcmp (directory, "..") == 0) {
      ret = rfc959_send_command (request, "CDUP\r\n");
   }
   else {
      tempstr = g_strconcat ("CWD ", directory, "\r\n", NULL);
      ret = rfc959_send_command (request, tempstr);
      g_free (tempstr);
   }
   if (ret != '2') return (-2);
   if (request->directory) {
      g_free (request->directory);
      request->directory = NULL;
   }
      
   /* We successfully changed to the new directory. Now get the new cwd */
   if (rfc959_send_command (request, "PWD\r\n") != '2') return (-2);
      
   tempstr = strchr (request->last_ftp_response, '"');
   if (tempstr != NULL) dir = tempstr + 1;
   else return (-2);
      
   tempstr = strchr (dir, '"');
   if (tempstr != NULL) *tempstr = '\0';
   else return (-2);
      
   request->directory = g_malloc (strlen (dir) + 1);
   strcpy (request->directory, dir);
   return (0);
}
/*****************************************************************************/
static int rfc959_rmdir (gftp_request *request, const char *directory) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (directory != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);

   tempstr = g_strconcat ("RMD ", directory, "\r\n", NULL);
   ret = rfc959_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}
/*****************************************************************************/
static int rfc959_rmfile (gftp_request *request, const char *file) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (file != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);

   tempstr = g_strconcat ("DELE ", file, "\r\n", NULL);
   ret = rfc959_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}
/*****************************************************************************/
static int rfc959_mkdir (gftp_request *request, const char *directory) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (directory != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);

   tempstr = g_strconcat ("MKD ", directory, "\r\n", NULL);
   ret = rfc959_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}
/*****************************************************************************/
static int rfc959_rename (gftp_request *request, const char *oldname, const char *newname) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (oldname != NULL, -2);
   g_return_val_if_fail (newname != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);

   tempstr = g_strconcat ("RNFR ", oldname, "\r\n", NULL);
   ret = rfc959_send_command (request, tempstr);
   g_free (tempstr);
   if (ret != '3') return (-2);

   tempstr = g_strconcat ("RNTO ", newname, "\r\n", NULL);
   ret = rfc959_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}
/*****************************************************************************/
static int rfc959_chmod (gftp_request *request, const char *file, int mode) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (file != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);

   tempstr = g_malloc (strlen (file) + (mode / 10) + 16);
   sprintf (tempstr, "SITE CHMOD %d %s\r\n", mode, file);
   ret = rfc959_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}
/*****************************************************************************/
static int rfc959_site (gftp_request *request, const char *command) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (command != NULL, -2);
   g_return_val_if_fail (request->sockfd != NULL, -2);
   
   tempstr = g_strconcat ("SITE ", command, "\r\n", NULL);
   ret = rfc959_send_command (request, tempstr);
   g_free (tempstr);
   return (ret);
}
/*****************************************************************************/
static char *parse_ftp_proxy_string (gftp_request *request) {
   char *startpos, *endpos, *oldstr, *newstr, *newval, *tempport;

   g_return_val_if_fail (request != NULL, NULL);
   
   newstr = g_malloc (1);
   *newstr = '\0';
   startpos = endpos = request->proxy_config;
   while (*endpos != '\0') {
      tempport = NULL;
      if (*endpos == '%' && tolower (*(endpos + 1)) == 'p') {
         switch (tolower (*(endpos + 2))) {
            case 'u':
               newval = request->proxy_username;
               break;
            case 'p':
               newval = request->proxy_password;
               break;
            case 'h':
               newval = request->proxy_hostname;
               break;
            case 'o':
               tempport = g_strdup_printf ("%d", request->proxy_port);
               newval = tempport;
               break;
            case 'a':
               newval = request->proxy_account;
               break;
            default:
               endpos++;
               continue;
         }
      }
      else if (*endpos == '%' && tolower (*(endpos + 1)) == 'h') {
         switch (tolower (*(endpos + 2))) {
            case 'u':
               newval = request->username;
               break;
            case 'p':
               newval = request->password;
               break;
            case 'h':
               newval = request->hostname;
               break;
            case 'o':
               tempport = g_strdup_printf ("%d", request->port);
               newval = tempport;
               break;
            case 'a':
               newval = request->account;
               break;
            default:
               endpos++;
               continue;
         }
      }
      else if (*endpos == '%' && tolower (*(endpos + 1)) == 'n') {
         *endpos = '\0';
         oldstr = newstr;
         newstr = g_strconcat (oldstr, startpos, "\r\n", NULL);
         g_free (oldstr);
         endpos += 2;
         startpos = endpos;
         continue;
      }
      else {
         endpos++;
         continue;
      }

      *endpos = '\0'; 
      oldstr = newstr;
      if (!newval) newstr = g_strconcat (oldstr, startpos, NULL);
      else newstr = g_strconcat (oldstr, startpos, newval, NULL);
      if (tempport) {
         g_free (tempport);
         tempport = NULL;
      }
      g_free (oldstr);
      endpos += 3;
      startpos = endpos;
   }
   return (newstr);
}
/*****************************************************************************/
