/*
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * Copyright (C) 1999  Rob Crittenden (rcrit@greyoak.com)
 * Copyright (C) 1999  Mark Baysinger (mbaysing@ucsd.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#define PREFS_INTERNAL_ACCESS
#include <stddef.h>
#include <stdio.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "compat/strdup.h"
#include <errno.h>
#include "compat/strerror.h"
#include <ctype.h>
#include "util.h"
#include "eventlog.h"
#include "prefs.h"


static int processDirective(char const * directive, char const * value, unsigned int curLine);
static char const * get_char_conf(char const * directive);
static unsigned int get_int_conf(char const * directive);
static int get_bool_conf(char const * directive);


#define NONE 0

/*    directive             type               defcharval           defintval         actual */
static Bconf_t conf_table[] =
{
    { "filedir",            conf_type_char,    BNETD_FILE_DIR,      NONE,              NULL, 0 }, 
    { "userdir",            conf_type_char,    BNETD_USER_DIR,      NONE,              NULL, 0 },
    { "logfile",            conf_type_char,    BNETD_LOG_FILE,      NONE,              NULL, 0 },
    { "loglevels",          conf_type_char,    BNETD_LOG_LEVELS,    NONE,              NULL, 0 },
    { "defacct",            conf_type_char,    BNETD_TEMPLATE_FILE, NONE,              NULL, 0 },
    { "motdfile",           conf_type_char,    BNETD_MOTD_FILE,     NONE,              NULL, 0 },
    { "newsfile",           conf_type_char,    BNETD_NEWS_DIR,      NONE,              NULL, 0 },
    { "channelfile",        conf_type_char,    BNETD_CHANNEL_FILE,  NONE,              NULL, 0 },
    { "pidfile",            conf_type_char,    BNETD_PID_FILE,      NONE,              NULL, 0 },
    { "adfile",             conf_type_char,    BNETD_AD_FILE,       NONE,              NULL, 0 },
    { "gameport",           conf_type_int,     NULL,                BNETD_GAME_PORT,   NULL, 0 },
    { "testport",           conf_type_int,     NULL,                BNETD_TEST_PORT,   NULL, 0 },
    { "usersync",           conf_type_int,     NULL,                BNETD_USERSYNC,    NULL, 0 },
    { "userflush",          conf_type_int,     NULL,                BNETD_USERFLUSH,   NULL, 0 },
    { "track",              conf_type_int,     NULL,                BNETD_TRACK_TIME,  NULL, 0 },
    { "location",           conf_type_char,    "",                  NONE,              NULL, 0 },
    { "description",        conf_type_char,    "",                  NONE,              NULL, 0 },
    { "url",                conf_type_char,    "",                  NONE,              NULL, 0 },
    { "contact_name",       conf_type_char,    "",                  NONE,              NULL, 0 },
    { "contact_email",      conf_type_char,    "",                  NONE,              NULL, 0 },
    { "latency",            conf_type_int,     NULL,                BNETD_LATENCY,     NULL, 0 },
    { "shutdown_delay",     conf_type_int,     NULL,                BNETD_SHUTDELAY,   NULL, 0 },
    { "shutdown_decr",      conf_type_int,     NULL,                BNETD_SHUTDECR,    NULL, 0 },
    { "new_accounts",       conf_type_bool,    NULL,                1,                 NULL, 0 },
    { "kick_old_login",     conf_type_bool,    NULL,                1,                 NULL, 0 },
    { "ask_new_channel",    conf_type_bool,    NULL,                1,                 NULL, 0 },
    { "hide_pass_games",    conf_type_bool,    NULL,                1,                 NULL, 0 },
    { "hide_temp_channels", conf_type_bool,    NULL,                0,                 NULL, 0 },
    { "gametrans",          conf_type_char,    "",                  NONE,              NULL, 0 },
    { "extra_commands",     conf_type_bool,    NULL,                0,                 NULL, 0 },
    { "reportdir",          conf_type_char,    BNETD_REPORT_DIR,    NONE,              NULL, 0 },
    { "report_all_games",   conf_type_bool,    NULL,                0,                 NULL, 0 },
    { "iconfile",           conf_type_char,    BNETD_ICON_FILE,     NONE,              NULL, 0 },
    { "tosfile",            conf_type_char,    BNETD_TOS_FILE,      NONE,              NULL, 0 },
    { "mpqfile",            conf_type_char,    BNETD_MPQ_FILE,      NONE,              NULL, 0 },
    { "mpqversion",         conf_type_char,    NULL,                BNETD_MPQ_VERSION, NULL, 0 },
    { "allow_autoupdate",   conf_type_bool,    NULL,                0,                 NULL, 0 },
    { "trackaddrs",         conf_type_char,    BNETD_TRACK_ADDRS,   NONE,              NULL, 0 },
    { "servaddrs",          conf_type_char,    BNETD_SERV_ADDRS,    NONE,              NULL, 0 },

    { NULL,                 conf_type_none,    NULL,                NONE,              NULL, 0 }
};

char const * preffile=NULL;


static int processDirective(char const * directive, char const * value, unsigned int curLine)
{
    unsigned int i;
    
    if (!directive || !value)
    {
	eventlog(eventlog_level_error,"processDirective","got NULL input(s)");
	return -1;
    }
    
    for (i=0; conf_table[i].directive; i++)
        if (strcasecmp(conf_table[i].directive,directive) == 0)
	{
            switch (conf_table[i].type)
            {
	    case conf_type_char:
		{
		    char const * temp;
		    
		    if (!(temp = strdup(value)))
		    {
			eventlog(eventlog_level_error,"processDirective","could not allocate memory for value");
			break;
		    }
		    if (conf_table[i].charval)
			pfree((void *)conf_table[i].charval,strlen(conf_table[i].charval)+1); /* avoid warning */
		    conf_table[i].charval = temp;
		}
		break;
		
	    case conf_type_int:
		{
		    unsigned int temp;
		    
		    if (str_to_uint(value,&temp)<0)
			eventlog(eventlog_level_error,"processDirective","invalid integer value \"%s\" for element \"%s\" at line %u",value,directive,curLine);
		    else
                	conf_table[i].intval = temp;
		}
		break;
		
	    case conf_type_bool:
		if (strcasecmp(value,"true")==0 ||
		    strcasecmp(value,"yes")==0 ||
		    strcasecmp(value,"on")==0 ||
		    strcmp(value,"1")==0)
		    conf_table[i].intval = 1;
		else if (strcasecmp(value,"false")==0 ||
			 strcasecmp(value,"no")==0 ||
			 strcasecmp(value,"off")==0 ||
			 strcmp(value,"0")==0)
		    conf_table[i].intval = 0;
		else
		    eventlog(eventlog_level_error,"processDirective","invalid boolean value for element \"%s\" at line %u",directive,curLine);
		break;
		
	    default:
		eventlog(eventlog_level_error,"processDirective","invalid type %d in table",(int)conf_table[i].type);
	    }
	    return 0;
	}
    
    eventlog(eventlog_level_error,"processDirective","unknown element \"%s\" at line %u",directive,curLine);
    return -1;
}


static char const * get_char_conf(char const * directive)
{
    unsigned int i;
    
    for (i=0; conf_table[i].directive; i++)
	if (conf_table[i].type==conf_type_char && strcasecmp(conf_table[i].directive,directive)==0)
	    return conf_table[i].charval;
    
    return NULL;
}


static unsigned int get_int_conf(char const * directive)
{
    unsigned int i;
    
    for (i=0; conf_table[i].directive; i++)
	if (conf_table[i].type==conf_type_int && strcasecmp(conf_table[i].directive,directive)==0)
	    return conf_table[i].intval;
    
    return 0;
}


static int get_bool_conf(char const * directive)
{
    unsigned int i;
    
    for (i=0; conf_table[i].directive; i++)
	if (conf_table[i].type==conf_type_bool && strcasecmp(conf_table[i].directive,directive)==0)
	    return conf_table[i].intval;
    
    return 0;
}


extern int prefs_load(char const * filename)
{
    /* restore defaults */
    {
	unsigned int i;
	
	for (i=0; conf_table[i].directive; i++)
	    switch (conf_table[i].type)
	    {
	    case conf_type_int:
	    case conf_type_bool:
		conf_table[i].intval = conf_table[i].defintval;
		break;
		
	    case conf_type_char:
		if (conf_table[i].charval)
		    pfree((void *)conf_table[i].charval,strlen(conf_table[i].charval)+1); /* avoid warning */
		if (!conf_table[i].defcharval)
		    conf_table[i].charval = NULL;
		else
		    if (!(conf_table[i].charval = strdup(conf_table[i].defcharval)))
		    {
			eventlog(eventlog_level_error,"prefs_load","could not allocate memory for conf_table[i].charval");
			return -1;
		    }
		break;
		
	    default:
		eventlog(eventlog_level_error,"prefs_load","invalid type %d in table",(int)conf_table[i].type);
		return -1;
	    }
    }
    
    /* load file */
    if (filename)
    {
	FILE *       fp;
	char *       buff;
	char *       cp;
	char *       temp;
	unsigned int currline;
	unsigned int j;
	char const * directive;
	char const * value;
	char *       rawvalue;
	
        if (!(fp = fopen(filename,"r")))
        {
            eventlog(eventlog_level_error,"prefs_load","could not open file \"%s\" for reading (fopen: %s)",filename,strerror(errno));
            return -1;
        }
	
	/* Read the configuration file */
	for (currline=1; (buff = file_get_line(fp)); currline++)
	{
	    cp = buff;
	    
            while (*cp=='\t' || *cp==' ') cp++;
	    if (*cp=='\0' || *cp=='#')
	    {
		pfree(buff,strlen(buff)+1);
		continue;
	    }
	    temp = cp;
	    while (*cp!='\t' && *cp!=' ' && *cp!='\0') cp++;
	    if (*cp!='\0')
	    {
		*cp = '\0';
		cp++;
	    }
	    if (!(directive = strdup(temp)))
	    {
		eventlog(eventlog_level_error,"prefs_load","could not allocate memory for directive");
		pfree(buff,strlen(buff)+1);
		continue;
	    }
            while (*cp=='\t' || *cp==' ') cp++;
	    if (*cp!='=')
	    {
		eventlog(eventlog_level_error,"prefs_load","missing = on line %u",currline);
		pfree((void *)directive,strlen(directive)+1); /* avoid warning */
		pfree(buff,strlen(buff)+1);
		continue;
	    }
	    cp++;
	    while (*cp=='\t' || *cp==' ') cp++;
	    if (*cp=='\0')
	    {
		eventlog(eventlog_level_error,"prefs_load","missing value after = on line %u",currline);
		pfree((void *)directive,strlen(directive)+1); /* avoid warning */
		pfree(buff,strlen(buff)+1);
		continue;
	    }
	    if (!(rawvalue = strdup(cp)))
	    {
		eventlog(eventlog_level_error,"prefs_load","could not allocate memory for rawvalue");
		pfree((void *)directive,strlen(directive)+1); /* avoid warning */
		pfree(buff,strlen(buff)+1);
		continue;
	    }
	    
	    if (rawvalue[0]=='"')
	    {
		char prev;
		
		for (j=1,prev='\0'; rawvalue[j]!='\0'; j++)
		{
		    switch (rawvalue[j])
		    {
		    case '"':
			if (prev!='\\')
			    break;
			prev = '"';
			continue;
		    case '\\':
			if (prev=='\\')
			    prev = '\0';
			else
			    prev = '\\';
			continue;
		    default:
			prev = rawvalue[j];
			continue;
		    }
		    break;
		}
		if (rawvalue[j]!='"')
		{
		    eventlog(eventlog_level_error,"processDirective","missing end quote for value of element \"%s\" on line %u",directive,currline);
		    pfree(rawvalue,strlen(rawvalue)+1);
		    pfree((void *)directive,strlen(directive)+1); /* avoid warning */
		    pfree(buff,strlen(buff)+1);
		    continue;
		}
		rawvalue[j] = '\0';
		if (rawvalue[j+1]!='\0' && rawvalue[j+1]!='#')
		{
		    eventlog(eventlog_level_error,"processDirective","extra characters after the value for element \"%s\" on line %u",directive,currline);
		    pfree(rawvalue,strlen(rawvalue)+1);
		    pfree((void *)directive,strlen(directive)+1); /* avoid warning */
		    pfree(buff,strlen(buff)+1);
		    continue;
		}
		value = &rawvalue[1];
            }
	    else
	    {
		unsigned int k;
		
		for (j=0; rawvalue[j]!='\0' && rawvalue[j]!=' ' && rawvalue[j]!='\t'; j++);
		k = j;
		while (rawvalue[k]==' ' || rawvalue[k]=='\t') k++;
		if (rawvalue[k]!='\0' && rawvalue[k]!='#')
		{
		    eventlog(eventlog_level_error,"processDirective","extra characters after the value for element \"%s\" on line %u (%s)",directive,currline,&rawvalue[k]);
		    pfree(rawvalue,strlen(rawvalue)+1);
		    pfree((void *)directive,strlen(directive)+1); /* avoid warning */
		    pfree(buff,strlen(buff)+1);
		    continue;
		}
		rawvalue[j] = '\0';
		value = rawvalue;
	    }
            
	    processDirective(directive,value,currline);
	    
	    pfree(rawvalue,strlen(rawvalue)+1);
	    pfree((void *)directive,strlen(directive)+1); /* avoid warning */
	    pfree(buff,strlen(buff)+1);
	}
    }
    
    return 0;
}


extern void prefs_unload(void)
{
    unsigned int i;
    
    for (i=0; conf_table[i].directive; i++)
	switch (conf_table[i].type)
	{
	case conf_type_int:
	case conf_type_bool:
	    break;
	    
	case conf_type_char:
	    if (conf_table[i].charval)
	    {
		pfree((void *)conf_table[i].charval,strlen(conf_table[i].charval)+1); /* avoid warning */
		conf_table[i].charval = NULL;
	    }
	    break;
	    
	default:
	    eventlog(eventlog_level_error,"prefs_unload","invalid type %d in table",(int)conf_table[i].type);
	    break;
	}
}


extern char const * prefs_get_userdir(void)
{
    return get_char_conf("userdir");
}


extern char const * prefs_get_filedir(void)
{
    return get_char_conf("filedir");
}


extern char const * prefs_get_logfile(void)
{
    return get_char_conf("logfile");
}


extern char const * prefs_get_loglevels(void)
{
    return get_char_conf("loglevels");
}


extern char const * prefs_get_defacct(void)
{
    return get_char_conf("defacct");
}


extern char const * prefs_get_motdfile(void)
{
    return get_char_conf("motdfile");
}


extern char const * prefs_get_newsfile(void)
{
    return get_char_conf("newsfile");
}


extern char const * prefs_get_adfile(void)
{
    return get_char_conf("adfile");
}


extern unsigned int prefs_get_gameport(void)
{
    return get_int_conf("gameport");
}


extern unsigned int prefs_get_testport(void)
{
    return get_int_conf("testport");
}


extern unsigned int prefs_get_user_sync_timer(void)
{
    return get_int_conf("usersync");
}


extern unsigned int prefs_get_user_flush_timer(void)
{
    return get_int_conf("userflush");
}


extern unsigned int prefs_get_track(void)
{
    return get_int_conf("track");
}


extern char const * prefs_get_location(void)
{
    return get_char_conf("location");
}


extern char const * prefs_get_description(void)
{
    return get_char_conf("description");
}


extern char const * prefs_get_url(void)
{
    return get_char_conf("url");
}


extern char const * prefs_get_contact_name(void)
{
    return get_char_conf("contact_name");
}


extern char const * prefs_get_contact_email(void)
{
    return get_char_conf("contact_email");
}


extern unsigned int prefs_get_latency(void)
{
    return get_int_conf("latency");
}


extern unsigned int prefs_get_shutdown_delay(void)
{
    return get_int_conf("shutdown_delay");
}


extern unsigned int prefs_get_shutdown_decr(void)
{
    return get_int_conf("shutdown_decr");
}


extern unsigned int prefs_get_allow_new_accounts(void)
{
    return get_bool_conf("new_accounts");
}


extern unsigned int prefs_get_kick_old_login(void)
{
    return get_bool_conf("kick_old_login");
}


extern char const * prefs_get_channelfile(void)
{
    return get_char_conf("channelfile");
}


extern unsigned int prefs_get_ask_new_channel(void)
{
    return get_bool_conf("ask_new_channel");
}


extern unsigned int prefs_get_hide_pass_games(void)
{
    return get_bool_conf("hide_pass_games");
}


extern unsigned int prefs_get_hide_temp_channels(void)
{
    return get_bool_conf("hide_temp_channels");
}


extern char const * prefs_get_gametrans(void)
{
    return get_char_conf("gametrans");
}


extern unsigned int prefs_get_extra_commands(void)
{
    return get_bool_conf("extra_commands");
}


extern char const * prefs_get_reportdir(void)
{
    return get_char_conf("reportdir");
}


extern unsigned int prefs_get_report_all_games(void)
{
    return get_bool_conf("report_all_games");
}


extern char const * prefs_get_pidfile(void)
{
    return get_char_conf("pidfile");
}


extern char const * prefs_get_iconfile(void)
{
    return get_char_conf("iconfile");
}


extern char const * prefs_get_tosfile(void)
{
    return get_char_conf("tosfile");
}


extern unsigned int prefs_get_allow_autoupdate(void)
{
    return get_bool_conf("allow_autoupdate");
}


extern char const * prefs_get_mpqfile(void)
{
    return get_char_conf("mpqfile");
}


extern unsigned int prefs_get_mpqversion(void)
{
    return get_int_conf("mpqversion");
}


extern char const * prefs_get_trackserv_addrs(void)
{
    return get_char_conf("trackaddrs");
}


extern char const * prefs_get_bnetdserv_addrs(void)
{
    return get_char_conf("servaddrs");
}

