/* $Id: ethtool.c,v 1.1.1.1 1998/12/19 15:07:09 davem Exp $
 * ethtool.c: SparcLinux ethernet device configuration tool.
 *
 * Copyright (C) 1998 David S. Miller (davem@dm.cobaltmicro.com)
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <net/if.h>

#include "ethtool.h"

/* Syntax:
 *
 *	ethtool DEVNAME
 *	ethtool -s DEVNAME [ speed {10,100,1000} ] \
 *		[ duplex {half,full} ] \
 *		[ port {tp,aui,mii,fibre} ] \
 *		[ autoneg {on,off} ] \
 *		[ phyad %d ] \
 *		[ xcvr {internal,external} ]
 */

static char *devname = NULL;
static int cmd = SPARC_ETH_GSET;
static int speed_wanted = -1;
static int duplex_wanted = -1;
static int port_wanted = -1;
static int autoneg_wanted = -1;
static int phyad_wanted = -1;
static int xcvr_wanted = -1;

static void show_usage(int badarg)
{
	fprintf(stderr, "Usage:\n");
	fprintf(stderr, "	ethtool DEVNAME\n");
	fprintf(stderr, "	ethtool -s DEVNAME \\\n");
	fprintf(stderr, "		[ speed {10,100,1000} ] \\\n");
	fprintf(stderr, "		[ duplex {half,full} ]	\\\n");
	fprintf(stderr, "		[ port {tp,aui,mii,fibre} ] \\\n");
	fprintf(stderr, "		[ autoneg {on,off} ] \\\n");
	fprintf(stderr, "		[ phyad %%d ] \\\n");
	fprintf(stderr, "		[ xcvr {internal,external} ] \\\n");
	exit(badarg);
}

static int parse_cmdline(int argc, char **argp)
{
	int i;

	for(i = 1; i < argc; i++) {
		switch(i) {
		case 1:
			if(!strcmp(argp[i], "-s"))
				cmd = SPARC_ETH_SSET;
			else if(!strcmp(argp[i], "-h"))
				show_usage(0);
			else
				devname = argp[i];
			break;
		case 2:
			if(cmd == SPARC_ETH_SSET) {
				devname = argp[i];
				break;
			}
			/* fallthrough */
		default:
			if(cmd == SPARC_ETH_GSET)
				show_usage(1);
			if(!strcmp(argp[i], "speed")) {
				i += 1;
				if(i >= argc)
					show_usage(1);
				if(!strcmp(argp[i], "10"))
					speed_wanted = SPEED_10;
				else if(!strcmp(argp[i], "100"))
					speed_wanted = SPEED_100;
				else if(!strcmp(argp[i], "1000"))
					speed_wanted = SPEED_1000;
				else
					show_usage(1);
				break;
			} else if(!strcmp(argp[i], "duplex")) {
				i += 1;
				if(i >= argc)
					show_usage(1);
				if(!strcmp(argp[i], "half"))
					duplex_wanted = DUPLEX_HALF;
				else if(!strcmp(argp[i], "full"))
					duplex_wanted = DUPLEX_FULL;
				else
					show_usage(1);
				break;
			} else if(!strcmp(argp[i], "port")) {
				i += 1;
				if(i >= argc)
					show_usage(1);
				if(!strcmp(argp[i], "tp"))
					port_wanted = PORT_TP;
				else if(!strcmp(argp[i], "aui"))
					port_wanted = PORT_AUI;
				else if(!strcmp(argp[i], "mii"))
					port_wanted = PORT_MII;
				else if(!strcmp(argp[i], "fibre"))
					port_wanted = PORT_FIBRE;
				else
					show_usage(1);
				break;
			} else if(!strcmp(argp[i], "autoneg")) {
				i += 1;
				if(i >= argc)
					show_usage(1);
				if(!strcmp(argp[i], "on"))
					autoneg_wanted = AUTONEG_ENABLE;
				else if(!strcmp(argp[i], "off"))
					autoneg_wanted = AUTONEG_DISABLE;
				else
					show_usage(1);
				break;
			} else if(!strcmp(argp[i], "phyad")) {
				i += 1;
				if(i >= argc)
					show_usage(1);
				phyad_wanted = strtol(argp[i], NULL, 10);
				if(phyad_wanted < 0)
					show_usage(1);
				break;
			} else if(!strcmp(argp[i], "xcvr")) {
				i += 1;
				if(i >= argc)
					show_usage(1);
				if(!strcmp(argp[i], "internal"))
					xcvr_wanted = XCVR_INTERNAL;
				else if(!strcmp(argp[i], "external"))
					xcvr_wanted = XCVR_EXTERNAL;
				else
					show_usage(1);
				break;
			}
			show_usage(1);
		}
	}
	if(devname == NULL)
		show_usage(1);
}

void dump_supported(struct ethtool_cmd *ep)
{
	u_int32_t mask = ep->supported;
	int cnt;

	fprintf(stdout, "	Supported ports: [ ");
	if(mask & SUPPORTED_TP)
		fprintf(stdout, "TP ");
	if(mask & SUPPORTED_AUI)
		fprintf(stdout, "AUI ");
	if(mask & SUPPORTED_MII)
		fprintf(stdout, "MII ");
	if(mask & SUPPORTED_FIBRE)
		fprintf(stdout, "FIBRE ");
	fprintf(stdout, "]\n");

	fprintf(stdout, "	Supported link modes:   ");
	cnt = 0;
	if(mask & SUPPORTED_10baseT_Half) {
		cnt++; fprintf(stdout, "10baseT/Half ");
	}
	if(mask & SUPPORTED_10baseT_Full) {
		cnt++; fprintf(stdout, "10baseT/Full ");
	}
	if(cnt != 0) {
		fprintf(stdout, "\n");
		fprintf(stdout, "	                        ");
		cnt = 0;
	}
	if(mask & SUPPORTED_100baseT_Half) {
		cnt++; fprintf(stdout, "100baseT/Half ");
	}
	if(mask & SUPPORTED_100baseT_Full) {
		cnt++; fprintf(stdout, "100baseT/Full ");
	}
	if(cnt != 0) {
		fprintf(stdout, "\n");
		fprintf(stdout, "	                        ");
		cnt = 0;
	}
	if(mask & SUPPORTED_1000baseT_Half) {
		cnt++; fprintf(stdout, "1000baseT/Half ");
	}
	if(mask & SUPPORTED_1000baseT_Full) {
		cnt++; fprintf(stdout, "1000baseT/Full ");
	}
	fprintf(stdout, "\n");

	fprintf(stdout, "	Supports auto-negotiation? ");
	if(mask & SUPPORTED_Autoneg)
		fprintf(stdout, "Yes\n");
	else
		fprintf(stdout, "No\n");
}

int dump_ecmd(struct ethtool_cmd *ep)
{
	fprintf(stdout, "Settings for %s:\n\n", devname);
	dump_supported(ep);

	fprintf(stdout, "	Speed: ");
	switch(ep->speed) {
	case SPEED_10:
		fprintf(stdout, "10Mb/s\n");
		break;
	case SPEED_100:
		fprintf(stdout, "100Mb/s\n");
		break;
	case SPEED_1000:
		fprintf(stdout, "1000Mb/s\n");
		break;
	default:
		fprintf(stdout, "Unknown!\n");
		break;
	};

	fprintf(stdout, "	Duplex: ");
	switch(ep->duplex) {
	case DUPLEX_HALF:
		fprintf(stdout, "Half\n");
		break;
	case DUPLEX_FULL:
		fprintf(stdout, "Full\n");
		break;
	default:
		fprintf(stdout, "Unknown!\n");
		break;
	};

	fprintf(stdout, "	Port: ");
	switch(ep->port) {
	case PORT_TP:
		fprintf(stdout, "Twisted Pair\n");
		break;
	case PORT_AUI:
		fprintf(stdout, "AUI\n");
		break;
	case PORT_MII:
		fprintf(stdout, "MII\n");
		break;
	case PORT_FIBRE:
		fprintf(stdout, "FIBRE\n");
		break;
	default:
		fprintf(stdout, "Unknown!\n");
		break;
	};

	fprintf(stdout, "	PHYAD: %d\n", ep->phy_address);
	fprintf(stdout, "	Transceiver: ");
	switch(ep->transceiver) {
	case XCVR_INTERNAL:
		fprintf(stdout, "internal\n");
		break;
	case XCVR_EXTERNAL:
		fprintf(stdout, "externel\n");
		break;
	default:
		fprintf(stdout, "Unknown!\n");
		break;
	};

	fprintf(stdout, "	Auto-negotiation: %s\n",
		(ep->autoneg == AUTONEG_DISABLE) ?
		"off" : "on");
	return 0;
}

int doit(void)
{
	struct ethtool_cmd ecmd;
	struct ifreq ifr;
	int fd, err;

	/* Setup our control structures. */
	memset(&ecmd, 0, sizeof(ecmd));
	memset(&ifr, 0, sizeof(ifr));
	strcpy(&ifr.ifr_name[0], devname);
	ifr.ifr_data = (caddr_t) &ecmd;
	ecmd.cmd = SPARC_ETH_GSET;

	/* Open control socket. */
	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if(fd < 0) {
		perror("Cannot get control socket");
		return 70;
	}

	/* Get current settings. */
	err = ioctl(fd, SIOCETHTOOL, &ifr);
	if(err < 0) {
		perror("Cannot get current device settings");
		return 71;
	}
	if(cmd == SPARC_ETH_GSET)
		return dump_ecmd(&ecmd);
	else if(cmd == SPARC_ETH_SSET) {
		/* Change everything the user specified. */
		if(speed_wanted != -1)
			ecmd.speed = speed_wanted;
		if(duplex_wanted != -1)
			ecmd.duplex = duplex_wanted;
		if(port_wanted != -1)
			ecmd.port = port_wanted;
		if(autoneg_wanted != -1)
			ecmd.autoneg = autoneg_wanted;
		if(phyad_wanted != -1)
			ecmd.phy_address = phyad_wanted;
		if(xcvr_wanted != -1)
			ecmd.transceiver = xcvr_wanted;

		/* Try to perform the update. */
		ecmd.cmd = SPARC_ETH_SSET;
		err = ioctl(fd, SIOCETHTOOL, &ifr);
		if(err < 0) {
			perror("Cannot update new settings");
			return 72;
		}
		return 0;
	}

	/* XXX */
	return 69;
}

int main(int argc, char **argp, char **envp)
{
	parse_cmdline(argc, argp);
	exit(doit());
}
