/* reconstructs the
incoming file(s) or outputs the EMWIN QBT block to stdout.

Copyright (C) 1999 Antonio Querubin

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

--

Compiles/runs on:  Solaris, BSDI, IRIX, Linux, OS X, OS/2, Windows
95/98/NT.

The GCC compiler is required to build this on UNIX and OS/2.  To build
Microsoft Windows binaries requires either GCC-Mingw32 or LCC-Win32.

To compile for most UNIXs:  gcc -O2 emwinmcr.c -o emwinmcr

To compile for SunOS/Solaris:  gcc -O2 emwinmcr.c -lsocket -lnsl -o emwinmcr

To compile for Mac OS X:  cc -O2 emwinmcr.c -o emwinmcr

To compile for OS/2 using emx-gcc:  gcc -O2 emwinmcr.c -lsocket
Depending on how you link the binary, it may require the EMX run-time DLL
for distribution.  It can be obtained from:
ftp://ftp.leo.org/pub/comp/os/os2/leo/gnu/emx+gcc/emxrt.zip

To compile for Windows using gcc-mingw32: 
gcc -O2 emwinmcr.c -lwsock32 -o emwinmcr.exe

To compile for Windows using lcc-win32, the wsock32.lib file must be added
as an additional library in the linker configuration.

Please send patches and/or bug reports to:  tony@dod.hawaii.gov

Contributions/bug fixes by:
  Julian Cowley
  Maitland Bottoms (AA4HS)

Revision history:
 6 Sep 1999, v0.1:  First alpha release for Solaris/BSDI. 
18 Sep 1999, v0.2:  Added Windows 95/98/NT support.
                    Added file reconstruction.
19 Sep 1999, v0.3:  Added Linux support (from Julian Cowley).
                    Removed dependency on drive C: (Windows version).
21 Sep 1999, v0.4:  Added better error checking for bogus network packets.
                    Added command line options.
23 Sep 1999, v0.5:  Fixed up socket error reporting.
                    Added more error checking of the incoming packet.
30 Sep 1999, v0.6:  More command line options and code cleanup.
                    Added Y2K rollover fix.
                    Restored signal handlers for Windows.
                    Added SIGHUP handler for UNIX.
 1 Oct 1999, v0.7:  Added unix daemon mode and syslog support (from AA4HS).
10 Oct 1999, v0.8:  Added OS/2 support and a better move completed file
                    routine.
11 Oct 1999, v0.9:  Now truncates trailing nulls in text files.
13 Oct 1999, v0.10: Changed the move routine to a copy and delete for
                    Windows and OS/2 so that it works across different
                    drives.
14 Oct 1999, v0.11: Fixed a bug in the stderr redirection for OS/2.  OS/2
                    wants to see NUL instead of NUL:
15 Oct 1999, v0.12: Tests for BSD-type SO_REUSEPORT instead of specific OS
                    at compile time.
19 Oct 1999, v0.13: Added call to unzip for ZIS files (output filename is
                    in lower-case).
                    Set umask to plug a security hole.
                    Added directory name error checking.
20 Oct 1999, v0.14: Fixed getcwd/chdir problem in OS/2.  
                    Added option to specify unzip path.
26 Oct 1999, v0.14amb: Split multicast from file handling
                    so that code can be shared with serial port listener.

Known bugs:

Multiple concurrent instances of the OS/2 version can't bind to the
address/port.  However, most users will only run a single copy anyway.

Windows Ctrl-C handler still isn't correct but showing some stats
is better than nothing.

--

Uniquely defined macros for various compilers and specific OS targets:
bsdi            gcc / BSDI
linux           gcc / Linux
__APPLE__       cc / OS X
__EMX__         emx-gcc / OS/2
WIN32           gcc-mingw32 or lcc-win32 / Microsoft Windows 95/98/NT

*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <ctype.h>
#include <sys/stat.h>
#include <errno.h>

#ifdef WIN32
#include <direct.h>
#include <winsock.h>
#else
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/param.h>
#ifndef __EMX__
#include <syslog.h>
#endif
#endif

#include "qbt.h"
extern unsigned char *tmpdir;

/* Parses and checks an EMWIN QBT block.
   Return values: -1 if any check fails, 0 otherwise.
   The attr structure is filled with the parsed block values if all
   checks are successful.  Otherwise the content of the structure is
   undefined. */
int checkblock(const struct QBT *blockp, struct PKTATTR *attr) {
  unsigned char tempbuf[sizeof(blockp->header) + 1];
  int i;
  unsigned char filename[13];
  unsigned long ourchecksum;
  time_t filetime;
  struct tm timest;
  unsigned char apm[3];

  /* The header isn't null-terminated so we have to add a NULL before
     parsing with sscanf. */
  memcpy(tempbuf,blockp->header,sizeof(blockp->header));
  tempbuf[sizeof(tempbuf)-1] = '\0';
  if (sscanf(tempbuf,"/PF%12c/PN %u /PT %u /CS %lu /FD%20c",
             filename, &(attr->blocknum), &(attr->totalblocks),
             &(attr->checksum), attr->datestamp) != 5) {
    fprintf(stderr,"Bad header.\n");
    return (-1);
  }
  filename[sizeof(filename)-1] = '\0';
  attr->datestamp[sizeof(attr->datestamp)-1] = '\0';

  /* Do a sanity check of the header. */

  /* Strip any trailing white-space from the filename. */
  if (sscanf(filename,"%s",attr->filename) != 1) {
    fprintf(stderr,"Missing filename.\n");
    return (-1);
  }
  for (i=0; i<strlen(attr->filename); i++) {
    if (!(isalnum(attr->filename[i]) || attr->filename[i] == '.')) {
      fprintf(stderr,"Invalid filename.\n");
      return (-1);
    }
  }

  /* Range check the block numbers. */
  if (attr->blocknum < 1 || attr->blocknum > attr->totalblocks ||
      attr->totalblocks > MAXBLOCKS) {
    fprintf(stderr,"Bad block number(s).\n");
    return (-1);
  }

  /* Validate the datestamp by trying to convert it to a time value. */
  memset(&timest,'\0',sizeof(timest));
  if (sscanf(attr->datestamp,"%u/%u/%u %u:%u:%u %2s",
             &timest.tm_mon,&timest.tm_mday,&timest.tm_year,
             &timest.tm_hour,&timest.tm_min,&timest.tm_sec,apm) != 7) {
    fprintf(stderr,"Bad date string.\n");
    return (-1);
  }
  if (strcmp(apm,"AM") && strcmp(apm,"PM")) {
    fprintf(stderr,"Bad date string.\n");
    return (-1);
  }
  /* Setup the time structure. */
  timest.tm_mon--;
  /* Adjust the hour for AM/PM. */
  if (timest.tm_hour == 12 && apm[0] == 'A') timest.tm_hour = 0;
  else if (timest.tm_hour < 12 && apm[0] == 'P') timest.tm_hour += 12;
  /* Y2K rollover patch. */
  if (timest.tm_year <= 38) timest.tm_year += 100;
  if ((filetime=mktime(&timest)) == -1) {
    fprintf(stderr,"Bad date/time.\n");
    return (-1);
  }

  fprintf(stderr,"%s, # %u of %u, %s",
          attr->filename, attr->blocknum, attr->totalblocks,
          ctime(&filetime));

  /* Checksum the data portion. */
  for (i=0, ourchecksum=0;
       i<sizeof(blockp->data);
       ourchecksum += (unsigned int) blockp->data[i++]);
  if (ourchecksum != attr->checksum) {
    fprintf(stderr,"Checksum = %lu, should be %lu.\n",
            ourchecksum, attr->checksum);
    return (-1);
  }

  attr->data = (unsigned char *)blockp->data;
  return (0);
} /* checkblock() */


/* Save the EMWIN block.
   Return values:
     -1=failure
      0=block saved
      1=file completed by this block
   savedfilepath is set to point at the full path to the saved output
   file. */
int saveblock(const struct PKTATTR *pktp, unsigned char *savedfilepath) {
  FILE *trackfp, *datafp = NULL;
  int i, n, newfile, datasize;
  unsigned char trackrec[sizeof(pktp->datestamp)+MAXBLOCKS],
                trackpath[MAXPATHLEN], datapath[MAXPATHLEN];

  /* Construct the full file paths. */
#if defined(WIN32) || defined(__EMX__)
  sprintf(datapath, "%s" PATHSEPARATOR "%s", TMPDIR, pktp->filename);
  sprintf(trackpath, "%s" PATHSEPARATOR "%s", TRACKDIR, pktp->filename);
#else
  /* In UNIX use the same tmpdir to store the tracking file but
     add a '.track' suffix to the filename. */
  sprintf(datapath, "%s" PATHSEPARATOR "%s", tmpdir, pktp->filename);
  sprintf(trackpath, "%s.track", datapath);
#endif

  newfile = FALSE;
  /* Index value into the track record for this block. */
  n = sizeof(pktp->datestamp) + pktp->blocknum - 1;

  /* Open the block tracking file first. */
  if ((trackfp=fopen(trackpath,"r+")) == NULL) newfile = TRUE;
  else {
    fgets(trackrec,sizeof(trackrec),trackfp);
    /* Drop the line terminator if it exists. */
    if (trackrec[strlen(trackrec)-1] == '\n')
      trackrec[strlen(trackrec)-1] = '\0';
    /* If the datestamp doesn't match then create a new file. */
    if (strncmp(pktp->datestamp,trackrec,sizeof(pktp->datestamp)-1)) {
      newfile = TRUE;
      fclose(trackfp);
    }
    /* Make sure the block numbers and track record length make sense. */
    else if (n >= strlen(trackrec) ||
             sizeof(pktp->datestamp)+pktp->totalblocks !=
                    strlen(trackrec)) {
      fclose(trackfp);
      fprintf(stderr,"Track record length error.\n");
      return(-1);
    }
    /* Do nothing if we already have the block. */
    else if (trackrec[n] == 'Y') {
      fclose(trackfp);
      return(0);
    }
    /* If the data file can't be opened, assume it must be created. */
    else if ((datafp=fopen(datapath,"r+b")) == NULL) {
      newfile = TRUE;
      fclose(trackfp);
    }
  }

  if (newfile) {
    if ((datafp=fopen(datapath,"w+b")) == NULL) {
      perror("saveblock, fopen data file");
      return(-1);
    }
    /* Construct a new track record.  It consists of the file
       datestamp, a comma, and a series of . or Y characters where Y
       indicates the block has already been received. */
    memcpy(trackrec,pktp->datestamp,sizeof(pktp->datestamp));
    trackrec[sizeof(pktp->datestamp)-1] = ',';
    memset(trackrec+sizeof(pktp->datestamp), '.', pktp->totalblocks);
    trackrec[sizeof(pktp->datestamp)+pktp->totalblocks] = '\0';
    if ((trackfp=fopen(trackpath,"w+")) == NULL) {
      perror("saveblock, fopen track file");
      return(-1);
    }
    fputs(trackrec,trackfp);
  }

  /* Did we already save this block? */
  if (trackrec[n] != 'Y') {
    /* Write the block. */
    if (fseek(datafp, sizeof(qbtpacket.data)*(pktp->blocknum - 1),
        SEEK_SET)) {
      perror("saveblock, fseek");
      return(-1);
    }

    if ((pktp->blocknum == pktp->totalblocks) && 
        ((strstr(pktp->filename,".TXT") != NULL) || 
         (strstr(pktp->filename,".HTM") != NULL))) {
      /* If this is the last block of a text file, truncate the trailing
         null characters by changing the number of bytes to write. */
      for(datasize=sizeof(qbtpacket.data); datasize > 0; datasize--)
        if (pktp->data[datasize-1] != '\0') break;
    }
    else datasize=sizeof(qbtpacket.data);
    if (fwrite(pktp->data, datasize, 1, datafp) != 1) {
      perror("saveblock, fwrite");
      return(-1);
    }

    /* Update the track record. */
    trackrec[n] = 'Y';
    rewind(trackfp);
    fputs(trackrec,trackfp);
  }
  fclose(datafp);
  fclose(trackfp);

  /* Check the track record to see if we have all the parts. */
  for (i=sizeof(pktp->datestamp);
       i<sizeof(pktp->datestamp) + pktp->totalblocks; i++)
  if (trackrec[i] != 'Y') return(0);

  strcpy(savedfilepath,datapath);
  return(1);
} /* saveblock() */

/* run-program is taken directly from the ppp-2.3.10 source
 * where it is used for example to run ip-up and ip-down scripts.
 */
/*
 * run-program - execute a program with given arguments,
 * but don't wait for it.
 * If the program can't be executed, logs an error unless
 * must_exist is 0 and the program file doesn't exist.
 * Returns -1 if it couldn't fork, 0 if the file doesn't exist
 * or isn't an executable plain file, or the process ID of the child.
 * If done != NULL, (*done)(arg) will be called later (within
 * reap_kids) iff the return value is > 0.
 */
pid_t
run_program(char *prog,char *args[],int must_exist,void (*done)((void *)),void *arg)
{
    int pid;
    struct stat sbuf;

    /*
     * First check if the file exists and is executable.
     * We don't use access() because that would use the
     * real user-id, which might not be root, and the script
     * might be accessible only to root.
     */
    errno = EINVAL;
    if (stat(prog, &sbuf) < 0 || !S_ISREG(sbuf.st_mode)
	|| (sbuf.st_mode & (S_IXUSR|S_IXGRP|S_IXOTH)) == 0) {
	if (must_exist || errno != ENOENT)
	    warn("Can't execute %s: %m", prog);
	return 0;
    }

    pid = fork();
    if (pid == -1) {
	error("Failed to create child process for %s: %m", prog);
	return -1;
    }
    if (pid == 0) {
	int new_fd;

	/* Leave the current location */
	(void) setsid();	/* No controlling tty. */
	(void) umask (S_IRWXG|S_IRWXO);
	(void) chdir ("/");	/* no current directory. */
	setuid(0);		/* set real UID = root */
	setgid(getegid());

	/* Ensure that nothing of our device environment is inherited. */
	sys_close();
	closelog();
	close (0);
	close (1);
	close (2);
	close (ttyfd);  /* tty interface to the ppp device */
	if (real_ttyfd >= 0)
	    close(real_ttyfd);

        /* Don't pass handles to the PPP device, even by accident. */
	new_fd = open (_PATH_DEVNULL, O_RDWR);
	if (new_fd >= 0) {
	    if (new_fd != 0) {
	        dup2  (new_fd, 0); /* stdin <- /dev/null */
		close (new_fd);
	    }
	    dup2 (0, 1); /* stdout -> /dev/null */
	    dup2 (0, 2); /* stderr -> /dev/null */
	}

#ifdef BSD
	/* Force the priority back to zero if pppd is running higher. */
	if (setpriority (PRIO_PROCESS, 0, 0) < 0)
	    warn("can't reset priority to 0: %m"); 
#endif

	/* SysV recommends a second fork at this point. */

	/* run the program */
	execve(prog, args, script_env);
	if (must_exist || errno != ENOENT) {
	    /* have to reopen the log, there's nowhere else
	       for the message to go. */
	    reopen_log();
	    syslog(LOG_ERR, "Can't execute %s: %m", prog);
	    closelog();
	}
	_exit(-1);
    }

#ifdef PPP
    if (debug)
	dbglog("Script %s started (pid %d)", prog, pid);
    record_child(pid, prog, done, arg);
#endif
    return pid;
}
