
#include "lispeval.h"
#include "lispuserfunc.h"
#include "standard.h"

#include "lispio.h"
#include "platfileio.h"
#include "infixparser.h"


LispUserFunction* GetUserFunction(LispEnvironment& aEnvironment,
                                  LispPtr* subList)
{
    LispObject* head = subList->Get();
    LispUserFunction* userFunc = aEnvironment.UserFunction(*subList);
    if (userFunc != NULL)
    {
        return userFunc;
    }
    else if (head->String()!=NULL)
    {
        LispMultiUserFunction* multiUserFunc =
        aEnvironment.MultiUserFunction(head->String());
        if (multiUserFunc->iFileToOpen!=NULL)
        {
            LispDefFile* def = multiUserFunc->iFileToOpen;
            /*Show loading...
             printf("%s->%s\n",head->String()->String(),def->iFileName->String());
             */
            multiUserFunc->iFileToOpen=NULL;
            InternalUse(aEnvironment,def->iFileName);

//            printf("Finished using %s\n",def->iFileName->String());

            userFunc = aEnvironment.UserFunction(*subList);
        }
    }
    return userFunc;
}



// Eval: evaluates an expression. The result of this operation must
// be a unique (copied) element! Eg. its Next might be set...
void BasicEvaluator::Eval(LispEnvironment& aEnvironment, LispPtr& aResult, LispPtr& aExpression)
{
    LISPASSERT(aExpression.Get() != NULL);

    aEnvironment.iEvalDepth++;
    Check(aEnvironment.iEvalDepth<aEnvironment.iMaxEvalDepth,
          KLispErrMaxRecurseDepthReached);

    LispStringPtr str = aExpression.Get()->String();

    // Evaluate an atom: find the bound value (treat it as a variable)
    if (str)
    {
        if (str->String()[0] == '\"')
        {
            aResult.Set(aExpression.Get()->Copy(LispFalse));
            goto FINISH;
        }

        LispPtr* val = aEnvironment.GetVariable(str);
        if (val)
        if (val->Get())
        {
            aResult.Set(val->Get()->Copy(LispFalse));
            goto FINISH;
        }
        aResult.Set(aExpression.Get()->Copy(LispFalse));
        goto FINISH;
    }

    {
        EvalFuncBase* func = aExpression.Get()->EvalFunc();
        LispPtr* subList = aExpression.Get()->SubList();
        if (func)
        {
            func->Evaluate(aResult, aEnvironment, *subList);
            goto FINISH;
        }
        if (subList)
        {
            LispObject* head = subList->Get();
            if (head)
            {
                LispEvaluator* evaluator =
                    aEnvironment.Commands().LookUp(head->String());

                // Try to find a built-in command
                if (evaluator)
                {
                aExpression.Get()->SetEvalFunc(evaluator);
                evaluator->Evaluate(aResult, aEnvironment, *subList);
                goto FINISH;
                }
                // Else try to find a user-defined function
                else
                {

                    LispUserFunction* userFunc;

                    userFunc = GetUserFunction(aEnvironment, subList);
                    if (userFunc != NULL)
                    {
                        aExpression.Get()->SetEvalFunc(userFunc);
                        userFunc->Evaluate(aResult,aEnvironment,*subList);
                        goto FINISH;
                    }
                }
            }
        }
        aResult.Set(aExpression.Get()->Copy(LispFalse));
    }
FINISH:

    aEnvironment.iEvalDepth--;
}

void TraceShowArg(LispEnvironment& aEnvironment,LispPtr& aParam,
                  LispPtr& aValue)
{
    LispInt i;
    StdUserOutput out;
    InfixPrinter infixprinter(aEnvironment.PreFix(),
                              aEnvironment.InFix(),
                              aEnvironment.PostFix(),
                              aEnvironment.Bodied());
    for (i=0;i<aEnvironment.iEvalDepth+2;i++)
        out.Write("  ");
    out.Write("ARG:");
    infixprinter.Print(aParam,out);
    out.Write(" <- ");
    infixprinter.Print(aValue,out);
    out.Write("\n");
}

void TraceShowEnter(LispEnvironment& aEnvironment,
                    LispPtr& aExpression)
{
    StdUserOutput out;
    LispInt i;
    for (i=0;i<aEnvironment.iEvalDepth;i++)
        out.Write("  ");
    out.Write("ENTER:");
    InfixPrinter infixprinter(aEnvironment.PreFix(),
                              aEnvironment.InFix(),
                              aEnvironment.PostFix(),
                              aEnvironment.Bodied());

    infixprinter.Print(aExpression, out);
    out.Write("\n");
}



void TraceShowLeave(LispEnvironment& aEnvironment, LispPtr& aResult,
                    LispPtr& aExpression)
{
    StdUserOutput out;
    LispInt i;
    for (i=0;i<aEnvironment.iEvalDepth;i++)
        out.Write("  ");
    out.Write("LEAVE:");
    InfixPrinter infixprinter(aEnvironment.PreFix(),
                              aEnvironment.InFix(),
                              aEnvironment.PostFix(),
                              aEnvironment.Bodied());

    infixprinter.Print(aExpression, out);
    out.Write(" -> ");
    infixprinter.Print(aResult, out);
    out.Write("\n");
}

void TracedEvaluator::Eval(LispEnvironment& aEnvironment, LispPtr& aResult,
                           LispPtr& aExpression)
{
    TraceShowEnter(aEnvironment, aExpression);
    BasicEvaluator::Eval(aEnvironment, aResult, aExpression);
    TraceShowLeave(aEnvironment, aResult, aExpression);
}



