/*
 * Copyright 1995,96 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: Literal.c,v 2.3 1996/07/18 17:37:04 bousch Exp $
 *
 * Literals are really just strings. Unlike other types, each object is
 * unique: different mnodes will never contain the same string.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "saml.h"
#include "mnode.h"
#include "builtin.h"

typedef struct _literal {
	struct mnode_header hdr;
	struct _literal *next;
	char name[0];
} literal;

#define INITIAL_HASHSIZE  59
static literal **hashtable = NULL;
static unsigned int hashsize = 0;
static unsigned int entries = 0;

static void literal_free (literal*);
static s_mnode* literal_build (const char*);
static gr_string* literal_stringify (literal*);
static int literal_differ (literal*, literal*);
static int literal_lessthan (literal*, literal*);

static unsafe_s_mtype MathType_Literal = {
	"Literal",
	literal_free, literal_build, literal_stringify,
	NULL, NULL,
	NULL, NULL, NULL, NULL, NULL,
	NULL, NULL, NULL, literal_differ, literal_lessthan,
	NULL, NULL, NULL, NULL, NULL
};

static unsigned int hash_value (const char *str)
{
	unsigned int hash = 0;
	char c;

	while ((c = *str++) != '\0')
		hash += (hash<<2) + (unsigned char)c;
	return hash % hashsize;
}

static void resize_htable (unsigned int new_size)
{
	literal *list, *p, *q;
	unsigned int i, h;

#ifdef DEBUG_LITERALS
	fprintf(stderr, "Resizing literal hash-table from %d to %d slots\n",
		lit_hsize, new_size);
#endif
	/* First step: link all nodes on the "list" */
	list = NULL;
	for (i = 0; i < hashsize; i++)
		for (p = hashtable[i]; p; p = q) {
			q = p->next;
			p->next = list;
			list = p;
		}
	hashtable = realloc(hashtable, new_size * sizeof(literal*));
	if (hashtable == NULL)
		panic_out_of_memory();
	hashsize = new_size;
	memset(hashtable, 0, hashsize * sizeof(literal*));

	/* And insert them again in the new table */
	for (p = list; p; p = q) {
		q = p->next;
		h = hash_value(p->name);
		p->next = hashtable[h];
		hashtable[h] = p;
	}
}

void init_MathType_Literal (void)
{
	register_mtype(ST_LITERAL, &MathType_Literal);
	resize_htable(INITIAL_HASHSIZE);
}

static void literal_free (literal *mn)
{
	literal *q, **old;
	unsigned int h = hash_value(mn->name);

	for (old = &hashtable[h]; (q = *old) != NULL; old = &q->next)
		if (q == mn) {
			*old = q->next;
			break;
		}
	assert(mn == q);
	free(mn);
	--entries;
}

static s_mnode* literal_build (const char *str)
{
	unsigned int h = hash_value(str);
	size_t length;
	literal *p;

	for (p = hashtable[h]; p; p = p->next)
		if (strcmp(str, p->name) == 0)
			return copy_mnode((s_mnode*)p);
	/*
	 * Not found: create a new literal
	 */
	length = sizeof(literal) + strlen(str) + 1;
	p = (literal*) __mnalloc(ST_LITERAL, length);
	strcpy(p->name, str);
	p->next = hashtable[h];
	hashtable[h] = p;

	if (++entries > hashsize)
		resize_htable(2*hashsize+1);
	return (s_mnode*) p;
}

static gr_string* literal_stringify (literal* p)
{
	char *name = p->name;
	gr_string *grs = new_gr_string(0);
	return grs_append(grs, name, strlen(name));
}

static int literal_differ (literal* p1, literal* p2)
{
	return (p1 != p2);
}

static int literal_lessthan (literal* p1, literal* p2)
{
	return (p1 < p2);
}
