/*
 * Copyright (c) 1994 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

/* Tcl_LdapSearchCmd - simple function to search/read an entry using LDAP */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <lber.h>
#include <ldap.h>
#include "tcl.h"
#define DEFSEP "="

/* Set the default base to 'base' */
static char *base = LDAP_SCOPE_BASE;
static char *sep = DEFSEP;
static char *sortattr = NULL;
static int verbose, not, allow_binary, vals2tmp;

static int dosearch _ANSI_ARGS_ ((Tcl_Interp * interp, LDAP * ld, char *base,
	int scope, char **attrs, int attrsonly, char *filtpatt, char *value));
static int print_entry _ANSI_ARGS_ ((Tcl_Interp * interp, LDAP * ld,
	LDAPMessage * entry, int attrsonly));

extern int ldcnt;
extern LDAP *ldprime[1024];

int Tcl_LdapSearchCmd (dummy, interp, argc, argv)
     ClientData dummy;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
    char *filtpattern, **attrs;
    int conn;
    int rc, scope, kerberos, deref, attrsonly;
    int lArgc;
    char *slevel, *dlevel;

    verbose = allow_binary = not = kerberos = vals2tmp = attrsonly = 0;
    scope = LDAP_SCOPE_SUBTREE;
    deref = LDAP_DEREF_NEVER;

    if (argc < 6) {
	Tcl_AppendResult (interp, "Too few arguments! Usage: \n", (char *) NULL);
	Tcl_AppendResult (interp, argv[0], " LdapConn ", (char *) NULL);
	Tcl_AppendResult (interp, "Scope Deref BaseDN Filter ", (char *) NULL);
	Tcl_AppendResult (interp, "[LIST of Attrs]", (char *) NULL);
	return TCL_ERROR;
    }

    conn = atoi (argv[1]);
    slevel = argv[2];
    dlevel = argv[3];
    base = argv[4];
    filtpattern = argv[5];
    if (argc > 6) {
	if ((Tcl_SplitList (interp, argv[6], &lArgc, &attrs)) != TCL_OK) {
	    Tcl_AppendResult (interp, "No LIST argument passed!", (char *) NULL);
	    return TCL_ERROR;
	}
    } else {
	attrs = NULL;
    }

    if (strncasecmp (slevel, "base", 4) == 0) {
	scope = LDAP_SCOPE_BASE;
    }
    if (strncasecmp (slevel, "one", 3) == 0) {
	scope = LDAP_SCOPE_ONELEVEL;
    }
    if (strncasecmp (slevel, "sub", 3) == 0) {
	scope = LDAP_SCOPE_SUBTREE;
    }
    if (strncasecmp (dlevel, "never", 5) == 0) {
	deref = LDAP_DEREF_NEVER;
    }
    if (strncasecmp (dlevel, "search", 5) == 0) {
	deref = LDAP_DEREF_SEARCHING;
    }
    if (strncasecmp (dlevel, "find", 4) == 0) {
	deref = LDAP_DEREF_FINDING;
    }
    if (strncasecmp (dlevel, "always", 6) == 0) {
	deref = LDAP_DEREF_ALWAYS;
    }

    ldprime[conn]->ld_deref = deref;
    ldprime[conn]->ld_timelimit = 0;
    ldprime[conn]->ld_sizelimit = 0;
    ldprime[conn]->ld_options = 0;

    rc = dosearch (interp, ldprime[conn], base, scope, attrs, attrsonly,
	filtpattern, "");

    return TCL_OK;
}

static int dosearch (interp, ld, base, scope, attrs, attrsonly, filtpatt, value)
     Tcl_Interp *interp;
     LDAP *ld;
     char *base;
     int scope;
     char **attrs;
     int attrsonly;
     char *filtpatt;
     char *value;
{
    char filter[BUFSIZ];
    int rc, first, matches;
    LDAPMessage *res, *e;

    sprintf (filter, filtpatt, value);

    if (ldap_search (ld, base, scope, filter, attrs, attrsonly) == -1) {
	Tcl_AppendResult (interp, "Search failed!", (char *) NULL);
	return TCL_ERROR;
    }

    matches = 0;
    first = 1;
    while ((rc = ldap_result (ld, LDAP_RES_ANY, sortattr ? 1 : 0, NULL, &res))
	== LDAP_RES_SEARCH_ENTRY) {
	matches++;
	e = ldap_first_entry (ld, res);
	if (!first) {
	    Tcl_AppendElement (interp, "");
	} else {
	    first = 0;
	}
	print_entry (interp, ld, e, attrsonly);
	Tcl_FreeResult (interp);
	ldap_msgfree (res);
    }
    if (rc == -1) {
	Tcl_AppendResult (interp, "Search failed to find anything!", (char *) NULL);
	return TCL_ERROR;
    }
    if ((rc = ldap_result2error (ld, res, 0)) != LDAP_SUCCESS) {
	Tcl_AppendResult (interp, "Search failed to find anything!", (char *) NULL);
    }

    ldap_msgfree (res);
    return (rc);
}

static int print_entry (interp, ld, entry, attrsonly)
     Tcl_Interp *interp;
     LDAP *ld;
     LDAPMessage *entry;
     int attrsonly;
{
    char *a, resultstr[8192];
    int i;
    BerElement *ber;
    struct berval **bvals;

    for (a = ldap_first_attribute (ld, entry, &ber); a != NULL;
	a = ldap_next_attribute (ld, entry, ber)) {
	bvals = ldap_get_values_len (ld, entry, a);
	for (i = 0; bvals[i] != NULL; i++) {
	    sprintf (resultstr, "%s%s%s", a, sep, bvals[i]->bv_val);
	    Tcl_AppendElement (interp, resultstr);
	}
	ber_bvecfree (bvals);
    }
    return (0);
}
