/*
 * Copyright (c) 1994 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 */

/* Tcl_LdapModifyCmd - simple function to modify an entry using LDAP */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <fcntl.h>
#include <unistd.h>
#include <lber.h>
#include <ldap.h>
#include "tcl.h"
#define safe_realloc(ptr,size) (ptr == NULL ? malloc(size) : realloc(ptr,size))

static int freepmods _ANSI_ARGS_ ((LDAPMod ** pmods));
static int addmodifyop _ANSI_ARGS_ ((LDAPMod *** pmodsp, char *buf));
static int new;

extern int ldcnt;
extern LDAP *ldprime[1024];

int Tcl_LdapModifyCmd (dummy, interp, argc, argv)
     ClientData dummy;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
    char *entrydn, *buf;
    int not, verbose, rc, linenum, conn;
    LDAPMod **pmods;
    int localArgc;
    char **localArgv;

    rc = not = verbose = 0;
    pmods = NULL;

    new = (strcmp (argv[0], "LdapAdd") == 0);

    if (argc < 4) {
	Tcl_AppendResult (interp, "Too few arguments! Usage: \n", (char *) NULL);
	Tcl_AppendResult (interp, argv[0], " LdapConn ", (char *) NULL);
	Tcl_AppendResult (interp, "DN [LIST of attr=value pairs]", (char *) NULL);
	return TCL_ERROR;
    }

    conn = atoi (argv[1]);
    entrydn = argv[2];

    if ((Tcl_SplitList (interp, argv[3], &localArgc, &localArgv)) != TCL_OK) {
	Tcl_AppendResult (interp, "No LIST argument passed!", (char *) NULL);
	return TCL_ERROR;
    }

    ldprime[conn]->ld_deref = LDAP_DEREF_NEVER;		/* this seems prudent */

    for (linenum = 0; linenum < localArgc; linenum++) {
	buf = localArgv[linenum];
	if ((addmodifyop (&pmods, buf)) != TCL_OK) {
	    Tcl_AppendResult (interp, "Problem with data!", (char *) NULL);
	    return TCL_ERROR;
	}
    }

    if (pmods != NULL) {
	if (new) {
	    if ((ldap_add_s (ldprime[conn], entrydn, pmods)) != LDAP_SUCCESS) {
		Tcl_AppendResult (interp, "Unable to add entry!", (char *) NULL);
		freepmods (pmods);
		return TCL_ERROR;
	    }
	} else {
	    if ((ldap_modify_s (ldprime[conn], entrydn, pmods)) != LDAP_SUCCESS) {
		Tcl_AppendResult (interp, "Unable to change entry!", (char *) NULL);
		freepmods (pmods);
		return TCL_ERROR;
	    }
	}
	freepmods (pmods);
    }

    return TCL_OK;
}				/* End of main function Tcl_LdapModify */

static int addmodifyop (pmodsp, buf)
     LDAPMod ***pmodsp;
     char *buf;
{
    char *p, *q, *attr, *value;
    LDAPMod **pmods;
    int i, j, modop;
    struct berval *bvp;

    pmods = *pmodsp;

    if ((p = strchr (buf, '=')) == NULL) {
	value = NULL;
	p = buf + strlen (buf);
    } else {
	*p++ = '\0';
	value = p;
    }

    for (attr = buf; *attr != '\0' && isspace (*attr); ++attr) {;
    }
    for (q = p - 1; q > attr && isspace (*q); --q) {
	*q = '\0';
    }

    if (value != NULL) {
	while (isspace (*value)) {
	    ++value;
	}
	for (q = value + strlen (value) - 1; q > value && isspace (*q); --q) {
	    *q = '\0';
	}
    } else {
	if (new) {
	    return TCL_ERROR;
	}
    }

    switch (*attr) {
    case '-':
	modop = LDAP_MOD_DELETE;
	++attr;
	break;
    case '+':
	modop = LDAP_MOD_ADD;
	++attr;
	break;
    default:
	modop = LDAP_MOD_REPLACE;
    }

    modop |= LDAP_MOD_BVALUES;

    i = 0;
    if (pmods != NULL) {
	for (; pmods[i] != NULL; ++i) {
	    if (strcasecmp (pmods[i]->mod_type, attr) == 0 && pmods[i]->mod_op
		== modop) {
		break;
	    }
	}
    }

    if (pmods == NULL || pmods[i] == NULL) {
	if ((pmods = (LDAPMod **) safe_realloc (pmods, (i + 2) *
		    sizeof (LDAPMod *))) == NULL) {
	    return TCL_ERROR;
	}
	*pmodsp = pmods;
	pmods[i + 1] = NULL;
	if ((pmods[i] = (LDAPMod *) calloc (1, sizeof (LDAPMod))) == NULL) {
	    return TCL_ERROR;
	}
	pmods[i]->mod_op = modop;
	if ((pmods[i]->mod_type = strdup (attr)) == NULL) {
	    return TCL_ERROR;
	}
    }

    if (value != NULL) {
	j = 0;
	if (pmods[i]->mod_bvalues != NULL) {
	    for (; pmods[i]->mod_bvalues[j] != NULL; ++j) {;
	    }
	}
	if ((pmods[i]->mod_bvalues =
		(struct berval **) safe_realloc (pmods[i]->mod_bvalues,
		    (j + 2) * sizeof (struct berval *))) == NULL) {
	    return TCL_ERROR;
	}
	pmods[i]->mod_bvalues[j + 1] = NULL;
	if ((bvp = (struct berval *) malloc (sizeof (struct berval))) == NULL) {
	    return TCL_ERROR;
	}
	pmods[i]->mod_bvalues[j] = bvp;

	bvp->bv_len = strlen (value);
	if ((bvp->bv_val = strdup (value)) == NULL) {
	    return TCL_ERROR;
	}
    }
    return TCL_OK;
}				/* End addmodifyop */

static int freepmods (pmods)
     LDAPMod **pmods;
{
    int i;

    for (i = 0; pmods[i] != NULL; ++i) {
	if (pmods[i]->mod_bvalues != NULL) {
	    ber_bvecfree (pmods[i]->mod_bvalues);
	}
    }
    free (pmods);
    return (0);
}				/* End of freepmods */
