/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysvideo.c,v 1.32 1999/08/06 21:10:09 slouken Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <windows.h>
#include <directx.h>

#include "SDL_error.h"
#include "SDL_timer.h"
#include "SDL_events.h"
#include "SDL_syswm.h"
#include "SDL_sysvideo.h"
#include "SDL_RLEaccel_c.h"
#include "SDL_video_c.h"
#include "SDL_syswm_c.h"
#include "SDL_blit.h"
#include "SDL_pixels_c.h"
#include "SDL_lowvideo.h"


/* Hardware Acceleration functions */
static int SDL_AllocHWSurface(SDL_Surface *surface);
static int SDL_CheckHWBlit(SDL_Surface *src, SDL_Surface *dst);
static int SDL_FillHWRect(SDL_Surface *dst, SDL_Rect *dstrect, Uint32 color);
static int SDL_SetHWColorKey(SDL_Surface *surface, Uint32 key);
static int SDL_SetHWAlpha(SDL_Surface *surface, Uint32 alpha);
static int SDL_LockHWSurface(SDL_Surface *surface);
static void SDL_UnlockHWSurface(SDL_Surface *surface);
static int SDL_FlipHWSurface(SDL_Surface *surface);
static void SDL_FreeHWSurface(SDL_Surface *surface);

static int SDL_AllocDDSurface(SDL_Surface *surface, 
				LPDIRECTDRAWSURFACE3 requested, Uint32 flag);

/* Various local structures */
static LPDIRECTDRAW2 ddraw2;
static LPDIRECTDRAWSURFACE3 SDL_primary;
static LPDIRECTDRAWCLIPPER SDL_clipper;
static LPDIRECTDRAWPALETTE SDL_palette;
static PALETTEENTRY SDL_colors[256];
static int colorchange_expected = 0;

#define NUM_MODELISTS	4		/* 8, 16, 24, and 32 bits-per-pixel */

static int SDL_nummodes[NUM_MODELISTS];
static SDL_Rect **SDL_modelist[NUM_MODELISTS];
static int SDL_modeindex[NUM_MODELISTS];

static HRESULT WINAPI CountModes(DDSURFACEDESC *desc, VOID *udata)
{
#if defined(NONAMELESSUNION)
	int bpp = desc->ddpfPixelFormat.u1.dwRGBBitCount;
#else
	int bpp = desc->ddpfPixelFormat.dwRGBBitCount;
#endif
	switch (bpp)  {
		case 8:
		case 16:
		case 24:
		case 32:
			++SDL_nummodes[(bpp/8)-1];
			break;
	}
	return(DDENUMRET_OK);
}

static HRESULT WINAPI EnumModes(DDSURFACEDESC *desc, VOID *udata)
{
#if defined(NONAMELESSUNION)
	int bpp = desc->ddpfPixelFormat.u1.dwRGBBitCount;
#else
	int bpp = desc->ddpfPixelFormat.dwRGBBitCount;
#endif
	int index;

	switch (bpp)  {
		case 8:
		case 16:
		case 24:
		case 32:
			bpp /= 8; --bpp;
			index = SDL_nummodes[bpp]-(++SDL_modeindex[bpp]);
			SDL_modelist[bpp][index]->x = 0;
			SDL_modelist[bpp][index]->y = 0;
			SDL_modelist[bpp][index]->w = (Uint16)desc->dwWidth;
			SDL_modelist[bpp][index]->h = (Uint16)desc->dwHeight;
			break;
		default:
#ifdef DDRAW_DEBUG
			fprintf(stderr, "Unknown bit-depth: %d\n", bpp);
#endif
			break;
	}
	return(DDENUMRET_OK);
}

static void SetDDerror(char *function, int code)
{
	static char *error;
	static char  errbuf[BUFSIZ];

	errbuf[0] = 0;
	switch (code) {
		case DDERR_GENERIC:
			error = "Undefined error!";
			break;
		case DDERR_EXCEPTION:
			error = "Exception encountered";
			break;
		case DDERR_INVALIDOBJECT:
			error = "Invalid object";
			break;
		case DDERR_INVALIDPARAMS:
			error = "Invalid parameters";
			break;
		case DDERR_NOTFOUND:
			error = "Object not found";
			break;
		case DDERR_INVALIDRECT:
			error = "Invalid rectangle";
			break;
		case DDERR_INVALIDCAPS:
			error = "Invalid caps member";
			break;
		case DDERR_INVALIDPIXELFORMAT:
			error = "Invalid pixel format";
			break;
		case DDERR_OUTOFMEMORY:
			error = "Out of memory";
			break;
		case DDERR_OUTOFVIDEOMEMORY:
			error = "Out of video memory";
			break;
		case DDERR_SURFACEBUSY:
			error = "Surface busy";
			break;
		case DDERR_SURFACELOST:
			error = "Surface was lost";
			break;
		case DDERR_WASSTILLDRAWING:
			error = "DirectDraw is still drawing";
			break;
		case DDERR_INVALIDSURFACETYPE:
			error = "Invalid surface type";
			break;
		case DDERR_NOEXCLUSIVEMODE:
			error = "Not in exclusive access mode";
			break;
		case DDERR_NOPALETTEATTACHED:
			error = "No palette attached";
			break;
		case DDERR_NOPALETTEHW:
			error = "No palette hardware";
			break;
		case DDERR_NOT8BITCOLOR:
			error = "Not 8-bit color";
			break;
		case DDERR_EXCLUSIVEMODEALREADYSET:
			error = "Exclusive mode was already set";
			break;
		case DDERR_HWNDALREADYSET:
			error = "Window handle already set";
			break;
		case DDERR_HWNDSUBCLASSED:
			error = "Window handle is subclassed";
			break;
		case DDERR_NOBLTHW:
			error = "No blit hardware";
			break;
		case DDERR_IMPLICITLYCREATED:
			error = "Surface was implicitly created";
			break;
		case DDERR_INCOMPATIBLEPRIMARY:
			error = "Incompatible primary surface";
			break;
		case DDERR_NOCOOPERATIVELEVELSET:
			error = "No cooperative level set";
			break;
		case DDERR_NODIRECTDRAWHW:
			error = "No DirectDraw hardware";
			break;
		case DDERR_NOEMULATION:
			error = "No emulation available";
			break;
		case DDERR_NOFLIPHW:
			error = "No flip hardware";
			break;
		case DDERR_NOTFLIPPABLE:
			error = "Surface not flippable";
			break;
		case DDERR_PRIMARYSURFACEALREADYEXISTS:
			error = "Primary surface already exists";
			break;
		case DDERR_UNSUPPORTEDMODE:
			error = "Unsupported mode";
			break;
		case DDERR_WRONGMODE:
			error = "Surface created in different mode";
			break;
		case DDERR_UNSUPPORTED:
			error = "Operation not supported";
			break;
		default:
			sprintf(errbuf, "%s: Unknown DirectDraw error: 0x%x",
								function, code);
			break;
	}
	if ( ! errbuf[0] ) {
		sprintf(errbuf, "%s: %s", function, error);
	}
	SDL_SetError("%s", errbuf);
	return;
}

static int SDL_UpdateVideoInfo(SDL_VideoInfo *info)
{
	/* This needs to be DDCAPS_DX5 for the DirectDraw2 interface */
	DDCAPS_DX5 DDCaps;
	HRESULT result;

	/* Fill in our hardware acceleration capabilities */
	memset(&DDCaps, 0, sizeof(DDCaps));
	DDCaps.dwSize = sizeof(DDCaps);
	result = IDirectDraw2_GetCaps(ddraw2, (DDCAPS *)&DDCaps, NULL);
	if ( result != DD_OK ) {
		SetDDerror("DirectDraw2::GetCaps", result);
		return(-1);
	}
	info->hw_available = 1;
	if ( (DDCaps.dwCaps & DDCAPS_BLT) == DDCAPS_BLT ) {
		info->blit_hw = 1;
	}
	if ( ((DDCaps.dwCaps & DDCAPS_COLORKEY) == DDCAPS_COLORKEY) &&
	     ((DDCaps.dwCKeyCaps & DDCKEYCAPS_SRCBLT) == DDCKEYCAPS_SRCBLT) ) {
		info->blit_hw_CC = 1;
	}
	if ( (DDCaps.dwCaps & DDCAPS_ALPHA) == DDCAPS_ALPHA ) {
		/* This is only for alpha channel, and DirectX 6
		   doesn't support 2D alpha blits yet, so set it 0
		 */
		info->blit_hw_A = 0;
	}
	if ( (DDCaps.dwCaps & DDCAPS_CANBLTSYSMEM) == DDCAPS_CANBLTSYSMEM ) {
		info->blit_sw = 1;
		/* This isn't necessarily true, but the HEL will cover us */
		info->blit_sw_CC = info->blit_hw_CC;
		info->blit_sw_A = info->blit_hw_A;
	}
	if ( (DDCaps.dwCaps & DDCAPS_BLTCOLORFILL) == DDCAPS_BLTCOLORFILL ) {
		info->blit_fill = 1;
	}

	/* Find out how much video memory is available */
	{ DDSCAPS ddsCaps;
	  DWORD total_mem;
		ddsCaps.dwCaps = DDSCAPS_VIDEOMEMORY;
		result = IDirectDraw2_GetAvailableVidMem(ddraw2,
						&ddsCaps, &total_mem, NULL);
		if ( result != DD_OK ) {
			total_mem = DDCaps.dwVidMemTotal; 
		}
		info->video_mem = total_mem/1024;
	}
	return(0);
}

int SDL_SYS_VideoInit(SDL_PixelFormat *vformat)
{
	HRESULT result;
	LPDIRECTDRAW ddraw;
	int i, j;
	HDC hdc;

	/* Intialize everything */
	ddraw2 = NULL;
	SDL_primary = NULL;
	SDL_clipper = NULL;
	SDL_palette = NULL;
	for ( i=0; i<NUM_MODELISTS; ++i ) {
		SDL_nummodes[i] = 0;
		SDL_modelist[i] = NULL;
		SDL_modeindex[i] = 0;
	}

	/* Create the window */
	if ( SDL_CreateWindow() < 0 ) {
		return(-1);
	}
	SDL_SYS_SoundFocus(SDL_Window);

	/* Create the DirectDraw object */
	result = DirectDrawCreate(NULL, &ddraw, NULL);
	if ( result != DD_OK ) {
		SetDDerror("DirectDrawCreate", result);
		return(-1);
	}
	result = IDirectDraw_QueryInterface(ddraw, &IID_IDirectDraw2,
							(LPVOID *)&ddraw2);
	IDirectDraw_Release(ddraw);
	if ( result != DD_OK ) {
		SetDDerror("DirectDraw::QueryInterface", result);
		return(-1);
	}

	/* Determine the screen depth */
	hdc = GetDC(SDL_Window);
	vformat->BitsPerPixel = GetDeviceCaps(hdc,PLANES) *
					GetDeviceCaps(hdc,BITSPIXEL);
	ReleaseDC(SDL_Window, hdc);

	/* Enumerate the available fullscreen modes */
	result = IDirectDraw2_EnumDisplayModes(ddraw2,0,NULL,NULL,CountModes);
	if ( result != DD_OK ) {
		SetDDerror("DirectDraw2::EnumDisplayModes", result);
		return(-1);
	}
	for ( i=0; i<NUM_MODELISTS; ++i ) {
		SDL_modelist[i] = (SDL_Rect **)
				malloc((SDL_nummodes[i]+1)*sizeof(SDL_Rect *));
		if ( SDL_modelist[i] == NULL ) {
			SDL_OutOfMemory();
			return(-1);
		}
		for ( j=0; j<SDL_nummodes[i]; ++j ) {
			SDL_modelist[i][j]=(SDL_Rect *)malloc(sizeof(SDL_Rect));
			if ( SDL_modelist[i][j] == NULL ) {
				SDL_OutOfMemory();
				return(-1);
			}
		}
		SDL_modelist[i][j] = NULL;
	}
	result = IDirectDraw2_EnumDisplayModes(ddraw2,0,NULL,NULL,EnumModes);
	if ( result != DD_OK ) {
		SetDDerror("DirectDraw2::EnumDisplayModes", result);
		return(-1);
	}

	/* Fill in some window manager capabilities */
	SDL_HWCaps.info.wm_available = 1;
	SDL_WMCaps.SetIcon = SDL_SYS_SetWMIcon;
	SDL_WMCaps.SetCaption = SDL_SYS_SetWMCaption;
	SDL_WMCaps.GetWMInfo = SDL_SYS_GetWMInfo;

	/* Fill in the video hardware capabilities */
	SDL_UpdateVideoInfo(&SDL_HWCaps.info);
	SDL_HWCaps.AllocHWSurface = SDL_AllocHWSurface;
	SDL_HWCaps.CheckHWBlit = SDL_CheckHWBlit;
	SDL_HWCaps.FillHWRect = SDL_FillHWRect;
	SDL_HWCaps.SetHWColorKey = SDL_SetHWColorKey;
	SDL_HWCaps.LockHWSurface = SDL_LockHWSurface;
	SDL_HWCaps.UnlockHWSurface = SDL_UnlockHWSurface;
	SDL_HWCaps.FlipHWSurface = SDL_FlipHWSurface;
	SDL_HWCaps.FreeHWSurface = SDL_FreeHWSurface;

	return(0);
}

SDL_Rect **SDL_SYS_ListModes(SDL_Surface *screen,
					SDL_PixelFormat *format, Uint32 flags)
{
	int bpp;

	bpp = format->BitsPerPixel;
	if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		/* FIXME:  No support for 1 bpp or 4 bpp formats */
		switch (bpp) {  /* Does windows support other BPP? */
			case 8:
			case 16:
			case 24:
			case 32:
				bpp = (bpp/8)-1;
				if ( SDL_nummodes[bpp] > 0 )
					return(SDL_modelist[bpp]);
				/* Fall through */
			default:
				return((SDL_Rect **)0);
		}
	} else {
		if ( screen->format->BitsPerPixel == bpp ) {
			return((SDL_Rect **)-1);
		} else {
			return((SDL_Rect **)0);
		}
	}
}

/* Various screen update functions available */
static void SDL_WindowUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);
static void SDL_DirectUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);

SDL_Surface *SDL_SYS_SetVideoMode(SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	SDL_Surface *video;
	HRESULT result;
	DWORD sharemode;
	DWORD style;
	const DWORD windowstyle = 
			(WS_OVERLAPPED|WS_CAPTION|WS_SYSMENU|WS_MINIMIZEBOX);
	const DWORD directstyle =
			(WS_POPUP);
	DDSURFACEDESC ddsd;
	LPDIRECTDRAWSURFACE  dd_surface1;
	LPDIRECTDRAWSURFACE3 dd_surface3;

#ifdef DDRAW_DEBUG
 fprintf(stderr, "Setting %dx%dx%d video mode\n", width, height, bpp);
#endif
	/* Clean up any previous DirectDraw surfaces */
	if ( current->hwdata ) {
		SDL_HWCaps.FreeHWSurface(current);
		current->hwdata = NULL;
	}
	if ( SDL_primary != NULL ) {
		IDirectDrawSurface3_Release(SDL_primary);
		SDL_primary = NULL;
	}

	/* Set the appropriate window style */
	ShowWindow(SDL_Window, SW_HIDE);
	style = GetWindowLong(SDL_Window, GWL_STYLE);
	if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		style &= ~windowstyle;
		style |= directstyle;
	} else {
		style &= ~directstyle;
		style |= windowstyle;
	}
	SetWindowLong(SDL_Window, GWL_STYLE, style);

	/* Set DirectDraw sharing mode.. exclusive when fullscreen */
	if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		sharemode = DDSCL_FULLSCREEN|DDSCL_EXCLUSIVE|DDSCL_ALLOWREBOOT;
	} else {
		sharemode = DDSCL_NORMAL;
	}
	result = IDirectDraw2_SetCooperativeLevel(ddraw2,SDL_Window,sharemode);
	if ( result != DD_OK ) {
		SetDDerror("DirectDraw2::SetCooperativeLevel", result);
		return(NULL);
	}

	/* Set the display mode, if we are in fullscreen mode */
	if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		/* Cover up desktop during mode change */
		ShowWindow(SDL_Window, SW_SHOW);
		while ( GetForegroundWindow() != SDL_Window ) {
			SetForegroundWindow(SDL_Window);
			SDL_Delay(100);
		}
		result = IDirectDraw2_SetDisplayMode(ddraw2, width, height,
								bpp, 0, 0);
		if ( result != DD_OK ) {
			/* We couldn't set fullscreen mode, try window */
			return(SDL_SYS_SetVideoMode(current,
				width, height, bpp, flags & ~SDL_FULLSCREEN)); 
		}
		SDL_DInputReset(1);
	} else {
		SDL_DInputReset(0);
	}
	SDL_UpdateVideoInfo(&SDL_HWCaps.info);

	/* Create a primary DirectDraw surface */
	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS;
	ddsd.ddsCaps.dwCaps = (DDSCAPS_PRIMARYSURFACE|DDSCAPS_VIDEOMEMORY);
	if ( (flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF ) {
		ddsd.dwFlags |= DDSD_BACKBUFFERCOUNT;
		ddsd.ddsCaps.dwCaps |= (DDSCAPS_COMPLEX|DDSCAPS_FLIP);
		ddsd.dwBackBufferCount = 1;
	}
	result = IDirectDraw2_CreateSurface(ddraw2, &ddsd, &dd_surface1, NULL); 
	if ( (result != DD_OK) && ((flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF) ) {
		ddsd.dwFlags &= ~DDSD_BACKBUFFERCOUNT;
		ddsd.ddsCaps.dwCaps &= ~(DDSCAPS_COMPLEX|DDSCAPS_FLIP);
		ddsd.dwBackBufferCount = 0;
		result = IDirectDraw2_CreateSurface(ddraw2,
						&ddsd, &dd_surface1, NULL); 
	}
	if ( result != DD_OK ) {
		SetDDerror("DirectDraw2::CreateSurface(PRIMARY)", result);
		return(NULL);
	}
	result = IDirectDrawSurface_QueryInterface(dd_surface1,
			&IID_IDirectDrawSurface3, (LPVOID *)&SDL_primary);
	if ( result != DD_OK ) {
		SetDDerror("DirectDrawSurface::QueryInterface", result);
		return(NULL);
	}
	result = IDirectDrawSurface_Release(dd_surface1);
	if ( result != DD_OK ) {
		SetDDerror("DirectDrawSurface::Release", result);
		return(NULL);
	}

	/* Get the format of the primary DirectDraw surface */
	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_PIXELFORMAT|DDSD_CAPS;
	result = IDirectDrawSurface3_GetSurfaceDesc(SDL_primary, &ddsd);
	if ( result != DD_OK ) {
		SetDDerror("DirectDrawSurface::Release", result);
		return(NULL);
	}
	if ( ! (ddsd.ddpfPixelFormat.dwFlags&DDPF_RGB) ) {
		SDL_SetError("Primary DDRAW surface is not RGB format");
		return(NULL);
	}

	/* Free old palette and create a new one if we're in 8-bit mode */
	if ( SDL_palette != NULL ) {
		IDirectDrawPalette_Release(SDL_palette);
		SDL_palette = NULL;
	}
#if defined(NONAMELESSUNION)
	if ( ddsd.ddpfPixelFormat.u1.dwRGBBitCount == 8 ) {
#else
	if ( ddsd.ddpfPixelFormat.dwRGBBitCount == 8 ) {
#endif
		int i;

		if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
			/* We have access to the entire palette */
			for ( i=0; i<256; ++i ) {
				SDL_colors[i].peFlags =
						(PC_NOCOLLAPSE|PC_RESERVED);
				SDL_colors[i].peRed = 0;
				SDL_colors[i].peGreen = 0;
				SDL_colors[i].peBlue = 0;
			}
		} else {
			/* First 10 colors are reserved by Windows */
			for ( i=0; i<10; ++i ) {
				SDL_colors[i].peFlags = PC_EXPLICIT;
				SDL_colors[i].peRed = i;
				SDL_colors[i].peGreen = 0;
				SDL_colors[i].peBlue = 0;
			}
			for ( i=10; i<(10+236); ++i ) {
				SDL_colors[i].peFlags = PC_NOCOLLAPSE;
				SDL_colors[i].peRed = 0;
				SDL_colors[i].peGreen = 0;
				SDL_colors[i].peBlue = 0;
			}
			/* Last 10 colors are reserved by Windows */
			for ( i=246; i<256; ++i ) {
				SDL_colors[i].peFlags = PC_EXPLICIT;
				SDL_colors[i].peRed = i;
				SDL_colors[i].peGreen = 0;
				SDL_colors[i].peBlue = 0;
			}
		}
		result = IDirectDraw2_CreatePalette(ddraw2,
		     			(DDPCAPS_8BIT|DDPCAPS_ALLOW256),
						SDL_colors, &SDL_palette, NULL);
		if ( result != DD_OK ) {
			SetDDerror("DirectDraw2::CreatePalette", result);
			return(NULL);
		}
		result = IDirectDrawSurface3_SetPalette(SDL_primary,
								SDL_palette);
		if ( result != DD_OK ) {
			SetDDerror("DirectDrawSurface3::SetPalette", result);
			return(NULL);
		}
	}

	/* Create our video surface using the same pixel format */
	video = current;
	if ( (width != video->w) || (height != video->h)
			|| (video->format->BitsPerPixel != 
#if defined(NONAMELESSUNION)
				ddsd.ddpfPixelFormat.u1.dwRGBBitCount) ) {
#else
				ddsd.ddpfPixelFormat.dwRGBBitCount) ) {
#endif 
		video = SDL_AllocSurface(SDL_SWSURFACE, 0, 0,
#if defined(NONAMELESSUNION)
				ddsd.ddpfPixelFormat.u1.dwRGBBitCount,
					ddsd.ddpfPixelFormat.u2.dwRBitMask,
					ddsd.ddpfPixelFormat.u3.dwGBitMask,
					ddsd.ddpfPixelFormat.u4.dwBBitMask,
#else
				ddsd.ddpfPixelFormat.dwRGBBitCount,
					ddsd.ddpfPixelFormat.dwRBitMask,
					ddsd.ddpfPixelFormat.dwGBitMask,
					ddsd.ddpfPixelFormat.dwBBitMask,
#endif
								0);
		if ( video == NULL ) {
			SDL_OutOfMemory();
			return(NULL);
		}
		video->w = width;
		video->h = height;
		video->pitch = (width*video->format->BytesPerPixel);
		/* Pitch needs to be QWORD (8-byte) aligned */
		video->pitch = (video->pitch + 7) & ~7;
	}
	video->flags = 0;	/* Clear flags */

	/* If not fullscreen, locking is possible, but it doesn't do what 
	   the caller really expects -- if the locked surface is written to,
	   the appropriate portion of the entire screen is modified, not 
	   the application window, as we would like.
	   Note that it is still possible to write directly to display
	   memory, but the application must respect the clip list of
	   the surface.  There might be some odd timing interactions
	   involving clip list updates and background refreshing as
	   Windows moves other windows across our window.
	   We currently don't support this, even though it might be a
	   good idea since BeOS has an implementation of BDirectWindow
	   that does the same thing.  This would be most useful for
	   applications that do complete screen updates every frame.
	    -- Fixme?
	*/
	if ( (flags & SDL_FULLSCREEN) != SDL_FULLSCREEN ) {
		/* Necessary if we're going from fullscreen to window */
		if ( video->pixels == NULL ) {
			video->pixels = (void *)malloc(video->w*video->pitch);
			if ( video->pixels == NULL ) {
				if ( video != current ) {
					SDL_FreeSurface(video);
				}
				SDL_OutOfMemory();
				return(NULL);
			}
		}
		dd_surface3 = NULL;
		video->flags |= SDL_SWSURFACE;
	} else {
		/* Necessary if we're going from window to fullscreen */
		if ( video->pixels != NULL ) {
			free(video->pixels);
			video->pixels = NULL;
		}
		dd_surface3 = SDL_primary;
		video->flags |= SDL_HWSURFACE;
	}
	if ( SDL_AllocDDSurface(video, dd_surface3, video->flags&SDL_HWSURFACE)
									< 0 ) {
		if ( video != current ) {
			SDL_FreeSurface(video);
		}
		return(NULL);
	}

	/* See if the surface we created has double-buffering */
	if ( (ddsd.ddsCaps.dwCaps & DDSCAPS_FLIP) == DDSCAPS_FLIP ) {
		video->flags |= SDL_DOUBLEBUF;
	}

	/* Use the appropriate blitting function */
	if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		video->flags |= SDL_FULLSCREEN;
		if ( video->format->palette != NULL ) {
			video->flags |= SDL_HWPALETTE;
		}
		SDL_SYS_UpdateRects = SDL_DirectUpdate;
	} else {
		SDL_SYS_UpdateRects = SDL_WindowUpdate;
	}

	/* Make our window the proper size, set the clipper, then show it */
	if ( (flags & SDL_FULLSCREEN) != SDL_FULLSCREEN ) {
		RECT bounds;
		int  x, y;

		/* Create and set a clipper on our primary surface */
		if ( SDL_clipper == NULL ) {
			result = IDirectDraw2_CreateClipper(ddraw2,
							0, &SDL_clipper, NULL);
			if ( result != DD_OK ) {
				if ( video != current ) {
					SDL_FreeSurface(video);
				}
				SetDDerror("DirectDraw2::CreateClipper",result);
				return(NULL);
			}
		}
		result = IDirectDrawClipper_SetHWnd(SDL_clipper, 0, SDL_Window);
		if ( result != DD_OK ) {
			if ( video != current ) {
				SDL_FreeSurface(video);
			}
			SetDDerror("DirectDrawClipper::SetHWnd", result);
			return(NULL);
		}
		result = IDirectDrawSurface3_SetClipper(SDL_primary,
								SDL_clipper);
		if ( result != DD_OK ) {
			if ( video != current ) {
				SDL_FreeSurface(video);
			}
			SetDDerror("DirectDrawSurface3::SetClipper", result);
			return(NULL);
		}

		/* Set the size of the window, centering and adjusting */
		SDL_resizing = 1;
		bounds.left = 0;
		bounds.top = 0;
		bounds.right = video->w;
		bounds.bottom = video->h;
		AdjustWindowRect(&bounds, GetWindowLong(SDL_Window, GWL_STYLE),
									FALSE);
		width = bounds.right-bounds.left;
		height = bounds.bottom-bounds.top;
		x = (GetSystemMetrics(SM_CXSCREEN)-width)/2;
		y = (GetSystemMetrics(SM_CYSCREEN)-height)/2;
		if ( y < 0 ) { /* Cover up title bar for more client area */
			y -= GetSystemMetrics(SM_CYCAPTION)/2;
		}
		SetWindowPos(SDL_Window, NULL, x, y, width, height,
					(SWP_NOCOPYBITS | SWP_NOZORDER));
		SDL_resizing = 0;
	}
	ShowWindow(SDL_Window, SW_SHOW);
	SetForegroundWindow(SDL_Window);

	/* We're live! */
	return(video);
}

struct private_hwdata {
	LPDIRECTDRAWSURFACE3 dd_surface;
	LPDIRECTDRAWSURFACE3 dd_writebuf;
};

static int SDL_AllocDDSurface(SDL_Surface *surface, 
				LPDIRECTDRAWSURFACE3 requested, Uint32 flag)
{
	LPDIRECTDRAWSURFACE  dd_surface1;
	LPDIRECTDRAWSURFACE3 dd_surface3;
	DDSURFACEDESC ddsd;
	HRESULT result;

	/* Clear the hardware flag, in case we fail */
	surface->flags &= ~flag;

	/* Allocate the hardware acceleration data */
	surface->hwdata = (struct private_hwdata *)
					malloc(sizeof(*surface->hwdata));
	if ( surface->hwdata == NULL ) {
		SDL_OutOfMemory();
		return(-1);
	}
	dd_surface3 = NULL;

	/* Set up the surface description */
	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = (DDSD_WIDTH|DDSD_HEIGHT|DDSD_CAPS|
					DDSD_PITCH|DDSD_PIXELFORMAT);
	ddsd.dwWidth = surface->w;
	ddsd.dwHeight= surface->h;
#if defined(NONAMELESSUNION)
	ddsd.u1.lPitch = surface->pitch;
#else
	ddsd.lPitch = surface->pitch;
#endif
	if ( (flag & SDL_HWSURFACE) == SDL_HWSURFACE ) {
		ddsd.ddsCaps.dwCaps =
				(DDSCAPS_OFFSCREENPLAIN|DDSCAPS_VIDEOMEMORY);
	} else {
		ddsd.ddsCaps.dwCaps =
				(DDSCAPS_OFFSCREENPLAIN|DDSCAPS_SYSTEMMEMORY);
	}
	ddsd.ddpfPixelFormat.dwSize = sizeof(ddsd.ddpfPixelFormat);
	ddsd.ddpfPixelFormat.dwFlags = DDPF_RGB;
	if ( surface->format->palette ) {
		ddsd.ddpfPixelFormat.dwFlags |= DDPF_PALETTEINDEXED8;
	}
#if defined(NONAMELESSUNION)
	ddsd.ddpfPixelFormat.u1.dwRGBBitCount = surface->format->BitsPerPixel;
	ddsd.ddpfPixelFormat.u2.dwRBitMask = surface->format->Rmask;
	ddsd.ddpfPixelFormat.u3.dwGBitMask = surface->format->Gmask;
	ddsd.ddpfPixelFormat.u4.dwBBitMask = surface->format->Bmask;
#else
	ddsd.ddpfPixelFormat.dwRGBBitCount = surface->format->BitsPerPixel;
	ddsd.ddpfPixelFormat.dwRBitMask = surface->format->Rmask;
	ddsd.ddpfPixelFormat.dwGBitMask = surface->format->Gmask;
	ddsd.ddpfPixelFormat.dwBBitMask = surface->format->Bmask;
#endif

	/* Create the DirectDraw video surface */
	if ( requested != NULL ) {
		dd_surface3 = requested;
	} else {
		result = IDirectDraw2_CreateSurface(ddraw2,
						&ddsd, &dd_surface1, NULL); 
		if ( result != DD_OK ) {
			SetDDerror("DirectDraw2::CreateSurface", result);
			goto error_end;
		}
		result = IDirectDrawSurface_QueryInterface(dd_surface1,
			&IID_IDirectDrawSurface3, (LPVOID *)&dd_surface3);
		IDirectDrawSurface_Release(dd_surface1);
		if ( result != DD_OK ) {
			SetDDerror("DirectDrawSurface::QueryInterface", result);
			goto error_end;
		}
	}

	if ( (flag & SDL_HWSURFACE) == SDL_HWSURFACE ) {
		/* Check to see whether the surface actually ended up
		   in video memory, and fail if not.  We expect the
		   surfaces we create here to actually be in hardware!
		*/
		result = IDirectDrawSurface3_GetCaps(dd_surface3,&ddsd.ddsCaps);
		if ( result != DD_OK ) {
			SetDDerror("DirectDrawSurface3::GetCaps", result);
			goto error_end;
		}
		if ( (ddsd.ddsCaps.dwCaps&DDSCAPS_VIDEOMEMORY) !=
							DDSCAPS_VIDEOMEMORY ) {
			SDL_SetError("No room in video memory");
			goto error_end;
		}
	} else {
		/* Try to hook our surface memory */
		ddsd.dwFlags = DDSD_LPSURFACE;
		ddsd.lpSurface = surface->pixels;
		result = IDirectDrawSurface3_SetSurfaceDesc(dd_surface3,
								&ddsd, 0);
		if ( result != DD_OK ) {
			SetDDerror("DirectDraw2::SetSurfaceDesc", result);
			goto error_end;
		}
	
	}

	/* Make sure the surface format was set properly */
	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	result = IDirectDrawSurface3_Lock(dd_surface3, NULL,
					&ddsd, DDLOCK_NOSYSLOCK, NULL);
	if ( result != DD_OK ) {
		SetDDerror("DirectDrawSurface3::Lock", result);
		goto error_end;
	}
	IDirectDrawSurface3_Unlock(dd_surface3, NULL);

	if ( (flag & SDL_HWSURFACE) == SDL_SWSURFACE ) {
		if ( ddsd.lpSurface != surface->pixels ) {
			SDL_SetError("DDraw didn't use SDL surface memory");
			goto error_end;
		}
	}
#if defined(NONAMELESSUNION)
	if ( (ddsd.ddpfPixelFormat.u1.dwRGBBitCount != 
					surface->format->BitsPerPixel) ||
	     (ddsd.ddpfPixelFormat.u2.dwRBitMask != surface->format->Rmask) ||
	     (ddsd.ddpfPixelFormat.u3.dwGBitMask != surface->format->Gmask) ||
	     (ddsd.ddpfPixelFormat.u4.dwBBitMask != surface->format->Bmask) ){
#else
	if ( (ddsd.ddpfPixelFormat.dwRGBBitCount != 
					surface->format->BitsPerPixel) ||
	     (ddsd.ddpfPixelFormat.dwRBitMask != surface->format->Rmask) ||
	     (ddsd.ddpfPixelFormat.dwGBitMask != surface->format->Gmask) ||
	     (ddsd.ddpfPixelFormat.dwBBitMask != surface->format->Bmask) ){
#endif
		SDL_SetError("DDraw didn't use SDL surface description");
		goto error_end;
	}
	if ( (ddsd.dwWidth != (DWORD)surface->w) ||
	     (ddsd.dwHeight != (DWORD)surface->h) ||
#if defined(NONAMELESSUNION)
	     (ddsd.u1.lPitch != (LONG)surface->pitch) ) {
#else
	     (ddsd.lPitch != (LONG)surface->pitch) ) {
#endif
		SDL_SetError("DDraw created surface with wrong size");
		goto error_end;
	}

	/* Set the surface private data */
	surface->flags |= flag;
	surface->hwdata->dd_surface = dd_surface3;
	if ( (surface->flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF ) {
		LPDIRECTDRAWSURFACE3 dd_writebuf;

		ddsd.ddsCaps.dwCaps = DDSCAPS_BACKBUFFER;
		result = IDirectDrawSurface3_GetAttachedSurface(dd_surface3,
						&ddsd.ddsCaps, &dd_writebuf);
		if ( result != DD_OK ) {
			SetDDerror("DirectDrawSurface3::GetAttachedSurface",
								result);
		} else {
			dd_surface3 = dd_writebuf;
		}
	}
	surface->hwdata->dd_writebuf = dd_surface3;

	/* We're ready to go! */
	return(0);

	/* Okay, so goto's are cheesy, but there are so many possible
	   errors in this function, and the cleanup is the same in 
	   every single case.  Is there a better way, other than deeply
	   nesting the code?
	*/
error_end:
	if ( (dd_surface3 != NULL) && (dd_surface3 != requested) ) {
		IDirectDrawSurface_Release(dd_surface3);
	}
	free(surface->hwdata);
	surface->hwdata = NULL;
	return(-1);
}

static int SDL_AllocHWSurface(SDL_Surface *surface)
{
	/* DDraw limitation -- you need to set cooperative level first */
	if ( SDL_primary == NULL ) {
		SDL_SetError("You must set the video mode first");
		return(-1);
	}
	return(SDL_AllocDDSurface(surface, NULL, SDL_HWSURFACE));
}

void PrintSurface(char *title, LPDIRECTDRAWSURFACE3 surface, Uint32 flags)
{
	DDSURFACEDESC ddsd;

	/* Lock and load! */
	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	if ( IDirectDrawSurface3_Lock(surface, NULL, &ddsd,
			(DDLOCK_NOSYSLOCK|DDLOCK_WAIT), NULL) != DD_OK ) {
		return;
	}
	IDirectDrawSurface3_Unlock(surface, NULL);
	
	fprintf(stderr, "%s:\n", title);
	fprintf(stderr, "\tSize: %dx%d in %s at %d bpp (pitch = %d)\n",
		ddsd.dwWidth, ddsd.dwHeight,
		(flags & SDL_HWSURFACE) ? "hardware" : "software",
#if defined(NONAMELESSUNION)
		ddsd.ddpfPixelFormat.u1.dwRGBBitCount, ddsd.u1.lPitch);
#else
		ddsd.ddpfPixelFormat.dwRGBBitCount, ddsd.lPitch);
#endif
	fprintf(stderr, "\tR = 0x%p, G = 0x%p, B = 0x%p\n", 
#if defined(NONAMELESSUNION)
	     		ddsd.ddpfPixelFormat.u2.dwRBitMask,
	     		ddsd.ddpfPixelFormat.u3.dwGBitMask,
	     		ddsd.ddpfPixelFormat.u4.dwBBitMask);
#else
	     		ddsd.ddpfPixelFormat.dwRBitMask,
	     		ddsd.ddpfPixelFormat.dwGBitMask,
	     		ddsd.ddpfPixelFormat.dwBBitMask);
#endif
}

static int SDL_HWAccelBlit(SDL_Surface *src, SDL_Rect *srcrect,
					SDL_Surface *dst, SDL_Rect *dstrect)
{
	LPDIRECTDRAWSURFACE3 src_surface;
	LPDIRECTDRAWSURFACE3 dst_surface;
	DWORD flags;
	RECT rect;
	HRESULT result;

	/* Set it up.. the desination must have a DDRAW surface */
	src_surface = src->hwdata->dd_writebuf;
	dst_surface = dst->hwdata->dd_writebuf;
	rect.top    = srcrect->y;
	rect.bottom = srcrect->y+srcrect->h;
	rect.left   = srcrect->x;
	rect.right  = srcrect->x+srcrect->w;
	if ( (src->flags & SDL_SRCCOLORKEY) == SDL_SRCCOLORKEY )
		flags = DDBLTFAST_SRCCOLORKEY;
	else
		flags = DDBLTFAST_NOCOLORKEY;
	/* FIXME:  We can remove this flag for _really_ fast blit queuing,
	           but it will affect the return values of locks and flips.
	 */
	flags |= DDBLTFAST_WAIT;

	/* Do the blit! */
	result = IDirectDrawSurface3_BltFast(dst_surface,
			dstrect->x, dstrect->y, src_surface, &rect, flags);
	if ( result != DD_OK ) {
		if ( result == DDERR_SURFACELOST ) {
			result = IDirectDrawSurface3_Restore(src_surface);
			result = IDirectDrawSurface3_Restore(dst_surface);
			/* The surfaces need to be reloaded with artwork */
			SDL_SetError("Blit surfaces were lost, reload them");
			return(-2);
		}
		SetDDerror("IDirectDrawSurface3::BltFast", result);
#ifdef DDRAW_DEBUG
 fprintf(stderr, "Original dest rect: %dx%d at %d,%d\n", dstrect->w, dstrect->h, dstrect->x, dstrect->y);
 fprintf(stderr, "HW accelerated %sblit to from 0x%p to 0x%p at (%d,%d)\n",
		(src->flags & SDL_SRCCOLORKEY) ? "colorkey " : "", src, dst,
					dstrect->x, dstrect->y);
  PrintSurface("SRC", src_surface, src->flags);
  PrintSurface("DST", dst_surface, dst->flags);
 fprintf(stderr, "Source rectangle: (%d,%d) - (%d,%d)\n",
		rect.left, rect.top, rect.right, rect.bottom);
#endif
		/* Unexpected error, fall back to software blit */
		return(src->map->sw_blit(src, srcrect, dst, dstrect));
	}
	return(0);
}

static int SDL_CheckHWBlit(SDL_Surface *src, SDL_Surface *dst)
{
	int accelerated;

	/* We need to have a DDraw surface for HW blits */
	if ( (src->flags & SDL_HWSURFACE) == SDL_SWSURFACE ) {
		/* Allocate a DDraw surface for the blit */
		if ( src->hwdata == NULL ) {
			SDL_AllocDDSurface(src, NULL, SDL_SWSURFACE);
		}
	}
	if ( src->hwdata == NULL ) {
		return(0);
	}

	/* Set initial acceleration on */
	src->flags |= SDL_HWACCEL;

	/* Set the surface attributes */
	if ( (src->flags & SDL_SRCCOLORKEY) == SDL_SRCCOLORKEY ) {
		if ( SDL_SetHWColorKey(src, src->format->colorkey) < 0 ) {
			src->flags &= ~SDL_HWACCEL;
		}
	}
	if ( (src->flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
		if ( SDL_SetHWAlpha(src, src->format->alpha) < 0 ) {
			src->flags &= ~SDL_HWACCEL;
		}
	}

	/* Check to see if final surface blit is accelerated */
	accelerated = !!(src->flags & SDL_HWACCEL);
	if ( accelerated ) {
#ifdef DDRAW_DEBUG
  fprintf(stderr, "Setting accelerated blit on 0x%p\n", src);
#endif
		src->map->hw_blit = SDL_HWAccelBlit;
	}
	return(accelerated);
}

static int SDL_FillHWRect(SDL_Surface *dst, SDL_Rect *dstrect, Uint32 color)
{
	LPDIRECTDRAWSURFACE3 dst_surface;
	RECT area;
	DDBLTFX bltfx;
	HRESULT result;

#ifdef DDRAW_DEBUG
 fprintf(stderr, "HW accelerated fill at (%d,%d)\n", dstrect->x, dstrect->y);
#endif
	dst_surface = dst->hwdata->dd_writebuf;
	area.top = dstrect->y;
	area.bottom = dstrect->y+dstrect->h;
	area.left = dstrect->x;
	area.right = dstrect->x+dstrect->w;
	bltfx.dwSize = sizeof(bltfx);
#if defined(NONAMELESSUNION)
	bltfx.u5.dwFillColor = color;
#else
	bltfx.dwFillColor = color;
#endif
	result = IDirectDrawSurface3_Blt(dst_surface,
			&area, NULL, NULL, DDBLT_COLORFILL|DDBLT_WAIT, &bltfx);
	if ( result == DDERR_SURFACELOST ) {
		IDirectDrawSurface3_Restore(dst_surface);
		result = IDirectDrawSurface3_Blt(dst_surface,
			&area, NULL, NULL, DDBLT_COLORFILL|DDBLT_WAIT, &bltfx);
	}
	if ( result != DD_OK ) {
		SetDDerror("IDirectDrawSurface3::Blt", result);
		return(-1);
	}
	return(0);
}

static int SDL_SetHWColorKey(SDL_Surface *surface, Uint32 key)
{
	DDCOLORKEY colorkey;
	HRESULT result;

	/* Set the surface colorkey */
	colorkey.dwColorSpaceLowValue = key;
	colorkey.dwColorSpaceHighValue = key;
	result = IDirectDrawSurface3_SetColorKey(
			surface->hwdata->dd_surface, DDCKEY_SRCBLT, &colorkey);
	if ( result != DD_OK ) {
		SetDDerror("IDirectDrawSurface3::SetColorKey", result);
		return(-1);
	}
	return(0);
}
static int SDL_SetHWAlpha(SDL_Surface *surface, Uint32 alpha)
{
	return(-1);
}

static int SDL_LockHWSurface(SDL_Surface *surface)
{
	HRESULT result;
	LPDIRECTDRAWSURFACE3 dd_surface;
	DDSURFACEDESC ddsd;

	/* Lock and load! */
	dd_surface = surface->hwdata->dd_writebuf;
	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);
	result = IDirectDrawSurface3_Lock(dd_surface, NULL, &ddsd,
					(DDLOCK_NOSYSLOCK|DDLOCK_WAIT), NULL);
	if ( result == DDERR_SURFACELOST ) {
		result = IDirectDrawSurface3_Restore(
						surface->hwdata->dd_surface);
		result = IDirectDrawSurface3_Lock(dd_surface, NULL, &ddsd, 
					(DDLOCK_NOSYSLOCK|DDLOCK_WAIT), NULL);
	}
	if ( result != DD_OK ) {
		SetDDerror("DirectDrawSurface3::Lock", result);
		return(-1);
	}
	/* Pitch might have changed -- recalculate pitch and offset */
#if defined(NONAMELESSUNION)
	if ( surface->pitch != ddsd.u1.lPitch ) {
		surface->pitch = ddsd.u1.lPitch;
#else
	if ( surface->pitch != ddsd.lPitch ) {
		surface->pitch = (Uint16)ddsd.lPitch;
#endif
		surface->offset =
			((ddsd.dwHeight-surface->h)/2)*surface->pitch +
			((ddsd.dwWidth-surface->w)/2)*
					surface->format->BytesPerPixel;
	}
	surface->pixels = ddsd.lpSurface;
	return(0);
}

static void SDL_UnlockHWSurface(SDL_Surface *surface)
{
	IDirectDrawSurface3_Unlock(surface->hwdata->dd_writebuf, NULL);
	surface->pixels = NULL;
}

static int SDL_FlipHWSurface(SDL_Surface *surface)
{
	HRESULT result;
	LPDIRECTDRAWSURFACE3 dd_surface;

	dd_surface = surface->hwdata->dd_surface;
	result = IDirectDrawSurface3_Flip(dd_surface, NULL, DDFLIP_WAIT);
	if ( result == DDERR_SURFACELOST ) {
		result = IDirectDrawSurface3_Restore(
						surface->hwdata->dd_surface);
		result = IDirectDrawSurface3_Flip(dd_surface,NULL,DDFLIP_WAIT);
	}
	if ( result != DD_OK ) {
		SetDDerror("DirectDrawSurface3::Flip", result);
		return(-1);
	}
	return(0);
}

static void SDL_FreeHWSurface(SDL_Surface *surface)
{
	if ( surface->hwdata->dd_surface != SDL_primary ) {
		IDirectDrawSurface3_Release(surface->hwdata->dd_surface);
	}
	free(surface->hwdata);
}

void SDL_WindowUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	int i;
	RECT src, dst;

	for ( i=0; i<numrects; ++i ) {
		src.top = rects[i].y;
		src.bottom = rects[i].y+rects[i].h;
		src.left = rects[i].x;
		src.right = rects[i].x+rects[i].w;
		dst.top = SDL_bounds.top+src.top;
		dst.left = SDL_bounds.left+src.left;
		dst.bottom = SDL_bounds.top+src.bottom;
		dst.right = SDL_bounds.left+src.right;
		IDirectDrawSurface3_Blt(SDL_primary, &dst, 
					screen->hwdata->dd_surface, &src,
							DDBLT_WAIT, NULL);
	}
}

void SDL_DirectUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
#if 0	/* This should no longer be necessary */

	/* If the application is writing to system memory, copy to video */
	if ( ((screen->flags & SDL_HWSURFACE) == SDL_SWSURFACE) && 
					(SDL_GetAppState()&SDL_APPACTIVE) ) {
		int i;
		HRESULT result;
		RECT src;

		for ( i=0; i<numrects; ++i ) {
			src.top = rects[i].y;
			src.bottom = rects[i].y+rects[i].h;
			src.left = rects[i].x;
			src.right = rects[i].x+rects[i].w;
			result = IDirectDrawSurface3_BltFast(
					SDL_primary, src.left, src.top,
					screen->hwdata->dd_surface, &src,
					(DDBLTFAST_NOCOLORKEY|DDBLTFAST_WAIT));
			if ( result == DDERR_SURFACELOST ) {
				result=IDirectDrawSurface3_Restore(SDL_primary);
				result=IDirectDrawSurface3_BltFast(
					SDL_primary, src.left, src.top,
					screen->hwdata->dd_surface, &src,
					(DDBLTFAST_NOCOLORKEY|DDBLTFAST_WAIT));
			}
			if ( result != DD_OK ) {
				SetDDerror("IDirectDrawSurface3::BltFast",
								result);
			}
		}
	}
#endif
}

/* Compress a full palette into the limited number of colors given to us
   by windows.

   The "best" way to do this is to sort the colors by diversity and place
   the most diverse colors into the limited palette.  Unfortunately this
   results in widely varying colors being displayed in the interval during
   which the windows palette has been set, and the mapping of the shadow
   surface to the new palette.  This is especially noticeable during fades.

   To deal with this problem, we can copy a predetermined portion of the
   full palette, and use that as the limited palette.  This allows colors
   to fade smoothly as the remapping is very similar on each palette change.
   Unfortunately, this breaks applications which partition the palette into
   distinct and widely varying areas, expecting all colors to be available.

   I'm making them both available, chosen at compile time.
   If you want the chunk-o-palette algorithm, define SIMPLE_COMPRESSION,
   otherwise the sort-by-diversity algorithm will be used.
*/
#define SIMPLE_COMPRESSION
#define CS_CS_DIST(A, B) ({						\
	int r = (A.r - B.r);						\
	int g = (A.g - B.g);						\
	int b = (A.b - B.b);						\
	(r*r + g*g + b*b);						\
})
static void SDL_CompressPalette(SDL_Color *colors, int ncolors, int maxcolors)
{
#ifdef SIMPLE_COMPRESSION
	int i, j;
#else
	static SDL_Color zero = { 0, 0, 0, 0 };
	int i, j;
	int max, dist;
	int prev, next;
	int *pool;
	int *seen, *order;
#endif

	/* Does this happen? */
	if ( maxcolors > ncolors ) {
		maxcolors = ncolors;
	}

#ifdef SIMPLE_COMPRESSION
	/* Just copy the first "maxcolors" colors */
	for ( j=10, i=0; i<maxcolors; ++i, ++j ) {
		SDL_colors[j].peRed = colors[i].r;
		SDL_colors[j].peGreen = colors[i].g;
		SDL_colors[j].peBlue = colors[i].b;
	}
#else
	/* Allocate memory for the arrays we use */
	pool = (int *)alloca(2*ncolors*sizeof(int));
	if ( pool == NULL ) {
		/* No worries, just return */;
		return;
	}
	seen = pool;
	memset(seen, 0, ncolors*sizeof(int));
	order = pool+ncolors;

	/* Start with the brightest color */
	max = 0;
	for ( i=0; i<ncolors; ++i ) {
		dist = CS_CS_DIST(zero, colors[i]);
		if ( dist >= max ) {
			max = dist;
			next = i;
		}
	}
	j = 0;
	order[j++] = next;
	seen[next] = 1;
	prev = next;

	/* Keep going through all the colors */
	while ( j < maxcolors ) {
		max = 0;
		for ( i=0; i<ncolors; ++i ) {
			if ( seen[i] ) {
				continue;
			}
			dist = CS_CS_DIST(colors[i], colors[prev]);
			if ( dist >= max ) {
				max = dist;
				next = i;
			}
		}
		order[j++] = next;
		seen[next] = 1;
		prev = next;
	}

	/* Compress the colors to the palette */
	for ( j=10, i=0; i<maxcolors; ++i, ++j ) {
		SDL_colors[j].peRed = colors[order[i]].r;
		SDL_colors[j].peGreen = colors[order[i]].g;
		SDL_colors[j].peBlue = colors[order[i]].b;
	}
#endif /* SIMPLE_COMPRESSION */
}

/* Set the system colormap in both fullscreen and windowed modes */
int SDL_SYS_SetColors(SDL_Surface *screen, int firstcolor, int ncolors)
{
	SDL_Palette *palette;
	int i;
	int alloct_all;

	/* Copy palette colors into display palette */
	palette = screen->format->palette;
	if ( SDL_palette != NULL ) {
		if ( (screen->flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
			/* We can set all entries explicitly */
			for ( i=firstcolor; i<(firstcolor+ncolors); ++i ) {
				SDL_colors[i].peRed = palette->colors[i].r;
				SDL_colors[i].peGreen = palette->colors[i].g;
				SDL_colors[i].peBlue = palette->colors[i].b;
			}
			IDirectDrawPalette_SetEntries(SDL_palette, 0,
				firstcolor, ncolors, &SDL_colors[firstcolor]);
			alloct_all = 1;
		} else {
			/* Grab the 236 most diverse colors in the palette */
			SDL_CompressPalette(palette->colors, palette->ncolors,
									236);
			/* This sends an WM_PALETTECHANGED message to us */
			colorchange_expected = 1;
			IDirectDrawPalette_SetEntries(SDL_palette, 0,
							0, 256, SDL_colors);
			alloct_all = 0;
		}
	}
	return(alloct_all);
}

void SDL_SYS_VideoQuit(SDL_Surface *screen)
{
	int i, j;

	/* Free any palettes we used */
	if ( SDL_palette != NULL ) {
		IDirectDrawPalette_Release(SDL_palette);
		SDL_palette = NULL;
	}

	/* Allow the primary surface to be freed */
	if ( SDL_primary != NULL ) {
		SDL_primary = NULL;
	}

	/* Free video mode lists */
	for ( i=0; i<NUM_MODELISTS; ++i ) {
		if ( SDL_modelist[i] != NULL ) {
			for ( j=0; SDL_modelist[i][j]; ++j )
				free(SDL_modelist[i][j]);
			free(SDL_modelist[i]);
			SDL_modelist[i] = NULL;
		}
	}

	/* Free the window */
	if ( SDL_Window ) {
		SDL_DestroyWindow();
	}

	/* Free our window icon */
	if ( screen_icn ) {
		DestroyIcon(screen_icn);
		screen_icn = NULL;
	}
}

void SDL_SYS_FinalQuit(void)
{
	/* Free DirectDraw object */
	if ( ddraw2 != NULL ) {
		IDirectDraw2_Release(ddraw2);
		ddraw2 = NULL;
	}
}

/* Exported for the windows message loop only */
void SDL_RealizePalette(void)
{
	if ( SDL_palette ) {
		IDirectDrawSurface3_SetPalette(SDL_primary, SDL_palette);
	}
}
static void SDL_Recolor8Bit(SDL_Surface *surface, Uint8 *mapping)
{
	int row, col;
	Uint8 *pixels;

	if ( surface->w && surface->h ) {
		if ( (surface->flags & SDL_HWSURFACE) == SDL_HWSURFACE ) {
			if ( SDL_HWCaps.LockHWSurface(surface) < 0 ) {
				return;
			}
		}
		for ( row=0; row<surface->h; ++row ) {
			pixels = (Uint8 *)surface->pixels+row*surface->pitch;
			for ( col=0; col<surface->w; ++col, ++pixels ) {
				*pixels = mapping[*pixels];
			}
		}
		if ( (surface->flags & SDL_HWSURFACE) == SDL_HWSURFACE ) {
			SDL_HWCaps.UnlockHWSurface(surface);
		}
		SDL_UpdateRect(surface, 0, 0, 0, 0);
	}
}
void SDL_PaletteChanged(HWND window)
{
	SDL_Palette *palette;
	SDL_Color *saved;
	HDC hdc;
	int i;
	PALETTEENTRY *entries;

	/* This is true when the window is closing */
	if ( (SDL_primary == NULL) || (SDL_VideoSurface == NULL) )
		return;

	/* We need to get the colors as they were set */
	palette = SDL_VideoSurface->format->palette;
	if ( palette == NULL ) { /* Sometimes we don't have a palette */
		return;
	}
	entries = (PALETTEENTRY *)alloca(palette->ncolors*sizeof(*entries));
	hdc = GetDC(window);
	GetSystemPaletteEntries(hdc, 0, palette->ncolors, entries);
	ReleaseDC(window, hdc);
	if ( ! colorchange_expected ) {
		saved = (SDL_Color *)alloca(palette->ncolors*sizeof(SDL_Color));
		memcpy(saved, palette->colors, 
					palette->ncolors*sizeof(SDL_Color));
	}
	for ( i=0; i<palette->ncolors; ++i ) {
		palette->colors[i].r = entries[i].peRed;
		palette->colors[i].g = entries[i].peGreen;
		palette->colors[i].b = entries[i].peBlue;
	}
	if ( ! colorchange_expected ) {
		Uint8 mapping[256];

		memset(mapping, 0, sizeof(mapping));
		for ( i=0; i<palette->ncolors; ++i ) {
			mapping[i] = SDL_FindColor(palette,
					saved[i].r, saved[i].g, saved[i].b);
		}
		SDL_Recolor8Bit(SDL_VideoSurface, mapping);
	}
	colorchange_expected = 0;

	/* Notify all mapped surfaces of the change */
	SDL_FormatChanged(SDL_VideoSurface);
}

/* Exported for the windows message loop only */
void SDL_WinPAINT(HDC hdc)
{
	SDL_UpdateRect(SDL_PublicSurface, 0, 0, 0, 0);
}
