/* File "malaga_files.h":
 * Defines the structure of compiled Malaga files. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* the header of a Malaga file ==============================================*/

#define MALAGA_LEN 6 /* magic code at the beginning of a Malaga file */

typedef struct /* the common header of every Malaga file */
{
  char malaga[MALAGA_LEN];    /* "MALAGA" to recognise Malaga files */
  short_t file_type;          /* SYMBOL_FILE, RULE_FILE or LEXICON_FILE */
  long_t code_version;        /* only load code of the current version */
  time_t symbol_compile_time; /* time when symbol file was compiled */
} common_header_t;

/* values for <file_type> */
#define SYMBOL_FILE 0
#define RULE_FILE 1
#define LEXICON_FILE 2

/* the format of a Malaga rule file =========================================*/

typedef struct 
{
  common_header_t common_header;
  long_t initial_rule_set; /* index of the initial rule set in rule_set */
  long_t initial_cat;      /* value index of the initial combi rule category */
  long_t robust_rule;      /* rule number of robust_rule */
  long_t pruning_rule;     /* rule number of pruning_rule */
  long_t allo_rule;        /* rule number of allo_rule */
  long_t filter_rule;      /* rule number of filter_rule */
  long_t input_rule;       /* rule number of input_rule */
  long_t rules_size;       /* number of rules in this file */
  long_t rule_sets_size;   /* size of rule set table */
  long_t instrs_size;      /* number of instructions in this file */
  long_t values_size;      /* size of Malaga value table */
  long_t src_lines_size;   /* number of correspondences
			    * between source lines and rule code */
  long_t vars_size;        /* number of variable names */
  long_t var_scopes_size;  /* number of variable scopes */
  long_t strings_size;     /* size of string table */

  /* the following blocks are of dynamic size: */
  /* rule_t rules [rules_size]; */
  /* long_t rule_sets [rule_sets_size]; */
  /* instr_t instrs [instrs_size]; */
  /* cell_t values [values_size]; */
  /* src_line_t src_lines [src_lines_size]; */
  /* var_t vars [vars_size]; */
  /* var_scope_t var_scopes [var_scopes_size]; */
  /* char strings [strings_size]; */
} rule_header_t;

#define RULE_CODE_VERSION 36

/* the format of a Malaga symbol file =======================================*/

typedef struct 
{
  common_header_t common_header;
  long_t symbols_size; /* number of symbols in this file */
  long_t values_size;  /* size of Malaga value table
			* (used to define multi-symbols) */
  long_t strings_size; /* size of string table (used to define symbol names) */

  /* the following blocks are of dynamic size:
   * symbol_entry_t symbols [symbols_size];
   * cell_t values [values_size];
   * char strings [strings_size]; */
} symbol_header_t;

#define SYMBOL_CODE_VERSION 5

/* the format of a Malaga lexicon file ======================================*/

typedef struct
{ 
  common_header_t common_header;
  long_t trie_size;      /* size of trie table */
  long_t trie_root;      /* index of root node in <trie> */
  long_t cat_lists_size; /* size of cat lists table */
  long_t values_size;    /* size of value table */
  long_t strings_size;   /* size of string table */

  /* the following blocks are of dynamic size:
   * long_t trie [trie_size];
   * long_t cat_lists [cat_lists_size];
   * cell_t values [values_size];
   * char strings [strings_size]; */
} lexicon_header_t;

#define LEXICON_CODE_VERSION 9

/* functions ================================================================*/

extern void check_header (common_header_t *header, 
			  string_t file_name, 
			  short_t file_type, 
			  long_t code_version);
/* Check if <header> is of <file_type> and <code_version>.
 * <file_name> is needed for error messages. */

extern void set_header (common_header_t *header, 
			short_t file_type,
  			long_t code_version);
/* Set header to be of <file_type> and <code_version>. */

/*---------------------------------------------------------------------------*/
