/* $Id: internal.c,v 1.6 1999/03/04 02:11:34 marcus Exp $
******************************************************************************

   Misc internal-only functions

   Copyright (C) 1998-1999  Andrew Apted  [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/internal.h>


int _ggi_mask2shift(uint32 mask)
{
	int shift;
	for (shift = 32; mask != 0; mask >>= 1) {
		shift--;
	}
	if (shift == 32) shift = 0;

	return shift;
}

void _ggi_build_pixfmt(ggi_pixelformat *pixfmt)
{
	int i, j = 0, bmsub = 0, oldbmsub = 0, bmtype = 0, oldbmtype = 0;

	for (i = 0; i < pixfmt->depth; i++) {
		int colsize = 0, h;
		if (pixfmt->clut_mask & (1<<i)) {
			bmsub = GGI_BM_SUB_CLUT;
			bmtype = GGI_BM_TYPE_COLOR;
			for (colsize = 0, h = i; (pixfmt->clut_mask & (1<<h));
			     h++, colsize++) ;
		} else if (pixfmt->red_mask & (1<<i)) {
			bmsub = GGI_BM_SUB_RED;
			bmtype = GGI_BM_TYPE_COLOR;
			for (colsize = 0, h = i; (pixfmt->red_mask & (1<<h));
			     h++, colsize++) ;
		} else if (pixfmt->green_mask & (1<<i)) {
			bmsub = GGI_BM_SUB_GREEN;
			bmtype = GGI_BM_TYPE_COLOR;
			for (colsize = 0, h = i; (pixfmt->green_mask & (1<<h));
			     h++, colsize++) ;
		} else if (pixfmt->blue_mask & (1<<i)) {
			bmsub = GGI_BM_SUB_BLUE;
			bmtype = GGI_BM_TYPE_COLOR;
			for (colsize = 0, h = i; (pixfmt->blue_mask & (1<<h));
			     h++, colsize++) ;
		} else if (pixfmt->alpha_mask & (1<<i)) {
			bmsub = GGI_BM_SUB_ALPHA;
			bmtype = GGI_BM_TYPE_ATTRIB;
			for (colsize = 0, h = i; (pixfmt->alpha_mask & (1<<h));
			     h++, colsize++) ;
		} else if (pixfmt->fg_mask & (1<<i)) {
			bmsub = GGI_BM_SUB_FGCOL;
			bmtype = GGI_BM_TYPE_ATTRIB;
			for (colsize = 0, h = i; (pixfmt->fg_mask & (1<<h));
			     h++, colsize++) ;
		} else if (pixfmt->bg_mask & (1<<i)) {
			bmsub = GGI_BM_SUB_BGCOL;
			bmtype = GGI_BM_TYPE_ATTRIB;
			for (colsize = 0, h = i; (pixfmt->bg_mask & (1<<h));
			     h++, colsize++) ;
		} else if (pixfmt->texture_mask & (1<<i)) {
			bmsub = GGI_BM_SUB_TEXNUM;
			bmtype = GGI_BM_TYPE_ATTRIB;
			for (colsize = 0, h = i;
			     (pixfmt->texture_mask & (1<<h)); h++, colsize++) ;
		} else {
			bmsub = 0;
		}
		if (bmsub != oldbmsub || bmtype != oldbmtype) {
			j = 256-colsize;
			oldbmsub = bmsub;
			oldbmtype = bmtype;
		}
		if (bmsub) {
			pixfmt->bitmeaning[i]
				= bmtype | bmsub | j;
			j++;
		}
	}
	pixfmt->red_shift     = _ggi_mask2shift(pixfmt->red_mask);
	pixfmt->green_shift   = _ggi_mask2shift(pixfmt->green_mask);
	pixfmt->blue_shift    = _ggi_mask2shift(pixfmt->blue_mask);
	pixfmt->alpha_shift   = _ggi_mask2shift(pixfmt->alpha_mask);
	pixfmt->clut_shift    = _ggi_mask2shift(pixfmt->clut_mask);
	pixfmt->fg_shift      = _ggi_mask2shift(pixfmt->fg_mask);
	pixfmt->bg_shift      = _ggi_mask2shift(pixfmt->bg_mask);
	pixfmt->texture_shift = _ggi_mask2shift(pixfmt->texture_mask);
}

int _ggi_match_palette(ggi_color *pal, int pal_len, ggi_color *col)
{
	int i, closest=0;
	int r = col->r, g = col->g, b = col->b;

	uint32 closest_dist = (1 << 31);

	for (i=0; i < pal_len; i++) {
#undef ABS
#define ABS(val)	((val) < 0 ? -(val) : val)
		int dist =
			ABS(r - pal[i].r) +
			ABS(g - pal[i].g) +
			ABS(b - pal[i].b);
#undef ABS

		if (dist < closest_dist) {
			closest = i;
			if (dist == 0) {
				/* Exact match */
				break;
			}
			closest_dist = dist;
		}
	}

	GGIDPRINT_COLOR("match-color: %02x%02x%02x -> %02x%02x%02x (%d).\n",
		     col->r >> 8, col->g >> 8, col->b >> 8,
		     pal[closest].r >> 8, pal[closest].g >> 8,
		     pal[closest].b >> 8, closest);

	return closest;
}

int _ggi_default_setreadframe(ggi_visual *vis, int num)
{
        ggi_directbuffer *db = _ggi_db_find_frame(vis, num);

        if (db == NULL) {
                return -1;
        }

        vis->r_frame_num = num;
        vis->r_frame = db;

        return 0;
}

int _ggi_default_setwriteframe(ggi_visual *vis, int num)
{
        ggi_directbuffer *db = _ggi_db_find_frame(vis, num);

        if (db == NULL) {
                return -1;
        }

        vis->w_frame_num = num;
        vis->w_frame = db;

        return 0;
}

void _ggi_smart_match_palettes(ggi_color *pal, int size,
                               ggi_color *ref_pal, int ref_size)
{
	/* This function moves the color entries in the palette `pal'
	 * around so to produce the best match against the reference
	 * palette.  Lower indexed entries in the reference palette have
	 * higher priority than higher indexed entries, so they are
	 * matched first.  The reference palette can be smaller *or*
	 * larger than the one to be re-arranged.
	 */
	
	int i;
	int smallest = (size < ref_size) ? size : ref_size;

	for (i=0; i < smallest; i++) {

		ggi_color tmp;

		int n = i + _ggi_match_palette(pal + i, size - i, 
		                               ref_pal + i);

		tmp = pal[i]; pal[i] = pal[n]; pal[n] = tmp;
	}
}

void _ggi_build_palette(ggi_color *pal, int num)
{
	ggi_color black  = { 0x0000, 0x0000, 0x0000 };
	ggi_color white  = { 0xffff, 0xffff, 0xffff };
	ggi_color blue   = { 0x0000, 0x0000, 0xffff };
	ggi_color yellow = { 0xffff, 0xffff, 0x0000 };

	int i, depth, N;

	int rnum,  gnum,  bnum;
	int rmask, gmask, bmask;
	
	/* handle small palettes */

	if (num == 0) return;

	pal[0] = black;   if (num == 1) return;
	pal[1] = white;   if (num == 2) return;
	pal[2] = blue;    if (num == 3) return;
	pal[3] = yellow;  if (num == 4) return;

	/* handle large palettes. */

	depth = 0;  /* work out depth so that (1 << depth) >= num */

	for (N = num-1; N > 0; N /= 2) {
		depth++;
	}

	gnum = (depth + 2) / 3;  /* green has highest priority */
	rnum = (depth + 1) / 3;  /* red has second highest priority */
	bnum = (depth + 0) / 3;  /* blue has lowest priority */

	gmask = (1 << gnum) - 1;
	rmask = (1 << rnum) - 1;
	bmask = (1 << bnum) - 1;

	for (i=0; i < num; i++) {

		/* When num < (1 << depth), we interpolate the values so
		 * that we still get a good range.  There's probably a
		 * better way...
		 */
		
		int j = i * ((1 << depth) - 1) / (num - 1);
		int r, g, b;
		
		b = j & bmask;  j >>= bnum;
		r = j & rmask;  j >>= rnum;
		g = j & gmask;

		pal[i].r = r * 0xffff / rmask;
		pal[i].g = g * 0xffff / gmask;
		pal[i].b = b * 0xffff / bmask;
	}
}
