/* 
   objc-api.h

   Copyright (C) 1997 NET-Community

   Author: Ovidiu Predescu <ovidiu@net-community.com>

   This file is part of the Foundation Extensions Library.

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies and that both that
   copyright notice and this permission notice appear in supporting
   documentation.

   We disclaim all warranties with regard to this software, including all
   implied warranties of merchantability and fitness, in no event shall
   we be liable for any special, indirect or consequential damages or any
   damages whatsoever resulting from loss of use, data or profits, whether in
   an action of contract, negligence or other tortious action, arising out of
   or in connection with the use or performance of this software.
*/

#ifndef __extensions_objc_api_h__
#define __extensions_objc_api_h__

#include <stdarg.h>
#include <objc/objc.h>

/* Filer types used to describe Ivars and Methods.  */
#define _C_ID       '@'
#define _C_CLASS    '#'
#define _C_SEL      ':'
#define _C_CHR      'c'
#define _C_UCHR     'C'
#define _C_SHT      's'
#define _C_USHT     'S'
#define _C_INT      'i'
#define _C_UINT     'I'
#define _C_LNG      'l'
#define _C_ULNG     'L'
#define _C_FLT      'f'
#define _C_DBL      'd'
#define _C_BFLD     'b'
#define _C_VOID     'v'
#define _C_UNDEF    '?'
#define _C_PTR      '^'
#define _C_CHARPTR  '*'
#define _C_ATOM     '%'
#define _C_ARY_B    '['
#define _C_ARY_E    ']'
#define _C_UNION_B  '('
#define _C_UNION_E  ')'
#define _C_STRUCT_B '{'
#define _C_STRUCT_E '}'


/*
** Error handling
**
** Call objc_error() or objc_verror() to record an error; this error
** routine will generally exit the program but not necessarily if the
** user has installed his own error handler.
**
** Call objc_set_error_handler to assign your own function for
** handling errors.  The function should return YES if it is ok
** to continue execution, or return NO or just abort if the
** program should be stopped.  The default error handler is just to
** print a message on stderr.
**
** The error handler function should be of type objc_error_handler
** The first parameter is an object instance of relevance.
** The second parameter is an error code.
** The third parameter is a format string in the printf style.
** The fourth parameter is a variable list of arguments.
*/
extern void objc_error(id object, int code, const char* fmt, ...);
extern void objc_verror(id object, int code, const char* fmt, va_list ap);
typedef BOOL (*objc_error_handler)(id, int code, const char *fmt, va_list ap);
objc_error_handler objc_set_error_handler(objc_error_handler func);

/*
** Error codes
** These are used by the runtime library, and your
** error handling may use them to determine if the error is
** hard or soft thus whether execution can continue or abort.
*/
#define OBJC_ERR_UNKNOWN 0             /* Generic error */

#define OBJC_ERR_OBJC_VERSION 1        /* Incorrect runtime version */
#define OBJC_ERR_GCC_VERSION 2         /* Incorrect compiler version */
#define OBJC_ERR_MODULE_SIZE 3         /* Bad module size */
#define OBJC_ERR_PROTOCOL_VERSION 4    /* Incorrect protocol version */

#define OBJC_ERR_MEMORY 10             /* Out of memory */

#define OBJC_ERR_RECURSE_ROOT 20       /* Attempt to archive the root
					  object more than once. */
#define OBJC_ERR_BAD_DATA 21           /* Didn't read expected data */
#define OBJC_ERR_BAD_KEY 22            /* Bad key for object */
#define OBJC_ERR_BAD_CLASS 23          /* Unknown class */
#define OBJC_ERR_BAD_TYPE 24           /* Bad type specification */
#define OBJC_ERR_NO_READ 25            /* Cannot read stream */
#define OBJC_ERR_NO_WRITE 26           /* Cannot write stream */
#define OBJC_ERR_STREAM_VERSION 27     /* Incorrect stream version */
#define OBJC_ERR_BAD_OPCODE 28         /* Bad opcode */

#define OBJC_ERR_UNIMPLEMENTED 30      /* Method is not implemented */

#define OBJC_ERR_BAD_STATE 40          /* Bad thread state */


typedef void* retval_t;		/* return value */
typedef void(*apply_t)(void);	/* function pointer */
typedef union {
  char *arg_ptr;
  char arg_regs[sizeof (char*)];
} *arglist_t;			/* argument frame */

#define __CLS_INFO(cls) ((cls)->info)
#define __CLS_ISINFO(cls, mask) ((__CLS_INFO(cls)&mask)==mask)
#define CLS_ISCLASS(cls) ((cls)&&__CLS_ISINFO(cls, CLS_CLASS))

#define class_pointer			isa


static inline Class
class_get_meta_class(Class class)
{
  return CLS_ISCLASS(class)?class->class_pointer:Nil;
}

static inline Class
class_get_super_class(Class class)
{
  return CLS_ISCLASS(class)?class->super_class:Nil;
}

static inline BOOL
object_is_instance(id object)
{
  return (object!=nil)&&CLS_ISCLASS(object->class_pointer);
}


/*
** Hook functions for allocating, copying and disposing of instances
*/
extern id (*_objc_object_alloc)(Class class);
extern id (*_objc_object_copy)(id object);
extern id (*_objc_object_dispose)(id object);

/*
** Standard functions for memory allocation and disposal.
** Users should use these functions in their ObjC programs so
** that they work properly with garbage collectors as well as
** can take advantage of the exception/error handling available.
*/
void *
objc_malloc(size_t size);

void *
objc_atomic_malloc(size_t size);

void *
objc_valloc(size_t size);

void *
objc_realloc(void *mem, size_t size);

void *
objc_calloc(size_t nelem, size_t size);

void
objc_free(void *mem);

#endif /* __extensions_objc_api_h__ */
