/*
 * Program to control ICOM radios
 *
 * Input/output interface
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "icom.h"

#ifndef MSDOS
#include <sys/inttypes.h>
#include <sys/fcntl.h>
#include <sys/termios.h>
#include <sys/unistd.h>
#else /* MSDOS */
#include <string.h>
#endif /* MSDOS */

/*
 * Parameters
 */
#ifdef MSDOS
#define XFRETRY	3000		/* interface timeout counter (MSDOS) */

/*
 * Define port and speed
 */
#define PORT	0x03f8		/* port address (COM1) */
/* #define PORT 0x02f8		/* port address (COM2) */
/* #define PORT 0x03e8		/* port address (COM3) */
/* #define PORT 0x02e8		/* port address (COM4) */
/* #define BAUD 384		/* baud rate 300 */
#define BAUD	96		/* baud rate 1200 */
/* #define BAUD 12		/* baud rate 9600 */

/*
 * Serial port definitions (8250)
 */
#define THR	0		/* transmitter holding register */
#define RBR	0		/* receiver buffer register */
#define DLL	0		/* divisor latch LSB */
#define DLM	1		/* divisor latch MSB */
#define LCR	3		/* line control register */
#define LCR_8BITS 3		/* 8 bit words */
#define   LCR_DLAB 0x80		/* divisor latch access bit */
#define MCR	4		/* modem control register */
#define   MCR_DTR 1		/* data terminal ready */
#define   MCR_RTS 2		/* request to send */
#define LSR	5		/* line status register */
#define   LSR_DR 0x01		/* data ready */
#define   LSR_THRE 0x20		/* trans line holding register empty */
#define   LSR_TSRE 0x40		/* transmitter shift register empty */
#else /* MSDOS */
#define DICOM	"/dev/icom"	/* serial line driver */
#define	BICOM	B1200		/* serial line speed */
#endif /* MSDOS */

/*
 * fsa definitions
 */
#define S_IDLE	0		/* idle */
#define S_HDR	1		/* header */
#define S_TX	2		/* address */
#define S_DATA	3		/* data */
#define S_ERROR	4		/* error */

/*
 * Global variables
 */
int retry;			/* max command retries */

/*
 * Local function prototypes
 */
static int sndoctet(int);	/* send octet */
static int rcvoctet();		/* receive octet */

/*
 * Local variables
 */
static int count;		/* retry counter */
static int fd_icom;		/* radio file descriptor */
static int state;		/* fsa state */
#ifdef MSDOS
static int inp(int);		/* input port byte */
static void outp(int, int);	/* output port byte */
static void outpw(int, int);	/* output port word */
#endif /* MSDOS */

/*
 * Packet routines
 *
 * These routines send a packet and receive the response. If an error
 * (collision) occurs on transmit, the packet is resent. If an error
 * occurs on receive (timeout), all input to the terminating FI is
 * discarded and the packet is resent. If the maximum number of retries
 * is not exceeded, the program returns the number of octets in the user
 * buffer; otherwise, it returns zero.
 *
 * ICOM frame format
 *
 * Frames begin with a two-octet preamble PR-PR followyd by the
 * transceiver address RE, controller address TX, control code CN, zero
 * or more data octets DA (depending on command), and terminator FI.
 * Since the bus is bidirectional, every octet output is echoed on
 * input. Every valid frame sent is answered with a frame in the same
 * format, but with the RE and TX fields interchanged. The CN field is
 * set to NAK if an error has occurred. Otherwise, the data are returned
 * in this and following DA octets. If no data are returned, the CN
 * octet is set to ACK.
 *
 *	+------+------+------+------+------+--//--+------+
 *	|  PR  |  PR  |  RE  |  TX  |  CN  |  DA  |  FI  |
 *	+------+------+------+------+------+--//--+------+
 */

/*
 * initpkt() - initialize serial interface
 *
 * This routine opens the serial interface for raw transmission; that
 * is, character-at-a-time, no stripping, checking or monkeying with the
 * bits. For Unix, an input operation ends either with the receipt of a
 * character or a 0.5-s timeout.
 */
void
initpkt()			/* no return value */
{
#ifdef MSDOS
	outp(PORT+LCR, LCR_DLAB); /* set baud */
	outpw(PORT+DLL, BAUD);
	outp(PORT+LCR, LCR_8BITS); /* set 8 bits, no parity */
	outp(PORT+MCR, MCR_DTR+MCR_RTS); /* wake up modem */
#else /* MSDOS */
	struct termios ttyb, *ttyp;

	ttyp = &ttyb;
	if ((fd_icom = open(DICOM, O_RDWR, 0777)) < 0) {
		printf("\n*** Unable to open serial line %s\n", DICOM);
		exit(1);
	}
	if (tcgetattr(fd_icom, ttyp) < 0) {
		printf("\n*** Unable to get line parameters\n");
		exit(1);
	}
	ttyp->c_iflag = 0;	/* input modes */
	ttyp->c_oflag = 0;	/* output modes */
	ttyp->c_cflag = BICOM|CS8|CREAD|CLOCAL;	/* control modes */
	ttyp->c_lflag = 0;	/* local modes */
	ttyp->c_cc[VMIN] = 0;	/* min chars */
	ttyp->c_cc[VTIME] = 5;	/* receive timeout */
	if (tcsetattr(fd_icom, TCSANOW, ttyp) < 0) {
		printf("\n*** Unable to set line parameters\n");
		exit(1);
	}
#endif /* MSDOS */
	retry = RETRY;
}

/*
 * sndpkt(r, x, y) - send packet and receive response
 *
 * This routine sends a command frame, which consists of all except the
 * preamble octets PR-PR. It then listens for the response frame and
 * returns the payload to the caller. The routine checks for correct
 * response header format; that is, the length of the response vector
 * returned to the caller must be at least 2 and the RE and TX octets
 * must be interchanged; otherwise, the operation is retried up to
 * the number of times specified in a global variable.
 *
 * The trace function, which is enabled by the P_TRACE bit of the global
 * flags variable, prints all characters received or echoed on the bus
 * preceded by a T (transmit) or R (receive). The P_ERMSG bit of the
 * flags variable enables printing of bus error messages.
 *
 * Note that the first octet sent is a PAD in order to allow time for
 * the radio to flush its receive buffer after sending the previous
 * response. Even with this precaution, some of the older radios
 * occasionally fail to receive a command and it has to be sent again.
 */
int
sndpkt(				/* returns octet count */
	int r,			/* radio address */
	int *x,			/* command vector */
	int *y			/* response vector */
	)
{
	int i, j, k, temp;	/* temps */

#ifndef MSDOS
	(void)tcflush(fd_icom, TCIOFLUSH);
#endif /* MSDOS */
	for (i = 0; i < retry; i++) {
		state = S_IDLE;

		/*
		 * Transmit packet.
		 */
		if (flags & P_TRACE)
			printf("T:");
		sndoctet(PAD);		/* send header */
		sndoctet(PR);
		sndoctet(PR);
		sndoctet(r);
		sndoctet(TX);
		for (j = 0; j < BMAX; j++) { /* send body */
			if (sndoctet(x[j]) == FI)
				break;
		}
		while (rcvoctet() != FI); /* purge echos */
		if (x[0] == V_FREQT || x[0] == V_MODET)
			return(0);	/* shortcut for broadcast */

		/*
		 * Receive packet. First, delete all characters
		 * preceeding a PR, then discard all PRs. Check that the
		 * RE and TX fields are correctly interchanged, then
		 * copy the remaining data and FI to the user buffer.
		 */
		if (flags & P_TRACE)
			printf("\nR:");
		j = 0;
		while ((temp = rcvoctet()) != FI) {
			switch (state) {

				case S_IDLE:
				if (temp != PR)
					continue;
				state = S_HDR;
				break;

				case S_HDR:
				if (temp == PR) {
					continue;
				} else if (temp != TX) {
					if (flags & P_ERMSG)
						printf(
						    "\n*** TX error\n");
					state = S_ERROR;
				}
				state = S_TX;
				break;

				case S_TX:
				if (temp != r) {
					if (flags & P_ERMSG)
						printf(
						    "\n*** RE error\n");
					state = S_ERROR;
				}
				state = S_DATA;
				break;

				case S_DATA:
				if (j >= BMAX ) {
					if (flags & P_ERMSG)
						printf(
					    "\n*** buffer overrun\n");
					state = S_ERROR;
					j = 0;
				}
				y[j++] = temp;
				break;

				case S_ERROR:
				break;
			}
		}
		if (flags & P_TRACE)
			printf("\n");
		if (j > 0)
			return(j);

	}
	if (flags & P_ERMSG)
		printf("*** retries exceeded\n");
	return(0);
}

/*
 * Interface routines
 *
 * These routines read and write octets on the bus. In case of receive
 * timeout a FI code is returned. In case of output collision (echo
 * does not match octet sent), the remainder of the collision frame
 * (including the trailing FI) is discarded.
 */

#ifdef MSDOS
/*
 * sndoctet(x) - send octet
 */
static int
sndoctet(			/* returns octet */
	int x;			/* octet */
	)
{
	int temp;

	inp(PORT+RBR);		/* flush spikes */
	outp(PORT+THR, x);	/* write octet */
	if (x == (temp = rcvoctet()))
		return(x);
	if (flags & P_ERMSG)
		printf("\n*** bus error %02x %02x\n", x, temp);
	while (rcvoctet() != FI); /* discard junk */
	return(FI);
}

/*
 * Receive octet
 *     returns octet received, force FI if error
 */
static int
rcvoctet()			/* returns octet */
{
	int temp, i;

	for (i = 0; i < XFRETRY; i++) /* wait for input */
	if ((inp(PORT+LSR))&LSR_DR != 0) {
		if (i > count)
			count = i;
		temp = inp(PORT+RBR); /* read octet */
		return(temp);
	}
	return(FI);
}
#else /* MSDOS */

/*
 * sndoctet(x) - send octet
 */
static int
sndoctet(			/* returns octet */
	int x			/* octet */
	)
{
	unsigned char y;

	y = (unsigned char)x;
	write(fd_icom, &y, 1);
	return(x);
}

/*
 * rcvoctet () - receive octet
 */
static int
rcvoctet()			/* returns octet */
{
	unsigned char y;

	if (read(fd_icom, &y, 1) < 1)
		y = FI;		/* come here if timeout */
	if (flags & P_TRACE && y != PAD)
		printf(" %02x", y);
	return((int)y);
}
#endif /* MSDOS */

/* end program */
