// Realistic Earth Terrain Scene
// Copyright (c) 1999, 2000 David A. Bartold
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// TODO (Some things to do or think about):
//
// 1. Apply real texture and bump maps to the moon.
// 2. Account for phases of the moon.
// 3. Sunrise and sunset should look different. (Perhaps sunrises have more
//    moisture/fog in the air?)
// 4. Add more variety to clouds.
// 5. Allow clouds to flow and move for animations.
// 6. Allow water to flow for animations.
// 7. Stretch landscape for bigger height field images (bigger than 400x400)
// 8. Too much dirt above the water (turbulence magnitude is too big,
      turbulence lateral scale too big)

// Fixed problems:
// 1. Rendering near sunset no longer messes up the lighting.

// New features:
// 1. Sun + Moon get smaller near their apexs.
// 2. Faster rendering for sunbeams.

/////// Start of User Definable Parameters and Their Defaults ///////////////

//#declare TF_CAMERA_LOCATION = <-0.5, 0.1, -0.5>;
//#declare TF_CAMERA_LOOK_AT = <-0.45, 0.1, -0.45>;
//#declare TF_HAVE_SUNBEAMS = true;
//#declare TF_FAST_SUNBEAMS = false;
//#declare TF_NORTH_DIR = -45.0;
//#declare TF_TIME_OF_DAY = 6.25;

// TF_HEIGHT_FIELD
//
// Declares the height field object that's positioned with one corner at
// the origin

#ifndef (TF_HEIGHT_FIELD)
#declare TF_HEIGHT_FIELD = height_field { tga "TF_FILE" smooth }
#end

// TF_WATER_LEVEL
//
// The parameter is a float between [0.0, 0.9].  The smaller the number,
// the more land that is above water.

#ifndef (TF_WATER_LEVEL)
#declare TF_WATER_LEVEL = 0.33;
#end

// TF_Y_SCALE
//
// This parameter is a float between (0.0, 1.0].  Basically, it sets the
// height of the rendered terrain

#ifndef (TF_Y_SCALE)
#declare TF_Y_SCALE = TF_YSCALE;
#end

// TF_REAL_LIGHTING
//
// Set true if you want a sun, moon, stars, sunrise, sunset, etc.  Otherwise,
// render with a single light to give a plastic model look.

#ifndef (TF_REAL_LIGHTING)
#declare TF_REAL_LIGHTING = true;
#end

// TF_HAVE_SUNBEAMS
//
// Set it to true if you want a sunbeam-like effect.  The parameter
// has an effect only if TF_REAL_LIGHTING is also set true.

#ifndef (TF_HAVE_SUNBEAMS)
#declare TF_HAVE_SUNBEAMS = false;
#end

// TF_FAST_SUNBEAMS
//
// Set it to render the sunbeams quickly, in exchange for smoothness.
// The sky will look slightly grainy if this is set.  This parameter
// only has an effect if sunbeams are turned on.

#ifndef (TF_FAST_SUNBEAMS)
#declare TF_FAST_SUNBEAMS = true;
#end

// TF_HAVE_WATER
//
// Self explanatory.

#ifndef (TF_HAVE_WATER)
#declare TF_HAVE_WATER = true;
#end

// TF_WATER_CLARITY
//
// Water clarity is expressed as a float from [0.0, 1.0].  It has no
// effect if water is disabled.
//
// e.x.  0.0  = Murky
//       0.95 = Clear freshwater

#ifndef (TF_WATER_CLARITY)
#declare TF_WATER_CLARITY = 0.95;
#end

// TF_TIME_OF_DAY
//
// Time of day is expressed as the number of hours given by a float
// in the range of [0.0, 24.0)
//
// e.x.  0.0  = (12:00am) Midnight
//       6.25 = ( 6:15am) Around sunrise
//      12.0  = (12:00pm) Noon

#ifndef (TF_TIME_OF_DAY)
#declare TF_TIME_OF_DAY = 12.0;
#end

// TF_HAVE_CLOUDS
//
// Self explanatory.

#ifndef (TF_HAVE_CLOUDS)
#declare TF_HAVE_CLOUDS = true;
#end

// TF_HAVE_FOG
//
// Turn on for a ground-level fog.

#ifndef (TF_HAVE_FOG)
#declare TF_HAVE_FOG = false;
#end

// TF_NORTH_DIR
//
// Set the direction for north.  Normally (when this parameter is 0.0),
// north is defined as the top of the height field image.  Direction
// is in degrees and positive numbers rotate the sun clockwise.

// e.x.  0.0: North = Up
//      90.0: North = Right

#ifndef (TF_NORTH_DIR)
#declare TF_NORTH_DIR = 0.0;
#end

// TF_CAMERA_LOCATION
//
// Set the camera's location.

#ifndef (TF_CAMERA_LOCATION)
#declare TF_CAMERA_LOCATION = <0, 0.75, -1.1>; 
#end

// TF_CAMERA_LOOK_AT
//
// Set the location to which the camera points.

#ifndef (TF_CAMERA_LOOK_AT)
#declare TF_CAMERA_LOOK_AT = <0, 0, 0>;
#end

/////// End of User Definable Parameters ////////////////////////////////////

// Include files:
#include "colors.inc"
#include "skies.inc"

#if (TF_REAL_LIGHTING)
// Earthly Lighting (sun, moon, stars)

// Noon:
#declare noon_sun_color = <1.0, 0.95, 0.8>;
#declare noon_sky_color_1 = <0.75, 0.85, 1.0>; // Low in sky
#declare noon_sky_color_2 = <0.75, 0.85, 1.0>; // Mid of sky
#declare noon_sky_color_3 = <0.75, 0.85, 1.0>; // High in sky
#declare noon_light_color = <1.0, 0.95, 0.8>;
#declare noon_apparent_brightness = 1.0;

// Sunrise/Sunset:
#declare srise_sun_color = <1.0, 0.7, 0.333>;
#declare srise_sky_color_1 = <1.0, 0.7, 0.4>;  // Low
#declare srise_sky_color_2 = <0.9, 0.9, 0.75>; // Mid
#declare srise_sky_color_3 = <0.7, 0.7, 0.95>; // High
#declare srise_light_color = <1.0, 0.8, 0.7>;
#declare srise_apparent_brightness = 0.9;

// Night time:
// The sun isn't visible then, but the reason why the sun's color at night
// is reddish is so that we can blend the color for sunrise/sunset.
#declare night_sun_color = <1.0, 0.6, 0.233>;
#declare night_sky_color_1 = <0.001, 0.001, 0.001>;
#declare night_sky_color_2 = <0.004, 0.004, 0.004>;
#declare night_sky_color_3 = <0.0, 0.0, 0.0>;
#declare night_light_color = <0.0, 0.0, 0.0>;
#declare night_apparent_brightness = 0.5;

// Other lights (These are always constant):
#declare moon_color = <0.0018, 0.0015, 0.0015>;
#declare stars_color = <0.0012, 0.0012, 0.0012>;

// Sky Color Blender:

#declare PURE_DAY_START = 7.0;
#declare PURE_DAY_STOP = 17.0;
#declare PURE_NIGHT_START = 19.0;
#declare PURE_NIGHT_STOP = 5.0;
#declare SUN_RISE = 6.0;
#declare SUN_SET = 18.0;
 
// Having these two conversions might be useful:

// #macro RGBToHSV(Clr)
// #end

// #macro HSVToRGB(Clr)
// #end

#macro Blend(MinPos,MinClr,MaxPos,MaxClr,Pos)
  MaxClr*((Pos-MinPos)/(MaxPos-MinPos)) + MinClr*((MaxPos-Pos)/(MaxPos-MinPos))
#end

#if (TF_TIME_OF_DAY>=PURE_DAY_START & TF_TIME_OF_DAY<=PURE_DAY_STOP)
#declare sun_color = noon_sun_color;
#declare sky_color_1 = noon_sky_color_1;
#declare sky_color_2 = noon_sky_color_2;
#declare sky_color_3 = noon_sky_color_3;
#declare light_color = noon_light_color;
#declare apparent_brightness = noon_apparent_brightness;
#else

#if (TF_TIME_OF_DAY>=PURE_NIGHT_START | TF_TIME_OF_DAY<PURE_NIGHT_STOP)
#declare sun_color = night_sun_color;
#declare sky_color_1 = night_sky_color_1;
#declare sky_color_2 = night_sky_color_2;
#declare sky_color_3 = night_sky_color_3;
#declare light_color = night_light_color;
#declare apparent_brightness = night_apparent_brightness;
#else

#if (TF_TIME_OF_DAY>=PURE_NIGHT_STOP & TF_TIME_OF_DAY<=PURE_DAY_START)

#if (TF_TIME_OF_DAY>=SUN_RISE)
#declare sun_color = Blend(SUN_RISE,srise_sun_color,PURE_DAY_START,noon_sun_color,TF_TIME_OF_DAY);
#declare sky_color_1 = Blend(SUN_RISE,srise_sky_color_1,PURE_DAY_START,noon_sky_color_1,TF_TIME_OF_DAY);
#declare sky_color_2 = Blend(SUN_RISE,srise_sky_color_2,PURE_DAY_START,noon_sky_color_2,TF_TIME_OF_DAY);
#declare sky_color_3 = Blend(SUN_RISE,srise_sky_color_3,PURE_DAY_START,noon_sky_color_3,TF_TIME_OF_DAY);
#declare light_color = Blend(SUN_RISE,srise_light_color,PURE_DAY_START,noon_light_color,TF_TIME_OF_DAY);
#declare apparent_brightness = Blend(SUN_RISE,srise_apparent_brightness,PURE_DAY_START,noon_apparent_brightness,TF_TIME_OF_DAY);
#else
// Time is between night and sunrise
#declare sun_color = Blend(PURE_NIGHT_STOP,night_sun_color,SUN_RISE,srise_sun_color,TF_TIME_OF_DAY);
#declare sky_color_1 = Blend(PURE_NIGHT_STOP,night_sky_color_1,SUN_RISE,srise_sky_color_1,TF_TIME_OF_DAY);
#declare sky_color_2 = Blend(PURE_NIGHT_STOP,night_sky_color_2,SUN_RISE,srise_sky_color_2,TF_TIME_OF_DAY);
#declare sky_color_3 = Blend(PURE_NIGHT_STOP,night_sky_color_3,SUN_RISE,srise_sky_color_3,TF_TIME_OF_DAY);
#declare light_color = Blend(PURE_NIGHT_STOP,night_light_color,SUN_RISE,srise_light_color,TF_TIME_OF_DAY);
#declare apparent_brightness = Blend(PURE_NIGHT_STOP,night_apparent_brightness,SUN_RISE,srise_apparent_brightness,TF_TIME_OF_DAY);

#end // Is between sunrise and day

#else
// Then time is during sunset

#if (TF_TIME_OF_DAY>=SUN_SET)
#declare sun_color = Blend(SUN_SET,srise_sun_color,PURE_NIGHT_START,night_sun_color,TF_TIME_OF_DAY);
#declare sky_color_1 = Blend(SUN_SET,srise_sky_color_1,PURE_NIGHT_START,night_sky_color_1,TF_TIME_OF_DAY);
#declare sky_color_2 = Blend(SUN_SET,srise_sky_color_2,PURE_NIGHT_START,night_sky_color_2,TF_TIME_OF_DAY);
#declare sky_color_3 = Blend(SUN_SET,srise_sky_color_3,PURE_NIGHT_START,night_sky_color_3,TF_TIME_OF_DAY);
#declare light_color = Blend(SUN_SET,srise_light_color,PURE_NIGHT_START,night_light_color,TF_TIME_OF_DAY);
#declare apparent_brightness = Blend(SUN_SET,srise_apparent_brightness,PURE_NIGHT_START,night_apparent_brightness,TF_TIME_OF_DAY);
#else
// Is between day and sunset
#declare sun_color = Blend(PURE_DAY_STOP,noon_sun_color,SUN_SET,srise_sun_color,TF_TIME_OF_DAY);
#declare sky_color_1 = Blend(PURE_DAY_STOP,noon_sky_color_1,SUN_SET,srise_sky_color_1,TF_TIME_OF_DAY);
#declare sky_color_2 = Blend(PURE_DAY_STOP,noon_sky_color_2,SUN_SET,srise_sky_color_2,TF_TIME_OF_DAY);
#declare sky_color_3 = Blend(PURE_DAY_STOP,noon_sky_color_3,SUN_SET,srise_sky_color_3,TF_TIME_OF_DAY);
#declare light_color = Blend(PURE_DAY_STOP,noon_light_color,SUN_SET,srise_light_color,TF_TIME_OF_DAY);
#declare apparent_brightness = Blend(PURE_DAY_STOP,noon_apparent_brightness,SUN_SET,srise_apparent_brightness,TF_TIME_OF_DAY);

#end // Is between sunset and night

#end // Is time during sunrise

#end // Is time during pure night

#end // Is time during pure day

#declare eye_sensitivity = apparent_brightness/(stars_color+sky_color_3);

light_source
{
  <-1.0, 1.0, -0.5>
  color sky_color_1*0.2*eye_sensitivity
  rotate y*TF_NORTH_DIR
  shadowless
}

#macro SPHERE_SIZE_FACTOR()
  ( abs( sin( TF_TIME_OF_DAY*360.0/24.0 ) )*0.6 + 0.4 )
#end

// Moon:
#if (TF_TIME_OF_DAY<PURE_DAY_START | TF_TIME_OF_DAY>PURE_DAY_STOP )
light_source
{
  <0.0, 7.0, 0.0>
  color moon_color*0.45*eye_sensitivity
  rotate z*(TF_TIME_OF_DAY*360.0/24.0)
  rotate y*TF_NORTH_DIR

  looks_like
  {
    sphere
    {
      <0.0, 0.0, 0.0>, 0.3*SPHERE_SIZE_FACTOR()

      pigment
      {
        bozo
        turbulence 0.5

        pigment_map
        {
          [0.0 color moon_color*eye_sensitivity]
          [1.0 color moon_color*0.8*eye_sensitivity]
        }
        scale 0.1
      }

      finish
      {
        ambient 1.0
        diffuse 0.0
      }
    }
  }
}

light_source
{
  <0.0, 7.0, 0.0>
  color moon_color*0.2*eye_sensitivity
}
#end

// Sun:
#if (TF_TIME_OF_DAY>PURE_NIGHT_STOP & TF_TIME_OF_DAY<PURE_NIGHT_START )

light_source
{
  <0.0, 1.0, 0.0>
  color (sky_color_3*0.2 + light_color*0.8)*eye_sensitivity
  rotate y*TF_NORTH_DIR
  shadowless
}

#declare Sun_Location = vrotate( vrotate( <0.0, -8.0, 0.0>,z*(TF_TIME_OF_DAY*360.0/24.0) ), y*TF_NORTH_DIR );

light_source
{
  Sun_Location
  color sun_color*0.7*eye_sensitivity

  spotlight
  radius 15
  falloff 20
  tightness 10

  point_at <0.0, 0.0, 0.0>

  looks_like
  {
    sphere
    {
      <0.0, 0.0, 0.0>, 0.5*SPHERE_SIZE_FACTOR()
      pigment
      {
        color sun_color*eye_sensitivity
      }

      finish
      {
        ambient 1.0
        diffuse 0.0
      }  
    }
  }

  media_attenuation on
  media_interaction on
}
#end

#declare P_cloud_gradient =
pigment
{
  planar
  turbulence 0.2
  omega 0.2
  pigment_map
  {
    [0.0 color sky_color_2 filter 0.8] // Mid in sky
    [1.0 color sky_color_1 filter 0.8] // Low in sky
  }
  scale 8
}

#if (TF_HAVE_SUNBEAMS)
media
{
#if (TF_FAST_SUNBEAMS)
  intervals 4
#else
  intervals 10
#end

  scattering { 2, rgb 0.03 }

  samples 1, 5
  confidence 0.99
  variance 1/100
  ratio 0.9
}
#end // TF_HAVE_SUNBEAMS

// Create some clouds:
#if (TF_HAVE_CLOUDS)
sphere
{
  <0.0, 0.0, 0.0>, 6.0

  hollow

  texture
  {
    pigment
    {
      bozo
      turbulence 0.5
      omega 0.5
      scale 0.05

      pigment_map
      {
        [0.0  P_cloud_gradient]
        [0.7  color rgbf 1.0]
        [1.0  color rgbf 1.0]
      }
    }

    finish
    {
      ambient 0.1
      diffuse 0.7
      reflection 0.0
    }

    scale <20, 5, 20>
  }
}
#end // TF_HAVE_CLOUDS

// Model stars and sky:
sphere
{
  <0.0, 0.0, 0.0>, 10.0
  hollow

  texture
  {
    pigment
    {
      bozo
      pigment_map
      {
        [0.0  color (stars_color+sky_color_3)*eye_sensitivity]
        [0.01 color sky_color_3*eye_sensitivity]
        [1.0  color sky_color_3*eye_sensitivity]
      }
    }
    scale 0.01
  }

  finish
  {
    ambient 1.0
    diffuse 0.0
    reflection 0.0
  }
}

#else

// Turn off realistic light sources to give a "toy model" look.
light_source
{
  <0, -2.0, 0.0>
  color <1.0, 0.95, 0.8>
  rotate z*(TF_TIME_OF_DAY*360.0/24.0)
  rotate y*TF_NORTH_DIR
}

#if (TF_HAVE_CLOUDS)
sky_sphere
{
  S_Cloud1
}
#end // TF_HAVE_CLOUDS

#end // TF_REAL_LIGHTING

#if (TF_HAVE_FOG)
fog
{
  distance 0.3
  color <0.65, 0.65, 0.65>
  fog_type 2
  fog_offset 0.072
  fog_alt 0.05
  turbulence 0.2
}
#end // TF_HAVE_FOG

camera
{
  location TF_CAMERA_LOCATION
  look_at TF_CAMERA_LOOK_AT
}

// Create water:
#if (TF_HAVE_WATER)
plane
{
  <0, 1, 0>, 0
  translate <0, TF_WATER_LEVEL*TF_Y_SCALE, 0>

  texture
  {
    pigment
    {
      color <0.65, 0.75, 0.8, TF_WATER_CLARITY>
    }

    finish
    {
      reflection 0.6
      diffuse 0.3
      roughness 0.01
    }

    normal
    {
      ripples 0.15
      frequency 1000.0
    }
  }

  interior
  {
    ior 1.33
  }
}
#end // TF_HAVE_WATER

// Create a bottommost plane for water
plane
{
  <0, 1, 0>, 0

  texture
  {
    pigment
    {
      color <0.05, 0.2, 0.4>  // Blue (water)
    }
  }
}

#declare T_silt =
texture
{
  pigment
  {
    // Sienna
    color <0.9, 0.8, 0.6>
  }
}

// Soil near shore-line -- light-colored and wet.
#declare T_soil =
texture
{
  // Burnt Sienna
  pigment
  {
    color <0.8, 0.6, 0.4>
  }

  finish
  {
    roughness 0.0001
    diffuse 0.85
  }
}

// Green stuff, like trees and shrubs
#declare T_shrubbery =
texture
{
  pigment
  {
    gradient y
    turbulence 0.4
    omega 20.0
    pigment_map
    {
      [ 0.0 color <0.1, 0.8, 0.05> ]    // Alive Green
      [ 1.0 color <0.05, 0.45, 0.2> ]   // Forest Green
    }
    scale 0.1
  }

  finish
  {
    reflection 0.0
    diffuse 0.5
    roughness 0.5
  }

  normal
  {
    bumps 0.4

    scale 0.0002
  }
}

// Deep green plants near rivers and lakes.
#declare T_greenery =
texture
{ 
  pigment
  {
    bozo
    color_map
    {
      [0 color <0.6, 0.75, 0.3>]
      [1 color <0.2, 0.5, 0.1>]
    }
    scale 0.005
  }

  normal
  {
    bumps 0.25
    scale 0.0001
  }
}

#declare T_yellowed =
texture
{
  pigment
  {
    gradient y
    turbulence 0.2
    omega 1.0
    pigment_map
    {
      [ 0.0 color <0.67, 0.7, 0.15> ]  // Yellow Green
      [ 0.5 color <0.8, 0.6, 0.5> ]    // Brownish Color
      [ 1.0 color <0.67, 0.7, 0.15> ]  // Yellow Green
    }
  }

  finish
  {
    diffuse 0.9
    roughness 1.0
  }

  normal
  {
    bumps 0.2
    scale 0.001
  }

  scale 0.05
}

#declare T_snowtop =
texture
{
  pigment
  {
    White
  }

  finish
  {
    specular 0.6
    reflection 0.0
    diffuse 0.9
    roughness 0.01
  }

  normal
  {
    bumps 0.1
    scale 0.0002
  }
}

#declare T_mountaincap =
texture
{
  planar
  turbulence 0.005
  omega 0.005
  texture_map
  {
    [ 0.000 T_snowtop ]
    [ 0.013 T_snowtop ]     // White (peaks)
    [ 0.030 T_yellowed ]    // Not-so-green foliage
    [ 0.070 T_shrubbery ]   // Green plants
    [ 1.000 T_shrubbery ]   // Green plants
  }
}

// Reddish Soil.
#declare P_reddish =
pigment
{
  color <0.65, 0.5, 0.3>
}

// Mixture of plants and sand near water sources.
#declare T_landwater =
texture
{
  pigment
  {
    gradient y
    turbulence 0.5
    omega 1.0
    pigment_map
    {
      [0.0 color <0.2, 0.5, 0.1> ]
      [0.10 color <0.2, 0.5, 0.1> ]
      [0.40 P_reddish ]
      [0.60 P_reddish ]
      [0.90 color <0.2, 0.5, 0.1> ]
      [1.0 color <0.2, 0.5, 0.1> ]
    }

    scale 0.01
  }

  normal
  {
    bumps 0.2
    scale 0.00005
  }
}

#declare T_landscape =
texture
{
  planar
  turbulence 0.01
  omega 0.75
  texture_map
  {
    [ 0.0500 T_mountaincap ]  // Additional foliage and possibly an ice cap
    [ 0.0655 T_greenery ]                            // Dark, Rough Green
    [ 0.0700 T_landwater ]                           // Between land and water
    [ 0.0725 pigment { P_reddish } ]                 // Reddish soil
    [ 0.0750 T_soil ]                                // Burnt Sienna
    [ 0.0780 T_silt ]                                // Sienna
    [ 0.0850 pigment   { color <0.3, 0.65, 0.9> } ]
    [ 0.1000 pigment   { color <0.05, 0.2, 0.4> } ]  // Blue (water)
  } 
  scale 10.0
  translate <0, -10, 0>

  scale (1.0-TF_WATER_LEVEL)/(1.0-0.24)
  translate <0, 1, 0>
  // Map terrain exists at 1.0-0.5 (mountain), WL (sealevel), 0.0 (seafloor)
}

// Create Land:
object
{
  TF_HEIGHT_FIELD

  texture
  {
    T_landscape
  }

  scale <1, TF_Y_SCALE, 1>
  translate <-0.5, 0.0, -0.5>
}
