/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "TFDialogTransform.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"
#include "GuiDialogOAC.h"


/*
 *  constructor: initialize all data members 
 */
TFDialogTransform::TFDialogTransform (HeightField *HF, HeightFieldDraw *HFD)
	        : TFPreviewDialog (HF, HFD, "Terraform Transform Dialog", 
				"Transform: set v for h = h^v", 6, 4), 
		  d_vbList (TRUE, 5),
		  d_hbTFactor (TRUE, 5),
		  d_hbInvert (TRUE, 5),
		  d_hbAdjustSea (TRUE, 5),
		  d_adjTFactor (1.5, 1.0, 5.0, 0.01),
		  d_hsTFactor (d_adjTFactor),
		  d_lblTFactor (_("Power Factor ")),
		  d_lblInvert (_("Invert Power Factor")),
		  d_lblAdjustSea (_("Adjust Waterlevel")),
		  d_frmOptions (_("Transform Parameters"))

{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ TFDialogTransform\n");
	buildDialogWindow ();
	this->setHFobjs (HF, HFD);
	sprintf (this->p_windowTitle, _("Transform: %s"), p_HF->getName());
	this->set_title (this->p_windowTitle);
	this->iterateEvents ();
	updatePreviewCallback ();

	connect_to_method (d_adjTFactor.value_changed, this, &TFDialogTransform::updatePreviewCallback);
	connect_to_method (d_btnInvert.clicked, this, &TFDialogTransform::updatePreviewCallback);
	connect_to_method (d_btnAdjustSea.clicked, this, &TFDialogTransform::updatePreviewCallback);
}


/*
 *  destructor: clean up 
 */
TFDialogTransform::~TFDialogTransform ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- TFDialogTransform\n");
}


/*
 *  insertOptions: insert the dialogs actual options 
 */
void TFDialogTransform::insertOptions ()
{
	SanityCheck::bailout ((!this->p_frmBase), "TFDialogTransform called without Frame label",
		"TFDialogTransform ::setHFobjs");

	d_frmOptions.set_shadow_type (GTK_SHADOW_ETCHED_IN);
	d_frmOptions.set_border_width (5);

	d_hbTFactor.pack_start (d_lblTFactor, TRUE, TRUE, GuiDialogOAC::s_HBOff);
	d_lblTFactor.show ();
	d_hsTFactor.set_update_policy (GTK_UPDATE_CONTINUOUS);
	d_hsTFactor.set_digits (2);
	d_hsTFactor.set_draw_value (TRUE);
	d_hbTFactor.pack_end (d_hsTFactor, TRUE, TRUE, GuiDialogOAC::s_HBOff);
	d_hsTFactor.show ();
	d_vbList.pack_start (d_hbTFactor, TRUE, TRUE, GuiDialogOAC::s_VBOff);
	d_hbTFactor.show ();

	d_hbInvert.pack_start (d_lblInvert, TRUE, TRUE, GuiDialogOAC::s_HBOff);
	d_lblInvert.show ();
	d_btnInvert.set_active (FALSE);
	connect_to_method (d_btnInvert.clicked, this, &TFDialogTransform::invertCBCallback);
	d_hbInvert.pack_end (d_btnInvert, TRUE, TRUE, GuiDialogOAC::s_HBOff);
	d_btnInvert.show ();
	d_vbList.pack_start (d_hbInvert, TRUE, TRUE, GuiDialogOAC::s_VBOff);
	d_hbInvert.show ();

	d_hbAdjustSea.pack_start (d_lblAdjustSea, TRUE, TRUE, GuiDialogOAC::s_HBOff);
	d_lblAdjustSea.show ();
	d_btnAdjustSea.set_active (FALSE);
	d_hbAdjustSea.pack_end (d_btnAdjustSea, TRUE, TRUE, GuiDialogOAC::s_HBOff);
	d_btnAdjustSea.show ();
	d_vbList.pack_start (d_hbAdjustSea, TRUE, TRUE, GuiDialogOAC::s_VBOff);
	d_hbAdjustSea.show ();

	d_frmOptions.add (d_vbList);
	this->p_tblMain->attach (d_frmOptions, 3, 6, 1, 3);
	d_vbList.show ();
	d_frmOptions.show ();
	this->p_frmBase->show ();
	this->get_vbox()->show ();

}



void TFDialogTransform::buildDialogWindow ()
{
	this->set_usize (450, 225);
	this->set_title (this->p_windowTitle);

	this->get_vbox()->set_border_width (2);

	this->setupVBox ();
	this->insertPreview ();
	insertOptions ();
	this->fillActionArea ();
	
	this->show ();
}


void TFDialogTransform::invertCBCallback ()
{	
	if (d_btnInvert.get_active())
		p_frmBase->set_label ("Transform: set v for h = h^(1/v)"); 
	else
		p_frmBase->set_label ("Transform: set v for h = h^v"); 

	// need to do this otherwise drawing on widget flashes out our redraw
	this->iterateEvents ();

	updatePreviewCallback ();
}


/*
 *  buttonCallbackApply: transform the height field
 */
void TFDialogTransform::buttonCallbackApply ()
{
	bool		adjustSea=d_btnAdjustSea.get_active (),
			invert=d_btnInvert.get_active ();
	float		value=d_adjTFactor.get_value ();

	SanityCheck::bailout ((!p_HFO), "p_HFO==NULL", "TFDialogTransform::buttonCallbackApply");

	if (invert)
		p_HFO->transform ((double)1/(double)value, adjustSea);
	else
		p_HFO->transform ((double)value, adjustSea);

	p_HFD->draw ();

	// only do this if window stays open 
	if (b_applyHit)
		{
		this->setHFobjs (p_HF, p_HFD);
		updatePreviewCallback ();
		}

	this->b_applyHit = TRUE;
}


/*
 *  updatePreviewCallback: update the preview after a widget has been updated. 
 */
void TFDialogTransform::updatePreviewCallback ()
{
	if (!d_cbUsePreview.get_active())
		return;

	bool		adjustSea=d_btnAdjustSea.get_active (),
			invert=d_btnInvert.get_active ();
	float		value=d_adjTFactor.get_value ();

	SanityCheck::bailout ((!p_HFPreview), "p_HFPreview==NULL", "TFDialogFill::updatePreviewCallback");
	SanityCheck::bailout ((!p_HFOPreview), "p_HFOPreview==NULL", "TFDialogFill::updatePreviewCallback");
	SanityCheck::bailout ((!p_HFDPreview), "p_HFDPreview==NULL", "TFDialogFill::updatePreviewCallback");
	this->previewUpToDate ();
	p_HFPreview->restoreBackup ();
	if (invert)
		p_HFOPreview->transform ((double)1/(double)value, adjustSea);
	else
		p_HFOPreview->transform ((double)value, adjustSea);
	p_HFDPreview->setColormap (p_HFD->getColormap());
	p_HFDPreview->draw ();
}


