/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "TFDialogPrintOptions.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"
#include "GuiBufferedDrawingArea.h"
#include "GuiColormap.h"
#include "MathTrig.h"
#include "TFWindowHandler.h"			// circular 
#include "PrintOptions.h"


// Paper definitions stollen from gnome-libs.
// All measurements are in centimetres. 
static const struct _dia_paper_metrics 
	{
	gchar *paper;
	gdouble pswidth, psheight;
	gdouble tmargin, bmargin, lmargin, rmargin;
	} 
	paper_metrics[] = 
		{
		{ "A3", 29.7, 42.0, 2.8222, 2.8222, 2.8222, 2.8222 },
		{ "A4", 21.0, 29.7, 2.8222, 2.8222, 2.8222, 2.8222 },
		{ "A5", 14.85, 21.0, 2.8222, 2.8222, 2.8222, 2.8222 },
		{ "B4", 25.7528, 36.4772, 2.1167, 2.1167, 2.1167, 2.1167 },
		{ "B5", 17.6389, 25.0472, 2.8222, 2.8222, 2.8222, 2.8222 },
		{ "B5-Japan", 18.2386, 25.7528, 2.8222, 2.8222, 2.8222, 2.8222 },
		{ "Letter", 21.59, 27.94, 2.54, 2.54, 2.54, 2.54 },
		{ "Legal", 21.59, 35.56, 2.54, 2.54, 2.54, 2.54 },
		{ "Half-Letter", 21.59, 14.0, 2.54, 2.54, 2.54, 2.54 },
		{ "Executive", 18.45, 26.74, 2.54, 2.54, 2.54, 2.54 },
		{ "Tabloid", 28.01, 43.2858, 2.54, 2.54, 2.54, 2.54 },
		{ "Monarch", 9.8778, 19.12, 0.3528, 0.3528, 0.3528, 0.3528 },
		{ "SuperB", 29.74, 43.2858, 2.8222, 2.8222, 2.8222, 2.8222 },
		{ "Envelope-Commercial", 10.5128, 24.2, 0.1764, 0.1764, 0.1764, 0.1764 },
		{ "Envelope-Monarch", 9.8778, 19.12, 0.1764, 0.1764, 0.1764, 0.1764 },
		{ "Envelope-DL", 11.0, 22.0, 0.1764, 0.1764, 0.1764, 0.1764 },
		{ "Envelope-C5", 16.2278, 22.9306, 0.1764, 0.1764, 0.1764, 0.1764 },
		{ "EuroPostcard", 10.5128, 14.8167, 0.1764, 0.1764, 0.1764, 0.1764 },
		{ NULL, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 }
		};

static const char* color_space[] = 
		{
		{ N_("Color") },
		{ N_("Grayscale") },
		{ N_("Monochrome") },
		{ NULL }
		};

enum {
	CHANGED,
	FITTOPAGE,
	LAST_SIGNAL
	};


/*
 *  TFDialogPrintOptions: provide a GUI interface to the print options
 * 	stored in PrintOptions.h
 */
TFDialogPrintOptions::TFDialogPrintOptions ()
	        : GuiDialogYN (_("Terraform Print Options Dialog"), _("Print Options Parameters"), 'N'),
		  d_hbOrientation (FALSE, 5),
		  d_vbPaperSize (FALSE, 5),
		  d_vbColorSpace (FALSE, 5),
		  d_vbScaling (FALSE, 5),
		  d_frmPaperSize (_("Paper Size")),
		  d_frmColorSpace (_("Colorspace")),
		  d_frmOrientation (_("Orientation")),
		  d_frmMargins (_("Margins")),
		  d_frmScaling (_("Scaling")),
		  d_lblPaperSize ("XXX"),
		  d_lblT (_("Top:")),
		  d_lblB (_("Bottom:")),
		  d_lblL (_("Left:")),
		  d_lblR (_("Right:")),
		  d_btnFitToPage (_("Fit to Page"))
			
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ TFDialogPrintOptions\n");

	d_paperSelected = 1;
	d_colorSpaceSelected = 1;
	p_col = new GdkColor ();
	p_colB = new GdkColor ();
	p_colB->red = p_colB->green = p_colB->blue = 0;
	p_colW = new GdkColor ();
	p_colW->red = p_colW->green = p_colW->blue = 255;

	p_adjMarginT=p_adjMarginB=p_adjMarginL=p_adjMarginR = NULL;
	p_pixPortrait=p_pixLandscape = NULL;
	p_rbOrientLandscape = p_rbOrientPortrait = NULL;
	p_menuPaperSize=p_menuColorSpace = NULL;
	p_optMenuPaperSize=p_optMenuColorSpace = NULL;
	p_spbtnMarginT=p_spbtnMarginB=p_spbtnMarginL=p_spbtnMarginR = NULL;
	p_rbOrientLandscape=p_rbOrientPortrait = NULL;
	p_tblMain=p_tblMargins = NULL;
	p_da = NULL;
	p_colorMap = NULL;

	buildDialogWindow ();

	// connect the signal handlers 
	connect_to_method (p_rbOrientPortrait->toggled, this, &TFDialogPrintOptions::orientChanged);
	connect_to_method (p_spbtnMarginT->changed, this, &TFDialogPrintOptions::marginChanged);
	connect_to_method (p_spbtnMarginB->changed, this, &TFDialogPrintOptions::marginChanged);
	connect_to_method (p_spbtnMarginL->changed, this, &TFDialogPrintOptions::marginChanged);
	connect_to_method (p_spbtnMarginR->changed, this, &TFDialogPrintOptions::marginChanged);
	connect_to_method (p_spbtnScaling->changed, this, &TFDialogPrintOptions::scaleChanged);
	connect_to_method (d_btnFitToPage.pressed, this, &TFDialogPrintOptions::fitToPagePressed);
}


/*
 *  destructor: clean up 
 */
TFDialogPrintOptions::~TFDialogPrintOptions ()
{
	delete p_col;
	delete p_colB;
	delete p_colW;
	delete p_adjMarginT;
	delete p_adjMarginB;
	delete p_adjMarginL;
	delete p_adjMarginR;
	delete p_pixPortrait;
	delete p_pixLandscape;
	delete p_menuPaperSize;
	delete p_menuColorSpace;
	delete p_optMenuPaperSize;
	delete p_optMenuColorSpace;
	delete p_spbtnMarginT;
	delete p_spbtnMarginB;
	delete p_spbtnMarginL;
	delete p_spbtnMarginR;
	delete p_rbOrientLandscape;
	delete p_rbOrientPortrait;
	delete p_tblMain;
	delete p_tblMargins;
	delete p_da;
	delete p_colorMap;

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- TFDialogPrintOptions\n");
}


/*
 *  init: assemble the dialog from it's component widgets
 */
void TFDialogPrintOptions::init ()
{
	Gtk_MenuItem	*menuitem;
	string		*s;
	gint 		i;
	char		buf[256];
	GdkColor	col;

	p_colorMap = new GuiColormap (this->get_toplevel(), 2);
	p_colorMap->addNamedColor (&col, "black");
	p_colorMap->addNamedColor (&col, "white");

	p_tblMain = new Gtk_Table (4, 2);
	p_tblMain->set_row_spacings (5);
	p_tblMain->set_col_spacings (5);
	this->p_frmBase->add (*p_tblMain);

	// 
	// paper size 
	// 
	p_tblMain->attach (d_frmPaperSize, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	d_frmPaperSize.show ();

	d_vbPaperSize.set_border_width (5);
	d_frmPaperSize.add (d_vbPaperSize);
	d_vbPaperSize.show ();

	p_optMenuPaperSize = new Gtk_OptionMenu ();
	d_vbPaperSize.pack_start (*p_optMenuPaperSize, TRUE, FALSE, 0);

	p_menuPaperSize = new Gtk_Menu ();
	for (i=0; paper_metrics[i].paper!=NULL; i++) 
		{
		menuitem = new Gtk_MenuItem (paper_metrics[i].paper);
		menuitem->set_user_data (GINT_TO_POINTER(i));
		connect_to_method (menuitem->activate, this, &TFDialogPrintOptions::paperSizeChange);
		p_menuPaperSize->add (*menuitem);
		menuitem->show ();
		}
	p_menuPaperSize->set_active (d_paperSelected);
	p_optMenuPaperSize->set_menu (*p_menuPaperSize);
	p_optMenuPaperSize->show ();
	
	d_vbPaperSize.pack_start (d_lblPaperSize, TRUE, TRUE, 0);
	d_lblPaperSize.show ();

	// 
	// orientation 
	// 
	p_tblMain->attach (d_frmOrientation, 1,2, 0,1, GTK_FILL|GTK_EXPAND, GTK_FILL, 0, 0);
	d_frmOrientation.show ();

	d_hbOrientation.set_border_width (5);
	d_frmOrientation.add (d_hbOrientation);
	d_hbOrientation.show ();

	p_rbOrientPortrait = new Gtk_RadioButton ((GSList*)NULL);
	sprintf (buf, "%s/%s", TF_DATADIR, "portrait.xpm");
	s = new string (buf);
	p_pixPortrait = new Gtk_Pixmap (*this, *s);
	p_rbOrientPortrait->add (*p_pixPortrait);
	delete s;
	p_pixPortrait->show ();

	d_hbOrientation.pack_start (*p_rbOrientPortrait, TRUE, TRUE, 0);
	p_rbOrientPortrait->show ();

	p_rbOrientLandscape = new Gtk_RadioButton (p_rbOrientPortrait->group());
	sprintf (buf, "%s/%s", TF_DATADIR, "landscape.xpm");
	s = new string (buf);
	p_pixLandscape = new Gtk_Pixmap (*this, *s);
	p_rbOrientLandscape->add (*p_pixLandscape);
	delete s;
	p_pixLandscape->show ();

	d_hbOrientation.pack_start (*p_rbOrientLandscape, TRUE, TRUE, 0);
	p_rbOrientLandscape->show ();

	// 
	// color space 
	// 
	p_tblMain->attach (d_frmColorSpace, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	d_frmColorSpace.show ();

	d_vbColorSpace.set_border_width (5);
	d_frmColorSpace.add (d_vbColorSpace);
	d_vbColorSpace.show ();

	p_optMenuColorSpace = new Gtk_OptionMenu ();
	d_vbColorSpace.pack_start (*p_optMenuColorSpace, TRUE, FALSE, 0);

	p_menuColorSpace = new Gtk_Menu ();
	for (i=0; color_space[i]!=NULL; i++) 
		{
		menuitem = new Gtk_MenuItem (_(color_space[i]));
		menuitem->set_user_data (GINT_TO_POINTER(i));
		connect_to_method (menuitem->activate, this, &TFDialogPrintOptions::colorSpaceChange);
		p_menuColorSpace->add (*menuitem);
		menuitem->show ();
		}
	p_menuColorSpace->set_active (0);
	p_optMenuColorSpace->set_menu (*p_menuColorSpace);
	p_optMenuColorSpace->show ();

	// 
	// margins 
	// 
	p_tblMain->attach (d_frmMargins, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 0, 0);
	d_frmMargins.show ();

	p_tblMargins = new Gtk_Table (4, 2, FALSE);
	p_tblMargins->set_border_width (5);
	p_tblMargins->set_row_spacings (5);
	p_tblMargins->set_col_spacings (5);
	d_frmMargins.add (*p_tblMargins);
	p_tblMargins->show ();

	d_lblT.set_alignment (1.0, 0.5);
	p_tblMargins->attach (d_lblT, 0, 1, 0, 1, GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
	d_lblT.show ();

	p_adjMarginT = new Gtk_Adjustment (1, 0,100, 0.1,10,10);
	p_spbtnMarginT = new Gtk_SpinButton (*p_adjMarginT);
	p_spbtnMarginT->set_digits (2);
	p_tblMargins->attach (*p_spbtnMarginT, 1,2, 0,1, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
	p_spbtnMarginT->show ();

	d_lblB.set_alignment (1.0, 0.5);
	p_tblMargins->attach (d_lblB, 0, 1, 1, 2, GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
	d_lblB.show ();

	p_adjMarginB = new Gtk_Adjustment (1, 0,100, 0.1,10,10);
	p_spbtnMarginB = new Gtk_SpinButton (*p_adjMarginB);
	p_spbtnMarginB->set_digits (2);
	p_tblMargins->attach (*p_spbtnMarginB, 1,2, 1,2, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
	p_spbtnMarginB->show ();

	d_lblL.set_alignment (1.0, 0.5);
	p_tblMargins->attach (d_lblL, 0, 1, 2, 3, GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
	d_lblL.show ();

	p_adjMarginL = new Gtk_Adjustment (1, 0,100, 0.1,10,10);
	p_spbtnMarginL = new Gtk_SpinButton (*p_adjMarginL);
	p_spbtnMarginL->set_digits (2);
	p_tblMargins->attach (*p_spbtnMarginL, 1,2, 2,3, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
	p_spbtnMarginL->show ();

	d_lblR.set_alignment (1.0, 0.5);
	p_tblMargins->attach (d_lblR, 0, 1, 3, 4, GTK_FILL, GTK_FILL|GTK_EXPAND, 0, 0);
	d_lblR.show ();

	p_adjMarginR = new Gtk_Adjustment (1, 0,100, 0.1,10,10);
	p_spbtnMarginR = new Gtk_SpinButton (*p_adjMarginR);
	p_spbtnMarginR->set_digits (2);
	p_tblMargins->attach (*p_spbtnMarginR, 1,2, 3,4, GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
	p_spbtnMarginR->show ();

	// 
	// Scaling 
	// 
	p_tblMain->attach (d_frmScaling, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 0, 0);
	d_frmScaling.show ();

	d_vbScaling.set_border_width (5);
	d_frmScaling.add (d_vbScaling);
	d_vbScaling.show ();

	p_adjScaling = new Gtk_Adjustment (100,1,10000, 1,10,10);
	p_spbtnScaling = new Gtk_SpinButton (*p_adjScaling);
	d_vbScaling.pack_start (*p_spbtnScaling, TRUE, TRUE, 0);
	p_spbtnScaling->show ();

	d_vbScaling.pack_start (d_btnFitToPage, TRUE, FALSE, 0);
	d_btnFitToPage.show ();

	// the drawing area 
	p_da = new GuiBufferedDrawingArea ();
	p_da->setSync (1);
	p_tblMain->attach (*p_da, 1,2, 1,4, 
		   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
	p_da->show ();
	p_tblMargins->show ();
	p_tblMain->show ();
}


/*
 *  setPaper: set the paper size 
 */
void TFDialogPrintOptions::setPaper (const gchar *paper)
{
	gint	i;

	for (i=0; paper_metrics[i].paper != NULL; i++) 
    		if (!g_strcasecmp(paper_metrics[i].paper, paper))
			break;

	if (paper_metrics[i].paper == NULL)
		i = 1; /* A4 */

	p_optMenuPaperSize->set_history(i);
}


/*
 *  getMargins: return the currently set margin values
 */
void TFDialogPrintOptions::getMargins ( gfloat *tmargin, gfloat *bmargin,
			    		gfloat *lmargin, gfloat *rmargin)
{
	if (tmargin)
		*tmargin = p_spbtnMarginT->get_value_as_float();
	if (bmargin)
		*bmargin = p_spbtnMarginB->get_value_as_float();
	if (lmargin)
		*lmargin = p_spbtnMarginL->get_value_as_float();
	if (rmargin)
		*rmargin = p_spbtnMarginR->get_value_as_float();
}


/*
 *  setMargins: set the Margins
 */
void TFDialogPrintOptions::setMargins ( gfloat tmargin, gfloat bmargin,
			    		gfloat lmargin, gfloat rmargin)
{
	p_spbtnMarginT->set_value (tmargin);
	p_spbtnMarginB->set_value (bmargin);
	p_spbtnMarginL->set_value (lmargin);
	p_spbtnMarginR->set_value (rmargin);
}



/*
 *  getOrientation: return the currently set orientation
 */
int TFDialogPrintOptions::getOrientation ()
{
	if (p_rbOrientPortrait->get_active())
		return 0; //DIA_PAGE_ORIENT_PORTRAIT;
	else
		return 1; //DIA_PAGE_ORIENT_LANDSCAPE;
}


/*
 *  setOrientation: set the orientation
 */
void TFDialogPrintOptions::setOrientation (int orient)
{
	switch (orient) 
		{
		case 0: //DIA_PAGE_ORIENT_PORTRAIT:
			p_rbOrientPortrait->set_active (TRUE);
	    		break;
		case 1: //DIA_PAGE_ORIENT_LANDSCAPE:
			p_rbOrientLandscape->set_active (TRUE);
			break;
		}
}


/*
 *  getScaling: return the currently set scaling factor
 */
gfloat TFDialogPrintOptions::getScaling ()
{
	return p_spbtnScaling->get_adjustment()->get_value()/100.0;
}


/*
 *  setScaling: set the scaling factor
 */
void TFDialogPrintOptions::setScaling (gfloat scaling)
{
//	p_spbtnScaling->
//  GTK_SPIN_BUTTON(self->scaling)->adjustment->value = scaling * 100.0;
//  gtk_adjustment_value_changed(GTK_SPIN_BUTTON(self->scaling)->adjustment);
}


/*
 *  getEffectiveArea: return the effective area the we can draw the page on
 */
void TFDialogPrintOptions::getEffectiveArea(gfloat *width, gfloat *height)
{
	gfloat 		h, w, scaling;

	if (p_rbOrientPortrait->get_active()) 
		{
		w = paper_metrics[d_paperSelected].pswidth;
		h = paper_metrics[d_paperSelected].psheight;
		} 
	else 
		{
		h = paper_metrics[d_paperSelected].pswidth;
		w = paper_metrics[d_paperSelected].psheight;
		}
	h -= p_spbtnMarginT->get_value_as_float();
	h -= p_spbtnMarginB->get_value_as_float();
	w -= p_spbtnMarginL->get_value_as_float();
	w -= p_spbtnMarginR->get_value_as_float();
	scaling = p_spbtnScaling->get_value_as_float() / 100.0;
	h /= scaling;
	w /= scaling;

	if (width)  
		*width = w;
	if (height) 
		*height = h;
}


/*
 *  getPaperSize: return the currently set paper size 
 */
void TFDialogPrintOptions::getPaperSize (const gchar *paper,
			       gfloat *width, gfloat *height)
{
	gint 	i;

	for (i = 0; paper_metrics[i].paper != NULL; i++) 
		{
		if (!g_strcasecmp(paper_metrics[i].paper, paper))
			break;
		}

	if (paper_metrics[i].paper == NULL)
		i = 1; 	// A4 
	if (width)
		*width = paper_metrics[i].pswidth;
	if (height)
		*height = paper_metrics[i].psheight;
}


/*
 *  getDefaultMargins: return the default margins for a given paper name
 */
void TFDialogPrintOptions::getDefaultMargins (const gchar *paper,
				    gfloat *tmargin, gfloat *bmargin,
				    gfloat *lmargin, gfloat *rmargin)
{
	gint 	i;

	for (i = 0; paper_metrics[i].paper != NULL; i++) 
		{
		if (!g_strcasecmp(paper_metrics[i].paper, paper))
			break;
		}

	if (paper_metrics[i].paper == NULL)
		i = 1; 	// A4
	if (tmargin)
		*tmargin = paper_metrics[i].tmargin;
	if (bmargin)
		*bmargin = paper_metrics[i].bmargin;
	if (lmargin)
		*lmargin = paper_metrics[i].lmargin;
	if (rmargin)
		*rmargin = paper_metrics[i].rmargin;
}


/*
 *  getDefaultMargins: return the default margins for a given paper name
 */
void TFDialogPrintOptions::fitToPagePressed ()
{
  //gtk_signal_emit(GTK_OBJECT(self), pl_signals[FITTOPAGE]);
}


/*
 *  sizePage: figure out what the page size accoring to orientation
 */
void TFDialogPrintOptions::sizePage ()
{
	Gtk_MenuItem	*menuitem = p_menuPaperSize->get_active();
	d_paperSelected = (int)(menuitem->get_user_data());
	d_width = p_da->width() - 3;
	d_height = p_da->height() - 3;

	// change to correct metrics 
	if (p_rbOrientPortrait->get_active())
		{
		if (d_width * paper_metrics[d_paperSelected].psheight >
	    	    d_height * paper_metrics[d_paperSelected].pswidth)
			d_width = (int)(d_height * paper_metrics[d_paperSelected].pswidth /
				  paper_metrics[d_paperSelected].psheight);
		else
			d_height = (int)(d_width * paper_metrics[d_paperSelected].psheight /
				   paper_metrics[d_paperSelected].pswidth);
		} 
	else 
		{
		if (d_width * paper_metrics[d_paperSelected].pswidth >
		    d_height * paper_metrics[d_paperSelected].psheight)
			d_width = (int)(d_height * paper_metrics[d_paperSelected].psheight /
				  paper_metrics[d_paperSelected].pswidth);
		else
		    d_height = (int)(d_width * paper_metrics[d_paperSelected].pswidth /
				paper_metrics[d_paperSelected].psheight);
		}

	d_offx = (p_da->width() - d_width - 3) / 2;
	d_offy = (p_da->height() - d_height - 3) / 2;
}


/*
 *  redraw: redraw the dialog after a setting has been changed
 */
gint TFDialogPrintOptions::redraw (GdkEventExpose *event)
{
	gfloat 		val;
	gint 		num;

	printf ("redraw ()\n");

	sizePage ();
	p_da->clear ();

	// draw the page image 
	p_da->setColors (p_colorMap->getColor(0));
	p_da->drawRectangle (d_offx+3, d_offy+3, d_width, d_height, TRUE);

	p_da->setColors (p_colorMap->getColor(1));
	p_da->drawRectangle (d_offx, d_offy, d_width, d_height, TRUE);

	p_da->setColors (p_colorMap->getColor(0));
	p_da->drawRectangle (d_offx, d_offy, d_width-1, d_height-1, FALSE);

	// draw margins 
	if (p_rbOrientPortrait->get_active()) 
		{
		val = p_spbtnMarginT->get_value_as_float();
		num = (int)(d_offy + val * d_height /paper_metrics[d_paperSelected].psheight);
		p_da->drawLine (d_offx+1, num, d_offx+d_width-2,num);

		val = p_spbtnMarginB->get_value_as_float();
		num = (int)(d_offy + d_height - val * d_height / paper_metrics[d_paperSelected].psheight);
		p_da->drawLine (d_offx+1, num, d_offx+d_width-2,num);

		val = p_spbtnMarginL->get_value_as_float(); 
		num = (int)(d_offx + val * d_width / paper_metrics[d_paperSelected].pswidth); 
		p_da->drawLine (num, d_offy+1,num,d_offy+d_height-2);

		val = p_spbtnMarginR->get_value_as_float(); 
		num = (int)(d_offx + d_width - val * d_width / paper_metrics[d_paperSelected].pswidth);
		p_da->drawLine (num, d_offy+1,num,d_offy+d_height-2);
  		} 
	else 
		{
		val = p_spbtnMarginT->get_value_as_float();
		num = (int)(d_offy + val * d_height /paper_metrics[d_paperSelected].pswidth);
		p_da->drawLine (d_offx+1, num, d_offx+d_width-2,num);

		val = p_spbtnMarginB->get_value_as_float();
		num = (int)(d_offy + d_height - val * d_height / paper_metrics[d_paperSelected].pswidth);
		p_da->drawLine (d_offx+1, num, d_offx+d_width-2,num);

		val = p_spbtnMarginL->get_value_as_float(); 
		num = (int)(d_offx + val * d_width / paper_metrics[d_paperSelected].psheight);
		p_da->drawLine (num, d_offy+1,num,d_offy+d_height-2);

		val = p_spbtnMarginR->get_value_as_float(); 
		num = (int)(d_offx + d_width - val * d_width / paper_metrics[d_paperSelected].psheight);
		p_da->drawLine (num, d_offy+1,num,d_offy+d_height-2);
		}
	p_da->setColors (p_colorMap->getColor(0));
	return FALSE;
}


/*
 *  paperSizeChange: process the changed paper size
 */
void TFDialogPrintOptions::paperSizeChange ()
{
	gchar 		buf[512];
	GtkAdjustment	*adj;
	float		w, h;

	//printf ("paper_size_change()\n");

	getEffectiveArea (&w, &h);
	//p_da->set_usize (w, h);

	redraw ();

	p_spbtnMarginT->set_value (paper_metrics[d_paperSelected].tmargin);
	p_spbtnMarginB->set_value (paper_metrics[d_paperSelected].bmargin);
	p_spbtnMarginL->set_value (paper_metrics[d_paperSelected].lmargin);
	p_spbtnMarginR->set_value (paper_metrics[d_paperSelected].rmargin);

	if (p_rbOrientPortrait->get_active ())
		{
		// plain Gtk magic to get around object limitation of setting values
		adj = (GtkAdjustment*)(p_spbtnMarginT->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].psheight;
		p_spbtnMarginT->set_adjustment(*p_spbtnMarginT->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginB->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].psheight;
		p_spbtnMarginB->set_adjustment(*p_spbtnMarginB->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginL->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].pswidth;
		p_spbtnMarginL->set_adjustment(*p_spbtnMarginL->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginR->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].pswidth;
		p_spbtnMarginR->set_adjustment(*p_spbtnMarginR->get_adjustment());
		} 
	else 
		{
		// plain Gtk magic to get around object limitation of setting values
		adj = (GtkAdjustment*)(p_spbtnMarginT->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].pswidth;
		p_spbtnMarginT->set_adjustment(*p_spbtnMarginT->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginB->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].pswidth;
		p_spbtnMarginB->set_adjustment(*p_spbtnMarginB->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginL->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].psheight;
		p_spbtnMarginL->set_adjustment(*p_spbtnMarginL->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginR->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].psheight;
		p_spbtnMarginR->set_adjustment(*p_spbtnMarginR->get_adjustment());
		}

	g_snprintf(buf, sizeof(buf), "%0.3gcm x %0.3gcm",
		   paper_metrics[d_paperSelected].pswidth,
		   paper_metrics[d_paperSelected].psheight);
	string 	s(buf);
	d_lblPaperSize.set_text (s);

	getEffectiveArea (&w, &h);
}


/*
 *  orientChanged: process the changed orientation 
 */
void TFDialogPrintOptions::orientChanged ()
{
	GtkAdjustment	*adj = NULL;

	//printf ("orient_changed()\n");

	redraw ();

	if (p_rbOrientPortrait->get_active ())
		{
		// plain Gtk magic to get around object limitation of setting values
		adj = (GtkAdjustment*)(p_spbtnMarginT->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].psheight;
		p_spbtnMarginT->set_adjustment(*p_spbtnMarginT->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginB->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].psheight;
		p_spbtnMarginB->set_adjustment(*p_spbtnMarginB->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginL->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].pswidth;
		p_spbtnMarginL->set_adjustment(*p_spbtnMarginL->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginR->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].pswidth;
		p_spbtnMarginR->set_adjustment(*p_spbtnMarginR->get_adjustment());
		} 
	else 
		{
		// plain Gtk magic to get around object limitation of setting values
		adj = (GtkAdjustment*)(p_spbtnMarginT->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].pswidth;
		p_spbtnMarginT->set_adjustment(*p_spbtnMarginT->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginB->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].pswidth;
		p_spbtnMarginB->set_adjustment(*p_spbtnMarginB->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginL->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].psheight;
		p_spbtnMarginL->set_adjustment(*p_spbtnMarginL->get_adjustment());

		adj = (GtkAdjustment*)(p_spbtnMarginR->get_adjustment()->gtkobj());
		adj->upper = paper_metrics[d_paperSelected].psheight;
		p_spbtnMarginR->set_adjustment(*p_spbtnMarginR->get_adjustment());
		}
}


/*
 *  colorSpaceChange: process a changed color space
 */
void TFDialogPrintOptions::colorSpaceChange ()
{
	Gtk_MenuItem	*menuitem = p_menuColorSpace->get_active();
	d_colorSpaceSelected = (int)(menuitem->get_user_data());
}


/*
 *  marginChanged: process the changed maring
 */
void TFDialogPrintOptions::marginChanged ()
{
	orientChanged ();
	redraw ();
}


/*
 *  scaleChanged: process the changed maring
 */
void TFDialogPrintOptions::scaleChanged ()
{
	int	value = p_spbtnScaling->get_value_as_int ();

	redraw ();
}


int TFDialogPrintOptions::idleRedraw ()
{
	paperSizeChange ();
	return 0;
}



/*
 *  buildDialogWindow: build the dialog window from it's components
 */
void TFDialogPrintOptions::buildDialogWindow ()
{
	this->set_usize (400, 400);
	this->set_title (this->p_windowTitle);

	this->get_vbox()->set_border_width (2);
	setupVBox ();

	SanityCheck::bailout ((!this->p_frmBase), "this->p_frmBase==NULL", "TFDialogPrintOptions::buildDialogWindow");

	init ();
	this->fillActionArea ();

	this->show ();

	// interate through all the pending events
	Gtk_Main *gtkMain = Gtk_Main::instance();
		while (gtkMain->events_pending())
			gtkMain->iteration ();

	connect_to_method (Gtk_Main::timeout(150), this, &TFDialogPrintOptions::idleRedraw);
}


/*
 *  buttonCallbackYes: process clicking the 'Yes' button
 */
void TFDialogPrintOptions::buttonCallbackYes ()
{
	PrintOptions::s_paperSize = paper_metrics[d_paperSelected].paper;
	if (p_rbOrientLandscape->get_active())
		PrintOptions::s_orientation='L';
	else
		PrintOptions::s_orientation='P';

	if (d_colorSpaceSelected == 0)
		PrintOptions::s_color='C';		// color
	else
	if (d_colorSpaceSelected == 1)
		PrintOptions::s_color='G';		// grayscale
	else
	if (d_colorSpaceSelected == 2)
		PrintOptions::s_color='M';		// mono

	PrintOptions::s_marginT = p_spbtnMarginT->get_value_as_float();
	PrintOptions::s_marginB = p_spbtnMarginB->get_value_as_float();
	PrintOptions::s_marginL = p_spbtnMarginL->get_value_as_float();
	PrintOptions::s_marginR = p_spbtnMarginR->get_value_as_float();

	PrintOptions::s_paperWidth = paper_metrics[d_paperSelected].pswidth;
	PrintOptions::s_paperWidth = paper_metrics[d_paperSelected].psheight;

	PrintOptions::s_scaleFactor = p_spbtnScaling->get_value_as_float ();

	this->hide ();
}


/*
 *  buttonCallbackNo: process clicking the 'No' button
 */
void TFDialogPrintOptions::buttonCallbackNo ()
{
	this->hide ();
}




/*
 *  cmbPaperSizeCallback: process a change in paper size selection 
 */
void TFDialogPrintOptions::cmbPaperSizeCallback ()
{
	string		entry;

	//entry = (d_cmbPaperSize->get_entry())->get_text();

	if (entry == "Custom")
		d_enPaperSize.set_sensitive (TRUE);
	else
		d_enPaperSize.set_sensitive (FALSE);
		
}


