/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "MathRandom.h"
#include "GlobalTrace.h"



/* 
 *  constructor 
 */ 
MathRandom::MathRandom (int seed)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ MathRandom\n");

	if (seed == -1)		// default
		init (time(NULL));
	else
		init (seed);
}


/*
MathRandom::~MathRandom ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- MathRandom\n");
}
*/


/* 
 *  rnd: return a number between floor and ceil (default 0 <= n <= 1)
 */ 
double MathRandom::rnd (float floor, float ceil)
{
	return (floor+((ceil-floor)*rnd1()));
}


void MathRandom::init (int seed)
{
	int 	i, j;

	i = abs(97 * seed) % 31329;  /* fixed abs(); overflow cond. 12/16/95 jpb */
	j = abs(33 * seed) % 30082;
	initSequence (i,j);
} 


/* 
C This is the initialization routine for the random number generator RANMAR()
C NOTE: The seed variables can have values between:    0 <= IJ <= 31328
C                                                      0 <= KL <= 30081
C The random number sequences created by these two seeds are of sufficient 
C length to complete an entire calculation with. For example, if several
C different groups are working on different parts of the same calculation,
C each group could be assigned its own IJ seed. This would leave each group
C with 30000 choices for the second seed. That is to say, this random 
C number generator can create 900 million different subsequences -- with 
C each subsequence having a length of approximately 10^30.
C 
C Use IJ = 1802 & KL = 9373 to test the random number generator. The
C subroutine RANMAR should be used to generate 20000 random numbers.
C Then display the next six random numbers generated multiplied by 4096*4096
C If the random number generator is working properly, the random numbers
C should be:
C           6533892.0  14220222.0  7275067.0
C           6172232.0  8354498.0   10633180.0
*/
void MathRandom::initSequence (int ij, int kl)
{
	int i, j, k, l, ii, jj, m;
	float s, t;
	
	if (GlobalTrace::isSet (GlobalTrace::TRACE_VERBOSE))
		printf ("MathRandom::initSequence: seed=%d, %d\n", ij,  kl);

	if (ij<0 || ij>31328 || kl<0 || kl>30081) 
		{
		puts("The first random number seed must have a value between 0 and 31328.");
		puts("The second seed must have a value between 0 and 30081.");
		exit(1);
		}
	
	i = (ij/177)%177 + 2;
	j = ij%177 + 2;
	k = (kl/169)%178 + 1;
	l = kl%169;
	
	for (ii=1; ii<=97; ii++) 
		{
		s = 0.0;
		t = 0.5;
		for (jj=1; jj<=24; jj++) 
			{
			m = (((i*j)%179)*k) % 179;
			i = j;
			j = k;
			k = m;
			l = (53*l + 1) % 169;
			if ((l*m)%64 >= 32) s += t;
			t *= 0.5;
			}
		u[ii] = s;
		}
	
	c = 362436.0 / 16777216.0;
	cd = 7654321.0 / 16777216.0;
	cm = 16777213.0 / 16777216.0;
	
	i97 = 97;
	j97 = 33;
}


/*  return a single floating-point random number  -mod jpb 7/23/95 */
/*  the output is a number between 0.0 and 1.0 */
float MathRandom::rnd1 ()
{
	float uni;  			// the random number itself 

	uni = u[i97] - u[j97];		// difference between two [0..1] #s 
	if (uni < 0.0) 
		uni += 1.0;
	u[i97] = uni;
	i97--;				// i97 ptr decrements and wraps around 
	if (i97==0) 
		i97 = 97;
	j97--;				// j97 ptr decrements likewise 
	if (j97==0) 
		j97 = 97;
	c -= cd; 			// finally, condition with c-decrement 
	if (c<0.0) 
		c += cm;		// cm > cd we hope! (only way c always >0) 
	uni -= c;
	if (uni<0.0) 
		uni += 1.0;

	//if (GlobalTrace::isSet (GlobalTrace::TRACE_DEBUG))
	//	printf ("MathRandom::rnd1: r=%f\n", uni);

	return(uni); 			// return the random # 
}  

