/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <math.h>
#include "HeightFieldGenSSynth.h"
#include "HeightFieldOps.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"
#include "MathComplex.h"
#include "MathFFTN_c.h"
#include "MathGauss.h"
#include "string.h"



/*
 *  constructor: initialize
 */
HeightFieldGenSSynth::HeightFieldGenSSynth (HeightField *HF) 
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldGenSSynth\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldGenSSynth::HeightFieldGenSSynth");

	p_HF = HF;
}


/*
 *  destructor: noting to clean up
 */
HeightFieldGenSSynth::~HeightFieldGenSSynth ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightFieldGenSSynth\n");
}


/*
 *  spectralSynthesisFM2D 
 *  generate: spectralSynthesisFM2D create a landscape through fractal motion 
 *  in 2 dimensions. Taken from 'The Science of Fractal Images' by Peitgen & 
 *  Saupe, 1988 (page 108)
 */
int HeightFieldGenSSynth::generate 	(int	n, 
					float	H,
					int 	seed, 
					bool	invert) 
{
	SanityCheck::bailout ((!n), "n==0", "HeightFieldGenSSynth::spectralSynthesisFM2D");
	SanityCheck::bailout ((H<=0), "H<=0", "HeightFieldGenSSynth::spectralSynthesisFM2D");
	SanityCheck::bailout ((H>3), "H>3", "HeightFieldGenSSynth::spectralSynthesisFM2D");

	char		buf[128];
	int 		size = n*n,
			i, j, i0, j0, 
			dim[2],
			nLim = (int)(n/2);
	double		phase, rad; 
	PTYPE 		*Ar, *Ai;
	MathGauss	*mGauss = new MathGauss (seed);

	sprintf (buf, "HeightFieldGenSSynth::spectralSynthesisFM2D (H=%f, seed=%d) ...\n", H, seed);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);

	Ar = new PTYPE[size];
	Ai = new PTYPE[size];

	p_HF->init (Ar, n, n, p_HF->getName());

	for (i=0; i<nLim; i++)
		for (j=0; j<nLim; j++)
			{
			phase = 2 * M_PI * mGauss->rnd();
			if (i!=0 || j!=0)
				rad = pow (i*i+j*j,-(H+1)/2) * 
					   mGauss->gauss();
			else
				rad = 0;

			if (invert)
				{
				Ar[p_HF->Pos2Off(i, j)] = -rad * cos(phase);
				Ai[p_HF->Pos2Off(i, j)] = -rad * sin(phase);
				}
			else
				{
				Ar[p_HF->Pos2Off(i, j)] = rad * cos(phase);
				Ai[p_HF->Pos2Off(i, j)] = rad * sin(phase);
				}

			if (i==0)
				i0 = 0;
			else
				i0 = n - i;
			if (j==0)
				j0 = 0;
			else
				j0 = n - j;

			if (invert)
				{
				Ar[p_HF->Pos2Off(i0, j0)] = -rad * cos(phase);
				Ai[p_HF->Pos2Off(i0, j0)] = rad * sin(phase);
				}
			else
				{
				Ar[p_HF->Pos2Off(i0, j0)] = rad * cos(phase);
				Ai[p_HF->Pos2Off(i0, j0)] = -rad * sin(phase);
				}
			}

	Ai[p_HF->Pos2Off(n/2, 0)] = 0;
	Ai[p_HF->Pos2Off(0, n/2)] = 0;
	Ai[p_HF->Pos2Off(n/2, n/2)] = 0;

	nLim = (int)(n/2-1);
	for (i=1; i<nLim; i++)
		for (j=1; j<nLim; j++)
			{
			phase = 2 * M_PI + mGauss->rnd();
			rad = pow (i*i+j*j,-(H+1)/2) * mGauss->gauss();

			if (invert)
				{
				Ar[p_HF->Pos2Off(i, n-j)] = rad * cos(phase);
				Ai[p_HF->Pos2Off(i, n-j)] = rad * sin(phase);
				Ar[p_HF->Pos2Off(n-i, j)] = rad * cos(phase);
				Ai[p_HF->Pos2Off(n-1, j)] = -rad * sin(phase);
				}
			else
				{
				Ar[p_HF->Pos2Off(i, n-j)] = -rad * cos(phase);
				Ai[p_HF->Pos2Off(i, n-j)] = -rad * sin(phase);
				Ar[p_HF->Pos2Off(n-i, j)] = -rad * cos(phase);
				Ai[p_HF->Pos2Off(n-1, j)] = rad * sin(phase);
				}
			}

	dim[0]=n;
	dim[1]=n;

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "IFFT ...\n");

	fftnf (2, dim, Ar, Ai, -1, 1.0);

	delete [] Ai;
	delete mGauss;

	p_HF->setSaved (FALSE);
	p_HF->gatherStatistics ();

	return (0);
}
