/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "HeightField.h" 
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"



int	HeightField::s_HFcount = 0;


/*
 *  constructor: take a hf and build the object from it
 */
HeightField::HeightField (PTYPE *hf, int xsize, int ysize, char *name) 
	    : HeightFieldCore (hf, xsize, ysize)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightField\n");
	p_name = NULL;
	p_hfBck = NULL;
	p_contourList = NULL;
	p_renderOptions = NULL;

	b_loadedFromFile = FALSE;
	b_isSaved = FALSE;
	d_updateTime = time(NULL);
	d_backupTime = d_contourTime = 0;

	setName (name);
}


/*
 *  constructor: create a new HF of specified size 
 */
HeightField::HeightField (int xsize, int ysize, char *name) 
		: HeightFieldCore (xsize, ysize)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightField\n");
	p_name = NULL;
	p_hfBck = NULL;
	p_contourList = NULL;
	p_renderOptions = NULL;

	b_loadedFromFile = FALSE;
	b_isSaved = FALSE;
	d_updateTime = time(NULL);
	d_backupTime = d_contourTime = 0;
	
	setName (name);
}


/*
 *  destructor: clean up, free allocated memory
 */
HeightField::~HeightField ()
{
	if (p_name)
		free (p_name);
	if (p_renderOptions)
		delete p_renderOptions;

	destroyBackup ();
	destroyContourList ();

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightField\n");
}


/*
 *  init: re-initialize the object properties according to the parameters 
 */
void HeightField::init (PTYPE *hf, int xsize, int ysize, 
			char *name, bool keepDerived)
{
	this->setData (hf, xsize, ysize);
	setName (name);

	if (hf)
		{
		d_updateTime = time (NULL);
		b_isSaved = FALSE;
		}
	else
		{
		d_updateTime = 0;
		b_isSaved = TRUE;
		}

	if (!keepDerived)
		{
		destroyBackup ();
		destroyContourList ();
		setRenderOptions (NULL);
		}

	b_loadedFromFile=FALSE;
}


/*
 *  setName: assign a new name to the HF. If a NULL name is passed, we 
 * 	generate a new one, otherwise we take a copy of name and discard 
 * 	the old name. 
 */
void HeightField::setName (char *name)
{
	// take a temp pointer to name so that we can handle name==p_name
	char 	*oldName = NULL;

	if (p_name)
		oldName = p_name;

	if (name)
		p_name = strdup (name);
	else
		p_name = getNewName ();

	if (oldName)
		free (oldName);
}


/*
 *  setSaved: set the b_isSaved flag and calculate statistics. It's a good
 *	idea to call this after having altered the height field because
 *	it ensures the consistency of the HF and some basic data.
 */
void HeightField::setSaved (bool isSaved, bool keepDerived)
{
        b_isSaved = isSaved;
        d_updateTime = time(NULL);

	if (!isSaved && !keepDerived)
		destroyContourList ();

        gatherStatistics ();
}


/*
 *  setContourList: assign a new contour list and delete the old one
 */
void HeightField::setContourList(TFGListFlexArr *l, time_t timestamp)
{
	// delete old list
	if (p_contourList)
		delete p_contourList;

	p_contourList = l;
	d_contourTime = timestamp;
}


/*
 *  setRenderOptions: assign a new RenderOptions class and delete the old one
 */
void HeightField::setRenderOptions (RenderOptions *r)
{
	// delete old RenderOptions
	if (p_renderOptions)
		delete p_renderOptions;

	p_renderOptions = r;
}


/*
 *  getNewName: return a newly allocated string for the next name.
 * 	receiver becomes the owner of this piece of memory!
 */
char *HeightField::getNewName ()
{
	char	buf[256];
	char	*newName;

	sprintf (buf, "%s-%02d.tga", TF_DEFAULT_FILENAME, ++s_HFcount);
	newName = strdup (buf);
	return (newName);
}


/*
 *  getCopyOfData: return a newly allocated copy of the actual data stream. 
 *	Calling this function returns a pointer which you then own. If you
 *	don't keep track of this pointer, you're going to have some really
 *	large memory leaks!
 */
PTYPE *HeightField::getCopyOfData ()
{
	SanityCheck::bailout ((!p_hf), "p_hf==NULL", "HeightField::getCopyOfData");

	PTYPE	 *hfnew = new PTYPE[d_size];

	memcpy (hfnew, this->p_hf, this->d_size*sizeof(PTYPE));
	return hfnew;
}


/*
 *  releaseData: release the data pointer and reset everything ... 
 *	Calling this function returns a pointer which you then own. If you
 *	don't keep track of this pointer, you're going to have some really
 *	large memory leaks!
 */
PTYPE *HeightField::releaseData ()
{
	SanityCheck::bailout ((!p_hf), "p_hf==NULL", "HeightField::releaseData");
	PTYPE	*data = p_hf;

	// setting HF::p_hf to null will prevent init/setData from deleting it
	this->p_hf = NULL;
	init (NULL, 0, 0, "empty");
	return data;
}


/*
 *  newHF: discard the current HF and allocate a new one of specified size 
 */
void HeightField::newHF (int xsize, int ysize, bool generateName)
{
	int	size = xsize * ysize;
	PTYPE 	*hf = new PTYPE[size];

	memset (hf, 0, size);
	init (hf, xsize, ysize, (generateName ? NULL : getName()));
}


/*
 *  backup: make a backup copy of the current height field
 */
void HeightField::backup ()
{
	SanityCheck::bailout ((!this->p_hf), "this->p_hf==NULL", "HeightField::backup");

	if (p_hfBck)
		delete [] p_hfBck;
	p_hfBck = new PTYPE[d_size];
	memcpy (p_hfBck, this->p_hf, sizeof(PTYPE)*this->d_size);
	p_hfBckSealevel = d_sealevel;
	d_backupTime = d_updateTime;
}


/*
 *  restore: restore a backup copy of the current height field. Doesn't
 *  	     calculate statistics! 
 */
void HeightField::restoreBackup ()
{
	SanityCheck::bailout ((!this->p_hf), "this->p_hf==NULL", "HeightField::restore");
	SanityCheck::bailout ((!p_hfBck), "p_hfBck==NULL", "HeightField::restore");

	memcpy (this->p_hf, p_hfBck, sizeof(PTYPE)*this->d_size);
	d_sealevel = p_hfBckSealevel;
	d_updateTime = d_backupTime;
}


/*
 *  destroyBackup: destroy the backup copy of the current height field and 
 * 	free the memory. 
 */
void HeightField::destroyBackup ()
{
	if (!p_hfBck)
		return;

	delete [] p_hfBck;
	p_hfBck = NULL;
	d_backupTime = 0;
}


/*
 *  destroyContourList: destroy the contour list
 */
void HeightField::destroyContourList ()
{
	if (!p_contourList)
		return;

	delete p_contourList;
	p_contourList = NULL;
	d_contourTime = 0;
}

