/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "FileIO.h"
#include "GlobalDefs.h"
#include "GlobalSanityCheck.h"
#include "GlobalTrace.h"



/*
 *  constructor: give everyting a meaningful default value
 */
FileIO::FileIO (const char *fName)
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ FileIO\n");

	SanityCheck::bailout ((!fName), "fName==NULL", "FileIO::FileIO");
	SanityCheck::bailout ((!strlen(fName)), "strlen(fName)==0", "FileIO::FileIO");

	d_File = NULL;
	d_fName = strdup (fName);
	d_baseName = NULL;
	d_extension = NULL;
}


/*
 *  destructor: clean up, delete any dyncamically allocated memory
 */
FileIO::~FileIO ()
{
	if (d_fName) 
		free (d_fName);			// C-style alloc by strdup
	if (d_baseName)
		free (d_baseName);		// C-style alloc by strdup
	if (d_extension)
		free (d_extension);		// C-style alloc by strdup

	if (d_File)
		this->close ();

	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- FileIO\n");
}


/*
 *  open: open the file with the specified mode and the file pointer
 */
FILE *FileIO::open (char *mode)
{
	SanityCheck::bailout ((!mode), "mode==NULL", "FileIO::open");
	d_File = fopen (d_fName, mode);
	if (!d_File)
		{
		char	buf[256];
		sprintf (buf, _("opening file [%s] failed!"), d_fName);
		SanityCheck::warning (TRUE, buf, "FileIO::open");
		}
	return d_File;
}


/*
 *  close: close the file descriptor
 */
int FileIO::close ()
{
	if (d_File)
		{
		FILE	*f = d_File;
		d_File = NULL;
		fflush (f);
		return (fclose (f));
		}
	else
		return (-1);
}


/*
 *  getExtension: return a pointer to the filename extension or NULL
 */
char *FileIO::getExtension (int include_dot)
{
	char	*last=NULL; 		// last found dot
	int	len; 			// full string length

	// only do this once
	if (d_extension)
		return (d_extension);

	len = strlen (d_fName);

	// find last dot
	for (int i=len-1; i>0; i--)
		if (d_fName[i] == '.')
			{
			last=&d_fName[i];
			i=0;
			}

	if (last)
		{
		if (!include_dot)
			last++;
		d_extension = strdup (last);
		}

	return (d_extension);
}


/*
 *  getBasename: return a pointer to a copy of the file basename
 */
char *FileIO::getBasename ()
{
	char	*dot; 
	int	len;

	// only do this once
	if (d_baseName)
		return (d_baseName);

	// find first dot in name 
	dot = strchr (d_fName, '.');

	if (!dot)
		return (d_fName);

	len = dot - d_fName;
	d_baseName = strdup (d_fName);
	d_baseName[len] = '\0';
	return (d_baseName);
}


/*
 *  exists: check if the file exists
 */
bool FileIO::exists ()
{
	struct stat	stat_info;

	// stat returns -1 on error, 0 on success
	if (stat (d_fName, &stat_info))
		return (FALSE);
	else
		return (TRUE);
}



/*
 *  promptOverwrite: ask the user if the file can be overwritten
 */
bool FileIO::promptOverwrite ()
{
	unsigned char t;	

	fprintf (stdout, _("File [%s] already exists.  Overwrite (Y/N)? "), d_fName);
	t = toupper (fgetc (stdin));

	if (t == 'Y')
		return TRUE;

	return (FALSE);
}



