/*
                     Object Management Utilities

	Functions:

        int DBCreateObjectWeapons(long object_num, int total_weapons)
        int DBCreateObjectScores(long object_num)
        int DBCreateObjectEconomy(long object_num)
        int DBCreateObjectEconomyProduct(
                long object_num,
                char *product_name,
                xswo_credits_t sell_price,
		xswo_credits_t buy_price,
                double product_amount,
                double product_max
        )

        int DBObjectTractor(long src_obj, long tar_obj)
	int DBIsObjectTractorablePtr(
		xsw_object_struct *src_obj_ptr,
		xsw_object_struct *tar_obj_ptr
	)
        void DBObjectUntractor(long src_obj, long tar_obj)

	void DBSortEconomyProducts(long object_num)

        char *DBGetTypeName(long type)
        char *DBGetFormalNameStr(long object_num)

        int DBValidateObjectName(char *name)
        int DBValidateObjectPassword(char *password)

        long DBGetTopObjectNumber()

 */

#include "../include/unvfile.h"
#include "swserv.h"



/*
 *      Allocates or deallocates weapons on object_num.
 */
int DBCreateObjectWeapons(long object_num, int total_weapons)
{
        int old_total_weapons;
        int weapon_num;
        xsw_object_struct *obj_ptr;
            
        
        /* Check if object_num is valid. */
        if(DBIsObjectGarbage(object_num))
            return(-1);
        else
            obj_ptr = xsw_object[object_num];
           
           
        /* Free all weapons on object? */
        if(total_weapons <= 0)
        {
            /* Deallocate all weapons. */
            for(weapon_num = 0;
                weapon_num < obj_ptr->total_weapons;
                weapon_num++
            )
            {
#ifdef DEBUG_MEM_FREE
if(obj_ptr->weapons[weapon_num] != NULL)
    printf("Object weapon %i: Free'ed.\n", weapon_num);
#endif
                free(obj_ptr->weapons[weapon_num]);
                obj_ptr->weapons[weapon_num] = NULL;
            }
#ifdef DEBUG_MEM_FREE
if(obj_ptr->weapons != NULL)
    printf("Object weapon pointers: Free'ed.\n");
#endif
            free(obj_ptr->weapons);
            obj_ptr->weapons = NULL;

            obj_ptr->total_weapons = 0;
            obj_ptr->selected_weapon = -1;

            return(0); 
        }


        /* Sanitize requested total_weapons. */
        if(total_weapons > MAX_WEAPONS)  
            total_weapons = MAX_WEAPONS;

        /* Sanitize total_weapons on object. */
        if(obj_ptr->total_weapons > MAX_WEAPONS)
            obj_ptr->total_weapons = MAX_WEAPONS;   
        if(obj_ptr->total_weapons < 0)
            obj_ptr->total_weapons = 0;

        /* Record old total weapons on object. */
        old_total_weapons = obj_ptr->total_weapons;


        /* ************************************************************ */
        /* Allocate or deallocate total weapons on object. */
             
        if(old_total_weapons == total_weapons)
        {
            /* Total weapons remains the same, do nothing. */
            return(0);
        }
        else if(old_total_weapons > total_weapons)
        {   
            /* Deallocate weapons on object. */
            for(weapon_num = old_total_weapons - 1;
                weapon_num >= total_weapons;
                weapon_num--
            )
            {
                /* Unselect weapon if it will be deleted. */
                if(obj_ptr->selected_weapon == weapon_num)
                    obj_ptr->selected_weapon = -1;
            
#ifdef DEBUG_MEM_FREE
if(obj_ptr->weapons[weapon_num] != NULL)
    printf("Object weapon %i: Free'ed.\n", weapon_num);
#endif
                free(obj_ptr->weapons[weapon_num]);
                obj_ptr->weapons[weapon_num] = NULL;
            }
            obj_ptr->weapons = (xsw_weapons_struct **)realloc(
                obj_ptr->weapons,
                total_weapons * sizeof(xsw_weapons_struct *)
            );
            if(obj_ptr->weapons == NULL)
            {
                obj_ptr->total_weapons = 0;
                obj_ptr->selected_weapon = -1;
                return(-1);
            }
            
            /* Adjust the object's total_weapons to reflect changes. */
            obj_ptr->total_weapons = total_weapons;
        }   
        else
        {
            /* Allocate more weapons on object. */
            obj_ptr->weapons = (xsw_weapons_struct **)realloc(
                obj_ptr->weapons,
                total_weapons * sizeof(xsw_weapons_struct *)
            );
            if(obj_ptr->weapons == NULL)
            {
                obj_ptr->total_weapons = 0;
                obj_ptr->selected_weapon = -1;
                return(-1);
            }

            for(weapon_num = old_total_weapons;
                weapon_num < total_weapons;
                weapon_num++
            )
            {
                obj_ptr->weapons[weapon_num] = (xsw_weapons_struct *)calloc(
                    1,
                    sizeof(xsw_weapons_struct)
                );
                if(obj_ptr->weapons[weapon_num] == NULL)
                {
                    obj_ptr->total_weapons = 0;
                    obj_ptr->selected_weapon = -1;
                    return(-1);
                }
            }

            /* Adjust the object's total_weapons to reflect changes. */
            obj_ptr->total_weapons = total_weapons;
        }


        /* Sanitize selected weapon. */
        if(obj_ptr->selected_weapon >= obj_ptr->total_weapons)
            obj_ptr->selected_weapon = obj_ptr->total_weapons - 1;
        if(obj_ptr->selected_weapon < 0)
            obj_ptr->selected_weapon = -1;


        return(0);
}


/*
 *      Allocates scores structure on object_num as needed.
 *      Returns -1 on error.
 */
int DBCreateObjectScores(long object_num)
{
        static xsw_object_struct *obj_ptr;


        /* Error checks. */
        if(DBIsObjectGarbage(object_num))
            return(-1);
        else
            obj_ptr = xsw_object[object_num];


        /* Allocate score structure as needed. */
        if(obj_ptr->score == NULL)
        {
            obj_ptr->score = (xsw_score_struct *)calloc(
                1,
                sizeof(xsw_score_struct)
            );
        }


        return((obj_ptr->score == NULL) ? -1 : 0);
}


/*          
 *      Allocates economy structure on object_num as needed.
 *	Returns -1 on error.
 */
int DBCreateObjectEconomy(long object_num)
{  
        static xsw_object_struct *obj_ptr;


        /* Error checks. */
        if(DBIsObjectGarbage(object_num))
            return(-1);
        else
            obj_ptr = xsw_object[object_num];


        /* Allocate economy structure as needed. */
        if(obj_ptr->eco == NULL)
        {
            obj_ptr->eco = (xsw_ecodata_struct *)calloc(
                1,
                sizeof(xsw_ecodata_struct)
            );
        }


        return((obj_ptr->eco == NULL) ? -1 : 0);
}


/*
 *      Allocates economy product structure on object_num as needed.
 *      Returns -1 on error.
 */
int DBCreateObjectEconomyProduct(
        long object_num,
        char *product_name,
        xswo_credits_t sell_price,
        xswo_credits_t buy_price,
        double product_amount,
        double product_max
)
{
        int i, n;
        xsw_ecodata_struct *eco_ptr;
        xsw_object_struct *obj_ptr;
   
             
        if(product_name == NULL)
            return(-2);

        if(DBIsObjectGarbage(object_num))
            return(-1);
        else
            obj_ptr = xsw_object[object_num];
        
              
        /* Allocate economy structure as needed. */
        if(obj_ptr->eco == NULL)
        {
            if(DBCreateObjectEconomy(object_num))
                return(-1);
        }
                
        /* Get pointer to eco data. */
        eco_ptr = obj_ptr->eco;
 
        /* Sanitize total. */
        if(eco_ptr->total_products < 0)
            eco_ptr->total_products = 0;
        
        /* Is product_name in list? */
        for(i = 0, n = -1; i < eco_ptr->total_products; i++)
        {
            if(eco_ptr->product[i] == NULL)
	    {
		n = i;
                continue;
	    }

            if(!strcasecmp(eco_ptr->product[i]->name, product_name))
                break;
        }
        if(i >= eco_ptr->total_products)
        {
            /* No such product, allocate a new one. */

	    if(n > -1)
	    {
		/* Have allocated pointer n. */

		i = n;
	    }
	    else
	    {
		/* No unallocated pointers, allocate more. */

                i = eco_ptr->total_products; 

                eco_ptr->total_products += 1;

                eco_ptr->product = (xsw_ecoproduct_struct **)realloc(
                    eco_ptr->product,
                    eco_ptr->total_products * sizeof(xsw_ecoproduct_struct *)
                );
                if(eco_ptr->product == NULL)
                {
                    eco_ptr->total_products = 0;
                    return(-1);
                }
	    }

	    /* Allocate structure. */
            eco_ptr->product[i] = (xsw_ecoproduct_struct *)calloc(
		1,
                sizeof(xsw_ecoproduct_struct)
            );
            if(eco_ptr->product[i] == NULL)
            {
                return(-1);
            }
        }


        /* New product index is now i. */
        strncpy(
            eco_ptr->product[i]->name,
            product_name,
            ECO_PRODUCT_NAME_MAX
        );
        eco_ptr->product[i]->name[ECO_PRODUCT_NAME_MAX - 1] = '\0';

        eco_ptr->product[i]->sell_price = sell_price;
	eco_ptr->product[i]->buy_price = buy_price;
        eco_ptr->product[i]->amount = product_amount;
        eco_ptr->product[i]->amount_max = product_max;



        return(0);
}


/*
 *      Sets src_obj to lock tractor beam on tar_obj.
 */
int DBObjectTractor(long src_obj, long tar_obj)
{
        static xsw_object_struct *obj_ptr;
            
        
        /* Error checks. */
        if(DBIsObjectGarbage(src_obj))
            return(-1);
        else
            obj_ptr = xsw_object[src_obj];
              
/* If tar_obj is -1 then unlock all tractor beam objects. */
if(tar_obj < 0)
{        
#ifdef DEBUG_MEM_FREE
if(obj_ptr->tractored_object != NULL)
    printf("Object tractored objects: Free'ed.\n");
#endif
        free(obj_ptr->tractored_object);
        obj_ptr->tractored_object = NULL; 
              
        obj_ptr->total_tractored_objects = 0;
   
        return(0);
}
 
 
        /* tar_obj is assumed valid. */
  
 
        /* Sanitize total_tractored_objects. */
        if(obj_ptr->total_tractored_objects < 0)
            obj_ptr->total_tractored_objects = 0;
        
        /* Skip if maximum is excceded. */
        if(obj_ptr->total_tractored_objects > MAX_TRACTORED_OBJECTS)
            return(-3);
        
            
        /* Allocate new tractored object. */
        obj_ptr->total_tractored_objects += 1;
        obj_ptr->tractored_object = (long *)realloc(
            obj_ptr->tractored_object,
            obj_ptr->total_tractored_objects * sizeof(long)
        );
        if(obj_ptr->tractored_object == NULL)
        {
            obj_ptr->total_tractored_objects = 0;
            return(-1);
        }
 
        /* Set new tractored object. */
        obj_ptr->tractored_object[obj_ptr->total_tractored_objects - 1] =
            tar_obj;


        return(0);
}


/*
 *	Checks if source object can tractor target object.
 */
int DBIsObjectTractorablePtr(
	xsw_object_struct *src_obj_ptr,
	xsw_object_struct *tar_obj_ptr
)
{
	if((src_obj_ptr == NULL) ||
           (tar_obj_ptr == NULL)
	)
	    return(0);

	/* Check if objects are in range. */
	if(!Mu3DInRangePtr(src_obj_ptr, tar_obj_ptr, MAX_TRACTOR_BEAM_LEN))
	    return(0);

	/*   Check if target object has shields up and if so, check
	 *   if target object's power is greater than the source
	 *   object's.
	 */
	if((tar_obj_ptr->shield_state == SHIELD_STATE_UP) &&
           ((tar_obj_ptr->power * tar_obj_ptr->power_purity) >=
            (src_obj_ptr->power * src_obj_ptr->power_purity)
	   )
	)
	    return(0);


	return(1);
} 


/*
 *      Looks for tar_obj in tractored objects list on
 *      src_obj and unlocks.
 */
void DBObjectUntractor(long src_obj, long tar_obj)
{
        static int i, n;
        static xsw_object_struct *obj_ptr;


        /* Error checks. */
        if(DBIsObjectGarbage(src_obj))
            return;
        else
            obj_ptr = xsw_object[src_obj];


        /* Sanitize total. */
        if(obj_ptr->total_tractored_objects < 0)   
            obj_ptr->total_tractored_objects = 0;

/* If tar_obj is -1 then unlock all tractor beam objects. */
if(tar_obj < 0)
{
#ifdef DEBUG_MEM_FREE
if(obj_ptr->tractored_object != NULL)
    printf("Object tractored objects: Free'ed.\n");
#endif
        free(obj_ptr->tractored_object);
        obj_ptr->tractored_object = NULL;
  
        obj_ptr->total_tractored_objects = 0;
 
        return;
}  


        /* tar_obj is assumed valid. */


        /* Untractor object. */
        for(i = 0; i < obj_ptr->total_tractored_objects; i++)
        {
            if(obj_ptr->tractored_object[i] == tar_obj)
                obj_ptr->tractored_object[i] = -1;
        }


        /* Reclaim tractored objects memory? */
        n = -1;
        for(i = 0; i < obj_ptr->total_tractored_objects; i++)
        {
            if(DBIsObjectGarbage(obj_ptr->tractored_object[i]))
                continue;
            else
                n = i;
        }

        /* Reclaim memory as needed. */
        obj_ptr->total_tractored_objects = n + 1;
        if(obj_ptr->total_tractored_objects <= 0)
        {
#ifdef DEBUG_MEM_FREE
if(obj_ptr->tractored_object != NULL)
    printf("Object tractored objects: Free'ed.\n");
#endif
            free(obj_ptr->tractored_object);
            obj_ptr->tractored_object = NULL;
            obj_ptr->total_tractored_objects = 0;
        }
        else
        {
            obj_ptr->tractored_object = (long *)realloc(
                obj_ptr->tractored_object,
                obj_ptr->total_tractored_objects * sizeof(char)
            );
            if(obj_ptr->tractored_object == NULL)
            {
                obj_ptr->total_tractored_objects = 0;
            }
        }


        return;
}



/*
 *	Alphabitizes the products on object_num by name of
 *	the product.  If object has no economy or has economy
 *	but no products, then no operation is performed.
 */
void DBSortEconomyProducts(long object_num)
{
	int i, n, k;
	xsw_ecodata_struct *eco_ptr;
	xsw_object_struct *obj_ptr;
	xsw_ecoproduct_struct **product_ptr;	/* To hold tmp ptrs. */
	char **strv;
	int strc;


	if(DBIsObjectGarbage(object_num))
	    return;
	else
	    obj_ptr = xsw_object[object_num];


	eco_ptr = obj_ptr->eco;
	if(eco_ptr == NULL)
	    return;

	if(eco_ptr->total_products <= 0)
	    return;


	/* Build products name list in strv. */
	for(i = 0, strv = NULL, strc = 0;
            i < eco_ptr->total_products;
            i++
	)
	{
	    if(eco_ptr->product[i] == NULL)
		continue;

	    /* Copy name to string list. */
	    strc++;
	    strv = (char **)realloc(strv, strc * sizeof(char *));
	    if(strv == NULL)
		return;

	    strv[strc - 1] = StringCopyAlloc(
		eco_ptr->product[i]->name
	    );
	    if(strv[strc - 1] == NULL)
	    {
		strc--;
		continue;
	    }
	}

	/* Got no strings? */
	if(strc == 0)
	    return;

	/* Sort strings. */
	strv = StringQSort(strv, strc);
	if(strv == NULL)
	    return;


	/* Copy products pointers to tempory pointer array. */
	product_ptr = (xsw_ecoproduct_struct **)calloc(
	    1,
	    eco_ptr->total_products * sizeof(xsw_ecoproduct_struct *)
	);
	memcpy(
	    product_ptr,	/* Destination. */
	    eco_ptr->product,	/* Source. */
	    eco_ptr->total_products * sizeof(xsw_ecoproduct_struct *)
	);


	/* Sort products with respect to the strv list. */
	for(i = 0, k = 0; i < strc; i++)
        {
	    if(k >= eco_ptr->total_products)
		break;

	    if(strv[i] == NULL)
		continue;

	    /* Get pointer matching string in tempory products list. */
	    for(n = 0; n < eco_ptr->total_products; n++)
	    {
                if(product_ptr[n] == NULL)
                    continue;

		if(!strcasecmp(product_ptr[n]->name, strv[i]))
		    break;
	    }
	    if(n < eco_ptr->total_products)
	    {
		/* Matched product name, now put it in order. */

		eco_ptr->product[k] = product_ptr[n];
	        k++;
	    }
	    else
	    {
		fprintf(stderr,
 "DBSortEconomyProducts(): Warning: Missing name from tmp products array (#%ld %i %i)\n",
		    object_num, strc, eco_ptr->total_products
		);
	    }
	}
	/* Reallocate the rest of the product pointers. */
	if(k > 0)
	{
	    eco_ptr->total_products = k;
	    eco_ptr->product = (xsw_ecoproduct_struct **)realloc(
		eco_ptr->product,
		eco_ptr->total_products * sizeof(xsw_ecoproduct_struct *)
	    );
	    if(eco_ptr->product == NULL)
	    {
		eco_ptr->total_products = 0;
		/* Don't return, need to free other stuff. */
	    }
	}
	else
	{
	    free(eco_ptr->product);
	    eco_ptr->product = NULL;

	    eco_ptr->total_products = 0;
	}


	/* Free tempory products pointers. */
	free(product_ptr);
	product_ptr = NULL;

	/* Free string array. */
	StringFreeArray(strv, strc);


	return;
}


/*
 *      Return a statically allocated string containing the official
 *      name for the given object type code.
 */     
char *DBGetTypeName(long type)
{
        switch(type)
        {
          case XSW_OBJ_TYPE_ERROR:
            return(XSW_TYPE_NAME_ERROR);
            break;

          case XSW_OBJ_TYPE_GARBAGE:
            return(XSW_TYPE_NAME_GARBAGE);
            break;

          case XSW_OBJ_TYPE_STATIC:
            return(XSW_TYPE_NAME_STATIC);
            break; 

          case XSW_OBJ_TYPE_DYNAMIC:
            return(XSW_TYPE_NAME_DYNAMIC);
            break;

          case XSW_OBJ_TYPE_CONTROLLED:
            return(XSW_TYPE_NAME_CONTROLLED);
            break;

          case XSW_OBJ_TYPE_PLAYER:
            return(XSW_TYPE_NAME_PLAYER);
            break;
  
          case XSW_OBJ_TYPE_WEAPON:
            return(XSW_TYPE_NAME_WEAPON);   
            break;

          case XSW_OBJ_TYPE_STREAMWEAPON:
            return(XSW_TYPE_NAME_STREAMWEAPON);
            break;

          case XSW_OBJ_TYPE_SPHEREWEAPON:
            return(XSW_TYPE_NAME_SPHEREWEAPON);
            break;

          case XSW_OBJ_TYPE_HOME:
            return(XSW_TYPE_NAME_HOME);
            break;

          case XSW_OBJ_TYPE_AREA:
            return(XSW_TYPE_NAME_AREA);
            break;

          case XSW_OBJ_TYPE_ANIMATED:
            return(XSW_TYPE_NAME_ANIMATED);
            break;

          default:
            return("*Unknown*");
            break;
        }

        return("*Unknown*");
}


/*
 *      Returns a statically allocated format name string in
 *      the format of Name(#12345)
 */
char *DBGetFormalNameStr(long object_num)
{
        if(DBIsObjectGarbage(object_num))
            return(XSW_OBJ_GARBAGE_NAME);
        else
            return(UNVGetObjectFormalName(
                xsw_object[object_num],
                object_num
            ));
}



/*      
 *      Validates name for an object.  Returns non-zero if it is
 *      NOT valid.
 *      
 *      -1  general error.
 *      0   no error.
 *      1   name too long.
 *      2   name too short.
 */        
int DBValidateObjectName(char *name)
{          
        int i, n;  
        char *strptr = INVALID_NAME_CHARACTERS;


        /* Null pointer? */
        if(name == NULL) return(-1);

        /* Empty string? */
        if(name[0] == '\0') return(2);

        /* Invalid characters? */
        for(i = 0; name[i] != '\0'; i++)   
        {
            if(i >= XSW_OBJ_NAME_MAX)
                return(1);

            for(n = 0; strptr[n] != '\0'; n++)
            {
                if(name[i] == strptr[n])
                    return(3);
            }
        }


        return(0);
}


/*         
 *      Checks if this is a valid password.
 */        
int DBValidateObjectPassword(char *password)
{




        return(0);
}



/*        
 *      Returns the number of the highest non-garbage XSW Object.
 */
long DBGetTopObjectNumber()
{
        static long current_object;
        static long last_valid_object; 


        /* No objects allocated? */
        if(total_objects <= 0)
            return(-1);
        else if(xsw_object == NULL)
        {
            fprintf(stderr,
                "DBGetTopObjectNumber(): Error: Object pointers is NULL.\n"
            );
            return(-1);  
        }

        /* Get last valid object. */
        last_valid_object = -1;
        for(current_object = 0;
            current_object < total_objects;
            current_object++
        )
        {
            if(xsw_object[current_object] == NULL)
            {
                fprintf(stderr,
    "DBGetTopObjectNumber(): Error: Object #%ld not allocated.\n",
                    current_object
                );
                continue;
            }
 
            if(xsw_object[current_object]->type <= XSW_OBJ_TYPE_GARBAGE)
                continue;

            last_valid_object = current_object;
        }
        
        
        return(last_valid_object); 
}
