## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##

# Display a tree in a canvas. Based on newtree.py by, a very unfinished
# widget by <flognat@fukt.hk-r.se>. In the end I would have been faster
# if I had written this entirely from scratch...

# imports
import os, string, types
import Tkinter

# Toolkit imports
from tkwidget import MfxScrolledCanvas                              #bundle#


# /***********************************************************************
# //
# ************************************************************************/

class _TkTreeBaseNode:
    def __init__(self, tree, parent, text):
        self.tree = tree
        self.parent = parent
        self.text = text
        self.color = "black"
        self.selected = 0
        # canvas item ids
        self.symbol_id = -1
        self.text_id = -1

    def whoami(self):
        if self.parent is None:
            return (self.text, )
        else:
            return self.parent.whoami() + (self.text, )

    def draw(self, x, y, lastx=None, lasty=None):
        canvas = self.tree.canvas
        style = self.tree.style
        topleftx = x + style.distx
        toplefty = y - style.height / 2
        # draw the horizontal line
        if lastx is not None:
            canvas.create_line(x, y, topleftx, y, stipple=style.linestyle, fill=style.linecolor)
        # draw myself
        self.symbol_id = self.drawSymbol(topleftx, toplefty)
        linestart = style.distx + style.width + 5
        self.text_id = canvas.create_text(x + linestart, y, text=self.text,
                                          justify="left", anchor="w",
                                          font=style.font)
        # link canvas items back to node
        self.tree.nodes[self.symbol_id] = self
        self.tree.nodes[self.text_id] = self
        return x, y, x, y + style.disty

    def drawSymbol(self, x, y):
        style = self.tree.style
        # note: outline is one pixel
        return self.tree.canvas.create_rectangle(
            x+1, y+1, x + style.width, y + style.height, fill=self.color)


class Leaf(_TkTreeBaseNode):
    def __init__(self, tree, parent, text, selected=0):
        _TkTreeBaseNode.__init__(self, tree, parent, text)
        self.selected = selected
        self.updateSymbol()

    def updateSymbol(self):
        if self.selected:
            self.color = "darkgreen"
        else:
            self.color = "green"
        if self.symbol_id >= 0:
            self.tree.canvas.itemconfig(self.symbol_id, fill=self.color)


class Node(_TkTreeBaseNode):
    def __init__(self, tree, parent, text, expanded=0):
        _TkTreeBaseNode.__init__(self, tree, parent, text)
        self.expanded = expanded
        self.subnodes = None
        self.updateSymbol()

    def updateSymbol(self):
        if self.expanded:
            self.color = "red"
        else:
            self.color = "pink"
        if self.symbol_id >= 0:
            self.tree.canvas.itemconfig(self.symbol_id, fill=self.color)

    def drawChildren(self, x, y, lastx, lasty):
        # get subnodes
        self.subnodes = self.tree.getContents(self)
        # draw subnodes
        lx, ly = lastx, lasty
        nx, ny = x, y
        for node in self.subnodes:
            # update tree
            node.tree = self.tree
            # draw node
            lx, ly, nx, ny = node.draw(nx, ny, lx, ly)
        # draw the vertical line
        if self.subnodes:
            style = self.tree.style
            self.tree.canvas.create_line(x, y-style.disty/2, nx, ly,
                                         stipple=style.linestyle, fill=style.linecolor)
        return ny


    def draw(self, x, y, ilastx=None, ilasty=None):
        # draw myself
        lx, ly, nx, ny = _TkTreeBaseNode.draw(self, x, y, ilastx, ilasty)
        if self.expanded:
            style = self.tree.style
            childx = nx + style.distx + style.width / 2
            childy = ny
            clastx = nx + style.distx + style.width / 2
            clasty = ly + style.height /2
            ny = self.drawChildren(childx, childy, clastx, clasty)
        return lx, ly, x, ny


class TreeInCanvas(MfxScrolledCanvas):
    class Style:
        def __init__(self):
            self.distx = 16
            self.disty = 18
            self.width = 16         # width of symbol
            self.height = 16        # height of symbol
            self.originx = 0
            self.originy = 0
            self.linestyle = "gray50"
            self.linecolor = "black"
            self.font = ("Helvetica", 10)
            if os.name == "nt":
                self.linestyle = ""                 # Tk bug ?
                self.linecolor = "gray50"
                self.font = ("MS Sans Serif", "8")

    def __init__(self, parent, rootnodes, c_width=400, c_height=300, **kw):
        self.nodes = {}
        self.rootnodes = rootnodes
        self.style = self.Style()
        #
        kw["bg"] = kw.get("bg", parent.cget("bg"))
        self.sc = apply(MfxScrolledCanvas, (parent,), kw)
        self.canvas = self.sc.canvas
        self.canvas.config(width=c_width, height=c_height)
        self.canvas.bind("<ButtonPress-1>", self.singleClick)
        #self.canvas.bind("<ButtonRelease-1>", xxx)
        ##self.canvas.bind("<Double-Button-1>", self.doubleClick)
        self.sc.frame.pack(fill=Tkinter.BOTH, expand=1)

    def destroy(self):
        for v in self.nodes.values():
            v.tree = None
        self.clear()
        self.sc.unbind(self)
        self.sc.destroy(self)

    def findNode(self):
        id = self.canvas.find_withtag(Tkinter.CURRENT)
        if id:
            return self.nodes.get(id[0])
        return None

    #
    # draw nodes
    #

    def draw(self):
        nx, ny = self.style.originx, self.style.originy
        # Account for initial offsets, see topleft[xy] in BaseNode.draw().
        # We do this so that our bounding box always starts at (0,0)
        # and the yscrollincrement works nicely.
        nx = nx - self.style.distx
        ny = ny + self.style.height / 2
        for node in self.rootnodes:
            # update tree
            node.tree = self
            # draw
            lx, ly = None, None
            lx, ly, nx, ny = node.draw(nx, ny, lx, ly)
        # set scroll region
        bbox = self.canvas.bbox("all")
        ##self.canvas.config(scrollregion=bbox)
        self.canvas.config(scrollregion=(0,0,bbox[2],bbox[3]))
        self.canvas.config(yscrollincrement=self.style.disty)

    def clear(self):
        self.nodes = {}
        self.canvas.delete("all")

    def redraw(self):
        oldcur = self.canvas["cursor"]
        self.canvas["cursor"] = "watch"
        self.canvas.update_idletasks()
        self.clear()
        self.draw()
        self.canvas["cursor"] = oldcur

    def getContents(self, node):
        # Overload this, supposed to return a list of subnodes of node.
        pass

    def singleClick(self, event=None):
        # Overload this if you want to know when a node is clicked on.
        pass

    def doubleClick(self, event=None):
        # Overload this if you want to know when a node is d-clicked on.
        self.singleClick(event)


# /***********************************************************************
# //
# ************************************************************************/

#%ifndef BUNDLE

class DirectoryBrowser(TreeInCanvas):
    def __init__(self, parent, dirs):
        nodes = []
        if type(dirs) == types.StringType:
            dirs = (dirs,)
        for dir in dirs:
            self.addNode(nodes, None, dir, dir)
        # note: best results if c_height is a multiple of style.disty
        TreeInCanvas.__init__(self, parent, nodes, c_height=25*18, hbar=1)
        self.draw()

    def addNode(self, list, node, filename, text):
        try:
            if os.path.isdir(filename):
                list.append(Node(self, node, text))
            else:
                list.append(Leaf(self, node, text))
        except EnvironmentError:
            pass

    def getContents(self, node):
        # use cached values
        if node.subnodes is not None:
            return node.subnodes
        #
        thepath = apply(os.path.join, node.whoami())
        print "Getting %s" % thepath
        try:
            filenames = os.listdir(thepath)
            filenames.sort()
        except EnvironmentError:
            return ()
        contents = []
        for filename in filenames:
            self.addNode(contents, node, os.path.join(thepath, filename), filename)
        ##print "gotten"
        return contents

    def singleClick(self, event=None):
        node = self.findNode()
        if node is None:
            return
        print "Pressed", node
        if isinstance(node, Leaf):
            node.selected = not node.selected
            node.updateSymbol()
        else:
            node.expanded = not node.expanded
            node.updateSymbol()
            self.redraw()


if __name__ == "__main__":
    tk = Tkinter.Tk()
    ##app = DirectoryBrowser(tk, "/")
    app = DirectoryBrowser(tk, ("/","/home"))
    app.sc.frame.mainloop()

#%endif

