#! /usr/bin/env python
##
## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys, os, glob, re, string, time, types
import traceback                                                    #bundle#

# PySol imports
from mfxutil import destruct, dumpmem, Struct, pickle, unpickle     #bundle#
from mfxutil import merge_dict, gethomedir, getprefdir, EnvError    #bundle#
from util import DataLoader, Timer, cyclops                         #bundle#
from util import Cardset, CardsetManager                            #bundle#
from util import PACKAGE, VERSION                                   #bundle#
from random import LCRandom64                                       #bundle#
from gamedb import GAME_DB, loadGame                                #bundle#
from pysolaudio import PysolAudio                                   #bundle#

# Toolkit imports
from pysoltk import tkname, withdraw                                #bundle#
from pysoltk import bind, unbind_destroy, after_idle                #bundle#
from pysoltk import MfxDialog, MfxExceptionDialog                   #bundle#
from pysoltk import TclError, MfxRoot, MfxCanvas, MfxImages         #bundle#
from pysoltk import PysolMenubar                                    #bundle#
from pysoltk import PysolProgressBar                                #bundle#
from pysoltk import PysolToolbar                                    #bundle#
from pysoltk import PysolStatusbar                                  #bundle#
from help import helpAbout                                          #bundle#


# /***********************************************************************
# // init games database
# ************************************************************************/

# import the games
#%ifndef BUNDLE
path = sys.path[:]
sys.path.insert(0, os.path.join(sys.path[0], "games"))
try:
    names = os.listdir(sys.path[0])
except EnvError:
    names = []
names.sort()
for name in names:
    m = re.search(r'^(.+)\.py$', name)
    if m:
##        try:
        exec "import " + m.group(1)
##        except:
##            traceback.print_exc()
sys.path = path
del path, names, name
#%endif


# /***********************************************************************
# // these classes must reside in module __main__ because of
# // pickling compatibility
# ************************************************************************/

class Options:
    def __init__(self):
        self.player = "unknown"     # initialized from Statistics.last_player
        # options menu:
        self.confirm = 1
        self.autofaceup = 0
        self.autodrop = 0
        self.autodeal = 1
        self.quickplay = 0
        self.undo = 1
        self.hint = 1
        self.highlight_piles = 1
        self.highlight_cards = 1
        self.highlight_samerank = 1
        self.cardset_name = None
        self.cardback_name = None
        if os.name == "posix":
            self.tabletile_name = "Fade_Green.ppm"
        else:
            self.tabletile_name = ""
        self.tablecolor = "#008200"
        self.animations = 2
        self.shadow = 0
        self.shade = 0
        self.hint_sleep = 1.5
        self.demo_sleep = 1.5
        self.toolbar = 1
        self.statusbar = 1
        # not changeable:
        self.sound = 0
        self.toolbar_dir = "classic-large"
        self.toolbar_dir = "gnome-large"
        self.toolbar_dir = "kde-large"
        self.toolbar_relief = 0
        self.dragcursor = 1
        self.magnetic_mouse = 0
        self.magnetic_mouse_time = 2.0      # seconds
        self.raise_card_sleep = 1.0
        self.highlight_piles_sleep = 1.5
        self.highlight_piles_colors = (None, "#ffc000")
        self.highlight_cards_sleep = 1.5
        self.highlight_cards_colors = (None, "#ffc000", None, "#0000ff")
        self.highlight_samerank_sleep = 1.5
        self.highlight_samerank_colors = (None, "#ffc000", None, "#0000ff")
        self.hintarrow_color = "#303030"

    def copy(self):
        opt = Options()
        merge_dict(opt.__dict__, self.__dict__)
        return opt


class Statistics:
    def __init__(self):
        # a dictionary of dictionaries of tuples (won/lost),
        # indexed by player and gameid
        self.stats = {}
        self.demo_stats = {}
        # a dictionary of lists of tuples
        self.prev_games = {}
        # additional startup information
        self.last_gameid = 0        # last game played
        self.last_player = None     # last player
        self.last_save_dir = None   # last directory for load/save

    #
    # player statistics
    #

    def resetStats(self, player, gameid):
        if player is None:
            self.__resetDemoStats(gameid)
            return
        assert player and gameid >= 0
        self.resetPrevGames(player, gameid)
        if not self.stats.has_key(player):
            return
        if gameid == 0:
            del self.stats[player]
        else:
            w0, l0 = self.getStats(player, 0)
            w1, l1 = self.getStats(player, gameid)
            self.stats[player][0] = (w0 - w1, l0 - l1)
            self.stats[player][gameid] = (0, 0)

    def resetPrevGames(self, player, gameid):
        assert player and gameid >= 0
        if not self.prev_games.has_key(player):
            return
        if gameid == 0:
            del self.prev_games[player]
        else:
            self.prev_games[player] = filter(lambda a, b=gameid: a[0] != b, self.prev_games[player])

    def getStats(self, player, gameid):
        if player is None:
            return self.__getDemoStats(gameid)
        assert player and gameid >= 0
        d = self.stats.get(player)
        if d is None:
            # create an entry for this player
            d = self.stats[player] = {}
        return d.get(gameid, (0,0))

    def updateStats(self, player, game, won, lost):
        if player is None:
            self.__updateDemoStats(game.id, won, lost)
            return
        assert player and game.id > 0
        w, l = self.getStats(player, 0)               # all games
        self.stats[player][0] = (w + won, l + lost)
        w, l = self.getStats(player, game.id)
        self.stats[player][game.id] = (w + won, l + lost)
        if not self.prev_games.has_key(player):
            self.prev_games[player] = []
        self.prev_games[player].append((game.id, game.getFullId(format=0),
                                        won, game.gstats.start_time,
                                        game.gstats.total_elapsed_time))

    #
    # demo statistics (deprecated interface)
    #

    def __resetDemoStats(self, gameid):
        # a dictionary of tuples (won/lost), indexed by gameid
        if gameid == 0:
            self.demo_stats = {}
        else:
            won, lost = self.__getDemoStats(gameid)
            self.__updateDemoStats(gameid, -won, -lost)

    def __getDemoStats(self, gameid):
        assert gameid >= 0
        return self.demo_stats.get(gameid, (0,0))

    def __updateDemoStats(self, gameid, won, lost):
        assert gameid > 0
        w, l = self.__getDemoStats(0)               # all games
        self.demo_stats[0] = (w + won, l + lost)
        w, l = self.__getDemoStats(gameid)
        self.demo_stats[gameid] = (w + won, l + lost)


# /***********************************************************************
# // Application
# // This is the glue between the toplevel window and a Game.
# // Also handles all global resources.
# ************************************************************************/

class Application:
    def __init__(self):
        self.starttimer = Timer("Application.__init__")
        self.opt = Options()
        self.stats = Statistics()
        self.splashscreen = 1
        self.debug = 0
        # visual compoments
        self.top = None                 # the root toplevel window
        self.top_cursor = None          # default cursor
        self.menubar = None
        self.toolbar = None
        self.canvas = None
        self.statusbar = None
        #
        self.game = None
        self.dataloader = None
        self.audio = None
        self.images = None
        self.jokers = []
        self.redeal_images = []
        self.progress_bg = None
        self.progress_images = []
        self.cm = CardsetManager()
        self.tiles = ()
        self.tabletile_index = 0
        self.cardset = None             # current cardset
        self.intro = Struct(
            progress = None,            # progress bar
        )
        # directory names
        home = os.path.normpath(gethomedir())
        config = os.path.normpath(getprefdir(PACKAGE, home))
        self.dn = Struct(
            home = home,
            config = config,
            plugins = os.path.join(config, "plugins"),
            savegames = os.path.join(config, "savegames"),
            maint = os.path.join(config, "maint"),          # debug
        )
        for k in self.dn.__dict__.keys():
            v = self.dn.__dict__[k]
            if os.name == "nt":
                v = os.path.normcase(v)
            v = os.path.normpath(v)
            self.dn.__dict__[k] = v
        # file names
        self.fn = Struct(
            opt   = os.path.join(self.dn.config, "options.dat"),
            stats = os.path.join(self.dn.config, "statistics.dat"),
            #$unbundle#lastgame = os.path.join(self.dn.config, "lastgame.pso"),
            lastgame = os.path.join(self.dn.config, "lastgamex.pso"), #bundle#
        )
        for k in self.fn.__dict__.keys():
            v = self.fn.__dict__[k]
            if os.name == "nt":
                v = os.path.normcase(v)
            v = os.path.normpath(v)
            self.fn.__dict__[k] = v
        # random generator
        self.gamerandom = LCRandom64()
        self.miscrandom = LCRandom64()
        # player
        player = string.strip(os.environ.get("USER",""))
        if not player:
            player = string.strip(os.environ.get("LOGNAME",""))
        if not player:
            player = "unknown"
        if len(player) > 30:
            player = player[:30]
        self.opt.player = player
        # misc
        self.nextgame = Struct(
            id = 0,                     # start which game
            random = None,              # use this random generator
            loadedgame = None,          # data for loaded game
            startdemo = 0,              # start demo ?
            cardset_index = 0,          # use which cardset
        )
        self.commandline = Struct(
            loadgame = None,            # load a game ?
        )
        self.demo_counter = 0


    # the PySol mainloop
    def mainloop(self):
        # try to load statistics
        try: self.loadStatistics()
        except: pass
        # startup information
        if self.getGameClass(self.stats.last_gameid):
            self.nextgame.id = self.stats.last_gameid
        if self.stats.last_player:
            self.opt.player = self.stats.last_player
        # check for commandline options
##        if 1 and self.debug and self.commandline.loadgame is None:
##            if os.path.isfile(self.fn.lastgame):
##                self.commandline.loadgame = self.fn.lastgame
        if self.commandline.loadgame:
            try:
                tmpgame = self.constructGame(2, seed=None)
                self.nextgame.loadedgame = tmpgame._loadGame(self.commandline.loadgame, self)
            except:
                pass
            tmpgame = None
        # create the menubar
        self.menubar = PysolMenubar(self, self.top)
        # create the statusbar (before the toolbar)
        self.statusbar = PysolStatusbar(self.top)
        self.statusbar.show(self.opt.statusbar)
        # create the toolbar
        dir = os.path.join(self.dataloader.dir, "toolbar", self.opt.toolbar_dir)
        self.toolbar = PysolToolbar(self.top, dir=dir)
        self.toolbar.setRelief(self.opt.toolbar_relief)
        self.toolbar.show(self.opt.toolbar)
        if self.intro.progress: self.intro.progress.update(step=1)
        # create the canvas
        self.canvas = MfxCanvas(self.top, bg=self.opt.tablecolor, highlightthickness=0)
        self.canvas.pack(fill="both", expand=1)
        tile = self.tiles[self.tabletile_index]
        self.canvas.setTile(tile[1])
        self.canvas.setTextColor(tile[3])
        try:
            # run
            while 1:
                id, random = self.nextgame.id, self.nextgame.random
                self.nextgame.id, self.nextgame.random = 0, None
                self.runGame(id, random)
                if self.nextgame.id <= 0:
                    f = os.path.join(self.fn.lastgame)
                    try: self.game._saveGame(f)
                    except: pass
                self.freeGame()
                self.dumpMem()                                      #bundle#
                if self.nextgame.id <= 0:
                    break
                # load new cardset
                if self.nextgame.cardset_index != self.cardset.index:
                    self.loadCardset()
        finally:
            # save statistics
            self.stats.last_gameid = id
            self.stats.last_player = self.opt.player
            try: self.saveStatistics()
            except: pass


    def runGame(self, id, random=None):
        self.top.connectApp(self)
        # create game instance
        g = self.getGameClass(id)
        if g is None:
            id = 2          # start Klondike as default game
            random = None
            g = self.getGameClass(id)
            if g is None:
                # start first available game
                id = self.getGamesIdSortedById()[0]
                g = self.getGameClass(id)
        assert g and type(g) == types.ClassType and id > 0
        self.game = self.constructGame(id, random=random)
        self.game.busy = 1
        # initialize game
        self.game.create(self)
        # delete intro progress bar
        if self.intro.progress:
            self.intro.progress.destroy()
            destruct(self.intro.progress)
        # connect with game
        self.menubar.connectGame(self.game)
        self.toolbar.connectGame(self.game, self.menubar)
        # start game
        if self.nextgame.loadedgame:
            self.game.restoreGame(self.nextgame.loadedgame)
            destruct(self.nextgame.loadedgame)
            self.nextgame.loadedgame = None
            autoplay = 0
        else:
            self.game.newGame(random=random, autoplay=0)
            autoplay = 1
        self.intro.progress = None
        if self.splashscreen and (0 or not self.debug):
            status = helpAbout(self, timeout=20000)
            if status == 2:                 # timeout - start a demo
                self.nextgame.startdemo = 1
        self.splashscreen = 0
        if self.nextgame.startdemo:
            after_idle(self.top, self.game.startDemo)
            self.nextgame.startdemo = 0
        elif autoplay:
            self.game.autoPlay()
            self.game.changed_index = self.game.moves.index
            self.game.stats.player_moves = 0
        # enter the Tk mainloop
        self.game.busy = 0
        self.top.mainloop()


    # free game
    def freeGame(self):
        # disconnect from game
        self.toolbar.connectGame(None, None)
        self.menubar.connectGame(None)
        # clean up the canvas
        unbind_destroy(self.canvas)
        self.canvas.deleteItems()
        self.canvas.update_idletasks()
        # destruct the game
        if self.game:
            # break circular dependencies
            for obj in self.game.cards:
                destruct(obj)
            for obj in self.game.allstacks:
                destruct(obj)
            destruct(self.game)
        self.game = None
        self.top.connectApp(None)


    # debug
    def dumpMem(self):
#%ifndef BUNDLE
        return
        ##print "freeGame:\n"; dumpmem(); print
        if 1 and cyclops: cyclops.clear()
        if 1 and cyclops: cyclops.register(self)
        if 1 and cyclops: cyclops.find_cycles()
        if 1 and cyclops: cyclops.show_stats()
        #if 1 and cyclops: cyclops.show_arcs()
        #if 1 and cyclops: cyclops.show_sccs()
        #if 1 and cyclops: cyclops.show_cycles()
        if 1 and cyclops: cyclops.clear()
#%endif
        pass


    def loadCardset(self):
        if self.nextgame.cardset_index == self.cardset.index:
            return
        cs = self.cm.get(self.nextgame.cardset_index)
        if not cs:
            return
        images = MfxImages(self.dataloader, cs)
        withdraw(self.top)
        self.top.busyUpdate()
        title = "Loading cardset " + cs.name + "..."
        color = self.opt.tablecolor
        if self.tabletile_index > 0:
            color = "#008200"
        pb = PysolProgressBar(self, self.top, title=title, color=color, bg=self.progress_bg, images=self.progress_images)
        try:
            images.load(progress=pb)
            destruct(self.images)
            self.cardset = cs
            self.images = images
            # update options
            self.opt.cardset_name = self.cardset.name
        except (Exception, TclError), ex:
            self.opt.cardset_index = self.cardset.index   # restore previous setting
            destruct(images)
            if 1 or self.debug: traceback.print_exc()               #bundle#
            d = MfxExceptionDialog(self.top, ex, title="Cardset load error",
                                   text="Error while loading cardset")
        self.menubar.updateBackgroundImages()
        self.intro.progress = pb


    #
    # options & statistics
    #

    def loadOptions(self):
        opt = unpickle(self.fn.opt)
        if opt:
            opt.toolbar_dir = None
            merge_dict(self.opt.__dict__, opt.__dict__)

    def saveOptions(self):
        opt = self.opt.copy()
        opt.toolbar_dir = None
        pickle(opt, self.fn.opt)

    def loadStatistics(self):
        stats = unpickle(self.fn.stats)
        if stats:
            merge_dict(self.stats.__dict__, stats.__dict__)

    def saveStatistics(self):
        pickle(self.stats, self.fn.stats, binmode=1)


    #
    # access games database
    #

    def constructGame(self, id, random=None):
        gameclass = self.getGameClass(id)
        if not gameclass:
            raise Exception, "Unknown game (id %d)" % id
        return gameclass(GAME_DB[id], random=random)

    def getGamesIdSortedById(self):
        return GAME_DB.getGamesIdSortedById()

    def getGamesIdSortedByName(self):
        return GAME_DB.getGamesIdSortedByName()

    def getGameInfo(self, id):
        return GAME_DB.get(id)

    def getGameClass(self, id):
        g = GAME_DB.get(id)
        if g is None: return None
        return g.gameclass

    def getGameTitleName(self, id):
        g = GAME_DB.get(id)
        if g is None: return None
        return g.name

    def getGameMenuitemName(self, id):
        g = GAME_DB.get(id)
        if g is None: return None
        return g.name

    def getGameRulesFilename(self, id):
        g = GAME_DB.get(id)
        if g is None: return None
        if g.rules_filename is not None:
            return g.rules_filename
        n = g.name
        n = re.sub(r'[\[\(].*$', '', n)
        n = re.sub(r'\W', '', n)
        n = string.lower(n) + ".html"
        g.rules_filename = n    # save the filename for next use
        return n

    def getGameSaveName(self, id):
        n = self.getGameTitleName(id)
        if not n: return None
        m = re.search(r'^(.*)([\[\(](\w+).*[\]\)])\s*$', n)
        if m:
            n = m.group(1) + "_" + string.lower(m.group(2))
        return re.sub(r'[^\w-]', '', n)

    def getRandomGameId(self):
        return self.miscrandom.choice(GAME_DB.getGamesIdSortedById())


    #
    # plugins
    #

    def loadPlugins(self, dir):
        if not dir or not os.path.isdir(dir):
            return
        names = os.listdir(dir)
        names.sort()
        for name in names:
            m = re.search(r'^(.+)\.py$', name)
            n = os.path.join(dir, name)
            if m and os.path.isfile(n):
                p = sys.path[:]
                try:
                    loadGame(m.group(1), n)
                except Exception, ex:
                    print "Error loading plugin " + n + ": " + str(ex)
                    sys.stdout.flush()
                    if 1 or self.debug: traceback.print_exc()       #bundle#
                sys.path = p


    #
    # cardsets
    #

    # read & parse a cardset config.txt file
    def _readCardsetConfig(self, dir, filename):
        f = None
        try:
            f = open(filename, "r")
            line = []
            for i in range(6):
                line.append(string.strip(f.readline()))
        finally:
            if f: f.close()
        cs = Cardset()
        cs.dir = dir
        # line[0]: magic identifier, possible version information
        if string.index(line[0], "PySol") != 0: return None
        fields = filter(None, re.split(r';', line[0]))
        fields = map(string.strip, fields)
        if len(fields) >= 2:
            m = re.search(r'^(\d+)$', fields[1])
            if m: cs.version = int(m.group(1))
        if cs.version >= 3:
            if len(fields) < 5:
                return None
            cs.ext = fields[2]
            m = re.search(r'^(\d+)$', fields[3])
            if not m: return None
            cs.type = int(m.group(1))
            m = re.search(r'^(\d+)$', fields[4])
            if not m: return None
            cs.ncards = int(m.group(1))
        if len(cs.ext) < 2 or cs.ext[0] != ".":
            return None
        # line[1]: identifier used in menubar
        if not line[1]: return None
        cs.ident = line[1]
        m = re.search(r'^(.*;)?([^;]+)$', cs.ident)
        if not m: return None
        cs.name = string.strip(m.group(2))
        # line[2]: CARDW, CARDH, CARDD
        m = re.search(r'^(\d+)\s+(\d+)\s+(\d+)', line[2])
        if not m: return None
        cs.CARDW, cs.CARDH, cs.CARDD = int(m.group(1)), int(m.group(2)), int(m.group(3))
        if cs.CARDD > self.top.winfo_screendepth(): return None
        # line[3]: CARD_UP_YOFFSET, CARD_DOWN_YOFFSET, SHADOW_XOFFSET, SHADOW_YOFFSET
        m = re.search(r'^(\d+)\s+(\d+)\s+(\d+)\s+(\d+)', line[3])
        if not m: return None
        cs.CARD_UP_YOFFSET = int(m.group(1))
        cs.CARD_DOWN_YOFFSET = int(m.group(2))      # not used
        cs.SHADOW_XOFFSET = int(m.group(3))
        cs.SHADOW_YOFFSET = int(m.group(4))
        # line[4]: default background
        back = line[4]
        if not back: return None
        # line[5]: all available backgrounds
        cs.backnames = re.split(r';', line[5])
        cs.backnames = map(string.strip, cs.backnames)
        cs.backnames = filter(None, cs.backnames)
        if back in cs.backnames:
            cs.back_index = cs.backnames.index(back)
        else:
            cs.backnames.insert(0, back)
            cs.back_index = 0
        return cs

    def initCardsets(self):
        # find all available cardsets
        sets = []
        dirs = string.split(os.environ.get("PYSOL_CARDSETS",""), os.pathsep)
        dirs.append(self.dataloader.dir)
        dirs.append(os.path.join(self.dataloader.dir, "cardsets"))
        if self.dn.maint:
            dirs.append(self.dn.maint)
            dirs.append(os.path.join(self.dn.maint, "cardsets"))
        dirs.append(self.dn.config)
        dirs.append(os.path.join(self.dn.config, "cardsets"))
        dirs.append(os.path.join(self.dn.config, "data"))
        for dir in dirs:
            dir = string.strip(dir)
            try:
                names = []
                if dir and os.path.isdir(dir):
                    names = os.listdir(dir)
                    names.sort()
                for name in names:
                    m = re.search(r'^cardset-', name, re.I)
                    if not m: continue
                    d = os.path.join(dir, name)
                    if not os.path.isdir(d): continue
                    f1 = os.path.join(d, "config.txt")
                    if os.path.isfile(f1):
                        try:
                            cs = self._readCardsetConfig(d, f1)
                            if cs:
                                ##print cs.__dict__
                                back = cs.backnames[cs.back_index]
                                f1 = os.path.join(d, back)
                                f2 = os.path.join(d, "shade" + cs.ext)
                                f3 = os.path.join(d, "02c" + cs.ext)
                                if os.path.isfile(f1) and os.path.isfile(f2) and os.path.isfile(f3):
                                    sets.append((cs.ident, cs))
                        except:
                            if self.debug: traceback.print_exc()    #bundle#
                            pass
            except EnvError, ex:
                pass
        # register cardsets
        sets.sort()
        for s in sets:
            cs = s[1]
            if not self.cm.getByName(cs.name):
                self.cm.register(cs)


    #
    # tiles
    #

    def initTiles(self):
        # find all available tiles
        ext_re       = re.compile(r'\.((gif)|(ppm))$', re.I)
        textcolor_re = re.compile(r'^(.+)-([0-9A-Fa-f]{6})$')
        tiles = []
        t = {}
        dirs = string.split(os.environ.get("PYSOL_TILES",""), os.pathsep)
        for dir in (self.dataloader.dir, self.dn.maint, self.dn.config):
            try:
                if dir and os.path.isdir(dir):
                    dir = os.path.normpath(os.path.join(dir, "tiles"))
                    d = glob.glob(dir + "*")
                    d.sort()
                    dirs = dirs + filter(os.path.isdir, d)
            except EnvError, ex:
                pass
        ##print dirs
        for dir in dirs:
            dir = string.strip(dir)
            try:
                names = []
                if dir and os.path.isdir(dir):
                    names = os.listdir(dir)
                    names.sort()
                for name in names:
                    if not name or not ext_re.search(name):
                        continue
                    f = os.path.join(dir, name)
                    if not os.path.isfile(f):
                        continue
                    n = string.strip(name)
                    n = ext_re.sub("", n)
                    textcolor = "#000000"
                    m = textcolor_re.search(n)
                    if m:
                        n = m.group(1)
                        textcolor = "#" + string.lower(m.group(2))
                    key = string.lower(name)
                    if not t.has_key(key):
                        t[key] = 1
                        n = re.sub("_", " ", n)
                        tiles.append((n, f, name, textcolor))
                        ##print (n, f, name, textcolor)
            except EnvError, ex:
                pass
        tiles.sort()
        self.tiles = self.tiles + tuple(tiles)
        ##print self.tiles


# /***********************************************************************
# // PySol main entry
# ************************************************************************/

def pysol_main(args):
    # create the application
    app = Application()

    # try to create the config directory
    try: os.mkdir(app.dn.config, 0750)
    except EnvError: pass
    try: os.mkdir(os.path.join(app.dn.savegames), 0750)
    except EnvError: pass
    try: os.mkdir(os.path.join(app.dn.config, "tiles"), 0750)
    except EnvError: pass

#%ifndef BUNDLE
    # import Cyclops
    if 1 and app.debug:
        global cyclops
        try:
            import Cyclops
            cyclops = Cyclops.CycleFinder()
        except ImportError: pass
#%endif

    # init DataLoader
    f = os.path.join("images", "redeal.gif")
    app.dataloader = DataLoader(args[0], f)

    # try to load plugins
    if not "noplugins" in args[1:]:
        for dir in (os.path.join(app.dataloader.dir, "games"),
                    os.path.join(app.dataloader.dir, "plugins"),
                    app.dn.plugins):
            try:
                app.loadPlugins(dir)
            except:
                if app.debug: traceback.print_exc()                 #bundle#
                pass

    # init commandline options (undocumented)
    prog = sys.executable
    if prog and os.path.isfile(prog):
        prog = os.path.abspath(prog)
    else:
        prog = "python"
    argv0 = os.path.normpath(args[0])
    wm_command = ""
    if os.path.isfile(argv0):
        wm_command = prog + " " + os.path.abspath(argv0)
    for a in args[1:]:
        if os.path.isfile(a):
            app.commandline.loadgame = a
        elif a == "nodebug":
            app.debug = 0
        elif a == "debug":
            app.debug = app.debug + 1
        # ignore other commandline options

    # print some debug info
    if 1 and not app.debug and app.opt.player == "mfx":             #bundle#
        app.debug = 1                                               #bundle#
    if not app.debug:
        app.dn.maint = None
    if 1 and app.debug >= 2:
        print "prog:", prog
        print "argv0:", argv0
        print "wm_command:", wm_command
        print "sys.executable:", sys.executable
        print "sys.version:", sys.version
        print "sys.argv:", sys.argv
        print "sys.path:", sys.path
        print "dataloader:", app.dataloader.__dict__
#%ifndef BUNDLE
    if 1 and app.debug >= 3:
        print len(GAME_DB.getAll())
        for id in app.getGamesIdSortedByName():
            print '    "%s",' % app.getGameMenuitemName(id)
    if 1 and app.debug:
        for id in app.getGamesIdSortedByName():
            f = os.path.join(app.dataloader.dir, "html", "rules", app.getGameRulesFilename(id))
            if not os.path.isfile(f):
                print "Warning:", f, "not found"
#%endif

    # try to load options
    try: app.loadOptions()
    except: pass

    # init audio
    app.audio = PysolAudio(app)

    # init toolkit
    top = MfxRoot(className=PACKAGE)
    top.wm_group(top)
    top.wm_title(PACKAGE + " " + VERSION)
    top.wm_iconname(PACKAGE + " " + VERSION)
    top.wm_minsize(200, 200)
    top.wm_protocol("WM_DELETE_WINDOW", top.wmDeleteWindow)
    if wm_command:
        top.wm_command(wm_command)
    sw, sh, sd = top.winfo_screenwidth(), top.winfo_screenheight(), top.winfo_screendepth()
    if sw < 640 or sh < 480 or sd < 8:
        withdraw(top)
        top.update()
        d = MfxDialog(top, title="PySol init error",
                      text="PySol requires at least\n640x480 in 256 colors.",
                      bitmap="error", strings=("Quit",))
        return 1
    if 1:
        # set expected window size to assist the layout of the window manager
        top.config(width=min(800,sw-64), height=min(600,sh-64), bg=app.opt.tablecolor)
    else:
        top.config(bg=app.opt.tablecolor)
    try: top.wm_iconbitmap("@" + app.dataloader.findFile("pysol.xbm"))
    except: pass
    app.top = top
    app.top_cursor = top.cget("cursor")

    # check games
    if len(app.getGamesIdSortedById()) < 1:
        withdraw(top)
        top.update()
        d = MfxDialog(top, title="PySol installation error",
                      text="No games were found !!!\n\nMain data directory is:\n" +
                           app.dataloader.dir + "\n\nPlease check your PySol installation.",
                      bitmap="error", strings=("Quit",))
        return 1

    # init cardsets
    app.initCardsets()
    if app.cm.len() == 0:
        withdraw(top)
        top.update()
        d = MfxDialog(top, title="PySol installation error",
                      text="No cardsets were found !!!\n\nMain data directory is:\n" +
                           app.dataloader.dir + "\n\nPlease check your PySol installation.",
                      bitmap="error", strings=("Quit",))
        ##raise Exception, "no cardsets found !"
        return 1

    # init tiles
    app.tiles = ( ("None", None, "", "#000000"), )
    app.initTiles()
    if app.opt.tabletile_name and top.winfo_screendepth() > 8:
        for i in range(len(app.tiles)):
            if app.opt.tabletile_name == app.tiles[i][2]:
                app.tabletile_index = i
                break

    # init image loader
    app.cardset = app.cm.getByName(app.opt.cardset_name)
    if app.cardset:
        # update default background of the cardset to match options
        b = app.opt.cardback_name
        if b and b in app.cardset.backnames:
            app.cardset.back_index = app.cardset.backnames.index(b)
    else:
        app.cardset = app.cm.get(0)
    app.images = MfxImages(app.dataloader, app.cardset)
    # update options
    app.opt.cardset_name = app.cardset.name
    app.nextgame.cardset_index = app.cardset.index

    # set global color scheme
    if os.name == "posix":              # Unix/X11
        ##color = "grey"
        color = "#d9d9d9"               # Tk 8.0p2 default
        ##top.tk_setPalette(color)
        top.tk_setPalette("background", color, "activeBackground", color)
    if os.name == "mac":
        color, priority = "#d9d9d9", "60"
        classes = (
            "Button", "Canvas", "Checkbutton", "Entry",
            "Frame", "Label", "Listbox", "Menubutton", ### "Menu",
            "Message", "Radiobutton", "Scale", "Scrollbar", "Text",
        )
        for c in classes:
            top.option_add("*" + c + "*background", color, priority)
            top.option_add("*" + c + "*activeBackground", color, priority)

    # create a progress bar
    app.progress_bg = "#c0c0c0"
    ##app.progress_bg = None
    dir = os.path.join("images", "jokers")
    app.jokers.append(app.images.loadImage(app.dataloader.findFile("joker07_40_774.gif", dir)))
    app.jokers.append(app.images.loadImage(app.dataloader.findFile("joker08_40_774.gif", dir)))
    app.jokers.append(app.dataloader.findFile("joker07_50_774.gif", dir))
    app.jokers.append(app.dataloader.findFile("joker08_50_774.gif", dir))
    app.jokers.append(app.dataloader.findFile("joker11_100_774.gif", dir))
    app.jokers.append(app.dataloader.findFile("joker10_100.gif", dir))
    if not app.progress_images:
        app.progress_images = (app.jokers[0], app.jokers[1])
    withdraw(top)
    title = "Welcome to " + PACKAGE
    color = app.opt.tablecolor
    if app.tabletile_index > 0:
        color = "#008200"
    app.intro.progress = PysolProgressBar(app, top, title=title, color=color, bg=app.progress_bg, images=app.progress_images)

    # prepare other images
    dir = os.path.join("images")
    ##for f in ("noredeal", "redeal"):
    for f in ("stopsign", "redeal"):
        app.redeal_images.append(app.dataloader.findFile(f+".gif", dir))

    # load all images (loading of transparent GIF images is *very* slow under Tk)
    for i in range(len(app.jokers)):
        if type(app.jokers[i]) == types.StringType:
            app.jokers[i] = app.images.loadImage(app.jokers[i])
            app.intro.progress.update(step=1)
    app.jokers = tuple(app.jokers)
    for i in range(len(app.redeal_images)):
        if type(app.redeal_images[i]) == types.StringType:
            app.redeal_images[i] = app.images.loadImage(app.redeal_images[i])
            app.intro.progress.update(step=1)
    app.redeal_images = tuple(app.redeal_images)
    app.images.load(progress=app.intro.progress)

    # let's go
#%ifdef BUNDLE
    #$unbundle#app.mainloop()
#%else
    try:
        app.mainloop()
    except AssertionError:
        if app.game: print app.game.game_info.__dict__
        raise
#%endif

    # clean up
    app.audio.destroy()         # shut down audio
    destruct(app.audio)
    if app.canvas is not None:
        app.canvas.destroy()
        destruct(app.canvas)
    app.toolbar.destroy()
    destruct(app.toolbar)
    destruct(app.menubar)
    destruct(cyclops)                                               #bundle#
    cyclops = None                                                  #bundle#
    destruct(app)
    app = None
    top.destroy()
    destruct(top)
    top = None
    return 0


# /***********************************************************************
# // main
# ************************************************************************/

def main(args=None):
    if args is None: args = ["./pysol"]                             #bundle#

    # check versions
    if sys.version < "1.5.2":
        print "%s needs Python 1.5.2 or better (you have %s)" % (PACKAGE, sys.version)
        return 1
    if tkname == "tk":
        import Tkinter                                              #bundle#
        if Tkinter.TkVersion < 8.0:
            ##import _tkinter; print _tkinter.TK_VERSION
            print "%s needs Tcl/Tk 8.0 or better (you have %s)" % (PACKAGE, str(Tkinter.TkVersion))
            return 1
        # check that Tkinter bindings are also at version 1.5.2
        if not hasattr(Tkinter.Wm, "wm_aspect"):
            print "%s: please update the Python-Tk bindings (aka Tkinter) to version 1.5.2 or better" % (PACKAGE,)
            return 1
    # check Python
    assert -1 % 13 == 12

    # run it
    r = pysol_main(args)
    ##print "FINAL\n"; dumpmem()
    return r


if __name__ == "__main__":
    sys.exit(main(sys.argv))

