/* This program takes a .psf font and generates a new one for use with
 * the Linux Snipes DOUBLEWIDE_FONT_HACK mode.  It also generates a
 * header file which is used by snipes to compile in support for the
 * DOUBLEWIDE_FONT_HACK mode.  This file is copyright 1999 by
 * Jeremy Boulton.  You may do anything you want with this file
 * except claim you wrote it.
 */

/*
 * The format of a .PSF file is as follows:
 *  
 * Byte    Size of field           Contents
 * 0       Word                    File ID, always 0436h
 * 2       Byte                    File mode
 * 
 * === FILE MODE 0:  256 character byte-aligned monofont (as used by EGA/VGA)
 * 
 * 3       Byte                    Height of characters in scan lines
 * 4...    256x[byte 3] bytes      Font data; each character contains one byte
 *                                     per scan line, characters continuous
 * 
 * === FILE MODE 1:  512 character byte-aligned monofont
 * 
 * 3       Byte                    Height of characters in scan lines
 * 4...    512x[byte 3] bytes      Font data; each character contains one byte
 * 
 *                                     per scan line, characters continuous
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>

void bit_double( char inval, char *out_high, char *out_low )
{
  int i;

  *out_low = 0;
  *out_high = 0;

  for( i=0; i<4; i++ )
    if( inval & (1<<i) )
      *out_low |= (3<<(2*i));

  for( ; i<8; i++ )
    if( inval & (1<<i) )
      *out_high |= (3<<(2*i-8));
}



void double_char( char *in_buf, char *out_buf, int char_height, int in_idx,
		  int out_idx )
{
  int out_idx2;
  int off;
  
  /* The 32 character range starting at 192 seems to get its 8th bit
   * duplicated into bit 9 by the video hardware.  This allows the
   * line drawing characters to appear without inter-character gaps
   * of one pixel.  We try to take some advantage of that to make
   * this hack look a bit better.
   */
  
  if( out_idx >= 128 )
    out_idx2 = out_idx+64;
  else
    out_idx2 = out_idx+1;

  for( off=0; off<char_height; off++ ) {
    bit_double(   in_buf[in_idx*char_height+off],
		&out_buf[out_idx*char_height+off],
		&out_buf[(out_idx2)*char_height+off] );
  }
}
  

int main( int argc, char *argv[] )
{
  int char_height;
  int num_chars;
  int i;
  int off;
  int fdi, fdo;
  int outfile_len;
  char header[4];
  char *in_buf, *out_buf, *in_fontstart, *out_fontstart;
  char zero_char=0;
  struct stat stat_buf;
  FILE *fptr;
  
  unsigned char font_map_table[256];

  /* Characters use by snipes.  Alphanumeric characters are handled
   * separately.
   */
  int in_idx[32] = {
	0xDA, 0xBF, 0xC0, 0xD9, 0x01, 0x02, 0x18, 0x19, 0x1D,
	'/' , '\\', 0x11, 0x10, 'O' , 0x93, 0x09, 0x2A, 0xB0,
	0xB2, 0xB3,  ' ', 0xBA, 0xCD, 0xC9, 0xC8, 0xBB, 0xBC,
	0xCC, 0xB9, 0xCA, 0xCB, 0xCE
  };
  
  for( i=0; i<256; i++ ) font_map_table[i]=0;

  if( argc != 2 ) {
    fprintf( stderr, "Usage: %s filename\n", argv[0] );
    exit(1);
  }
  
  if( (fdi = open( argv[1], O_RDONLY )) == -1 ) {
    fprintf( stderr, "Could not open file \"%s\" for read.\n", argv[1] );
    exit(1);
  }

  fstat( fdi, &stat_buf );
  
  if( stat_buf.st_size == 0 ) {
    close( fdi );
    fprintf( stderr, "Input file is too short.\n" );
    exit(1);
  }

  if( (in_buf = mmap( 0, stat_buf.st_size, PROT_READ, MAP_SHARED, fdi, 0 )) == MAP_FAILED ) {
    close( fdi );
    perror( "Could not mmap input file" );
    exit(1);
  }
  
  /* Check for valid input file. */

  if( stat_buf.st_size == 4096 && (in_buf[1] != 0x04 || in_buf[0] != 0x36) ) {
    fprintf( stderr, "Not a PSF file.  Assuming 256 16-line characters.\n" );
    
    char_height = 16;
    num_chars = 256;
    outfile_len = 4100;
    in_fontstart = in_buf;

    header[0] = 0x36;		/* magic number */
    header[1] = 0x04;		/* magic number */
    header[2] = 0x00;		/* mode */
    header[3] = char_height;	/* char height */
  }
  else {
    if( in_buf[1] != 0x04 || in_buf[0] != 0x36 ) {
      munmap( in_buf, stat_buf.st_size );
      close( fdi );
      fprintf( stderr, "Invalid magic number in input file.\n" );
      exit(1);
    }

    if( in_buf[2] == 0 )
      fprintf( stderr, "FILE MODE 0:  256 character byte-aligned monofont (as used by EGA/VGA)\n" );
    else if( in_buf[2] == 1 )
      fprintf( stderr, "FILE MODE 1:  512 character byte-aligned monofont\n" );
    else {
      munmap( in_buf, stat_buf.st_size );
      close( fdi );
      fprintf( stderr, "Invalid mode number in input file.\n" );
      exit(1);
    }
    
    outfile_len = stat_buf.st_size;
    char_height = (int)in_buf[3];
    num_chars = (stat_buf.st_size-4)/char_height;
    in_fontstart = &in_buf[4];
    memcpy( header, in_buf, 4 );
  }
  
  fprintf( stderr, "Original font contains %d characters.\n", num_chars );

  /* Output file */

  if( (fdo = open( "newfont", O_RDWR|O_CREAT|O_TRUNC, S_IRUSR|S_IWUSR )) == -1 ) {
    munmap( in_buf, stat_buf.st_size );
    close( fdi );
    fprintf( stderr, "Could not open file \"%s\" for write.\n", "newfont" );
    exit(1);
  }
  
  lseek( fdo, outfile_len-1, SEEK_SET );
  write( fdo, &zero_char, 1 );
  
  if( (out_buf = mmap( 0, outfile_len, PROT_READ|PROT_WRITE, MAP_SHARED, fdo, 0 )) == MAP_FAILED ) {
    close( fdo );
    munmap( in_buf, stat_buf.st_size );
    close( fdi );
    perror( "Could not mmap output file" );
    exit(1);
  }
  
  /* Copy header */
  memcpy( out_buf, header, 4 );
  out_fontstart = &out_buf[4];

  for( i=0; i<32; i++ ) {
    double_char( in_fontstart, out_fontstart, char_height, in_idx[i], 128+i );
    font_map_table[in_idx[i]] = 128+i;
  }
  for( i=0; i<26; i++ ) {
    double_char( in_fontstart, out_fontstart, char_height, 'A'+i, 34+2*i );
    font_map_table['A'+i] = 34+2*i;
    font_map_table['a'+i] = 34+2*i;
  }
  for( i=0; i<10; i++ ) {
    double_char( in_fontstart, out_fontstart, char_height, '0'+i, 86+2*i );
    font_map_table['0'+i] = 86+2*i;
  }

  munmap( out_buf, outfile_len );
  close( fdo );
  munmap( in_buf, stat_buf.st_size );
  close( fdi );
  
  if( (fptr = fopen( "newfont.h", "w" )) == NULL ) {
    fprintf( stderr, "Could not open file \"%s\" for write.\n", "newfont.map" );
    exit(1);
  }

  fprintf( fptr, "#define CLR\t32\n\n" );
  fprintf( fptr, "unsigned char first_char_table[256] = {" );
  for( i=0; i<256; i++ ) {
    if( i != 0 )
      fprintf( fptr, "," );

    if( i%8 == 0 ) {
      if( i%16 == 0 )
	fprintf( fptr, "\n/* %X */", i/16 );
      else
	fprintf( fptr, "\n       " );
    }
    if( font_map_table[i] )
      fprintf( fptr, "\t%4d", font_map_table[i] );
    else {
      if( i<0x20 )
	fprintf( fptr, "\t0x%02X", i );
      else
	fprintf( fptr, "\t CLR" );
    }
  }
  fprintf( fptr, "\n};\n" );
  
  fclose( fptr );

  return 0;
}
