#include <string.h>

#include "aiinterface.h"
#include "../coordinate.h"
#include "../game.h"

#define _THREAT_MAX	(BOARD_XSIZE*BOARD_YSIZE*12)

extern Coordinate direction[8];
extern Square board[BOARD_XSIZE][BOARD_YSIZE];

extern char _player_char[3];

typedef struct {
	ThinkAccelData *data;
	int e_turn;
} _WinFindData;

static int _threat_value[_THREAT_MAX];
static int _think_map_inc[5];

void _threat_value_init(void)
{
	int i;

	for(i=0; i<_THREAT_MAX; i++)
		_threat_value[i] = i*10;
}

void _think_map_inc_init(void)
{
	int i;

	for(i=0; i<5; i++)
		_think_map_inc[i] = i*i;
}

void ai_think_accel_init(void)
{
	_threat_value_init();
	_think_map_inc_init();
}

static int _threats_win_find_per_square(Coordinate *c, _WinFindData *data)
{
	if(ARRAY_COOR(board, *c) != NONE)
		return 0;
	if(ARRAY_COOR(data->data->threats[data->e_turn].threat, *c).count>1)
		return 1;
	return 0;
}

int ai_think_accel_threats_win_find(ThinkAccelData *think_accel_data, int e_turn, int *best, Coordinate *best_move, int level)
{
	_WinFindData data;

	if(!think_accel_data->threats[e_turn].threat_count[1])
		return 0;
	*best = WIN_VALUE;
	if(level)
		return 1;

	data.data = think_accel_data;
	data.e_turn = e_turn;
	coordinate_foreach(best_move, 0, 0, BOARD_XSIZE-1, BOARD_YSIZE-1, (CoordinateFunc)_threats_win_find_per_square, &data);

	return 1;
}

typedef struct {
	ThinkAccelData *data;
	int e_turn; /* e_turn of opponent */
	int *best;
	int turn, level, energy;
	Coordinate *best_move, *last_move;
} _LoseFindData;

int _threats_lose_find_per_square(Coordinate *c, _LoseFindData *data)
{
	Threats *threats = &data->data->threats[data->e_turn^1];
	Threat *t = &ARRAY_COOR(threats->threat, *c);

	if (ARRAY_COOR(board, *c) != NONE)
		return 0;
	if (ARRAY_COOR(data->data->threats[data->e_turn].threat, *c).count)
		return ai_think_move_test(c, data->data, data->best_move, data->best, data->last_move, data->e_turn, data->turn, data->energy-1, data->level);
	if (t->count > 2) {
		ai_think_move_test(c, data->data, data->best_move, data->best, data->last_move, data->e_turn, data->turn, data->energy-1, data->level);
		return 1;
	}
	if (t->count >= threats->threat_count[1])
		return ai_think_move_test(c, data->data, data->best_move, data->best, data->last_move, data->e_turn, data->turn, data->energy-1, data->level);
	return 0;	
}

int ai_think_accel_threats_lose_find(ThinkAccelData *think_accel_data, int *best, Coordinate *best_move, Coordinate *last_move, int e_turn, int turn, int energy, int level)
{
	Coordinate c;
	_LoseFindData data;

	if(!think_accel_data->threats[e_turn^1].threat_count[1])
		return 0;
	data.data = think_accel_data;
	data.e_turn = e_turn;
	data.best = best;
	data.turn = turn;
	data.level = level;
	data.energy = energy;
	data.last_move = last_move;
	data.best_move = best_move;
	coordinate_foreach(&c, 0, 0, BOARD_XSIZE-1, BOARD_YSIZE-1, (CoordinateFunc)_threats_lose_find_per_square, &data);
	return *(data.best) > (-WIN_VALUE-RANGE_CONSTANT);
}

/* Finds the threat described. */
static inline int _threat_find(Threat *threat, Coordinate *block)
{
	int i;

	for(i = 0; i < threat->count; i++)
		if(!coordinate_compare(&threat->block[i], block))
			return i;

	return -1;
}

static inline void _threat_add(Threat *t, Coordinate *threat_space, int e_turn)
{
	if(_threat_find(t, &threat_space[e_turn^1]) == -1) {
		coordinate_copy(&(t->block[t->count]), &threat_space[e_turn^1]);
		++t->count;
	}
}

static inline void _threat_del(Threat *t, Coordinate *threat_space, int e_turn)
{
	int which, tmp;

	if((which = _threat_find(t, &threat_space[e_turn^1])) != -1) {
		t->count--;
		if((tmp = t->count-which) > 0)
			memmove(&t->block[which], &t->block[which+1], tmp*sizeof(Coordinate));
	}
}

/* Finds the 2 spaces in a Five that can form an immediate threat.
 */
static void _spaces_find_in_five(Coordinate c, Coordinate *spaces, int dir)
{
	int i, count = 0;

	for(i=0; i<5; i++) {
		if(ARRAY_COOR(board, c) == NONE)
			coordinate_copy(&spaces[count++], &c);
		if(i<4 && coordinate_add_with_clip(&c, &direction[dir]))
			break;
	}
}

static inline void _threat_modify(ThinkAccelData *data, Coordinate *c, int dir, int e_turn, int action)
{
	Threat *t;
	Coordinate threat_space[2];
	int i;

 	_spaces_find_in_five(*c, threat_space, dir);
	for(i=0; i<2; i++) {
		t = &ARRAY_COOR(data->threats[e_turn].threat, threat_space[i]);
		if(action) {
			_threat_add(t, threat_space, i);
			data->threats[e_turn].threat_count[0]++;
			if(t->count == 2)
				data->threats[e_turn].threat_count[1]++;
			else if(t->count == 3)
				data->threats[e_turn].threat_count[2]++;
		} else {
			data->threats[e_turn].threat_count[0]--;
			if(t->count == 2)
				data->threats[e_turn].threat_count[1]--;
			else if(t->count == 3)
				data->threats[e_turn].threat_count[2]--;
			_threat_del(t, threat_space, i);
		}
	}
}

static void _threat_check_modify(ThinkAccelData *data, Coordinate *c, int dir, Five *five, int action)
{
	int i;

	switch(five->amount[0]) {
	case 0:
	case 1:
		if(!(five->amount[1] && five->amount[2]))
			data->state[five->amount[0]] += action ? 1 : -1;
		break;
	case 2:
		for(i=0; i<2; i++)
			if(five->amount[i+1] == 3)
				_threat_modify(data, c, dir, i, action);
	}
}

static inline int _five_valid(Coordinate c, int dir)
{
	int i;

	for(i=0; i<4; i++)
		if(coordinate_add_with_clip(&c, &direction[dir]))
			break;
	return i == 4;
}

static int _fives_init_per_square(Coordinate *c, Fives *fives)
{
	int dir;

	for(dir=0; dir<4; dir++)
		ARRAY_COOR(fives->five[dir], *c).amount[0] = _five_valid(*c, dir) ? 5 : -1;
	return 0;
}

static inline void _fives_init(Fives *fives)
{
	Coordinate c;

	coordinate_foreach(&c, 0, 0, BOARD_XSIZE-1, BOARD_YSIZE-1, (CoordinateFunc)_fives_init_per_square, fives);
}

typedef struct {
	ThinkAccelData *think_accel_data;
	Square board_copy[BOARD_XSIZE][BOARD_YSIZE];
} _CreateData;

static int _create_per_square(Coordinate *coor, _CreateData *data)
{
	Square tmp;

	if((tmp = ARRAY_COOR(data->board_copy, *coor)) != NONE)
		ai_think_accel_update(data->think_accel_data, coor, tmp-1, 1, NONE, tmp);

	return 0;
}

void ai_think_accel_create(ThinkAccelData *think_accel_data)
{
	Coordinate c;
	_CreateData data = {think_accel_data};

	memset(think_accel_data, 0, sizeof(*think_accel_data));
	_fives_init(&think_accel_data->fives);
	memcpy(data.board_copy, board, sizeof(board));
	memset(board, 0, sizeof(board));
	coordinate_foreach(&c, 0, 0, BOARD_XSIZE-1, BOARD_YSIZE-1, (CoordinateFunc)_create_per_square, &data);
}

static inline int _update_directional_five(ThinkAccelData *data, Coordinate *c, Coordinate *move, int e_turn, int dir, int inc, Square old, Square vmove)
{
	Five *tmp;
	int tm_inc = 0;

	tmp = &(ARRAY_COOR(data->fives.five[dir], *c));
	ARRAY_COOR(board, *move) = old;
	_threat_check_modify(data, c, dir, tmp, 0);
	if(!tmp->amount[1])
		tm_inc = -_think_map_inc[tmp->amount[2]];
	else if(!tmp->amount[2])
		tm_inc = -_think_map_inc[tmp->amount[1]];
	tmp->amount[e_turn+1] += inc;
	tmp->amount[0] -= inc;
	if(!tmp->amount[1])
		tm_inc += _think_map_inc[tmp->amount[2]];
	else if(!tmp->amount[2])
		tm_inc += _think_map_inc[tmp->amount[1]];
	ARRAY_COOR(board, *move) = vmove;
	_threat_check_modify(data, c, dir, tmp, 1);

	return tm_inc;
}

static inline void _update_directional(ThinkAccelData *data, Coordinate *move, int e_turn, int inc, int dir, Square old, Square vmove)
{
	int count, ocount, tm_inc[5], inc_sum;
	Coordinate c;

	coordinate_copy(&c, move);
	for(count = 0; count<5; count++) {
		tm_inc[count] = _update_directional_five(data, &c, move, e_turn, dir, inc, old, vmove);
		coordinate_add(&c, &direction[dir^4]);
		if(count < 4 && coordinate_clip(&c))
			break;
	}
	inc_sum = 0;
	for(ocount = count-1; ocount >= 0; ocount--) {
		coordinate_add(&c, &direction[dir]);
		ARRAY_COOR(data->think_map.val, c) += (inc_sum += tm_inc[ocount]);
	}
	for(ocount = 4; ocount >= count; ocount--)
		if(coordinate_add_with_clip(&c, &direction[dir]))
			return;
		else
			ARRAY_COOR(data->think_map.val, c) += inc_sum;
	for(; ocount >= 0; ocount--)
		if(coordinate_add_with_clip(&c, &direction[dir]))
			return;
		else
			ARRAY_COOR(data->think_map.val, c) += (inc_sum -= tm_inc[ocount]);
}

int ai_think_accel_win_test(ThinkAccelData *data)
{
	return data->state[0];
}

void ai_think_accel_update(ThinkAccelData *data, Coordinate *move, int e_turn, int inc, Square old, Square vmove)
{
	int dir;

	for(dir=0; dir<4; dir++)
		_update_directional(data, move, e_turn, inc, dir, old, vmove);
}

static int _immediate_test_directional(ThinkAccelData *data, Coordinate *move, Coordinate coor, int e_turn, int dir)
{
	Five *tmp;
	int i;

	for(i=0; i<5; i++) {
		tmp = &ARRAY_COOR(data->fives.five[dir], coor);
		if(tmp->amount[e_turn+1] == 4 && tmp->amount[0] == 1) {
			_spaces_find_in_five(coor, move, dir);
			return 1;
		}
		if(i<4 && coordinate_add_with_clip(&coor, &direction[dir^4]))
			break;
	}

	return 0;
}

int ai_think_accel_immediate_test(ThinkAccelData *data, Coordinate *move, Coordinate *coor, int e_turn)
{
	int dir;

	if(!data->state[1])
		return 0;
	for(dir=0; dir<4; dir++)
		if(_immediate_test_directional(data, move, *coor, e_turn, dir) != 0)
			return 1;

	return 0;
}

inline int ai_think_accel_relevance(ThinkAccelData *think_accel_data, Coordinate *c)
{
	return ARRAY_COOR(think_accel_data->think_map.val, *c);
}

inline int ai_think_accel_board_value(ThinkAccelData *think_accel_data, int e_turn)
{
	return  (_threat_value[think_accel_data->threats[e_turn].threat_count[0]] -
		_threat_value[think_accel_data->threats[e_turn^1].threat_count[0]]);
}
