// Copyright (c) 1996-1999 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu

//---------------------------------------------------------------------------
//
// $Id: IIRScram_SignalAssignmentStatement.cc,v 1.2 1999/03/09 20:56:36 dmartin Exp $
//
//---------------------------------------------------------------------------
#include "IIR_SignalAssignmentStatement.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_Label.hh"
#include "IIR_WaveformElement.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "set.hh"
#include "symbol_table.hh"

extern symbol_table *cgen_sym_tab_ptr;

IIRScram_SignalAssignmentStatement::~IIRScram_SignalAssignmentStatement() {}

void
IIRScram_SignalAssignmentStatement::_publish_vhdl(ostream &_vhdl_out) {

  if (get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ": ";
  }

  get_target()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " <= ";

  if(get_delay_mechanism() == IIR_TRANSPORT_DELAY) {
    _vhdl_out << "transport ";
  } else if (get_delay_mechanism() == IIR_INERTIAL_DELAY) {
    _vhdl_out << "inertial ";
  } else {
    if (get_reject_time_expression() != NULL) {
      _vhdl_out << "reject ";
      get_reject_time_expression()->_publish_vhdl(_vhdl_out);
      _vhdl_out << " inertial ";
    }
  }

  waveform._publish_vhdl(_vhdl_out);
}

void
IIRScram_SignalAssignmentStatement::_publish_cc() {
  ASSERT(get_target()->_is_resolved());
  IIR_WaveformElement* waveform_element = waveform.first();
  // Indicates the first itereation.  We need this to publish the
  // waveforms other than the first with transport delay mechanism.
  IIR_Boolean first = TRUE;

  while(waveform_element != NULL) {

    if (get_target()->get_kind() == IIR_AGGREGATE) {
      // Create a temporary aggregate and use that in the signal assignment
      _cc_out << "{\n";
      ASSERT ( get_target()->_get_subtype() != NULL );
      get_target()->_get_subtype()->_publish_cc_type_name();
      _cc_out << " tempAggregate(";
      get_target()->_publish_cc_constructor_args();
      get_target()->_publish_cc_initialization_value();
      _cc_out << ");\n";
    }
    
    if(_is_currently_publishing_subprogram() == TRUE) {
      _cc_out << "processPtr->";
    }

    if(waveform_element->_is_null_transaction() == TRUE) {
      _cc_out << "assignNullTransaction(";
    }
    else {
      _cc_out << "assignSignal(";
    }

    switch(get_target()->get_kind()) {
      IIR_Declaration *destsig;
    case IIR_SLICE_NAME:
      destsig = (IIR_Declaration*)get_target()->_get_prefix_declaration();
      destsig->_publish_cc();
      if (!cgen_sym_tab_ptr->in_scope(destsig)) {
	cgen_sym_tab_ptr->add_declaration(destsig);
      }
      break;
    case IIR_INDEXED_NAME:
    case IIR_SELECTED_NAME:
      get_target()->_publish_cc();
      destsig = get_target()->_get_prefix_declaration();
      if (!cgen_sym_tab_ptr->in_scope(destsig)) {
	cgen_sym_tab_ptr->add_declaration(destsig);
      }
      break;

    case IIR_AGGREGATE:
      _cc_out << "tempAggregate";
      destsig = get_target()->_get_prefix_declaration();
      if ((destsig != NULL) && 
	  (cgen_sym_tab_ptr->in_scope(destsig) == FALSE)) {
	cgen_sym_tab_ptr->add_declaration(destsig);
      }
      break;
    
    default:
      get_target()->_publish_cc();
      destsig = get_target()->_get_prefix_declaration();
      if ((destsig != NULL) && 
	  (cgen_sym_tab_ptr->in_scope(destsig) == FALSE)) {
	cgen_sym_tab_ptr->add_declaration(destsig);
      }
    }
    
    if(_is_currently_publishing_subprogram() == TRUE) {
      _cc_out << ", processPtr->myId(), ";
    } else {
      _cc_out << ", id, ";
    }      
    waveform_element->_publish_cc(); // data source and delay value
    
    if(first == FALSE) {
      _cc_out << "SAVANT_ZERO_TIME";
    } else if (get_delay_mechanism() == IIR_TRANSPORT_DELAY) {
      _cc_out << "SAVANT_ZERO_TIME";
    } else {
      //       ASSERT(get_reject_time_expression() != NULL);
      if(get_reject_time_expression() == NULL) {
	if(waveform.first()->get_time() != NULL) {
	  waveform.first()->get_time()->_publish_cc();
	} else {
	  _cc_out << "SAVANT_ZERO_TIME";
	}
      } else {
	get_reject_time_expression()->_publish_cc();
      } 
    }
    _cc_out << ", ";
    
    get_target()->_publish_cc_bounds();
    _cc_out << ", ";
    waveform_element->_publish_cc_bounds();
    _cc_out << ");" << endl;
    waveform_element = waveform.successor(waveform_element);
    first = FALSE;

    if (get_target()->get_kind() == IIR_AGGREGATE) {
      _cc_out << "}\n";
    }
  }
}

void 
IIRScram_SignalAssignmentStatement::_type_check(){
  _type_check_target_and_waveform( &waveform );
  _type_check_mechanism_and_time( &waveform );
}

void
IIRScram_SignalAssignmentStatement::_get_list_of_input_signals(set<IIR_Declaration>* list) {
  waveform._get_list_of_input_signals(list);
}


IIR_Boolean
IIRScram_SignalAssignmentStatement::_is_sequential_signal_assignment() {
  return TRUE;
}


void
IIRScram_SignalAssignmentStatement::_get_signal_source_info(set<IIR_Declaration>* siginfo) {

  ASSERT(get_target()->_is_signal() == TRUE);
  IIR* my_target = get_target();
  ASSERT(my_target != NULL);
  if(my_target->get_kind() == IIR_ALIAS_DECLARATION) {
    siginfo->add((IIR_Declaration*)((IIR_AliasDeclaration*)my_target)->get_name());
  } 
  else if((my_target->_is_name() == TRUE) || 
	  (my_target->_is_aggregate_expression() == TRUE)){
    my_target->_get_signal_source_info(siginfo);
  }
  else {
    ASSERT(my_target->_is_iir_declaration() == TRUE);
    siginfo->add((IIR_Declaration*)my_target);
  }
}

IIR *
IIRScram_SignalAssignmentStatement::_clone() {
  IIR_SignalAssignmentStatement *stmt;
  IIR *target;
  IIR_WaveformElement *newwave, *oldwave;

  stmt = new IIR_SignalAssignmentStatement();
  IIRScram_SequentialStatement::_clone(stmt);

  target = get_target();
  target = target->_clone();
  stmt->set_target(target);
  stmt->set_delay_mechanism(get_delay_mechanism());
  stmt->set_reject_time_expression(get_reject_time_expression());

  oldwave = waveform.first();
  while(oldwave != NULL) {
    newwave = (IIR_WaveformElement*)oldwave->_clone();
    stmt->waveform.append(newwave);
    oldwave = waveform.successor(oldwave);
  }
  return stmt;
}


IIR *
IIRScram_SignalAssignmentStatement::_get_target(){
  return get_target();
}

void 
IIRScram_SignalAssignmentStatement::_set_target( IIR *new_target ){
  set_target( new_target );
}

IIR_DelayMechanism 
IIRScram_SignalAssignmentStatement::_get_delay_mechanism(){
  return get_delay_mechanism();
}

void 
IIRScram_SignalAssignmentStatement::_set_delay_mechanism( IIR_DelayMechanism new_mech ){
  set_delay_mechanism( new_mech );
}

IIR *
IIRScram_SignalAssignmentStatement::_get_reject_time_expression(){
  return get_reject_time_expression();
}

IIR_Boolean 
IIRScram_SignalAssignmentStatement::_is_resolved(){
  IIR_Boolean retval = TRUE;

  if( get_target() && get_target()->_is_resolved() == FALSE ){
    retval = FALSE;
  }
  else if( get_reject_time_expression() && get_reject_time_expression()->_is_resolved() == FALSE ){
    retval = FALSE;
  }
  else if ( waveform._is_resolved() == FALSE ){
    retval = FALSE;
  }

  return retval;
}

void 
IIRScram_SignalAssignmentStatement::_set_reject_time_expression( IIR *new_reject_time ){
  set_reject_time_expression( new_reject_time );
}
