
// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_DeclarationList.cc,v 1.7 1999/10/26 16:44:41 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIRScram_DeclarationList.hh"
#include "IIR_AttributeSpecification.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_LibraryClause.hh"
#include "IIR_LibraryDeclaration.hh"
#include "IIR_SelectedName.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_UseClause.hh"
#include "IIR_DisconnectSpecification.hh"
#include "error_func.hh"
#include "symbol_table.hh"
#include "set.hh"
#include "IIR_FileDeclaration.hh"

IIRScram_DeclarationList::~IIRScram_DeclarationList() {}

void
IIRScram_DeclarationList::_publish_vhdl(ostream &_vhdl_out) {
  IIR_Declaration *decl = first();
  while (decl != NULL) {
    decl->_publish_vhdl(_vhdl_out);
    decl = successor(decl);
  }
}

void 
IIRScram_DeclarationList::_publish_vhdl_decl(ostream &_vhdl_out) {
  IIR_Declaration *decl = first();
  
  while (decl != NULL){
    decl->_publish_vhdl_decl(_vhdl_out);
    decl = successor(decl);
  }
}

void
IIRScram_DeclarationList::_publish_cc_type_info(){
  IIR_Declaration *decl;
  set<IIR> headerSet;
  IIR_Boolean has_sub_prog = FALSE;
  PublishedUnit oldUnit = _get_currently_publishing_unit();

  _set_currently_publishing_unit(TYPE);
  
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      // Include this type in the current scope if it is not incomplete
      if (decl->_is_incomplete_type_declaration() == FALSE) {
	headerSet.add(decl);
      }
      break;
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      decl->_get_headers(headerSet);
      has_sub_prog = TRUE;
      break;
    default:
      break;
    }
    decl = successor(decl);
  }
  
  IIR *typDecl = headerSet.get_element();
  while (typDecl != NULL) {
    _cc_out << "#include \"";
    typDecl->_publish_cc_type_name();
    _cc_out << ".hh\"\n";
    typDecl = headerSet.get_next_element();
  }
    
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      if (decl->_is_incomplete_type_declaration() == FALSE) {
	decl->_publish_cc_type_info();
      }
      break;
    }
    decl = successor(decl); 
  }
  
  _set_currently_publishing_unit(oldUnit);
}

void
IIRScram_DeclarationList::_publish_cc_extern_type_info(){
  IIR_Declaration *decl;
  set<IIR> headerSet;
  IIR_Boolean has_sub_prog = FALSE;
  PublishedUnit oldUnit = _get_currently_publishing_unit();

  _set_currently_publishing_unit(TYPE);
  
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      // Include this type in the current scope if it is not incomplete
      if (decl->_is_incomplete_type_declaration() == FALSE) {
	headerSet.add(decl);
      }
      break;
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      decl->_get_headers(headerSet);
      has_sub_prog = TRUE;
      break;
    default:
      break;
    }
    decl = successor(decl);
  }
  
  IIR *typDecl = headerSet.get_element();
  while (typDecl != NULL) {
    _cc_out << "#include \"";
    typDecl->_publish_cc_type_name();
    _cc_out << ".hh\"\n";
    typDecl = headerSet.get_next_element();
  }
    
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      if (decl->_is_incomplete_type_declaration() == FALSE) {
	decl->_publish_cc_extern_type_info();
      }
      break;
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      decl->_publish_cc_extern_type_info();
      break;
    }
    decl = successor(decl); 
  }
  
  _set_currently_publishing_unit(oldUnit);
}

void 
IIRScram_DeclarationList::_publish_cc() {
  IIR_Declaration *decl;
  IIR_Char* temp_file_name = NULL;
  IIR_Char* file_name = NULL;
  set<IIR> headerSet;
  IIR_Boolean has_sub_prog = FALSE;

  // First collect all the includes needed for this declartive region
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      // Include this type in the current scope if it is not incomplete!!
      if (decl->_is_incomplete_type_declaration() == FALSE) {
	headerSet.add(decl);
      }
      break;
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      decl->_get_headers(headerSet);
      has_sub_prog = TRUE;
      break;

    case IIR_CONSTANT_DECLARATION:
      headerSet.add(decl->_get_subtype());
      break;
      
    default:
      break;
    }
    decl = successor(decl);
  }

  temp_file_name = _cc_out.get_file_name();
  if(temp_file_name != NULL) {
    file_name = new char[strlen(temp_file_name) + 1];
    strcpy(file_name, temp_file_name);
  }

  if(file_name != NULL) {
    _cc_out.set_file(file_name);

    // If the declarative region has sub programs then include these stuff
    if (has_sub_prog == TRUE) {
      _cc_out << "#include \"VHDLKernel.hh\"" << endl
	      << "#include \"FileQueue.hh\"" << endl
	      << "#include \"textio.hh\"" << endl;
    }
    // Walk through the collected list and publish the headers
    IIR *typDecl = headerSet.get_element();
    while (typDecl != NULL) {
      _cc_out << "#include \"";
      if (typDecl->_is_iir_type_definition() == FALSE &&
	  typDecl->_get_subtype()->_is_scalar_type() == TRUE &&
	  typDecl->_get_subtype()->_is_kernel_type() == FALSE) {
	typDecl->_get_subtype()->_publish_cc_kernel_type();
      }
      else if (typDecl->_is_iir_type_definition() == TRUE  &&
	       typDecl->_is_scalar_type() == TRUE &&
	       typDecl->_is_kernel_type() == FALSE) {
	typDecl->_publish_cc_kernel_type();
      }
      else {
	typDecl->_publish_cc_type_name();
      }
      _cc_out << ".hh\"\n";
      typDecl = headerSet.get_next_element();
    }
  }

   decl = first();
  while (decl != NULL) {
    if(file_name != NULL) {
      _cc_out.set_file(file_name);
    }
    switch (decl->get_kind()) {
    case IIR_COMPONENT_DECLARATION:
      _cc_out << "#include \"SCOM";
      (decl->_get_declarator())->_publish_cc_elaborate();
      _cc_out << "_elab.hh\"\n";
      break;
    case IIR_ATTRIBUTE_SPECIFICATION:
      _cc_out << endl
	      << "extern ";
      decl->_publish_cc_decl();
      break;
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      if (decl->_is_incomplete_type_declaration() == FALSE) {
	decl->_publish_cc_decl();
      }
      break;
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      ((IIR_SubprogramDeclaration *)decl)->_publish_cc_prototype();
      break;
    case IIR_USE_CLAUSE:
      decl->_publish_cc_headers();
      break;

    case IIR_CONSTANT_DECLARATION:
      _cc_out << "extern ";
      decl->_publish_cc_type_name();
      _cc_out << " ";
      decl->_get_declarator()->_publish_cc();
      _cc_out << ";\n";
      break;
      
    default:
      break;
    }
    decl = successor(decl);
  }
  
  if (file_name != NULL) {
    delete [] file_name;
  }
}

void 
IIRScram_DeclarationList::_publish_cc_decl() {
  IIR_Declaration *decl;
  IIR_Char* temp_file_name = NULL;
  IIR_Char* file_name = NULL;
  IIR_Int32 num_decls_in_file = 0;

  temp_file_name = _cc_out.get_file_name();
  if(temp_file_name != NULL) {
    file_name = new char[strlen(temp_file_name) + 1];
    strcpy(file_name, temp_file_name);
  }
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_SIGNAL_DECLARATION:
    case IIR_VARIABLE_DECLARATION:
    case IIR_SIGNAL_INTERFACE_DECLARATION:
    case IIR_VARIABLE_INTERFACE_DECLARATION:
    case IIR_FILE_DECLARATION:
    case IIR_CONFIGURATION_SPECIFICATION:
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
    case IIR_ALIAS_DECLARATION:
    case IIR_ATTRIBUTE_DECLARATION:
    case IIR_USE_CLAUSE:
      break;

     case IIR_CONSTANT_DECLARATION:
       decl->_publish_cc_type_name();
       _cc_out << " ";
       decl->_get_declarator()->_publish_cc();

       if ((decl->get_value() != NULL) &&
	   (decl->get_value()->_is_static_expression() == TRUE)) {
	 decl->_publish_cc_constructor_args();
	 _cc_out << ";\n";
       }
       else {
	 _cc_out << "(ObjectBase::VARIABLE);\n";
       }
       break;
       
    case IIR_COMPONENT_DECLARATION:
      decl->_publish_cc_elaborate();
      // The above call will change the name of the output .cc file so
      // we need to restore the name of the file after the above call...
      _cc_out.set_file(FALSE, file_name);
      break;
      
    case IIR_ATTRIBUTE_SPECIFICATION:
      ((IIR_AttributeSpecification*)decl)->_publish_cc_decl_with_constructor();
      break;
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      decl->_publish_cc_decl();
      num_decls_in_file++;
      break;
    default:
      cerr << "Unimplemented declaration type: " << decl->get_kind_text()
	   << " in IIRScram_DeclarationList::_publish_cc_decl()." << endl;
    }
    decl = successor(decl);
  }
  // WARNING: This place is spurious!!!
  // If the file does not have any declarations, just remove it.  It
  // causes g++ to really bitch wierdly during linking.
  if(num_decls_in_file == 0) {
    switch (_get_currently_publishing_unit()) {
    case IIRScram::ENTITY_DECL: {
      IIR_EntityDeclaration* entity_decl = (IIR_EntityDeclaration*)_current_publish_node;
      if(entity_decl->generic_clause.num_elements() == 0) {
	// ::remove(file_name);
      }
      break;
    }
    case IIRScram::BLOCK: {
      IIR_BlockStatement *block = (IIR_BlockStatement*)_current_publish_node;
      if (block->generic_clause.num_elements() == 0) {
	// ::remove(file_name);
      }
      break;
    }
    default:
      // ::remove(file_name);
      break;
    }
  }
  if(file_name != NULL) {
    delete [] file_name;
  }
}
  
void 
IIRScram_DeclarationList::_publish_cc_package_declarations() {
  IIR_Declaration *decl;
  IIR_Char* temp_file_name = NULL;
  IIR_Char* file_name = NULL;

  temp_file_name = _cc_out.get_file_name();
  if(temp_file_name != NULL) {
    file_name = new char[strlen(temp_file_name) + 1];
    strcpy(file_name, temp_file_name);
  }

  decl = first();
  while (decl != NULL) {
    if(file_name != NULL) {
      _cc_out.set_file(file_name);
    }
    switch (decl->get_kind()) {
    case IIR_COMPONENT_DECLARATION:
      decl->_publish_cc_elaborate();
      break;
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      decl->_publish_cc_decl();
      break;
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      ((IIR_SubprogramDeclaration *)decl)->_publish_cc_prototype();
      break;
    case IIR_ALIAS_DECLARATION:
      decl->_publish_cc_decl();
      break;
    case IIR_USE_CLAUSE:
    case IIR_ATTRIBUTE_DECLARATION:
      break;
    case IIR_ATTRIBUTE_SPECIFICATION:
      _cc_out << "extern ";
      decl->_publish_cc_decl();
      break;      
    case IIR_CONSTANT_DECLARATION:
      _cc_out << "extern ";
      if (decl->_get_subtype()->_is_scalar_type() == TRUE &&
	  decl->_get_subtype()->_is_kernel_type() == FALSE){
	decl->_get_subtype()->_publish_cc_kernel_type();
      }
      else {
	decl->_get_subtype()->_publish_cc_type_name();
      }
      _cc_out << " ";
      decl->_publish_cc();
      _cc_out << ";" << endl;
      break;
    case IIR_SIGNAL_DECLARATION:
      _cc_out << "extern ";
      if (decl->_get_subtype()->_is_scalar_type() == TRUE &&
	  decl->_get_subtype()->_is_kernel_type() == FALSE){
	decl->_get_subtype()->_publish_cc_kernel_type();
      }
      else {
	decl->_get_subtype()->_publish_cc_type_name();
      }
      _cc_out << " ";
      decl->_publish_cc();
      _cc_out << "_info;" << endl;
      break;
    default:
      cerr << "Unimplemented declaration type: " << decl->get_kind_text()
	   << " in IIRScram_DeclarationList::_publish_cc_package_declarations()." 
	   << endl;
    }
    decl = successor(decl);
  }
  if(file_name != NULL) {
    delete [] file_name;
  }
}

void 
IIRScram_DeclarationList::_publish_cc_decl_subprograms_and_types() {
  IIR_Declaration *decl;
  IIR_Char *temp_file_name = _cc_out.get_file_name();
  IIR_Char *file_name = NULL;

  if(temp_file_name != NULL) {
    file_name = new char[strlen(temp_file_name) + 1];
    strcpy(file_name, temp_file_name);
  }

  decl = first();
  while (decl != NULL) {
    if(decl->_is_subprogram() == TRUE ||
       decl->_is_type() == TRUE) {
      decl->_publish_cc_decl();
    }
    decl = successor(decl);

    // The file that _cc_out writes to might be changed in the function
    // call.  We have to reset it here.
    if(file_name != NULL) {
      _cc_out.set_file(file_name);
    }
  }

  if(file_name != NULL) {
    delete [] file_name;
  }
}  

void 
IIRScram_DeclarationList::_publish_cc_prototype() {
  IIR_Declaration *decl;

  decl = first();
  while (decl != NULL) {
    if(decl->_is_subprogram() == TRUE) {
      ((IIR_SubprogramDeclaration *) decl)->_publish_cc_prototype();
    }
    decl = successor(decl);
  }
}  

set<IIR_Declaration> *
IIRScram_DeclarationList::_find_declarations( IIR_Name *to_find ){
  ASSERT( to_find != NULL );
  set<IIR_Declaration> *retval = NULL;
  // The following assertion holds true because an indexed name doesn't generally
  // map into a declaration.  So, we can't expect to look at a list of declarations,
  // searching for an indexed name, and really find anything.
  ASSERT( to_find->get_kind() != IIR_INDEXED_NAME );
  
  switch( to_find->get_kind() ){
  case IIR_SIMPLE_NAME:{
    ASSERT( to_find->get_prefix() != NULL );
    ASSERT( to_find->get_prefix()->_is_iir_text_literal() == TRUE );
    IIR_TextLiteral *prefix = (IIR_TextLiteral *)to_find->get_prefix();
    retval = _find_declarations( prefix );
    break;
  }
  case IIR_SELECTED_NAME:{
    IIR_SelectedName *as_selected_name = (IIR_SelectedName *)to_find;
    ASSERT( to_find->get_prefix() != NULL );
    ASSERT( to_find->get_prefix()->_is_iir_name() == TRUE );
    
    set<IIR_Declaration> *found = _find_declarations( (IIR_Name *)to_find->get_prefix() );
    if( found == NULL ){
      retval = NULL;
    }
    else{
      ASSERT( as_selected_name->get_suffix() != NULL );
      ASSERT( as_selected_name->get_suffix()->_is_iir_name() == TRUE );
      set<IIR_Declaration> *current_set;
      retval = new set<IIR_Declaration>;

      IIR_Declaration *current_decl = found->get_element();
      while( current_decl != NULL ){
	current_set = 
	  current_decl->_find_declarations( (IIR_Name *)as_selected_name->get_suffix() );
	if( current_set != NULL ){
	  retval->add( current_set );
	  delete current_set;
	}
	current_decl = found->get_next_element();
      }

      if( retval->num_elements() == 0 ){
	delete retval;
	retval = NULL;
      }
    }
    delete found;

    break;
  }
  default:{
    ostrstream err;
    err << "Internal error in IIRScram_DeclarationList::_find_declarations - "
	<< " don't know how to find a " << to_find->get_kind_text() << ends;
    report_error( to_find, err );
  }
  }

  return retval;
}

set<IIR_Declaration> *
IIRScram_DeclarationList::_find_declarations( IIR_TextLiteral *to_find ){
  set<IIR_Declaration> *retval = new set<IIR_Declaration>;
  ASSERT( to_find != NULL );
  
  IIR_Declaration *current = first();
  while( current != NULL ){
    if( IIR_TextLiteral::_cmp( current->get_declarator(), to_find ) == 0 ){
      retval->add( current );
    }
    else{
      if( current->get_kind() == IIR_TYPE_DECLARATION ){
	// We need to search the implicit declarations for whatever we're
	// looking for.
	IIR_TypeDeclaration *as_type_decl = (IIR_TypeDeclaration *)current;
	ASSERT( as_type_decl->_get_implicit_declarations() != NULL );
	IIR_Declaration *current_implicit;
	current_implicit = as_type_decl->_get_implicit_declarations()->get_element();
	while( current_implicit != NULL ){
	  if( IIR_TextLiteral::_cmp( current_implicit->get_declarator(), to_find ) == 0 ){
	    retval->add( current_implicit );
	  }
	  current_implicit = as_type_decl->_get_implicit_declarations()->get_next_element();
	}

	// If we have an enumeration type, we need to search for the our
	// culprit in the literal list.
	if( current->_is_enumeration_type() == TRUE ){
	  set<IIR_Declaration> *enumeration_literals = current->_find_declarations( to_find );
	  if( enumeration_literals != NULL ){
	    ASSERT( enumeration_literals->num_elements() == 1 );
	    retval->add( enumeration_literals );
	  }
	}
      }
    }
    current = successor( current );
  }

  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }

  return retval;
}

void
IIRScram_DeclarationList::_publish_cc_constants() {
  IIR_Declaration *current_declaration = first();
  
  while (current_declaration != NULL) {
    if (current_declaration->_is_constant() == TRUE) {
      current_declaration->_publish_cc_type_name();
      _cc_out << " ";
      current_declaration->_get_declarator()->_publish_cc();
      _cc_out << ";\n";
    }
    
    current_declaration = successor(current_declaration);
  }
}

void
IIRScram_DeclarationList::_publish_cc_global_constants_assignments() {
  IIR_Declaration *current_declaration = first();
  
  while (current_declaration != NULL) {
    if (current_declaration->get_kind() == IIR_CONSTANT_DECLARATION) {
      _cc_out << "::";
      current_declaration->_get_declarator()->_publish_cc();
      _cc_out << " = ";
      current_declaration->_publish_cc_type_name();
      current_declaration->_publish_cc_constructor_args();
      _cc_out << ";\n";
    }
    
    current_declaration = successor(current_declaration);
  }
}

void
IIRScram_DeclarationList::_publish_cc_file_objects() {
  IIR_Declaration *decl = first();
  
  while (decl != NULL) {
    if (decl->get_kind() == IIR_FILE_DECLARATION) {
      if (((IIR_FileDeclaration *) decl)->_is_write_file() == TRUE) {
	decl->_get_subtype()->_publish_cc_type_name();
	_cc_out << " ";
	decl->_get_declarator()->_publish_cc_elaborate();
	_cc_out << ";\n";
      }
    }
    decl = successor(decl);
  }
}

void
IIRScram_DeclarationList::_publish_cc_file_objects_init() {
  IIR_Declaration *decl = first();
  
  while (decl != NULL) {
    if (decl->get_kind() == IIR_FILE_DECLARATION) {
      if (((IIR_FileDeclaration *) decl)->_is_write_file() == TRUE) {
	decl->_get_declarator()->_publish_cc();
	_cc_out << ".reset(";
	((IIR_FileDeclaration*)decl)->get_file_logical_name()->_publish_cc();
	_cc_out << ");\n";
      }
    }
    decl = successor(decl);
  }
}


IIR_Boolean
IIRScram_DeclarationList::_publish_cc_constants_init(IIR_Boolean firstDecl) {
  IIR_Declaration *current_declaration = first();
  IIR_Boolean     flag = FALSE;
  
  while (current_declaration != NULL) {
    switch (current_declaration->get_kind()) {
    case IIR_CONSTANT_DECLARATION:
      if (firstDecl == TRUE) {
	_cc_out << " : ";
	firstDecl = FALSE;
      }
      else {
	_cc_out << ",\n";
      }
      
      current_declaration->_get_declarator()->_publish_cc();
      current_declaration->_publish_cc_constructor_args();
      flag = TRUE;
      break;
    }
    
    current_declaration = successor(current_declaration);
  }

  return flag;
}

void 
IIRScram_DeclarationList::_type_check_configuration_specifications( IIR_List &statement_list ){
  IIR_Declaration *current_declaration = first();
  while( current_declaration != NULL ){
    if( current_declaration->_is_iir_configuration_specification() == TRUE ){
      IIR_ConfigurationSpecification *as_specification;
      as_specification = (IIR_ConfigurationSpecification *)current_declaration;
      as_specification->_type_check( statement_list );
    }
    current_declaration = successor( current_declaration );
  }
}

void 
IIRScram_DeclarationList::_type_check_disconnection_specifications( ){
  IIR_Declaration *current_declaration = first();
  while( current_declaration != NULL ){
    if( current_declaration->_is_iir_disconnect_specification() == TRUE ){
      IIR_DisconnectSpecification *as_specification;
      as_specification = (IIR_DisconnectSpecification *)current_declaration;
      as_specification->_type_check( *(IIR_DeclarationList *)this );
    }
    current_declaration = successor( current_declaration );
  }
}

void 
IIRScram_DeclarationList::_type_check_attribute_specifications( IIR_List &statement_list ){
  IIR_Declaration *current_declaration = first();
  while( current_declaration != NULL ){
    if( current_declaration->_is_iir_attribute_specification() == TRUE ){
      IIR_AttributeSpecification *as_specification;
      as_specification = (IIR_AttributeSpecification *)current_declaration;
      as_specification->_type_check( *((IIR_DeclarationList *)this) );
    }
    current_declaration = successor( current_declaration );
  }
}

void 
IIRScram_DeclarationList::_make_visible_as_context_list( symbol_table *sym_tab ){
  IIR_Declaration *current = first();
  while( current != NULL ){
    ASSERT( current->get_kind() == IIR_LIBRARY_CLAUSE || 
	    current->get_kind() == IIR_USE_CLAUSE );
    
    if( current->get_kind() == IIR_LIBRARY_CLAUSE ){
      IIR_LibraryClause *as_clause = (IIR_LibraryClause *)current;
      if( as_clause->_get_library_declaration() != NULL ){
	sym_tab->make_visible( as_clause->_get_library_declaration() );
      }
    }
    else{
      ASSERT( current->get_kind() == IIR_USE_CLAUSE );
      IIR_UseClause *as_use_clause = (IIR_UseClause *)current;
      ASSERT( as_use_clause->get_selected_name() != NULL );
      current->_make_interface_visible( sym_tab );
    }
    
    current = successor( current );
  }
}



IIR_DeclarationList *
IIRScram_DeclarationList::_build_constrained_list( IIR_Boolean (IIR::*constraint_function)() ){
  ASSERT ( constraint_function != 0 );
  
  // Not using clone because clone copies the elements in the list.
  // We want a new list that contains the exact same elements.
  IIR_DeclarationList *cloned_list = new IIR_DeclarationList();
  //  copy_location( this, cloned_list );
  _listcopy( *cloned_list, *this );

  ASSERT( cloned_list->get_kind() == get_kind() );
  ASSERT( cloned_list->num_elements() == num_elements() );

  IIR_Declaration *next = NULL;
  IIR_Declaration *current = cloned_list->first();
  while( current != NULL ){
    next = cloned_list->successor( current );    
    if( (current->*constraint_function)() == FALSE ){
      cloned_list->remove( current );
    }
    current = next;
  }
  
  return cloned_list;
}


IIR *
IIRScram_DeclarationList::_clone(){
  IIR_DeclarationList *my_clone = new IIR_DeclarationList();
  IIR_List::_clone( my_clone );

  ASSERT( get_kind() == my_clone->get_kind() );

  return my_clone;
}
