;;; x-symbol-tex.el --- token language "TeX macro" for package x-symbol

;; Copyright (C) 1996-1999 Free Software Foundation, Inc.
;;
;; Author: Christoph Wedler <wedler@fmi.uni-passau.de>
;; Maintainer: (Please use `M-x x-symbol-package-bug' to contact the maintainer)
;; Version: $Id: x-symbol-tex.el,v 3.3 1999/01/18 14:17:12 wedler Exp wedler $
;; Keywords: WYSIWYG, LaTeX, wp, math, internationalization
;; X-URL: http://www.fmi.uni-passau.de/~wedler/x-symbol/

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; If you want to use package x-symbol, please visit the URL (use
;; \\[x-symbol-package-web]) and read the info (use \\[x-symbol-package-info]).

;; Token language tex is registered in x-symbol-hooks.

;;; Code:

(provide 'x-symbol-tex)
(require 'x-symbol-vars)


;;;===========================================================================
;;;  General language accesses, see `x-symbol-language-access-alist'
;;;===========================================================================

(defcustom x-symbol-tex-modeline-name "tex"
  "*String naming the language TeX in the modeline."
  :group 'x-symbol-tex
  :type 'string)

(defcustom x-symbol-tex-header-groups-alist nil
  "*If non-nil, used in TeX specific grid/menu.
See `x-symbol-header-groups-alist'."
  :group 'x-symbol-tex
  :group 'x-symbol-input-init
  :type 'x-symbol-headers)

(defcustom x-symbol-tex-electric-ignore 'x-symbol-tex-default-electric-ignore
  "*Additional TeX version of `x-symbol-electric-ignore'."
  :group 'x-symbol-tex
  :group 'x-symbol-input-control
  :type 'x-symbol-function-or-regexp)
;;;  :options '(x-symbol-tex-default-electric-ignore)) ; any use?

(defcustom x-symbol-tex-electric-ignore-regexp "[A-Za-z]~\\'"
  "*Regexp match contexts not to be used in TeX for input method ELECTRIC.
Used by `x-symbol-tex-default-electric-ignore'."
  :group 'x-symbol-tex
  :group 'x-symbol-input-control
  :type '(choice (const :tag "None" nil) regexp))

(defvar x-symbol-tex-required-fonts nil
  "List of features providing fonts for language `tex'.")
(defvar x-symbol-tex-case-insensitive nil
  "If non-nil, TeX macros are case insensitive.  They are not.")
(defvar x-symbol-tex-token-shape '(?\\ "\\\\[@A-Za-z]+\\'" . "[@A-Za-z]")
  "Shape of TeX macros.  See `x-symbol-language-access-alist'.")
(defvar x-symbol-tex-exec-specs
  '(nil (nil . "\\\\[@A-Za-z]+"))
  "Specification for TeX's executables.
See `x-symbol-exec-create' and `x-symbol-language-access-alist'.")

(defvar x-symbol-tex-token-list 'x-symbol-tex-default-token-list
  "Function returning a list of TeX macros from table specification.
The TOKEN-SPEC in tex tables look like (STRING...) or STRING.  See
`x-symbol-init-language' and `x-symbol-tex-default-token-list'.")

(defvar x-symbol-tex-input-token-ignore "\\`{.+}\\'"
  "Regexp matching TeX macros not being used for input method TOKEN.
See `x-symbol-init-language'.")

(defvar x-symbol-tex-user-table nil
  "User table defining TeX macros, used in `x-symbol-tex-table'.")


;;;===========================================================================
;;;  Image support
;;;===========================================================================

(defcustom x-symbol-tex-master-directory 'x-symbol-tex-default-master-directory
  "Function returning the directory of the master file or nil.
See `x-symbol-image-parse-buffer'."
  :group 'x-symbol-tex
  :group 'x-symbol-image-language
  :type 'function)

(defcustom x-symbol-tex-image-searchpath
  (let ((dirs (parse-colon-path (or (getenv "TEXPICTS") (getenv "TEXINPUTS"))))
	result)
    (while dirs
      (or (null (car dirs))
	  (string-equal (car dirs) "/") ; not ending with //
	  (member (car dirs) result)
	  (push (car dirs) result))
      (setq dirs (cdr dirs)))
    (nreverse (if (member "./" result) result (cons "./" result))))
  "Search path used for implicitly relative image file names.
See `x-symbol-image-use-remote'."
  :group 'x-symbol-tex
  :group 'x-symbol-image-language
  :type '(repeat directory))

(defcustom x-symbol-tex-image-cached-dirs '("figures/")
  "Directory parts of image file names stored in the memory cache.
See `x-symbol-image-use-remote'."
  :group 'x-symbol-tex
  :group 'x-symbol-image-language
  :type '(repeat string))

(defcustom x-symbol-tex-image-keywords
  ;; keep it short!
  '("\\.\\(eps\\|ps\\(tex\\)?\\|gif\\|png\\|jpe?g\\)\\'"
    ("\\\\epsf\\(box\\|file\\)[ \t]*\\(\\[[^][\n]*\\]\\)?{\\([^ \t\n,{}]+\\.e?ps\\)}" 3)
    ("\\\\e?psfig[ \t]*{file=\\([^ \t\n,{}]+\\.e?ps\\)[^\n{}]*}" 1)
    ("\\\\includegraphics\\*?[ \t]*\\(\\[[^][\n]*\\]\\)?\\(\\[[^][\n]*\\]\\)?{\\([^ \t\n,{}]+\\)}" 3 ".\\.[^./]+\\'" ".eps")
    ("\\\\input[ \t]*{\\([^ \t\n,{}]+\\.pstex\\)_t}" 1))
  "Keywords used to find image insertion commands.
See `x-symbol-image-parse-buffer'."
  :group 'x-symbol-tex
  :group 'x-symbol-image-language
  :type 'x-symbol-image-keywords)


;;;===========================================================================
;;;  Super- and Subscripts
;;;===========================================================================

(defvar x-symbol-tex-font-lock-keywords
  '((x-symbol-tex-match-subscript
     (1 x-symbol-invisible-face t)
     (2 (if (eq (char-after (match-beginning 1)) ?\_)
	    'x-symbol-sub-face 'x-symbol-sup-face)
	prepend))
    (x-symbol-tex-match-simple-subscript
     (1 x-symbol-invisible-face t)
     (2 (if (eq (char-after (match-beginning 1)) ?\_)
	    'x-symbol-sub-face 'x-symbol-sup-face)
	prepend)))
  "TeX font-lock keywords for super- and subscripts.")

(defcustom x-symbol-tex-font-lock-allowed-faces
  '(font-lock-string-face font-lock-doc-string-face font-latex-math-face)
  "*Faces which are allowed when fontifying simple super- and subscripts.
Package x-symbol only uses super- and subscripts if they are in braces,
if the \"^\"/\"_\" has not been fontified yet or is only fontified with
faces which appear in this list.  Value t means, always use super- and
subscripts."
  :group 'x-symbol-tex
  :type '(repeat (symbol :tag "Face name"))) ; face would create faces... :-(

(defcustom x-symbol-tex-font-lock-limit-regexp "[\n^_]"
  "Regexp matching the limit for the end of super-/subscripts.
This regexp should match the end of line."
  :group 'x-symbol-tex
  :type 'regexp)

(defvar x-symbol-tex-font-lock-prefix-regexp
  "[^\000-\040\134\177-\237]\\([_^]\\)" ; control, spaces, backslash
  "Regexp matching the prefix of super-/subscripts.
The first regexp group should match the super-/subscript command.")

(defcustom x-symbol-tex-font-lock-simple-regexp
  (concat x-symbol-tex-font-lock-prefix-regexp
	  "\\([^ \t\n\f%\\{}^_$#&~]\\|\\\\[@A-Za-z]+\\)")
  "Regexp matching the prefix of simple super-/subscripts.
The first regexp group should match the super-/subscript command, the
second the simple super-/subscript."
  :group 'x-symbol-tex
  :type 'regexp)

(defcustom x-symbol-tex-font-lock-brace-regexp
  (concat x-symbol-tex-font-lock-prefix-regexp "\\({\\)")
  "Regexp matching the prefix of super-/subscripts in braces.
The first regexp group should match the super-/subscript command, the
second the opening brace."
  :group 'x-symbol-tex
  :type 'regexp)


;;;===========================================================================
;;;  Charsym Info
;;;===========================================================================

(defface x-symbol-tex-math-face
  '((((class color) (background light))
     (:foreground "purple3")))
  "*Face, normally used for tokens only allowed in TeX's math mode.
Used in `x-symbol-tex-class-face-alist'."
  :group 'x-symbol-tex
  :group 'x-symbol-info-general)

(defface x-symbol-tex-text-face
  '((((class color) (background light))
     (:foreground "Royalblue")))
  "*Face, normally used for tokens only allowed in TeX's text mode.
Used in `x-symbol-tex-class-face-alist'."
  :group 'x-symbol-tex
  :group 'x-symbol-info-general)

(defcustom x-symbol-tex-class-alist
  '((text)
    (math)
    (accent "accent" (x-symbol-info-face))
    (aletter "acc.letter" (x-symbol-info-face))
    (letter "letter" (x-symbol-info-face))
    (greek "greek" (x-symbol-info-face))
    (ordinary "ordinary" (x-symbol-info-face))
    (binop "binop" (x-symbol-info-face))
    (bigop "bigop" (x-symbol-info-face))
    (relation "relation" (x-symbol-info-face))
    (delim "delimiter" (x-symbol-info-face))
    (punct "punctuation" (x-symbol-info-face))
    (quote "quote" (x-symbol-info-face))
    (space "space" (x-symbol-info-face))
    (special "special" (x-symbol-info-face))
    (latexsym "latexsym.sty" (red x-symbol-info-face)) ; w/ latexsym or amssymb
    (amssymb "amssymb.sty" (red x-symbol-info-face))
    (T1 "T1 fontenc.sty" (red x-symbol-info-face))
    (correct-T1 "correct: T1 fontenc.sty" (x-symbol-info-face))
    (inputenc "inputenc.sty" (red x-symbol-info-face)) ; v0.9z (teTeX-0.9)
    (new-inputenc "new inputenc.sty" (red x-symbol-info-face)) ; v0.97
    (inputenc-unavail "inputenc.sty: unavailable" (red x-symbol-info-face))
;;;    (inputenc-incorrect "old inputenc: incorrect" . red) ; IMHO
    (user "user" (red x-symbol-info-face))
    (VALID "unknown TeX class" (red x-symbol-info-face))
    (INVALID "no TeX macro" (red x-symbol-info-face)))
  "Alist for TeX's token classes displayed by info in echo area.
See `x-symbol-language-access-alist' for details."
  :group 'x-symbol-tex
  :group 'x-symbol-info-strings
  :set 'x-symbol-set-cache-variable
  :type 'x-symbol-class-info)

(defcustom x-symbol-tex-class-face-alist
  '((math x-symbol-tex-math-face (x-symbol-tex-math-face))
    (text x-symbol-tex-text-face (x-symbol-tex-text-face)))
  "Alist for TeX's color scheme in TeX's grid and info.
See `x-symbol-language-access-alist' for details."
  :group 'x-symbol-tex
  :group 'x-symbol-input-init
  :group 'x-symbol-info-general
  :set 'x-symbol-set-cache-variable
  :type 'x-symbol-class-faces)


;;;===========================================================================
;;;  Misc
;;;===========================================================================

(defun x-symbol-tex-default-master-directory ()
  "Convert NAME to absolute file name, respecting `TeX-master'.
Variable `TeX-master' should be buffer-local and a string to be used.
Used as default value of `x-symbol-tex-master-directory'."
  (and (local-variable-p 'TeX-master (current-buffer))
       (stringp TeX-master)
       (file-name-directory (expand-file-name TeX-master))))

(defun x-symbol-tex-default-electric-ignore (context charsym)
  "Non nil, if CONTEXT should not be replaced by input method ELECTRIC.
Return non-nil if `x-symbol-tex-electric-ignore-regexp' matches CONTEXT
or if CHARSYM represents a TeX macro which can only be used in math mode
whereas point is in a text area or vice versa.  This function uses
package \"texmathp\" whose variables you might want to customize.  Used
as default value for `x-symbol-tex-electric-ignore'."
  (or (and x-symbol-tex-electric-ignore-regexp
	   (string-match x-symbol-tex-electric-ignore-regexp context))
      (condition-case nil
	  (let ((class (car (plist-get (get charsym 'x-symbol-classes) 'tex))))
	    (cond ((eq class 'math) (not (texmathp)))
		  ((eq class 'text) (texmathp))))
	(error nil))))


;;;===========================================================================
;;;  The tables
;;;===========================================================================

(defun x-symbol-tex-default-token-list (tokens)
  "Return list of tokens from token specification TOKENS.
If TOKENS is a string \"MACRO\", return (\"{MACRO}\" \"MACRO\"),
otherwise return TOKENS."
  (if (stringp tokens)
      (list (concat "{" tokens "}") tokens)
    tokens))

(defun x-symbol-tex-extended-token-list (tokens)
  "Return list of tokens from token specification TOKENS.
If TOKENS is a string \"MACRO\", return (\"{MACRO}\" \"MACRO\").  If
MACRO is an accent command with an argument, i.e., \"ACC LETTER\",
return (\"{ACC LETTER}\" \"ACC LETTER\" \"{ACC{LETTER}}\" \"ACC{LETTER}\".
Return TOKENS otherwise."
  (if (stringp tokens)
      (if (string-match
	   "\\`\\\\\\([`'^\"~=.]\\|[uvHtcdb]\\([ ]\\)\\)\
\\([A-Za-z]\\|\\\\[A-Za-z ]+\\)\\'"
	    tokens)
	  (let ((x-token (concat
			  (substring tokens 0
				     (or (match-beginning 2) (match-end 1)))
			  "{" (match-string 3 tokens) "}")))
	    (list (concat "{" tokens "}") tokens
		  (concat "{" x-token "}") x-token))
	(list (concat "{" tokens "}") tokens))
    tokens))

(defvar x-symbol-tex-latin1-table
  '((nobreakspace (space) . "\\nobreakspace")
    (exclamdown (text punct) . "\\textexclamdown")
    (cent (text inputenc-unavail) . "\\textcent")
    (sterling (ordinary) . "\\pounds")
    (currency (text inputenc) . "\\textcurrency")
    (yen (text inputenc-unavail) . "\\textyen")
    (brokenbar (text inputenc-unavail) . "\\textbrokenbar")
    (section (ordinary) . "\\S")
    (diaeresis (text accent) . "\\\"{}")
    (copyright (text ordinary) . "\\textcopyright")
    (ordfeminine (text ordinary new-inputenc) . "\\textordfeminine")
    (guillemotleft (text quote T1) . "\\guillemotleft")
    (notsign (math ordinary) "\\lnot" "\\neg")
    (hyphen (special) "\\-")
    (registered (text ordinary) . "\\textregistered")
    (macron (text accent) . "\\={}")
    (degree (text ordinary new-inputenc) . "\\textdegree")
    (plusminus (math binop) "\\pm")
    (twosuperior (math ordinary inputenc) "\\mathtwosuperior")
    (threesuperior (math ordinary inputenc) "\\maththreesuperior")
    (acute (text accent) . "\\'{}")
    (mu1 (math greek user) "\\mathmicro")
    (paragraph (ordinary) . "\\P")
    (periodcentered (text punct) . "\\textperiodcentered")
    (cedilla (text accent) . "\\c\\ ")
    (onesuperior (math ordinary inputenc) "\\mathonesuperior")
    (masculine (text ordinary new-inputenc) . "\\textordmasculine")
    (guillemotright (text quote T1) . "\\guillemotright")
    (onequarter (text ordinary inputenc) . "\\textonequarter")
    (onehalf (text ordinary inputenc) . "\\textonehalf")
    (threequarters (text ordinary inputenc) . "\\textthreequarters")
    (questiondown (text punct) . "\\textquestiondown")
    (Agrave (text aletter) . "\\`A")
    (Aacute (text aletter) . "\\'A")
    (Acircumflex (text aletter) . "\\^A")
    (Atilde (text aletter) . "\\~A")
    (Adiaeresis (text aletter) . "\\\"A")
    (Aring (text aletter) . "\\AA")
    (AE (text letter) . "\\AE")
    (Ccedilla (text aletter) . "\\c C")
    (Egrave (text aletter) . "\\`E")
    (Eacute (text aletter) . "\\'E")
    (Ecircumflex (text aletter) . "\\^E")
    (Ediaeresis (text aletter) . "\\\"E")
    (Igrave (text aletter) . "\\`I")
    (Iacute (text aletter) . "\\'I")
    (Icircumflex (text aletter) . "\\^I")
    (Idiaeresis (text aletter) . "\\\"I")
    (ETH (text letter T1) . "\\DH")
    (Ntilde (text aletter) . "\\~N")
    (Ograve (text aletter) . "\\`O")
    (Oacute (text aletter) . "\\'O")
    (Ocircumflex (text aletter) . "\\^O")
    (Otilde (text aletter) . "\\~O")
    (Odiaeresis (text aletter) . "\\\"O")
    (multiply (math binop) "\\times")
    (Ooblique (text letter) . "\\O")
    (Ugrave (text aletter) . "\\`U")
    (Uacute (text aletter) . "\\'U")
    (Ucircumflex (text aletter) . "\\^U")
    (Udiaeresis (text aletter) . "\\\"U")
    (Yacute (text aletter) . "\\'Y")
    (THORN (text letter T1) . "\\TH")
    (ssharp (text letter) . "\\ss")
    (agrave (text aletter) . "\\`a")
    (aacute (text aletter) . "\\'a")
    (acircumflex (text aletter) . "\\^a")
    (atilde (text aletter) . "\\~a")
    (adiaeresis (text aletter) . "\\\"a")
    (aring (text aletter) . "\\aa")
    (ae (text letter) . "\\ae")
    (ccedilla (text aletter) . "\\c c")
    (egrave (text aletter) . "\\`e")
    (eacute (text aletter) . "\\'e")
    (ecircumflex (text aletter) . "\\^e")
    (ediaeresis (text aletter) . "\\\"e")
    (igrave (text aletter) . "\\`\\i")
    (iacute (text aletter) . "\\'\\i")
    (icircumflex (text aletter) . "\\^\\i")
    (idiaeresis (text aletter) . "\\\"\\i")
    (eth (text letter T1) . "\\dh")
    (ntilde (text aletter) . "\\~n")
    (ograve (text aletter) . "\\`o")
    (oacute (text aletter) . "\\'o")
    (ocircumflex (text aletter) . "\\^o")
    (otilde (text aletter) . "\\~o")
    (odiaeresis (text aletter) . "\\\"o")
    (division (math binop) "\\div")
    (oslash (text letter) . "\\o")
    (ugrave (text aletter) . "\\`u")
    (uacute (text aletter) . "\\'u")
    (ucircumflex (text aletter) . "\\^u")
    (udiaeresis (text aletter) . "\\\"u")
    (yacute (text aletter) . "\\'y")
    (thorn (text letter T1) . "\\th")
    (ydiaeresis (text aletter) . "\\\"y"))
  "Table defining TeX macros, see `x-symbol-tex-table'.")

(defvar x-symbol-tex-latinN-table
  '((Aogonek (text aletter T1) . "\\k A")
    (breve (text accent) . "\\u{}")
    (Lslash (text letter) . "\\L")
    (Lcaron (text aletter correct-T1) . "\\v L")
    (Sacute (text aletter) . "\\'S")
    (Scaron (text aletter) . "\\v S")
    (Scedilla (text aletter) . "\\c S")
    (Tcaron (text aletter) . "\\v T")
    (Zacute (text aletter) . "\\'Z")
    (Zcaron (text aletter) . "\\v Z")
    (Zdotaccent (text aletter) . "\\.Z")
    (aogonek (text aletter T1) . "\\k a")
    (ogonek (text accent T1) . "\\k\\ ")
    (lslash (text letter) . "\\l")
    (lcaron (text aletter correct-T1) . "\\v l")
    (sacute (text aletter) . "\\'s")
    (caron (text accent) . "\\v{}")
    (scaron (text aletter) . "\\v s")
    (scedilla (text aletter) . "\\c s")
    (tcaron (text aletter correct-T1) . "\\v t")
    (zacute (text aletter) . "\\'z")
    (hungarumlaut (text accent) . "\\H{}")
    (zcaron (text aletter) . "\\v z")
    (zdotaccent (text aletter) . "\\.z")
    (Racute (text aletter) . "\\'R")
    (Abreve (text aletter) . "\\u A")
    (Lacute (text aletter) . "\\'L")
    (Cacute (text aletter) . "\\'C")
    (Ccaron (text aletter) . "\\v C")
    (Eogonek (text aletter T1) . "\\k E")
    (Ecaron (text aletter) . "\\v E")
    (Dcaron (text aletter) . "\\v D")
    (Dbar (text letter T1) . "\\DJ")	; new-inputenc
    (Nacute (text aletter) . "\\'N")
    (Ncaron (text aletter) . "\\v N")
    (Ohungarumlaut (text aletter) . "\\H O")
    (Rcaron (text aletter) . "\\v R")
    (Uring (text aletter) . "\\r U")
    (Uhungarumlaut (text aletter) . "\\H U")
    (Tcedilla (text aletter) . "\\c T")
    (racute (text aletter) . "\\'r")
    (abreve (text aletter) . "\\u a")
    (lacute (text aletter) . "\\'l")
    (cacute (text aletter) . "\\'c")
    (ccaron (text aletter) . "\\v c")
    (eogonek (text aletter T1) . "\\k e")
    (ecaron (text aletter) . "\\v e")
    (dcaron (text aletter correct-T1) . "\\v d")
    (dbar (text letter T1) . "\\dj")	; new-inputenc
    (nacute (text aletter) . "\\'n")
    (ncaron (text aletter) . "\\v n")
    (ohungarumlaut (text aletter) . "\\H o")
    (rcaron (text aletter) . "\\v r")
    (uring (text aletter) . "\\r u")
    (uhungarumlaut (text aletter) . "\\H u")
    (tcedilla (text aletter) . "\\c t")
    (dotaccent (text accent) . "\\.{}")
    (Hbar (text letter inputenc-unavail) . "\\textmalteseH")
    (Hcircumflex (text aletter) . "\\^H")
    (Idotaccent (text aletter) . "\\.I")
    (Gbreve (text aletter) . "\\u G")
    (Jcircumflex (text aletter) . "\\^J")
    (hbar (text letter inputenc-unavail) . "\\textmalteseh")
    (hcircumflex (text aletter) . "\\^h")
    (dotlessi (text letter) . "\\i")
    (gbreve (text aletter) . "\\u g")
    (jcircumflex (text aletter) . "\\^j")
    (Cdotaccent (text aletter) . "\\.C")
    (Ccircumflex (text aletter) . "\\^C")
    (Gdotaccent (text aletter) . "\\.G")
    (Gcircumflex (text aletter) . "\\^G")
    (Ubreve (text aletter) . "\\u U")
    (Scircumflex (text aletter) . "\\^S")
    (cdotaccent (text aletter) . "\\.c")
    (ccircumflex (text aletter) . "\\^c")
    (gdotaccent (text aletter) . "\\.g")
    (gcircumflex (text aletter) . "\\^g")
    (ubreve (text aletter) . "\\u u")
    (scircumflex (text aletter) . "\\^s"))
  "Table defining TeX macros, see `x-symbol-tex-table'.")

;; Characters w/ NEW weren't defined before, w/ (NEW) were defined at other
;; positions.  If we get problems in the nomule version (e.g., w/ font-lock),
;; we could be forced to move these characters to the xsymb1 font.
(defvar x-symbol-tex-xsymb0-table
  ;; With elems (SYMBOL (TEX-CLASS ...) TEX-MACRO ...)
  '((numbersign1 (ordinary) "\\#")	; NEW
    ;;(existential)
    (suchthat (math relation) "\\ni" "\\owns")
    (asterisk1 (math binop) "\\ast")		; NEW
    ;;(comma1 (mark) "\\quotesinglbase")	; not in {}! (spacing)
    (period1 (math punct) "\\ldotp")		; (NEW)
    (colon1 (math punct) "\\colon")		; (NEW)
    (congruent (math relation) "\\cong")
    (Delta (math greek) "\\Delta")
    (Phi (math greek) "\\Phi")
    (Gamma (math greek) "\\Gamma")
    (theta1 (math greek) "\\vartheta")
    (Lambda (math greek) "\\Lambda")
    (Pi (math greek) "\\Pi")
    (Theta (math greek) "\\Theta")
    (Sigma (math greek) "\\Sigma")
    (sigma1 (math greek) "\\varsigma")
    (Omega (math greek) "\\Omega")
    (Xi (math greek) "\\Xi")
    (Psi (math greek) "\\Psi")
    ;;(therefore (math relation) "\\therefore")
    (perpendicular (math ordinary) "\\bot")	; (NEW)
    (underscore1 (ordinary) "\\_")	; NEW
    ;;(radicalex)
    (alpha (math greek) "\\alpha")
    (beta (math greek) "\\beta")
    (chi (math greek) "\\chi")
    (delta (math greek) "\\delta")
    (epsilon (math greek) "\\epsilon")
    (phi (math greek) "\\phi")
    (gamma (math greek) "\\gamma")
    (eta (math greek) "\\eta")
    (iota (math greek) "\\iota")
    (phi1 (math greek) "\\varphi")
    (kappa (math greek) "\\kappa")
    (lambda (math greek) "\\lambda")
    (mu (math greek) "\\mu")
    (nu (math greek) "\\nu")
    (pi (math greek) "\\pi")
    (theta (math greek) "\\theta")
    (rho (math greek) "\\rho")
    (sigma (math greek) "\\sigma")
    (tau (math greek) "\\tau")
    (upsilon (math greek) "\\upsilon")
    (omega1 (math greek) "\\varpi")
    (omega (math greek) "\\omega")
    (xi (math greek) "\\xi")
    (psi (math greek) "\\psi")
    (zeta (math greek) "\\zeta")
    (bar1 (math relation) "\\mid")		; (NEW)
    (similar (math relation) "\\sim")
    (Upsilon1 (math greek) "\\Upsilon")
    (minute (math ordinary) "\\prime")
    (lessequal (math relation) "\\leq" "\\le")
    (fraction (ordinary) "\\slash")
    (infinity (math ordinary) "\\infty")
    (florin (text user) . "\\textflorin")	; NEW
    (club (math ordinary) "\\clubsuit")
    (diamond (math ordinary) "\\diamondsuit")
    (heart (math ordinary) "\\heartsuit")
    (spade (math ordinary) "\\spadesuit")
    (arrowboth (math relation) "\\leftrightarrow")
    (arrowleft (math relation) "\\gets" "\\leftarrow")
    (arrowup (math relation delim) "\\uparrow")
    (arrowright (math relation) "\\to" "\\rightarrow")
    (arrowdown (math relation delim) "\\downarrow")
    (ring (text accent) . "\\r{}")	; NEW
    ;;(second)
    (greaterequal (math relation) "\\geq" "\\ge")
    (proportional (math relation) "\\propto")
    (partialdiff (math ordinary) "\\partial")
    (bullet (math binop) "\\bullet")
    (notequal (math relation) "\\neq" "\\ne")
    (equivalence (math relation) "\\equiv")
    (approxequal (math relation) "\\approx")
    (ellipsis (ordinary) "\\ldots")
    ;;(carriagereturn)
    (aleph (math letter) "\\aleph")
    (Ifraktur (math letter) "\\Im")
    (Rfraktur (math letter) "\\Re")
    (weierstrass (math letter) "\\wp")
    (circlemultiply (math binop) "\\otimes")
    (circleplus (math binop) "\\oplus")
    (emptyset (math ordinary) "\\emptyset")
    (intersection (math binop) "\\cap")
    (union (math binop) "\\cup")
    (propersuperset (math relation) "\\supset")
    (reflexsuperset (math relation) "\\supseteq")
    (notsubset (math relation user) "\\nsubset")
    (propersubset (math relation) "\\subset")
    (reflexsubset (math relation) "\\subseteq")
    (element (math relation) "\\in")
    (notelement (math relation) "\\notin")
    (angle (ordinary) "\\angle")
    (gradient (math ordinary) "\\nabla")
    (product (math bigop) "\\prod")
    (radical (math ordinary) "\\surd")
    (periodcentered1 (math binop) "\\cdot")	; (NEW)
    (logicaland (math binop) "\\land" "\\wedge")
    (logicalor (math binop) "\\lor" "\\vee")
    (arrowdblboth (math relation) "\\Leftrightarrow" "\\lequiv")
    (arrowdblleft (math relation) "\\Leftarrow")
    (arrowdblup (math relation delim) "\\Uparrow")
    (arrowdblright (math relation) "\\Rightarrow")
    (arrowdbldown (math relation delim) "\\Downarrow")
    (lozenge (math ordinary amssymb) "\\lozenge")
    (angleleft (math delim) "\\langle")	; (NEW)
    (trademark (text ordinary) . "\\texttrademark")
    (summation (math bigop) "\\sum")
    (angleright (math delim) "\\rangle") ; (NEW)
    (integral (math bigop) "\\int"))
  "Table defining TeX macros, see `x-symbol-tex-table'.")

(defvar x-symbol-tex-xsymb1-table
  ;; With elems (SYMBOL (TEX-CLASS ...) TEX-MACRO ...)
  '((verticaldots (ordinary) "\\vdots")
    (backslash1 (text ordinary) . "\\textbackslash")
    (dagger (ordinary) . "\\dag")
    (percent2 (ordinary) "\\%")		; NEW
    (guilsinglright (text quote T1) . "\\guilsinglright")
    (NG (text letter T1) . "\\NG")
    (OE (text letter) . "\\OE")
    (dotlessj (text letter) . "\\j")
    (ng (text letter T1) . "\\ng")
    (oe (text letter) . "\\oe")
    (sharp (math ordinary) "\\sharp")
    (ceilingleft (math delim) "\\lceil")
    (ceilingright (math delim) "\\rceil")
    (star (math binop) "\\star")
    (lozenge1 (math ordinary latexsym) "\\Diamond")
    (braceleft2 (delim) "\\{" "\\lbrace")
    (circleslash (math binop) "\\oslash")
    (braceright2 (delim) "\\}" "\\rbrace")
    (triangle1 (math binop) "\\bigtriangleup")
    (smltriangleright (math binop) "\\triangleright")
    (triangleleft (math binop latexsym) "\\lhd")
    (triangle (math ordinary) "\\triangle")
    (triangleright (math binop latexsym) "\\rhd")
    (trianglelefteq (math binop latexsym) "\\unlhd")
    (trianglerighteq (math binop latexsym) "\\unrhd")
    (periodcentered2 (math punct) "\\cdotp")
    (dotequal (math relation) "\\doteq")
    (wrong (math binop) "\\wr")
    (natural (math ordinary) "\\natural")
    (flat (math ordinary) "\\flat")
    (epsilon1 (math greek) "\\varepsilon")
    (hbarmath (math letter) "\\hbar")
    (imath (math letter) "\\imath")
    (kappa1 (math greek amssymb) "\\varkappa")
    (jmath (math letter) "\\jmath")
    (ell (math letter) "\\ell")
    (amalg (math binop) "\\amalg")
    (rho1 (math greek) "\\varrho")
    (top (math ordinary) "\\top")
    (Mho (math greek latexsym) "\\mho")
    (floorleft (math delim) "\\lfloor")
    (floorright (math delim) "\\rfloor")
    (perpendicular1 (math relation) "\\perp")
    (box (math ordinary latexsym) "\\Box")
    (asciicircum1 (text ordinary) . "\\textasciicircum")
    (asciitilde1 (text ordinary) . "\\textasciitilde")
    (leadsto (math relation latexsym) "\\leadsto")
    (longarrowleft (math relation) "\\longleftarrow")
    (arrowupdown (math relation delim) "\\updownarrow")
    (longarrowright (math relation) "\\longrightarrow")
    (longmapsto (math relation) "\\longmapsto")
    (longarrowdblboth (math relation) "\\Longleftrightarrow")
    (longarrowdblleft (math relation) "\\Longleftarrow")
    (arrowdblupdown (math relation delim) "\\Updownarrow")
    (longarrowdblright (math relation) "\\Longrightarrow")
    (mapsto (math relation) "\\mapsto")
    (iff (math relation) "\\iff")
    (hookleftarrow (math relation) "\\hookleftarrow")
    (hookrightarrow (math relation) "\\hookrightarrow")
    (arrownortheast (math relation) "\\nearrow")
    (arrowsoutheast (math relation) "\\searrow")
    (arrownorthwest (math relation) "\\nwarrow")
    (arrowsouthwest (math relation) "\\swarrow")
    (rightleftharpoons (math relation) "\\rightleftharpoons")
    (leftharpoondown (math relation) "\\leftharpoondown")
    (rightharpoondown (math relation) "\\rightharpoondown")
    (leftharpoonup (math relation) "\\leftharpoonup")
    (rightharpoonup (math relation) "\\rightharpoonup")
    (bardbl (math ordinary delim) "\\|" "\\Vert")
    (bardbl1 (math relation) "\\parallel")
    (backslash2 (math ordinary delim) "\\backslash")
    (backslash3 (math binop) "\\setminus")
    (diagonaldots (math ordinary) "\\ddots")
    (simequal (math relation) "\\simeq")
    (digamma (math ordinary amssymb) "\\digamma")
    (asym (math relation) "\\asymp")
    (minusplus (math binop) "\\mp")
    (bowtie (math relation) "\\bowtie")
    (centraldots (math ordinary) "\\cdots")
    (visiblespace (text ordinary) . "\\textvisiblespace")
    (dagger1 (math binop) "\\dagger")
    (circledot (math binop) "\\odot")
    (propersqsuperset (math relation latexsym) "\\sqsupset")
    (reflexsqsuperset (math relation) "\\sqsupseteq")
    (gradient1 (math binop) "\\bigtriangledown")
    (propersqsubset (math relation latexsym) "\\sqsubset")
    (reflexsqsubset (math relation) "\\sqsubseteq")
    (smllozenge (math binop) "\\diamond")
    (lessless (math relation) "\\ll")
    (greatergreater (math relation) "\\gg")
    (unionplus (math binop) "\\uplus")
    (sqintersection (math binop) "\\sqcap")
    (squnion (math binop) "\\sqcup")
    (frown (math relation) "\\frown")
    (smile (math relation) "\\smile")
    (reflexprec (math relation) "\\preceq")
    (reflexsucc (math relation) "\\succeq")
    (properprec (math relation) "\\prec")
    (propersucc (math relation) "\\succ")
    (bardash (math relation) "\\vdash")
    (dashbar (math relation) "\\dashv")
    (bardashdbl (math relation) "\\models")
    (smlintegral (math ordinary) "\\smallint")
    (circleintegral (math bigop) "\\oint")
    (coproduct (math bigop) "\\coprod")
    (bigcircledot (math bigop) "\\bigodot")
    (bigcirclemultiply (math bigop) "\\bigotimes")
    (bigcircleplus (math bigop) "\\bigoplus")
    (biglogicaland (math bigop) "\\bigwedge")
    (biglogicalor (math bigop) "\\bigvee")
    (bigintersection (math bigop) "\\bigcap")
    (bigunion (math bigop) "\\bigcup")
    (bigunionplus (math bigop) "\\biguplus")
    (bigsqunion (math bigop) "\\bigsqcup")
    (bigcircle (math binop) "\\bigcirc")
    (guilsinglleft (text quote T1) . "\\guilsinglleft")
    (circleminus (math binop) "\\ominus")
    (smltriangleleft (math binop) "\\triangleleft")
    (existential1 (math ordinary) "\\exists")
    (daggerdbl1 (math binop) "\\ddagger")
    (daggerdbl (ordinary) . "\\ddag")
    (bigbowtie (math relation latexsym) "\\Join")
    (circ (math binop) "\\circ")
    (grave (text accent) . "\\`{}")	; NEW
    (circumflex (text accent) . "\\^{}") ; NEW
    (tilde (text accent) . "\\~{}")	; NEW
    (longarrowboth (math relation) "\\longleftrightarrow")
    (endash (text ordinary) . "\\textendash") ; NEW
    (emdash (text ordinary) . "\\textemdash")
    (Ydiaeresis (text aletter) . "\\\"Y") ; NEW
    (ampersand2 (ordinary) "\\&")	; NEW
    (universal1 (math ordinary) "\\forall")
    (booleans (math letter user) "\\setB")
    (complexnums (math letter user) "\\setC")
    (natnums (math letter user) "\\setN")
    (rationalnums (math letter user) "\\setQ")
    (realnums (math letter user) "\\setR")
    (integers (math letter user) "\\setZ")
    (lesssim (math relation amssymb) "\\lesssim")
    (greatersim (math relation amssymb) "\\gtrsim")
    (lessapprox (math relation amssymb) "\\lessapprox")
    (greaterapprox (math relation amssymb) "\\gtrapprox")
    (definedas (math relation amssymb) "\\triangleq")
    (circleminus1 (math binop amssymb) "\\circleddash")
    (circleasterisk (math binop amssymb) "\\circledast")
    (circlecirc (math binop amssymb) "\\circledcirc")
    (dollar1 (ordinary) "\\$")
    (therefore1 (math relation amssymb) "\\therefore")
    (coloncolon (math relation user) "\\coloncolon")
    (semanticsleft (math delim user) "\\lsemantics")
    (semanticsright (math delim user) "\\rsemantics")
    ;;(quotedblbase (mark T1) "\\quotedblbase") ; not in {}! (spacing)
    ;;(quotedblleft (mark) . "\\textquotedblleft") ; not in {}! (spacing)
    ;;(quotedblright (mark) . "\\textquotedblright") ; not in {}! (spacing)
    ;;(perthousand)
    )
  "Table defining TeX macros, see `x-symbol-tex-table'.")

(defvar x-symbol-tex-table
  (append x-symbol-tex-user-table
	  '(nil)
	  x-symbol-tex-latin1-table
	  x-symbol-tex-latinN-table
	  x-symbol-tex-xsymb0-table
	  x-symbol-tex-xsymb1-table)
  "Complete table defining TeX macros.
Use `x-symbol-tex-user-table' to define/shadow private TeX macros.
See `x-symbol-init-language' and `x-symbol-tex-token-list'.")


;;;===========================================================================
;;;  Internal
;;;===========================================================================

(defvar x-symbol-tex-menu-alist nil
  "Internal.  Alist used for TeX specific menu.")
(defvar x-symbol-tex-grid-alist nil
  "Internal.  Alist used for TeX specific grid.")

(defvar x-symbol-tex-decode-atree nil
  "Internal.  Atree used by `x-symbol-token-input'.")
(defvar x-symbol-tex-decode-alist nil
  "Internal.  Alist used for decoding of TeX macros.")
(defvar x-symbol-tex-encode-alist nil
  "Internal.  Alist used for encoding to TeX macros.")

(defvar x-symbol-tex-nomule-decode-exec nil
  "Internal.  File name of TeX decode executable.")
(defvar x-symbol-tex-nomule-encode-exec nil
  "Internal.  File name of TeX encode executable.")


;;;===========================================================================
;;;  Super- and Subscripts
;;;===========================================================================

(defun x-symbol-tex-match-simple-subscript (limit)
  ;; checkdoc-params: (limit)
  "Match and skip over simple super- and subscripts.
Return nil if `x-symbol-mode' or `x-symbol-subscripts' is nil.  Uses
`x-symbol-tex-font-lock-simple-regexp'.  Do not match super-/subscripts
with a face not mentioned in `x-symbol-tex-font-lock-allowed-faces'."
  (and x-symbol-mode x-symbol-subscripts
       (let (pos)
	 (or (bolp) (backward-char))
	 (while (and (setq pos (re-search-forward
				x-symbol-tex-font-lock-simple-regexp limit t))
		     (null (eq x-symbol-tex-font-lock-allowed-faces t))
		     (let ((faces (plist-get (text-properties-at
					      (match-beginning 1))
					     'face)))
		       (when faces
			 (or (listp faces) (setq faces (list faces)))
			 (catch 'x-symbol-tex-match-simple-subscript
			   (while faces
			     (or (memq (pop faces)
				       x-symbol-tex-font-lock-allowed-faces)
				 (progn
				   (setq pos nil)
				   (throw 'x-symbol-tex-match-simple-subscript
					  t))))
			   nil)))))
	 pos)))

(defun x-symbol-tex-match-subscript (limit)
  ;; checkdoc-params: (limit)
  "Match and skip over simple super- and subscripts.
Return nil if `x-symbol-mode' or `x-symbol-subscripts' is nil.  Uses
`x-symbol-tex-font-lock-brace-regexp'."
  (and x-symbol-mode x-symbol-subscripts
       (let (pos data)
	 (or (bolp) (backward-char))
	 (while (and (setq pos (re-search-forward
				x-symbol-tex-font-lock-brace-regexp limit t))
		     (not
		      (and (setq data (match-data)
				 pos
				 (save-restriction
				   (narrow-to-region
				    (point)
				    (save-excursion
				      (re-search-forward
				       x-symbol-tex-font-lock-limit-regexp
				       limit 'limit)
				      (point)))
				   (scan-lists (point) 1 1 nil t)))
			   (eq (char-before pos) ?})))))
	 (when pos
	   (setcar (cdr data) pos)
	   (setcar (last data) pos)
	   (store-match-data data)
	   (goto-char pos)
	   t))))


;;;===========================================================================
;;;  Functions for AucTeX
;;;===========================================================================

(defun x-symbol-tex-math-insert (string)
  "Insert the character for \\STRING.
Used as value for `LaTeX-math-insert-function'."
  (let ((cstring (get (cdr (assoc (concat "\\" string)
				  x-symbol-tex-decode-alist))
		      'x-symbol-cstring)))
    (if cstring
	(insert cstring)
      (TeX-insert-macro string))))

(defun x-symbol-tex-init-auctex-region ()
  "Inherit X-Symbol specific variables from master buffer.
Used as function in `TeX-region-hook'."
  (and (boundp 'master-buffer)
       (buffer-live-p master-buffer)
       (let (mode-on coding 8bits language)
	 (save-excursion
	   (set-buffer master-buffer)
	   (setq mode-on  x-symbol-mode
		 coding   x-symbol-coding
		 8bits	  x-symbol-8bits
		 language x-symbol-language))
	 (setq x-symbol-mode     mode-on
	       x-symbol-coding   coding
	       x-symbol-8bits	 8bits
	       x-symbol-language language))))

;;; Local IspellPersDict: .ispell_xsymb
;;; x-symbol-tex.el ends here
