/*
   Editor functions
   --------------------------------------------------------------------
   VCHE - Virtual Console Hex Editor

   Copyright (C) 1998, 1999 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

#if defined(VCHE_VC) || defined(VCHE_RAW)
    #define EDITOR_COLOR_BLUE     VC_COLOR_BACKBLUE
    #define EDITOR_COLOR_BG_LGRAY VC_COLOR_BG_LIGHTGRAY
    #define EDITOR_COLOR_WHITE    VC_COLOR_WHITE
#elif defined(VCHE_NC)
    #define EDITOR_COLOR_BLUE     BUF_BBLUE
    #define EDITOR_COLOR_BG_LGRAY BUF_BLGRAY
    #define EDITOR_COLOR_WHITE    BUF_WHITE
#endif

extern char hex_trans_byte1[];
extern char hex_trans_byte2[];

extern char *hbuffer;

int edit_mask;        /* Edit mask for XOR/etc F* keys                  */
static char *cbuffer; /* Color buffer                                   */
static int tab_flag;  /* Switch from hex to ascii editing and viceversa
                         (i.e. left side = hex, right side = ascii edit */
int ex, ey;           /* X and Y coordinates                            */
static int hx, hy;    /* Hex side x, y                                  */
static int ax, ay;    /* Ascii side x, y                                */
static int f_digit;   /* First or second digit (for hex editing)        */
static long posxyh;   /* xy position on /dev/vcsa* (hex side)           */
static long posxya;   /* xy position on /dev/vcsa* (ascii side)         */

static void edit_calculate_xy_position( void);
static void edit_change_color_at_cursor( int what_attr);
static void edit_insert( char c);
static void edit_insert_in_ascii( char c);
static void edit_insert_in_hex( char c);
#define edit_set_color_buffer() edit_change_color_at_cursor( cbuffer[ ey * bytes_per_row + ex])
#define edit_set_color_in_buffer( attr) cbuffer[ ey * bytes_per_row + ex] = attr
static void edit_show_edited_data( void);
static void edit_set_mask( void);
static void edit_bitops( int bitop);
#define edit_xor() edit_bitops( EDIT_BITOP_XOR)
#define edit_or()  edit_bitops( EDIT_BITOP_OR )
#define edit_and() edit_bitops( EDIT_BITOP_AND)
#define edit_neg() edit_bitops( EDIT_BITOP_NEG)
#define edit_add() edit_bitops( EDIT_BITOP_ADD)
#define edit_sub() edit_bitops( EDIT_BITOP_SUB)
static int edit_ask_save( int pos);
static void edit_save_data( void);

void edit( void)
{
 int c;

 cbuffer = ( char *) allocate_mem( ( vc_lines - 2 ) * vc_cols);

 memset( cbuffer, EDITOR_COLOR_BLUE, ( vc_lines - 2 ) * vc_cols);

 ll_edit_head = NULL;

 ex = ey  = 0;
 f_digit  = 0;
 tab_flag = 0;

 hex_showfile();
 lib_flush();

 draw_fkeys( fkeys[ 2]);

 term_unhide_cursor();

 while( 1) {
        if( ex <  0) {
            ex = bytes_per_row - 1;
            ey--;
        }

        if( ex > bytes_per_row - 1) {
            ex = 0;
            ey++;
        }

        edit_change_color_at_cursor( EDITOR_COLOR_WHITE);

        c = key_get();

        switch( c) {
                case '\t'        : tab_flag = tab_flag ? 0 : 1;
                                   continue;
                case VC_KEY_F2   : if( edit_ask_save( EDIT_ASK_SAVE) == -1) {
                                       edit_set_color_buffer();
                                       term_hide_cursor();
                                       free( cbuffer);
                                       return;
                                   }
                                   continue;
                case VC_KEY_F3   : edit_set_mask();
                                   continue;
                case VC_KEY_F4   : edit_xor(); 
                                   continue;
                case VC_KEY_F5   : edit_or();
                                   continue;
                case VC_KEY_F6   : edit_and();
                                   continue;
                case VC_KEY_F7   : edit_neg();
                                   continue;
                case VC_KEY_F8   : edit_add();
                                   continue;
                case VC_KEY_F9   : edit_sub();
                                   continue;
                case VC_KEY_F10  : if( edit_ask_save( EDIT_ASK_QUIT) == -1) {
                                       edit_set_color_buffer();
                                       term_hide_cursor();
                                       free( cbuffer);
                                       return;
                                   }
                                   continue;
                case VC_KEY_UP   : edit_set_color_buffer();
                                   ey--;
                                   if( ey < 0) {
                                       ey = 0;
                                       file_prev_line();
                                       edit_show_edited_data();
                                   }
                                   continue;
                case VC_KEY_DOWN : edit_set_color_buffer();
                                   ey++;
                                   if( ey > vc_lines - 3) {
                                       ey = vc_lines - 3;
                                       file_next_line();
                                       edit_show_edited_data();
                                   }
                                   continue;
                case VC_KEY_RIGHT: if( !tab_flag) {
                                       if( f_digit) {
                                           edit_set_color_buffer();
                                           ex++;
                                       }
                                       f_digit = f_digit ? 0 : 1;
                                   } else {
                                       edit_set_color_buffer();
                                       ex++;
                                   }
                                   if( ex > bytes_per_row - 1 && ey == vc_lines - 3) {
                                       ex = 0;
                                       file_next_line();
                                       edit_show_edited_data();
                                       continue;
                                   }
                                   continue;
                case VC_KEY_LEFT : if( !tab_flag) {
                                       if( !f_digit) {
                                           edit_set_color_buffer();
                                           ex--;
                                       }
                                       f_digit = f_digit ? 0 : 1;
                                   } else {
                                       edit_set_color_buffer();
                                       ex--;
                                   }
                                   if( ex < 0 && ey == 0) {
                                       ex = bytes_per_row - 1;
                                       file_prev_line();
                                       edit_show_edited_data();
                                       continue;
                                   }
                                   continue;
                case VC_KEY_HOME : file_go_home();
                                   edit_show_edited_data();
                                   ex = ey = 0;
                                   continue;
                case VC_KEY_END  : file_go_end();
                                   edit_show_edited_data();
                                   ex = ey = 0;
                                   continue;
                case VC_KEY_PGUP : file_prev_block();
                                   edit_show_edited_data();
                                   ex = ey = 0;
                                   continue;
                case VC_KEY_PGDN : file_next_block();
                                   edit_show_edited_data();
                                   ex = ey = 0;
                                   continue;
        } /* end switch( c) */

        edit_insert( ( char) c);

 } /* end while( 1) */

}

/*
 * It takes our current ex and ey position and do some math
 * to fill other variables used to edit in the hex or the
 * ascii segment.
 */
static void edit_calculate_xy_position( void)
{
 /* The hex numbers begin at ey + 1 */
 hy = ey + 1;
 hx = 3.25 * ex + 12;

 /*
  *  Q: Where the f*** you get "hx = 3.25 * ex + 12" ???????????????
  *  A: Simple!
  *
  *  01234567890123456789012345678901234567890123456789012345678901
  *
  *  0000000000  68 65 6C 6C  6F 20 77 6F  72 6C 64 20  21 21 21 21  hello world
  *  0000000010  21 21 21 21  21 21 0A 0A  0A 0A       |             !!!!!!
  *              |              |__|__|   |____________|
  *              |____the hex      |                   |
  *                   numbers      |                   |
  *                   begin at     |                   |
  *                   ex = 12      |               at ex / 4 we
  *                                |               add one space
  *                             they are
  *                             separated
  *                             by ex * 2
  *
  *  Then we have this formula:   hx = ex + 12 + ( ex * 2) + ( ex / 4);
  *  We could render it this way: hx = ex + (2 * ex) + (1/4 * ex) + 12
  *  We simplify:                 hx = (1 + 2 + 1/4) * ex + 12
  *  Solution:                    hx = 3.25 * ex + 12
  *
  */

 ay = hy;
 ax = ex + vc_cols - bytes_per_row;

 posxyh = 4 + 2 *( hy * vc_cols + hx);
 posxya = 4 + 2 *( ay * vc_cols + ax);
}

/*
 * Change the color at current position and set the cursor there.
 */
static void edit_change_color_at_cursor( int what_attr)
{
#if defined(VCHE_VC)

 char buf[ 2];

 edit_calculate_xy_position();

 buf[ 0] = what_attr;

 lseek( vcsafd, posxyh + 1, SEEK_SET);
 write( vcsafd, &buf, 1);

 lseek( vcsafd, posxyh + 3, SEEK_SET);
 write( vcsafd, &buf, 1);

 lseek( vcsafd, posxya + 1, SEEK_SET);
 write( vcsafd, &buf, 1);

#elif defined(VCHE_RAW)

 char buffer[ 16];
 int co2 = vc_cols * 2;
 int posh, posa;

 edit_calculate_xy_position();

 posh = posxyh - (4 + co2) + co2;
 posa = posxya - (4 + co2) + co2;

 term_color( what_attr);

 buffer[ 0] = restore_buffer[ posh    ];
 buffer[ 1] = restore_buffer[ posh + 2];

 raw_puts( hx, hy, 2, buffer);

 buffer[ 0] = restore_buffer[ posa    ];

 raw_putc( ax, ay, buffer[ 0]);

#elif defined(VCHE_NC)

 attr_t eattr = 0;
 short ecolor = 0;
 chtype c = 0;

 edit_calculate_xy_position();

 translate_buffer_colors( what_attr, &eattr, &ecolor);

 term_goto( hx, hy);
 c = inch();
 c &= A_CHARTEXT;
 attrset( COLOR_PAIR( ecolor) | eattr);
 addch( c);

 term_goto( hx + 1, hy);
 c = inch();
 c &= A_CHARTEXT;
 attrset( COLOR_PAIR( ecolor) | eattr);
 addch( c);

 term_goto( ax, ay);
 c = inch();
 c &= A_CHARTEXT;
 attrset( COLOR_PAIR( ecolor) | eattr);
 addch( c);

 refresh();
#endif

 if( tab_flag)
     term_goto( ax, ay);
 else {
     if( f_digit) 
         term_goto( hx + 1, hy);
     else
         term_goto( hx, hy);
 }
}

/*
 * Insert the byte with his color attribute in obuffer
 * (which is the buffer used to flush the screen data to stdout)
 */
void edit_insert_in_obuffer( char byte, int what_attr)
{
#if defined(VCHE_RAW)
 int co2 = vc_cols * 2;
#endif
 int posh, posa;

 edit_calculate_xy_position();

 posh = posxyh - (4 + 2 * vc_cols);
 posa = posxya - (4 + 2 * vc_cols);

#if defined(VCHE_VC)
 obuffer[ posh    ] = hex_trans_byte1[ (unsigned char) byte];
 obuffer[ posh + 1] = what_attr;
 obuffer[ posh + 2] = hex_trans_byte2[ (unsigned char) byte];
 obuffer[ posh + 3] = what_attr;

 obuffer[ posa    ] = byte;
 obuffer[ posa + 1] = what_attr;
#elif defined(VCHE_RAW)
 obuffer[ posh    ] = restore_buffer[ posh     + co2 ] = hex_trans_byte1[ (unsigned char) byte];
 obuffer[ posh + 1] = restore_buffer[ posh + 1 + co2 ] = what_attr;
 obuffer[ posh + 2] = restore_buffer[ posh + 2 + co2 ] = hex_trans_byte2[ (unsigned char) byte];
 obuffer[ posh + 3] = restore_buffer[ posh + 3 + co2 ] = what_attr;

 obuffer[ posa    ] = restore_buffer[ posa     + co2 ] = byte > 31 && byte < 127 ? byte : '.';
 obuffer[ posa + 1] = restore_buffer[ posa + 1 + co2 ] = what_attr;
#elif defined(VCHE_NC)
 obuffer[ posh    ] = hex_trans_byte1[ (unsigned char) byte];
 obuffer[ posh + 1] = what_attr;
 obuffer[ posh + 2] = hex_trans_byte2[ (unsigned char) byte];
 obuffer[ posh + 3] = what_attr;

 obuffer[ posa    ] = byte > 31 && byte < 127 ? byte : '.';
 obuffer[ posa + 1] = what_attr;
#endif

}

/*
 * If we are in a valid position, insert the byte in the correct
 * place calling the appropiate function.
 */
static void edit_insert( char c)
{

 if( ( ey * bytes_per_row + ex) >= bread) {
     term_beep();
     return;
 }

 if( tab_flag) { /* ascii edit */
     edit_insert_in_ascii( c);
 } else {        /* hex edit   */
     if( isxdigit( c))
         edit_insert_in_hex( c);
     else
         term_beep();
 }

}

/*
 * Inserts the byte in the linked list and show the
 * changes (byte and attributes) in the screen.
 * The byte is inserted from the ascii segment of the
 * screen, this function takes care of that and show
 * the correct changes in both segments.
 */

static void edit_insert_in_ascii( char c)
{
 struct ll_edit *node = ll_edit_head;
 struct ll_edit *tmp  = ll_edit_head;
#if defined(VCHE_VC) || defined(VCHE_RAW)
 char buf[ 16];
#endif
 int pos;
 int temp;

 edit_calculate_xy_position();

 pos = ey * bytes_per_row + ex;

 while( node != NULL) {
        if( node->filepos < ( fpos + pos)) {
            tmp  = node;
            node = node->next;
        } else
            break;
 }

 temp = (unsigned char) c;

#if defined(VCHE_VC)

 buf[ 1] = buf[ 3] = VC_COLOR_WHITE;
 buf[ 0] = hex_trans_byte1[ temp];
 buf[ 2] = hex_trans_byte2[ temp];

 lseek( vcsafd, posxyh, SEEK_SET);
 write( vcsafd, &buf, 4);

 buf[ 0] = c; 
 buf[ 1] = VC_COLOR_WHITE;

 lseek( vcsafd, posxya, SEEK_SET);
 write( vcsafd, &buf, 2);

#elif defined(VCHE_RAW)

 term_color( VC_COLOR_WHITE);

 buf[ 0] = hex_trans_byte1[ temp];
 buf[ 1] = hex_trans_byte2[ temp];

 raw_puts( hx, hy, 2, buf);

 raw_putc( ax, ay, c > 31 && c < 127 ? c : '.');

#elif defined(VCHE_NC)
 attrset( VC_COLOR_WHITE);
 mvaddch( hy, hx    , hex_trans_byte1[ temp]);
 mvaddch( hy, hx + 1, hex_trans_byte2[ temp]);

 mvaddch( ay, ax, c > 31 && c < 127 ? c : '.');

 refresh();
#endif

 if( hbuffer[ pos] != c) {

     if( node == NULL)
         ll_edit_add( c, fpos + pos, tmp);
     else 
         if( node->filepos == ( fpos + pos)){
             node->byte = c;
         }
         else
             ll_edit_add( c, fpos + pos, tmp);

     edit_change_color_at_cursor( EDITOR_COLOR_BG_LGRAY);
     edit_set_color_in_buffer( EDITOR_COLOR_BG_LGRAY);
 } else {
     if( node != NULL) {
         if( node->filepos == ( fpos + pos)) {
             tmp->next = node->next;
             if( ll_edit_head == node)
                 ll_edit_head = tmp->next;
             free( node);
         }
     }
     edit_change_color_at_cursor( EDITOR_COLOR_BLUE);
     edit_set_color_in_buffer( EDITOR_COLOR_BLUE);
 }

 ex++;

 if( ex > bytes_per_row - 1 && ey == vc_lines - 3) {
     ex = 0;
     file_next_line();
     edit_show_edited_data();
 }

}

/*
 * Inserts the byte in the linked list and show the
 * changes (byte and attributes) in the screen.
 * The byte is inserted from the hex segment of the
 * screen, this function takes care of that and show
 * the correct changes in both segments.
 */

static void edit_insert_in_hex( char c)
{
 struct ll_edit *node = ll_edit_head;
 struct ll_edit *tmp  = ll_edit_head;
 char buf[ 16];
 int temp;
 int pos;
 int b1, b2;

 edit_calculate_xy_position();

 pos = ey * bytes_per_row + ex;

 while( node != NULL) {
        if( node->filepos < ( fpos + pos)) {
            tmp  = node;
            node = node->next;
        } else
            break;
 }

#if defined(VCHE_VC)
 buf[ 1] = buf[ 3] = VC_COLOR_WHITE;
#elif defined(VCHE_RAW)
 term_color( VC_COLOR_WHITE);
#endif

#if defined(VCHE_VC) || defined(VCHE_NC)
 if( node == NULL) {
     buf[ 0] = hex_trans_byte1[ (unsigned char) hbuffer[ pos]];
     buf[ 2] = hex_trans_byte2[ (unsigned char) hbuffer[ pos]];
 } else {
     if( node->filepos == ( fpos + pos)) {
         buf[ 0] = hex_trans_byte1[ (unsigned char) node->byte];
         buf[ 2] = hex_trans_byte2[ (unsigned char) node->byte];
     } else {
         buf[ 0] = hex_trans_byte1[ (unsigned char) hbuffer[ pos]];
         buf[ 2] = hex_trans_byte2[ (unsigned char) hbuffer[ pos]];
     }
 }

 c = toupper( c);

 if( f_digit)
     buf[ 2] = c;
 else
     buf[ 0] = c;

 b1 = hex2bin( buf, 0);
 b2 = hex2bin( buf, 2);
#elif defined(VCHE_RAW)

 if( node == NULL) {
     buf[ 0] = hex_trans_byte1[ (unsigned char) hbuffer[ pos]];
     buf[ 1] = hex_trans_byte2[ (unsigned char) hbuffer[ pos]];
 } else {
     if( node->filepos == ( fpos + pos)) {
         buf[ 0] = hex_trans_byte1[ (unsigned char) node->byte];
         buf[ 1] = hex_trans_byte2[ (unsigned char) node->byte];
     } else {
         buf[ 0] = hex_trans_byte1[ (unsigned char) hbuffer[ pos]];
         buf[ 1] = hex_trans_byte2[ (unsigned char) hbuffer[ pos]];
     }
 }

 c = toupper( c);

 if( f_digit)
     buf[ 1] = c;
 else
     buf[ 0] = c;

 b1 = hex2bin( buf, 0);
 b2 = hex2bin( buf, 1);

#endif

 temp = b1 * 16 + b2;

#if defined(VCHE_VC)
 lseek( vcsafd, posxyh, SEEK_SET);
 write( vcsafd, &buf, 4);

 buf[ 0] = temp;
 lseek( vcsafd, posxya, SEEK_SET);
 write( vcsafd, &buf, 2);
#elif defined(VCHE_RAW)
 raw_puts( hx, hy, 2, buf);

 buf[ 0] = temp > 31 && temp < 127 ? temp : '.';

 raw_putc( ax, ay, buf[ 0]);
#elif defined(VCHE_NC)
 temp = b1 * 16 + b2;

 attrset( VC_COLOR_WHITE);
 mvaddch( hy, hx    , buf[ 0]);
 mvaddch( hy, hx + 1, buf[ 2]);

 mvaddch( ay, ax, temp > 31 && temp < 127 ? temp : '.');

 refresh();
#endif

 if( hbuffer[ pos] != (char) temp) {
     if( node == NULL)
         ll_edit_add( temp, fpos + pos, tmp);
     else
         if( node->filepos == ( fpos + pos))
             node->byte = temp;
         else
             ll_edit_add( temp, fpos + pos, tmp);

     edit_change_color_at_cursor( EDITOR_COLOR_BG_LGRAY);
     edit_set_color_in_buffer( EDITOR_COLOR_BG_LGRAY);
 } else {
     if( node != NULL) {
         if( node->filepos == ( fpos + pos)) {
             tmp->next = node->next;
             if( ll_edit_head == node)
                 ll_edit_head = tmp->next;
             free( node);
         }
     }
     edit_change_color_at_cursor( EDITOR_COLOR_BLUE);
     edit_set_color_in_buffer( EDITOR_COLOR_BLUE);
 }

 if( f_digit) {
     ex++;

     if( ex > bytes_per_row - 1 && ey == vc_lines - 3) {
         ex = 0;
         file_next_line();
         edit_show_edited_data();
     }
 }

 f_digit = f_digit ? 0 : 1;
}

/*
 * Show with a special color the edited data.
 * And flush the output buffer to the screen (even if there
 * is not any edited data)
 */
static void edit_show_edited_data( void)
{
 struct ll_edit *node = ll_edit_head;
 int tmpx, tmpy;
 int bytes;

 memset( cbuffer, EDITOR_COLOR_BLUE, ( vc_lines - 2 ) * vc_cols);

 /* Navigate through the linked list looking for data
    at our current file position */
 while( node != NULL) {
        if( node->filepos < fpos)
            node = node->next;
        else
            break;
 }

 /* Check if there is edited data */
 if( node != NULL) {
     tmpx = ex;
     tmpy = ey;

     bytes = node->filepos - fpos;

     while( bytes < bread && node != NULL && node->filepos <= (fpos + bytes)) {

          ey = bytes / bytes_per_row;
          ex = - ey * bytes_per_row + bytes;

          /* We found edited data, insert it in the output buffer */
          edit_insert_in_obuffer( node->byte, EDITOR_COLOR_BG_LGRAY);
          edit_set_color_in_buffer( EDITOR_COLOR_BG_LGRAY);

          node = node->next;
          ex++;
          if( ex > bytes_per_row - 1) {
              ex = 0;
              ey++;
          }
          if( node != NULL)
              bytes = node->filepos - fpos;
     }

     ex = tmpx;
     ey = tmpy;
 }

 /* Finally flush the changes to the screen */
 lib_flush();
}

/*
 * Set the edit mask used by the XOR, OR, AND, etc functions.
 */
static void edit_set_mask( void)
{
#if defined(VCHE_NC)
 WINDOW *w;
#endif
 char emask[ 256];

#if defined(VCHE_VC) || defined(VCHE_RAW)
 lib_save_screen();
#endif

 memset( &emask, 0, sizeof( emask));
 sprintf( emask, "0x%X", edit_mask);

#if defined(VCHE_NC)
 w =
#endif

 lib_dialog( "Enter edit mask: ", emask, 38, VC_COLOR_BACKCYAN, VC_COLOR_WHITE);

#if defined(VCHE_VC) || defined(VCHE_RAW)
 term_goto( (vc_cols - 38) / 2, vc_lines / 2);
 key_scanf( emask, 38);
#elif defined(VCHE_NC)
 term_wgoto( w, 2, 2);
 key_scanf( w, emask, 38);
#endif

 term_unhide_cursor();

 if( !strncasecmp( emask, "0x", 2))
     sscanf( emask, "%x", &edit_mask);
 else
     edit_mask = atoi( emask);

#if defined(VCHE_VC) || defined(VCHE_RAW)
 lib_load_screen();
#elif defined(VCHE_NC)
 delwin( w);
#endif
}


/*
 * Execute the XOR, OR, AND, etc bit operations.
 */

static void edit_bitops( int bitop)
{
 struct ll_edit *node = ll_edit_head;
 struct ll_edit *tmp  = ll_edit_head;
 char buf[ 16];
 char c = 0;
 int pos;
 int temp;

 edit_calculate_xy_position();

 pos = ey * bytes_per_row + ex;

 if( pos >= bread) {
     term_beep();
     return;
 }

 while( node != NULL) {
        if( node->filepos < ( fpos + pos)) {
            tmp  = node;
            node = node->next;
        } else
            break;
 }

 /* If we found data in the linked list for the
    current file position, then apply the bit operations
    over that data */
 if( node != NULL && node->filepos == ( fpos + pos))
     switch( bitop) {
             case EDIT_BITOP_XOR: c =  node->byte ^ edit_mask;
                                  break;
             case EDIT_BITOP_OR : c =  node->byte | edit_mask;
                                  break;
             case EDIT_BITOP_AND: c =  node->byte & edit_mask;
                                  break;
             case EDIT_BITOP_NEG: c = ~node->byte;
                                  break;
             case EDIT_BITOP_ADD: c = node->byte;
                                  c++; 
                                  break;
             case EDIT_BITOP_SUB: c = node->byte;
                                  c--;
                                  break;
     }
 else /* Else apply the bit operations on a temporal variable */
     switch( bitop) {
             case EDIT_BITOP_XOR: c =  hbuffer[ pos] ^ edit_mask;
                                  break;
             case EDIT_BITOP_OR : c =  hbuffer[ pos] | edit_mask;
                                  break;
             case EDIT_BITOP_AND: c =  hbuffer[ pos] & edit_mask;
                                  break;
             case EDIT_BITOP_NEG: c = ~hbuffer[ pos];
                                  break;
             case EDIT_BITOP_ADD: c = hbuffer[ pos];
                                  c++;
                                  break;
             case EDIT_BITOP_SUB: c = hbuffer[ pos];
                                  c--;
                                  break;
     }

 temp = (unsigned char) c;

 /* Show the changes in the screen */
#if defined(VCHE_VC)

 buf[ 1] = buf[ 3] = VC_COLOR_WHITE;
 buf[ 0] = hex_trans_byte1[ temp];
 buf[ 2] = hex_trans_byte2[ temp];

 lseek( vcsafd, posxyh, SEEK_SET);
 write( vcsafd, &buf, 4);

 buf[ 0] = c;
 buf[ 1] = VC_COLOR_WHITE;

 lseek( vcsafd, posxya, SEEK_SET);
 write( vcsafd, &buf, 2);

#elif defined(VCHE_RAW)

 term_color( VC_COLOR_WHITE);
 buf[ 0] = hex_trans_byte1[ temp];
 buf[ 1] = hex_trans_byte2[ temp];

 raw_puts( hx, hy, 2, buf);

 raw_putc( ax, ay, c > 31 && c < 127 ? c : '.');

#elif defined(VCHE_NC)

 buf[ 0] = hex_trans_byte1[ temp];
 buf[ 2] = hex_trans_byte2[ temp];

 attrset( VC_COLOR_WHITE);
 mvaddch( hy, hx    , buf[ 0]);
 mvaddch( hy, hx + 1, buf[ 2]);

 mvaddch( ay, ax, c > 31 && c < 127 ? c : '.');

 refresh();

#endif

 /* Insert the changed data in the linked list
    of edited data */

 if( hbuffer[ pos] != c) {

     if( node == NULL)
         ll_edit_add( c, fpos + pos, tmp);
     else
         if( node->filepos == ( fpos + pos)){
             node->byte = c;
         }
         else
             ll_edit_add( c, fpos + pos, tmp);
     edit_change_color_at_cursor( EDITOR_COLOR_BG_LGRAY);
     edit_set_color_in_buffer( EDITOR_COLOR_BG_LGRAY);
 } else {
     if( node != NULL) {
         if( node->filepos == ( fpos + pos)) {
             tmp->next = node->next;
             if( ll_edit_head == node)
                 ll_edit_head = tmp->next;
             free( node);
         }
     }
     edit_change_color_at_cursor( EDITOR_COLOR_BLUE);
     edit_set_color_in_buffer( EDITOR_COLOR_BLUE);
 }

 ex++;

 if( ex > bytes_per_row - 1 && ey == vc_lines - 3) {
     ex = 0;
     file_next_line();
     edit_show_edited_data();
 }

}

static int edit_ask_save( int pos)
{
 int i;

 if( ll_edit_head == NULL)
     return -1;

 i = lib_ask( "The file was modified, save the changes?", " Yes ", " No ", " Cancel ", pos);

 if( i == 0) {
     edit_save_data();
     return -1;
 } else 
     if( i == 1)
         return -1;

 return 0;
}

static void edit_save_data( void)
{
 struct ll_edit *node = ll_edit_head;
 struct ll_edit *temp = ll_edit_head;
#if defined(VCHE_NC)
 WINDOW *w;
#endif

#if defined(VCHE_VC) || defined(VCHE_RAW)
 lib_save_screen();
#endif

#if defined(VCHE_NC)
 w =
#endif
 lib_message( " Saving... ", 0);

 while( node != NULL) {
        lseek( fd, node->filepos, SEEK_SET);
        write( fd, &node->byte, 1);
        temp = node;
        node = node->next;
        free( temp);
 }

 lseek( fd, (long) fpos, SEEK_SET);

 ll_edit_head = NULL;

#if defined(VCHE_VC) || defined(VCHE_RAW)
 lib_load_screen();
#endif

#if defined(VCHE_NC)
 delwin( w);
#endif
}

