/****************************************************************/
/*  Utility functions for scanning documents for typographical	*/
/*  errors.							*/
/****************************************************************/

#   include	"config.h"

#   include	"indlocal.h"
#   include	<debugon.h>

/****************************************************************/
/*  Cleanup routine: Free a possibility list.			*/
/****************************************************************/
void	indFreePossibilities(	PossibleWord *	pw	)
    {
    while( pw )
	{
	PossibleWord *	next= pw->pwNext;

	free( (char *)pw );

	pw= next;
	}

    return;
    }

/****************************************************************/
/*  Start a new possibility.					*/
/****************************************************************/
PossibleWord *	indNewPossibility(	int		startPosition,
					PossibleWord *	next,
					int		firstCharacter	)
    {
    PossibleWord *	pw= (PossibleWord *)malloc(sizeof(PossibleWord));

    if  ( ! pw )
	{ XDEB(pw); return pw;	}

    pw->pwStartPosition= startPosition;
    pw->pwInsertionPoint= 1;
    pw->pwRejectedAt= -1;
    pw->pwAcceptedAt= -1;
    pw->pwNext= next;
    pw->pwForm[0]= firstCharacter;
    pw->pwForm[1]= '\0';

    return pw;
    }

/****************************************************************/
/*  Add a character to all the possibilities in a list.		*/
/****************************************************************/
void indAddCharacterToPossibilities(	PossibleWord *	pw,
					int		c	)
    {
    while( pw )
	{
	if  ( pw->pwRejectedAt == -1 )
	    {
	    if  ( pw->pwInsertionPoint >= FORM_MAX )
		{
		pw->pwRejectedAt= pw->pwStartPosition+ pw->pwInsertionPoint;
		}
	    else{
		pw->pwForm[pw->pwInsertionPoint++]= c;
		pw->pwForm[pw->pwInsertionPoint  ]= '\0';
		}
	    }

	pw= pw->pwNext;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  1)  Give a judgement on the validity of a word.			*/
/*									*/
/*  2)  In a list of possiblities, count the number that is not yet	*/
/*	rejected.							*/
/*									*/
/************************************************************************/

/*  1  */
static int indCheckWord(	const unsigned char *	word,
				SpellCheckContext *	scc,
				int			asPrefix )
    {
    int				accepted;
    int				ignoredHow;

    if  ( ! asPrefix && isdigit( word[0] ) )
	{ return 0;	}

    if  ( ! asPrefix && scc->sccForgotInd )
	{
	/*  a  */
	if  ( indGet( &accepted, scc->sccForgotInd, word ) >= 0	&&
	      accepted						)
	    { return -1;	}
	}

    /*  b  */
    if  ( ! indGetWord( &ignoredHow, scc->sccStaticInd, word, asPrefix,
				    scc->sccCharKinds, scc->sccCharShifts ) )
	{ return 0;	}

    /*  c  */
    if  ( scc->sccLearntInd &&
	  indGet( &accepted, scc->sccLearntInd, word ) >= 0	&&
	  ( accepted || asPrefix )				)
	{ return 0; }

    return -1;
    }

/*  2  */
int indCountPossibilities(	PossibleWord *		pw,
				int			position,
				SpellCheckContext *	scc,
				int			characterThatFollows )
    {
    int		count= 0;

    while( pw )
	{
	if  ( pw->pwRejectedAt == -1 )
	    {
	    int	rejectedAsWord;
	    int	rejectedAsPrefix;

	    rejectedAsWord= indCheckWord( pw->pwForm, scc, 0 );

	    /* SLDEB(pw->pwForm,rejected); */

	    if  ( ! rejectedAsWord )
		{ pw->pwAcceptedAt= position; }

	    pw->pwForm[pw->pwInsertionPoint  ]= characterThatFollows;
	    pw->pwForm[pw->pwInsertionPoint+1]= '\0';
	    rejectedAsPrefix= indCheckWord( pw->pwForm, scc, 1 );
	    /* SLDEB(pw->pwForm,rejected); */
	    pw->pwForm[pw->pwInsertionPoint  ]= '\0';

	    if  ( rejectedAsPrefix )
		{ pw->pwRejectedAt= position;	}

	    if  ( ! rejectedAsWord || ! rejectedAsPrefix )
		{ count++;	}
	    }

	pw= pw->pwNext;
	}

    return count;
    }

/****************************************************************/
/*  Make a list of the possibilities.				*/
/****************************************************************/
void	indLogPossibilities(	PossibleWord *	pw	)
    {
    while( pw )
	{
	printf( "\"%s\": Rejected %d Accepted %d\r\n",
			pw->pwForm, pw->pwRejectedAt, pw->pwAcceptedAt );

	pw= pw->pwNext;
	}

    return;
    }

/****************************************************************/
/*  Cleanup routine: Remove rejected possibilities.		*/
/****************************************************************/
PossibleWord * indRejectPossibilities(	PossibleWord *	pw	)
    {
    PossibleWord *	rval= pw;
    PossibleWord *	next;

    while( rval						&&
	   rval->pwRejectedAt != -1			&&
	   rval->pwRejectedAt >= rval->pwAcceptedAt	)
	{
	next= rval->pwNext;

	free( (char *)rval );

	rval= next;
	}

    if  ( ! rval )
	{ return rval; }

    pw= rval;
    while( pw )
	{
	next= pw->pwNext;

	while( next					&&
	       next->pwRejectedAt != -1			&&
	       next->pwRejectedAt >= next->pwAcceptedAt	)
	    {
	    next= next->pwNext;

	    free( (char *)pw->pwNext );
	    pw->pwNext= next;
	    }

	pw= next;
	}

    return rval;
    }

/****************************************************************/
/*  Find a maximal possibility to reject. This is rather	*/
/*  arbitrary.							*/
/****************************************************************/
PossibleWord * indMaximalPossibility(	PossibleWord *	pw	)
    {
    PossibleWord *	rval= pw;

    if  ( ! rval )
	{ return rval;	}

    while( rval->pwAcceptedAt >= 0 )
	{ rval= rval->pwNext; }

    if  ( ! rval )
	{ return rval;	}

    pw= rval->pwNext;

    while( pw )
	{
	if  ( pw->pwAcceptedAt == -1 )
	    {
	    if  ( pw->pwStartPosition < rval->pwStartPosition )
		{ rval= pw; }
	    else{
		if  ( pw->pwStartPosition == rval->pwStartPosition	&&
		      pw->pwRejectedAt    >  rval->pwRejectedAt   	)
		    { rval= pw; }
		}
	    }

	pw= pw->pwNext;
	}

    return rval;
    }
