#   include	"config.h"

#   include	"indlocal.h"
#   include	<debugon.h>

#   ifdef NeXT
#	include	<libc.h>
#	include	<mach/mach.h>
#   endif

#   ifdef __linux__
#	include	<sys/types.h>
#	include	<sys/mman.h>
#	include	<errno.h>
#	ifndef MAP_FAILED
#	    define	MAP_FAILED	((void *)-1)
#	endif
/* #	define	IND_MMAP requires 512 byte boundaries */
#   endif

#   include	<unistd.h>
#   include	<fcntl.h>
#   include	<utilEndian.h>

/************************************************************************/
/*  Swap a buffer							*/
/************************************************************************/
static void	indSwapInt( char * buf )
    {
    register char	c;

    c= buf[0]; buf[0]= buf[3]; buf[3]= c;
    c= buf[1]; buf[1]= buf[2]; buf[2]= c;
    }

static void	indSwapInd( IND * ind )
    {
    indSwapInt( (char *)&(ind->ind_magic) );
    indSwapInt( (char *)&(ind->ind_fd) );
    indSwapInt( (char *)&(ind->ind_start) );
    indSwapInt( (char *)&(ind->ind_readonly) );
    indSwapInt( (char *)&(ind->ind_nnode) );
    indSwapInt( (char *)&(ind->indAllocatedNodes) );
    indSwapInt( (char *)&(ind->indAllocatedLinks) );
    indSwapInt( (char *)&(ind->ind_lfree) );
    indSwapInt( (char *)&(ind->ind_lfull) );
    indSwapInt( (char *)&(ind->ind_ccount) );
    indSwapInt( (char *)&(ind->ind_cfree) );
    indSwapInt( (char *)&(ind->ind_cfull) );

    return;
    }

/************************************************************************/
/*  Make an index.							*/
/************************************************************************/
IND *	indINDmake( int	readonly )
    {
    IND *	rval= (IND *)malloc( sizeof( IND ) );

    if  ( rval )
	{
	rval->ind_magic= INDMAGIC;
	rval->ind_fd= -1;

	rval->ind_start= -1;
	rval->ind_readonly= readonly;

	rval->ind_nodes= (TrieNode *)0;
	rval->indNodePages= (TrieNode **)0;
	rval->ind_nnode= 0;
	rval->indAllocatedNodes= 0;

	rval->ind_links= (TrieLink *)0;
	rval->indLinkPages= (TrieLink **)0;
	rval->indAllocatedLinks= 0;
	rval->ind_lfree= 0;
	rval->ind_lfull= 0;

	rval->ind_classes= (int *)0;
	rval->ind_cpages= (int **)0;
	rval->ind_ccount= 0;
	rval->ind_cfree= 0;
	rval->ind_cfull= 0;
	}

    return rval;
    }

void	indINDfree( IND *	ind )
    {
    int		i;
    int		page;
    char *	p;
    int		pagesAreMemoryMapped= 0;

#   ifdef NeXT
    if  ( ind->ind_readonly )
	{
	kern_return_t	ret;

	if  ( ind->ind_nodes )
	    {
	    ret= vm_deallocate( task_self(), (vm_address_t)ind->ind_nodes,
				ind->indAllocatedNodes* sizeof(TrieNode) );
	    if  ( ret != KERN_SUCCESS )
		{ ; }
	    }
	if  ( ind->ind_links )
	    {
	    ret= vm_deallocate( task_self(), (vm_address_t)ind->ind_links,
				ind->indAllocatedLinks* sizeof(TrieLink) );
	    if  ( ret != KERN_SUCCESS )
		{ ; }
	    }
	if  ( ind->ind_classes )
	    {
	    ret= vm_deallocate( task_self(), (vm_address_t)ind->ind_classes,
				ind->ind_ccount* sizeof( int ) );
	    if  ( ret != KERN_SUCCESS )
		{ ; }
	    }

	pagesAreMemoryMapped= 1;
	}
#   endif


#   ifdef IND_MMAP
    if  ( ind->ind_readonly )
	{
	int	ret;

	if  ( ind->ind_nodes )
	    {
	    ret= munmap( (void *)ind->ind_nodes,
				ind->indAllocatedNodes* sizeof(TrieNode) );
	    if  ( ret )
		{ ; }
	    }
	if  ( ind->ind_links )
	    {
	    ret= munmap( (void *)ind->ind_links,
				ind->indAllocatedLinks* sizeof(TrieLink) );
	    if  ( ret )
		{ ; }
	    }
	if  ( ind->ind_classes )
	    {
	    ret= munmap( (void *)ind->ind_classes,
				ind->ind_ccount* sizeof( int ) );
	    if  ( ret )
		{ ; }
	    }

	pagesAreMemoryMapped= 1;
	}
#   endif

    if  ( ind->indNodePages )
	{
	page= 0;
	if  ( ! pagesAreMemoryMapped )
	    {
	    for ( i= 0; i < ind->indAllocatedNodes; i += TNsBLOCK )
		{
		p= (char *)ind->indNodePages[page++];
		if  ( p )
		    { free( p );	}
		}
	    }

	free( (char *)ind->indNodePages );
	}

    if  ( ind->indLinkPages )
	{
	page= 0;
	if  ( ! pagesAreMemoryMapped )
	    {
	    for ( i= 0; i < ind->indAllocatedLinks; i += TLsBLOCK )
		{
		p= (char *)ind->indLinkPages[page++];
		if  ( p )
		    { free( p );	}
		}
	    }

	free( (char *)ind->indLinkPages );
	}

    if  ( ind->ind_cpages )
	{
	page= 0;
	if  ( ! pagesAreMemoryMapped )
	    {
	    for ( i= 0; i < ind->ind_ccount; i += ITsBLOCK )
		{
		p= (char *)ind->ind_cpages[page++];
		if  ( p )
		    { free( p );	}
		}
	    }

	free( (char *)ind->ind_cpages );
	}

    free( (char *)ind );
    }

/************************************************************************/
/*  Write an index to file						*/
/************************************************************************/
static int	indWriteData(	IND *		ind,
				int		fd,
				int		swap )
    {
    int		n;
    int		i;
    unsigned	sz;
    int		page;

    page= 0;
    for ( n= 0; n < ind->indAllocatedNodes; n += TNsBLOCK )
	{
	sz= TNsBLOCK * sizeof( TrieNode );
	if  ( swap )
	    {
	    for ( i= 0; i < TNsBLOCK; i++ )
		{
		indSwapInt( (char *)&ind->indNodePages[page][i].tn_transitions );
		indSwapInt( (char *)&ind->indNodePages[page][i].tn_items );
		}
	    }
	if  ( write( fd, (char *)ind->indNodePages[page], sz ) != sz )
	    { return -1; }

	if  ( swap )
	    {
	    for ( i= 0; i < TNsBLOCK; i++ )
		{
		indSwapInt( (char *)&ind->indNodePages[page][i].tn_transitions );
		indSwapInt( (char *)&ind->indNodePages[page][i].tn_items );
		}
	    }
	page++;
	}

    page= 0;
    for ( n= 0; n < ind->indAllocatedLinks; n += TLsBLOCK )
	{
	sz= TLsBLOCK * sizeof( TrieLink );
	if  ( swap )
	    {
	    for ( i= 0; i < TLsBLOCK; i++ )
		{ indSwapInt( (char *)&ind->indLinkPages[page][i] ); }
	    }
	if  ( write( fd, (char *)ind->indLinkPages[page], sz ) != sz )
	    { LDEB(fd); return -1; }
	if  ( swap )
	    {
	    for ( i= 0; i < TLsBLOCK; i++ )
		{ indSwapInt( (char *)&ind->indLinkPages[page][i] ); }
	    }
	page++;
	}

    page= 0;
    for ( n= 0; n < ind->ind_ccount; n += ITsBLOCK )
	{
	sz= ITsBLOCK * sizeof( int );
	if  ( swap )
	    {
	    for ( i= 0; i < ITsBLOCK; i++ )
		{ indSwapInt( (char *)&ind->ind_cpages[page][i] ); }
	    }
	if  ( write( fd, (char *)ind->ind_cpages[page], sz ) != sz )
	    { LDEB(fd); return -1; }
	if  ( swap )
	    {
	    for ( i= 0; i < ITsBLOCK; i++ )
		{ indSwapInt( (char *)&ind->ind_cpages[page][i] ); }
	    }
	page++;
	}

    return 0;
    }

int	indINDwrite(	IND *		ind,
			const char *	filename )
    {
    int		fd= creat( filename, 0666 );
    int		rval= 0;
    IND	swapped;

    if  ( fd < 0 )
	{ return -1; }

#   if 0
    printf( "WRITE: ROOT= %d, nnode= %d, ncount= %d lcount= %d\n",
	ind->ind_start,	ind->ind_nnode, ind->indAllocatedNodes,
	ind->indAllocatedLinks );
#   endif

    if  ( write( fd, ind, sizeof(IND) ) != sizeof(IND) )
	{ LDEB(fd); rval= -1;	}
    else{
	if  ( indWriteData( ind, fd, 0 ) )
	    { rval= -1;	}
	}

    if  ( ! rval )
	{
	swapped= *ind;
	indSwapInd( &swapped );

	if  ( write( fd, &swapped, sizeof(IND) ) != sizeof(IND) )
	    { rval= -1;	}
	else{
	    if  ( indWriteData( ind, fd, 1 ) )
		{ rval= -1;	}
	    }
	}

    if  ( close( fd ) )
	{ rval= -1;	}

    return rval;
    }

/************************************************************************/
/*  Read an index from file						*/
/************************************************************************/
IND *	indINDread(	const char *	filename,
			int		readonly )
    {
    int			fd;

    IND *		ind;
    TrieNode *		nodes;
    TrieLink *		links;
    int *		items;
    int			n;
    unsigned		sz;
    int			page;
    long		offset= 0;
    int			memory_mapped= 0;

    unsigned short	one= 1;
    int			machineBigEndian= ( *(unsigned char *)&one == 0 );

    unsigned char	hd[72];

    if  ( sizeof(TrieNode) != 12 )
	{ LLDEB(sizeof(TrieNode),12); return (IND *)0;	}
    if  ( sizeof(TrieLink) != 4 )
	{ LLDEB(sizeof(TrieLink),4); return (IND *)0;	}

    fd= open( filename, 0 );
    if  ( fd < 0 )
	{ SLDEB(filename,readonly); return (IND *)0;	}

    ind= indINDmake( readonly );

    if  ( ! ind )
	{ XDEB(ind); goto failure;	}

    ind->ind_readonly= readonly;

    if  ( read( fd, hd, sizeof(hd) ) != sizeof(hd) )
	{ LDEB(sizeof(hd)); goto failure;	}

    if  ( machineBigEndian )
	{
	ind->ind_magic= utilEndianExtractBeInt32( hd );

	if  ( ind->ind_magic == INDMAGIC )
	    {
	    ind->ind_start= utilEndianExtractBeInt32( hd+ 8 );
	    ind->ind_nnode= utilEndianExtractBeInt32( hd+ 24 );
	    ind->indAllocatedNodes= utilEndianExtractBeInt32( hd+ 28 );
	    ind->indAllocatedLinks= utilEndianExtractBeInt32( hd+ 40 );
	    ind->ind_lfree= utilEndianExtractBeInt32( hd+ 44 );
	    ind->ind_lfull= utilEndianExtractBeInt32( hd+ 48 );
	    ind->ind_ccount= utilEndianExtractBeInt32( hd+ 60 );
	    ind->ind_cfree= utilEndianExtractBeInt32( hd+ 64 );
	    ind->ind_cfull= utilEndianExtractBeInt32( hd+ 68 );

	    offset += sizeof(hd);
	    }
	else{
	    if  ( ind->ind_magic == INDMAGIC_R )
		{
		offset= sizeof(hd);
		offset += utilEndianExtractLeInt32( hd+ 28 );
		offset += utilEndianExtractLeInt32( hd+ 40 );
		offset += utilEndianExtractLeInt32( hd+ 60 );

		if  ( lseek( fd, offset, SEEK_SET ) != offset	||
		      read( fd, hd, sizeof(hd) ) != sizeof(hd)	)
		    { LDEB(offset); goto failure;	}

		ind->ind_start= utilEndianExtractLeInt32( hd+ 8 );
		ind->ind_nnode= utilEndianExtractLeInt32( hd+ 24 );
		ind->indAllocatedNodes= utilEndianExtractLeInt32( hd+ 28 );
		ind->indAllocatedLinks= utilEndianExtractLeInt32( hd+ 40 );
		ind->ind_lfree= utilEndianExtractLeInt32( hd+ 44 );
		ind->ind_lfull= utilEndianExtractLeInt32( hd+ 48 );
		ind->ind_ccount= utilEndianExtractLeInt32( hd+ 60 );
		ind->ind_cfree= utilEndianExtractLeInt32( hd+ 64 );
		ind->ind_cfull= utilEndianExtractLeInt32( hd+ 68 );

		offset += sizeof(hd);
		}
	    else{ LDEB(ind->ind_magic); goto failure;	}
	    }
	}
    else{
	ind->ind_magic= utilEndianExtractLeInt32( hd );

	if  ( ind->ind_magic == INDMAGIC )
	    {
	    ind->ind_start= utilEndianExtractLeInt32( hd+ 8 );
	    ind->ind_nnode= utilEndianExtractLeInt32( hd+ 24 );
	    ind->indAllocatedNodes= utilEndianExtractLeInt32( hd+ 28 );
	    ind->indAllocatedLinks= utilEndianExtractLeInt32( hd+ 40 );
	    ind->ind_lfree= utilEndianExtractLeInt32( hd+ 44 );
	    ind->ind_lfull= utilEndianExtractLeInt32( hd+ 48 );
	    ind->ind_ccount= utilEndianExtractLeInt32( hd+ 60 );
	    ind->ind_cfree= utilEndianExtractLeInt32( hd+ 64 );
	    ind->ind_cfull= utilEndianExtractLeInt32( hd+ 68 );

	    offset += sizeof(hd);
	    }
	else{
	    if  ( ind->ind_magic == INDMAGIC_R )
		{
		offset= sizeof(hd);
		offset += utilEndianExtractBeInt32( hd+ 28 );
		offset += utilEndianExtractBeInt32( hd+ 40 );
		offset += utilEndianExtractBeInt32( hd+ 60 );

		if  ( lseek( fd, offset, SEEK_SET ) != offset	||
		      read( fd, hd, sizeof(hd) ) != sizeof(hd)	)
		    { LDEB(offset); goto failure;	}

		ind->ind_start= utilEndianExtractBeInt32( hd+ 8 );
		ind->ind_nnode= utilEndianExtractBeInt32( hd+ 24 );
		ind->indAllocatedNodes= utilEndianExtractBeInt32( hd+ 28 );
		ind->indAllocatedLinks= utilEndianExtractBeInt32( hd+ 40 );
		ind->ind_lfree= utilEndianExtractBeInt32( hd+ 44 );
		ind->ind_lfull= utilEndianExtractBeInt32( hd+ 48 );
		ind->ind_ccount= utilEndianExtractBeInt32( hd+ 60 );
		ind->ind_cfree= utilEndianExtractBeInt32( hd+ 64 );
		ind->ind_cfull= utilEndianExtractBeInt32( hd+ 68 );

		offset += sizeof(hd);
		}
	    else{ LDEB(ind->ind_magic); goto failure;	}
	    }
	}

#   if 0
    printf( "READ:  ROOT= %d, nnode= %d, ncount= %d lcount= %d\n",
	ind->ind_start,	ind->ind_nnode, ind->indAllocatedNodes,
	ind->indAllocatedLinks );
#   endif

#   ifdef NeXT
    if  ( readonly && ! swap )
	{
	kern_return_t	ret;

	ret= map_fd( fd, (vm_offset_t)offset,
		    (vm_offset_t *)&ind->ind_nodes, 1,
		    ind->indAllocatedNodes* sizeof(TrieNode) );
	if  ( ret != KERN_SUCCESS )
	    { goto failure;	}

	offset += ind->indAllocatedNodes* sizeof(TrieNode);

	ret= map_fd( fd, (vm_offset_t)offset,
		    (vm_offset_t *)&ind->ind_links, 1,
		    ind->indAllocatedLinks* sizeof(TrieLink) );
	if  ( ret != KERN_SUCCESS )
	    { goto failure;	}

	offset += ind->indAllocatedLinks* sizeof(TrieLink);

	ret= map_fd( fd, (vm_offset_t)offset,
		    (vm_offset_t *)&ind->ind_classes, 1,
		    ind->ind_ccount* sizeof(int) );

	if  ( ret != KERN_SUCCESS )
	    { goto failure;	}

	offset += ind->ind_ccount* sizeof(int);

	memory_mapped= 1;
	}
#   endif

#   ifdef IND_MMAP
    if  ( readonly )
	{
	void *	mmapResult;

	mmapResult= mmap( (void *)0,
			    ind->indAllocatedNodes* sizeof(TrieNode),
			    PROT_READ, MAP_SHARED, fd, offset );

	if  ( mmapResult == MAP_FAILED )
	    { LSDEB(errno,strerror(errno)); goto failure;	}

	ind->ind_nodes= mmapResult;
	offset += ind->indAllocatedNodes* sizeof(TrieNode);

	mmapResult= mmap( (void *)0,
			    ind->indAllocatedLinks* sizeof(TrieLink),
			    PROT_READ, MAP_SHARED, fd, offset );

	if  ( mmapResult == MAP_FAILED )
	    { LSDEB(errno,strerror(errno)); goto failure;	}

	ind->ind_links= mmapResult;
	offset += ind->indAllocatedLinks* sizeof(TrieLink);

	mmapResult= mmap( (void *)0,
			    ind->ind_ccount* sizeof(int),
			    PROT_READ, MAP_SHARED, fd, offset );

	if  ( mmapResult == MAP_FAILED )
	    { LSDEB(errno,strerror(errno)); goto failure;	}

	ind->ind_classes= mmapResult;
	offset += ind->ind_ccount* sizeof(int);

	memory_mapped= 1;
	}
#   endif

    sz= (ind->indAllocatedNodes/TNsBLOCK)* sizeof( TrieNode * );
    ind->indNodePages= (TrieNode **)malloc( sz );
    if  ( ! ind->indNodePages )
	{ goto failure;	}
    page= 0;
    for ( n= 0; n < ind->indAllocatedNodes; n += TNsBLOCK )
	{ ind->indNodePages[page++]= (TrieNode *)0; }

    sz= (ind->indAllocatedLinks/TLsBLOCK)* sizeof( TrieLink * );
    ind->indLinkPages= (TrieLink **)malloc( sz );
    if  ( ! ind->indLinkPages )
	{ goto failure;	}
    page= 0;
    for ( n= 0; n < ind->indAllocatedLinks; n += TLsBLOCK )
	{ ind->indLinkPages[page++]= (TrieLink *)0; }

    if  ( ind->ind_ccount > 0 )
	{
	sz= (ind->ind_ccount/ITsBLOCK)* sizeof( int * );
	ind->ind_cpages= (int **)malloc( sz );
	if  ( ! ind->ind_cpages )
	    { goto failure;	}
	page= 0;
	for ( n= 0; n < ind->ind_ccount; n += ITsBLOCK )
	    { ind->ind_cpages[page++]= (int *)0; }
	}
    else{ ind->ind_cpages= (int **)0;	}

    page= 0;
    for ( n= 0; n < ind->indAllocatedNodes; n += TNsBLOCK )
	{
	if  ( ! memory_mapped )
	    {
	    sz= TNsBLOCK * sizeof( TrieNode );
	    ind->indNodePages[page++]= nodes= (TrieNode *)malloc( sz );
	    if  ( ! nodes )
		{ goto failure;	}

	    if  ( read( fd, (char *)nodes, sz ) != sz )
		{ goto failure;	}
	    }
	else{
	    ind->indNodePages[page++]= ind->ind_nodes+ n;
	    }
	}

    page= 0;
    for ( n= 0; n < ind->indAllocatedLinks; n += TLsBLOCK )
	{
	if  ( ! memory_mapped )
	    {
	    sz= TLsBLOCK * sizeof( TrieLink );
	    ind->indLinkPages[page++]= links= (TrieLink *)malloc( sz );
	    if  ( ! links )
		{ goto failure;	}

	    if  ( read( fd, (char *)links, sz ) != sz )
		{ goto failure;	}
	    }
	else{
	    ind->indLinkPages[page++]= ind->ind_links+ n;
	    }
	}

    page= 0;
    for ( n= 0; n < ind->ind_ccount; n += ITsBLOCK )
	{
	if  ( ! memory_mapped )
	    {
	    sz= ITsBLOCK * sizeof( int );
	    ind->ind_cpages[page++]= items= (int *)malloc( sz );
	    if  ( ! items )
		{ goto failure;	}

	    if  ( read( fd, (char *)items, sz ) != sz )
		{ goto failure;	}
	    }
	else{ ind->ind_cpages[page]= ind->ind_classes+ n; page++; }
	}

    if  ( close( fd ) )
	{ indINDfree( ind ); return (IND *)0; }

    return ind;

failure:
    close( fd );
    if  ( ind )
	{ indINDfree( ind );	}
    return (IND *)0;
    }
