/************************************************************************/
/*									*/
/*  Consult the operating system about certain things.			*/
/*									*/
/*  0)  mkdir() is not declared on linux, suppress warning.		*/
/*  1)  Note that we use shell commands to consult the file system.	*/
/*	this is the easiest way.					*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>

#   include	<sys/types.h>
#   include	<pwd.h>
#   include	<unistd.h>
#   include	<sys/stat.h>
#   include	<time.h>
#   include	<errno.h>

#   include	<appSystem.h>

#   include	<debugon.h>

    /*  0  */
#   ifdef __linux__
    extern int mkdir( const char * name, mode_t mode );
#   endif

#   define	USE_STAT	1
#   define	USE_ACCESS	1
#   define	USE_GETCWD	1

#   if  HAVE_DIRENT_H
#	define	USE_OPENDIR	1
#	include	<dirent.h>
#   endif

/************************************************************************/
/*									*/
/*  Determine the home directory of the user.				*/
/*									*/
/************************************************************************/

int appHomeDirectory(	char *		home,
			int		len )
    {
    struct passwd *	pwd;
    int			l;

    pwd= getpwuid( getuid() );
    if  ( ! pwd )
	{ LXDEB(getuid(),pwd); return -1;	}

    if  ( ! pwd->pw_dir )
	{ XDEB(pwd->pw_dir);	}

    l= strlen( pwd->pw_dir );
    if  ( l > len )
	{ SLDEB(pwd->pw_dir,len); return -1;	}

    strcpy( home, pwd->pw_dir ); return l;
    }

/************************************************************************/
/*									*/
/*  Check whether a directory exists.					*/
/*									*/
/************************************************************************/

#if USE_STAT
int appTestDirectory(	const char *	dir )
    {
    struct stat	st;

    if  ( stat( dir, &st ) )
	{
	if  ( errno != ENOENT )
	    { SSDEB(dir,strerror(errno)); }

	return -1;
	}

    if  ( ! S_ISDIR( st.st_mode ) )
	{ return -1;	}

    return 0;
    }
#else
int appTestDirectory(	const char *	dir )
    {
    char	scratch[1001];

    sprintf( scratch, "test -d '%s'", dir );

    if  ( system( scratch ) )
	{ return -1;	}

    return 0;
    }
#endif

#if USE_ACCESS
int appTestFileWritable( const char *	file )
    {
    if  ( access( file, W_OK ) )
	{ return -1;	}

    return 0;
    }
#else
int appTestFileWritable( const char *	file )
    {
    char	scratch[1001];

    sprintf( scratch, "test -w '%s'", file );

    if  ( system( scratch ) )
	{ return -1;	}

    return 0;
    }
#endif

#if USE_ACCESS
int appTestFileReadable( const char *	file )
    {
    if  ( access( file, R_OK ) )
	{ return -1;	}

    return 0;
    }
#else
int appTestFileReadable( const char *	file )
    {
    char	scratch[1001];

    sprintf( scratch, "test -w '%s'", file );

    if  ( system( scratch ) )
	{ return -1;	}

    return 0;
    }
#endif

#if USE_STAT
int appTestFileExists(	const char *	file )
    {
    struct stat	st;

    if  ( stat( file, &st ) )
	{
	if  ( errno != ENOENT )
	    { SSDEB(file,strerror(errno)); }

	return -1;
	}

    if  ( ! S_ISREG( st.st_mode ) )
	{ return -1;	}

    return 0;
    }
#else
int appTestFileExists( const char *	file )
    {
    char	scratch[1001];

    sprintf( scratch, "test -f '%s'", file );

    if  ( system( scratch ) )
	{ return -1;	}

    return 0;
    }
#endif

/************************************************************************/
/*									*/
/*  Make a Directory.							*/
/*									*/
/************************************************************************/

int appMakeDirectory(	const char *	dir )
    {
    if  ( mkdir( dir, 0777 ) )
	{ SSDEB(dir,strerror(errno)); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Make a Directory.							*/
/*									*/
/************************************************************************/

long appGetTimestamp( void )
    {
    long	now;

    now= time( (long *)0 ); sleep( 1 );

    return now;
    }

/************************************************************************/
/*									*/
/*  Translate a file name to an absolute name.				*/
/*									*/
/************************************************************************/
int appAbsoluteName(	char *		absolute,
			int		len,
			const char *	filename )
    {
    int		rellen;
    int		abslen;

    rellen= strlen( filename );

    if  ( filename[0] == '/' )
	{
	if  ( rellen > len )
	    { SLDEB(filename,len); return -1;	}

	strcpy( absolute, filename ); return rellen;
	}

    if  ( rellen > len- 2 )
	{ SLDEB(filename,len); return -1;	}

#   if USE_GETCWD
    if  ( ! getcwd( absolute, len- rellen- 2 ) )
	{ LDEB(len); SSDEB(filename,strerror(errno)); return -1; }

    abslen= strlen( absolute );

#   else
    {
    FILE *	f= popen( "pwd", "r" );

    if  ( ! f )
	{ XDEB(f); return -1;	}

    if  ( ! fgets( absolute, len- rellen- 2, f ) )
	{ LDEB(1); pclose( f ); return -1;	}

    abslen= strlen( absolute );
    if  ( abslen < 1 || absolute[abslen -1] != '\n' )
	{ SDEB(absolute); return -1;	}

    absolute[abslen--]= '\0';

    pclose( f );
    }
#   endif

    absolute[abslen]= '/';

    strcpy( absolute+ abslen+ 1, filename );

    return abslen+ 1+ rellen;
    }

/************************************************************************/
/*									*/
/*  Remove a file.							*/
/*									*/
/************************************************************************/

int appRemoveFile(	const char *	filename )
    {
    if  ( remove( filename ) )
	{ SSDEB(filename,strerror(errno)); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Call a function for all files matching a certain criterion.		*/
/*									*/
/************************************************************************/

#   define	FILEL		1000

#if USE_OPENDIR
int appForAllFiles(	const char *		dir,
			const char *		ext,
			void *			through,
			FILE_CALLBACK		callback )
    {
    int			rval= 0;
    DIR *		d= opendir( dir );
    struct dirent *	de;

    if  ( ! d )
	{ SSDEB(dir,strerror(errno)); return -1;	}

    de= readdir( d );
    while( de )
	{
	char *	dot= strrchr( de->d_name, '.' );

	if  ( dot					&&
	      ! strcmp( dot+ 1, ext )			)
	    {
	    char	scratch[FILEL+1];

	    sprintf( scratch, "%s/%s", dir, de->d_name );

	    if  ( (*callback)( scratch, through )	)
		{ SDEB(scratch); rval= -1; break;	}
	    }

	de= readdir( d );
	}

    if  ( closedir( d ) )
	{ SSDEB(dir,strerror(errno)); rval= -1;	}

    return rval;
    }
#else

int appForAllFiles(	const char *		dir,
			const char *		ext,
			void *			through,
			FILE_CALLBACK		callback )
    {
    int			rval= 0;
    char		scratch[FILEL+1];
    char *		format= "( ls '%s'/*.'%s' ) 2>/dev/null";
    FILE *		names;

    sprintf( scratch, format, dir, ext );
    names= popen( scratch, "r" );
    if  ( ! names )
	{ SXDEB(scratch,names); return -1;	}

    while( fgets( scratch, FILEL, names ) )
	{
	int	len= strlen( scratch );

	if  ( len > 0 && scratch[len -1] == '\n' )
	    { scratch[len -1]= '\0';		}
	else{ SLDEB(scratch,len); continue;	}

	if  ( (*callback)( scratch, through ) )
	    { SDEB(scratch); rval= -1; break;	}
	}

    if  ( pclose( names ) )
	{ LDEB(1); rval= -1;	}

    return rval;
    }

#endif
