/************************************************************************/
/*									*/
/*  Editor functionality, indepentent of the application.		*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedApp.h"

#   include	<X11/Xatom.h>

#   include	<debugon.h>

/************************************************************************/
/*									*/
/*  Change the attributes of the selection.				*/
/*									*/
/*  1)  Adjust the start line for recalculating the layout of the first	*/
/*	paragraph in the selection.					*/
/*  2)  If necessary, split the first particule to change the text	*/
/*	attributes of the second half.					*/
/*  3)  For all paragraphs in the selection upto the last one..		*/
/*  4)  Change the text attributes.					*/
/*									*/
/************************************************************************/

static int tedChangeItemProperties( unsigned int *		pPpChgMask,
				    unsigned int *		pTaChgMask,
				    BufferItem *		bi,
				    int				partFrom,
				    int				partUpto,
				    AppDrawingData *		add,
				    DocumentFontList *		dfl,
				    unsigned int		taUpdMask,
				    TextAttribute		taNew,
				    unsigned int		ppUpdMask,
				    const ParagraphProperties *	ppNew )
    {
    unsigned int	ppChgMask= 0;
    unsigned int	taChgMask= 0;

    if  ( taUpdMask )
	{
	TextParticule *		tp= bi->biParaParticules+ partFrom;
	int			part;

	tp= bi->biParaParticules+ partFrom;
	for ( part= partFrom; part < partUpto; tp++, part++ )
	    {
	    taChgMask |= docAttributeDifference(
				    tp->tpTextAttribute, taNew, taUpdMask );

	    if  ( tedChangeParticuleAttribute( add, tp, dfl,
						    taUpdMask, taNew ) )
		{ XDEB(taUpdMask); return -1;	}
	    }
	}

    if  ( ppUpdMask )
	{
	if  ( tedUpdateParaProperties( &ppChgMask, add, bi,
						    ppUpdMask, ppNew ) )
	    { XDEB(ppUpdMask); return -1;	}
	}

    *pTaChgMask= taChgMask;
    *pPpChgMask= ppChgMask;
    return 0;
    }

int tedChangeSelectionProperties( EditDocument *		ed,
				unsigned int			taUpdMask,
				TextAttribute			taNew,
				unsigned int			ppUpdMask,
				const ParagraphProperties *	ppNew )
    {
    AppDrawingData *		add= &(ed->edDrawingData);

    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    const BufferSelection *	bs= &(td->tdSelection);
    BufferDocument *		bd= td->tdDocument;
    DocumentFontList *		dfl= &(bd->bdFontList);

    int				part;

    TextParticule *		tp;

    BufferItem *		bi= bs->bsBegin.bpBi;
    int				oldY1;

    DocumentRectangle		drChanged;
    
    int				firstParticuleSplit= 0;

    unsigned int		selPpChgMask= 0;
    unsigned int		selTaChgMask= 0;
    unsigned int		paraPpChgMask;
    unsigned int		paraTaChgMask;

    if  ( ! bi )
	{ XDEB(bi); return -1; }

    if  ( tedHasIBarSelection( td ) )
	{ taUpdMask= 0;	}

    part= bs->bsBegin.bpParticule;

    oldY1= bi->biY1;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    /*  1  */
    tp= bi->biParaParticules+ part;

    /*  2  */
    if  ( (unsigned)bs->bsBegin.bpStroff > tp->tpStroff			&&
	  docAttributeDifference( tp->tpTextAttribute, taNew,
							    taUpdMask )	)
	{
	int		stroff= bs->bsBegin.bpStroff;

	if  ( (unsigned)stroff < tp->tpStroff+ tp->tpStrlen )
	    {
	    TextParticule *	newTp;

	    newTp= docCopyParticule( bi, part+ 1, stroff,
			tp->tpStroff+ tp->tpStrlen- stroff, DOCkindTEXT, tp );
	    if  ( ! newTp )
		{ XDEB(newTp); return -1;	}

	    tp= bi->biParaParticules+ part;
	    tp->tpStrlen= stroff- tp->tpStroff;
	    tp= newTp; part++;

	    if  ( bs->bsEnd.bpBi == bi )
		{ firstParticuleSplit= 1;	}
	    }
	else{ tp++; part++; }
	}

    /*  3  */
    while( bi != bs->bsEnd.bpBi )
	{
	int	col;
	int	partUpto= bi->biParaParticuleCount;

	col= bi->biParent->biNumberInParent;

	if  ( bs->bsCol0 < 0					||
	      bs->bsCol1 < 0					||
	      ( col >= bs->bsCol0 && col <= bs->bsCol1 )	)
	    {
	    /*  4  */
	    if  ( tedChangeItemProperties( &paraPpChgMask, &paraTaChgMask,
				    bi, part, partUpto, add, dfl,
				    taUpdMask, taNew, ppUpdMask, ppNew ) )
		{ XXDEB(taUpdMask,ppUpdMask);	}

	    if  ( paraPpChgMask || paraTaChgMask )
		{
		tedApplyItemFormat( bi, bd, add, &drChanged );

		if  ( paraPpChgMask && bi == bs->bsBegin.bpBi )
		    { tedDocAdaptHorizontalRuler( ed, bi, 1 );	}

		selPpChgMask |= paraPpChgMask;
		selTaChgMask |= paraTaChgMask;
		}
	    }

	bi= docNextParagraph( bi );
	if  ( ! bi )
	    { XDEB(bi); return -1;	}

	part= 0; firstParticuleSplit= 0;
	}

    /*  4  */
    if  ( tedChangeItemProperties( &paraPpChgMask, &paraTaChgMask, bi,
			part+ firstParticuleSplit,
			bs->bsEnd.bpParticule+ firstParticuleSplit, add, dfl,
			taUpdMask, taNew, ppUpdMask, ppNew ) )
	{ XXDEB(taUpdMask,ppUpdMask);	}

    part= bs->bsEnd.bpParticule+ firstParticuleSplit;
    tp= bi->biParaParticules+ part;

    if  ( part < bi->biParaParticuleCount )
	{
	int	strend= tp->tpStroff+ tp->tpStrlen;

	if  ( bs->bsEnd.bpStroff < strend )
	    {
	    TextParticule *	newTp;

	    newTp= docCopyParticule( bi, part+ 1, bs->bsEnd.bpStroff,
		    strend- bs->bsEnd.bpStroff,
		    DOCkindTEXT, tp );

	    if  ( ! newTp )
		{ XDEB(newTp); return -1;	}

	    tp= bi->biParaParticules+ part;
	    tp->tpStrlen= bs->bsEnd.bpStroff- tp->tpStroff;
	    }

	paraTaChgMask |= docAttributeDifference( tp->tpTextAttribute,
							taNew, taUpdMask );

	/*  2  */
	if  ( tedChangeParticuleAttribute( add, tp, dfl, taUpdMask, taNew ) )
	    { LDEB(1);	}
	}

    if  ( paraPpChgMask || paraTaChgMask )
	{
	tedApplyItemFormat( bi, bd, add, &drChanged );

	if  ( paraPpChgMask && bi == bs->bsBegin.bpBi )
	    { tedDocAdaptHorizontalRuler( ed, bi, 1 );	}

	selPpChgMask |= paraPpChgMask;
	selTaChgMask |= paraTaChgMask;
	}

    if  ( selPpChgMask || selTaChgMask )
	{
	tedExposeRectangle( ed, &drChanged, /*scrolledX,Y*/ 0,0 );

	docFindLineAndParticule( td->tdSelection.bsBegin.bpBi,
		    td->tdSelection.bsBegin.bpStroff,
		    &(td->tdSelection.bsBegin), 1 );
	docFindLineAndParticule( td->tdSelection.bsEnd.bpBi,
		    td->tdSelection.bsEnd.bpStroff,
		    &(td->tdSelection.bsEnd), 0 );

	tedCalculateSelectedLines( td );
	tedSelectionCoordinates( &(td->tdSelection), add );

	tedSelectionRectangle( &(td->tdSelectedRectangle),
					     add, &(td->tdSelection) );

	if  ( add->addBackRect.drY1 != oldY1 )
	    {
	    appDocSetScrollbarValues( ed );
	    appSetShellConstraints( ed );
	    }

	tedAdaptToolsToSelection( ed );

	appDocumentChanged( ed->edApplication, ed, 1 );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Set an I Bar selection and cause the I Bar to be drawn.		*/
/*									*/
/*  1)  As an extra service, remove empty fields.			*/
/*									*/
/************************************************************************/

int tedEditSetIBarSelection(	EditDocument *	ed,
				BufferItem *	bi,
				int *		pScrolledX,
				int *		pScrolledY,
				int		stroff )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    const int		andExposeRuler= 1;
    BufferPosition *	bp= &(td->tdSelection.bsBegin);

    if  ( td->tdObjectSelected )
	{
	XUnmapWindow( XtDisplay( ed->edDocumentWidget ), td->tdObjectWindow );
	td->tdObjectSelected= 0;
	}

    if  ( docFindLineAndParticule( bi, stroff, bp, 1 ) )
	{ LDEB(stroff); return -1;	}

    /*  1  */
    if  ( bp->bpParticule > 0 )
	{
	int		i;
	TextParticule *	tp= bi->biParaParticules+ bp->bpParticule;
	TextLine *	tl= bi->biParaLines+ bp->bpLine;

	if  ( tp[-1].tpKind == DOCkindFIELDSTART	&&
	      ! tp->tpTextAttribute.taInField		)
	    {
SDEB("NO!");
	    for ( i= bp->bpParticule;
				i < bi->biParaParticuleCount- 1; tp++, i++ )
		{ tp[-1]= tp[0];	}

	    if  ( bp->bpLine > 0					&&
		  tl[-1].tlFirstParticule+tl[-1].tlParticuleCount ==
						    bp->bpParticule	)
		{
		tl[-1].tlParticuleCount--;
		tl[ 0].tlFirstParticule--;
		}
	    else{
		if  ( tl->tlFirstParticule == bp->bpParticule )
		    { LLDEB(tl->tlFirstParticule,bp->bpParticule);	}

		tl->tlParticuleCount--;
		}

	    tl++;
	    for ( i= bp->bpLine+ 1; i < bi->biParaLineCount; tl++, i++ )
		{ tl->tlFirstParticule--;	}

	    bi->biParaParticuleCount--;
	    bp->bpParticule--;
	    }
	}

    tedSetSelectedPosition( ed, td, bp, pScrolledX, pScrolledY );

    tedAdaptToolsToPosition( ed, andExposeRuler );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Replace the selection in the document with new text.		*/
/*									*/
/*  b)  Replace the tail of the beginning paragraph with the new text.	*/
/*  c)  Merge the two paragraphs.					*/
/*  d)  Erase all paragraphs after the beginning of the selection	*/
/*	to, including the end.						*/
/*  e)  Update the paragraph buffer and the particule administration.	*/
/*									*/
/*  B)  The first particule of the line was split, also reformat the	*/
/*	previous line.							*/
/*	If paragraphs were merged, redraw the whole neighbourhood.	*/
/*  C)  - Recalculate the geometry from the start of the selection to	*/
/*	  the end of the paragraph.					*/
/*	- Redraw the affected part of the paragraph.			*/
/*  6)  Update the notion of current particule and current line.	*/
/*  7)  Redraw the I-Bar.						*/
/*									*/
/************************************************************************/

/*  B  */
static void tedAdjustRedrawBegin( TedDocument *			td,
				const DocumentRectangle *	drBack,
				DocumentRectangle *		drChanged,
				int *				pLine,
				int				part )
    {
    TextLine *	tl= td->tdSelection.bsBegin.bpBi->biParaLines+ *pLine;

    if  ( *pLine == 0 )
	{
	if  ( td->tdSelection.bsBegin.bpBi != td->tdSelection.bsEnd.bpBi )
	    {
	    drChanged->drX0= drBack->drX0;
	    drChanged->drX1= drBack->drX1;

	    if  ( drChanged->drY0 > tl->tlY0 )
		{ drChanged->drY0=  tl->tlY0;	}
	    }

	return;
	}

    if  ( part == tl->tlFirstParticule					||
	  td->tdSelection.bsBegin.bpBi != td->tdSelection.bsEnd.bpBi	)
	{
	(*pLine)--; tl--;

	drChanged->drX0= drBack->drX0;
	drChanged->drX1= drBack->drX1;

	if  ( drChanged->drY0 > tl->tlY0 )
	    { drChanged->drY0=  tl->tlY0;	}
	}

    return;
    }

void tedReplaceSelection(	EditDocument *		ed,
				const unsigned char *	addedText,
				int			addedLength )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    Widget			w= ed->edDocumentWidget;
    Window			win= XtWindow( w );
    GC				gc= ed->edGc;
	
    int				stroffShift= 0;
    int				partShift= 0;

    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
	
    int				line;
    int				part;

    AppDrawingData *		add= &(ed->edDrawingData);

    int				oldBackY1= add->addBackRect.drY1;
    int				refY1;
    int				scrolledX= 0;
    int				scrolledY= 0;

    int				yShift= 0;
    const DocumentRectangle *	drBack= &(add->addBackRect);
    DocumentRectangle		drChanged;
    int				stroff;

    TextLine *			tl;
    int				upto;

    stroff= td->tdSelection.bsBegin.bpStroff;
    part= td->tdSelection.bsBegin.bpParticule;
    line= td->tdSelection.bsBegin.bpLine;

    tl= bi->biParaLines+ line;
    upto= tl->tlStroff+ tl->tlStrlen;

    refY1= td->tdSelection.bsEnd.bpBi->biY1;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX1= drBack->drX1;

    /*  b,c,d,e  */
    if  ( docReplaceSelection( bd, &(td->tdSelection),
		    &partShift, &stroffShift,
		    addedText, addedLength, td->tdCurrentTextAttribute,
		    (void *)add->addDisplay, tedCloseObject ) )
	{ LDEB(addedLength); return;	}

    /*  B  */
    tedAdjustRedrawBegin( td, drBack, &drChanged, &line, part );

    /*  C  */
    if  ( tedRecalculateParaLayout( bi, bd, add, line, yShift,
		    stroffShift, upto+ addedLength, refY1, &drChanged ) < 0 )
	{ LDEB(line); }

    if  ( tedOpenItemObjects( bi, &ed->edColors,
				    &(bd->bdFontList), add, win, gc ) )
	{ LDEB(1);	}

    /*  6,7  */
    td->tdVisibleSelectionCopied= 0;

    tedEditSetIBarSelection( ed, bi, &scrolledX, &scrolledY,
							stroff+ addedLength );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    return;
    }

/************************************************************************/
/*									*/
/*  Merge the selection into one paragraph.				*/
/*									*/
/************************************************************************/

static int tedIsIndentationParticule(	const BufferItem *	bi,
					const TextParticule *	tp )
    {
    const unsigned char *	s= bi->biParaString+ tp->tpStroff;
    int				i;

    if  ( tp->tpKind == DOCkindTAB )
	{ return 1;	}

    if  ( tp->tpKind != DOCkindTEXT )
	{ return 0;	}

    for ( i= 0; i < tp->tpStrlen; i++ )
	{
	if  ( *(s++) != ' ' )
	    { return 0;	}
	}

    return 1;
    }

void tedMergeParagraphsInSelection(	EditDocument *	ed )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDrawingData);

    BufferItem *		biTo;
    BufferItem *		biFrom;

    int				oldBackY1= add->addBackRect.drY1;

    int				yShift= 0;
    const DocumentRectangle *	drBack= &(add->addBackRect);
    DocumentRectangle		drChanged;
    int				refY1;

    int				endMoved= 0;

    refY1= td->tdSelection.bsEnd.bpBi->biY1;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX1= drBack->drX1;

    if  ( td->tdSelection.bsBegin.bpBi == td->tdSelection.bsEnd.bpBi )
	{ LDEB(1); return; }

    biTo= td->tdSelection.bsBegin.bpBi;
    biFrom= docNextParagraph( biTo );
    if  ( ! biTo )
	{ XDEB(biTo); return;	}

    endMoved= biTo->biParaStrlen;

    for (;;)
	{
	int	partFrom= 0;

	int	particulesInserted= 0;
	int	charactersCopied= 0;


	while( partFrom < biFrom->biParaParticuleCount- 1		&&
	       tedIsIndentationParticule( biFrom,
				biFrom->biParaParticules+ partFrom )	)
	    { partFrom++; }

	if  ( partFrom < biFrom->biParaParticuleCount )
	    {
	    int		toCount= biTo->biParaParticuleCount;

	    if  ( toCount > 0						&&
		  biTo->biParaParticules[toCount-1].tpKind ==
							DOCkindTEXT	&&
		  biTo->biParaString[biTo->biParaStrlen-1] != ' '	)
		{
		if  ( docInflateTextString( biTo, 1 ) )
		    { LDEB(biTo->biParaStrlen); return; }

		biTo->biParaString[biTo->biParaStrlen++]= ' ';
		biTo->biParaString[biTo->biParaStrlen  ]= '\0';
		biTo->biParaParticules[toCount- 1].tpStrlen++;
		endMoved++;
		}

	    if  ( docCopyParticules( biTo, biFrom,
			biTo->biParaParticuleCount, partFrom,
			biFrom->biParaParticuleCount- partFrom,
			&particulesInserted, &charactersCopied, (char *)0 ) )
		{ LDEB(biFrom->biParaParticuleCount); return;	}

	    }

	if  ( biFrom == td->tdSelection.bsEnd.bpBi )
	    {
	    endMoved -= biFrom->biParaParticules[partFrom].tpStroff;
	    break;
	    }

	endMoved += charactersCopied;

	biFrom= docNextParagraph( biFrom );
	if  ( ! biFrom )
	    { XDEB(biFrom); return;	}
	}

    biTo= td->tdSelection.bsBegin.bpBi;
    biFrom= docNextParagraph( biTo );
    if  ( ! biTo )
	{ XDEB(biTo); return;	}

    for (;;)
	{
	BufferItem *	nextBi= docNextParagraph( biFrom );

	docDeleteItems( bd, biFrom->biParent, biFrom->biNumberInParent, 1 );

	if  ( biFrom == td->tdSelection.bsEnd.bpBi )
	    { break;	}

	if  ( ! nextBi )
	    { XDEB(nextBi); return;	}
	biFrom= nextBi;
	}

    if  ( tedRecalculateParaLayout( biTo, bd, add, /*line*/ 0, yShift,
				/* stroffShift */ 0,
				biTo->biParaStrlen, refY1, &drChanged ) < 0 )
	{ LDEB(1); return; }

    tedExposeRectangle( ed, &drChanged, /*scrolledX,Y*/ 0,0 );

    td->tdSelection.bsEnd.bpBi= biTo;
    td->tdSelection.bsEnd.bpStroff += endMoved;

    docFindLineAndParticule( td->tdSelection.bsEnd.bpBi,
		td->tdSelection.bsEnd.bpStroff, &(td->tdSelection.bsEnd), 0 );

    tedCalculateSelectedLines( td );
    tedSelectionCoordinates( &(td->tdSelection), add );

    tedSelectionRectangle( &(td->tdSelectedRectangle),
					     add, &(td->tdSelection) );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Replace the selection in a document with another document.		*/
/*  ( Used with 'paste', 'undo', 'redo'. )				*/
/*									*/
/*  B)  The first particule of the line was split, probably, part fits	*/
/*	on the previous line. Reformat from the previous particule.	*/
/*	If paragraphs were merged, redraw the whole neighbourhood.	*/
/*									*/
/*  1)  Replace the selection of the target with the text of those	*/
/*	particules at the beginning of the source that have the same	*/
/*	attributes.							*/
/*  2)  Insert the rest of the first paragraph of the source into the	*/
/*	target.								*/
/*  4)  If the insertion consists of more than one paragraph, split the	*/
/*	target paragraph.						*/
/*  5)  Insert the last particule of the insertion as text.		*/
/*									*/
/*  z)  Copy all paragraphs between the first and last (exclusive) of	*/
/*	the source to the target.					*/
/*									*/
/************************************************************************/

static int tedIncludeParagraphs(	const char *		filename,
					int *			pEndedInTable,
					int			startWithTable,
					BufferItem *		biFrom,
					BufferItem *		biTo,
					BufferPosition *	bpEndFrom,
					BufferDocument *	bdTo,
					AppDrawingData *	add,
					int			y,
					int *			pY,
					int *			pLastY,
					AppColors *		ac,
					Window			win,
					GC			gc )
    {
    BufferItem *	rowBi= (BufferItem *)0;
    int			inTable= 0;

    if  ( ! startWithTable )
	{
	biFrom= docNextParagraph( biFrom );
	if  ( ! biFrom )
	    { XDEB(biFrom); return -1;	}

	if  ( biFrom == bpEndFrom->bpBi )
	    { *pY= y; return 0; }
	}

    for (;;)
	{
	BufferItem *	insBi;

	if  ( biFrom->biParaInTable )
	    {
	    BufferPosition	bp;

	    BufferItem *	biRowTo= biTo->biParent->biParent;
	    BufferItem *	biRowFrom= biFrom->biParent->biParent;

	    inTable= 1;

	    if  ( biTo->biNumberInParent !=
					biTo->biParent->biGroupChildCount -1 )
		{
		if  ( docSplitGroupItem( &insBi,
				biTo->biParent, biTo->biNumberInParent+ 1 ) )
		    { LDEB(1); return -1;	}
		}

	    if  ( biTo->biParent->biNumberInParent !=
					    biRowTo->biGroupChildCount -1 )
		{
		if  ( docSplitGroupItem( &insBi,
			    biRowTo, biTo->biParent->biNumberInParent+ 1 ) )
		    { LDEB(1); return -1;	}

		biRowTo= insBi;
		}

	    rowBi= docCopyRowItem( bdTo, biRowTo->biParent,
			biRowTo->biNumberInParent+ 1, biRowFrom, filename );

	    if  ( ! rowBi )
		{ XDEB(rowBi); return -1;	}
	    rowBi->biY0= rowBi->biY1= y- 1;

	    if  ( tedLayoutItem( rowBi, bdTo, add, y, &y, pLastY ) )
		{ LDEB(1); return -1;	}

	    if  ( tedOpenItemObjects( rowBi, ac,
					&(bdTo->bdFontList), add, win, gc ) )
		{ LDEB(1); return -1;	}

	    if  ( docLastPosition( biRowFrom, &bp ) )
		{ LDEB(1); break;	}

	    if  ( biFrom == bpEndFrom->bpBi )
		{ break;	}

	    biFrom= docNextParagraph( bp.bpBi );
	    if  ( ! biFrom )
		{ break;	}

	    if  ( biFrom == bpEndFrom->bpBi && ! biFrom->biParaInTable )
		{ break;	}

	    if  ( docLastPosition( rowBi, &bp ) )
		{ LDEB(1); break;	}

	    biTo= bp.bpBi;
	    }
	else{
	    if  ( rowBi )
		{
		insBi= docInsertItem( rowBi->biParent,
				    rowBi->biNumberInParent+ 1, DOClevROW );
		if  ( ! insBi )
		    { XDEB(insBi); return -1;	}
		insBi->biY0= insBi->biY1= y- 1;

		insBi= docInsertItem( insBi, 0, DOClevCELL );
		if  ( ! insBi )
		    { XDEB(insBi); return -1;	}
		insBi->biY0= insBi->biY1= y- 1;

		insBi= docCopyParaItem( bdTo, insBi, 0, biFrom, filename );
		insBi->biY0= insBi->biY1= y- 1;
		}
	    else{
		insBi= docCopyParaItem( bdTo, biTo->biParent,
				biTo->biNumberInParent+ 1, biFrom, filename );
		insBi->biY0= insBi->biY1= y- 1;
		}

	    inTable= 0;

	    if  ( ! insBi )
		{ XDEB(insBi); return -1;	}

	    if  ( tedLayoutItem( insBi, bdTo, add, y, &y, pLastY ) )
		{ LDEB(1); return -1;	}

	    if  ( tedOpenItemObjects( insBi, ac, &(bdTo->bdFontList),
							    add, win, gc ) )
		{ LDEB(1); return -1;	}

	    biFrom= docNextParagraph( biFrom );
	    if  ( ! biFrom )
		{ XDEB(biFrom); return -1;	}

	    if  ( biFrom == bpEndFrom->bpBi )
		{ break;	}

	    biTo= insBi;
	    rowBi= (BufferItem *)0;
	    }
	}

    *pEndedInTable= inTable; *pY= y; return 0;
    }

/************************************************************************/
/*									*/
/*  Include the tail of a paragraph in another paragraph.		*/
/*									*/
/*  It is known by the caller that at the beginning if the series of	*/
/*  particules that is to be included, no particule merging is		*/
/*  possible/necessary.							*/
/*									*/
/*  1)  Inclusion at the end of the paragraph... No particule merging.	*/
/*  2)  If the last particule of the source, and the subsequent		*/
/*	particule of the target have the same attributes, let the	*/
/*	text replacement routine decide whether to merge them or not.	*/
/*  3)  Copy those particules that are to be copied 'as is.'		*/
/*  4)  Insert the last paricule as text if this is desirable.		*/
/*									*/
/************************************************************************/

static int tedIncludeTail(	BufferDocument *	bdTo,
				BufferItem *            biTo,
				const BufferItem *	biFrom,
				int			partTo,
				int			partFrom,
				int *			pPartShift,
				int *			pStroffShift,
				Display *		display,
				const char *		refFileName )
    {
    int			copyLastAsText;
    int			particulesCopied;

    TextParticule *	tpTo;
    TextParticule *	tpFrom;

    int			particulesInserted= 0;
    int			charactersCopied= 0;

    /*  1  */
    if  ( biTo->biParaStrlen == 0		||
	  partTo == biTo->biParaParticuleCount	)
	{ copyLastAsText= 0;	}
    else{
	tpTo= biTo->biParaParticules+ partTo;
	tpFrom= biFrom->biParaParticules+ biFrom->biParaParticuleCount- 1;

	/*  2  */
	if  ( tpFrom->tpKind == DOCkindTEXT				&&
	      tpTo->tpKind == DOCkindTEXT				&&
	      docEqualTextAttributes( &(tpTo->tpTextAttribute),
					&(tpFrom->tpTextAttribute) )	)
	    { copyLastAsText= 1;	}
	else{ copyLastAsText= 0;	}
	}

    /*  3  */
    particulesCopied= biFrom->biParaParticuleCount- partFrom- copyLastAsText;
    if  ( particulesCopied > 0 )
	{
	if  ( docCopyParticules( biTo, biFrom, partTo, partFrom,
				    particulesCopied, &particulesInserted,
				    &charactersCopied, refFileName ) )
	    { LDEB(1); return -1;	}
	}
    else{
	if  ( particulesCopied < 0 )
	    { LDEB(particulesCopied); particulesCopied= 0;	}
	}

    /*  4  */
    if  ( copyLastAsText > 0 )
	{
	tpTo= biTo->biParaParticules+ partTo+ particulesCopied;

	tpFrom= biFrom->biParaParticules+
				biFrom->biParaParticuleCount- copyLastAsText;

	if  ( docParaReplaceText( bdTo, biTo, partTo+ particulesCopied,
		    tpTo->tpStroff,
		    &particulesInserted, &charactersCopied, tpTo->tpStroff,
		    biFrom->biParaString+ tpFrom->tpStroff,
		    tpFrom->tpStrlen,
		    tpFrom->tpTextAttribute,
		    (void *)display, tedCloseObject ) )
	    { LDEB(1); return -1;	}
	}

    *pPartShift += particulesInserted;
    *pStroffShift += charactersCopied;

    return 0;
    }

static int tedIncludeHead(	BufferItem *		biTo,
				const BufferItem *	biFrom,
				const int		part,
				const int		stroff,
				TedDocument *		tdTo,
				int *			pPartShift,
				int *			pStroffShift,
				int *			pLine,
				int *			pCharactersCopied,
				DocumentRectangle *	drChanged,
				const AppDrawingData *	add,
				const char *		refFileName )
    {
    const DocumentRectangle *	drBack= &(add->addBackRect);

    TextParticule *		tpTo;
    TextParticule *		tpFrom;

    int				partTo= part;
    unsigned int		stroffTo= stroff;

    int				charactersCopied= 0;
    int				particulesCopied;
    int				particuleSplit= 0;

    BufferDocument *		bdTo= tdTo->tdDocument;

    tpTo= biTo->biParaParticules+ tdTo->tdSelection.bsBegin.bpParticule;
    tpFrom= biFrom->biParaParticules;

    if  ( biTo->biParaStrlen == 0 )
	{
	int		toInTable= biTo->biParaInTable;
	FormattingFrame	ff;

	if  ( docCopyParagraphRuler( &biTo->biParaProperties,
					    &biFrom->biParaProperties )	)
	    { LDEB(1); }

	biTo->biParaInTable= toInTable;
	tedParaScreenGeometry( biTo, &ff, add->addMagnifiedPixelsPerTwip );
	}

    /*  B  */
    tedAdjustRedrawBegin( tdTo, drBack, drChanged, pLine, partTo );

    /*  1  */
    particulesCopied= 0;
    while( particulesCopied < biFrom->biParaParticuleCount		&&
	   tpTo->tpKind == DOCkindTEXT					&&
	   tpFrom[particulesCopied].tpKind == DOCkindTEXT		&&
	   docEqualTextAttributes( &(tpTo->tpTextAttribute),
			&(tpFrom[particulesCopied].tpTextAttribute) )	)
	{ particulesCopied++; }

    if  ( particulesCopied > 0 )
	{
	charactersCopied= tpFrom[particulesCopied-1].tpStroff+
				    tpFrom[particulesCopied-1].tpStrlen;
			    
	if  ( docReplaceSelection( bdTo, &(tdTo->tdSelection),
		    pPartShift, pStroffShift,
		    biFrom->biParaString+ tpFrom->tpStroff, charactersCopied,
		    tpFrom[particulesCopied-1].tpTextAttribute,
		    (void *)add->addDisplay, tedCloseObject ) )
	    { LDEB(tpFrom->tpStrlen); return -1;	}
	}
    else{
	if  ( ! tedHasIBarSelection( tdTo )				&&
	      docReplaceSelection( bdTo, &(tdTo->tdSelection),
		    pPartShift, pStroffShift,
		    (unsigned char *)0, 0, tdTo->tdCurrentTextAttribute,
		    (void *)add->addDisplay, tedCloseObject )		)
	    { LDEB(0); return -1;	}
	}

    stroffTo += charactersCopied;

    if  ( stroffTo >= biTo->biParaStrlen )
	{ partTo= biTo->biParaParticuleCount;	}
    else{
	partTo= 0; tpTo= biTo->biParaParticules;
	while( tpTo->tpStroff+ tpTo->tpStrlen < stroffTo )
	    { partTo++; tpTo++;	}

	if  ( tpTo->tpStroff != stroffTo				&&
	      tpTo->tpStroff+ tpTo->tpStrlen != stroffTo		)
	    { particuleSplit= 1;	}

	if  ( particulesCopied < biFrom->biParaParticuleCount	&&
	      particuleSplit					)
	    {
	    if  ( ! docCopyParticule( biTo, partTo+ 1, stroffTo,
		    tpTo->tpStroff+ tpTo->tpStrlen- stroffTo,
							DOCkindTEXT, tpTo ) )
		{ LDEB(1); return -1;	}

	    tpTo= biTo->biParaParticules+ partTo;
	    tpTo->tpStrlen= stroffTo- tpTo->tpStroff;

	    (*pPartShift)++; partTo++;
	    }
	}

    if  ( particulesCopied < biFrom->biParaParticuleCount )
	{
	int	partFrom= particulesCopied;

	if  ( tedIncludeTail( bdTo, biTo, biFrom, partTo, partFrom, pPartShift,
				pStroffShift, add->addDisplay, refFileName ) )
	    { LLDEB(partTo,partFrom); return -1;	}

	charactersCopied= biFrom->biParaStrlen;
	}

    *pCharactersCopied= charactersCopied;
    return 0;
    }

static int tedIncludeDocument(	Widget			w,
				EditDocument *		edTo,
				BufferDocument *	bdFrom )
    {
    AppDrawingData *		add= &(edTo->edDrawingData);

    TedDocument *		tdTo= (TedDocument *)edTo->edPrivateData;
    BufferDocument *		bdTo= tdTo->tdDocument;

    Display *			display= XtDisplay( w );
    Window			win= XtWindow( w );
    GC				gc= edTo->edGc;
	
    int				stroffShift= 0;
    int				partShift= 0;
    int				endedInTable= 0;

    BufferPosition		bpBeginFrom;
    BufferPosition		bpEndFrom;

    BufferItem *		biFrom;

    BufferItem *		biTo;

    int				line;
    int				part;
    int				stroff;
    int				stroffEnd;

    int				charactersCopied= 0;

    int				oldBackY1= add->addBackRect.drY1;
    int				scrolledX;
    int				scrolledY;
    int				refY1;

    int				yShift= 0;
    const DocumentRectangle *	drBack= &(add->addBackRect);
    DocumentRectangle		drChanged;

    if  ( docFirstPosition( &(bdFrom->bdItem), &bpBeginFrom ) )
	{ LDEB(1); return -1;	}
    if  ( docLastPosition( &(bdFrom->bdItem), &bpEndFrom ) )
	{ LDEB(1); return -1;	}

    biFrom= bpBeginFrom.bpBi;
    biTo= tdTo->tdSelection.bsBegin.bpBi;

    stroff= tdTo->tdSelection.bsBegin.bpStroff;
    part= tdTo->tdSelection.bsBegin.bpParticule;
    line= tdTo->tdSelection.bsBegin.bpLine;

    refY1= tdTo->tdSelection.bsEnd.bpBi->biY1;

    drChanged= tdTo->tdSelectedRectangle;
    drChanged.drX0= drBack->drX0;
    drChanged.drX1= drBack->drX1;

    if  ( ! biFrom->biParaInTable )
	{
	if  ( tedIncludeHead( biTo, biFrom, part, stroff, tdTo,
		    &partShift, &stroffShift, &line,
		    &charactersCopied, &drChanged, add, edTo->edFilename ) )
	    { LDEB(1); return -1;	}
	}

    if  ( bpEndFrom.bpBi != biFrom )
	{
	int		y;
	int		oldY1= biTo->biY1;
	BufferItem *	newBi;

	const int	particulesCopied= 0;
	TextParticule *	tpFrom;

	/*  4  */
	if  ( docSplitParaItem( bdTo, &newBi, biTo, stroff+ charactersCopied ) )
	    { LLDEB(part+ partShift,stroff+ charactersCopied); return -1; }

	/*  C  */
	tedLayoutItem( biTo, bdTo, add, biTo->biY0, &y, &drChanged.drY1 );
	newBi->biY0= newBi->biY1= y;

	if  ( tedOpenItemObjects( biTo, &edTo->edColors,
					&(bdTo->bdFontList), add, win, gc ) )
	    { LDEB(1); return -1;	}

	if  ( tedIncludeParagraphs( edTo->edFilename, &endedInTable,
				    biFrom->biParaInTable,
				    biFrom, biTo, &bpEndFrom,
				    bdTo, add,
				    y, &y, &drChanged.drY1,
				    &edTo->edColors, win, gc ) )
	    { LDEB(1); return -1;	}

	part= 0; stroff= 0; partShift= 0; stroffShift= 0;

	biFrom= bpEndFrom.bpBi;
	tpFrom= biFrom->biParaParticules;

	if  ( ! endedInTable && biFrom->biParaStrlen > 0 )
	    {
	    int		partTo= part;
	    int		partFrom= particulesCopied;
	    int		wasEmpty= newBi->biParaStrlen == 0;
	    int		savedInTable= newBi->biParaInTable;

	    if  ( tedIncludeTail( bdTo, newBi, biFrom, partTo, partFrom,
			&partShift, &stroffShift, display, edTo->edFilename ) )
		{ LLDEB(partTo,partFrom); return -1;	}

	    if  ( wasEmpty &&
		    docCopyParagraphProperties( &newBi->biParaProperties,
						&biFrom->biParaProperties ) )
		{ LDEB(wasEmpty);	}

	    newBi->biParaInTable= savedInTable;
	    }

	newBi->biY0= newBi->biY1= y- 1;
	if  ( tedLayoutItem( newBi, bdTo, add, y, &y, &drChanged.drY1 ) )
	    { LDEB(1); return -1;	}

	if  ( tedOpenItemObjects( newBi, &edTo->edColors,
					&(bdTo->bdFontList), add, win, gc ) )
	    { LDEB(1); return -1;	}

	if  ( drChanged.drY1 < y- 1 )
	    { drChanged.drY1= y- 1;	}

	biTo= newBi;
	yShift= y- oldY1;
	stroffEnd= 0+ stroffShift;
	}
    else{
	stroffEnd= tdTo->tdSelection.bsBegin.bpStroff+ charactersCopied;

	/*  C  */
	if  ( tedRecalculateParaLayout( biTo, bdTo, add, line, yShift,
			    stroffShift, stroffEnd, refY1, &drChanged ) < 0 )
	    { LDEB(1); }

	if  ( tedOpenItemObjects( biTo, &edTo->edColors,
					&(bdTo->bdFontList), add, win, gc ) )
	    { LDEB(1); return -1;	}
	}

    tdTo->tdVisibleSelectionCopied= 0;
    tedEditSetIBarSelection( edTo, biTo, &scrolledX, &scrolledY, stroffEnd );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( edTo );
	appSetShellConstraints( edTo );
	}

    tedExposeRectangle( edTo, &drChanged, scrolledX, scrolledY );

    return 0;
    }

int tedIncludePlainDocument(	Widget			w,
				EditDocument *		ed,
				BufferDocument *	bdFrom )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferPosition		bpBeginTo;
    BufferPosition		bpBeginFrom;
    BufferItem *		bi;

    TextAttribute		ta;

    bpBeginTo= td->tdSelection.bsBegin;
    if  ( ! bpBeginTo.bpBi				||
	  bpBeginTo.bpBi->biParaParticuleCount == 0	)
	{ LDEB(1); return -1;	}

    ta= bpBeginTo.bpBi->biParaParticules[bpBeginTo.bpParticule].tpTextAttribute;

    if  ( docFirstPosition( &(bdFrom->bdItem), &bpBeginFrom ) )
	{ LDEB(1); return -1;	}
    bi= bpBeginFrom.bpBi;

    for (;;)
	{
	TextParticule *		tp;
	int			part;

	tp= bi->biParaParticules;

	for ( part= 0; part < bi->biParaParticuleCount; part++, tp++ )
	    { tp->tpTextAttribute= ta; }

	bi= docNextParagraph( bi );
	if  ( ! bi )
	    { break;	}
	}

    return tedIncludeDocument( w, ed, bdFrom );
    }

int tedIncludeRtfDocument(	Widget			w,
				EditDocument *		ed,
				BufferDocument *	bdFrom )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferPosition		bpBeginFrom;
    BufferItem *		bi;

    int *			fontMap= (int *)0;
    int *			fontUsed= (int *)0;

    int				fromFont;
    int				fromFontCount;
    DocumentFont *		fromFonts;

    int				toFont;
    int				toFontCount;
    DocumentFont *		toFonts;

    if  ( bdFrom->bdItem.biDocContainsTables )
	{
	if  ( td->tdSelection.bsCol0 >= 0 )
	    { return 0;	}
	if  ( td->tdSelection.bsBegin.bpBi->biParaInTable )
	    { return 0;	}
	if  ( td->tdSelection.bsEnd.bpBi->biParaInTable )
	    { return 0;	}
	}

    fromFontCount= bdFrom->bdFontList.dflCount;
    fromFonts= bdFrom->bdFontList.dflFonts;

    toFontCount= td->tdDocument->bdFontList.dflCount;
    toFonts= td->tdDocument->bdFontList.dflFonts;

    toFonts= (DocumentFont *)realloc( toFonts,
		    ( toFontCount+ fromFontCount )* sizeof(DocumentFont) );
    if  ( ! toFonts )
	{ LXDEB(toFontCount+ fromFontCount,toFonts); return -1;	}
    td->tdDocument->bdFontList.dflFonts= toFonts;

    fontMap= (int *)malloc( fromFontCount* sizeof( int ) );
    if  ( ! fontMap )
	{ LXDEB(fromFontCount,fontMap); return -1; }

    fontUsed= (int *)malloc( fromFontCount* sizeof( int ) );
    if  ( ! fontUsed )
	{ LXDEB(fromFontCount,fontUsed); return -1; }

    for ( fromFont= 0; fromFont < fromFontCount; fromFont++ )
	{ fontUsed[fromFont]= 0;	}

    if  ( docFirstPosition( &(bdFrom->bdItem), &bpBeginFrom ) )
	{ LDEB(1); return -1;	}
    bi= bpBeginFrom.bpBi;

    for (;;)
	{
	TextParticule *		tp;
	int			part;

	tp= bi->biParaParticules;

	for ( part= 0; part < bi->biParaParticuleCount; part++, tp++ )
	    { fontUsed[tp->tpTextAttribute.taFontNumber]= 1; }

	bi= docNextParagraph( bi );
	if  ( ! bi )
	    { break;	}
	}

    for ( fromFont= 0; fromFont < fromFontCount; fromFont++ )
	{
	if  ( ! fontUsed[fromFont] )
	    { continue;	}

	for ( toFont= 0; toFont < toFontCount; toFont++ )
	    {
	    if  ( toFonts[toFont].dfDocFamilyNumber < 0 )
		{ continue;	} 

	    if  ( ! toFonts[toFont].dfFamilyStyle )
		{ XDEB(toFonts[toFont].dfFamilyStyle); continue;	}

	    if  ( ! strcmp( fromFonts[fromFont].dfFamilyStyle,
					toFonts[toFont].dfFamilyStyle )	&&
		  ! strcmp( fromFonts[fromFont].dfName,
					toFonts[toFont].dfName )	)
		{ break;	}
	    }

	if  ( toFont >= toFontCount )
	    {
	    DocumentFont *	df;

	    df= docInsertFont( &(td->tdDocument->bdFontList),
				    -1,
				    fromFonts[fromFont].dfFamilyStyle,
				    fromFonts[fromFont].dfName );

	    if  ( ! df )
		{ XDEB(df); return -1;	}

	    toFont= df->dfDocFamilyNumber;

	    toFonts= td->tdDocument->bdFontList.dflFonts;
	    toFontCount= td->tdDocument->bdFontList.dflCount;
	    }

	fontMap[fromFont]= toFont;
	}

    if  ( docFirstPosition( &(bdFrom->bdItem), &bpBeginFrom ) )
	{ LDEB(1); return -1;	}
    bi= bpBeginFrom.bpBi;

    for (;;)
	{
	TextParticule *		tp;
	int			part;

	tp= bi->biParaParticules;

	for ( part= 0; part < bi->biParaParticuleCount; part++, tp++ )
	    {
	    if  ( tp->tpTextAttribute.taFontNumber < 0			||
		  tp->tpTextAttribute.taFontNumber >= fromFontCount	)
		{
		LDEB(part);
		LLDEB(tp->tpTextAttribute.taFontNumber,fromFontCount);
		tp->tpTextAttribute.taFontNumber= 0;
		}
	    tp->tpTextAttribute.taFontNumber=
				    fontMap[tp->tpTextAttribute.taFontNumber];
	    }

	bi= docNextParagraph( bi );
	if  ( ! bi )
	    { break;	}
	}

    if  ( fontMap )
	{ free( fontMap );	}
    if  ( fontUsed )
	{ free( fontUsed );	}

    return tedIncludeDocument( w, ed, bdFrom );
    }

/************************************************************************/
/*									*/
/*  Split a paragraph.							*/
/*									*/
/*  1)  Split in the buffer administration.				*/
/*  2)  Recalculate the geometry from the start of the selection to the	*/
/*	end of the paragraph.						*/
/*  3)  Redraw the affected part of the paragraph.			*/
/*  4)  Layout the new paragraph.					*/
/*									*/
/************************************************************************/

int tedSplitParaContents(	BufferItem **		pNewBi,
				DocumentRectangle *	drChanged,
				EditDocument *		ed,
				int			splitLevel,
				int			onNewPage )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferDocument *	bd= td->tdDocument;
    BufferItem *	bi= td->tdSelection.bsBegin.bpBi;

    BufferItem *	newBi;

    AppDrawingData *	add= &(ed->edDrawingData);

    Widget		w= ed->edDocumentWidget;
    Window		win= XtWindow( w );
    GC			gc= ed->edGc;

    int			n;
    int			level;

    int			y;

    FormattingFrame	ff;

    /*  1  */
    if  ( docSplitParaItem( bd, &newBi, bi, td->tdSelection.bsBegin.bpStroff ) )
	{ LDEB(td->tdSelection.bsBegin.bpStroff); return -1;	}

    newBi->biParaStartsOnNewPage= ! newBi->biParaInTable && onNewPage;
    
    tedParagraphFrame( &ff, add, -1, bi );
    tedParaScreenGeometry( newBi, &ff, add->addMagnifiedPixelsPerTwip );

    /*  3  */
    *drChanged= td->tdSelectedRectangle;
    drChanged->drX0= add->addBackRect.drX0;
    drChanged->drX1= add->addBackRect.drX1; 

    if  ( bi->biParaBottomBorder.bpIsSet )
	{ bi->biParaBottomBorder.bpIsSet= 0;	}
    if  ( bi->biParaSpaceAfterTwips > 0 )
	{
	bi->biParaSpaceAfterTwips= 0;
	bi->biParaSpaceAfterPixels= 0;
	}

    if  ( newBi->biParaTopBorder.bpIsSet )
	{ newBi->biParaTopBorder.bpIsSet= 0;	}
    if  ( newBi->biParaSpaceBeforeTwips > 0 )
	{
	newBi->biParaSpaceBeforeTwips= 0;
	newBi->biParaSpaceBeforePixels= 0;
	}

    /*  2  */
    if  ( tedLayoutItem( bi, bd, add, bi->biY0, &y, &drChanged->drY1 ) < 0 )
	{ LDEB(1); return -1;	}

    /*  4  */
    newBi->biY0= newBi->biY1= y- 1;
    if  ( tedLayoutItem( newBi, bd, add, y, &y, &drChanged->drY1 ) < 0 )
	{ LDEB(1); return -1;	}

    if  ( tedOpenItemObjects( newBi, &ed->edColors,
					    &(bd->bdFontList), add, win, gc ) )
	{ LDEB(1); return -1;	}

    level= DOClevPARA;
    n= newBi->biNumberInParent;
    while( level > splitLevel )
	{
	BufferItem *	splitBi;

	if  ( docSplitGroupItem( &splitBi, bi->biParent, n ) )
	    { LDEB(1); return -1;	}

	n= splitBi->biNumberInParent+ 1;
	bi= splitBi; level--;
	}

    *pNewBi= newBi; return 0;
    }

void tedSplitParagraph(		EditDocument *		ed,
				int			onNewPage )
    {
    AppDrawingData *	add= &(ed->edDrawingData);
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferItem *	newBi;

    int			oldBackY1= add->addBackRect.drY1;
    int			scrolledX= 0;
    int			scrolledY= 0;

    DocumentRectangle	drChanged;

    if  ( tedSplitParaContents( &newBi, &drChanged,
						ed, DOClevPARA, onNewPage ) )
	{ LDEB(1); return;	}

    /*  7,8  */
    td->tdVisibleSelectionCopied= 0;
    tedEditSetIBarSelection( ed, newBi, &scrolledX, &scrolledY, 0 );

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a 'Tab'.							*/
/*									*/
/************************************************************************/
void tedSetTab(		Widget			w,
			EditDocument *		ed )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    TextParticule *		tp;

    int				yShift= 0;
    int				stroffShift;
    int				partShift;

    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    int				oldX0;

    int				line;
    int				part;

    int				oldBackY1= add->addBackRect.drY1;
    int				scrolledX= 0;
    int				scrolledY= 0;

    DocumentRectangle		drChanged;
    const DocumentRectangle *	drBack= &(add->addBackRect);

    part= td->tdSelection.bsBegin.bpParticule;
    line= td->tdSelection.bsBegin.bpLine;

    /*  1  */
    oldX0= bi->biParaParticules[part].tpX0;

    tp= docParaSpecialParticule( bi, DOCkindTAB,
				    part, td->tdSelection.bsEnd.bpStroff,
				    &partShift, &stroffShift );
    if  ( ! tp )
	{ LXDEB(part,tp); return;	}

    drChanged= td->tdSelectedRectangle;
    drChanged.drX1= drBack->drX1;

    /*  3,4,5  */
    if  ( tedRecalculateParaLayout( bi, bd, add, line, yShift,
			    stroffShift, td->tdSelection.bsBegin.bpStroff,
			    bi->biY1, &drChanged ) < 0 )
	{ LDEB(1); }

    /*  6,7  */
    td->tdVisibleSelectionCopied= 0;
    tedEditSetIBarSelection( ed, bi, &scrolledX, &scrolledY,
			    td->tdSelection.bsBegin.bpStroff+ stroffShift );

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Insert an 'Object'.							*/
/*									*/
/************************************************************************/
int tedInsertNewObject(		Widget			w,
				InsertedObject *	io,
				EditDocument *		ed )
    {
    AppDrawingData *		add= &(ed->edDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    TextParticule *		tp;

    int				yShift= 0;
    int				stroffShift;
    int				partShift;

    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    int				oldX0;

    int				line;
    int				part;

    int				oldBackY1= add->addBackRect.drY1;
    int				scrolledX;
    int				scrolledY;

    DocumentRectangle		drChanged;
    const DocumentRectangle *	drBack= &(add->addBackRect);

    part= td->tdSelection.bsBegin.bpParticule;
    line= td->tdSelection.bsBegin.bpLine;

    /*  1  */
    oldX0= bi->biParaParticules[part].tpX0;

    tp= docParaSpecialParticule( bi, DOCkindOBJECT,
				    part, td->tdSelection.bsEnd.bpStroff,
				    &partShift, &stroffShift );
    if  ( ! tp )
	{ LXDEB(part,tp); return -1;	}

    tp->tpObjectNumber= bi->biParaObjectCount++;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX1= drBack->drX1;

    /*  3,4,5  */
    if  ( tedRecalculateParaLayout( bi, bd, add, line, yShift,
			    stroffShift, td->tdSelection.bsBegin.bpStroff,
			    bi->biY1, &drChanged ) < 0 )
	{ LDEB(1); }

    /*  6,7  */
    td->tdVisibleSelectionCopied= 0;
    tedEditSetIBarSelection( ed, bi, &scrolledX, &scrolledY,
			    td->tdSelection.bsBegin.bpStroff+ stroffShift );

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    tp->tpPhysicalFont= 0;

    return 0;
    }

/************************************************************************/
/*									*/
/*  An object has been resized: Redo layout.				*/
/*									*/
/************************************************************************/
int tedResizeObject(		EditDocument *		ed,
				BufferItem *		bi,
				int			line,
				TextParticule *		tp,
				InsertedObject *	io,
				int			newWidth,
				int			newHeight,
				int			ox,
				int			oy )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferDocument *	bd= td->tdDocument;
    AppDrawingData *	add= &(ed->edDrawingData);

    Widget		w= ed->edDocumentWidget;
    Window		win= XtWindow( w );
    GC			gc= ed->edGc;

    int			yShift= 0;
    int			stroffShift= 0;

    int			oldBackY1= add->addBackRect.drY1;
    DocumentRectangle	drChanged;

    drChanged= td->tdSelectedRectangle;
    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;

    io->ioPixelsWide= newWidth;
    io->ioPixelsHigh= newHeight;
    io->ioScaleX= ( 100.0* io->ioPixelsWide )/
	    TWIPStoPIXELS( add->addMagnifiedPixelsPerTwip, io->ioTwipsWide );
    io->ioScaleY= ( 100.0* io->ioPixelsHigh )/
	    TWIPStoPIXELS( add->addMagnifiedPixelsPerTwip, io->ioTwipsHigh );

    tp->tpPixelsWide= newWidth;

    if  ( tedReopenObject( bd, bi, tp, &ed->edColors,
					    &(bd->bdFontList), add, win, gc ) )
	{ LDEB(1); return -1;	}

    if  ( tedRecalculateParaLayout( bi, bd, add, line, yShift,
				stroffShift, td->tdSelection.bsBegin.bpStroff,
				bi->biY1, &drChanged ) < 0 )
	{ LDEB(1); return -1; }

    tedExposeRectangle( ed, &drChanged, /*scrolledX,Y*/ 0,0 );

    tedCalculateSelectedLines( td );
    tedSelectionCoordinates( &(td->tdSelection), add );

    tedSelectionRectangle( &(td->tdSelectedRectangle),
					     add, &(td->tdSelection) );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Apply a change to the layout of a paragraph.			*/
/*									*/
/************************************************************************/
void tedEditApplyItemFormat(	EditDocument *	ed,
				BufferItem *	bi )
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    BufferDocument *	bd= td->tdDocument;
    AppDrawingData *	add= &(ed->edDrawingData);

    DocumentRectangle	drChanged;

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= bi->biY0;
    drChanged.drY1= bi->biY1;

    if  ( drChanged.drY0 > 0 )
	{ drChanged.drY0--;	}

    tedApplyItemFormat( bi, bd, add, &drChanged );

    tedExposeRectangle( ed, &drChanged, /*scrolledX,Y*/ 0,0 );

    tedCalculateSelectedLines( td );
    tedSelectionCoordinates( &(td->tdSelection), add );

    tedSelectionRectangle( &(td->tdSelectedRectangle),
						 add, &(td->tdSelection) );

    return;
    }

/************************************************************************/
/*									*/
/*  Replace the selection with something new.				*/
/*									*/
/*  1)  No selection... Just refuse.					*/
/*  2)  Replace nothing with nothing: Do nothing.			*/
/*									*/
/************************************************************************/
void tedAppReplaceSelection(	EditDocument *		ed,
				const unsigned char *	word,
				int			len	)
    {
    TedDocument *	td= (TedDocument *)ed->edPrivateData;

    /*  1  */
    if  ( ! td->tdSelection.bsBegin.bpBi )
	{ XDEB(td->tdSelection.bsBegin.bpBi); return; }

    /*  2  */
    if  ( tedHasIBarSelection( td ) && len == 0	)
	{ return;	}

    tedReplaceSelection( ed, word, len );

    appDocumentChanged( ed->edApplication, ed, 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Delete the current paragraph from a document.			*/
/*									*/
/************************************************************************/

int tedDeleteCurrentParagraph(	EditApplication *	ea )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    BufferItem *		bi= td->tdSelection.bsBegin.bpBi;
    AppDrawingData *		add= &(ed->edDrawingData);

    int				y0;
    int				y1;

    BufferPosition		bpNew;
    BufferItem *		parentBi;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    int				scrolledX= 0;
    int				scrolledY= 0;

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= bi->biY0- 1;
    drChanged.drY1= bi->biY1;

    y0= bi->biY0;
    y1= bi->biY1;

    docInitPosition( &bpNew );
    if  ( docLastPosition( bi, &bpNew ) 	||
	  docNextPosition( &bpNew )		)
	{
	const int	lastOne= 0;

	docInitPosition( &bpNew );
	if  ( docFirstPosition( bi, &bpNew )		||
	      docPrevPosition( &bpNew, lastOne )	)
	    { docInitPosition( &bpNew ); }
	}

    parentBi= bi->biParent;
    if  ( parentBi->biGroupChildCount < 2 )
	{
	int		part;
	TextParticule *	tp= bi->biParaParticules;
	TextAttribute	ta= tp->tpTextAttribute;
	Display *	display= XtDisplay( ed->edDocumentWidget );

	bi->biParaStrlen= 0;

	for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	    {
	    tedCloseObject( bd, bi, tp, (void *)display );
	    docCleanParticuleObject( bi, tp );
	    }

	y0= bi->biY0;
	y1= bi->biY1;

	docDeleteParticules( bi, 0, bi->biParaParticuleCount );
	docDeleteLines( bi, 0, bi->biParaLineCount );

	if  ( ! docInsertTextParticule( bi, 0, 0, 0, DOCkindTEXT, ta ) )
	    { LDEB(1); return -1;	}

	if  ( docFirstPosition( bi, &bpNew ) )
	    { LDEB(1); return -1;	}

	tedLayoutItem( bi, bd, add, y0, &y0, &drChanged.drY1 );
	}
    else{
	docDeleteItems( bd, parentBi, bi->biNumberInParent, 1 );

	y0= parentBi->biY0;
	y1= parentBi->biY1;

	tedLayoutItem( parentBi, bd, add, y0, &y0, &drChanged.drY1 );
	}

    td->tdVisibleSelectionCopied= 0;

    if  ( bpNew.bpBi )
	{
	tedEditSetIBarSelection( ed, bpNew.bpBi, &scrolledX, &scrolledY,
							    bpNew.bpStroff );
	}
    else{ XDEB(bpNew.bpBi);	}

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    appDocumentChanged( ed->edApplication, ed, 1 );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a paragraph Before/After the selection, depending on the	*/
/*  value of the 'after' argument.					*/
/*									*/
/************************************************************************/

int tedInsertParagraph(	EditApplication *	ea,
			int			after )
    {
    EditDocument *		ed= ea->eaCurrentDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDrawingData);

    BufferItem *		bi;
    BufferItem *		newBi;

    int				y0;

    TextAttribute		ta;
    BufferItem *		parentBi;
    int				pos;

    int				oldBackY1= add->addBackRect.drY1;
    DocumentRectangle		drChanged;

    int				scrolledX= 0;
    int				scrolledY= 0;

    if  ( after )
	{
	bi= td->tdSelection.bsEnd.bpBi;
	if  ( ! bi )
	    { XDEB(bi); return -1;	}

	pos= bi->biNumberInParent+ 1;
	ta= bi->biParaParticules[bi->biParaParticuleCount-1].tpTextAttribute;
	y0= bi->biY1+ 1;
	}
    else{
	bi= td->tdSelection.bsBegin.bpBi;
	if  ( ! bi )
	    { XDEB(bi); return -1;	}

	pos= bi->biNumberInParent;
	ta= bi->biParaParticules[0].tpTextAttribute;
	y0= bi->biY0;
	}

    drChanged.drX0= add->addBackRect.drX0;
    drChanged.drX1= add->addBackRect.drX1;
    drChanged.drY0= bi->biY0- 1;
    drChanged.drY1= bi->biY1;

    parentBi= bi->biParent;

    newBi= docInsertItem( parentBi, pos, DOClevPARA );
    if  ( ! newBi )
	{ XDEB(newBi); return -1;	}

    if  ( docCopyParagraphProperties( &(newBi->biParaProperties),
						&(bi->biParaProperties) ) )
	{ LDEB(1);	}

    newBi->biY0= newBi->biY1= y0;

    if  ( ! docInsertTextParticule( newBi, 0, 0, 0, DOCkindTEXT, ta ) )
	{ LDEB(1); return -1;	}

    tedLayoutItem( newBi, bd, add, y0, &y0, &drChanged.drY1 );

    td->tdVisibleSelectionCopied= 0;

    tedEditSetIBarSelection( ed, newBi, &scrolledX, &scrolledY, 0 );

    tedExposeRectangle( ed, &drChanged, scrolledX, scrolledY );

    appDocumentChanged( ed->edApplication, ed, 1 );

    if  ( add->addBackRect.drY1 != oldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    return 0;
    }
