/************************************************************************/
/*									*/
/*  Buffer manipulation routines.					*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<debugon.h>

#   include	"docBuf.h"

/*
# define DEB_PARTICULES
*/

/************************************************************************/
/*									*/
/*  Administration of fields that are deleted in the editing process.	*/
/*									*/
/************************************************************************/

static int docEditClaimFieldParticules(	TextParticule **	pTp,
					int			count )
    {
    TextParticule *	fresh;

    fresh= (TextParticule *)realloc( *pTp, count* sizeof( TextParticule ) );
    if  ( ! fresh )
	{ LXDEB(count,fresh); return -1;	}

    *pTp= fresh; return 0;
    }

/************************************************************************/
/*									*/
/*  Cleanup the set of fields that have been set apart during a		*/
/*  text substitution in a paragrapph.					*/
/*									*/
/*  1)  Bookmarks that begin and end in the text that is replaced are	*/
/*	dropped.							*/
/*  2)  Fields that begin and end in the text that is replaced are	*/
/*	dropped.							*/
/*  3)  Skip particules that have been deleted.				*/
/*  4)  Ends of bookmarks and fields are considered later on.		*/
/*  5)  Index entries and table of contents entries from the old text	*/
/*	are dropped.							*/
/*									*/
/*  a)  Superfluous as the array is reorganised.			*/
/*									*/
/************************************************************************/

static int docCleanupFields(	BufferDocument *	bd,
				BufferItem *		bi,
				TextParticule *		fieldParticules,
				int			fieldCount )
    {
    int		from;
    int		to;

    to= 0;
    for ( from= 0; from < fieldCount; from++ )
	{
	DocumentField *		dfFrom;
	DocumentField *		dfI;
	int			i;

	switch( fieldParticules[from].tpKind )
	    {
	    /*  1  */
	    case DOCkindBKMKSTART:
		dfFrom= bi->biParaFieldList.dflFields+
					fieldParticules[from].tpObjectNumber;
		dfI= (DocumentField *)0;
		for ( i= from+ 1; i < fieldCount; i++ )
		    {
		    if  ( fieldParticules[i].tpKind == DOCkindBKMKEND )
			{
			dfI= bi->biParaFieldList.dflFields+
					fieldParticules[i].tpObjectNumber;

			if  ( dfFrom->dfInstructions.odSize ==
					    dfI->dfInstructions.odSize	&&
			      ! memcmp( dfFrom->dfInstructions.odBytes,
					    dfI->dfInstructions.odBytes,
					    dfI->dfInstructions.odSize ) )
			    { break;	}
			}
		    }

		if  ( i < fieldCount )
		    {
		    int		numberInDocument= dfFrom->dfNumberInDocument;

		    docCleanField( dfI ); docInitField( dfI );
		    docCleanField( dfFrom ); docInitField( dfFrom );
		    dfI= bd->bdBookmarks.dflFields+ numberInDocument;
		    docCleanField( dfI ); docInitField( dfI );

		    fieldParticules[from].tpKind= DOCkindUNKNOWN; /* a */
		    fieldParticules[i].tpKind= DOCkindUNKNOWN;
		    }
		else{ fieldParticules[to++]= fieldParticules[from]; }
		break;

	    /*  2  */
	    case DOCkindFIELDSTART:
		dfI= (DocumentField *)0;
		dfFrom= bi->biParaFieldList.dflFields+
					fieldParticules[from].tpObjectNumber;
		for ( i= from+ 1; i < fieldCount; i++ )
		    {
		    if  ( fieldParticules[i].tpKind == DOCkindFIELDEND )
			{ break;	}
		    }

		if  ( i < fieldCount )
		    {
		    docCleanField( dfFrom ); docInitField( dfFrom );
		    fieldParticules[from].tpKind= DOCkindUNKNOWN; /* a */
		    fieldParticules[i].tpKind= DOCkindUNKNOWN;
		    }
		else{ fieldParticules[to++]= fieldParticules[from]; }
		break;

	    /*  3  */
	    case DOCkindUNKNOWN:
		continue;

	    /*  4  */
	    case DOCkindFIELDEND:
	    case DOCkindBKMKEND:
		fieldParticules[to++]= fieldParticules[from];
		break;

	    /*  5  */
	    case DOCkindXE:
	    case DOCkindTC:
		dfFrom= bi->biParaFieldList.dflFields+
					fieldParticules[from].tpObjectNumber;
		docCleanField( dfFrom ); docInitField( dfFrom );
		fieldParticules[from].tpKind= DOCkindUNKNOWN; /* a */
		continue;

	    default:
		LDEB(fieldParticules[from].tpKind);
		return -1;
	    }
	}

    return to;
    }

/************************************************************************/
/*									*/
/*  Reinsert a field particule that was contained in text that was	*/
/*  removed from the paragraph. Ends are only reinserted when the	*/
/*  corresponding beginning is also there.				*/
/*									*/
/************************************************************************/

static int docReinsertFieldParticule(	BufferItem *		oldBi,
					int			stroff,
					int			part,
					TextParticule *		tpField,
					TextParticule *		tpTo )
    {
    TextParticule *	tpFrom;
    int			i;

    DocumentField *	dfField;
    DocumentField *	dfBegin;

    switch( tpField->tpKind )
	{
	case DOCkindBKMKSTART:
	case DOCkindFIELDSTART:
	case DOCkindXE:
	case DOCkindTC:
	    break;

	case DOCkindBKMKEND:
	    tpFrom= oldBi->biParaParticules+ part- 1;
	    dfField= oldBi->biParaFieldList.dflFields+ tpField->tpObjectNumber;
	    dfBegin= (DocumentField *)0;

	    for ( i= part- 1; i >= 0; tpFrom--, i-- )
		{
		if  ( tpFrom->tpKind == DOCkindBKMKSTART )
		    {
		    dfBegin=  oldBi->biParaFieldList.dflFields+
						    tpFrom->tpObjectNumber;
		    if  ( dfField->dfInstructions.odSize ==
					dfBegin->dfInstructions.odSize	&&
			      ! memcmp( dfField->dfInstructions.odBytes,
					    dfBegin->dfInstructions.odBytes,
					    dfBegin->dfInstructions.odSize ) )
			    { break;	}
		    }
		}

	    if  ( i < 0 )
		{ return 0;	}

	    break;

	case DOCkindFIELDEND:
	    tpFrom= oldBi->biParaParticules+ part- 1;
	    for ( i= part- 1; i >= 0; tpFrom--, i-- )
		{
		if  ( tpFrom->tpKind == DOCkindFIELDSTART )
		    { break;	}
		}

	    if  ( i < 0 )
		{ return 0;	}

	    break;

	case DOCkindUNKNOWN:
	default:
	    LDEB(tpField->tpKind);
	    return -1;
	}

    if  ( tpTo )
	{
	*tpTo= *tpField;
	tpTo->tpStroff= stroff;
	}
    else{
	tpTo= docInsertTextParticule( oldBi, part, stroff,
		tpField->tpStrlen, tpField->tpKind, tpField->tpTextAttribute );
	}

#   ifdef DEB_PARTICULES
    DEBFUN( "NFL %3d: [%4d..%4d] %s <%s> \"%.*s\" len= %d\n", part,
		    tpTo->tpStroff,
		    tpTo->tpStroff+ tpTo->tpStrlen,
		    docKindStr( tpTo->tpKind ),
		    docAttributeStr( tpTo->tpTextAttribute ),
		    (int)tpTo->tpStrlen,
		    oldBi->biParaString+ tpTo->tpStroff,
		    tpTo->tpStrlen );
#   endif

    return 1;
    }

/************************************************************************/
/*									*/
/*  Replace text in a paragraph.					*/
/*									*/
/*  1)  Claim space to set apart special particules for field,		*/
/*	bookmarks, index entries &c.					*/
/*  2)  Make sure that enough space is allocated for the paragraph	*/
/*	contents.							*/
/*  3)  If the substitution actually is a deletion, compare attributes	*/
/*	to the current position.					*/
/*									*/
/*  4)  Find the last particule that does not end before the beginning	*/
/*	of the substitution.						*/
/*	NOTE that this automatically includes all particules that start	*/
/*	or end bookmarks.						*/
/*  5)  If the substitution starts at the beginning of a particule, and	*/
/*	the previous one has the same attributes as the inserted text,	*/
/*	start the redivision of the text in particules one particule	*/
/*	back.								*/
/*									*/
/*  6)  Find the last particule that does not begin after the end of	*/
/*	the substitution.						*/
/*	NOTE that this automatically includes all particules that start	*/
/*	or end bookmarks.						*/
/*  7)  If the substitution ends at the end of a particule, and	the	*/
/*	next one has the same attributes as the inserted text, include	*/
/*	the next one in the redivision of the text in particules.	*/
/*									*/
/*  8)  Decide whether the start particule needs to be split.		*/
/*  9)  Decide whether the end particule needs to be split.		*/
/*  9b) If the substitution is in the middle of a particule, it is not	*/
/*	possible to just adjust the begin of the particule, as it is	*/
/*	needed to do so for the first half. So make a new one.		*/
/*  10) Adapt the range of particules that are to be replaced to the	*/
/*	outcome of the decisions.					*/
/*									*/
/*  11) Delete particules corresponding to what is deleted, and to what	*/
/*	is to be redivided in particules.				*/
/*									*/
/*  12) Change the string contents of the paragraph.			*/
/*  13) Get rid of fields and bookmarks that start and end in de	*/
/*	deleted range.							*/
/*  14) Does the replacemeny start in a field?				*/
/*  15) Reinsert field and bookmark starts defore the new contents.	*/
/*									*/
/*  16) Sanity check.							*/
/*  17) Divide the replacement and its surroundings in particules.	*/
/*									*/
/*  18) Insert the remaining field, bookmark and index entry related	*/
/*	particules after the replacement.				*/
/*  19) Delete the particumes that are left over.			*/
/*  20) If the final result is an empty paragraph, insert a dummy	*/
/*	particule.							*/
/*									*/
/*  21) Adjust the offsets for the particules in the rest of the	*/
/*	paragraph.							*/
/*									*/
/*  22) Realise that the TextLine administration is left in an		*/
/*	inconsistent state.						*/
/*									*/
/************************************************************************/

static void docFindSubstitutionBegin(	int *			pPartBegin,
					TextParticule **	pTpBegin,
					const BufferItem *	bi,
					TextAttribute		addedAttr,
					int			stroffBegin,
					int			partBegin )
    {
    TextParticule *	tpBegin= bi->biParaParticules+ partBegin;

    /*  4  */
    while( partBegin > 0						&&
	   tpBegin[-1].tpStroff+ tpBegin[-1].tpStrlen > stroffBegin	)
	{ tpBegin--; partBegin--; }

    while( partBegin > 0						&&
	   tpBegin[-1].tpKind == DOCkindTEXT				&&
	   tpBegin[-1].tpStroff+ tpBegin[-1].tpStrlen >= stroffBegin	)
	{ tpBegin--; partBegin--; }

    /*  5  */
    if  ( partBegin > 0							&&
	  tpBegin[-1].tpKind == DOCkindTEXT				&&
	  tpBegin->tpStroff == stroffBegin				&&
	  docEqualTextAttributes( &addedAttr,
					&tpBegin[-1].tpTextAttribute )	)
	{ tpBegin--; partBegin--; }

    *pPartBegin= partBegin; *pTpBegin= tpBegin; return;
    }

static void docFindSubstitutionEnd(	int *			pPartEnd,
					TextParticule **	pTpEnd,
					const BufferItem *	bi,
					TextAttribute		addedAttr,
					int			stroffEnd,
					int			partBegin )
    {
    int				partEnd= partBegin;
    TextParticule *		tpEnd= bi->biParaParticules+ partEnd;

    /*  6  */
    while( partEnd < bi->biParaParticuleCount- 1			&&
	   tpEnd[1].tpStroff < stroffEnd				)
	{ tpEnd++; partEnd++;	}

    while( partEnd < bi->biParaParticuleCount- 1			&&
	   tpEnd[1].tpKind == DOCkindTEXT				&&
	   tpEnd[1].tpStroff <= stroffEnd				)
	{ tpEnd++; partEnd++;	}

    /*  7  */
    if  ( partEnd < bi->biParaParticuleCount- 1				&&
	   tpEnd[1].tpKind == DOCkindTEXT				&&
	  tpEnd->tpStroff+ tpEnd->tpStrlen == stroffEnd			&&
	  docEqualTextAttributes( &addedAttr,
					&tpEnd[1].tpTextAttribute )	)
	{ tpEnd++; partEnd++;	}

    *pPartEnd= partEnd; *pTpEnd= tpEnd; return;
    }

int docParaReplaceText(	BufferDocument *	bd,
			BufferItem *		bi,
			int			partBegin,
			unsigned int		stroffBegin,
			int *			pPartShift,
			int *			pStroffShift,
			unsigned int		stroffEnd,
			const unsigned char *	addedString,
			unsigned int		addedStrlen,
			TextAttribute		addedAttr,
			void *			voiddisplay,
			DOC_CLOSE_OBJECT	closeObject )
    {
    int				stroffShift;
    int				partShift;
    int				oldCount;
    int				newCount;

    TextParticule		firstParticule;

    int				part;
    TextParticule *		tp;

    int				partEnd;
    TextParticule *		tpBegin;
    TextParticule *		tpEnd;
    int				excludeBegin= 0;
    int				excludeEnd= 0;
    int				newBegin;
    int				newEnd;
    int				stroffEndEnd;

    int				i;
    TextParticule *		tpField;
    int				res;


    static TextParticule *	fieldParticules;
    int				fieldCount= 0;

    /*  1  */
    if  ( docEditClaimFieldParticules( &fieldParticules,
						bi->biParaParticuleCount ) )
	{ LDEB(bi->biParaParticuleCount); return -1;	}

    /*  2  */
    stroffShift= addedStrlen- stroffEnd+ stroffBegin;
    partShift= 0;
    if  ( stroffShift > 0				&&
	  docInflateTextString( bi, stroffShift )	)
	{ LLDEB(bi->biParaStrlen,stroffShift); return -1;	}

    /*  3  */
    if  ( addedStrlen == 0 )
	{
	tpBegin= bi->biParaParticules+ partBegin;
	addedAttr= tpBegin->tpTextAttribute;
	}

    /*  4,5  */
    docFindSubstitutionBegin( &partBegin, &tpBegin, bi,
				    addedAttr, stroffBegin, partBegin );

    /*  G  */
    firstParticule= *tpBegin;

    /*  6,7  */
    docFindSubstitutionEnd( &partEnd, &tpEnd, bi,
				    addedAttr, stroffEnd, partBegin );
    stroffEndEnd= tpEnd->tpStroff+ tpEnd->tpStrlen;

    /*  8  */
    if  ( tpBegin->tpKind == DOCkindTEXT				    &&
	  docEqualTextAttributes( &addedAttr, &(tpBegin->tpTextAttribute) ) )
	{ newBegin= tpBegin->tpStroff;			}
    else{
	newBegin= stroffBegin;

	if  ( newBegin > tpBegin->tpStroff )
	    { excludeBegin= 1; }
	}

    /*  9  */
    if  ( tpEnd->tpKind == DOCkindTEXT					  &&
	  docEqualTextAttributes( &addedAttr, &(tpEnd->tpTextAttribute) ) )
	{ newEnd= tpEnd->tpStroff+ tpEnd->tpStrlen;	}
    else{
	newEnd= stroffEnd;

	if  ( newEnd < tpEnd->tpStroff+ tpEnd->tpStrlen )
	    { excludeEnd= 1; }
	}

    /*  8  */
    if  ( excludeBegin )
	{
	tpBegin->tpStrlen= newBegin- tpBegin->tpStroff;

#	ifdef DEB_PARTICULES
	DEBFUN( "<<= %3d: [%4d..%4d] %s <%s> \"%.*s\"\n", partBegin,
				tpBegin->tpStroff,
				tpBegin->tpStroff+ tpBegin->tpStrlen,
				docKindStr( tpBegin->tpKind ),
				docAttributeStr( tpBegin->tpTextAttribute ),
				(int)tpBegin->tpStrlen,
				bi->biParaString+ tpBegin->tpStroff );
#	endif
	}

    /*  9  */
    if  ( excludeEnd )
	{
	/*  9b  */
	if  ( partEnd == partBegin	&&
	      excludeBegin		)
	    {
	    tp= docInsertTextParticule( bi, partEnd+ 1,
					newEnd, stroffEndEnd- newEnd,
					DOCkindTEXT, tpEnd->tpTextAttribute );
	    if  ( ! tp )
		{ XDEB(tp); return -1;	}

	    partShift++;
	    tpEnd= tp;
	    }
	else{
	    tpEnd->tpStroff= newEnd;
	    tpEnd->tpStrlen= stroffEndEnd- newEnd;
	    }

#	ifdef DEB_PARTICULES
	DEBFUN( "=>> %3d: [%4d..%4d] %s <%s> \"%.*s\"\n", partEnd,
				tpEnd->tpStroff,
				tpEnd->tpStroff+ tpEnd->tpStrlen,
				docKindStr( tpEnd->tpKind ),
				docAttributeStr( tpEnd->tpTextAttribute ),
				(int)tpEnd->tpStrlen,
				bi->biParaString+ tpEnd->tpStroff );
#	endif
	}

    newEnd += stroffShift;

    /*  10  */
    oldCount= 0;
    if  ( excludeBegin )
	{ partBegin++;	}
    if  ( excludeEnd )
	{ partEnd--;	}

    tpBegin= bi->biParaParticules+ partBegin;
    tpEnd= bi->biParaParticules+ partEnd;

    /*  11  */
    tp= tpBegin;
    for ( part= partBegin; part <= partEnd; tp++, part++ )
	{
#	ifdef DEB_PARTICULES
	DEBFUN( "OLD %3d: [%4d..%4d] %s <%s> \"%.*s\"\n", oldCount,
		    tp->tpStroff,
		    tp->tpStroff+ tp->tpStrlen,
		    docKindStr( tp->tpKind ),
		    docAttributeStr( tp->tpTextAttribute ),
		    (int)tp->tpStrlen,
		    bi->biParaString+ tp->tpStroff );
#	endif

	(*closeObject)( bd, bi, tp, voiddisplay );
	docCleanParticuleObject( bi, tp );

	if  ( tp->tpKind == DOCkindBKMKSTART	||
	      tp->tpKind == DOCkindBKMKEND		||
	      tp->tpKind == DOCkindFIELDSTART	||
	      tp->tpKind == DOCkindFIELDEND		||
	      tp->tpKind == DOCkindXE		||
	      tp->tpKind == DOCkindTC		)
	    { fieldParticules[fieldCount++]= *tp;	}

	partShift--; oldCount++;
	}

    /*  12  */
    if  ( bi->biParaStrlen > 0 )
	{
	memmove( bi->biParaString+ stroffEnd+ stroffShift,
		bi->biParaString+ stroffEnd, bi->biParaStrlen- stroffEnd+ 1 );
	}

    memcpy( bi->biParaString+ stroffBegin, addedString, addedStrlen );

    if  ( stroffShift != 0 )
	{
	bi->biParaString[bi->biParaStrlen+ stroffShift]= '\0';
	bi->biParaStrlen += stroffShift;
	}

    /*  13  */
    if  ( fieldCount > 0 )
	{ fieldCount= docCleanupFields( bd, bi, fieldParticules, fieldCount ); }
    if  ( fieldCount < 0 )
	{ LDEB(fieldCount); return -1;	}

    /*  14  */
    newCount= 0;
    addedAttr.taInField= 0;
    for ( i= 0; i < partBegin; i++ )
	{
	if  ( bi->biParaParticules[i].tpKind == DOCkindFIELDSTART )
	    { addedAttr.taInField= 1;	}
	if  ( bi->biParaParticules[i].tpKind == DOCkindFIELDEND )
	    { addedAttr.taInField= 0;	}
	}

    /*  15  */
    tp= tpBegin;
    part= partBegin;
    tpField= fieldParticules;
    for ( i= 0; i < fieldCount; tpField++, i++ )
	{
	if  ( tpField->tpKind != DOCkindBKMKSTART	&&
	      tpField->tpKind != DOCkindFIELDSTART	)
	    { continue;	}

	if  ( newCount < oldCount )
	    {
	    res= docReinsertFieldParticule( bi, newBegin, part,
							tpField, tp );
	    }
	else{
	    res= docReinsertFieldParticule( bi, newBegin, part,
					    tpField, (TextParticule *)0 );
	    }

	if  ( res < 0 )
	    { LDEB(res); return -1;	}

	if  ( res > 0 )
	    { partShift++; newCount++; part++, tp++;	}

	if  ( tpField->tpKind == DOCkindFIELDSTART )
	    { addedAttr.taInField= 1;	}

	tpField->tpKind= DOCkindUNKNOWN;
	}

    /*  16  */
    if  ( newBegin > newEnd )
	{ LLDEB(newBegin,newEnd); }

    /*  17  */
    while( newBegin < newEnd )
	{
	int	len= 0;
#	ifdef DEB_PARTICULES
	char *	label= "???";
#	endif

	while( newBegin+ len < newEnd			&&
	       bi->biParaString[newBegin+ len] != ' '	)
	    { len++;	}
	while( newBegin+ len < newEnd			&&
	       bi->biParaString[newBegin+ len] == ' '	)
	    { len++;	}

	if  ( newCount < oldCount )
	    {
	    *tp= firstParticule;
	    tp->tpStroff= newBegin;
	    tp->tpStrlen= len;
	    tp->tpKind= DOCkindTEXT;
	    tp->tpTextAttribute= addedAttr;
	    tp->tpPhysicalFont= -1;
#	    ifdef DEB_PARTICULES
	    label= "NW.";
#	    endif
	    }
	else{
	    if  ( len == 0 )
		{ /* LDEB(len); */ break;	}

	    tp= docInsertTextParticule( bi, part,
				newBegin, len, DOCkindTEXT, addedAttr );
	    if  ( ! tp )
		{ XDEB(tp); return -1;	}
#	    ifdef DEB_PARTICULES
	    label= "NW+";
#	    endif
	    }

#	ifdef DEB_PARTICULES
	DEBFUN( "%s %3d: [%4d..%4d] %s <%s> \"%.*s\" len= %d\n", label, part,
		    tp->tpStroff,
		    tp->tpStroff+ tp->tpStrlen,
		    docKindStr( tp->tpKind ),
		    docAttributeStr( tp->tpTextAttribute ),
		    (int)tp->tpStrlen,
		    bi->biParaString+ tp->tpStroff,
		    tp->tpStrlen );
#	endif

	newBegin += len; partShift++; newCount++; part++, tp++;
	}

    /*  18  */
    tpField= fieldParticules;
    for ( i= 0; i < fieldCount; tpField++, i++ )
	{
	if  ( tpField->tpKind == DOCkindUNKNOWN	)
	    { continue;	}

	if  ( tpField->tpKind == DOCkindFIELDSTART )
	    { addedAttr.taInField= 1;	}
	if  ( tpField->tpKind == DOCkindFIELDEND )
	    { addedAttr.taInField= 0;	}

	if  ( newCount < oldCount )
	    {
	    res= docReinsertFieldParticule( bi, newBegin, part,
							tpField, tp );
	    }
	else{
	    res= docReinsertFieldParticule( bi, newBegin, part,
					    tpField, (TextParticule *)0 );
	    }

	if  ( res < 0 )
	    { LDEB(res); return -1;	}

	if  ( res > 0 )
	    { partShift++; newCount++; part++, tp++;	}
	}

    /*  19  */
    if  ( newCount < oldCount )
	{
	if  ( newCount == 0				&&
	      bi->biParaParticuleCount == oldCount	)
	    {
	    /*  20  */
	    if  ( oldCount > 1 )
		{ docDeleteParticules( bi, 1, oldCount- 1 );	}

	    tp->tpKind= DOCkindTEXT;
	    tp->tpStrlen= 0;
	    tp->tpPixelsWide= 0;
	    tp->tpTextAttribute= addedAttr;
	    part= 1;
	    }
	else{ docDeleteParticules( bi, part, oldCount- newCount ); }
	}

    /*  21  */
    while( part < bi->biParaParticuleCount )
	{ tp->tpStroff += stroffShift; part++; tp++; }

#   ifdef DEB_PARTICULES
    DEBFUN( "\n" );

    part= 0; tp= bi->biParaParticules;
    while( part < bi->biParaParticuleCount )
	{
	DEBFUN( "=== %3d: [%4d..%4d] %s <%s> \"%.*s\"\n", part,
		    tp->tpStroff,
		    tp->tpStroff+ tp->tpStrlen,
		    docKindStr( tp->tpKind ),
		    docAttributeStr( tp->tpTextAttribute ),
		    (int)tp->tpStrlen,
		    bi->biParaString+ tp->tpStroff );
	part++; tp++;
	}
#   endif

    /*  22  */
    *pPartShift += partShift;
    *pStroffShift += stroffShift;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Append the tail of a paragraph to another paragraph.		*/
/*									*/
/*  1)  Claim memory for the appended contents.				*/
/*  2)  Claim memory for the appended particules.			*/
/*  3)  Append the contents of the first particule as text. This will	*/
/*	merge it with the last one of the target if necessary.		*/
/*  4)  Copy contents.							*/
/*  5)  Copy particules.						*/
/*									*/
/************************************************************************/
static int docParaAppend(	BufferDocument *	bdTo,
				BufferItem *		biTo,
				BufferItem *		biFrom,
				int			part,
				unsigned int		stroffFrom,
				void *			voiddisplay,
				DOC_CLOSE_OBJECT	closeObject )
    {
    TextParticule *	tpFrom= biFrom->biParaParticules+ part;
    TextParticule *	newTp;

    int			partShift= 0;
    int			stroffShift= 0;
    int			particuleCount;

    int			stroffTo;

    if  ( part < biFrom->biParaParticuleCount- 1		&&
	  stroffFrom == tpFrom->tpStroff+ tpFrom->tpStrlen	)
	{ tpFrom++; part++;	}

    stroffShift= biFrom->biParaStrlen- stroffFrom;

    /*  2  */
    particuleCount= biTo->biParaParticuleCount+
				    biFrom->biParaParticuleCount- part+ 1;
    newTp= (TextParticule *)realloc( biTo->biParaParticules,
				    particuleCount* sizeof(TextParticule) );
    if  ( ! newTp )
	{ LXDEB(particuleCount,newTp); return -1;	}
    biTo->biParaParticules= newTp;

    if  ( biTo->biParaParticuleCount > 0 )
	{
	newTp=	biTo->biParaParticules+ biTo->biParaParticuleCount- 1;

	/*  3  */
	if  ( tpFrom->tpKind == DOCkindTEXT				&&
	      docEqualTextAttributes( &(tpFrom->tpTextAttribute),
				      &(newTp->tpTextAttribute) )	)
	    {
	    int		ignored;

	    if  ( docParaReplaceText( bdTo, biTo, biTo->biParaParticuleCount- 1,
				biTo->biParaStrlen,
				&partShift, &ignored,
				biTo->biParaStrlen,
				biFrom->biParaString+ stroffFrom,
				tpFrom->tpStroff+ tpFrom->tpStrlen- stroffFrom,
				tpFrom->tpTextAttribute,
				voiddisplay, closeObject ) )
		{ LDEB(tpFrom->tpStrlen); return -1;	}

	    stroffFrom= tpFrom->tpStroff+ tpFrom->tpStrlen;
	    part++; tpFrom++;
	    }
	}

    /*  1  */
    if  ( docInflateTextString( biTo, biFrom->biParaStrlen- stroffFrom ) )
	{ LDEB(biFrom->biParaStrlen- stroffFrom); return -1;	}

    /*  4  */
    memcpy( biTo->biParaString+ biTo->biParaStrlen,
	    biFrom->biParaString+ stroffFrom,
	    biFrom->biParaStrlen- stroffFrom );
    stroffTo= biTo->biParaStrlen+biFrom->biParaStrlen- stroffFrom;
    biTo->biParaString[stroffTo]= '\0';

    /*  5  */
    stroffTo= biTo->biParaStrlen;

    while( part < biFrom->biParaParticuleCount )
	{
	int	len= tpFrom->tpStroff+ tpFrom->tpStrlen- stroffFrom;

	newTp= docCopyParticule( biTo, -1, stroffTo, len,
						    tpFrom->tpKind, tpFrom );

	if  ( ! newTp )
	    { LDEB(biTo->biParaParticuleCount); return -1;	}

	if  ( docCopyParticuleData( biTo, biFrom, newTp, tpFrom ) )
	    { LDEB(part); return -1;	}

	/*
	DEBFUN( "NEW %3d: [%4d..%4d] %s \"%.*s\"\n",
					    biTo->biParaParticuleCount- 1,
		    newTp->tpStroff,
		    newTp->tpStroff+ newTp->tpStrlen,
		    docKindStr( tpFrom->tpKind ),
		    (int)newTp->tpStrlen,
		    biTo->biParaString+ newTp->tpStroff );
	*/

	stroffTo += len; stroffFrom += len;
	tpFrom++; part++; partShift++;
	}

    biTo->biParaStrlen= stroffTo;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Replace a selection with a piece of text.				*/
/*									*/
/*  b)  Replace the tail of the beginning paragraph with the new text.	*/
/*  c)  Merge the two paragraphs.					*/
/*  d)  Erase all paragraphs after the beginning of the selection	*/
/*	to, including the end.						*/
/*  e)  Update the paragraph buffer and the particule administration.	*/
/*									*/
/************************************************************************/

int docReplaceSelection(	BufferDocument *	bd,
				const BufferSelection *	bs,
				int *			pPartShift,
				int *			pStroffShift,
				const unsigned char *	addedString,
				int			addedStrlen,
				TextAttribute		addedAttribute,
				void *			voiddisplay,
				DOC_CLOSE_OBJECT	closeObject )
    {
    /*  a  */
    if  ( bs->bsBegin.bpBi != bs->bsEnd.bpBi )
	{
	BufferItem *	nextBi;

	/*  b  */
	if  ( docParaReplaceText( bd, bs->bsBegin.bpBi,
		    bs->bsBegin.bpParticule, bs->bsBegin.bpStroff,
		    pPartShift, pStroffShift,
		    bs->bsBegin.bpBi->biParaStrlen,
		    addedString, addedStrlen, addedAttribute,
		    voiddisplay, closeObject ) )
	    { LDEB(addedStrlen); return -1;	}

	/*  c  */
	if  ( bs->bsEnd.bpStroff < bs->bsEnd.bpBi->biParaStrlen )
	    {
	    if  ( docParaAppend( bd, bs->bsBegin.bpBi, bs->bsEnd.bpBi,
				    bs->bsEnd.bpParticule, bs->bsEnd.bpStroff,
				    voiddisplay, closeObject ) )
		{ LDEB(addedStrlen); return -1;	}
	    }

	nextBi= docNextParagraph( bs->bsBegin.bpBi );
	if  ( ! nextBi )
	    { XDEB(nextBi); return -1;	}

	/*  d  */
	for (;;)
	    {
	    BufferItem *	par= nextBi->biParent;
	    BufferItem *	pp;

	    docCloseItemObjects( bd, nextBi, voiddisplay, closeObject );
	    docDeleteItem( bd, nextBi );

	    if  ( nextBi == bs->bsEnd.bpBi )
		{
		while( par && par->biGroupChildCount == 0 )
		    { pp= par->biParent; docDeleteItem( bd, par ); par= pp; }

		break;
		}

	    nextBi= docNextParagraph( bs->bsBegin.bpBi );
	    if  ( ! nextBi )
		{ XDEB(nextBi); return -1;	}

	    while( par && par->biGroupChildCount == 0 )
		{ pp= par->biParent; docDeleteItem( bd, par ); par= pp; }
	    }
	}
    else{
	/*  e  */
	if  ( docParaReplaceText( bd, bs->bsBegin.bpBi, bs->bsBegin.bpParticule,
			    bs->bsBegin.bpStroff,
			    pPartShift, pStroffShift,
			    bs->bsEnd.bpStroff,
			    addedString, addedStrlen, addedAttribute,
			    voiddisplay, closeObject ) )
	    { LDEB(addedStrlen); return -1;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Split a paragraph.							*/
/*									*/
/*  a)  Find the first particule that ends on or after the given	*/
/*	position.							*/
/*  1)  Insert a paragraph AFTER the current one.			*/
/*  1a) Copy indents and tab stops.					*/
/*  2)  Allocate space for the string of the successor.			*/
/*  3)  If the position is in the middle of a particule, split it.	*/
/*  4)  Move the rest of the particules to the new paragraph.		*/
/*									*/
/************************************************************************/

int docSplitParaItem(	BufferDocument *	bd,
			BufferItem **		pNewBi,
			BufferItem *		oldBi,
			int			stroff )
    {
    BufferItem *	newBi;
    int			part= 0;
    TextParticule *	tp;
    TextParticule *	newTp;

    int			splittingField= 0;
    int			splittingBookmark= 0;

    int			partShift= 0;
    int			newStrlen= 0;

    int			truncatedParticuleCount;

    static TextParticule *	fieldParticules;
    int				fieldCount= 0;

    if  ( docEditClaimFieldParticules( &fieldParticules,
						oldBi->biParaParticuleCount ) )
	{ LDEB(oldBi->biParaParticuleCount); return -1;	}

    /*  a,b  */
    if  ( docFindParticule( oldBi, stroff, &part, /* lastOne */ 0 ) )
	{ LLDEB(oldBi->biParaStrlen,stroff); return -1;	}

    tp= oldBi->biParaParticules+ part;

    /*  1  */
    newBi= docInsertItem( oldBi->biParent,
				oldBi->biNumberInParent+ 1, oldBi->biLevel );
    if  ( ! newBi )
	{ XDEB(newBi); return -1;	}

    /*  1a  */
    if  ( docCopyParagraphProperties( &(newBi->biParaProperties),
						&(oldBi->biParaProperties) ) )
	{ LDEB(oldBi->biParaTabCount); return -1;	}

    /*  2  */
    if  ( docInflateTextString( newBi, oldBi->biParaStrlen- stroff ) )
	{
	LLDEB(oldBi->biParaStrlen,stroff);
	docDeleteItems( bd, oldBi->biParent, oldBi->biNumberInParent+ 1, 1 );
	return -1;
	}

    memcpy( newBi->biParaString, oldBi->biParaString+ stroff,
						oldBi->biParaStrlen- stroff );
    newBi->biParaStrlen= oldBi->biParaStrlen- stroff;
    newBi->biParaString[newBi->biParaStrlen]= '\0';

    splittingField= docParticuleInField( oldBi, part );
    splittingBookmark= docParticuleInBookmark( oldBi, part );

    /*  3  */
    if  ( stroff > tp->tpStroff )
	{
	if  ( stroff == tp->tpStroff+ tp->tpStrlen )
	    {
	    newStrlen= 0;
	    truncatedParticuleCount= part+ 1;
	    part++; tp++;
	    }
	else{
	    newTp= docCopyParticule( newBi, partShift++,
				newStrlen, tp->tpStroff+ tp->tpStrlen- stroff,
				DOCkindTEXT, tp );
	    if  ( ! newTp )
		{ LDEB(partShift); return -1;	}

	    newTp->tpTextAttribute.taInField= 0;

	    newStrlen= tp->tpStroff+ tp->tpStrlen- stroff;
	    tp->tpStrlen= stroff- tp->tpStroff;
	    truncatedParticuleCount= part+ 1;
	    part++; tp++;
	    }
	}
    else{
	newTp= docCopyParticule( newBi, partShift++, newStrlen, tp->tpStrlen,
							    tp->tpKind, tp );
	if  ( ! newTp )
	    { LDEB(partShift); return -1;	}

	if  ( docCopyParticuleData( newBi, oldBi, newTp, tp ) )
	    { LDEB(part); return -1;	}

	newStrlen= tp->tpStrlen;
	truncatedParticuleCount= part;

	part++; tp++;
	}

    /*  4  */
    while( part < oldBi->biParaParticuleCount )
	{
	switch( tp->tpKind )
	    {
	    default:
		LDEB(tp->tpKind);
		newStrlen += tp->tpStrlen; part++; tp++;
		continue;

	    case DOCkindFIELDEND:
		fieldParticules[fieldCount++]= *tp;
		if  ( splittingField )
		    {
		    splittingField= 0;
		    newStrlen += tp->tpStrlen; part++; tp++;
		    continue;
		    }
		break;

	    case DOCkindBKMKEND:
		fieldParticules[fieldCount++]= *tp;
		if  ( splittingBookmark )
		    {
		    splittingBookmark= 0;
		    newStrlen += tp->tpStrlen; part++; tp++;
		    continue;
		    }
		break;

	    case DOCkindFIELDSTART:
	    case DOCkindBKMKSTART:
	    case DOCkindXE:
	    case DOCkindTC:
		break;

	    case DOCkindOBJECT:
	    case DOCkindTEXT:
	    case DOCkindTAB:
		break;
	    }

	newTp= docCopyParticule( newBi, partShift++, newStrlen, tp->tpStrlen,
							    tp->tpKind, tp );
	if  ( ! newTp )
	    { LXDEB(partShift,newTp); return -1;	}

	if  ( splittingField )
	    {
	    newTp->tpTextAttribute.taInField= 0;
	    newTp->tpTextAttribute.taIsUnderlined= 0;
	    }

	if  ( docCopyParticuleData( newBi, oldBi, newTp, tp ) )
	    { LDEB(part); return -1;	}

	newStrlen += tp->tpStrlen; part++; tp++;
	}

    if  ( fieldCount > 0 )
	{
	int		i;
	TextParticule *	tpField;

	fieldCount= docCleanupFields( bd, oldBi, fieldParticules, fieldCount );

	if  ( fieldCount < 0 )
	    { LDEB(fieldCount); return -1;	}

	tpField= fieldParticules;
	tp= oldBi->biParaParticules+ truncatedParticuleCount;
	for ( i= 0; i < fieldCount; tpField++, i++ )
	    {
	    int		res;

	    res= docReinsertFieldParticule( oldBi, stroff,
					truncatedParticuleCount, tpField, tp );

	    if  ( res < 0 )
		{ LDEB(res); return -1;	}

	    if  ( res > 0 )
		{ truncatedParticuleCount++; tp++;	}
	    }
	}

    oldBi->biParaParticuleCount= truncatedParticuleCount;
    oldBi->biParaStrlen= stroff;

    if  ( newBi->biParaParticuleCount == 0 )
	{
	tp= oldBi->biParaParticules+ oldBi->biParaParticuleCount- 1;

	newTp= docCopyParticule( newBi, -1, 0, 0, DOCkindTEXT, tp );
	if  ( ! newTp )
	    { LDEB(newBi->biParaParticuleCount); return -1;	}

	if  ( newTp->tpTextAttribute.taInField )
	    {
	    newTp->tpTextAttribute.taInField= 0;
	    newTp->tpTextAttribute.taIsUnderlined= 0;
	    }
	}

    if  ( oldBi->biParaParticuleCount == 0 )
	{
	tp= oldBi->biParaParticules;

	newTp= docCopyParticule( oldBi, -1, 0, 0, DOCkindTEXT,
						    oldBi->biParaParticules );
	if  ( ! newTp )
	    { LDEB(oldBi->biParaParticuleCount); return -1;	}

	if  ( newTp->tpTextAttribute.taInField )
	    {
	    newTp->tpTextAttribute.taInField= 0;
	    newTp->tpTextAttribute.taIsUnderlined= 0;
	    }
	}

    *pNewBi= newBi;
    return 0;
    }

int docSplitGroupItem(	BufferItem **		pNewBi,
			BufferItem *		oldBi,
			int			n )
    {
    int			i;
    BufferItem *	newBi;

    if  ( ! oldBi->biParent )
	{ XDEB(oldBi->biParent); return -1;	}
    if  ( n < 1 || n >= oldBi->biGroupChildCount )
	{ LLDEB(n,oldBi->biGroupChildCount); return -1;	}

    newBi= docInsertItem( oldBi->biParent,
				oldBi->biNumberInParent, oldBi->biLevel );
    if  ( ! newBi )
	{ XDEB(newBi); return -1;	}

    newBi->biY0= oldBi->biY0;
    newBi->biY1= oldBi->biY1;

    switch( oldBi->biLevel )
	{
	case DOClevSECT:
	    if  ( docCopySectProperties( &newBi->biSectProperties,
						&oldBi->biSectProperties ) )
		{ LDEB(1); return -1;	}

	    newBi->biY1= oldBi->biGroupChildren[n-1]->biY1;
	    oldBi->biY0= oldBi->biGroupChildren[n]->biY0;

	    break;
	case DOClevCELL:
	    if  ( docCopyCellProperties( &newBi->biCellProperties,
						&oldBi->biCellProperties ) )
		{ LDEB(1); return -1;	}

	    newBi->biCellProperties= oldBi->biCellProperties;

	    newBi->biY1= oldBi->biGroupChildren[n-1]->biY1;
	    oldBi->biY0= oldBi->biGroupChildren[n]->biY0;

	    break;
	case DOClevROW:
	    if  ( oldBi->biRowHasTableParagraphs )
		{
		LLDEB(oldBi->biLevel,oldBi->biRowHasTableParagraphs);
		return -1;
		}

	    if  ( docCopyRowProperties( &newBi->biRowProperties,
						&oldBi->biRowProperties ) )
		{ LDEB(1); return -1;	}

	    newBi->biY1= oldBi->biGroupChildren[n-1]->biY1;
	    oldBi->biY0= oldBi->biGroupChildren[n]->biY0;

	    break;
	default:
	    LDEB(oldBi->biLevel); return -1;
	}

    newBi->biGroupChildren= (BufferItem **)malloc( n* sizeof(BufferItem *) );
    if  ( ! newBi->biGroupChildren )
	{ XDEB(newBi->biGroupChildren); return -1;	}

    for ( i= 0; i < n; i++ )
	{
	newBi->biGroupChildren[i]= oldBi->biGroupChildren[i];
	newBi->biGroupChildren[i]->biParent= newBi;
	}

    newBi->biGroupChildCount= n;
    oldBi->biGroupChildCount -= n;

    for ( i= 0; i < oldBi->biGroupChildCount; i++ )
	{
	oldBi->biGroupChildren[i]= oldBi->biGroupChildren[i+ n];
	oldBi->biGroupChildren[i]->biNumberInParent -= n;
	}

    *pNewBi= newBi; return 0;
    }

/************************************************************************/
/*									*/
/*  Shift the tail of a document.					*/
/*									*/
/************************************************************************/
static void docShiftItemDown(	BufferItem *	bi,
				int		shift	)
    {
    int		i;

    bi->biY0 += shift;
    bi->biY1 += shift;

    if  ( bi->biLevel == DOClevPARA )
	{
	TextLine *	tl= bi->biParaLines;

	for ( i= 0; i < bi->biParaLineCount; tl++, i++ )
	    {
	    tl->tlY0 += shift;
	    tl->tlY  += shift;
	    tl->tlY1 += shift;
	    }
	}
    else{
	BufferItem **	child= bi->biGroupChildren;

	for ( i= 0; i < bi->biGroupChildCount; child++, i++ )
	    { docShiftItemDown( *child, shift ); }
	}
    }

/************************************************************************/
/*									*/
/*  Move to the first/last position.					*/
/*									*/
/************************************************************************/
int docFirstPosition(	BufferItem *		bi,
			BufferPosition *	bp	)
    {
    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA )
	    {
	    bp->bpBi= bi;
	    bp->bpStroff= 0;
	    bp->bpParticule= 0;
	    bp->bpLine= 0;

	    return 0;
	    }

	if  ( bi->biGroupChildCount == 0 )
	    { LDEB(bi->biGroupChildCount); return -1;	}

	bi= bi->biGroupChildren[0];
	}

    XDEB(bi); return -1;
    }

int docLastPosition(	BufferItem *		bi,
			BufferPosition *	bp	)
    {
    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA )
	    {
	    bp->bpBi= bi;
	    bp->bpStroff= bi->biParaStrlen;
	    bp->bpParticule= bi->biParaParticuleCount- 1;
	    bp->bpLine= bi->biParaLineCount- 1;

	    return 0;
	    }

	if  ( bi->biGroupChildCount == 0 )
	    { LDEB(bi->biGroupChildCount); return -1;	}

	bi= bi->biGroupChildren[bi->biGroupChildCount- 1];
	}

    XDEB(bi); return -1;
    }

/************************************************************************/
/*									*/
/*  Move to the next/previous position.					*/
/*									*/
/************************************************************************/
BufferItem *	docNextParagraph(	BufferItem *	bi	)
    {
    for (;;)
	{
	if  ( ! bi->biParent )
	    { return (BufferItem *)0;	}

	if  ( bi->biNumberInParent < bi->biParent->biGroupChildCount- 1 )
	    {
	    bi= bi->biParent->biGroupChildren[bi->biNumberInParent+ 1];

	    while( bi->biLevel != DOClevPARA	&&
		   bi->biGroupChildCount > 0	)
		{ bi= bi->biGroupChildren[0]; }

	    if  ( bi->biLevel == DOClevPARA )
		{ return bi;	}
	    }
	else{ bi= bi->biParent;	}
	}

    /*  Not reached..
    return bi;
    */
    }

BufferItem *	docPrevParagraph(	BufferItem *	bi	)
    {
    for (;;)
	{
	if  ( ! bi->biParent )
	    { return (BufferItem *)0;	}

	if  ( bi->biNumberInParent > 0 )
	    {
	    bi= bi->biParent->biGroupChildren[bi->biNumberInParent- 1];

	    while( bi->biLevel != DOClevPARA	&&
		   bi->biGroupChildCount > 0	)
		{ bi= bi->biGroupChildren[bi->biGroupChildCount- 1]; }

	    if  ( bi->biLevel == DOClevPARA )
		{ return bi;	}
	    }
	else{ bi= bi->biParent;	}
	}

    /*  Not reached..
    return bi;
    */
    }

int docNextPosition(	BufferPosition *	bp )
    {
    BufferItem *	bi= bp->bpBi;

    int			stroff= bp->bpStroff+ 1;

    int			lastOne= 1;

    while( bi )
	{
	if  ( stroff <= bi->biParaStrlen	)
	    {
	    docFindLineAndParticule( bi, stroff, bp, lastOne );

	    return 0;
	    }

	bi= docNextParagraph( bi );
	stroff= 0;

	if  ( bi && bi->biParaStrlen == 0 )
	    {
	    docFindLineAndParticule( bi, stroff, bp, lastOne );

	    return 0;
	    }
	}

    return -1;
    }

int docPrevPosition(	BufferPosition *	bp,
			int			lastOne )
    {
    BufferItem *	bi= bp->bpBi;

    int			stroff= bp->bpStroff- 1;

    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA		&&
	      stroff >= 0			)
	    {
	    docFindLineAndParticule( bi, stroff, bp, lastOne );

	    return 0;
	    }

	bi= docPrevParagraph( bi );
	if  ( bi )
	    { stroff= bi->biParaStrlen; }
	}

    return -1;
    }


int docPrevLine(	TextParticule **	pTp,
			TextLine **		pTl,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;
    BufferItem *	bi= bp->bpBi;

    int			part= bp->bpParticule;
    int			line= bp->bpLine- 1;

    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA		&&
	      line >= 0				)
	    {
	    tl= bi->biParaLines+ line;
	    tp= bi->biParaParticules+ part;

	    while( part > tl->tlFirstParticule		&&
		  part > 0				)
		{ tp--; part--;	}
	    
	    bp->bpBi= bi;
	    bp->bpStroff= tl->tlStroff;
	    bp->bpParticule= part;
	    bp->bpLine= line;

	    if  ( pTp )
		{ *pTp= tp;	}
	    if  ( pTl )
		{ *pTl= tl;	}

	    return 0;
	    }

	bi= docPrevParagraph( bi );
	if  ( bi )
	    {
	    part= bi->biParaParticuleCount- 1;
	    line= bi->biParaLineCount- 1;
	    }
	}

    return -1;
    }

int docNextLine(	TextParticule **	pTp,
			TextLine **		pTl,
			BufferPosition *	bp	)
    {
    TextLine *		tl;
    TextParticule *	tp;
    BufferItem *	bi= bp->bpBi;

    int			part= bp->bpParticule;
    int			line= bp->bpLine+ 1;

    while( bi )
	{
	if  ( bi->biLevel == DOClevPARA		&&
	      line < bi->biParaLineCount	)
	    {
	    tl= bi->biParaLines+ line;
	    tp= bi->biParaParticules+ part;

	    while( part < tl->tlFirstParticule		&&
		  part < bi->biParaParticuleCount	)
		{ tp++; part++;	}
	    
	    bp->bpBi= bi;
	    bp->bpStroff= tl->tlStroff;
	    bp->bpParticule= part;
	    bp->bpLine= line;
	    *pTp= tp;
	    *pTl= tl;

	    return 0;
	    }

	bi= docNextParagraph( bi );
	part= 0; line= 0;
	}

    return -1;
    }

/************************************************************************/
/*									*/
/*  Compare two positions in a document.				*/
/*									*/
/*  Comparison is in terms of the reading order in the document.	*/
/*									*/
/************************************************************************/

int docCompareItemPositions(	const BufferItem *	bi1,
				const BufferItem *	bi2 )
    {
    while( bi1->biParent && bi1->biLevel > bi2->biLevel )
	{ bi1= bi1->biParent;	}

    while( bi2->biParent && bi2->biLevel > bi1->biLevel )
	{ bi2= bi2->biParent;	}

    if  ( bi1 == bi2 )
	{ return 0;	}

    while( bi1->biParent			&&
	   bi2->biParent			&&
	   ( bi1->biParent != bi2->biParent )	)
	{ bi1= bi1->biParent; bi2= bi2->biParent; }

    if  ( bi1->biParent == bi2->biParent )
	{
	if  ( bi1->biNumberInParent > bi2->biNumberInParent )
	    { return  1;	}

	if  ( bi1->biNumberInParent < bi2->biNumberInParent )
	    { return -1;	}
	}

    return 0;
    }

int docComparePositions(	const BufferPosition *	bp1,
				const BufferPosition *	bp2,
				int			mindLine )
    {
    const BufferItem *	bi1= bp1->bpBi;
    const BufferItem *	bi2= bp2->bpBi;

    if  ( bi1 == bi2 )
	{
	if  ( bp1->bpStroff > bp2->bpStroff )
	    { return  1;	}

	if  ( bp1->bpStroff < bp2->bpStroff )
	    { return -1;	}

	if  ( mindLine )
	    {
	    if  ( bp1->bpLine > bp2->bpLine )
		{ return  1;	}

	    if  ( bp1->bpLine < bp2->bpLine )
		{ return -1;	}
	    }

	return 0;
	}

    return docCompareItemPositions( bi1, bi2 );
    }

/************************************************************************/
/*									*/
/*  Translate the result of a search in a paragraph to a selection.	*/
/*									*/
/************************************************************************/

void docSetSelection(	BufferSelection *	bs,
			BufferItem *		bi,
			int			direction,
			int			stroff,
			int			length	)
    {
    docFindLineAndParticule( bi, stroff, &(bs->bsBegin), 1 );
    docFindLineAndParticule( bi, stroff+ length, &(bs->bsEnd), 0 );

    bs->bsDirection= direction;

    bs->bsCol0= bs->bsCol1= -1;

    if  ( direction >= 0 )
	{ bs->bsAnchor= bs->bsBegin;	}
    else{ bs->bsAnchor= bs->bsEnd;	}

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a tab.							*/
/*									*/
/*  1)  Can be inserted before the current particule.			*/
/*  2)  Can be inserted after the current particule.			*/
/*  3)  First split the particule,					*/
/*  4)  Then insert between the two halves.				*/
/*									*/
/************************************************************************/
TextParticule * docParaSpecialParticule(BufferItem *		bi,
					int			kind,
					int			part,
					int			stroff,
					int *			pPartShift,
					int *			pStroffShift )
    {
    TextParticule *	tp= bi->biParaParticules+ part;
    TextParticule *	tpRet;

    if  ( docInflateTextString( bi, 1 )	)
	{ LLDEB(bi->biParaStrlen,1); return (TextParticule *)0;	}

    memmove( bi->biParaString+ stroff+ 1, bi->biParaString+ stroff,
						bi->biParaStrlen- stroff+ 1 );
    bi->biParaString[stroff]= ' ';
    bi->biParaStrlen++;

    if  ( bi->biParaParticuleCount == 1			&&
	  bi->biParaStrlen == 1				&&
	  bi->biParaParticules->tpKind == DOCkindTEXT	)
	{
	bi->biParaParticules->tpKind= kind;
	bi->biParaParticules->tpStrlen= 1;

	*pPartShift= 0; *pStroffShift= 1; return bi->biParaParticules;
	}

    /*  1  */
    if  ( stroff == tp->tpStroff )
	{
	tpRet= docCopyParticule( bi, part, stroff, 1, kind, tp );
	if  ( ! tpRet )
	    { LXDEB(part+1,tpRet); return (TextParticule *)0;	}

	tpRet->tpObjectNumber= -1; /* set later on */

	*pPartShift= 1;
	part++;
	}
    else{
	/*  2  */
	if  ( stroff == tp->tpStroff+ tp->tpStrlen	)
	    {
	    tpRet= docCopyParticule( bi, part+ 1, stroff, 1, kind, tp );
	    if  ( ! tpRet )
		{ LXDEB(part+1,tpRet); return (TextParticule *)0;	}

	    tpRet->tpObjectNumber= -1; /* set later on */

	    *pPartShift= 2;
	    part += 2;
	    }
	else{
	    /*  3  */
	    if  ( ! docCopyParticule( bi, part+ 1, stroff,
			tp->tpStroff+ tp->tpStrlen- stroff, DOCkindTEXT, tp ) )
		{ LDEB(part+1); return (TextParticule *)0;	}

	    tp= bi->biParaParticules+ part++;
	    tp->tpStrlen= stroff- tp->tpStroff;

	    /*  4  */
	    tpRet= docCopyParticule( bi, part, stroff, 1, kind, tp );
	    if  ( ! tpRet )
		{ LXDEB(part+1,tpRet); return (TextParticule *)0;	}

	    tpRet->tpObjectNumber= -1; /* set later on */

	    *pPartShift= 2;
	    part++;
	    }
	}

    tp= bi->biParaParticules+ part;
    while( part < bi->biParaParticuleCount )
	{ tp->tpStroff++; part++; tp++;	}

    *pStroffShift= 1;

    return tpRet;
    }


/************************************************************************/
/*									*/
/*  Close the objects in a buffer item.					*/
/*									*/
/************************************************************************/

static void docCloseParaObjects(	BufferDocument *	bd,
					BufferItem *		bi,
					void *			voiddisplay,
					DOC_CLOSE_OBJECT	closeObject )
    {
    int			part;
    TextParticule *	tp;

    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	if  ( tp->tpKind != DOCkindOBJECT	||
	      tp->tpPhysicalFont != 0		)
	    { continue;	}

	(*closeObject)( bd, bi, tp, voiddisplay );
	}

    return;
    }

void docCloseItemObjects(	BufferDocument *	bd,
				BufferItem *		bi,
				void *			voiddisplay,
				DOC_CLOSE_OBJECT	closeObject )
    {
    int		i;

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevROW:
	case DOClevCELL:
	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		docCloseItemObjects( bd, bi->biGroupChildren[i],
						    voiddisplay, closeObject );
		}
	    break;
	case DOClevPARA:
	    docCloseParaObjects( bd, bi, voiddisplay, closeObject );
	    break;
	default:
	    LDEB(bi->biLevel); return;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Substitute the page number in the text of an item.			*/
/*									*/
/*  1)  Though the routine works for multiple occurences of a page	*/
/*	number in a paragraph, the assumption is that there is only one.*/
/*	Substitution of multiple page numbers is simply inefficient as	*/
/*	the text is shifted for every page number.			*/
/*									*/
/************************************************************************/

int docSubstitutePageNumber(	BufferItem *		bi,
				int			pageNumber )
    {
    char		scratch[20];
    int			len;

    int			i;
    int			stroffShift;

    int			part;
    TextParticule *	tp;

    switch( bi->biLevel )
	{
	case DOClevDOC:
	case DOClevSECT:
	case DOClevROW:
	case DOClevCELL:
	    for ( i= 0; i < bi->biGroupChildCount; i++ )
		{
		if  ( docSubstitutePageNumber( bi->biGroupChildren[i],
								pageNumber ) )
		    { LDEB(pageNumber); return -1;	}
		}

	    return 0;
	case DOClevPARA:
	    break;
	default:
	    LDEB(bi->biLevel); return -1;
	}


    sprintf( scratch, "%d", pageNumber+ 1 );
    len= strlen( scratch );

    stroffShift= 0;
    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	DocumentField *		df;
	int			d;

	tp->tpStroff += stroffShift;

	if  ( tp->tpKind != DOCkindFIELDSTART )
	    { continue;	}

	df= bi->biParaFieldList.dflFields+ tp->tpObjectNumber;
	if  ( df->dfKind != DOCfkPAGEFIELD )
	    { continue;	}

	if  ( part > bi->biParaParticuleCount- 3 )
	    { LLDEB(part,bi->biParaParticuleCount); continue;	}

	if  ( bi->biParaParticules[part+ 2].tpKind != DOCkindFIELDEND )
	    { LDEB(bi->biParaParticules[part+ 2].tpKind); continue;	}

	/*  !!  */
	tp++; part++;
	tp->tpStroff += stroffShift;

	d= len- tp->tpStrlen;

	if  ( d > 0				&&
	      docInflateTextString( bi, d )	)
	    { LDEB(d); return -1;	}

	if  ( d != 0 )
	    {
	    int		past= tp->tpStroff+ tp->tpStrlen;

	    memmove( bi->biParaString+ past+ d, bi->biParaString+ past,
						bi->biParaStrlen- past+ 1 );
	    }

	memcpy( bi->biParaString+ tp->tpStroff+ stroffShift, scratch, len );

	tp->tpStrlen += d; /* = len */
	stroffShift += d;
	bi->biParaStrlen += d;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the range of rows for a 'table'				*/
/*									*/
/************************************************************************/
int docDelimitTable(	BufferItem *		paraBi,
			BufferItem **		pSectBi,
			int *			pCol,
			int *			pRow0,
			int *			pRow,
			int *			pRow1 )
    {
    BufferItem *	rowBi;
    BufferItem *	sectBi;
    int			col;

    int			row;
    int			row0;
    int			row1;

    if  ( paraBi->biLevel != DOClevPARA		||
	  ! paraBi->biParaInTable		)
	{ /* LLDEB(paraBi->biLevel,paraBi->biParaInTable); */ return -1; }

    rowBi= paraBi->biParent->biParent;
    sectBi= rowBi->biParent;
    col= paraBi->biParent->biNumberInParent;

    row= rowBi->biNumberInParent;
    row0= rowBi->biNumberInParent;
    row1= rowBi->biNumberInParent;

    while( row0 > 0							&&
	   docAlignedColumns( &(rowBi->biRowProperties),
		&(sectBi->biGroupChildren[row0-1]->biRowProperties) )	)
	{ row0--;	}

    while( row1 < sectBi->biGroupChildCount- 1				&&
	   docAlignedColumns( &(rowBi->biRowProperties),
		&(sectBi->biGroupChildren[row1+1]->biRowProperties) )	)
	{ row1++;	}

    *pSectBi= sectBi; *pCol= col;
    *pRow0= row0; *pRow= row; *pRow1= row1;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Are we in a something.						*/
/*									*/
/************************************************************************/

int docParticuleInField(	BufferItem *	bi,
				int		part )
    {
    TextParticule *	tp;

    part--;

    tp= bi->biParaParticules+ part;
    while( part >= 0 )
	{
	if  ( tp->tpKind == DOCkindFIELDEND )
	    { return 0;	}
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    { return 1;	}

	tp--; part--;
	}

    return 0;
    }

int docParticuleInBookmark(		BufferItem *	bi,
				int		part )
    {
    TextParticule *	tp;

    part--;

    tp= bi->biParaParticules+ part;
    while( part >= 0 )
	{
	if  ( tp->tpKind == DOCkindBKMKEND )
	    { return 0;	}
	if  ( tp->tpKind == DOCkindBKMKSTART )
	    { return 1;	}

	tp--; part--;
	}

    return 0;
    }
