/* vi:set ts=8 sts=0 sw=8:
 * $Id: doc.c,v 1.87 2000/03/01 01:46:03 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <sys/stat.h>
#include "main.h"
#include <gdk/gdkkeysyms.h>
#ifdef USE_GTKHTML
#include <gtkhtml/gtkhtml.h>
#endif
#if defined(APP_GNP) && defined(USE_GTKXMHTML)
#include <gtk-xmhtml/gtk-xmhtml.h>
#include "htmlview.h"
#endif
#include "gtkefilesel.h"
#include "doc.h"
#include "menu.h"
#include "misc.h"
#include "file.h"
#include "dialog.h"
#include "toolbar.h"
#include "msgbar.h"
#include "msgbox.h"
#include "prefs.h"
#include "recent.h"
#include "undo.h"
#ifdef APP_GNP
#include "project.h"
#include "print.h"
#endif

#include "gnpintl.h"

/*** global variables ***/
#if defined(USE_TOOLBARS) && defined(USE_DOCLIST)
toolbar_data_t dlw_tbdata[] = {
	{
		N_(" Save "),
		N_("Save file"),
		"dlw Save",
		"tb_save.xpm",
		GTK_SIGNAL_FUNC(doc_save_cb),
		NULL,
		"main_tb_save"
	},
	{
		N_(" Close "),
		N_("Close current file"),
		"dlw Close",
		"tb_cancel.xpm",
		GTK_SIGNAL_FUNC(doc_close_cb),
		NULL,
		"main_tb_close"
	},
	{
		N_(" Print "),
		N_("Print file"),
		"dlw Print",
		"tb_print.xpm",
		GTK_SIGNAL_FUNC(print_cb),
		NULL,
		"main_tb_print"
	},
	{
		" SPACE "
	},
	{
		N_(" Ok "),
		N_("Close list window"),
		"dlw Ok",
		"tb_exit.xpm",
		GTK_SIGNAL_FUNC(doc_list_destroy),
		NULL,
		"main_tb_exit"
	},
	{ NULL }
};
#endif	/* USE_TOOLBARS && USE_DOCLIST */


/*** external functions ***/
extern void win_set_title(doc_t *);


/*** local types ***/
#define INSERT	0
#define APPEND	1
#define FLW_NUM_COLUMNS	3

#ifdef USE_FILEINFO
typedef struct {
	GtkWidget *toplev;
	GSList *adv_list;
	GtkWidget *adv_button;
} doc_info_t;
#endif	/* USE_FILEINFO */


/*** local variables ***/
#ifdef APP_DEBUG
static unsigned nextdocid = 0;
#endif


/*** local function prototypes ***/
static void	doc_add_to_notebook(doc_t *d);
static bool_t	doc_close_execute(win_t *w, bool_t set_title);
static bool_t	doc_close_verify_yes(win_t *w);
static bool_t	doc_close_verify(win_t *w);
static void	doc_free(doc_t *);
#ifdef USE_FILEINFO
static void	doc_info_advanced(GtkWidget *wgt, gpointer cbdata);
static void	doc_info_close(GtkWidget *wgt, gpointer cbdata);
static void	doc_info_destroy(GtkWidget *wgt, gpointer cbdata);
static void	doc_info_table_add(GtkWidget *, doc_info_t *, char *, int, int,
				   int , int, bool_t);
#endif	/* USE_FILEINFO */
#ifdef USE_DOCLIST
static void	doc_list_add(win_t *w, doc_t *, int , char *, int, bool_t);
static int	doc_list_select(GtkWidget *, int, int, GdkEventButton *,
				gpointer);
static void	doc_list_update(win_t *w, doc_t *d, int rownum, char *text);
static void	doc_list_remove(win_t *w, int num);
#else
# define	doc_list_add(w, d, n, s, i, b)
# define	doc_list_select(wgt, a, b, e, gp)
# define	doc_list_update(w, d, r, t)
# define	doc_list_remove(w, num)
#endif	/* USE_DOCLIST */
static void	doc_open_destroy(GtkWidget *wgt, gpointer cbdata);
static void	doc_open_cancel(GtkWidget *wgt, gpointer cbdata);
static void	doc_open_ok(GtkWidget *wgt, gpointer cbdata);
static void	doc_open_ok_common(char *fname, win_t *w, gboolean update);
static doc_t *	doc_remove(win_t *w, bool_t set_title);
static void	doc_saveas_destroy(GtkWidget *wgt, gpointer cbdata);
static void	doc_saveas_ok(GtkWidget *wgt, gpointer data);
static void	doc_saveas_cancel_cb(GtkWidget *wgt, gpointer cbdata);
#if 0
static gboolean doc_key_press(GtkWidget *, GdkEventKey *, gpointer cbdata);
#endif
#ifdef GTK_HAVE_FEATURES_1_1_0
static void	doc_drag_data_get(GtkWidget *wgt, GdkDragContext *context,
				GtkSelectionData *selection_data, guint info,
				guint time, gpointer cbdata);
static void	doc_drag_data_delete(GtkWidget *wgt, GdkDragContext *context,
				gpointer cbdata);
#endif


/*** global function definitions ***/
/*
 * PUBLIC: doc_basefname
 *
 * convenience routine.  should probably be a macro, but oh well.
 */
char *
doc_basefname(doc_t *d)
{
	return (d->fname) ? (char *)my_basename(d->fname) : UNTITLED;
} /* doc_basefname */


#ifdef USE_RECENT
/*
 * PUBLIC: doc_by_name
 *
 * goes to a document by its filename.  if the document is not already opened,
 * then open it.  currently used only as a menu callback for the recent doc
 * list.  see recent_list_add() and recent_list_init().
 */
void
doc_by_name(GtkWidget *wgt, gpointer cbdata)
{
	GSList *dlp;
	char *full;
	int num;
	doc_t *d;
	doc_new_name_t *dnnp = (doc_new_name_t *)cbdata;

	g_assert(dnnp->w != NULL);

	/* see if we already have it open.  if so, go to it immediately */
	for (num = 0, dlp = dnnp->w->doclist; dlp; num++, dlp = dlp->next) {
		d = (doc_t *)(dlp->data);
		g_assert(d != NULL);
		if (d->fname) {
			/*
			 * since full pathnames are always stored in dnnp, we
			 * need to construct a full pathname for the doc's
			 * fname as well.
			 */
			full = file_full_pathname_make(d->fname);
			if (strcmp(full, dnnp->fname) == 0) {
				g_free(full);
				gtk_notebook_set_page(
						GTK_NOTEBOOK(dnnp->w->nb), num);
				break;
			}
			g_free(full);
		}
	} /* foreach dlp */

	/* not open, so open it, and close 'Untitled' if necessary */
	if (dlp == NULL) {
		d = DOC_CURRENT(dnnp->w);
		gtk_signal_disconnect_by_func(GTK_OBJECT(dnnp->w->nb),
				GTK_SIGNAL_FUNC(doc_switch_page), dnnp->w);

		if (dnnp->w->numdoc == 1 &&
		    !d->changed &&
		    !d->fname) {
			doc_close_common(dnnp->w, FALSE);
		}

		doc_new(dnnp->w, dnnp->fname, DocText,
			(UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));

		(void)gtk_signal_connect_after(GTK_OBJECT(dnnp->w->nb), 
					"switch_page",
					GTK_SIGNAL_FUNC(doc_switch_page), 
					dnnp->w);
	}
} /* doc_by_name */
#endif	/* USE_RECENT */


/*
 * PUBLIC: doc_changed_cb
 *
 * callback to indicate document changed.  also called by file_save_execute(),
 * since after saving a file, the document is no longer "changed".
 */
void
doc_changed_cb(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;

	d->changed = TRUE;
	gtk_signal_disconnect(GTK_OBJECT(d->data), d->changed_id);
	d->changed_id = FALSE;
	doc_info_label_update(d->w);
} /* doc_changed_cb */


/*
 * PUBLIC: doc_check_if_any_changed
 *
 * checks if any documents in the given window has been changed/updated.
 * returns 0 if none or changed, or docnum+1 of the docnum that was changed.
 */
int
doc_check_if_any_changed(win_t *w)
{
	GSList *dp;
	doc_t *d;
	int ret = 0;

	g_assert(w != NULL);
	dp = w->doclist;

	while (dp) {
		d = (doc_t *)(dp->data);
		g_assert(d != NULL);
		ret++;
		if (d->changed)
			return ret;
		dp = dp->next;
	}

	return 0;
} /* doc_check_if_any_changed */


/*
 * PUBLIC: doc_close_cb
 *
 * close file callback
 */
void
doc_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	g_assert(d != NULL);

	/* if all we have is a blank, Untitled doc, then return immediately */
	if (!d->changed && w->numdoc == 1 && !d->fname)
		return;

	(void)doc_close_common(w, TRUE);

	recent_list_write(w);

	if (w->numdoc < 1)
		doc_new(w, NULL, DocText,
			(UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));
} /* doc_close_cb */


/*
 * PUBLIC: doc_close_all_cb
 *
 * callback for closing all files
 */
void
doc_close_all_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;
	bool_t allclosed;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	g_assert(d != NULL);

	/* if all we have is a blank, Untitled doc, then return immediately */
	if (!d->changed && w->numdoc == 1 && !d->fname)
		return;

	msgbar_printf(w, _("Closing all files... (please wait)"));
	while (gtk_events_pending())
		gtk_main_iteration_do(TRUE);
	allclosed = doc_close_all_common(w);

	/*
	 * doc_close_all_common() won't create an "untitled" document, so
	 * do it here.
	 */
	if (w->numdoc < 1)
		doc_new(w, NULL, DocText,
			(UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));

	if (allclosed)
		msgbar_printf(w, _("All files closed..."));
} /* doc_close_all_cb */


/*
 * PUBLIC: doc_close_all_common
 *
 * common routine for closing all open files.  called from the menu callback
 * routine as well as win_close_common().  returns TRUE if all the files were
 * closed, FALSE if not.
 */
bool_t
doc_close_all_common(win_t *w)
{
	bool_t allclosed = TRUE;
	int anychanged = 0;

	g_assert(w != NULL);

	/*
	 * we temporarily disable the switch_page signal to avoid loading files
	 * that are going to be closed anyway.
	 */
	gtk_signal_disconnect_by_func(GTK_OBJECT(w->nb),
				      GTK_SIGNAL_FUNC(doc_switch_page), w);

#ifdef USE_DOCLIST
	if (w->dlw && GTK_WIDGET_VISIBLE(w->dlw))
		gtk_widget_hide(w->dlw);
#endif	/* USE_DOCLIST */

	/* first try to close any changed documents */
	while ((anychanged = doc_check_if_any_changed(w))) {
		gtk_notebook_set_page(GTK_NOTEBOOK(w->nb), anychanged - 1);
		allclosed = doc_close_common(w, FALSE);
		if (!allclosed)
			break;
	}

	/*
	 * if there are no more changed ones left, hide the notebook (so that
	 * it doesn't blink/switch constantly) and close all remaining docs.
	 */
	if (allclosed || !anychanged) {
		if (GTK_WIDGET_VISIBLE(w->nb))
			gtk_widget_hide(w->nb);

		while (w->numdoc > 0) {
			allclosed = doc_close_common(w, FALSE);
			gtk_main_iteration_do(FALSE);
			if (allclosed == FALSE)	/* not closed */
				break;
		}
	}

	/*
	 * now that we're done closing whatever we could, re-establish the
	 * switch_page signal on the notebook.
	 */
	(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				       GTK_SIGNAL_FUNC(doc_switch_page), w);

	if (!GTK_WIDGET_VISIBLE(w->nb))
		gtk_widget_show(w->nb);

	recent_list_write(w);

	return allclosed;
} /* doc_close_all_common */


/*
 * PUBLIC: doc_foreach
 *
 * given a window, traverses all documents of that window and calls func()
 * on each.
 */
void
doc_foreach(win_t *wp, void (*func)(void *))
{
	GSList *dlp;
	doc_t *dp;

	for (dlp = wp->doclist; dlp; dlp = dlp->next) {
		dp = (doc_t *)dlp->data;
		func(dp);
	}
} /* doc_foreach */


#if defined(USE_FILEINFO) || defined(GTK_HAVE_FEATURES_1_1_0)
/*
 * PUBLIC: doc_info_init
 *
 * creates the document info button
 */
void
doc_info_init(win_t *w, GtkWidget *parent)
{
#ifdef GTK_HAVE_FEATURES_1_1_0
#include <gtk/gtkselection.h>
	GtkTargetEntry dragtypes[] = {
		{ "STRING",     0, 0 },
		{ "text/uri-list", 0, 0 },
		{ "text/plain", 0, 0 }
	};
#endif

	w->docinfo = gtk_button_new();
#ifdef USE_FILEINFO
	w->docinfo_label = gtk_label_new(_(" File Info "));
#else
	w->docinfo_label = gtk_label_new(_(" DnD File "));
#endif	/* USE_FILEINFO */
	gtk_misc_set_alignment(GTK_MISC(w->docinfo_label), 0.5, 0.5);
	gtk_container_add(GTK_CONTAINER(w->docinfo), w->docinfo_label);

#if defined(USE_FILEINFO) || defined(GTK_HAVE_FEATURES_1_1_0)
	gtk_box_pack_end(GTK_BOX(parent), w->docinfo, FALSE, FALSE, 0);
# ifdef USE_FILEINFO
	(void)gtk_signal_connect(GTK_OBJECT(w->docinfo), "clicked",
				 GTK_SIGNAL_FUNC(doc_info_cb), w);
# endif	/* USE_FILEINFO */
#endif

	/* set according to preferences */
	if (IS_SHOW_MSGBAR())
		gtk_widget_show_all(w->docinfo);

#ifdef GTK_HAVE_FEATURES_1_1_0
	/* drag and drop */
	gtk_drag_source_set(w->docinfo, GDK_BUTTON1_MASK,
			    dragtypes, 3,
			    (GdkDragAction)(GDK_ACTION_COPY | GDK_ACTION_MOVE));
	gtk_signal_connect(GTK_OBJECT(w->docinfo), "drag_data_get",
			   GTK_SIGNAL_FUNC(doc_drag_data_get), w);
	gtk_signal_connect(GTK_OBJECT(w->docinfo), "drag_data_delete",
			   GTK_SIGNAL_FUNC(doc_drag_data_delete), NULL);
#endif

} /* docinfo_init */
#endif


#ifdef USE_FILEINFO
/*
 * PUBLIC: doc_info_cb
 *
 * callback invoked when docinfo button is clicked
 */
void
doc_info_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget *table, *tmp, *vbox;
	char *s;
	doc_info_t *di;
	doc_t *d;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	g_assert(d != NULL);
	di = (doc_info_t *)(d->docinfo);
	if (di) {
		(void)misc_show_and_raise(di->toplev);
		return;
	}

	di = g_new0(doc_info_t, 1);
	d->docinfo = di;
	di->toplev = gtk_dialog_new();
	(void)gtk_signal_connect(GTK_OBJECT(di->toplev), "destroy",
				 GTK_SIGNAL_FUNC(doc_info_destroy), d);
	gtk_window_set_title(GTK_WINDOW(di->toplev), _("File Information"));

	vbox = GTK_DIALOG(di->toplev)->vbox;
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);

	table = gtk_table_new(19, 2, FALSE);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, FALSE, 0);
	gtk_widget_show(table);

	/* blech, this is a whole lot of code just for some basic info */
	/* i wish there was a more efficient way of doing this */
	doc_info_table_add(table, di, _("Name:"),            0, 1, 0, 1, FALSE);
	doc_info_table_add(table, di, " ",                   0, 1, 1, 2, FALSE);
	doc_info_table_add(table, di, _("Modified:"),        0, 1, 2, 3, FALSE);
	doc_info_table_add(table, di, _("Read Only:"),       0, 1, 3, 4, FALSE);
	doc_info_table_add(table, di, " ",                   0, 1, 4, 5, FALSE);
	doc_info_table_add(table, di, _("Bytes in memory:"), 0, 1, 5, 6, FALSE);
	doc_info_table_add(table, di, _("Bytes on disk:"),   0, 1, 6, 7, FALSE);

	doc_info_table_add(table, di, doc_basefname(d),      1, 2, 0, 1, FALSE);
	doc_info_table_add(table, di, "                ",    1, 2, 1, 2, FALSE);
	doc_info_table_add(table, di,
		(d->changed) ? _("Yes") : _("No"),           1, 2, 2, 3, FALSE);
	doc_info_table_add(table, di, (d->sb) ?
		((file_is_readonly(d)) ?
		_("Yes") : _("No") ) : _("No"),              1, 2, 3, 4, FALSE);
	doc_info_table_add(table, di, " ",                   1, 2, 4, 5, FALSE);
	if (GTK_IS_TEXT(d->data))
		doc_info_table_add(table, di,
			ltoa((long)gtk_text_get_length(GTK_TEXT(d->data))),
			1, 2, 5, 6, FALSE);
	else
		doc_info_table_add(table, di, _("n/a"), 1, 2, 5, 6, FALSE);
	doc_info_table_add(table, di,
		(d->sb) ? ltoa((long)d->sb->st_size) : UNKNOWN,
		1, 2, 6, 7, FALSE);

	tmp = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), tmp, 0, 2, 7, 8);
	di->adv_list = g_slist_prepend(di->adv_list, tmp);

	doc_info_table_add(table, di, _("Full pathname:"), 0, 1, 8, 9, TRUE);
	doc_info_table_add(table, di, _("Inode:"),         0, 1, 9, 10, TRUE);
	doc_info_table_add(table, di, _("Mode:"),          0, 1, 10, 11, TRUE);
	doc_info_table_add(table, di, _("# of Links:"),    0, 1, 11, 12, TRUE);
	doc_info_table_add(table, di, _("UID:"),           0, 1, 12, 13, TRUE);
	doc_info_table_add(table, di, _("GID:"),           0, 1, 13, 14, TRUE);
	doc_info_table_add(table, di, _("Last accessed:"), 0, 1, 14, 15, TRUE);
	doc_info_table_add(table, di, _("Last modified:"), 0, 1, 15, 16, TRUE);
	doc_info_table_add(table, di, _("Last status change:"),0, 1, 16, 17, TRUE);
#ifndef __EMX__
		/* OS/2's EMX (Unix porting libs) does not have these */
	doc_info_table_add(table, di, _("File block size:"),   0, 1, 17, 18, TRUE);
	doc_info_table_add(table, di, _("Blocks allocated:"),  0, 1, 18, 19, TRUE);
#endif

	s = file_full_pathname_make(d->fname);
	doc_info_table_add(table, di, (s ? s : UNTITLED), 1, 2, 8, 9, TRUE);
	g_free(s);
	if (d->sb) {
		doc_info_table_add(table, di, ltoa((long)d->sb->st_ino),
				   1, 2, 9, 10, TRUE);
		doc_info_table_add(table, di, file_perm_string(d->sb),
				   1, 2, 10, 11, TRUE);
		doc_info_table_add(table, di, ltoa((long)d->sb->st_nlink),
				   1, 2, 11, 12, TRUE);
		doc_info_table_add(table, di, ltoa((long)d->sb->st_uid),
				   1, 2, 12, 13, TRUE);
		doc_info_table_add(table, di, ltoa((long)d->sb->st_gid),
				   1, 2, 13, 14, TRUE);
		doc_info_table_add(table, di, ttoa(d->sb->st_atime, FALSE),
				   1, 2, 14, 15, TRUE);
		doc_info_table_add(table, di, ttoa(d->sb->st_mtime, FALSE),
				   1, 2, 15, 16, TRUE);
		doc_info_table_add(table, di, ttoa(d->sb->st_ctime, FALSE),
				   1, 2, 16, 17, TRUE);
#ifndef __EMX__
		/* OS/2's EMX (Unix porting libs) does not have these */
		doc_info_table_add(table, di, ltoa((long)d->sb->st_blksize),
				   1, 2, 17, 18, TRUE);
		doc_info_table_add(table, di, ltoa((long)d->sb->st_blocks),
				   1, 2, 18, 19, TRUE);
#endif
	} else {
		int i;
		for (i = 9; i <= 18; i++)
			doc_info_table_add(
				table, di, UNKNOWN, 1, 2, i, i + 1, TRUE);
	}

	/* buttons */
	di->adv_button = misc_button_new_w_label(_("Advanced"), NULL,
					GTK_SIGNAL_FUNC(doc_info_advanced), di,
					GTK_DIALOG(di->toplev)->action_area,
					PACK_START | PACK_FILL | PACK_EXPAND |
					CANCEL_DEFAULT, 0);
	(void)misc_button_new_w_label(_(BUTTON_CLOSE), GNOME_STOCK_BUTTON_CLOSE,
				      GTK_SIGNAL_FUNC(doc_info_close), d,
				      GTK_DIALOG(di->toplev)->action_area,
				      PACK_START | PACK_FILL | PACK_EXPAND |
				      CANCEL_DEFAULT | GRAB_DEFAULT, 0);

	gtk_widget_show_all(GTK_DIALOG(di->toplev)->action_area);
	gtk_widget_show(di->toplev);
} /* doc_info_cb */


/*
 * PUBLIC: doc_info_label_update
 *
 * updates the file info button's label to reflect readonly and modified status
 * of the current document.
 */
void
doc_info_label_update(win_t *w)
{
	doc_t *d;
	char *buf;
	bool_t ro, isdir;

	d = DOC_CURRENT(w);
	buf = g_new(char, strlen(_(" File Info ")) + 8);
	strcpy(buf, _(" File Info "));

	ro = file_is_readonly(d);
	isdir = (d->sb && S_ISDIR(d->sb->st_mode));

	if (d->changed || ro || isdir) {
		strcat(buf, " (");
		if (d->changed)
			strcat(buf, "*");
		if (ro)
			strcat(buf, "R");
		if (isdir)
			strcat(buf, "D");
		strcat(buf, ")");
	}

	gtk_label_set(GTK_LABEL(w->docinfo_label), buf);
	g_free(buf);
} /* doc_info_label_update */
#endif	/* USE_FILEINFO */


#ifdef USE_DOCLIST
/*
 * PUBLIC: doc_list_add
 *
 * appends an entry to the files list window.
 */
void
doc_list_add(win_t *w, doc_t *d, int rownum, char *text, int which,
	     bool_t freezethaw)
{
	char numstr[4], sizestr[16];
	char *rownfo[FLW_NUM_COLUMNS];
	GtkCList *clist;
	int i;
	
	g_assert(w != NULL);
	if (w->dlw == NULL)
		return;

	clist = GTK_CLIST(w->dlw_data);

	/*
	 * currently, when there are no documents opened, we always create a
	 * new document called "Untitled".  if all we have "open" right now is
	 * the "Untitled" doc, we need to remove it before appending the new
	 * entry to the list.  (alternatively, we could also just set the clist
	 * field to the new column info too.)
	 */
	if (rownum == 0 && text != NULL) {
		/*
		 * this variable shouldn't need to be static/global (AFAIK),
		 * but i get a segv on my Linux box at home if i do NOT have
		 * compiler optimization present.  yet, this works fine on my
		 * Digital Unix box at work.  there's either something screwy
		 * with GTK or my Linux box (which is a little dated: a Debian
		 * 1.2 system with gcc 2.7.2.1 and libc5).
		 */
		static char *t;
		gtk_clist_get_text(clist, rownum, FnameCol, &t);
		if (t != NULL && strcmp(t, UNTITLED) == 0) {
			if (freezethaw)
				gtk_clist_freeze(clist);
			gtk_clist_remove(clist, rownum);
			if (freezethaw)
				gtk_clist_thaw(clist);
		}
	}

	g_snprintf(numstr, 4, "%d", rownum + 1);
	rownfo[0] = numstr;
	if (d->sb == NULL)
		rownfo[1] = UNKNOWN;
	else {
		g_snprintf(sizestr, 16, "%8lu ", (gulong)(d->sb->st_size));
		rownfo[1] = sizestr;
	}

	if (text && strlen(text) > 0) {
		rownfo[2] =  text;
	} else {
		rownfo[1] = UNKNOWN;
		rownfo[2] = UNTITLED;
	}
	if (freezethaw)
		gtk_clist_freeze(clist);
	if (which == INSERT)
		gtk_clist_insert(clist, rownum, rownfo);
	else
		gtk_clist_append(clist, rownfo);
	gtk_clist_select_row(clist, rownum, FnumCol);

	/* renumber any existing entries */
	if (clist->rows > 1) {
		for (i = 1; i < clist->rows; i++) {
			g_snprintf(numstr, 4, "%d", i + 1);
			gtk_clist_set_text(clist, i, FnumCol, numstr);
		}
	}

	if (freezethaw)
		gtk_clist_thaw(clist);
} /* doc_list_add */


/*
 * PUBLIC: doc_list_destroy
 *
 * zap!
 */
void
doc_list_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (w->dlw) {
		gtk_widget_destroy(w->dlw);
		w->dlw = NULL;
		w->dlw_data = NULL;
	}
} /* doc_list_destroy */


/*
 * PRIVATE: doc_list_remove
 *
 * removes an entry from the files list window.
 */
static void
doc_list_remove(win_t *w, int num)
{
	char buf[4];
	int i;
	GtkCList *clist;
	
	g_assert(w != NULL);
	if (w->dlw == NULL)
		return;

	clist = GTK_CLIST(w->dlw_data);
	g_assert(clist != NULL);
	gtk_clist_freeze(clist);
	gtk_clist_remove(clist, num);

	/* renumber any entries starting with the row just removed */
	if (clist->rows > 1) {
		for (i = num; i < clist->rows; i++) {
			g_snprintf(buf, 4, "%d", i + 1);
			gtk_clist_set_text(clist, i, FnumCol, buf);
		}
	}

	/* highlight the current document in the files list window */
	num = gtk_notebook_current_page(GTK_NOTEBOOK(w->nb));
	gtk_clist_select_row(clist, num, FnumCol);
	g_snprintf(buf, 4, "%d", num + 1);
	gtk_clist_set_text(clist, num, FnumCol, buf);
	gtk_clist_thaw(clist);
} /* doc_list_remove */


/*
 * PUBLIC: doc_list_show
 *
 * creates a new popup window containing a list of open files/documents for the
 * window from which it was invoked.
 */
void
doc_list_show(GtkWidget *wgt, gpointer cbdata)
{
	int num;
	GSList *dp;
	doc_t *d;
	GtkWidget *tmp, *vbox, *scrolled_win;
	char *titles[] = { N_(" # "), N_(" Size (bytes) "), N_(" File name ") };
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (misc_show_and_raise(w->dlw))
		return;

	for (num = 0; num < sizeof(titles)/sizeof(char*); num++)
		titles[num] = gettext(titles[num]);

	msgbar_printf(w, _("Creating doc list... (this might take a minute)"));

	w->dlw = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(w->dlw), _("Currently Open Files"));
	(void)gtk_signal_connect(GTK_OBJECT(w->dlw), "destroy",
				 GTK_SIGNAL_FUNC(doc_list_destroy), cbdata);

	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(w->dlw), vbox);
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);
	gtk_widget_show(vbox);

	w->dlw_data = gtk_clist_new_with_titles(FLW_NUM_COLUMNS, titles);
	gtk_widget_show(w->dlw_data);

#if 0
	GtkStyle *style;
	d = DOC_CURRENT(w);
	style = gtk_style_copy(GTK_WIDGET(d->data)->style);
	gdk_font_unref(style->font);
	gtk_widget_push_style(style);
#ifdef GTK_HAVE_FEATURES_1_1_0
	/* XXX - there seems to be a problem in gtk-1.0.x */
	gtk_widget_set_style(w->dlw_data, style);
#endif
	gtk_widget_pop_style();
#endif

	tmp = w->dlw_data;
	gtk_clist_column_titles_passive(GTK_CLIST(tmp));
#ifdef GTK_HAVE_FEATURES_1_1_0
	gtk_clist_set_column_auto_resize(GTK_CLIST(tmp), FnumCol, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(tmp), FsizeCol, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(tmp), FnameCol, TRUE);
	gtk_clist_set_column_min_width(GTK_CLIST(tmp), FnameCol, 64);
#else
	gtk_clist_set_column_width(GTK_CLIST(tmp), FnumCol, 25);
	gtk_clist_set_column_width(GTK_CLIST(tmp), FsizeCol, 90);
#endif
	scrolled_win = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(scrolled_win), tmp);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled_win, TRUE, TRUE, 0);
	gtk_widget_show(scrolled_win);

	gtk_clist_freeze(GTK_CLIST(w->dlw_data));
	for (num = 0, dp = w->doclist; dp; num++, dp = dp->next) {
		d = (doc_t *)(dp->data);
		doc_list_add(w, d, num, doc_basefname(d), APPEND, FALSE);
		gtk_main_iteration_do(FALSE);
	}
	gtk_clist_thaw(GTK_CLIST(w->dlw_data));

	/*
	 * MUST connect this signal **after** adding fnames to the clist, else
	 * we keep changing the cur doc to the newly added file.
	 */
	(void)gtk_signal_connect(GTK_OBJECT(w->dlw_data), "select_row",
				 GTK_SIGNAL_FUNC(doc_list_select), w);

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, TRUE, 0);
	gtk_widget_show(tmp);

	toolbar_create(w, vbox, dlw_tbdata, NULL);

	/*
	 * there should be a way to auto-detect how wide to make these widgets
	 * instead of having to use arbitrary values
	 */
	gtk_widget_set_usize(w->dlw, 250, 350);

	/*
	 * since we're currently inserting documents at the head of the
	 * notebook, the notebook list and the list here are in reverse order.
	 * hence, manually select the actual current file.
	 */
	num = g_slist_index(w->doclist, DOC_CURRENT(w));
	gtk_clist_select_row(GTK_CLIST(w->dlw_data), num, FnumCol);

	gtk_widget_show(w->dlw);
} /* doc_list_show */
#endif	/* USE_DOCLIST */


/*
 * PUBLIC: doc_new
 *
 * creates a new document
 *
 * flags: UPDATE_MSGBAR, DO_LOAD, UPDATE_TITLE
 */
void
doc_new(win_t *w, char *fname, doc_type_t doctype, long flags)
{
	char *cwd, *fn;
	doc_t *d;

	d = g_new0(doc_t, 1);
#ifdef APP_DEBUG
	d->docid = nextdocid++;
#endif
	d->w = w;
	d->fname = g_strdup(fname);
	d->sb = file_get_stats(d->fname);

	if (flags & FORCE_READONLY)
		d->force_ro = TRUE;
#ifdef WANT_DOCUNLOAD
	if (flags & NEVER_UNLOAD)
		d->never_unload = TRUE;
#endif

	w->curdoc = d;
	w->numdoc++;

	switch (doctype) {
	case DocText:
		/* text widget */
		d->data = gtk_text_new(NULL, NULL);

		doc_tab_stop_set((void *)d);

		gtk_text_set_editable(GTK_TEXT(d->data), TRUE);
		gtk_text_set_point(GTK_TEXT(d->data), 0);
		gtk_text_set_word_wrap(GTK_TEXT(d->data), IS_USE_WORDWRAP());
		gtk_widget_show(d->data);
#if 0
		gtk_signal_connect(GTK_OBJECT(d->data), "key_press_event",
				   GTK_SIGNAL_FUNC(doc_key_press), NULL);
#endif
#if defined(USE_UNDOREDO) && defined(GTK_HAVE_FEATURES_1_1_0)
		d->ins_txt_id = gtk_signal_connect(GTK_OBJECT(d->data),
					"insert_text",
					GTK_SIGNAL_FUNC(doc_insert_text_cb), d);
		d->del_txt_id = gtk_signal_connect(GTK_OBJECT(d->data),
					"delete_text",
					GTK_SIGNAL_FUNC(doc_delete_text_cb), d);
#endif	/* USE_UNDOREDO */

		/* quickmenu */
		(void)gtk_signal_connect_object(GTK_OBJECT(d->data), "event",
						GTK_SIGNAL_FUNC(quickmenu_show),
						GTK_OBJECT(w->quickmenu));
		break;
#ifdef USE_GTKXMHTML
	case DocXmHtml:
		d->data = gtk_xmhtml_new();
		gtk_widget_show(d->data);
		(void)file_open_execute(d);
		break;
#endif
#ifdef USE_GTKHTML
	case DocGtkHtml:
		d->data = gtk_html_new(NULL, NULL);
		gtk_signal_connect(GTK_OBJECT(d->data), "url_requested",
				   GTK_SIGNAL_FUNC(html_view_url_requested),
				   NULL);
		gtk_widget_show(d->data);
		(void)file_open_execute(d);
		break;
#endif
	default:
		g_warning("doc_new: unsupported doctype = %d\n", doctype);
		break;
	} /* switch (doctype) */

	/* tab label */
	d->tablabel = gtk_label_new(doc_basefname(d));
	GTK_WIDGET_UNSET_FLAGS(d->tablabel, GTK_CAN_FOCUS);
	if (IS_SHOW_TABS())
		gtk_widget_show(d->tablabel);

	/* add to notebook, update doclist, and update winlist popup */
	doc_add_to_notebook(d);
	win_list_set_curdoc(w);
	win_list_set_numdoc(w);

	if (GTK_IS_TEXT(d->data)) {
		/* now open the file and stuff the text into the text widget */
		gtk_widget_realize(d->data);
		if ((flags & DO_LOAD) && d->sb) {
			if (S_ISDIR(d->sb->st_mode)) {
				GNPDBG_DOC(( _("doc_new: '%s' is a directory\n"),
					    d->fname));
				msgbox_printf( _("doc_new: '%s' is a directory\n"),
					      d->fname);
				if (flags & UPDATE_MSGBAR)
					msgbar_printf(w, _("'%s' is a directory"),
						      d->fname);
			} else {
				GNPDBG_DOC(( _("doc_new: loading '%s'\n"),
					    d->fname));
				gtk_text_freeze(GTK_TEXT(d->data));
#ifdef WANT_DOCUNLOAD
				if (file_open_execute(d) == 0) {
					d->loaded = TRUE;
					if (d->adjvalue) {
						gtk_adjustment_set_value(
							GTK_ADJUSTMENT(
								GTK_TEXT(
								d->data)->vadj),
							d->adjvalue);
					}
				}
#else
				(void)file_open_execute(d);
#endif

				/* hack alert!  it seems that even if the text
				 * properties (font, fg and bg colors) are
				 * updated, we can't use them in
				 * gtk_text_insert because the text widget
				 * itself already has the settings which were
				 * loaded at app startup.  so by putting the
				 * redraw here, we force it to load whatever
				 * the current settings are, as opposed to what
				 * was found during app startup.  sometimes,
				 * the way GTK does is just totally
				 * non-intuitive (chalk another one up to lack
				 * of documentation). */
				doc_redraw(d);
				gtk_text_thaw(GTK_TEXT(d->data));
			}
		} else
			doc_redraw(d);
	}

	/* various status updates */
	if (flags & UPDATE_TITLE)
		win_set_title(d);

	doc_list_add(w, d, 0, doc_basefname(d), INSERT, TRUE);

	/* skip printing cwd in full pathname */
	fn = d->fname;
	if ((cwd = getcwd(NULL, 0)) != NULL) {
		if (fn && strstr(fn, cwd))
			fn = d->fname + strlen(cwd) + 1;
		free(cwd);
	}

	/* changed */
	d->changed = FALSE;
	if (GTK_IS_TEXT(d->data)) {
		d->changed_id = gtk_signal_connect(GTK_OBJECT(d->data),
					"changed",
					GTK_SIGNAL_FUNC(doc_changed_cb), d);
	}
#ifdef WANT_DOCUNLOAD
	d->loaded = (flags & DO_LOAD) ? TRUE : FALSE;
#endif

#if defined(GTK_HAVE_FEATURES_1_1_0) && defined(USE_UNDOREDO) && \
    defined(USE_TOOLBARS)
	if (w->main_tb) {
		tb_item_enable(w->main_tb, MAIN_TB_STR_UNDO,
			       (d->undolist) ? TRUE : FALSE);
		tb_item_enable(w->main_tb, MAIN_TB_STR_REDO,
			       (d->redolist) ? TRUE : FALSE);
	}
#endif	/* USE_UNDOREDO */

	if (flags & UPDATE_MSGBAR)
		msgbar_printf(w, _("Opened %s"), fn ? fn : UNTITLED);
	msgbox_printf( _("opened %s"), fn ? fn : UNTITLED);
} /* doc_new */


void
doc_tab_stop_set(void *data)
{
	doc_t *d = (doc_t *)data;
	GtkText *text = GTK_TEXT(d->data);

	/* this is a hack to get consistent tab stops for the text widget */
	text->default_tab_width = prefs.tab_stop;
	text->tab_stops = g_list_remove(text->tab_stops, text->tab_stops->data);
	text->tab_stops = g_list_remove(text->tab_stops, text->tab_stops->data);
	text->tab_stops = NULL;
	text->tab_stops = g_list_prepend(text->tab_stops,
					 GINT_TO_POINTER(prefs.tab_stop));
	text->tab_stops = g_list_prepend(text->tab_stops,
					 GINT_TO_POINTER(prefs.tab_stop));
} /* doc_tab_stop_set */


#ifdef GTK_HAVE_FEATURES_1_1_0
/*
 * PUBLIC: doc_insert_text_cb
 *
 * callback for text widget for text insertion from keyboard typing.  note that
 * this routine and the next one (doc_delete_text_cb()), gets invoked for EVERY
 * key typed by the user.
 */
void
doc_insert_text_cb(GtkEditable *editable, const char *text, int len,
		int *pos, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;
	char *buf;

	g_assert(d != NULL);
	buf = g_new0(char, len + 1);
	strncpy(buf, text, len);
	GNPDBG_UNDO(("doc_insert_text_cb: '%s'\n", buf));
	undo_list_add(d, buf, *pos, *pos + len, UndoInsert);
} /* doc_insert_text_cb */


/*
 * PUBLIC: doc_delete_text_cb
 *
 * callback for text widget for text deletion from keyboard typing
 */
void
doc_delete_text_cb(GtkEditable *editable, int start, int end, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;
	char *text;

	g_assert(d != NULL);
	text = gtk_editable_get_chars(editable, start, end);
	undo_list_add(d, text, start, end, UndoDelete);
} /* doc_delete_text_cb */


static void
doc_drag_data_get(GtkWidget *wgt, GdkDragContext *context,
		  GtkSelectionData *selection_data, guint info,
		  guint time, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d = DOC_CURRENT(w);
	char *full, *uri;

	/*
	 * build a string with the URI-list representing the current file.
	 *
	 * URI: "file:" + fullpathname + "\r\n"
	 */
	if (d->fname) {
		full = file_full_pathname_make(d->fname);
		uri = g_new(char, 5 + strlen(full) + 3);
		sprintf(uri, "file:%s\r\n", full);
		g_free(full);
		gtk_selection_data_set(selection_data, selection_data->target,
				       8, (guchar *)uri, strlen(uri));
		g_free(uri);
	} else
		gtk_selection_data_set(selection_data, selection_data->target,
				       8, NULL, 0);
} /* doc_drag_data_get */


static void
doc_drag_data_delete(GtkWidget *wgt, GdkDragContext *context, gpointer cbdata)
{
	g_print("delete the data\n");
} /* doc_drag_data_delete */
#endif

#if 0
static gboolean
doc_key_press(GtkWidget *txt, GdkEventKey *event, gpointer cbdata)
{
	char key = (char)(event->keyval);

	if ((event->state & GDK_MOD1_MASK)) {
		printf("alt-%c hit\n", key);
		return TRUE;
	}
	return FALSE;
}
#endif


/*
 * PUBLIC: doc_new_cb
 *
 * new document (file) callback
 */
void
doc_new_cb(GtkWidget *wgt, gpointer cbdata)
{
	doc_new((win_t *)cbdata, NULL, DocText,
		(UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));
} /* doc_new_cb */


/*
 * PUBLIC: doc_open_cb
 *
 * file open callback to create the file selection dialog box.
 */
void
doc_open_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	if (misc_show_and_raise(w->filesel))
		return;

	g_assert(!w->filesel);
	w->filesel = misc_open_dialog_create((void *)w,
					     _("Open File..."),
					     doc_open_ok,
					     doc_open_cancel,
					     doc_open_destroy);
	gtk_widget_show(w->filesel);
} /* doc_open_cb */


/*
 * PUBLIC: doc_open_in_new_win
 *
 * closes current doc and opens it in a new window.  it might be slower this
 * way because we have to reread the file from disk again, but there's no easy
 * way around reparenting all the widgets associated with a doc_t structure.
 * look at it this way: at least we're not copying the entire contents of the
 * text buffer and consuming a boatload of memory (this braindead method is
 * what's used in gedit).
 */
void
doc_open_in_new_win(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	win_t *neww;
	doc_t *d;
	char *fname;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	if (!d->changed && !d->fname)
		return;

	fname = g_strdup(d->fname);
	if (!doc_close_common(w, TRUE)) {
		g_free(fname);
		return;
	}
	if (w->numdoc < 1)
		doc_new(w, NULL, DocText,
			(UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));
	recent_list_write(w);

	neww = win_new();
	doc_new(neww, fname, DocText, (UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));
	recent_list_add(neww, fname);
	g_free(fname);

	gtk_widget_show(neww->toplev);
	(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				       GTK_SIGNAL_FUNC(doc_switch_page), w);

	d = DOC_CURRENT(neww);
	recent_list_add(neww, d->fname);

	msgbar_printf(neww, _("Opened %s"), doc_basefname(d));
	msgbox_printf( _("opened %s"), doc_basefname(d));
} /* doc_open_in_new_win */


void
doc_flags_set(doc_t *d, long flags)
{
	if (flags & FORCE_READONLY)
		d->force_ro = TRUE;
#ifdef WANT_DOCUNLOAD
	if (flags & NEVER_UNLOAD)
		d->never_unload = TRUE;
#endif
} /* doc_flags_set */


/*
 * PUBLIC: doc_redraw
 *
 * redraws the widgets for a text document
 */
void
doc_redraw(void *cbdata)
{
#ifdef GTK_HAVE_FEATURES_1_1_0
	doc_t *d = (doc_t *)cbdata;

	prefs_update_text_widget_style(d->data);
#endif
} /* doc_redraw */


/*
 * PUBLIC: doc_save_cb
 *
 * file save callback
 */
void
doc_save_cb(GtkWidget *wgt, gpointer cbdata)
{
	(void)doc_save_common((win_t *)cbdata);
} /* doc_save_cb */


/*
 * PUBLIC: doc_save_common
 *
 * common routine for saving a document.  called from the menu callback as well
 * as get_filename(), when saving a file before printing.
 */
bool_t
doc_save_common(win_t *w)
{
	doc_t *d;
	bool_t ret;
	int num;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);

	if (!d->fname)
		ret = doc_saveas_common(w);
	else
		ret = file_save_execute(d, FALSE);

	if (ret) {
		num = g_slist_index(w->doclist, DOC_CURRENT(w));
		doc_list_update(w, d, num, d->fname); 
		doc_info_label_update(w);
	}

	recent_list_write(w);

	return ret;
} /* doc_save_common */


/*
 * PUBLIC: doc_saveas_cb
 *
 * file save-as callback
 */
void
doc_saveas_cb(GtkWidget *wgt, gpointer cbdata)
{
	(void)doc_saveas_common((win_t *)cbdata);
} /* doc_save_cb */


/*
 * PUBLIC: doc_saveas_common
 *
 * common routine for doing a document save-as.  called from the menu callback
 * as well as get_filename(), when saving a file before printing.
 */
bool_t
doc_saveas_common(win_t *w)
{
	GtkWidget *tmp;
	doc_t *d;
	guint last_id;

	if (misc_show_and_raise(w->saveas)) {
		/* allow only one saveas box.  wait until that one goes away */
		while (w->saveas)
			gtk_main_iteration_do(TRUE);
	}

	d = DOC_CURRENT(w);
	last_id = d->changed_id;

	w->saveas = gtk_efilesel_new( _("Save As..."));
	gtk_efilesel_set_mode(GTK_EFILESEL(w->saveas), GTK_SELECTION_SINGLE);
	tmp = w->saveas;
	(void)gtk_signal_connect(
			GTK_OBJECT(GTK_EFILESEL(tmp)->ok_button),
			"clicked", GTK_SIGNAL_FUNC(doc_saveas_ok), w);
	(void)gtk_signal_connect(
			GTK_OBJECT(GTK_EFILESEL(tmp)->cancel_button),
			"clicked", GTK_SIGNAL_FUNC(doc_saveas_cancel_cb), w);
	(void)gtk_signal_connect(GTK_OBJECT(tmp), "destroy",
				 GTK_SIGNAL_FUNC(doc_saveas_destroy), w);

	gtk_widget_show(w->saveas);

	while (w->saveas != NULL)
		gtk_main_iteration_do(TRUE);

	return (d->changed_id != last_id);
} /* doc_saveas_common */


void
doc_load(doc_t *d)
{
#ifdef WANT_DOCUNLOAD
	if (d && !d->changed && !d->loaded) {
#else
	if (d && !d->changed) {
#endif
		if (GTK_IS_TEXT(d->data)) {
			GNPDBG_FILE(("doc_load: loading file\n"));
			if (file_open_execute(d) == 0) {
#ifdef WANT_DOCUNLOAD
				if (d->adjvalue) {
					gtk_adjustment_set_value(
						GTK_ADJUSTMENT(GTK_TEXT(
							d->data)->vadj),
						d->adjvalue);
				}
#endif
			}
		}
#ifdef WANT_DOCUNLOAD
		d->loaded = TRUE;
#endif
	}
} /* doc_load */


void
doc_unload(doc_t *d)
{
#ifdef WANT_DOCUNLOAD
	if (d && !d->changed && d->loaded && !d->never_unload) {
#else
	if (d && !d->changed) {
#endif
		if (GTK_IS_TEXT(d->data)) {
			GNPDBG_FILE(("doc_unload: UNloading file\n"));
			gtk_text_freeze(GTK_TEXT(d->data));
#ifdef WANT_DOCUNLOAD
			d->adjvalue =
				GTK_ADJUSTMENT(GTK_TEXT(d->data)->vadj)->value;
#endif
			gtk_text_set_point(GTK_TEXT(d->data), 0);
			gtk_text_forward_delete(
				GTK_TEXT(d->data),
				gtk_text_get_length(GTK_TEXT(d->data)));
			gtk_text_thaw(GTK_TEXT(d->data));

#if defined(USE_UNDOREDO) && defined(GTK_HAVE_FEATURES_1_1_0)
			/* since we're unloading, blow away the undo and redo
			 * lists */
			d->undolist = undo_list_delete(d->undolist);
			d->undotail = NULL;
			g_assert(d->undolist == NULL);
			d->redolist = undo_list_delete(d->redolist);
			d->redotail = NULL;
			g_assert(d->redolist == NULL);
#endif	/* USE_UNDOREDO */
		}
#ifdef WANT_DOCUNLOAD
		d->loaded = FALSE;
#endif
	}
} /* doc_unload */


/*
 * PUBLIC: doc_switch_page
 *
 * callback invoked when a switch page has occured on the document notebook.
 *
 * NOTE regarding loading/unloading: the reasoning for loading an unloading is
 * to reduce memory usage.  the strategy is to only keep the current (active)
 * file in memory.  all other files do NOT get loaded from disk until they are
 * the active file.  hence, whenever switching to a different file this needs
 * to be done:
 *
 *	- before switching to the new page, if the file has NOT been changed
 *	AND it has been loaded, then unload it and clear the "loaded" flag.
 *	otherwise, do nothing and keep it in memory.
 *
 *	- after switching to the new page, if the file has NOT been changed AND
 *	it isn't already loaded, then load it and set the "loaded" flag.
 *
 * the downside to this is that if you are switching to a very very large file,
 * it may take a while to load.  but then again, i think it's better to do it
 * now instead of at startup, because otherwise, it will take forever to start.
 *
 * also NOTE, that by doing this, when the user selects to close a window or to
 * close all files, we disable this signal handler temporarily before closing
 * the files, then reenable it after we're done.  see doc_close_all_common()
 * for more details.
 */
void
doc_switch_page(GtkWidget *wgt, GtkNotebookPage *page, int num, gpointer cbdata)
{
	doc_t *d;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (w->doclist == NULL)
		return;

	if (!GTK_WIDGET_REALIZED(w->toplev))
		return;

#ifdef WANT_DOCUNLOAD
	/* if we can, unload the file before going to the new document/page */
	d = DOC_CURRENT(w);
	doc_unload(d);

	/* change to the new document */
	w->curdoc = g_slist_nth_data(w->doclist, num);
	d = DOC_CURRENT(w);

	/* load the file */
	doc_load(d);
#endif

#ifdef USE_DOCLIST
	if (w->dlw_data)
		gtk_clist_select_row(GTK_CLIST(w->dlw_data), num, FnumCol);
#endif	/* USE_DOCLIST */
	if ((d = DOC_CURRENT(w)) != NULL) {
		gtk_widget_grab_focus(d->data);
		win_set_title(d);
		win_list_set_curdoc(w);
		doc_info_label_update(w);
	}

#if defined(GTK_HAVE_FEATURES_1_1_0) && defined(USE_UNDOREDO) && \
    defined(USE_TOOLBARS)
	if (w->main_tb) {
		tb_item_enable(w->main_tb, MAIN_TB_STR_UNDO,
			       (d->undolist) ? TRUE : FALSE);
		tb_item_enable(w->main_tb, MAIN_TB_STR_REDO,
			       (d->redolist) ? TRUE : FALSE);
	}
#endif	/* USE_UNDOREDO */
} /* doc_switch_page */


/*** local function definitions ***/
/*
 * PRIVATE: doc_add_to_notebook
 *
 * add a newly created document to the notebook
 */
static void
doc_add_to_notebook(doc_t *d)
{
	GtkWidget *vsb, *vbox2;
	GtkWidget *table, *vbox1;
	GtkStyle *style;

	vbox1 = gtk_vbox_new(TRUE, TRUE);
	gtk_widget_show(vbox1);

	table = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(table), 0, 2);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 2);
	gtk_box_pack_start(GTK_BOX(vbox1), table, TRUE, TRUE, 1);
	gtk_widget_show(table);

	gtk_table_attach_defaults(GTK_TABLE(table), d->data, 0, 1, 0, 1);

	/*
	 * it seems to be needed to avoid a timing(?) problem when changing
	 * font and color settings in the text widget.
	 */
	style = gtk_style_new();
	gtk_widget_set_style(GTK_WIDGET(d->data), style);
	gtk_widget_set_rc_style(GTK_WIDGET(d->data));
	gtk_widget_ensure_style(GTK_WIDGET(d->data));
	gtk_style_unref(style);

	if (GTK_IS_TEXT(d->data)) {
		vbox2 = gtk_vbox_new(TRUE, TRUE);
		gtk_widget_show(vbox2);

		vsb = gtk_vscrollbar_new(GTK_TEXT(d->data)->vadj);
		gtk_box_pack_start(GTK_BOX(vbox2), vsb, TRUE, TRUE, 0);
		gtk_table_attach(GTK_TABLE(table), vbox2, 1, 2, 0, 1,
#ifdef GTK_HAVE_FEATURES_1_1_0
				 (GtkAttachOptions)GTK_FILL,
				 (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
#else
				 GTK_FILL, GTK_EXPAND | GTK_FILL,
#endif
				 0, 0);
		GTK_WIDGET_UNSET_FLAGS(vsb, GTK_CAN_FOCUS);
		gtk_widget_show(vsb);
	}
#ifdef USE_GTKHTML
	else if (GTK_IS_HTML(d->data)) {
		vbox2 = gtk_vbox_new(TRUE, TRUE);
		gtk_widget_show(vbox2);

		vsb = gtk_vscrollbar_new(GTK_LAYOUT(d->data)->vadjustment);
		gtk_box_pack_start(GTK_BOX(vbox2), vsb, TRUE, TRUE, 0);
		gtk_table_attach(GTK_TABLE(table), vbox2, 1, 2, 0, 1,
#ifdef GTK_HAVE_FEATURES_1_1_0
				 (GtkAttachOptions)GTK_FILL,
				 (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
#else
				 GTK_FILL, GTK_EXPAND | GTK_FILL,
#endif
				 0, 0);
		GTK_WIDGET_UNSET_FLAGS(vsb, GTK_CAN_FOCUS);
		gtk_widget_show(vsb);
	}
#endif

	/*
	 * if there are a lot of files open, appending is really slow since it
	 * traverses the child list from start to end.  until this changes
	 * (e.g., the notebook widget keeps a pointer to the last child so
	 * appending is fast), we'll prepend documents.  i know this looks kind
	 * of weird, but try opening a couple hundred files and see what
	 * happens.
	 *
	 * gtk_notebook_append_page(GTK_NOTEBOOK(d->w->notebook), vbox1,
	 * 	d->tablabel);
	 */
	d->w->doclist = g_slist_prepend(d->w->doclist, d);
	gtk_notebook_insert_page(GTK_NOTEBOOK(d->w->nb), vbox1, d->tablabel, 0);

	gtk_notebook_set_page(GTK_NOTEBOOK(d->w->nb), 0);

	gtk_widget_grab_focus(d->data);
} /* doc_add_to_notebook */


/*
 * PUBLIC: doc_close_common
 *
 * common routine for closing a document.
 */
bool_t
doc_close_common(win_t *w, bool_t set_title)
{
	doc_t *d;

	g_assert(w != NULL);
	d = DOC_CURRENT(w);

	if (d->changed)
		return doc_close_verify(w);

	return doc_close_execute(w, set_title);
} /* doc_close_common */


/*
 * PRIVATE: doc_close_execute
 *
 * actually close the document (remove from the notebook, remove from the
 * doclist, etc.  always return TRUE.
 */
static bool_t
doc_close_execute(win_t *w, bool_t set_title)
{
	doc_t *d;

	d = doc_remove(w, set_title);
	g_assert(d != NULL);
	doc_free(d);

	return TRUE;
} /* doc_close_execute */


/*
 * PRIVATE: doc_close_verify
 *
 * creates popup verify box to ask user whether or not to save a file before
 * closing.
 */
static bool_t
doc_close_verify(win_t *w)
{
	int ret;
	char *title, *msg, *fn;
	doc_t *d = DOC_CURRENT(w);
    char *close_title = _("Save File");
    char *close_msg = _("has been modified. \n Do you wish to save it?");

	fn = d->fname;
	if (fn) {
		title = g_new(char, strlen(close_title) + strlen(fn) + 5);
		msg   = g_new(char, strlen(close_msg) + strlen(fn) + 6);
	} else {
		title = g_new(char, strlen(close_title) + strlen(UNTITLED) + 5);
		msg   = g_new(char, strlen(close_msg) + strlen(UNTITLED) + 6);
	}
	sprintf(title, "%s '%s'?", close_title, fn ? fn : UNTITLED);
	sprintf(msg  , " '%s' %s ", fn ? fn : UNTITLED, close_msg);

	ret = do_dialog_yes_no_cancel(title, msg);

	g_free(title);
	g_free(msg);

	switch (ret) {
	case DIALOG_YES:
		return doc_close_verify_yes(w);
	case DIALOG_NO:
		d->changed = FALSE;
		return doc_close_execute(w, TRUE);
	case DIALOG_CANCEL:
		break;
	default:
		printf("doc_close_verify: do_dialog returned %d\n", ret);
		exit(-1);
	} /* switch */

	return FALSE;
} /* doc_close_verify */


/*
 * PRIVATE: doc_close_verify_yes
 *
 * callback invoked when user selected "Yes" to save a file before closing
 */
static bool_t
doc_close_verify_yes(win_t *w)
{
	bool_t saved;
	doc_t *d = DOC_CURRENT(w);

	if (!d->fname) {
		doc_saveas_cb(NULL, w);
		if (d->changed == FALSE)
			return doc_close_execute(w, TRUE);
	} else {
		saved = file_save_execute(d, FALSE);
		if (saved)
			return doc_close_execute(w, TRUE);
	}

	return FALSE;
} /* doc_close_verify_yes */


/*
 * PRIVATE: doc_free
 *
 * free space used by a doc_t.  currently, only used after doc_remove(), so
 * there is some g_slist magic happening.
 */
static void
doc_free(doc_t *d)
{
#ifdef USE_FILEINFO
	doc_info_t *di = (doc_info_t *)(d->docinfo);

	if (di)
		doc_info_close(NULL, d);
#endif	/* USE_FILEINFO */

	g_free(d->fname);
	g_free(d->sb);
	if (d->printwin)
		gtk_widget_destroy(d->printwin);
	if (d->printentry)
		gtk_widget_destroy(d->printentry);
/*
	if (d->data)
		gtk_widget_destroy(d->data);
	if (d->tablabel)
		gtk_widget_destroy(d->tablabel);
*/
	g_free(d);
} /* doc_free */


#ifdef USE_FILEINFO
/*
 * PRIVATE: doc_info_advanced
 *
 * callback invoked when user clicks on "Advanced" button in the doc info popup
 * window.  traverses the list of widgets not shown and shows them.
 */
static void
doc_info_advanced(GtkWidget *wgt, gpointer cbdata)
{
	doc_info_t *di = (doc_info_t *)cbdata;
	GSList *alp;
	GtkWidget *wp;

	g_assert(di != NULL);
	alp = di->adv_list;
	while (alp) {
		wp = (GtkWidget *)(alp->data);
		gtk_widget_show(wp);
		alp = alp->next;
	}

	gtk_widget_destroy(di->adv_button);
	gtk_window_set_title(GTK_WINDOW(di->toplev),
			     _("Advanced File Information"));
} /* doc_info_advanced */


/*
 * PRIVATE: doc_info_close
 *
 * callback for "Close" button on doc info window.
 */
static void
doc_info_close(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;
	doc_info_t *di;

	g_assert(d != NULL);
	di = (doc_info_t *)(d->docinfo);
	gtk_widget_destroy(di->toplev);	/* goes to doc_info_destroy */
} /* doc_info_close */


/*
 * PRIVATE: doc_info_destroy
 *
 * callback for the "destroy" event on the top level doc info window.
 */
static void
doc_info_destroy(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;
	doc_info_t *di;
	GSList *alp;
	GtkWidget *wp;

	g_assert(d != NULL);
	di = (doc_info_t *)(d->docinfo);
	alp = di->adv_list;
	while (alp) {
		wp = (GtkWidget *)(alp->data);
		gtk_widget_destroy(wp);
		alp = alp->next;
	}
	g_slist_free(di->adv_list);
	g_free(di);
	d->docinfo = NULL;
} /* doc_info_destroy */


/*
 * PRIVATE: doc_info_table_add
 *
 * adds new entry into table.  since we keep all the widgets in one table (so
 * that alignment is consistent), we build a list of widgets representing the
 * "advanced" file infomation.  these widgets are *not* shown initially.
 */
static void
doc_info_table_add(GtkWidget *table, doc_info_t *di, char *text,
	int lft, int rht, int top, int bot, bool_t advanced)
{
	GtkWidget *tmp;

	tmp = gtk_label_new(text);
	gtk_misc_set_alignment(GTK_MISC(tmp), (rht % 2), 0.5);
	gtk_table_attach_defaults(GTK_TABLE(table), tmp, lft, rht, top, bot);

	if (advanced)
		di->adv_list = g_slist_prepend(di->adv_list, tmp);
	else
		gtk_widget_show(tmp);
} /* doc_info_table_add */
#endif	/* USE_FILEINFO */


#ifdef USE_DOCLIST
/*
 * PRIVATE: doc_list_select
 *
 * callback routine when selecting a file in the files list window.
 * effectively, sets the notebook to show the selected file.  see select_row()
 * in gtkclist.c to match function parameters.
 */
static int
doc_list_select(GtkWidget *wgt, int row, int column,
	GdkEventButton *event, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	/*
	 * note: gtk_notebook_set_page() emits a switch_page signal.
	 * consequently, we do not have to manually update our curdoc pointer
	 * in the win_t field at this point since it's down in our switch_page
	 * callback (see notebook_switch_page()).
	 */
	gtk_notebook_set_page(GTK_NOTEBOOK(w->nb), row);

	return TRUE;
} /* doc_list_select */


/*
 * PRIVATE: doc_list_update
 *
 * currently only called from doc_save_common(), this routine updates the clist
 * entry after saving a file.  this is necessary because the filename could
 * have changed, and most likely, the file size would have changed as well.
 */
static void
doc_list_update(win_t *w, doc_t *d, int rownum, char *text)
{
	GtkCList *clist;
	char buf[16];
	
	g_assert(w != NULL);

	if (w->dlw == NULL)
		return;

	clist = GTK_CLIST(w->dlw_data);

	if (d->sb == NULL) {
		d->sb = g_new(struct stat, 1);
		if (stat(text, d->sb) == -1) {
			/* print warning */
			gtk_clist_set_text(clist, rownum, FsizeCol, UNKNOWN);
			g_free(d->sb);
			d->sb = NULL;
		} else {
			g_snprintf(buf, 16, "%8lu ", (gulong)(d->sb->st_size));
			gtk_clist_set_text(clist, rownum, FsizeCol, buf);
		}
	} else {
		g_snprintf(buf, 16, "%8lu ", (gulong)(d->sb->st_size));
		gtk_clist_set_text(clist, rownum, FsizeCol, buf);
	}
	gtk_clist_set_text(clist, rownum, FnameCol, text);
} /* doc_list_update */
#endif	/* USE_DOCLIST */


/*
 * PRIVATE: doc_open_ffsel_destroy
 *
 * a destroy signal was sent from the window manager.  we need to set the
 * filesel field to NULL so we know we need to recreate it next time.
 */
static void
doc_open_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	w->filesel = NULL;
} /* doc_open_destroy */


/*
 * PRIVATE: doc_open_cancel
 *
 * The "Cancel" button was hit.  Just hide the dialog box.
 */
static void
doc_open_cancel(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	gtk_widget_hide(w->filesel);
} /* doc_open_cancel */


/*
 * PRIVATE: doc_open_ok
 *
 * callback invoked when user has clicked the "Ok" button from the file
 * selector for opening a file
 */
static void
doc_open_ok(GtkWidget *wgt, gpointer cbdata)
{
#ifdef USE_ENHANCED_FSEL
	GSList *fnlp, *tmp;
	guint len;
#endif
	char *fname;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
#ifdef USE_ENHANCED_FSEL
	fnlp = gtk_efilesel_get_filename_list(GTK_EFILESEL(w->filesel));
	len = g_slist_length(fnlp);
	for (tmp = fnlp; tmp; tmp = tmp->next) {
		fname = (char *)tmp->data;
		doc_open_ok_common(fname, w, (len == 1));
		len--;
	}
#else
	fname = gtk_efilesel_get_filename(GTK_EFILESEL(w->filesel));
	doc_open_ok_common(fname, w, TRUE);
#endif	/* USE_ENHANCED_FSEL */

	gtk_widget_hide(w->filesel);
} /* doc_open_ok */


/*
 * PRIVATE: doc_open_ok_common
 *
 * common routine for loading a file.  Only called from doc_open_ok().
 */
static void
doc_open_ok_common(char *fname, win_t *w, gboolean update)
{
	struct stat sb;
	doc_t *d;

	if (!fname || fname == "")
		return;

	if (stat(fname, &sb) == -1)
		return;	/* XXX - need to print warning */

	if (S_ISDIR(sb.st_mode)) {
		gtk_efilesel_set_filename(
			GTK_EFILESEL(w->filesel), fname);
		(void)do_dialog_ok(_(" Cannot open directories "), NULL);
		return;
	}

	gtk_signal_disconnect_by_func(GTK_OBJECT(w->nb),
				      GTK_SIGNAL_FUNC(doc_switch_page),
				      w);
	/*
	 * if there's only one document, and it's an empty "Untitled"
	 * doc, then remove it before adding the one we want
	 */
	d = DOC_CURRENT(w);
	if (w->numdoc == 1 && !d->changed && !d->fname)
		doc_close_common(w, FALSE);

	if (update)
		doc_new(w, fname, DocText,
			(UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));
	else
		doc_new(w, fname, DocText, 0);

	(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				       GTK_SIGNAL_FUNC(doc_switch_page),
				       w);
	recent_list_add(w, fname);
} /* doc_open_ok_common */


/*
 * PRIVATE: doc_remove
 *
 * removes current document from window and returns pointer to it.  currently
 * only called from doc_close_execute().
 */
static doc_t *
doc_remove(win_t *w, bool_t set_title)
{
	GSList *dlp;
	GtkNotebook *nb;
	char *cwd, *fn;
	doc_t *d;
	int num;

	d = DOC_CURRENT(w);
	nb = GTK_NOTEBOOK(w->nb);
	g_assert(nb != NULL);

	/* remove from notebook */
	num = gtk_notebook_current_page(nb);
	gtk_notebook_remove_page(nb, num);

	/*
	 * remove from doclist and doc list window.  update numdoc in winlist.
	 * contents of the doc gets freed in doc_free().
	 */
	dlp = g_slist_find(w->doclist, d);
	w->doclist = g_slist_remove_link(w->doclist, dlp);
	g_slist_free_1(dlp);
	w->numdoc--;
	doc_list_remove(w, num);
	win_list_set_numdoc(w);
#if defined(GTK_HAVE_FEATURES_1_1_0) && defined(USE_UNDOREDO)
	d->undolist = undo_list_delete(d->undolist);
	d->redolist = undo_list_delete(d->redolist);
#endif	/* USE_UNDOREDO */

	/* sanity checks */
	g_assert(w->numdoc == g_list_length(nb->children));
	g_assert(w->numdoc == g_slist_length(w->doclist));

	/* skip printing cwd in full pathname */
	fn = d->fname;
	if ((cwd = getcwd(NULL, 0)) != NULL) {
		if (fn && strstr(fn, cwd))
			fn = d->fname + strlen(cwd) + 1;
		free(cwd);
	}

	msgbar_printf(w, _("Closed %s"), fn ? fn : UNTITLED);
	msgbox_printf( _("closed %s"), fn ? fn : UNTITLED);

	/* update curdoc pointer */
	num = gtk_notebook_current_page(nb);
	if (num == -1) {	/* closed last doc */
		w->curdoc = NULL;
		return d;
	}

	w->curdoc = g_slist_nth_data(w->doclist, num);

	if (set_title)
		win_set_title(DOC_CURRENT(w));

	return d;
} /* doc_remove */


/*
 * PRIVATE: doc_saveas_destroy
 *
 * blow away the save-as dialog
 */
static void
doc_saveas_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	w->saveas = NULL;
} /* doc_saveas_destroy */


/*
 * PRIVATE: doc_saveas_ok
 *
 * callback invoked when user clicked "Ok" on the file save-as dialog.
 */

static void
doc_saveas_ok(GtkWidget *wgt, gpointer cbdata)
{
	int ret;
	doc_t *d;
	bool_t saved;
	char *fname, *title, *msg;
    char *ow_title = _("Overwrite");
    char *ow_msg = _("exists.  Overwrite?");

	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	g_assert(w->saveas != NULL);

	fname = gtk_efilesel_get_filename(GTK_EFILESEL(w->saveas));
	if (fname == NULL || fname[0] == '\0')
		return;

	if (file_exist(fname)) {
		title = g_new(char, strlen(ow_title) + strlen(fname) + 5);
		msg   = g_new(char, strlen(ow_msg) + strlen(fname) + 6);
		sprintf(title, "%s '%s'?", ow_title, fname);
		sprintf(msg  , " '%s' %s ", fname, ow_msg);

		ret = do_dialog_yes_no_cancel(title, msg);

		g_free(title);
		g_free(msg);

		switch (ret) {
		case DIALOG_YES:
			goto saveas_do_save;	/* ugh, just had to use goto */
			break;
		case DIALOG_NO:
			/* don't overwrite; let user decide another name */
			break;
		case DIALOG_CANCEL:
			/* canceled */
			gtk_widget_destroy(w->saveas);
			break;
		default:
			printf("doc_saveas_ok: do_dialog returnd %d\n", ret);
			exit(-1);
		} /* switch */
	} else {
saveas_do_save:
		d = DOC_CURRENT(w);
		g_free(d->fname);
		d->fname = g_strdup(fname);
		d->changed = TRUE;
		saved = file_save_execute(d, FALSE);
		if (saved) {
			gtk_widget_destroy(w->saveas);
			recent_list_add(w, d->fname);
			doc_info_label_update(w);
		}
	}
} /* doc_saveas_ok */


static void
doc_saveas_cancel_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_widget_destroy(w->saveas);
} /* doc_saveas_cancel_cb */


/* the end */
