(**
   Multiline formatted text. The text is draggable.

  TODO
  * Rethink integration of U.EscapeString into Text
**)

MODULE VOText;

(*
    Implements a text image.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VODisplay,
       DD  := VODragDrop,
       F   := VOFrame,
       G   := VOGUIObject,
       U   := VOUtil,
       V   := VOVecImage,

              Ascii,
       str := Strings;

CONST

  (* PartDesc.style *)

  smart      * =  D.maxStyle+1;  (* that nice 3-D effect with bright color on dark color *)
  highlight  * =  D.maxStyle+2;  (* uses shinePen instead of textpen                     *)
  warn       * =  D.maxStyle+3;  (* Use a color for warnings                             *)

  (* LineDesc.justification *)

  leftAlligned  * = 0;
  rightAlligned * = 1;
  centered      * = 2;

  (* Image.type *)

  returnImage  = 0;
  escapeImage  = 1;
  defaultImage = 2;

  (*
    This is the special escape-character introducting special commands
    for the layout-engine.

    Format is always ESCAPE <command>, where command is a special character
    introduced below...

    An extension of the syntax for command like ESCAPE <command> <onechaparameter>
    should be made. For example for colordefinitions.

    It is of cause no problem to implement commands
    for drawing horizontal lines or other nifty things...

    A usefull thing of cause would be support for datatype-objects...
  *)
  ESC         = Ascii.esc;

  (* The different letters than can follow escape *)

  NORMAL     = "n";
  BOLD       = "b";
  ITALIC     = "i";
  SLANTED    = "a";
  UNDERLINED = "u";
  SMART      = "s";
  HIGHLIGHT  = "h";
  WARN       = "w";

  LEFT       = "l";
  RIGHT      = "r";
  CENTERED   = "c";

  RETURNI    = "R"; (* Insert the return image at this place *)
  ESCAPE     = "E"; (* Insert the escape image at this place *)
  DEFAULT    = "D"; (* Insert the default image at this place *)

  FIXED      = "f";
  PROP       = "p";

  (* This are also characters which need special care: *)

  EOL  = Ascii.lf;  (* This one marks the end of a line                       *)
  TAB  = Ascii.ht;  (* Tabulator positions, work even with proportional fonts *)
  EOT  = Ascii.nul; (* Should be obsolent now, checking against LEN(text)     *)


  (* --------------------------------------------------------- *)


TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 frame*     : LONGINT; (* the frame to use for the button *)
                 highlight* : BOOLEAN; (* highlight if selected *)
               END;


  Part      = POINTER TO PartDesc;
  PartDesc  = RECORD
                next   : Part;     (* all parts of a line are linked    *)
                width,             (* width and height of this part
                                      calculated by TextImage.CalcSize  *)
                height : LONGINT;
              END;

  TextPart     = POINTER TO TextPartDesc;
  TextPartDesc = RECORD (PartDesc)
                   style   : SET;       (* every part has a special drawmode *)
                   aColor,              (* every part has special colors
                                           this is not supported yet         *)
                   bColor  : INTEGER;
                   text    : U.Text;    (* The text to be drawn              *)
                   font    : LONGINT;

                   offset  : LONGINT;
                 END;

  TabPart     = POINTER TO TabPartDesc;
  TabPartDesc = RECORD (PartDesc)
                  nextTab : TabPart;
                END;

  ImagePart     = POINTER TO ImagePartDesc;
  ImagePartDesc = RECORD (PartDesc)
                    imageType : LONGINT;
                    image     : V.VecImage;
                  END;

  Line     = POINTER TO LineDesc;
  LineDesc = RECORD
               next, last    : Line;      (* lines are double-linked, maybe someday we add a scroller *)
               justification : SHORTINT;  (* Every line can have it's special justification           *)
               parts         : Part;      (* Every line consists of a number of parts, each of them
                                             with special layout-paameters                            *)
               lastPart      : Part;      (* last entry in parts-list                                 *)
               currentPart   : Part;      (* curent, not yet appended entry                           *)
               currentFont   : LONGINT;
               width,                     (* width and height of line -> TextClass.CalcSize           *)
               height        : LONGINT;
               tabCount      : LONGINT;   (* Number of tabs in line. Usefull for optimations in
                                             TextClass.CalcSize                                       *)
             END;

  TabList      = POINTER TO ARRAY OF LONGINT;

  (* --------------------------------------------------------- *)

  Text *    = POINTER TO TextDesc;
  TextDesc* = RECORD (G.ImageDesc)
                prefs        : Prefs;
                textList     : Line;          (* A text is first just a number of lines    *)
                lastLine     : Line;          (* last line of textList                     *)
                currentLine  : Line;          (* Pointer to current line, not in textList  *)
                lines        : LONGINT;       (* number of lines in text                   *)
                currentJust  : SHORTINT;      (* current justification for current and
                                                 following lines/parts                     *)
                currentStyle : SET;           (* current style for current and
                                                 following lines/parts                     *)
                currentFont  : LONGINT;
                textWidth-,                   (* with and high of total text
                                                 calclated by TextClass.CalcSize           *)
                textHeight-,
                lineSpace    : LONGINT;       (* Space between lines                       *)
                tabList      : TabList;
                calced       : BOOLEAN;

                defaultJust  : SHORTINT;
                defaultDraw  : SET;
                defaultFont  : LONGINT;
              END;

VAR
  prefs* : Prefs;

  (* --------------------------------------------------------- *)

  (* Appends given part to the list of pats for this line *)

  PROCEDURE (l : Line) AppendPart (p : Part);

  BEGIN
    IF l.parts=NIL THEN
      l.parts:=p;
    ELSE
      l.lastPart.next:=p;
    END;
    l.lastPart:=p;
  END AppendPart;

  (* --------------------------------------------------------- *)

  (*
     Closes current part
     If there is a currentPart with a given text, it will appended to the list
     of parts of the current line (Line.AddPart) and currentPart will be set to NIL
   *)

  PROCEDURE (l : Line) ClosePart;

  BEGIN
    IF l.currentPart#NIL THEN
      IF (l.currentPart IS TextPart) & (l.currentPart(TextPart).text=NIL) THEN
        l.currentPart:=NIL;
      ELSE
        l.AppendPart(l.currentPart);
      END;
    END;
    l.currentPart:=NIL;
  END ClosePart;

  (* --------------------------------------------------------- *)

  (* Closes last part and creates a new TextPart in Line.currentPart *)

  PROCEDURE (l : Line) NewTextPart(defaultStyle : SET; defaultFont : LONGINT);

  VAR
    help : TextPart;

  BEGIN
    l.ClosePart;
    NEW(help);
    help.next:=NIL;
    help.text:=NIL;
    l.currentPart:=help;
    l.currentPart(TextPart).style:=defaultStyle;
    l.currentPart(TextPart).font:=defaultFont;
  END NewTextPart;

  (* --------------------------------------------------------- *)

  (* Closes last part and creates a new TabPart in Line.currentPart *)

  PROCEDURE (l : Line) NewTabPart(t : Text);

  VAR
    newOne : TabList;
    x      : LONGINT;
    help   : TabPart;

  BEGIN
    l.ClosePart;
    NEW(help);
    help.next:=NIL;
    help.nextTab:=NIL;
    l.currentPart:=help;

    IF (t.tabList=NIL) THEN
      NEW(t.tabList,10);
      FOR x:=0 TO LEN(t.tabList^)-1 DO
        t.tabList[x]:=0;
      END;
    END;

    IF l.tabCount>LEN(t.tabList^) THEN
      NEW(newOne,LEN(t.tabList^)+10);
      FOR x:=0 TO LEN(t.tabList^)-1 DO
        newOne[x]:=t.tabList[x];
      END;
      FOR x:=LEN(t.tabList^) TO LEN(newOne^)-1 DO
        newOne[x]:=0;
      END;
      t.tabList:=newOne;
    END;

    INC(l.tabCount);
  END NewTabPart;

  (* --------------------------------------------------------- *)

  (* Closes last part and creates a new ImagePart in Line.currentPart *)

  PROCEDURE (l : Line) NewImagePart(imageType : LONGINT);

  VAR
    help  : ImagePart;
    image : V.VecImage;

  BEGIN
    l.ClosePart;
    NEW(help);
    help.next:=NIL;
    help.image:=NIL;
    l.currentPart:=help;
    l.currentPart(ImagePart).imageType:=imageType;

    NEW(image);
    image.Init;
    CASE imageType OF
      returnImage:
        image.Set(V.return);
    | escapeImage:
        image.Set(V.escape);
    | defaultImage:
        image.Set(V.default);
    END;
    l.currentPart(ImagePart).image:=image;
  END NewImagePart;


  (* --------------------------------------------------------- *)

  (* Sets the curent style *)

  PROCEDURE (l : Line) SetStyle(style : SET);

  BEGIN
    l.currentPart(TextPart).style:=style;
  END SetStyle;

  (* --------------------------------------------------------- *)


  (* Sets the curent Font *)

  PROCEDURE (l : Line) SetFont(font : LONGINT);

  BEGIN
    l.currentPart(TextPart).font:=font;
  END SetFont;

  (* --------------------------------------------------------- *)
  (* Appends the given line to TextClassDesc.textList *)

  PROCEDURE (t : Text) AppendLine(l : Line);

  BEGIN
    IF t.textList=NIL THEN
      t.textList:=l;
    ELSE
      t.lastLine.next:=l;
      l.last:=t.lastLine;
    END;
    t.lastLine:=l;
    INC(t.lines);
  END AppendLine;

  (* --------------------------------------------------------- *)

  (*
     Checks if there is a valid line in TextClassDesc.currentLine and
     appends it to TextClassDesc.textList using AppendLine
   *)

  PROCEDURE (t : Text) CloseLine;

  BEGIN
    IF t.currentLine#NIL THEN
      t.AppendLine(t.currentLine);
      t.currentLine:=NIL;
    END;
  END CloseLine;

  (* --------------------------------------------------------- *)

  (*
     Handles existing TextClassDesc.currentLine, appends its using AppendLine
     and creates a new line in TextClassDesc.curentLine
   *)

  PROCEDURE (t : Text) NewLine;

  BEGIN

    (* Allready a line on the stack -> Close the line and append it *)

    t.CloseLine;

    NEW(t.currentLine);
    t.currentLine.next:=NIL;
    t.currentLine.last:=NIL;
    t.currentLine.parts:=NIL;
    t.currentLine.lastPart:=NIL;
    t.currentLine.currentPart:=NIL;
    t.currentLine.justification:=t.currentJust;
    t.currentLine.NewTextPart(t.currentStyle,t.currentFont);
  END NewLine;

  (* --------------------------------------------------------- *)

  (* Adds the given text to the part in t.currentLine.currentPart *)

  PROCEDURE (t : Text) AddTextToPart(text : ARRAY OF CHAR; start, stop : LONGINT);

  BEGIN
    IF stop-start>=0 THEN
      NEW(t.currentLine.currentPart(TextPart).text,stop-start+2);
      str.Extract(text,SHORT(start),SHORT(stop-start+1),t.currentLine.currentPart(TextPart).text^);
    END;
    t.currentLine.ClosePart;
    t.currentLine.NewTextPart(t.currentStyle,t.currentFont);
  END AddTextToPart;

  (* --------------------------------------------------------- *)

  (* Sets current justification *)

  PROCEDURE (t : Text) SetJustification(type : SHORTINT);

  BEGIN
    t.currentLine.justification:=type;
    t.currentJust:=type;
  END SetJustification;

  (* --------------------------------------------------------- *)

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.dottedFocus;
    p.highlight:=TRUE;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(t : Text);

  BEGIN
    t.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      t.SetBackgroundObject(p.background.Copy());
      t.backgroundObject.source:=t;
    END;
  END SetPrefs;

  (* --------------------------------------------------------- *)

  PROCEDURE (t : Text) Init*;

  BEGIN
    t.Init^;

    prefs.SetPrefs(t);

    EXCL(t.flags,G.stdFocus);
    INCL(t.flags,G.canDisable);

(*    IF t.prefs.highlight THEN
      EXCL(t.flags,G.noHighlight);
    ELSE
      INCL(t.flags,G.noHighlight);
    END;*)

    t.textHeight:=0;
    t.textWidth:=0;

    t.lines:=0;
    t.textList:=NIL;
    t.lastLine:=NIL;
    t.currentLine:=NIL;
    t.lineSpace:=0;
    t.tabList:=NIL;

    t.defaultJust:=leftAlligned;
    t.defaultDraw:={};
    t.defaultFont:=D.normalFont;

    (* Initialize text to an empty line *)

    t.NewLine;
    t.CloseLine;
  END Init;

  (* ---- Drag and drop stuff *)

  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  **)

  PROCEDURE (t : Text) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  BEGIN
    IF t.visible & t.PointIsIn(x,y) THEN
      (* we can drag and drop *)
      RETURN t;
    ELSE
      RETURN NIL;
    END;
  END GetDnDObject;

  (**
    Return a list of supported datatypes
  **)

  PROCEDURE (t : Text) GetDragInfo*(VAR dragInfo : DD.DnDDataInfo);

  BEGIN
    dragInfo.AddDataType(DD.text,DD.none,{DD.copy},DD.copy);
  END GetDragInfo;

  (**
    Called if we want to get the drag & drop data from the object.
    data will we garanted to be initialized.
  **)

  PROCEDURE (t : Text) GetDragData*(group, type, action : LONGINT):DD.DnDData;

  VAR
    data : DD.DnDStringData;
    size : LONGINT;
    l    : Line;
    p    : Part;
    help : ARRAY 2 OF CHAR;

  BEGIN
    IF group=DD.text THEN

      help[0]:=Ascii.lf;
      help[1]:=0X;

      size:=0;
      l:=t.textList;
      WHILE l#NIL DO
        p:=l.parts;
        WHILE p#NIL DO
          WITH p : TextPart DO
            INC(size,LEN(p.text^)-1);
          ELSE
          END;
          p:=p.next;
        END;

        IF l.next#NIL THEN
          INC(size);
        END;
        l:=l.next;
      END;

      NEW(data);
      NEW(data.string,size+1);
      data.string[0]:=0X;

      l:=t.textList;
      WHILE l#NIL DO
        p:=l.parts;
        WHILE p#NIL DO
          WITH p : TextPart DO
            str.Append(p.text^,data.string^);
          ELSE
          END;
          p:=p.next;
        END;
        IF l.next#NIL THEN
          str.Append(help,data.string^);
        END;
        l:=l.next;
      END;
      data.string[LEN(data.string^)-1]:=0X;

      RETURN data;
    ELSE
      RETURN NIL;
    END;
  END GetDragData;

  (*
    Calculates the size of TextClass.textList in respect to given font and rastPort.
  *)

  PROCEDURE (t : Text) CalcTextSize*(display : D.Display);

  VAR
    l        : Line;
    p,last   : Part;
    mSize,
    tabCount,
    tSize    : LONGINT;

    extent   : D.FontExtentDesc;

    help     : ARRAY 2 OF CHAR;

    font     : D.Font;

  BEGIN
    t.textHeight:=0;
    t.textWidth:=0;
    t.lineSpace:=0; (* Under X the space between lines seems to be part of the text *)

    mSize:=0;

    l:=t.textList;
    WHILE l#NIL DO

      tabCount:=0;
      tSize:=0;
      p:=l.parts;
      last:=NIL;
      WHILE p#NIL DO

        WITH
          p : TextPart DO
              font:=display.GetFont(p.font);
              p.width:=font.TextWidth(p.text^,LEN(p.text^)-1,p.style);
              p.height:=font.height;
              p.offset:=0;

              IF (last=NIL) OR ~(last IS TextPart) THEN
                help[0]:=p.text[0];
                help[1]:=0X;
                font.TextExtent(help,1,p.style,extent);
                p.offset:=extent.lbearing;
                DEC(p.width,extent.lbearing);
              END;

              IF p.next=NIL THEN
                help[0]:=p.text[LEN(p.text^)-2];
                help[1]:=0X;
                font.TextExtent(help,1,p.style,extent);
                DEC(p.width,extent.width-extent.rbearing);
              END;

              IF (smart IN p.style) OR (G.canDisable IN t.flags) THEN
                INC(p.height);
                INC(p.width);
              END;

              INC(tSize,p.width);

        | p : TabPart DO
              INC(tSize,mSize);
              p.height:=0; (*font.height;*)

              IF tSize>t.tabList[tabCount] THEN
                t.tabList[tabCount]:=tSize;
              END;

              tSize:=0;
              INC(tabCount);

        | p : ImagePart DO
              p.image.CalcSize(display);
              p.height:=p.image.height;
              p.width:=p.image.width+2*display.spaceWidth;
        END;

        INC(l.width,p.width);

        last:=p;
        p:=p.next;
      END;

      l:=l.next;
    END;

    l:=t.textList;
    WHILE l#NIL DO

      l.width:=0;
      IF l.parts#NIL THEN
        l.height:=0;
      ELSE
        font:=display.GetFont(D.normalFont);
        l.height:=font.height;
      END;

      tabCount:=0;
      tSize:=0;
      p:=l.parts;
      WHILE p#NIL DO

        WITH

          p : TextPart DO
              INC(tSize,p.width);
              IF p.height > l.height THEN
                l.height:=p.height;
              END;

        | p : TabPart DO
              p.width:=t.tabList[tabCount]-tSize+1;

              tSize:=0;
              INC(tabCount);

        | p : ImagePart DO
              INC(tSize,p.width);

        END;

        INC(l.width,p.width);
        IF p.height>l.height THEN
          l.height:=p.height;
        END;

        p:=p.next;
      END;

      IF l.width>t.textWidth THEN
        t.textWidth:=l.width;
      END;

      INC(t.textHeight,l.height);

      l:=l.next;
    END;

    INC(t.textHeight,(t.lines-1)*t.lineSpace);

  END CalcTextSize;

  (* --------------------------------------------------------- *)

  (* Parses text and builds up TextClass.textList *)

  PROCEDURE (t : Text) SetText*(text : ARRAY OF CHAR);

  VAR
    start,
    stop,
    pos    : LONGINT;

  BEGIN
    start:=0;
    stop:=0;
    pos:=0;
    t.lines:=0;
    t.textList:=NIL;
    t.lastLine:=NIL;
    t.currentLine:=NIL;
    t.currentJust:=t.defaultJust;
    t.currentStyle:=t.defaultDraw;
    t.currentFont:=t.defaultFont;
    t.lineSpace:=0;
    t.tabList:=NIL;

    t.NewLine;

    LOOP

      IF pos>=LEN(text) THEN
        stop:=pos-1;
        t.AddTextToPart(text,start,stop);
        t.CloseLine;
        EXIT;
      END;

      CASE text[pos] OF

      | EOT : stop:=pos-1;
              t.AddTextToPart(text,start,stop);
              t.CloseLine;
              EXIT;
      | EOL : stop:=pos-1;
              t.AddTextToPart(text,start,stop);
              t.CloseLine;
              t.NewLine;
              start:=pos+1;
      | ESC : stop:=pos-1;
              IF stop>=start THEN
                t.AddTextToPart(text,start,stop);
                t.currentLine.NewTextPart(t.currentStyle,t.currentFont);
              END;

              INC(pos);

              CASE text[pos] OF

              | NORMAL:
                  t.currentStyle:={};
                  t.currentLine.SetStyle(t.currentStyle);
              | BOLD:
                  INCL(t.currentStyle,D.bold);
                  t.currentLine.SetStyle(t.currentStyle);
              | ITALIC:
                  INCL(t.currentStyle,D.italic);
                  t.currentLine.SetStyle(t.currentStyle);
              | SLANTED:
                  INCL(t.currentStyle,D.slanted);
                  t.currentLine.SetStyle(t.currentStyle);
              | UNDERLINED:
                  INCL(t.currentStyle,D.underlined);
                  t.currentLine.SetStyle(t.currentStyle);
              | SMART:
                  INCL(t.currentStyle,smart);
                  t.currentLine.SetStyle(t.currentStyle);
              | HIGHLIGHT:
                  INCL(t.currentStyle,highlight);
                  t.currentLine.SetStyle(t.currentStyle);
              | WARN:
                  INCL(t.currentStyle,warn);
                  t.currentLine.SetStyle(t.currentStyle);

              | LEFT:
                  t.SetJustification(leftAlligned);
              | RIGHT:
                  t.SetJustification(rightAlligned);
              | CENTERED:
                  t.SetJustification(centered);

              | RETURNI:
                  t.currentLine.NewImagePart(returnImage);
                  t.currentLine.NewTextPart(t.currentStyle,t.currentFont);
              | ESCAPE:
                  t.currentLine.NewImagePart(escapeImage);
                  t.currentLine.NewTextPart(t.currentStyle,t.currentFont);
              | DEFAULT:
                  t.currentLine.NewImagePart(defaultImage);
                  t.currentLine.NewTextPart(t.currentStyle,t.currentFont);

              | FIXED:
                  t.currentLine.NewTextPart(t.currentStyle,D.fixedFont);
              | PROP:
                  t.currentLine.NewTextPart(t.currentStyle,D.normalFont);
              | "0".."9":
                  t.currentLine.NewTextPart(t.currentStyle,ORD(text[pos])-ORD("0"));
              ELSE
                t.AddTextToPart(text,pos,pos);
                t.currentLine.NewTextPart(t.currentStyle,t.currentFont);
              END;

              start:=pos+1;

      | TAB : stop:=pos-1;
              IF stop>=start THEN
                t.AddTextToPart(text,start,stop);
                t.currentLine.NewTextPart(t.currentStyle,t.currentFont);
              END;
              start:=pos+1; (* skip next char *)
              t.currentLine.NewTabPart(t);
              t.currentLine.NewTextPart(t.currentStyle,t.currentFont);

      ELSE
        (* Wer are skipping "normal" characters *)
      END;

      INC(pos);
    END;

    IF t.calced THEN
      t.CalcTextSize(t.display);
      t.Redraw;
    END;
  END SetText;

  (* --------------------------------------------------------- *)


  PROCEDURE (t : Text) SetDefault*(adjustment : SHORTINT; drawMode : SET; font : LONGINT);

  BEGIN
    t.defaultJust:=adjustment;
    t.defaultDraw:=drawMode;
    t.defaultFont:=font;
  END SetDefault;

  (* --------------------------------------------------------- *)


  PROCEDURE (t : Text) CalcSize*(display : D.Display);

  BEGIN
    t.CalcTextSize(display);

    t.calced:=TRUE;
    t.width:=t.textWidth;
    t.height:=t.textHeight;
    t.minWidth:=t.textWidth;
    t.minHeight:=t.textHeight;

    t.CalcSize^(display);
  END CalcSize;

  (* --------------------------------------------------------- *)

  (*
    Draws the text in TextClass.textList based upon the calculations of
    TextClass.CalcSize in the given window
  *)

  PROCEDURE (t : Text) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    l     : Line;
    p     : Part;

    xPos  : LONGINT;
    yPos,
    y2    : LONGINT;

    font  : D.Font;

    frame : F.Frame;

  BEGIN
    t.Draw^(x,y,draw);

    IF (D.selected IN draw.mode) & ~(G.noHighlight IN t.flags) THEN
      draw.PushForeground(D.fillColor);
      draw.FillRectangle(t.x,t.y,t.width,t.height);
      draw.PopForeground;
    ELSE
      t.DrawBackground(t.x,t.y,t.width,t.height);
    END;

    IF (D.selected IN draw.mode) & t.prefs.highlight THEN
      draw.PushForeground(D.fillTextColor);
    ELSE
      draw.PushForeground(D.textColor);
    END;

    draw.InstallClip;
    draw.AddRegion(t.x,t.y,t.width,t.height);

    yPos:=t.y;

    IF t.height>t.textHeight THEN
      INC(yPos,(t.height-t.textHeight) DIV 2);
    END;

    l:=t.textList;
    WHILE l#NIL DO

      CASE l.justification OF
      | leftAlligned  : xPos:=t.x;
      | rightAlligned : xPos:=t.x+t.width-t.textWidth+(t.textWidth-l.width);
      | centered      : xPos:=t.x+(t.width-t.textWidth) DIV 2 + (t.textWidth-l.width) DIV 2;
      END;

      p:=l.parts;
      WHILE p#NIL DO

        WITH
          p : TextPart DO

            font:=draw.display.GetFont(p.font);

            draw.PushFont(p.font,p.style);

            y2:=yPos+font.ascent;

            IF (t.disabled) OR (D.disabled IN draw.mode) THEN

              draw.PushForeground(D.halfShadowColor);
              draw.DrawString(xPos-p.offset,y2,p.text^,LEN(p.text^)-1);
              draw.PopForeground;

              draw.PushForeground(D.halfShineColor);
              draw.DrawString(xPos-p.offset+1,y2+1,p.text^,LEN(p.text^)-1);
              draw.PopForeground;

            ELSE

              IF highlight IN p.style THEN
                draw.PushForeground(D.whiteColor);
              ELSIF warn IN p.style THEN
                draw.PushForeground(D.warnColor);
              END;

              IF smart IN p.style THEN
                draw.DrawString(xPos+1-p.offset,y2+1,p.text^,LEN(p.text^)-1);
                IF D.underlined IN p.style THEN
                  draw.DrawLine(xPos+1,y2+1,xPos+p.width+1,y2+1);
                END;
                draw.PushForeground(D.whiteColor);
                draw.DrawString(xPos-p.offset,y2,p.text^,LEN(p.text^)-1);
                IF D.underlined IN p.style THEN
                  draw.DrawLine(xPos,y2,xPos+p.width,y2);
                END;
              ELSE
                draw.DrawString(xPos-p.offset,y2,p.text^,LEN(p.text^)-1);

                IF D.underlined IN p.style THEN
                  draw.DrawLine(xPos,y2,xPos+p.width,y2);
                END;
              END;

              IF (smart IN p.style) OR (highlight IN p.style) OR (warn IN p.style) THEN
                draw.PopForeground;
              END;

            END;

            INC(xPos,p.width);

            draw.PopFont;

        | p : TabPart DO
            INC(xPos,p.width);

        | p : ImagePart DO
            p.image.Draw(xPos+(p.width-p.image.width) DIV 2,
                               yPos+(l.height-p.image.height) DIV 2,
                               draw);
        END;

        p:=p.next;
      END;

      INC(yPos,l.height+t.lineSpace);
      l:=l.next;
    END;

    IF t.DisplayFocus() THEN
      IF t.focus=NIL THEN
        NEW(frame);
        frame.Init;
        frame.SetFlags({G.horizontalFlex,G.verticalFlex});
        t.focus:=frame;
      ELSE
        frame:=t.focus(F.Frame);
      END;
      frame.SetInternalFrame(t.prefs.frame);
      frame.CalcSize(t.display);

      IF (t.width>=t.textWidth+frame.leftBorder+frame.rightBorder+4)
      & (t.height>=t.textHeight+frame.topBorder+frame.bottomBorder+4) THEN
        frame.Resize(t.width-4,t.height-4);
      ELSE
        frame.Resize(t.width,t.height);
      END;
      frame.Draw(t.x+(t.width-frame.width) DIV 2,
                 t.y+(t.height-frame.height) DIV 2,t.draw);
    END;

    draw.FreeLastClip;
    draw.PopForeground;
  END Draw;

  PROCEDURE (t : Text) DrawFocus*;

  BEGIN
    INCL(t.flags,G.showFocus);
    t.Redraw;
  END DrawFocus;

  PROCEDURE (t : Text) HideFocus*;

  BEGIN
    EXCL(t.flags,G.showFocus);
(*    IF t.focus#NIL THEN
      t.focus.Hide;
    END;*)
    t.Redraw;
  END HideFocus;

  PROCEDURE (t : Text) Hide*;

  BEGIN
    IF t.visible THEN
      IF t.DisplayFocus() & (t.focus#NIL) THEN
        t.focus.Hide;
      END;
      t.DrawHide;
     t.Hide^;
    END;
  END Hide;

  PROCEDURE MakeLeftText*(string : ARRAY OF CHAR):Text;

  VAR
    text : Text;
    help : U.Text;

  BEGIN
    NEW(text);
    text.Init;
    text.SetFlags({G.horizontalFlex,G.verticalFlex});
    help:=U.EscapeString(string);
    text.SetText(help^);
    RETURN text;
  END MakeLeftText;

  PROCEDURE MakeCenterText*(string : ARRAY OF CHAR):Text;

  VAR
    text : Text;
    help : U.Text;

  BEGIN
    NEW(text);
    text.Init;
    text.SetFlags({G.horizontalFlex,G.verticalFlex});
    text.SetDefault(centered,{},D.normalFont);
    help:=U.EscapeString(string);
    text.SetText(help^);
    RETURN text;
  END MakeCenterText;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOText.