(**
   A simple led with two states: on and off.
**)

MODULE VOLED;

(*
    Implements a simple led with two states: on and off.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D := VODisplay,
       G := VOGUIObject,
       O := VOObject,
       P := VOPrefs,
       V := VOValue;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
               END;


  LED*     = POINTER TO LEDDesc;
  LEDDesc* = RECORD (G.ImageDesc)
               prefs : Prefs;
               model : V.ValueModel;
             END;

VAR
  prefs : Prefs;

  PROCEDURE (l : LED) Init*;

  BEGIN
    l.Init^;

    l.prefs:=prefs;

    l.model:=NIL;
  END Init;

  (**
    Assign the boolean model to the LED. The LED will be on, when the value of the
    model is true, else it will off.
  **)

  PROCEDURE (l : LED) SetModel*(model : V.ValueModel);

  BEGIN
    IF l.model#NIL THEN
      l.UnattachModel(l.model);
    END;
    l.model:=model;
    l.AttachModel(model);
  END SetModel;

  PROCEDURE (l : LED) CalcSize*(display : D.Display);

  BEGIN
    IF display.spaceWidth>display.spaceHeight THEN
      l.width:=2*display.spaceWidth;
      l.height:=2*display.spaceWidth;
    ELSE
      l.width:=2*display.spaceHeight;
      l.height:=2*display.spaceHeight;
    END;

    l.minWidth:=l.width;
    l.minHeight:=l.height;

    l.CalcSize^(display);
  END CalcSize;

  PROCEDURE (l : LED) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    l.Draw^(x,y,draw);

    IF (l.model#NIL) & ~l.model.IsNull() & l.model.GetBoolean() THEN
      IF l.display.colorMode=D.monochromeMode THEN
        draw.PushForeground(D.whiteColor);
      ELSE
        draw.PushForeground(D.warnColor);
      END;
      draw.FillArc(l.x,l.y,l.width-1,l.height-1,0*64,360*64);
      draw.PopForeground;
    ELSE
      draw.PushForeground(D.shadowColor);
      draw.FillArc(l.x,l.y,l.width-1,l.height-1,0*64,360*64);
      draw.PopForeground;
    END;

    draw.PushForeground(D.shineColor);
    draw.DrawArc(l.x+l.width DIV 4,l.y+l.height DIV 4,
                 l.width DIV 2,l.height DIV 2,90*64,60*64);
    draw.PopForeground;

    draw.PushForeground(D.shadowColor);
    draw.DrawArc(l.x,l.y,l.width,l.height,0*64,360*64);
    draw.PopForeground;
  END Draw;

  PROCEDURE (l : LED) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF l.visible THEN
      l.Redraw;
    END;
  END Resync;


  PROCEDURE (p : Prefs) Init*;

  BEGIN
    (* nothing to do yet *)
  END Init;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOLED.