(**
   a group object that puts a free defineable frame around its
   single child object.
**)

MODULE VOFrameGroup;

(*
    Makes an frame around an object.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D := VODisplay,
       F := VOFrame,
       G := VOGUIObject,
       T := VOText,
       U := VOUtil;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 mode : SET;
               END;


  FrameGroup*     = POINTER TO FrameGroupDesc;
  FrameGroupDesc* = RECORD (G.GroupDesc)
                      prefs       : Prefs;
                      frame       : F.Frame;
                      horizSpace,
                      vertSpace   : LONGINT;
                      spaceHValue,
                      spaceHType,
                      spaceVValue,
                      spaceVType  : LONGINT;
                    END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.mode:={T.smart};
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(f : FrameGroup);

  BEGIN
    f.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      f.SetBackgroundObject(p.background.Copy());
      f.backgroundObject.source:=f;
    END;
  END SetPrefs;

  PROCEDURE (f : FrameGroup) Init*;

  BEGIN
    f.Init^;

    prefs.SetPrefs(f);

    f.spaceHValue:=0;
    f.spaceHType:=G.sizeGiven;
    f.spaceVValue:=0;
    f.spaceVType:=G.sizeGiven;

    NEW(f.frame);
    f.frame.Init;
    f.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    f.frame.SetInternalFrame(F.group3D);
  END Init;

  PROCEDURE (f : FrameGroup) Add*(object : G.Object);

  BEGIN
    f.list:=object;
    object.parent:=f;
  END Add;

  PROCEDURE (f : FrameGroup) SetFrame*(type : INTEGER);

  BEGIN
    f.frame.SetFrame(type);
  END SetFrame;

  (**
    Set the space beween the frame and its contens in percent. Default to
    value=100 and type=VOGUIObject.spaceFontRel.

    NOTE
    VOGUIObject.sizeScreenRel is not supported.
  **)

  PROCEDURE (f : FrameGroup) SetSpace*(horizValue, horizType, vertValue, vertType : LONGINT);

  BEGIN
    f.spaceHValue:=horizValue;
    f.spaceHType:=horizType;
    f.spaceVValue:=vertValue;
    f.spaceVType:=vertType;
  END SetSpace;

  PROCEDURE (f : FrameGroup) SetLabel*(object : G.Object);

  BEGIN
    f.CopyBackground(object);
    f.frame.SetLabel(object);
  END SetLabel;

  PROCEDURE (f : FrameGroup) SetTextLabel*(string : ARRAY OF CHAR);

  VAR
    text : T.Text;
    help : U.Text;

  BEGIN
    NEW(text);
    text.Init;
    text.SetFlags({G.horizontalFlex,G.verticalFlex});
    text.SetDefault(T.leftAlligned,f.prefs.mode,D.normalFont);
    help:=U.EscapeString(string);
    text.SetText(help^);
    f.frame.SetLabel(text);
  END SetTextLabel;

  PROCEDURE (f : FrameGroup) CalcSize*(display : D.Display);

  BEGIN
    f.horizSpace:=2*f.GetSize(f.spaceHValue,f.spaceHType,0,display.scrWidth,display.spaceWidth,TRUE,display);
    f.vertSpace:=2*f.GetSize(f.spaceVValue,f.spaceVType,0,display.scrHeight,display.spaceHeight,FALSE,display);

    f.frame.CalcSize(display);
    f.width:=f.frame.leftBorder+f.frame.rightBorder+f.horizSpace;
    f.height:=f.frame.topBorder+f.frame.bottomBorder+f.vertSpace;

    f.minWidth:=f.width;
    f.minHeight:=f.height;

    IF f.list#NIL THEN
      f.list.CalcSize(display);
      INC(f.width,f.list.oWidth);
      INC(f.height,f.list.oHeight);
      INC(f.minWidth,f.list.oMinWidth);
      INC(f.minHeight,f.list.oMinHeight);
    END;

    f.minWidth:=G.MaxLong(f.minWidth,f.frame.oMinWidth);
    f.minHeight:=G.MaxLong(f.minHeight,f.frame.oMinHeight);
    f.width:=G.MaxLong(f.width,f.frame.oWidth);
    f.height:=G.MaxLong(f.height,f.frame.oHeight);

    f.CalcSize^(display);
  END CalcSize;

  PROCEDURE (f : FrameGroup) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    f.Draw^(x,y,draw);

    f.DrawBackground(f.x,f.y,f.width,f.height);

    f.frame.Resize(f.width,f.height);
    f.frame.Draw(f.x,f.y,draw);
    IF f.list#NIL THEN
      f.list.Resize(f.width-f.frame.leftBorder-f.frame.rightBorder-f.horizSpace,
                    f.height-f.frame.topBorder-f.frame.bottomBorder-f.vertSpace);
      f.list.Draw(f.x+(f.width-f.list.oWidth) DIV 2,
                  f.y+f.frame.topBorder+(f.height-f.frame.topBorder-f.frame.bottomBorder-f.list.oHeight) DIV 2,
                  draw);
    END;
  END Draw;


  PROCEDURE (f : FrameGroup) Refresh*(x,y,w,h : LONGINT);

  BEGIN
    IF f.visible & f.Intersect(x,y,w,h) THEN
      f.RestrictToBounds(x,y,w,h);
      f.DrawBackground(x,y,w,h);
      f.frame.Refresh(x,y,w,h);
      IF f.list#NIL THEN
        f.list.Refresh(x,y,w,h);
      END;
    END;
  END Refresh;


  PROCEDURE (f : FrameGroup) Hide*;

  BEGIN
    IF f.visible THEN
      f.frame.Hide;
      IF f.list#NIL THEN
        f.list.Hide;
      END;
      f.DrawHide;
      f.Hide^;
    END;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOFrameGroup.