(**
  A image class that implements a number of frames. Use the frames
  implemented here everywhere where possible. Do @emph{not} implement
  your own frames, add them here. Add your frames also here, when you
  want to implement another look.

  TODO
  * nicer frame for string gadget

  * Make some frames support label (groupFrame)

  * Isn't a line a special frame?
**)

MODULE VOFrame;

(*
    Frames.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT D   := VODisplay,
       G   := VOGUIObject,
       P   := VOPrefs;

CONST

  (* Internal Frames *)

  none              * =  0;
  noneSingle        * =  1;
  noneDouble        * =  2;
  single3DIn        * =  3;
  single3DOut       * =  4;
  single            * =  5;
  double            * =  6;
  double3DIn        * =  7;
  double3DOut       * =  8;
  double3DTool      * =  9;
  group3D           * = 10;
  ridge             * = 11;
  doubleFocus       * = 12; (* A thicker frame for motif-like focus *)
  dottedFocus       * = 13; (* A dotted-line focus frame            *)
  w95BO             * = 14; (* Win95 button, released               *)
  w95BI             * = 15; (* Win95 button, pressed                *)
  w95IO             * = 16; (* Win95 image button, released         *)
  w95II             * = 17; (* Win95 image button, pressed          *)
  w95CI             * = 18; (* Container in w95 look                *)
  w95CO             * = 19; (* Container in w95 look                *)
  BBBI              * = 20; (* BeBox Button in                      *)
  BBBO              * = 21; (* BeBox Button out                     *)

  internalFrameCount* = 22;

  (* External Frames, mapped onto the internal ones *)

  buttonFrame       * =  0; (* used by buttons                *)
  sButtonFrame      * =  1; (* Frame for smaller buttons      *)
  groupFrame        * =  2; (* used to group objects          *)
  containerInFrame  * =  3; (* for containers like scroller   *)
  containerFrame    * =  containerInFrame; (* For compability *)
  containerOutFrame * =  4; (* for raised containers          *)
  stringFrame       * =  5; (* string gadget container        *)
  listFrame         * =  6; (* for list-a-like gadgets        *)
  rListFrame        * =  7; (* for list-a-like gadgets        *)
  focusFrame        * =  8; (* A solid focus for showing the keyboard focus *)
  focus2Frame       * =  9; (* A dashed focus for showing focus within a gadget *)
  popupFrame        * = 10; (* Frame for borderless popupwindows *)
  helpFrame         * = 11; (* Frame for QuickHelps *)

  externalFrameCount* = 12;

TYPE
  FrameName       = ARRAY 30 OF CHAR;

  Prefs*     = POINTER TO PrefsDesc;
  PrefsDesc* = RECORD (P.PrefsDesc)
                 frames* : ARRAY internalFrameCount OF LONGINT;
               END;

  Frame *     = POINTER TO FrameDesc;
  FrameDesc * = RECORD (G.ImageDesc)
                  prefs-         : Prefs;
                  type-,
                  intType        : LONGINT;
                  topBorder-,
                  bottomBorder-,
                  leftBorder-,
                  rightBorder-   : LONGINT;
                  frameLabel     : G.Object;
                END;


VAR
  prefs* : Prefs;

  internalFrames- : ARRAY internalFrameCount OF FrameName;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frames[buttonFrame]      :=double3DOut;
    p.frames[sButtonFrame]     :=single3DOut;
    p.frames[groupFrame]       :=group3D;
    p.frames[containerInFrame] :=double3DIn;
    p.frames[containerOutFrame]:=double3DOut;
    p.frames[stringFrame]      :=double3DIn;
    p.frames[listFrame]        :=double3DIn;
    p.frames[rListFrame]       :=double3DOut;
    p.frames[focusFrame]       :=doubleFocus;
    p.frames[focus2Frame]      :=dottedFocus;
    p.frames[popupFrame]       :=double3DOut;
    p.frames[helpFrame]        :=single;
  END Init;

  PROCEDURE (f : Frame) Init*;

  BEGIN
    f.prefs:=prefs;

    f.Init^;
    f.type:=groupFrame;
    f.intType:=-1;
    f.frameLabel:=NIL;
  END Init;

  (**
    Set the frame to be used.

    NOTE
    The frame can be changed at runtime.
  **)

  PROCEDURE (f : Frame) SetFrame*(type : LONGINT);

  BEGIN
    IF f.visible THEN
      f.Hide;
      f.type:=type;
      f.intType:=-1;
      f.Draw(f.x,f.y,f.draw);
    ELSE
      f.type:=type;
      f.intType:=-1;
    END;
  END SetFrame;

  (**
    Set the frame to be used.

    NOTE
    The frame can be changed at runtime.
  **)

  PROCEDURE (f : Frame) SetInternalFrame*(type : LONGINT);

  BEGIN
    IF f.visible THEN
      f.Hide;
      f.intType:=type;
      f.Draw(f.x,f.y,f.draw);
    ELSE
      f.intType:=type;
    END;
  END SetInternalFrame;


  PROCEDURE (f : Frame) SetLabel*(label : G.Object);

  BEGIN
    f.frameLabel:=label;
  END SetLabel;

  PROCEDURE (f : Frame) CalcSize*(display : D.Display);

  VAR
    type : LONGINT;

  BEGIN
    f.width:=0;
    f.height:=0;

    IF f.intType>=0 THEN
      type:=f.intType;
    ELSE
      type:=f.prefs.frames[f.type];
    END;

    CASE type OF
      none:
        f.topBorder:=0;
        f.bottomBorder:=0;
        f.leftBorder:=0;
        f.rightBorder:=0;
    | single3DIn,
      single3DOut,
      noneSingle,
      single:
        f.topBorder:=1;
        f.bottomBorder:=1;
        f.leftBorder:=1;
        f.rightBorder:=1;
    | noneDouble,
      double,
      double3DIn,
      double3DOut,
      double3DTool,
      ridge:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | group3D:
        IF f.frameLabel#NIL THEN
          f.frameLabel.CalcSize(display);
          f.topBorder:=G.MaxLong(f.frameLabel.height,2);
          f.width:=f.frameLabel.width+4*display.spaceWidth;
        ELSE
          f.topBorder:=2;
        END;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | doubleFocus:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | dottedFocus:
        f.topBorder:=1;
        f.bottomBorder:=1;
        f.leftBorder:=1;
        f.rightBorder:=1;
    | w95BO,
      w95BI:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | w95CI,
      w95CO:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | w95II,
      w95IO:
        f.topBorder:=2;
        f.bottomBorder:=2;
        f.leftBorder:=2;
        f.rightBorder:=2;
    | BBBI,
      BBBO:
        f.topBorder:=4;
        f.bottomBorder:=4;
        f.leftBorder:=4;
        f.rightBorder:=4;
    ELSE
        f.topBorder:=0;
        f.bottomBorder:=0;
        f.leftBorder:=0;
        f.rightBorder:=0;
    END;

    INC(f.width,f.leftBorder+f.rightBorder);
    INC(f.height,f.topBorder+f.bottomBorder);
    f.minWidth:=f.width;
    f.minHeight:=f.height;

    f.CalcSize^(display);
  END CalcSize;

  PROCEDURE (f : Frame) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    type,shine,
    shadow,
    halfShadow,
    halfShine,
    top        : LONGINT;

  BEGIN
    f.Draw^(x,y,draw);

    shine:=D.shineColor;
    shadow:=D.shadowColor;
    halfShadow:=D.halfShadowColor;
    halfShine:=D.halfShineColor;


    IF f.intType>=0 THEN
      type:=f.intType;
    ELSE
      type:=f.prefs.frames[f.type];
    END;

    IF D.selected IN draw.mode THEN
      IF type=double3DOut THEN
        type:=double3DIn;
      ELSIF type=single3DOut THEN
        type:=single3DIn;
      ELSIF type=w95BO THEN
        type:=w95BI;
      ELSIF type=BBBO THEN
        type:=BBBI;
      ELSIF type=w95IO THEN
        type:=w95II;
      END;
    ELSIF D.hidden IN draw.mode THEN
      shine:=D.backgroundColor;
      shadow:=D.backgroundColor;
      halfShadow:=D.backgroundColor;
      halfShine:=D.backgroundColor;
    END;

    CASE type OF
      none,
      noneSingle,
      noneDouble:

    | double:
        draw.PushForeground(shadow);
        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y+1);
        draw.DrawLine(f.x,f.y,f.x+f.width-1,f.y);
        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+1,f.y+2);
        draw.DrawLine(f.x+1,f.y+1,f.x+f.width-2,f.y+1);

        draw.DrawLine(f.x+f.width-1,f.y+1,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-2,f.y+2,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x+f.width-2,f.y+f.height-2,f.x+1,f.y+f.height-2);
        draw.PopForeground;

    | double3DIn,
      double3DOut:
        IF type=double3DIn THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y+1);
        draw.DrawLine(f.x,f.y,f.x+f.width-1,f.y);
        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+1,f.y+2);
        draw.DrawLine(f.x+1,f.y+1,f.x+f.width-2,f.y+1);
        draw.PopForeground;

        IF type=double3DIn THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(f.x+f.width-1,f.y+1,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-2,f.y+2,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x+f.width-2,f.y+f.height-2,f.x+1,f.y+f.height-2);
        draw.PopForeground;

    | double3DTool:
        IF (draw.mode={}) OR (draw.mode={D.hidden}) THEN
          draw.PushForeground(D.backgroundColor);
        ELSIF draw.mode={D.selected} THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;

        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y+1);
        draw.DrawLine(f.x,f.y,f.x+f.width-1,f.y);
(*        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+1,f.y+2);
        draw.DrawLine(f.x+1,f.y+1,f.x+f.width-2,f.y+1);*)
        draw.PopForeground;


        IF (draw.mode={}) OR (draw.mode={D.hidden}) THEN
          draw.PushForeground(D.backgroundColor);
        ELSIF draw.mode={D.selected} THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(f.x+f.width-1,f.y+1,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,f.y+f.height-1);
(*        draw.DrawLine(f.x+f.width-2,f.y+2,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x+f.width-2,f.y+f.height-2,f.x+1,f.y+f.height-2);*)
        draw.PopForeground;

    | single3DIn,
      single3DOut :
        IF type=single3DIn THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y+1);
        draw.DrawLine(f.x,f.y,f.x+f.width-1,f.y);
        draw.PopForeground;

        IF type=single3DIn THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(f.x+f.width-1,f.y+1,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,f.y+f.height-1);
        draw.PopForeground;

    | single:
        draw.PushForeground(shadow);
        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y+1);
        draw.DrawLine(f.x,f.y,f.x+f.width-1,f.y);
        draw.DrawLine(f.x+f.width-1,f.y+1,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,f.y+f.height-1);
        draw.PopForeground;

    | group3D:
        IF f.frameLabel#NIL THEN
          top:=f.y+f.frameLabel.height DIV 2;
        ELSE
          top:=f.y;
        END;
        draw.PushForeground(shadow);
        draw.DrawLine(f.x,f.y+f.height-1,f.x,top);
        draw.DrawLine(f.x+f.width-2,top+1,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+f.width-2,f.y+f.height-2);
        IF f.frameLabel#NIL THEN
          draw.DrawLine(f.x,top,f.x+f.display.spaceWidth-1,top);
          draw.DrawLine(f.x+3*f.display.spaceWidth+f.frameLabel.width+1,top,f.x+f.width-1,top);
          draw.DrawLine(f.x+3*f.display.spaceWidth+f.frameLabel.width,top,f.x+3*f.display.spaceWidth+f.frameLabel.width,top+1);
        ELSE
          draw.DrawLine(f.x,top,f.x+f.width-1,top);
        END;
        draw.PopForeground;

        draw.PushForeground(shine);
        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+1,top+1);
        draw.DrawLine(f.x+f.width-1,top+1,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+1,f.y+f.height-1,f.x+f.width-2,f.y+f.height-1);
        IF f.frameLabel#NIL THEN
          draw.DrawLine(f.x+2,top+1,f.x+f.display.spaceWidth-1,top+1);
          draw.DrawLine(f.x+3*f.display.spaceWidth+f.frameLabel.width+1,top+1,f.x+f.width-2,top+1);
          draw.DrawLine(f.x+f.display.spaceWidth,top,f.x+f.display.spaceWidth,top+1);
        ELSE
          draw.DrawLine(f.x+2,top+1,f.x+f.width-2,top+1);
        END;
        draw.PopForeground;
        IF f.frameLabel#NIL THEN
          IF ~(D.hidden IN draw.mode) THEN
            f.frameLabel.Draw(f.x+f.leftBorder+2*f.display.spaceWidth,f.y,draw);
          ELSE
            f.frameLabel.Hide;
          END;
        END;

    | ridge:
        top:=f.y;
        draw.PushForeground(shine);
        draw.DrawLine(f.x,f.y,f.x+f.width-2,f.y);
        draw.DrawLine(f.x+2,f.y+f.height-2,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y);
        draw.DrawLine(f.x+f.width-2,f.y+2,f.x+f.width-2,f.y+f.height-2);
        draw.PopForeground;

        draw.PushForeground(shadow);
        draw.DrawLine(f.x+f.width-1,f.y,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+1,f.y+1);
        draw.DrawLine(f.x+1,f.y+f.height-1,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+1,f.y+1,f.x+f.width-2,f.y+1);
        draw.PopForeground;

    | doubleFocus:
        IF D.hidden IN draw.mode THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(D.focusColor);
        END;
        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y+1);
        draw.DrawLine(f.x,f.y,f.x+f.width-1,f.y);
        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+1,f.y+2);
        draw.DrawLine(f.x+1,f.y+1,f.x+f.width-2,f.y+1);
        draw.DrawLine(f.x+f.width-1,f.y+1,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-2,f.y+2,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x+f.width-2,f.y+f.height-2,f.x+1,f.y+f.height-2);
        draw.PopForeground;
    | dottedFocus:
        IF D.hidden IN draw.mode THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(D.textColor);
        END;
        draw.PushDash(D.sPointLine,D.fMode);
        draw.DrawLine(f.x          ,f.y,           f.x+f.width-1,f.y);
        draw.DrawLine(f.x+f.width-1,f.y,           f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,          f.y+f.height-1);
        draw.DrawLine(f.x          ,f.y+f.height-1,f.x,          f.y);
        draw.PopDash;
        draw.PopForeground;
    | w95BO,
      w95BI:
        IF type=w95BI THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y+1);
        draw.DrawLine(f.x,f.y,f.x+f.width-2,f.y);
        draw.PopForeground;

        IF type=w95BI THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(D.backgroundColor);
        END;
        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+1,f.y+2);
        draw.DrawLine(f.x+1,f.y+1,f.x+f.width-3,f.y+1);
        draw.PopForeground;

        IF type=w95BI THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(f.x+f.width-1,f.y,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,f.y+f.height-1);
        draw.PopForeground;

        IF type=w95BI THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(halfShadow);
        END;
        draw.DrawLine(f.x+f.width-2,f.y+1,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x+f.width-2,f.y+f.height-2,f.x+1,f.y+f.height-2);
        draw.PopForeground;
    | w95CI,
      w95CO:
        IF type=w95CI THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(halfShine);
        END;
        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y+1);
        draw.DrawLine(f.x,f.y,f.x+f.width-2,f.y);
        draw.PopForeground;

        IF type=w95CI THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+1,f.y+2);
        draw.DrawLine(f.x+1,f.y+1,f.x+f.width-3,f.y+1);
        draw.PopForeground;

        IF type=w95CI THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(f.x+f.width-1,f.y,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,f.y+f.height-1);
        draw.PopForeground;

        IF type=w95CI THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(halfShadow);
        END;
        draw.DrawLine(f.x+f.width-2,f.y+1,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x+f.width-2,f.y+f.height-2,f.x+1,f.y+f.height-2);
        draw.PopForeground;
    | w95II,
      w95IO:
        IF type=w95II THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(D.backgroundColor);
        END;
        draw.DrawLine(f.x,f.y+f.height-1,f.x,f.y+1);
        draw.DrawLine(f.x,f.y,f.x+f.width-2,f.y);
        draw.PopForeground;

        IF type=w95II THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(f.x+1,f.y+f.height-2,f.x+1,f.y+2);
        draw.DrawLine(f.x+1,f.y+1,f.x+f.width-3,f.y+1);
        draw.PopForeground;

        IF type=w95II THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(f.x+f.width-1,f.y,f.x+f.width-1,f.y+f.height-1);
        draw.DrawLine(f.x+f.width-1,f.y+f.height-1,f.x,f.y+f.height-1);
        draw.PopForeground;

        IF type=w95II THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(halfShadow);
        END;
        draw.DrawLine(f.x+f.width-2,f.y+1,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x+f.width-2,f.y+f.height-2,f.x+1,f.y+f.height-2);
        draw.PopForeground;
    | BBBI,
      BBBO:
        (* outest frame *)
        IF D.hidden IN draw.mode THEN
          draw.PushForeground(D.backgroundColor);
        ELSE
          draw.PushForeground(D.blackColor);
        END;
        draw.DrawLine(f.x+1,f.y,f.x+f.width-2,f.y);
        draw.DrawLine(f.x+1,f.y+f.height-1,f.x+f.width-2,f.y+f.height-1);
        draw.DrawLine(f.x,f.y+1,f.x,f.y+f.height-2);
        draw.DrawLine(f.x+f.width-1,f.y+1,f.x+f.width-1,f.y+f.height-2);
        draw.PopForeground;

        (* 2nd frame *)

        draw.PushForeground(D.backgroundColor);
        draw.DrawLine(f.x+1,f.y+1,f.x+f.width-2,f.y+1);
        draw.DrawLine(f.x+1,f.y+2,f.x+1,f.y+f.height-2);
        draw.PopForeground;

        IF type=BBBI THEN
          draw.PushForeground(shine);
        ELSE
          draw.PushForeground(shadow);
        END;
        draw.DrawLine(f.x+f.width-2,f.y+2,f.x+f.width-2,f.y+f.height-2);
        draw.DrawLine(f.x+2,f.y+f.height-2,f.x+f.width-2,f.y+f.height-2);
        draw.PopForeground;

        (* 3rd frame *)

        IF type=BBBI THEN
          draw.PushForeground(halfShadow);
        ELSE
          draw.PushForeground(halfShine);
        END;
        draw.DrawLine(f.x+2,f.y+2,f.x+f.width-3,f.y+2);
        draw.DrawLine(f.x+2,f.y+3,f.x+2,f.y+f.height-3);
        draw.PopForeground;

        IF type=BBBI THEN
          draw.PushForeground(halfShine);
        ELSE
          draw.PushForeground(halfShadow);
        END;
        draw.DrawLine(f.x+f.width-3,f.y+3,f.x+f.width-3,f.y+f.height-3);
        draw.DrawLine(f.x+4,f.y+f.height-3,f.x+f.width-4,f.y+f.height-3);
        draw.PopForeground;

        (* 4th frame *)

        IF type=BBBI THEN
          draw.PushForeground(shadow);
        ELSE
          draw.PushForeground(shine);
        END;
        draw.DrawLine(f.x+3,f.y+3,f.x+f.width-4,f.y+3);
        draw.DrawLine(f.x+3,f.y+4,f.x+3,f.y+f.height-4);
        draw.PopForeground;
    END;
  END Draw;

  PROCEDURE (f : Frame) Hide*;

  BEGIN
    IF f.visible THEN
      f.draw.mode:={D.hidden};
      f.Redraw;
      f.draw.mode:={};
      f.Hide^;
    END;
  END Hide;

  PROCEDURE GetFrameEntry*(name : ARRAY OF CHAR):LONGINT;

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=0 TO internalFrameCount-1 DO
      IF name=internalFrames[x] THEN
        RETURN x;
      END;
    END;
    RETURN -1;
  END GetFrameEntry;

BEGIN
  internalFrames[none]        :="none";
  internalFrames[noneSingle]  :="noneSingle";
  internalFrames[noneDouble]  :="noneDouble";
  internalFrames[single3DIn]  :="single3DIn";
  internalFrames[single3DOut] :="single3DOut";
  internalFrames[single]      :="single";
  internalFrames[double]      :="double";
  internalFrames[double3DIn]  :="double3DIn";
  internalFrames[double3DOut] :="double3DOut";
  internalFrames[double3DTool]:="double3DTool";
  internalFrames[group3D]     :="group3D";
  internalFrames[ridge]       :="ridge";
  internalFrames[doubleFocus] :="doubleFocus";
  internalFrames[dottedFocus] :="dottedFocus";
  internalFrames[w95BO]       :="w95BO";
  internalFrames[w95BI]       :="w95BI";
  internalFrames[w95IO]       :="w95IO";
  internalFrames[w95II]       :="w95II";
  internalFrames[w95CI]       :="w95CI";
  internalFrames[w95CO]       :="w95CO";
  internalFrames[BBBI]        :="BBBI";
  internalFrames[BBBO]        :="BBBO";

  NEW(prefs);
  prefs.Init;
END VOFrame.