(**
  modified by Marco in some places
 **)


(**
  This module is responsible for the handling of the display and all
  its attributes like color, font and that sort. It also exports the
  class @code{DrawInfo} which implements an abstract, that means
  OS-independend, drawing engine.

  TODO
  * Busy function for *all* windows

  * Rethink pens
**)

MODULE VODisplay;

(*
    Classes for visualisation.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT s    := SYSTEM,

       D    := VODragDrop,
       E    := VOEvent,
       LI   := VOLLInt,
       LM   := VOLLMac,
       O    := VOObject,
       P    := VOPrefs,
       U    := VOUtil,

       a    := Xatom,
               C,
       c    := Calendar,
       ch   := Channel,
       co   := IntStr,
               Err,
 Out,      p    := PosixFileDescr,
       sc   := SysClock,
       str  := Strings,
       str2 := Strings2,
       t    := Time,

<* IF LIB_HAVE_LIBIMLIB=TRUE THEN *>

               Imlib,

<* END *>

               X11,
       xu   := Xutil;

CONST
  contextTimeOutSec = 3; (* seconds until context help will displayed *)
  contextTimeOutMil = 0; (* microseconds until context help will displayed *)

  (* Size of the various stack arrays *)
  initialStackSize   = 30;
  stackSizeIncrement = 30;

  (* Different messages the display can listen to *)

  exitMsg*    = 0;
  timeOutMsg* = 1;

  sleepCursor = 150; (* from /usr/include/X11/cursorFont.h *)
  popCursor   =  94;
  dndCursor   =  50;

  (* Different modes of display color (Display.colorMode) *)

  monochromeMode * = 0;
  greyScaleMode  * = 1;
  colorMode      * = 2;

  (* Pens *)

  backgroundColor       * =  0; (* background color of windows *)
  tableBackgroundColor  * =  1; (* background of table like object *)
  tableTextColor        * =  2;
  textBackgroundColor   * =  3; (* backgrond of text pinput objects *)
  buttonBackgroundColor * =  4; (* background color of button-like objects, should be close to backgroundColor *)
  textColor             * =  5; (* normal text background *)
  shineColor            * =  6;
  halfShineColor        * =  7;
  halfShadowColor       * =  8;
  shadowColor           * =  9;
  fillColor             * = 10; (* color for visualizing selection of filling *)
  fillTextColor         * = 11; (* textcolor while selecting *)
  warnColor             * = 12; (* warn color *)
  disabledColor         * = 13; (* color of "disable"grid *)
  focusColor            * = 14; (* color of focus frame *)
  blackColor            * = 15;
  whiteColor            * = 16;
  helpBackgroundColor   * = 17; (* background color of tooltips *)

  colorCount            * = 18;

  (* The different supported fonts  *)

  tinyFont       * =  0;
  scriptFont     * =  1;
  footnoteFont   * =  2;
  smallFont      * =  3;
  normalFont     * =  4;
  largeFont      * =  5;
  LargeFont      * =  6;
  LARGEFont      * =  7;
  hugeFont       * =  8;
  HUGEFont       * =  9;

  bigFont        * = LargeFont;

  smallFixedFont * = 10;
  fixedFont      * = 11;
  hugeFixedFont  * = 12;

  fontCount      * = 13;

  (*
    The different styles a font can have.
    NOTE
    Not all styles can used together. italic and slanted f.e. are
    exclusive.
  *)

  bold        * = 0;
  italic      * = 1;
  underlined  * = 2;
  slanted     * = 3;

  maxStyle    * = 3;
  maxStyleNum * = 2*2*2*2;
  styleMask   * = {bold,italic,underlined,slanted};

  (*
    List of features a font can have. A Font structure has a number of
    attributes. The feature list tells which features are known to this font.
    (A font selection must only that parameter ofa font that are within its
    feature list.

    The other way round, when loading a font you fill in a number of features
    about that font to specify it. VisualOberon is free to ignore some features.
  *)

  fontName*        =  0; (* This should always be set *)
  fontHeight*      =  1;
  fontAscent*      =  2; (* I'm not sure if these two are really  feature *)
  fontDescent*     =  3;
  fontPointHeight* =  4;
  fontHorizRes*    =  5;
  fontVertRes*     =  6;
  fontAvWidth*     =  7;
  fontSetWidth*    =  8;
  fontSpacing*     =  9;
  fontCharSet*     = 10;


  (*
    font width for the above fontSetWidth feature
  *)

  fontNormal *   = 0;
  fontCondensed* = 1;
  fontNarrow*    = 2;
  fontDouble*    = 3;

  (*
    font spacing for the above fontSpacing feature
  *)

  fontFixed*        = 0;
  fontProportional* = 1;
  (* there is also "c", whats that? *)



  (*
    Different DrawInfo.mode
    Not all modes other than "normal" need to be implemented
  *)

  selected*  = 0; (* Not allways supported *)
  hidden*    = 1; (* Can be used interaly, need not to be supported *)
  activated* = 2; (* Active but not selected *)
  disabled*  = 3; (* Draw yourself disabled *)

  (* pen styles *)

  normalPen * = 0;
  roundPen  * = 1;

  (* draw mode *)

  copy   * = 0;
  invert * = 1;

  (* dash modes *)

  fMode * = 0; (* Draw dashes in foreground color *)
  fbMode* = 1; (* Draw dashes in foreground color, gaps in background color *)

  (* pattern styles *)

  fgPattern* = 0; (* set bits are draw in foreground color, unset bits arn't used *)
  fbPattern* = 1; (* set bits are draw in foreground color, unset in background   *)

  (* various types of data *)

  none * = 0;
  text * = 1;

  (* flags for window positioning *)

  manualPos*      = 0;
  centerOnParent* = 1;
  centerOnScreen* = 2;

  (* constants for Xdnd *)

  XDND_VERSION = 3;

TYPE
  ColorName     = ARRAY 30 OF CHAR;
  FontName      = ARRAY 100 OF CHAR;

  Window*       = POINTER TO WindowDesc;
  Display*      = POINTER TO DisplayDesc;
  ClipEntry     = POINTER TO ClipEntryDesc;
  DrawInfo*     = POINTER TO DrawInfoDesc;
  Font*         = POINTER TO FontDesc;

  PrefsCallback = PROCEDURE (appName : ARRAY OF CHAR; display : Display);

  FreeEntry     = POINTER TO FreeEntryDesc;
  FreeEntryDesc = RECORD
                    next   : FreeEntry;
                    object : O.Object;
                  END;


  Color*        = LONGINT;


  Object*     = POINTER TO ObjectDesc;

  (**
    The display defines its GUIobject to be able to store pointers
    to GUIobjects. VOGUIObject.Object inherits from this object.
  **)

  ObjectDesc* = RECORD (O.MsgObjectDesc)
                END;

  (**
    The abstract baseclass for all windows. All windows should derive
    from this class and call the apropiate methods where stated.

    NOTE
    Window inherits from O.MsgObject, you can send messages to it.
  **)

  WindowDesc*   = RECORD (O.MsgObjectDesc)
                    last,next      : Window;
                    window*,
                    parentWin-     : X11.Window;
                    parent-        : Window;

                    display-       : Display;
                    draw-          : DrawInfo;

                    title-         : U.Text;

                    modalCount-    : LONGINT;

                    background-    : Color;

                    draws          : LONGINT; (* The number of calls to draw since opening. TODO: change to bool *)

                    x-,y-,
                    width-,
                    height-,
                    minWidth-,
                    minHeight-,
                    maxWidth-,
                    maxHeight-     : LONGINT;

                    horizontalPos-,
                    verticalPos-   : LONGINT;

                    freeList       : FreeEntry;

                    modal,
                    maped-,
                    focus-,
                    exposed,
                    inited,
                    grab,
                    createAllways,
                    borderless,
                    open-          : BOOLEAN;
                  END;

  Bitmap*       = POINTER TO BitmapDesc;
  BitmapDesc*   = RECORD
                    draw-   : DrawInfo;
                    width-,
                    height- : LONGINT;
                    pixmap- : X11.Pixmap; (* only in X11 implementation! *)
                  END;

  (**
    Abstract fontdefinition.
  **)

  FontDesc*     = RECORD
                    next-,
                    last-        : Font;
                    fullName     : U.Text;
                    features*    : SET;
                    name*,
                    charSet*     : U.Text;
                    ascent*,
                    descent*,
                    height*,
                    pointHeight*,
                    horizRes*,
                    vertRes*,
                    avWidth*,
                    setWidth*,
                    spacing*     : LONGINT;
                    loaded       : BOOLEAN;
                    fonts        : ARRAY maxStyleNum OF X11.XFontStructPtr;
                    useCount     : LONGINT;
                    id           : LONGINT;
                  END;

  (**
    Returned by Font.FontExtent.
  **)

  FontExtentDesc* = RECORD
                      lbearing-,
                      rbearing-,
                      width-,
                      height-,
                      ascent-,
                      descent-  : LONGINT;
                    END;

  TimeOut*    = POINTER TO TimeOutDesc;

  (**
    This is the class, timeout information are stored in. VODisplay
    bilds a sorted list of TimeOuts to store all current timeouts in it.
  **)

  TimeOutDesc = RECORD
                  next   : TimeOut;
                  time   : t.TimeStamp;
                  object : O.MsgObject;
                END;

  Sleep*          = POINTER TO SleepDesc;

  (**
    This is the class, sleep information are stored in. VODisplay
    bilds a sorted list of Sleeps to store all current sleep callbacks in it.
  **)

  SleepDesc       = RECORD
                       next   : Sleep;
                       object : O.MsgObject;
                     END;

  Channel*        = POINTER TO ChannelDesc;

  (**
    This structure holds information about channels the main event loop
    should wait for. The display object notifies the given object if the
    handed channel becomes available (currently this means, there is some data
    in the channel that can be read, the event loop engine may support
    event sneding on channel getting available for writing in the future, too).
  **)

  ChannelDesc     = RECORD
                       next    : Channel;
                       channel : ch.Channel;
                       object  : O.MsgObject;
                     END;

  ColorEntry*     = POINTER TO ColorEntryDesc;

  (**
    Internal structure to store information about on color entry.

    NOTE
    color should hold information about OS specific colorinformation.
  **)

  ColorEntryDesc* = RECORD
                      useCount : LONGINT;
                      color    : X11.XColor;
                    END;

 (**
   Internaly used for storing user allocated colors.

   NOTE
   I'm not sure of that is the way to go. If the user allocated a huge amount
   of colors (1 million f.e.) we get a huge amount of storage overhead.
 **)

  ColorList = POINTER TO ARRAY OF ColorEntryDesc;

  (**
    Class that abstracts the properties of a display.
  **)

  DisplayDesc * = RECORD (O.MsgObjectDesc)
                    appName-     : POINTER TO ARRAY OF CHAR;

                    display*     : X11.DisplayPtr;
                    name-        : ARRAY 100 OF CHAR;
                    scrNum*,
                    scrWidth-,
                    scrHeight-   : C.int;

                    visual*      : X11.VisualPtr;

<* IF LIB_HAVE_LIBIMLIB=TRUE THEN *>
                    imlib*       : Imlib.DataPtr;
<* END *>

                    (* various cursors *)
                    sleepCursor-,
                    popCursor-,
                    dndCursor-,
                    copyCursor-,
                    moveCursor-,
                    linkCursor-  : X11.Cursor;

                    (* general sizing information *)
                    spaceHeight-,
                    spaceWidth-  : LONGINT;

                    (* global focusing object *)
                    focusBorder- : LONGINT; (* The border width of the focus frame
                                               must be hardcoded :-| *)

                    (* colors *)
                    colorMode-   : LONGINT;
                    colorDepth*  : LONGINT;
                    colorMap*    : X11.Colormap;
                    color-       : ARRAY colorCount OF ColorEntryDesc;
                    userColor-   : ColorList;

                    (* fonts *)
                    font         : ARRAY fontCount OF Font;
                    fonts        : Font;
                    usrFontID    : LONGINT;

                    deleteProt-  : ARRAY 1 OF X11.Atom;

                    (* windowing information *)
                    mainWin-     : X11.Window;
                    winList      : Window;
                    winCount-    : LONGINT;
                    currentWin-  : Window;
                    newWin-      : Window;

                    (* selection stuff *)
                    selectObject,
                    querySelectObject : Object;

                    (* timeouts *)
                    timeOutList  : TimeOut;
                    timeOut      : TimeOut; (* The current timeout for context help *)
                    contextHelp  : BOOLEAN;

                    (* sleep stuff *)
                    sleepList    : Sleep;

                    (* fd stuff *)

                    channelList  : Channel;

                    (* X property types *)
                    atom         : X11.Atom;

                    (* Xdnd stuff *)
                    XdndAware,
                    XdndEnter,
                    XdndLeave,
                    XdndPosition,
                    XdndStatus,
                    XdndFinished,
                    XdndDrop,
                    XdndActionCopy,
                    XdndActionMove,
                    XdndActionLink,
                    XdndActionAsk,
                    XdndActionPrivate,
                    XdndActionList,
                    XdndSelection,
                    XdndTypeList : X11.Atom;

                    (* Motif drag & drop *)
                    dndMessage,
                    dndIniProp,
                    dndRecProp,
                    dndSuccess,
                    dndFailure,
                    xSelection,
                    xDrop,
                    dSelection,
                    wmState      : X11.Atom;
                    dropWindow   : X11.Window;

                    (* general DnD *)
                    dragObject   : Object;
                    dragInfo     : D.DnDDataInfo;
                    dndAction    : LONGINT;
                    dragX,dragY  : LONGINT; (* x,y coord of the drag start point *)
                    dragging,               (* we are currently dragging (not handled yet) *)
                    dragStart    : BOOLEAN; (* Button down, check for dragging on mouse move *)

                    exit         : BOOLEAN;
(** Modified by Marco **)
(* new elements for doubleClick, see Display.DoubleClicked *)
                    lastEventIsButtonPress : BOOLEAN;
                    lastButton             : SHORTINT;
                    lastBut1Button         : SHORTINT;
                    lastPressTime          : X11.Time;
                    lastBut1PressTime      : X11.Time;
                    multiClickTime         : X11.Time;
(* end of modification *)
                  END;


  DisplayPrefs*     = POINTER TO DisplayPrefsDesc;

  (**
    In this class all preferences stuff of the display is stored.
  **)

  DisplayPrefsDesc* = RECORD (P.PrefsDesc)
                        display- : Display;
                        colors*  : ARRAY colorCount OF ColorName;
                        fonts*   : ARRAY fontCount  OF FontName;
                      END;

  ExitMsg*     = POINTER TO ExitMsgDesc;

  (**
    Send this message to Display if you want to leave the application.
  **)

  ExitMsgDesc*    = RECORD (O.MessageDesc)
                    END;

  TimeOutMsg*     = POINTER TO TimeOutMsgDesc;

  (**
    The message an objects gets, when the given timeout runs out.
  **)

  TimeOutMsgDesc* = RECORD (O.MessageDesc)
                      timeOut* : TimeOut;
                    END;

  SleepMsg*       = POINTER TO SleepMsgDesc;

  (**
    The message an objects gets, when a registered sleep event get called.
  **)

  SleepMsgDesc*   = RECORD (O.MessageDesc)
                      sleep* : Sleep;
                    END;

  ChannelMsg*     = POINTER TO ChannelMsgDesc;

  (**
    The message an objects gets, when a registered fd event get called.
  **)

  ChannelMsgDesc* = RECORD (O.MessageDesc)
                      channel* : Channel;
                    END;

  Msg2Exit*       = POINTER TO Msg2ExitDesc;

  (**
    A converter that throuws the incomming message away and generates a
    ExitMsg for Display.
  **)

  Msg2ExitDesc* = RECORD (O.HandlerDesc)
                  END;


  ClipEntryDesc = RECORD
                    last    : ClipEntry;
                    region  : xu.Region;
                    draw    : DrawInfo;
                  END;


  PenColor      = POINTER TO PenColorDesc;
  PenColorDesc  = RECORD
                    next  : PenColor;
                    color : Color;
                  END;

  DrawMode      = POINTER TO DrawModeDesc;
  DrawModeDesc  = RECORD
                    next : DrawMode;
                    mode : LONGINT;
                  END;

  PenStyle      = POINTER TO PenStyleDesc;
  PenStyleDesc  = RECORD
                    next  : PenStyle;
                    size,
                    mode,
                    cap,
                    join  : LONGINT;
                  END;

  PenDash       = POINTER TO PenDashDesc;
  PenDashDesc   = RECORD
                    next : PenDash;
                    list : POINTER TO ARRAY OF CHAR;
                    mode : LONGINT;
                  END;

  Pattern       = POINTER TO PatternDesc;
  PatternDesc   = RECORD
                    next   : Pattern;
                    pixMap : X11.Pixmap;
                    mode   : LONGINT;
                  END;

  FontEntry     = POINTER TO FontEntryDesc;
  FontEntryDesc = RECORD
                    next  : FontEntry;
                    font  : Font;
                    style : SET;
                  END;

  (**
    A point has an x and an y member with size INTEGER.
    However the X11-function wants a XPoint as parameter,
    so we simply alias from it. Ports may implement this
    differently but they must have two members x and y.
  **)

  PointDesc* = X11.XPoint;

  ClipStack = POINTER TO ARRAY OF ClipEntryDesc;
  PenStack  = POINTER TO ARRAY OF PenColorDesc;
  FontStack = POINTER TO ARRAY OF FontEntryDesc;

  (**
    The class for all abstract drawings.
  **)

  DrawInfoDesc* = RECORD
                    display-      : Display;
                    window-       : X11.Window;
                    vWindow-      : Window;
                    gc-           : X11.GC;

                    clipStack     : ClipStack;
                    clipPos       : LONGINT;
                    fPenStack     : PenStack;
                    fPenPos       : LONGINT;
                    bPenStack     : PenStack;
                    bPenPos       : LONGINT;
                    styleStack    : PenStyle;
                    dashStack     : PenDash;
                    patternStack  : Pattern;
                    fontStack     : FontStack;
                    fontPos       : LONGINT;
                    modeStack     : DrawMode;

                    mode*         : SET;
                 END;

(*  Data*     = POINTER TO DataDesc;
  DataDesc* = RECORD
                type*   : LONGINT;
                length* : LONGINT;

                string* : U.Text;
                ints*   : POINTER TO ARRAY OF INTEGER;
                longs*  : POINTER TO ARRAY OF LONGINT;

                xData   : C.charPtr1d;
              END;*)

CONST
  disableWidth*  = 8;
  disableHeight* = 2;

VAR

  (*
    This function will be called when all inportant display filed are filled
    and before the corresponding preferences settings are evaluated.
  *)

  prefsCallback* : PrefsCallback;

  (* Pattern for disabling of gadgets *)

  disablePattern- : ARRAY 2 OF CHAR;

  (* copy cursor *)

  copyCursorData,
  copyMaskData   : ARRAY 30 OF CHAR;

  moveCursorData,
  moveMaskData   : ARRAY 26 OF CHAR;

  linkCursorData,
  linkMaskData   : ARRAY 34 OF CHAR;

  (* Diverse patterns for dashlines *)

  sPointLine- : ARRAY 2 OF CHAR; (* small  points *)
  mPointLine- : ARRAY 2 OF CHAR; (* medium points *)
  lPointLine- : ARRAY 2 OF CHAR; (* large points  *)

  colorNames- : ARRAY colorCount OF ColorName;
  fontNames-  : ARRAY fontCount OF FontName;

  prefs*,
  monoPrefs*,
  greyPrefs*,
  colorPrefs* : DisplayPrefs;

  (* ------------ Utility-stuff --------------- *)

  (**
     Frees memory allocated using X11 if the pointer is not @code{NIL}.
  **)

  PROCEDURE XFree*(pointer : s.PTR);

  BEGIN
    IF pointer#NIL THEN
      X11.XFree(s.VAL(LONGINT,pointer));
    END;
  END XFree;

  (**
    Returns "l" if we are on a little endian machine, else "B".
    This is usefull for analysing Motif drag and drop messages.
  **)

(*  PROCEDURE GetEndianess():CHAR;

  VAR
    integer : INTEGER;
    bytes   : ARRAY SIZE(INTEGER) OF CHAR;

  BEGIN
    integer:=1;
    s.MOVE(s.ADR(integer),s.ADR(bytes),SIZE(INTEGER));

    IF bytes[0]#0X THEN
      RETURN "l";
    ELSE
      RETURN "B";
    END;
  END GetEndianess;*)


  (* ------------ object stuff --------------- *)

  (**
    Before the data drop actually occurs the drag object will be asked
    for a list of supported datatypes. This will then be handed to the
    drop object which than can select a apropiate datatype. This type will
    the requested from the drag object and after that will be handed to the
    drop object.

    NOTE
    The is no way to cancel the action at that point the object has already
    made itself valid by answering VOGUIObject.Object.GetDnDObject.

    This is only true for internal drag and drop. External drag and drop may
    leave some of the described steps. An object must be prepared of that.
  **)

  PROCEDURE (object : Object) GetDragInfo*(VAR dragInfo : D.DnDDataInfo);

  BEGIN
  END GetDragInfo;

  (**
    All objects support drag actions. Return a derived class from
    VODragDrop.DnDData if you can offer drag data of the given type
    or NIL.

    NOTE
    data should really be of type DataDesc, but there seems to be
    compiler bugs with it. We will change this later.
  **)

  PROCEDURE (object : Object) GetDragData*(group, type, action : LONGINT):D.DnDData;

  BEGIN
    RETURN NIL;
  END GetDragData;

  (**
    The object gets a list of supported datatypes of the drag object and
    has to choose one of them. If there is no fitting datatype it must return
    FALSE.
  **)

  PROCEDURE (object : Object) GetDropDataType*(VAR dragInfo : D.DnDDataInfo;
                                               VAR group,type,action : LONGINT):BOOLEAN;

  BEGIN
    RETURN FALSE;
  END GetDropDataType;

  (**
    All object can handle drop actions. Return TRUE if you have
    handled the drop event.
  **)

  PROCEDURE (object : Object) HandleDrop*(data : D.DnDData; action : LONGINT):BOOLEAN;

  BEGIN
    RETURN FALSE;
  END HandleDrop;

  (**
    Gets called, when the object has registered a selection using
    Display.RegisterSelection and now should deselect itself because
    somebody else has registered an selection.
  **)

  PROCEDURE (object : Object) Deselect*;

  BEGIN
  END Deselect;

  (* ------------ Data Exchange stuff --------------- *)

  (**
    Initializes an Data object.
  **)

(*  PROCEDURE (data : Data) Init*;

  BEGIN
    data.type:=none;
    data.length:=0;

    data.string:=NIL;
    data.ints:=NIL;
    data.longs:=NIL;

    data.xData:=NIL;
  END Init;

  PROCEDURE (data : Data) Free*;

  BEGIN
    IF data.xData#NIL THEN
      XFree(data.xData);
    END;
  END Free;

  PROCEDURE (data : Data) InitFromX(window : Window; event : X11.XSelectionEvent):BOOLEAN;

  VAR
    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    delete      : X11.Bool;

  BEGIN
    data.Init;

    IF event.property=X11.None THEN
      Err.String("Selection denied"); Err.Ln;
      RETURN FALSE;
    END;

    IF window.display.dragging THEN
      delete:=X11.False;
    ELSE
      delete:=X11.True;
    END;

    IF X11.XGetWindowProperty(window.display.display,
        event.requestor,event.property,0,MAX(LONGINT),delete,
        X11.AnyPropertyType,retType,retFormat,itemsReturn,bytesLeft,
        data.xData)#X11.Success THEN
        Err.String("Cannot get property"); Err.Ln;
        RETURN FALSE;
    END;

    IF retType=X11.None THEN
      RETURN FALSE;
    END;

    CASE retType OF
      a.XA_STRING: data.type:=text;
    ELSE
      RETURN FALSE;
    END;

    CASE retFormat OF
       8: NEW(data.string,itemsReturn+1);
          s.MOVE(s.VAL(LONGINT,data.xData),
                 s.VAL(LONGINT,data.string),itemsReturn);
          data.string[itemsReturn]:=0X;

          Err.String(">>"); Err.String(data.string^); Err.String("<<"); Err.Ln;

    | 16: NEW(data.ints,itemsReturn+1);
          s.MOVE(s.VAL(LONGINT,data.xData),
                 s.VAL(LONGINT,data.ints),itemsReturn*2);
    | 32: NEW(data.longs,itemsReturn+1);
          s.MOVE(s.VAL(LONGINT,data.xData),
                 s.VAL(LONGINT,data.longs),itemsReturn*4);
    END;

    data.length:=itemsReturn;

    IF window.display.dragging THEN
      Err.String("Finishing drop selection"); Err.Ln;
      X11.XConvertSelection(window.display.display,
                            window.display.dSelection,
                            window.display.dndSuccess,
                            window.display.xSelection,
                            window.window,X11.CurrentTime);
      window.display.dragging:=FALSE;
    END;

    RETURN TRUE;
  END InitFromX;

  PROCEDURE (data : Data) InitToX(VAR event : X11.XSelectionEvent):BOOLEAN;

  VAR
    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;

  BEGIN
    IF (event.target=a.XA_STRING) & (data.string#NIL) THEN

      IF X11.XGetWindowProperty(event.display,
        event.requestor,event.property,0,MAX(LONGINT),X11.False,
        event.target,retType,retFormat,itemsReturn,bytesLeft,
        data.xData)#X11.Success THEN
          Err.String("Cannot get property"); Err.Ln;
          RETURN FALSE;
      END;

      X11.XChangeProperty(event.display,event.requestor,event.property,
                          a.XA_STRING,8,
                          X11.PropModeReplace,data.string^,
                          str.Length(data.string^));
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END InitToX;*)

  PROCEDURE (display : Display) AddFont(font : Font);

  BEGIN
    font.last:=NIL;
    font.next:=display.fonts;
    IF display.fonts#NIL THEN
      display.fonts.last:=font;
    END;
    display.fonts:=font;
  END AddFont;

  (* ------------ Font ----------------- *)

  (**
    Initialize a font object.
  **)

  PROCEDURE (font : Font) Init*;

  VAR
    x : LONGINT;

  BEGIN
    font.last:=NIL;
    font.next:=NIL;

    font.loaded:=FALSE;
    font.features:={};
    font.useCount:=0;
    font.id:=0;

    FOR x:=0 TO LEN(font.fonts)-1 DO
      font.fonts[x]:=NIL;
    END;
  END Init;

  PROCEDURE (font : Font) InitFromFontInfo(name : ARRAY OF CHAR; info : X11.XFontStructPtr);

  VAR
    buffer,
    buffer2 : ARRAY 256 OF CHAR;
    res     : co.ConvResults;

    PROCEDURE GetField(VAR source : ARRAY OF CHAR; pos : INTEGER; VAR string : ARRAY OF CHAR);

    VAR
      x1,x2 : INTEGER;

    BEGIN
      x1:=0;
      WHILE (source[x1]#0X) & (pos>0) DO
        IF source[x1]="-" THEN
          DEC(pos);
        END;
        INC(x1);
      END;

      x2:=x1;
      WHILE (source[x2+1]#0X) & (source[x2+1]#"-") DO
        INC(x2);
      END;

      str.Extract(source,x1,x2-x1+1,string);
    END GetField;

  BEGIN
    GetField(name,1,buffer);
    GetField(name,2,buffer2);
    NEW(font.name,str.Length(buffer)+1+str.Length(buffer2)+1);
    COPY(buffer,font.name^);
    str.Append("-",font.name^);
    str.Append(buffer2,font.name^);
    INCL(font.features,fontName);

    GetField(name,5,buffer);
    IF buffer="normal" THEN
      font.setWidth:=fontNormal;
      INCL(font.features,fontSetWidth);
    ELSIF buffer="semicondensed" THEN
      font.setWidth:=fontCondensed;
      INCL(font.features,fontSetWidth);
    ELSIF buffer="narrow" THEN
      font.setWidth:=fontNarrow;
      INCL(font.features,fontSetWidth);
    ELSIF buffer="double" THEN
      font.setWidth:=fontDouble;
      INCL(font.features,fontSetWidth);
    END;

    GetField(name,7,buffer);
    co.StrToInt(buffer,font.height,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,fontHeight);

    GetField(name,8,buffer);
    co.StrToInt(buffer,font.pointHeight,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,fontPointHeight);

    GetField(name,9,buffer);
    co.StrToInt(buffer,font.horizRes,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,fontHorizRes);

    GetField(name,10,buffer);
    co.StrToInt(buffer,font.vertRes,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,fontVertRes);

    GetField(name,11,buffer);
    IF buffer="p" THEN
      font.spacing:=fontProportional;
      INCL(font.features,fontSpacing);
    ELSIF buffer="m" THEN
      font.spacing:=fontFixed;
      INCL(font.features,fontSpacing);
    END;

    GetField(name,12,buffer);
    co.StrToInt(buffer,font.avWidth,res);
    ASSERT(res=co.strAllRight);
    INCL(font.features,fontAvWidth);

    GetField(name,13,buffer);
    GetField(name,14,buffer2);
    NEW(font.charSet,str.Length(buffer)+1+str.Length(buffer2)+1);
    COPY(buffer,font.charSet^);
    str.Append("-",font.charSet^);
    str.Append(buffer2,font.charSet^);
    INCL(font.features,fontCharSet);

(*    IF info#NIL THEN
      font.ascent:=info.ascent;
      INCL(font.features,fontAscent);

      font.descent:=info.descent;
      INCL(font.features,fontDescent);
    END;*)

  END InitFromFontInfo;

  (**
    Calculates the position in the fontstyle array for the given style.
  **)

  PROCEDURE (font : Font) StyleToPos(style : SET):LONGINT;

  VAR
    x,y,val,res  : LONGINT;

  BEGIN
    res:=0;
    style:=style*styleMask;
    FOR x:=0 TO SIZE(SET)-1 DO
      IF x IN style THEN
        val:=1;
        FOR y:=1 TO x DO
          val:=val*2;
        END;
        INC(res,val);
      END;
    END;

    RETURN res;
  END StyleToPos;

  (**
    Returns the offset in the font array, that corresponds to the given style.
    uses Font.StyleToPos.

    NOTE
    GetFontPos tries to find a "close" font if a font with the given style bits
    does not exist. Currently it sees italic and slanted as equal, so if no italic
    font does exist, it tries a slanted one and vice versa.
  **)

  PROCEDURE (font : Font) GetFontPos(style : SET):LONGINT;

  VAR
    x : LONGINT;

  BEGIN
    x:=font.StyleToPos(style);
    IF font.fonts[x]#NIL THEN
      RETURN x;
    END;

    IF slanted IN style THEN
      EXCL(style,slanted);
      INCL(style,italic);
    ELSIF italic IN style THEN
      EXCL(style,italic);
      INCL(style,slanted);
    END;

    x:=font.StyleToPos(style);
    IF font.fonts[x]#NIL THEN
      RETURN x;
    END;

    EXCL(style,slanted);
    EXCL(style,italic);

    RETURN 0;
  END GetFontPos;

  (**
    Loads the given font. First tries to find an ressource entry
    with the given name and load the font stored there before falling
    back to the given default.

    This method is X11 specific.
  **)

  PROCEDURE (font : Font) LoadFont(display : Display; name: ARRAY OF CHAR; full : BOOLEAN):BOOLEAN;

  VAR
    help : ARRAY 256 OF CHAR;

    PROCEDURE ConvertField(VAR source : ARRAY OF CHAR; pos : INTEGER; string : ARRAY OF CHAR);

    VAR
      x1,x2 : INTEGER;

    BEGIN
      x1:=0;
      WHILE (source[x1]#0X) & (pos>0) DO
        IF source[x1]="-" THEN
          DEC(pos);
        END;
        INC(x1);
      END;

      x2:=x1;
      WHILE (source[x2+1]#0X) & (source[x2+1]#"-") DO
        INC(x2);
      END;

      str.Delete(source,x1,x2-x1+1);
      str.Insert(string,x1,source);
    END ConvertField;


  BEGIN
    NEW(font.fullName,str.Length(name)+1);
    COPY(name,font.fullName^);

    IF ~full THEN
      font.fonts[0]:=X11.XLoadQueryFont(display.display,name);
      IF font.fonts[0]=NIL THEN
        RETURN FALSE;
      END;
      font.height:=font.fonts[0].ascent+font.fonts[0].descent;
      font.ascent:=font.fonts[0].ascent;
      font.descent:=font.fonts[0].descent;
    ELSE
      font.loaded:=TRUE;
      font.useCount:=1;

      font.features:={fontHeight,fontAscent,fontDescent};

      IF str2.Match("-*-*-medium-r-*--*-*-*-*-*-*-*-*",name) THEN

        (* bold *)
        COPY(name,help);
        ConvertField(help,3,"bold");
        font.fonts[font.StyleToPos({bold})]:=X11.XLoadQueryFont(display.display,help);

        (* italic *)
        COPY(name,help);
        ConvertField(help,4,"i");
        font.fonts[font.StyleToPos({italic})]:=X11.XLoadQueryFont(display.display,help);

        (* slanted *)
        COPY(name,help);
        ConvertField(help,4,"o");
        font.fonts[font.StyleToPos({slanted})]:=X11.XLoadQueryFont(display.display,help);

        (* bold+italic *)
        COPY(name,help);
        ConvertField(help,3,"bold");
        ConvertField(help,4,"i");
        font.fonts[font.StyleToPos({bold,italic})]:=X11.XLoadQueryFont(display.display,help);

        (* bold+slanted *)
        COPY(name,help);
        ConvertField(help,3,"bold");
        ConvertField(help,4,"o");
        font.fonts[font.StyleToPos({bold,slanted})]:=X11.XLoadQueryFont(display.display,help);
      END;
    END;
    RETURN TRUE;
  END LoadFont;

  (**
    Loads the font. The OS specific font will be matched by evaluating the
    handed features. Every font loaded must be free using Free.

    RETURNS
    If the font can be loaded the method returns a new instance of font that must be used
    for future uses exspecially when calling Free.
  **)

  PROCEDURE (font : Font) Load(display : Display; VAR id : LONGINT):BOOLEAN;

  VAR
    name   : ARRAY 256 OF CHAR;
    buffer : ARRAY 20 OF CHAR;
    f      : Font;

  BEGIN
    name:="-";

    (* name *)
    IF fontName IN font.features THEN
      str.Append(font.name^,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-medium-r-",name);

    (* setWidth *)
    IF fontSetWidth IN font.features THEN
      CASE font.setWidth OF
        fontNormal:
          str.Append("normal",name);
      | fontCondensed:
          str.Append("semicondensed",name);
      | fontNarrow:
          str.Append("narrow",name);
      | fontDouble:
          str.Append("double",name);
      END;
    ELSE
      str.Append("*",name);
    END;

    str.Append("--",name);

    (* height *)
    IF fontHeight IN font.features THEN
      co.IntToStr(font.height,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* pointHeight *)
    IF fontPointHeight IN font.features THEN
      co.IntToStr(font.pointHeight,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* horizRes *)
    IF fontHorizRes IN font.features THEN
      co.IntToStr(font.horizRes,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* vertRes *)
    IF fontVertRes IN font.features THEN
      co.IntToStr(font.vertRes,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* spacing *)
    IF fontSpacing IN font.features THEN
      CASE font.spacing OF
        fontFixed:
          str.Append("m",name);
      | fontProportional:
          str.Append("p",name);
      END;
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);

    (* avWidth *)
    IF fontAvWidth IN font.features THEN
      co.IntToStr(font.avWidth,buffer);
      str.Append(buffer,name);
    ELSE
      str.Append("*",name);
    END;

    str.Append("-",name);
    IF fontCharSet IN font.features THEN
      str.Append(font.charSet^,name);
    ELSE
      str.Append("*",name);
    END;

    f:=display.fonts;
    WHILE (f#NIL) & ~str2.Match(name,f.fullName^) DO
      f:=f.next;
    END;

    IF f#NIL THEN
      INC(f.useCount);
      id:=f.id;
      RETURN TRUE;
    END;

    NEW(f);
    IF f.LoadFont(display,name,FALSE) THEN
      display.AddFont(f);
      f.id:=display.usrFontID;
      id:=f.id;
      INC(display.usrFontID);
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END Load;

  PROCEDURE (font : Font) Free(display : Display);

  VAR
    x : LONGINT;

  BEGIN
    DEC(font.useCount);

    IF font.useCount<=0 THEN
      FOR x:=0 TO LEN(font.fonts)-1 DO
        IF font.fonts[x]#NIL THEN
          X11.XFreeFont(display.display,font.fonts[x]);
        END;
      END;

      IF font=display.fonts THEN
        display.fonts:=font.next;
        IF display.fonts#NIL THEN
          display.fonts.last:=NIL;
        END;
      ELSE
        IF font.last#NIL THEN
          font.last.next:=font.next;
        END;
        IF font.next#NIL THEN
          font.next.last:=font.last;
        END;
      END;
    END;
  END Free;

  (**
    Returns the width of the given text in pixels in respect to the font.
  **)

  PROCEDURE (font : Font) TextWidth*(text : ARRAY OF CHAR; length : LONGINT; style : SET):LONGINT;

  BEGIN
    RETURN X11.XTextWidth(font.fonts[font.GetFontPos(style)],text,length);
  END TextWidth;

  (**
    Returns a FontExtent giving more information about the size of the string.
  **)

  PROCEDURE (font : Font) TextExtent*(text : ARRAY OF CHAR; length : LONGINT; style : SET; VAR extent : FontExtentDesc);

  VAR
    a,b,c : LONGINT;
    char  : X11.XCharStruct;

  BEGIN
    X11.XTextExtents(font.fonts[font.GetFontPos(style)],text,length,a,b,c,char);
    extent.lbearing:=char.lbearing;
    extent.rbearing:=char.rbearing;
    extent.width:=char.width;
    extent.ascent:=char.ascent;
    extent.descent:=char.descent;
    extent.height:=char.ascent+char.descent;
  END TextExtent;

  (**
    Returns a pointer to the OS-specific font handler.
  **)

  PROCEDURE (font : Font) GetFontHandle(style : SET):X11.XFontStructPtr;

  BEGIN
    RETURN font.fonts[font.GetFontPos(style)];
  END GetFontHandle;

  (**
    Return the font for the given font id.
  **)

  PROCEDURE (d : Display) GetFont*(id : LONGINT):Font;

  VAR
    font : Font;

  BEGIN
    IF id>=0 THEN
      font:=d.font[id];
    ELSE
      font:=d.fonts;
      WHILE (font#NIL) & (font.id#id) DO
        font:=font.next;
      END;
      ASSERT(font#NIL);
    END;
    IF ~font.loaded THEN
      IF font.LoadFont(d,font.fullName^,TRUE) THEN
      END;
    END;
    RETURN font;
  END GetFont;

  (**
    Loads the font. The OS specific font will be matched by evaluating the
    handed features. Every font loaded must be free using Free.

    RETURNS
    If the font can be loaded the method returns a new instance of font that must be used
    for future uses exspecially when calling Free.
  **)

  PROCEDURE (d : Display) LoadFont*(font : Font; VAR id : LONGINT):BOOLEAN;

  BEGIN
    RETURN font.Load(d,id);
  END LoadFont;

  (**
    Free a font loaded using LoadFont.
  **)

  PROCEDURE (d : Display) FreeFont*(id : LONGINT);

  VAR
    font : Font;

  BEGIN
    ASSERT(id<0);

    font:=d.GetFont(id);
    ASSERT(font#NIL);

    font.Free(d);
  END FreeFont;

  (* -------- color stuff of display ------------- *)

  (**
    Converts the nummerical VO color id to a X11 color id.

    NOTE
    Though this method s is exported you should not use it. It is only for some
    lowlevel classes like XpmImage, which need direct access to the X11 color values.
  **)

  PROCEDURE (d : Display) GetX11Color*(color : Color):C.longint;

  BEGIN
    IF color>=0 THEN
      RETURN d.color[color].color.pixel;
    ELSE
      RETURN d.userColor[-(color+1)].color.pixel;
    END;
  END GetX11Color;

  (**
    Allocate a color with the given rgb values. Since colors cannot be
    garbage collected you must store the color id and free the color later.

    RESULT
    A color id. You need to store this to free the color later.

    NOTE
    You cannot be sure that all bits of the color values will be used.
    X11 f.e. only uses 16 bit for color description.
  **)

  PROCEDURE (d : Display) AllocateColor*(r,g,b,default : LONGINT):Color;

  VAR
    x,pos : LONGINT;
    help  : ColorList;

  BEGIN
    FOR x:=0 TO LEN(d.color)-1 DO
      IF (d.color[x].color.red=r)
       & (d.color[x].color.green=g)
       & (d.color[x].color.blue=b) THEN
        RETURN x;
      END;
    END;
    pos:=-1;
    IF d.userColor#NIL THEN
      FOR x:=0 TO LEN(d.userColor^)-1 DO
        IF (d.userColor[x].color.red=r)
         & (d.userColor[x].color.green=g)
         & (d.userColor[x].color.blue=b) THEN
           INC(d.userColor[x].useCount);
          RETURN -(x+1);
        END;
        IF d.userColor[x].useCount=0 THEN
          IF pos<0 THEN
            pos:=x;
          END;
        END;
      END;
    END;

    IF pos<0 THEN
      IF d.userColor#NIL THEN
        help:=d.userColor;
        NEW(d.userColor,LEN(d.userColor^)+10);
        FOR x:=0 TO LEN(help^)-1 DO
          d.userColor[x]:=help[x];
        END;
        pos:=LEN(help^);
        FOR x:=LEN(help^) TO LEN(d.userColor^)-1 DO
          d.userColor[x].useCount:=0;
        END;
      ELSE
        NEW(d.userColor,10);
        FOR x:=0 TO LEN(d.userColor^)-1 DO
          d.userColor[x].useCount:=0;
        END;
        pos:=0;
      END;
    END;

    (*
      Scaling down color values from 31 bit to 16 bit,
      because X11 only supports 16 color values.
    *)
    d.userColor[pos].color.red:=SHORT(r DIV 32768);
    d.userColor[pos].color.green:=SHORT(g DIV 32768);
    d.userColor[pos].color.blue:=SHORT(b DIV 32768);

    IF X11.XAllocColor(d.display,d.colorMap,d.userColor[pos].color)=0 THEN
      IF default<=0 THEN
        INC(d.userColor[-(default+1)].useCount);
      END;
      RETURN default;
    ELSE
      d.userColor[pos].useCount:=1;
      RETURN -(pos+1);
    END;
  END AllocateColor;

  PROCEDURE (d : Display) AllocateNamedColor*(name : ARRAY OF CHAR; default : Color):Color;
  VAR
    exact,color : X11.XColor;

  BEGIN
    IF X11.XLookupColor(d.display,d.colorMap,name,exact,color)#0 THEN
      RETURN d.AllocateColor(color.red*32768,color.green*32768,color.blue*32768,default);
    ELSE
      RETURN default;
    END;
  END AllocateNamedColor;

  PROCEDURE (d : Display) IsAllocatedColor*(color : Color):BOOLEAN;

  BEGIN
    RETURN color>=0;
  END IsAllocatedColor;

  PROCEDURE (d : Display) FreeColor*(color : Color);

  VAR
    pixel : ARRAY 1 OF LONGINT;

  BEGIN
    IF color<0 THEN
      DEC(d.userColor[-color-1].useCount);
      IF d.userColor[-color-1].useCount<=0 THEN
        pixel[0]:=d.userColor[-color-1].color.pixel;
        X11.XFreeColors(d.display,d.colorMap,pixel,1,0);
      END;
    END;

  END FreeColor;

  (* -------- drawinfo stuff ------------- *)

  (**
    Do a display beep.
  **)

  PROCEDURE (d : DrawInfo) Beep*;

  BEGIN
    X11.XBell(d.display.display,100);
  END Beep;

  (* ----------- Clipping ---------------- *)

  (**
    Recalces the current clipping regions by analysing the
    current clipping stack.
  **)

  PROCEDURE (d : DrawInfo) RecalcClipRegion;

  VAR
    pos    : LONGINT;
    region : xu.Region;

  BEGIN
    region:=xu.XCreateRegion();
    pos:=0;
    IF d.clipPos>=0 THEN
      xu.XUnionRegion(d.clipStack[0].region,region,region);
      INC(pos);
      WHILE pos<=d.clipPos DO
        xu.XIntersectRegion(d.clipStack[pos].region,region,region);
        INC(pos);
      END;
    END;
    xu.XSetRegion(d.display.display,d.gc,region);
    xu.XDestroyRegion(region);
  END RecalcClipRegion;

  (**
    Initializes a clip entry object.
  **)

  PROCEDURE (VAR c : ClipEntryDesc) Init(draw : DrawInfo);

  BEGIN
    c.last:=NIL;
    c.region:=xu.XCreateRegion();
    c.draw:=draw;
  END Init;

  (**
    Recalcs the clipping rectangle.
  **)

  PROCEDURE (VAR c : ClipEntryDesc) Install;

  BEGIN
    c.draw.RecalcClipRegion;
  END Install;

  (**
    adds the given rectangle to the current clip entry.
  **)

  PROCEDURE (VAR c : ClipEntryDesc) Add(x,y,width,height : LONGINT);

  VAR
    rect : X11.XRectanglePtr;

  BEGIN
    s.NEW(rect,SIZE(X11.XRectangle));
    rect.x:=SHORT(x);
    rect.y:=SHORT(y);
    rect.width:=SHORT(width);
    rect.height:=SHORT(height);
    xu.XUnionRectWithRegion(rect,c.region,c.region);
    c.draw.RecalcClipRegion;
  END Add;

  (**
    Frees the given clip entry.
  **)

  PROCEDURE (VAR c : ClipEntryDesc) Free;

  BEGIN
    xu.XDestroyRegion(c.region);
  END Free;

  (**
    Start a new clipping. Calling this function does not start
    any cliping (in fact it copies the cliping rectangles of the
    current cliping) you must add cliping rectangles using
    DrawInfo.AddRegion.

    Clips are stacked and must be freed in order.
  **)

  PROCEDURE (d : DrawInfo) InstallClip*();

  VAR
    help : ClipStack;
    pos  : LONGINT;

  BEGIN
    IF d.clipPos>=LEN(d.clipStack^)-1 THEN
      NEW(help,LEN(d.clipStack^)+stackSizeIncrement);
      FOR pos:=0 TO LEN(d.clipStack^)-1 DO
        help[pos]:=d.clipStack[pos];
      END;
      d.clipStack:=help;
    END;
    INC(d.clipPos);

    d.clipStack[d.clipPos].Init(d);
    d.clipStack[d.clipPos].Install;
  END InstallClip;

  PROCEDURE (d : DrawInfo) ReinstallClip*;

  BEGIN
    d.clipStack[d.clipPos].Install;
  END ReinstallClip;

  (**
    Add a new cliping rectangle to the current clip.
  **)

  PROCEDURE (d : DrawInfo) AddRegion*(x,y,width,height : LONGINT);

  BEGIN
    d.clipStack[d.clipPos].Add(x,y,width,height);
  END AddRegion;

  (**
    Returns the outer box of the current clip region.
  **)

  PROCEDURE (d : DrawInfo) GetClipRegion*(VAR x,y,w,h : LONGINT);

  VAR
    rect : X11.XRectangle;

  BEGIN
    xu.XClipBox(d.clipStack[d.clipPos].region,rect);
    x:=rect.x;
    y:=rect.y;
    w:=rect.width;
    h:=rect.height;
  END GetClipRegion;

  (**
    Free the current clip and reset clipping the the last stacked clip.
  **)

  PROCEDURE (d : DrawInfo) FreeLastClip*;

  BEGIN
    d.clipStack[d.clipPos].Free;
    DEC(d.clipPos);
    IF d.clipPos>=0 THEN
      d.clipStack[d.clipPos].Install;
    END;
  END FreeLastClip;

  (**
    Push the given font on the stack and make it the current one.
  **)

  PROCEDURE (d : DrawInfo) PushFont*(id : LONGINT; style : SET);

  VAR
    help   : FontStack;
    pos    : LONGINT;
    handle : X11.XFontStructPtr;

  BEGIN
    IF d.fontPos>=LEN(d.fontStack^)-1 THEN
      NEW(help,LEN(d.fontStack^)+stackSizeIncrement);
      FOR pos:=0 TO LEN(d.fontStack^)-1 DO
        help[pos]:=d.fontStack[pos];
      END;
      d.fontStack:=help;
    END;
    INC(d.fontPos);

    d.fontStack[d.fontPos].font:=d.display.GetFont(id);
    d.fontStack[d.fontPos].style:=style;
    handle:=d.fontStack[d.fontPos].font.GetFontHandle(style);
    X11.XSetFont(d.display.display,d.gc,handle.fid);
  END PushFont;

  (**
    Pops the last stacked font and reactivates the last stacked.
  **)

  PROCEDURE (d : DrawInfo) PopFont*;

  VAR
    handle : X11.XFontStructPtr;

  BEGIN
    DEC(d.fontPos);
    IF d.fontPos>=0 THEN
      handle:=d.fontStack[d.fontPos].font.GetFontHandle(d.fontStack[d.fontPos].style);
      X11.XSetFont(d.display.display,d.gc,handle.fid);
    END;
  END PopFont;

  (**
    Draws the given string in the current font at the given position.
  **)

  PROCEDURE (d : DrawInfo) DrawString*(x,y : LONGINT; text : ARRAY OF CHAR; length : LONGINT);

  BEGIN
    X11.XDrawString(d.display.display,d.window,d.gc,x,y,text,length);
  END DrawString;

  (**
    Draws the given string in the current font at the given position.
    The string will be cliped to the given width and height.
  **)

(*
  PROCEDURE (d : DrawInfo) DrawClipString*(x,y,w,h : LONGINT; text : ARRAY OF CHAR; length : LONGINT);

  VAR
    rect,
    new    : X11.XRectanglePtr;
    region : xu.Region;


  BEGIN
    s.NEW(rect,SIZE(X11.XRectangle));
    s.NEW(new,SIZE(X11.XRectangle));
    new.x:=SHORT(x);
    new.y:=SHORT(y);
    new.width:=SHORT(w);
    new.height:=SHORT(h);

    xu.XClipBox(d.clipStack.region,rect^);

    region:=xu.XCreateRegion();
    xu.XUnionRectWithRegion(rect,region,region);
    xu.XUnionRectWithRegion(new,region,region);
    xu.XSetRegion(d.display.display,d.gc,region);
    xu.XDestroyRegion(region);

    X11.XDrawString(d.display.display,d.window,d.gc,x,y,text,length);

    region:=xu.XCreateRegion();
    xu.XUnionRectWithRegion(rect,region,region);
    xu.XSetRegion(d.display.display,d.gc,region);
    xu.XDestroyRegion(region);
  END DrawClipString;
*)
  (**
    Draws the given string in the current font at the given position.
    The background will be filled with the current background color.
  **)

  PROCEDURE (d : DrawInfo) DrawFillString*(x,y : LONGINT; text : ARRAY OF CHAR; length : LONGINT);

  BEGIN
    X11.XDrawImageString(d.display.display,d.window,d.gc,x,y,text,length);
  END DrawFillString;

  (**
    Draws the given string in the current font at the given position.
    The background will be filled with the current background color.
    The string will be cliped to the given width and height.
  **)
(*
  PROCEDURE (d : DrawInfo) DrawClipFillString*(x,y : LONGINT; text : ARRAY OF CHAR; length : LONGINT);

  BEGIN
    X11.XDrawImageString(d.display.display,d.window,d.gc,x,y,text,length);
  END DrawClipFillString;
*)
  (**
    Push the given color on the stack and make it the current foreground
    color.
  **)

  PROCEDURE (d : DrawInfo) PushForeground*(color : Color);

  VAR
    help : PenStack;
    pos  : LONGINT;

  BEGIN
    IF d.fPenPos>=LEN(d.fPenStack^)-1 THEN
      NEW(help,LEN(d.fPenStack^)+stackSizeIncrement);
      FOR pos:=0 TO LEN(d.fPenStack^)-1 DO
        help[pos]:=d.fPenStack[pos];
      END;
      d.fPenStack:=help;
    END;
    INC(d.fPenPos);

    d.fPenStack[d.fPenPos].color:=color;

    X11.XSetForeground(d.display.display,d.gc,d.display.GetX11Color(color));
  END PushForeground;

  (**
    Pop the last pushed foreground color from the stack and thus reinstalls
    the previous color.
  **)

  PROCEDURE (d : DrawInfo) PopForeground*;

  BEGIN
    DEC(d.fPenPos);
    IF d.fPenPos>=0 THEN
      X11.XSetForeground(d.display.display,d.gc,
                         d.display.GetX11Color(d.fPenStack[d.fPenPos].color));
    END;
  END PopForeground;

  (**
    Push the given draw mode on the stack and make it the current draw mode.
  **)

  PROCEDURE (d : DrawInfo) PushDrawMode*(mode : LONGINT);

  VAR
    m     : DrawMode;
    xMode : INTEGER;

  BEGIN
    NEW(m);
    m.mode:=mode;
    CASE mode OF
      copy:
        xMode:=X11.GXcopy;
    | invert:
        xMode:=X11.GXinvert;
    END;
    X11.XSetFunction(d.display.display,d.gc,xMode);
    m.next:=d.modeStack;
    d.modeStack:=m;
  END PushDrawMode;

  (**
    Pop the last pushed draw mode from the stack and thus reinstalls
    the previous draw mode.
  **)

  PROCEDURE (d : DrawInfo) PopDrawMode*;

  VAR
   xMode : INTEGER;

  BEGIN
    d.modeStack:=d.modeStack.next;
    IF d.modeStack#NIL THEN
      CASE d.modeStack.mode OF
        copy:
          xMode:=X11.GXcopy;
      | invert:
          xMode:=X11.GXinvert;
      END;
      X11.XSetFunction(d.display.display,d.gc,xMode);
    ELSE
      X11.XSetFunction(d.display.display,d.gc,X11.GXcopy);
    END;
  END PopDrawMode;

  (**
    Push the given color on the stack and make it the current background
    color.
  **)

  PROCEDURE (d : DrawInfo) PushBackground*(color : Color);

  VAR
    help : PenStack;
    pos  : LONGINT;

  BEGIN
    IF d.bPenPos>=LEN(d.bPenStack^)-1 THEN
      NEW(help,LEN(d.bPenStack^)+stackSizeIncrement);
      FOR pos:=0 TO LEN(d.bPenStack^)-1 DO
        help[pos]:=d.bPenStack[pos];
      END;
      d.bPenStack:=help;
    END;
    INC(d.bPenPos);

    d.bPenStack[d.bPenPos].color:=color;

    X11.XSetBackground(d.display.display,d.gc,d.display.GetX11Color(color));
  END PushBackground;

  (**
    Pop the last pushed background color from the stack and thus reinstalls
    the previous color.
  **)

  PROCEDURE (d : DrawInfo) PopBackground*;

  BEGIN
    DEC(d.bPenPos);
    IF d.bPenPos>=0 THEN
      X11.XSetBackground(d.display.display,d.gc,
                         d.display.GetX11Color(d.bPenStack[d.bPenPos].color));
    END;
  END PopBackground;

  (**
    Push the drawinfo mode and a pensize on the stack.
  **)

  PROCEDURE (d : DrawInfo) PushStyle*(size, mode : LONGINT);

  VAR
    pen  : PenStyle;
    lMode : LONGINT;

  BEGIN
    NEW(pen);
    pen.size:=size;
    pen.mode:=mode;
    IF mode=roundPen THEN
      pen.cap:=X11.CapRound;
      pen.join:=X11.JoinRound;
    ELSE
      pen.cap:=X11.CapButt;
      pen.join:=X11.JoinBevel;
    END;

    IF d.dashStack#NIL THEN
      lMode:=d.dashStack.mode;
    ELSE
      lMode:=X11.LineSolid;
    END;

    X11.XSetLineAttributes(d.display.display,d.gc,pen.size,lMode,pen.cap,pen.join);
    pen.next:=d.styleStack;
    d.styleStack:=pen;

  END PushStyle;

  (**
    Pop the last pushed style from the stack and thus reinstalls
    the previous style.
  **)

  PROCEDURE (d : DrawInfo) PopStyle*;

  VAR
    mode : LONGINT;

  BEGIN
    d.styleStack:=d.styleStack.next;

    IF d.dashStack#NIL THEN
      mode:=d.dashStack.mode;
    ELSE
      mode:=X11.LineSolid;
    END;

    IF d.styleStack#NIL THEN
      X11.XSetLineAttributes(d.display.display,d.gc,d.styleStack.size,mode,d.styleStack.cap,d.styleStack.join);
    ELSE
      X11.XSetLineAttributes(d.display.display,d.gc,0,mode,X11.CapButt,X11.JoinBevel);
    END;
  END PopStyle;

  (**
    Pushes the a new style for drawing lines on the stack.
  **)

  PROCEDURE (d : DrawInfo) PushDash*(dashList : ARRAY OF CHAR; mode : LONGINT);

  VAR
    dash   : PenDash;
    size,
    cap,
    join,x : LONGINT;

  BEGIN
    NEW(dash);
    dash.next:=NIL;
    NEW(dash.list,LEN(dashList));
    FOR x:=0 TO LEN(dashList)-1 DO
      dash.list[x]:=dashList[x];
    END;

    IF mode=fMode THEN
      dash.mode:=X11.LineOnOffDash;
    ELSE
      dash.mode:=X11.LineDoubleDash;
    END;

    IF d.styleStack#NIL THEN
      size:=d.styleStack.size;
      cap:=d.styleStack.cap;
      join:=d.styleStack.join;
    ELSE
      size:=0;
      cap:=X11.CapButt;
      join:=X11.JoinBevel;
    END;

    X11.XSetLineAttributes(d.display.display,d.gc,size,dash.mode,cap,join);
    X11.XSetDashes(d.display.display,d.gc,0,dash.list^,LEN(dashList));

    dash.next:=d.dashStack;
    d.dashStack:=dash;
  END PushDash;

  (**
    Pop the last pushed dash style from the stack and thus reinstalls
    the previous dash style.
  **)

  PROCEDURE (d : DrawInfo) PopDash*;

  VAR
    size,
    cap,
    join : LONGINT;

  BEGIN
    d.dashStack:=d.dashStack.next;

    IF d.styleStack#NIL THEN
      size:=d.styleStack.size;
      cap:=d.styleStack.cap;
      join:=d.styleStack.join;
    ELSE
      size:=0;
      cap:=X11.CapButt;
      join:=X11.JoinBevel;
    END;


    IF d.dashStack#NIL THEN
      X11.XSetLineAttributes(d.display.display,d.gc,size,d.dashStack.mode,cap,join);
    ELSE
      X11.XSetLineAttributes(d.display.display,d.gc,0,X11.LineSolid,X11.CapButt,X11.JoinBevel);
    END;
  END PopDash;

  (**
    Push a new pattern for filling onto the stack.
  **)

  PROCEDURE (d : DrawInfo) PushPattern*(pattern : ARRAY OF CHAR; width, height : LONGINT; mode : LONGINT);

  VAR
    pat : Pattern;

  BEGIN
    NEW(pat);
    pat.pixMap:=X11.XCreateBitmapFromData(d.display.display,
                    X11.XRootWindow(d.display.display,d.display.scrNum),
                    pattern,width,height);
    IF pat.pixMap=0 THEN
      Err.String("Cannot create pimap!");Err.Ln;
    END;
    pat.mode:=mode;

    X11.XSetStipple(d.display.display,d.gc,pat.pixMap);
    CASE mode OF
      fgPattern:
        X11.XSetFillStyle(d.display.display,d.gc,X11.FillStippled);
    | fbPattern:
        X11.XSetFillStyle(d.display.display,d.gc,X11.FillOpaqueStippled);
    ELSE
      Err.String("Unsuported patternMode!"); Err.Ln;
    END;
    pat.next:=d.patternStack;
    d.patternStack:=pat;
  END PushPattern;

  (**
    Pop the last pushed pattern from the stack and thus reinstalls
    the previous pattern.
  **)

  PROCEDURE (d : DrawInfo) PopPattern*;

  BEGIN
    X11.XFreePixmap(d.display.display,d.patternStack.pixMap);
    d.patternStack:=d.patternStack.next;
    IF d.patternStack#NIL THEN

      CASE d.patternStack.mode OF
        fgPattern:
          X11.XSetFillStyle(d.display.display,d.gc,X11.FillStippled);
      | fbPattern:
          X11.XSetFillStyle(d.display.display,d.gc,X11.FillOpaqueStippled);
      ELSE
       Err.String("Unsuported patternMode!"); Err.Ln;
      END;
      X11.XSetStipple(d.display.display,d.gc,d.patternStack.pixMap);

    ELSE
      X11.XSetFillStyle(d.display.display,d.gc,X11.FillSolid);
    END;
  END PopPattern;

  (**
    This methods tries to create a unique fill pattern on the stack. VisualOberon tries
    to select the optimal fill pattern for the given color mode.VisualOberon cannot offer
    an unlimited number of different fill patterns. After a not specified amount of patterns
    (at three) VisualOberon will reuse the patterns. E.g. when VisualOberon offers three
    patterns the fourth one will be equal to the first one. VisualOberon will also garantee
    that the the last pattern will not be equal to the first pattern. Garanteeing this, you
    need not take special care when filling f.e. a pie chart.

    PARAMETER
      - pos is the running number of the pattern.
      - maximum is the maximum number of patterns you want. Only when this value is
        correctly set VisualOberon will garantee the above fact. If you don't want
        VisualOberon to take special care just hand -1.
  **)

  PROCEDURE (d : DrawInfo) PushUniqueFillPattern*(pos, maximum : LONGINT);

  VAR
   pattern : LONGINT;

  BEGIN
    pattern:=pos MOD 3;
    IF (pos=maximum) & (pattern=1) THEN
      pattern:=2;
    END;

    CASE d.display.colorMode OF
      colorMode:
        CASE pattern OF
          0: d.PushForeground(warnColor);
        | 1: d.PushForeground(fillColor);
        | 2: d.PushForeground(halfShadowColor);
        END;
    | greyScaleMode:
        CASE pattern OF
          0: d.PushForeground(shineColor);
        | 1: d.PushForeground(halfShineColor);
        | 2: d.PushForeground(halfShadowColor);
        END;
    | monochromeMode: (* only use black and white and some shadings *)
        CASE pattern OF
        ELSE
          ASSERT(FALSE);
(*          0: d.PushForeground(whiteColor);
        | 1: d.PushForeground(whiteColor);
             d.PushBackground(whiteColor);
             d.PushPattern(disablePattern,disableWidth,disableHeight,fbPattern);
        | 2: d.PushForeground(blackColor);*)
        END;
    END;
  END PushUniqueFillPattern;

  (**
    Pop the pushed pattern from the stack.
  **)

  PROCEDURE (d : DrawInfo) PopUniqueFillPattern*(pos, maximum : LONGINT);

  VAR
   pattern : LONGINT;

  BEGIN
    pattern:=pos MOD 3;
    IF (pos=maximum) & (pattern=1) THEN
      pattern:=2;
    END;

    IF (d.display.colorMode=monochromeMode) & (pattern=1) THEN
      d.PopForeground;
      d.PopBackground;
      d.PopPattern;
    ELSE
      d.PopForeground;
    END;
  END PopUniqueFillPattern;

  (* Drawing functions *)

  (**
    Draws a point at the given position.
  **)

  PROCEDURE (d : DrawInfo) DrawPoint*(x,y : LONGINT);

  BEGIN
    X11.XDrawPoint(d.display.display,d.window,d.gc,x,y);
  END DrawPoint;

  (**
    Draws a line from x1,y1 to x2,y2.
  **)

  PROCEDURE (d : DrawInfo) DrawLine*(x1,y1,x2,y2 : LONGINT);

  BEGIN
    X11.XDrawLine(d.display.display,d.window,d.gc,x1,y1,x2,y2);
  END DrawLine;

  (**
    Draws afilled rectangle with the uper left corner at x,y and with the
    given width and height.
  **)

  PROCEDURE (d : DrawInfo) FillRectangle*(x,y,width,height : LONGINT);

  BEGIN
    X11.XFillRectangle(d.display.display,d.window,d.gc,x,y,width,height);
  END FillRectangle;

  PROCEDURE (d : DrawInfo) DrawArc*(x,y,width,height,angle1,angle2 : LONGINT);

  BEGIN
    X11.XDrawArc(d.display.display,d.window,d.gc,x,y,width-1,height-1,angle1,angle2);
  END DrawArc;

  PROCEDURE (d : DrawInfo) FillArc*(x,y,width,height,angle1,angle2 : LONGINT);

  BEGIN
    X11.XFillArc(d.display.display,d.window,d.gc,x,y,width,height,angle1,angle2);
  END FillArc;

  PROCEDURE (d : DrawInfo) FillPolygon*(points : ARRAY OF PointDesc; count : LONGINT);

  BEGIN
    X11.XFillPolygon(d.display.display,d.window,d.gc,points,count,X11.Complex,X11.CoordModeOrigin);
  END FillPolygon;

  (**
    Fill the given rectangle with the background color defined in Display.
    This is a high level function. You should use it whenever you want to clear
    a area and give it the background color.
  **)

  PROCEDURE (d : DrawInfo) FillBackground*(x,y,width,height : LONGINT);

(*  VAR
    Pat : ARRAY 2 OF CHAR;*)

  BEGIN
(*    Pat[0]:=CHR(170);  (* 10101010 *)
    Pat[1]:=CHR(85);   (* 01010101 *)*)
    d.PushForeground(backgroundColor);
(*    d.PushBackground(halfShineColor);
    d.PushPattern(Pat,8,2,fbPattern);*)
    X11.XFillRectangle(d.display.display,d.window,d.gc,x,y,width,height);
(*    d.PopPattern;*)
    d.PopForeground;
(*    d.PopBackground;*)
  END FillBackground;

  (**
    Copies the given area to a new place.
  **)

  PROCEDURE (d : DrawInfo) CopyArea*(sX,sY,width,height,dX,dY : LONGINT);

  BEGIN
    X11.XSetGraphicsExposures(d.display.display,d.gc,X11.True);
    X11.XCopyArea(d.display.display,d.window,d.window,d.gc,
                  sX,sY,width,height,dX,dY);
    X11.XSetGraphicsExposures(d.display.display,d.gc,X11.False);
  END CopyArea;

  PROCEDURE (d : DrawInfo) CopyFromBitmap*(bitmap : Bitmap; sX,sY,width,height,dX,dY : LONGINT);

  BEGIN
    X11.XCopyArea(d.display.display,bitmap.pixmap,d.window,d.gc,
                  sX,sY,width,height,dX,dY);
  END CopyFromBitmap;

  PROCEDURE (d : DrawInfo) CopyToBitmap*(sX,sY,width,height,dX,dY : LONGINT; bitmap : Bitmap);

  BEGIN
    X11.XSetGraphicsExposures(d.display.display,d.gc,X11.True);
    X11.XCopyArea(d.display.display,bitmap.pixmap,d.window,d.gc,
                  sX,sY,width,height,dX,dY);
    X11.XSetGraphicsExposures(d.display.display,d.gc,X11.False);
  END CopyToBitmap;

  (**
    Initialize an instance of the  DrawInfo class.
  **)

  PROCEDURE (d : DrawInfo) Init(display : Display; window : X11.Drawable; vWindow : Window);

  VAR
    mask   : X11.ulongmask;
    values : X11.XGCValues;

  BEGIN
    d.display:=display;
    d.window:=window;
    d.vWindow:=vWindow;
    mask:={};
    d.gc:=X11.XCreateGC(display.display,window,mask,values);
    X11.XSetArcMode(display.display,d.gc,X11.ArcPieSlice);
    d.mode:={};

    (* The array stacks *)
    d.fPenPos:=-1;
    NEW(d.fPenStack,initialStackSize);
    d.bPenPos:=-1;
    NEW(d.bPenStack,initialStackSize);
    d.fontPos:=-1;
    NEW(d.fontStack,initialStackSize);
    d.clipPos:=-1;
    NEW(d.clipStack,initialStackSize);

    (* The list stacks *)
    d.styleStack:=NIL;
    d.dashStack:=NIL;
    d.patternStack:=NIL;
    d.modeStack:=NIL;

    (* convinience stuff *)
    d.PushFont(normalFont,{});
  END Init;

  (**
    Deinitializes the drawInfo
  **)

  PROCEDURE (d : DrawInfo) Deinit;

  BEGIN
    d.PopFont;

    ASSERT(d.fPenPos=-1);
    ASSERT(d.bPenPos=-1);
    ASSERT(d.fontPos=-1);
    ASSERT(d.styleStack=NIL);
    ASSERT(d.dashStack=NIL);
    ASSERT(d.clipPos=-1);
    ASSERT(d.patternStack=NIL);
    ASSERT(d.modeStack=NIL);
  END Deinit;

  (* ------------ Display ----------------- *)

  (**
    Add a timeout. The object will be send a TimeOutMsg directly after
    the given (relative) time has been elapsed.

    NOTE
    It is garanteed that the time has been elasped when the object
    get notified. However, since there is no real multitasking involved,
    it may get called later.

    If the timeout has been elasped and the message has been send, there is
    no need to remove the TimeOut. Dipslay does this for you.
  **)

  PROCEDURE (d : Display) AddTimeOut*(sec, msec : LONGINT; object : O.MsgObject):TimeOut;

  VAR
    timeOut,
    help     : TimeOut;
    dt       : sc.DateTime;
    interval : t.Interval;
    milli    : LONGINT;

  BEGIN
    NEW(timeOut);
    timeOut.object:=object;
    sc.GetClock(dt);
    c.SetTimeStamp(dt,timeOut.time);

    milli:=sec*1000+msec;

    t.InitInterval(interval,0,milli);

    timeOut.time.Add(interval);

    IF (d.timeOutList=NIL) OR (timeOut.time.Cmp(d.timeOutList.time)<=0) THEN
      timeOut.next:=d.timeOutList;
      d.timeOutList:=timeOut;
    ELSE
      help:=d.timeOutList;
      WHILE (help.next#NIL) & (help.next.time.Cmp(timeOut.time)<0) DO
        help:=help.next;
      END;
      timeOut.next:=help.next;
      help.next:=timeOut;
    END;

    RETURN timeOut;
  END AddTimeOut;

  (**
    Remove a TimeOut before it has been elasped. It is save to call this function with
    a TimeOut-instance of an already elapsed time event.
  **)

  PROCEDURE (d : Display) RemoveTimeOut*(timeOut : TimeOut);

  VAR
    help,
    last : TimeOut;

  BEGIN
    IF d.timeOutList=NIL THEN
      RETURN;
    END;

    IF d.timeOutList=timeOut THEN
      d.timeOutList:=d.timeOutList.next;
      RETURN;
    END;

    help:=d.timeOutList.next;
    last:=d.timeOutList;
    WHILE (help#NIL) & (help#timeOut) DO
      last:=help;
      help:=help.next;
    END;
    IF help#NIL THEN
      last.next:=help.next;
    END;
  END RemoveTimeOut;

  (**
    Check the list of registered timeouts and send a message for every elapsed
    timeout.
  **)

  PROCEDURE (d : Display) CheckTimeOuts;

  VAR
    timeOut,
    help     : TimeOut;
    msg      : TimeOutMsg;
    dt       : sc.DateTime;
    time     : t.TimeStamp;

  BEGIN
    sc.GetClock(dt);
    c.SetTimeStamp(dt,time);

    timeOut:=d.timeOutList;
    WHILE timeOut#NIL DO
      IF (timeOut.time.Cmp(time)<0) THEN
        NEW(msg);
        msg.timeOut:=timeOut;
        help:=timeOut.next;
        d.RemoveTimeOut(timeOut);
        timeOut.object.Receive(msg);
        timeOut:=help;
      ELSE
        timeOut:=timeOut.next;
      END;
    END;
  END CheckTimeOuts;

  (**
    Return the next pending future time event. If there is no event pending it
    will return a event 10 seconds in the future.
  **)

  PROCEDURE (d : Display) GetNextTimeOut(VAR interval : t.Interval);

  VAR
    dt   : sc.DateTime;
    time : t.TimeStamp;

  BEGIN
    IF d.timeOutList#NIL THEN

      sc.GetClock(dt);
      c.SetTimeStamp(dt,time);
      d.timeOutList.time.Delta(time,interval);
    ELSE
      t.InitInterval(interval,0,30*1000);
    END;
  END GetNextTimeOut;

  (**
    Using this method, you can add an object to be called whenever the application is
    not busy. This way you can write applications that take as much processor time as
    possible while still listening to windowing events.

    Note, that you must design the action to be taken when the object is called to be
    rather short, since all event handling will be blocked during this time. If you
    want do handle long time actions you must split them into short partial actions.
  **)

  PROCEDURE (d : Display) AddSleep*(object : O.MsgObject):Sleep;

  VAR
    sleep : Sleep;

  BEGIN
    NEW(sleep);
    sleep.object:=object;

    sleep.next:=d.sleepList;
    d.sleepList:=sleep;

    RETURN sleep;
  END AddSleep;

  (**
    Removes the given sleep notifier.
  **)

  PROCEDURE (d : Display) RemoveSleep*(sleep : Sleep);

  VAR
    help,
    last : Sleep;

  BEGIN
    IF d.sleepList=NIL THEN
      RETURN;
    END;

    IF d.sleepList=sleep THEN
      d.sleepList:=d.sleepList.next;
      RETURN;
    END;

    help:=d.sleepList.next;
    last:=d.sleepList;
    WHILE (help#NIL) & (help#sleep) DO
      last:=help;
      help:=help.next;
    END;
    IF help#NIL THEN
      last.next:=help.next;
    END;
  END RemoveSleep;

  (**
    Check the list of registered sleeps and send a message for every elapsed
    timeout.
  **)

  PROCEDURE (d : Display) CheckSleeps;

  VAR
    sleep : Sleep;
    msg   : SleepMsg;

  BEGIN
    sleep:=d.sleepList;
    WHILE sleep#NIL DO
      NEW(msg);
      msg.sleep:=sleep;
      sleep.object.Receive(msg);
      sleep:=sleep.next;
    END;
  END CheckSleeps;

  (**
    Using this method, you can add an object to be called whenever the handed
    channel gets available.

    Note, that you must design the action to be taken when the object is
    called to be rather short, since the complete event handling will be
    blocked during this time. If you want do handle long time actions you
    must split them into short partial actions.
  **)

  PROCEDURE (d : Display) AddChannel*(channel : ch.Channel; object : O.MsgObject):Channel;

  VAR
    entry : Channel;

  BEGIN
    NEW(entry);
    entry.channel:=channel;
    entry.object:=object;

    entry.next:=d.channelList;
    d.channelList:=entry;

    RETURN entry;
  END AddChannel;

  (**
    Removes the given sleep notifier.
  **)

  PROCEDURE (d : Display) RemoveChannel*(channel : Channel);

  VAR
    help,
    last : Channel;

  BEGIN
    IF d.channelList=NIL THEN
      RETURN;
    END;

    IF d.channelList=channel THEN
      d.channelList:=d.channelList.next;
      RETURN;
    END;

    help:=d.channelList.next;
    last:=d.channelList;
    WHILE (help#NIL) & (help#channel) DO
      last:=help;
      help:=help.next;
    END;
    IF help#NIL THEN
      last.next:=help.next;
    END;
  END RemoveChannel;

  (**
    Check the list of registered file notifier and send a message for every
    notifier matching the file descriptor.
  **)

  PROCEDURE (entry : Channel) SendNotify;

  VAR
    msg : ChannelMsg;

  BEGIN
    NEW(msg);
    msg.channel:=entry;
    entry.object.Receive(msg);
    entry:=entry.next;
  END SendNotify;

  (**
    Call this method if you want the Display to stop generating
    QuickHelp calls to windows. This is necessesarry, if you are
    opening a QuickHelp and don't want to have a second after the
    second timeout.
  **)

  PROCEDURE (d : Display) StopContextHelp*;

  BEGIN
    IF d.contextHelp THEN
(*      Err.String("Stop contexthelp"); Err.Ln;*)
      d.contextHelp:=FALSE;
      IF d.timeOut#NIL THEN
        d.RemoveTimeOut(d.timeOut);
        d.timeOut:=NIL;
      END;
    END;
  END StopContextHelp;

  (**
    Restart the generation of QuickHelp calls to windows stoped
    with Display.StopContextHelp.
  **)

  PROCEDURE (d : Display) RestartContextHelp*;

  BEGIN
    IF ~d.contextHelp THEN
(*      Err.String("Restart contexthelp"); Err.Ln;*)
      d.timeOut:=d.AddTimeOut(contextTimeOutSec,contextTimeOutMil,d);
      d.contextHelp:=TRUE;
    END;
  END RestartContextHelp;

  (**
    Restart the generation of QuickHelp calls to windows stoped
    with Display.StopContextHelp.
  **)

  PROCEDURE (d : Display) StartContextHelp*;

  BEGIN
    IF ~d.contextHelp THEN
(*      Err.String("Start contexthelp"); Err.Ln;*)
      d.timeOut:=d.AddTimeOut(contextTimeOutSec,contextTimeOutMil,d);
      d.contextHelp:=TRUE;
    END;
  END StartContextHelp;

  (**
    Adds window to the internal list of windows.
  **)

  PROCEDURE (d : Display) AddWindow*(w : Window);

  BEGIN
    IF d.winList#NIL THEN
      w.last:=NIL;
      w.next:=d.winList;
      d.winList.last:=w;
    ELSE
      w.last:=NIL;
      w.next:=NIL;
    END;
    d.winList:=w;

    INC(d.winCount);
  END AddWindow;

  (**
    Removes window from the internal list of windows.
  **)

  PROCEDURE (d : Display) RemoveWindow*(w : Window);

  BEGIN
    IF d.winList=w THEN
      d.winList:=d.winList.next;
    END;

    IF w.last#NIL THEN
      w.last.next:=w.next;
    END;

    IF w.next#NIL THEN
      w.next.last:=w.last;
    END;

    DEC(d.winCount);
  END RemoveWindow;

  (**
    This will be set before a window calls its Preinit method. If you overload
    the Preinit method to add some custom stuff (common way to create custom
    dialogs).

    NOTE
    I'm not sure why I wrote this one, however XpmImage uses it!?
  **)

  PROCEDURE (d : Display) SetNewWindow(window : Window);

  BEGIN
    d.newWin:=window;
  END SetNewWindow;

  (**
    Return the window set using SetNewWindow.
  **)

  PROCEDURE (d : Display) GetNewWindow*():Window;

  BEGIN
    RETURN d.newWin;
  END GetNewWindow;

  PROCEDURE (d : Display) CreateBitmap*(width, height : LONGINT):Bitmap;

  VAR
    bitmap : Bitmap;

  BEGIN
    NEW(bitmap);
    bitmap.pixmap:=X11.XCreatePixmap(d.display,
                                     X11.XDefaultRootWindow(d.display),
                                     width,
                                     height,d.colorDepth);

    ASSERT(bitmap.pixmap#0);

    NEW(bitmap.draw);
    bitmap.draw.Init(d,bitmap.pixmap,NIL);

    bitmap.width:=width;
    bitmap.height:=height;

    RETURN bitmap;
  END CreateBitmap;

  PROCEDURE (d : Display) FreeBitmap*(bitmap : Bitmap);

  BEGIN
    X11.XFreePixmap(d.display,bitmap.pixmap);
  END FreeBitmap;

  (* ------------ Window ----------------- *)

  (**
    Initialize an instance of the window class.

    NOTE
    You must call this before using a window.
  **)

  PROCEDURE (w : Window) Init*;

  BEGIN
    w.modal:=FALSE;
    w.modalCount:=0;
    w.parentWin:=0;
    w.parent:=NIL;

    w.last:=NIL;
    w.next:=NIL;
    NEW(w.title,1);
    COPY("",w.title^);

    w.window:=0;

    w.background:=backgroundColor;

    w.freeList:=NIL;

    w.maped:=FALSE;
    w.exposed:=FALSE;
    w.focus:=FALSE;
    w.inited:=FALSE;
    w.grab:=FALSE;
    w.createAllways:=FALSE;
    w.borderless:=FALSE;
    w.open:=FALSE;

    w.x:=0;
    w.y:=0;

    w.width:=0;
    w.height:=0;

    w.minWidth:=0;
    w.minHeight:=0;

    w.maxWidth:=MAX(LONGINT);
    w.maxHeight:=MAX(LONGINT);

    w.horizontalPos:=manualPos;
    w.verticalPos:=manualPos;
  END Init;

  (**
    Sets the parent window of the current window. VisualOberon will try to
    make use of the supplied information about the hierachical structure of
    the window tree.

    NOTE
    Parent will only be evaluated before first opening.

  **)

  PROCEDURE (w : Window) SetParent*(parent : Window);

  BEGIN
    IF ~w.inited THEN
      w.parent:=parent;
    END;
  END SetParent;

  (**
    Sets the title of the window.

    NOTE
    Value will only be used before first opening.
  **)

  PROCEDURE (w : Window) SetTitle*(name : ARRAY OF CHAR);

  BEGIN
    NEW(w.title,str.Length(name)+1);
    COPY(name,w.title^);
    IF w.inited THEN
      X11.XStoreName(w.display.display,w.window,name);
    END;
  END SetTitle;

  (**
    Sets the display for the window.

    NOTE
    Only evaluated before first opening.
  **)

  PROCEDURE(w : Window) SetDisplay*(display : Display);

  BEGIN
    IF ~w.inited THEN
      w.display:=display;
    END;
  END SetDisplay;

  (**
    Make the window modal.
  **)

  PROCEDURE (w : Window) SetModal*;

  BEGIN
    w.modal:=TRUE;
  END SetModal;

  (**
    Sets the background color.
  **)

  PROCEDURE (w : Window) SetBackground*(color : Color);

  BEGIN
    w.background:=color;
  END SetBackground;

  (**
    Make the window borderless.

    NOTE
    Only evaluated before first opening.
  **)

  PROCEDURE (w : Window) Borderless*(borderless : BOOLEAN);

  BEGIN
    w.borderless:=borderless;
  END Borderless;

  (**
    Tells the window to allways create a new window of the underlying
    windowing-system on a call to Window.Open. This is most usefull for
    unmodal windows or popups where it is difficult to track open and close
    calls.
  **)

  PROCEDURE (w : Window) CreateAllways*(create : BOOLEAN);

  BEGIN
    IF ~w.maped THEN
      w.createAllways:=create;
    END;
  END CreateAllways;

  (**
    Sets the width and height of the window.

    NOTE
    Does only work, if the windows is not open. If the window  is open,
    use Window.Resize instead.
  **)

  PROCEDURE (w : Window) SetSize*(width,height : LONGINT);

  BEGIN
    IF ~w.maped THEN
      w.width:=width;
      w.height:=height;
    END;
  END SetSize;

  (**
    Sets the minimal width and height of the window.

    NOTE
    Does only work, if the windows is not open. If the window  is open,
    use Window.Resize instead.
  **)

  PROCEDURE (w : Window) SetMinSize*(width,height : LONGINT);

  BEGIN
    IF ~w.maped THEN
      IF width>=0 THEN
        w.minWidth:=width;
      END;
      IF height>=0 THEN
        w.minHeight:=height;
      END;
    END;
  END SetMinSize;

  (**
    Sets the maximal width and height of the window.

    NOTE
    Does only work, if the windows is not open. If the window  is open,
    use Window.Resize instead.
  **)

  PROCEDURE (w : Window) SetMaxSize*(width,height : LONGINT);

  BEGIN
    IF ~w.maped THEN
      IF width>=0 THEN
        w.maxWidth:=width;
      END;
      IF height>=0 THEN
        w.maxHeight:=height;
      END;
    END;
  END SetMaxSize;

  (**
    Sets the top left of the window.

    NOTE
    Does only work, if the windows is not open. If the window  is open,
    use Window.Resize instead.
  **)

  PROCEDURE (w : Window) SetPos*(x,y : LONGINT);

  BEGIN
    IF ~w.maped THEN
      w.x:=x;
      w.y:=y;
    END;
  END SetPos;

  (**
    Set the position modes for vertical and horizotal positioning of the window
    on the display. If no modes are explicitly set, the x,y position of the
    window will be used. This position is the position set with SetPos
    or defaults to zero. In this case the windowmanager might position the
    window (it might, too, if you set a mode, since a windowmanager may ignore
    these values).
  **)

  PROCEDURE (w : Window) SetPosition*(horiz, vert : LONGINT);

  BEGIN
    w.horizontalPos:=horiz;
    w.verticalPos:=vert;
  END SetPosition;

  (**
    Using this function you can link object to the window. If the window gets
    deleted (you must call Window.Delete when the window is not used anymore!)
    all Object.Free gets called for all objects in the freelist. This way you
    can asure that objects that allocate OS ressources (like bitmaps, colors,
    fonts etc...) free this ressources.

    The order of calls to Free for the registered objects is not determinated.
    Also can cannot be sure that Free gets called only ones. For windows which
    have createAllways set, Free gets called every time after the window gets
    closed. So you should allocate your ressources and
    register yourself in GObject.CalcSize to asure reentrace. Object.Free is
    currently not called on program exit! You can call AddFreeList more than
    once.

    You can get a pointer to the window via Display.currentWin while in
    Object.CalcSize.
  **)

  PROCEDURE (w : Window) AddFreeList*(object : O.Object);

  VAR
    entry : FreeEntry;

  BEGIN
    entry:=w.freeList;
    WHILE (entry#NIL) & (entry.object#object) DO
      entry:=entry.next;
    END;

    IF (entry#NIL) & (entry.object=object) THEN
      RETURN;
    END;

    NEW(entry);
    entry.object:=object;
    entry.next:=w.freeList;
    w.freeList:=entry;
  END AddFreeList;

  (**
    Resize the window to the given size. Be carefull to not resize the window
    below the minimal bounds of the top object. Best is to leave resize handling
    completely to the derived window class.

    NOTE
    Derived class should size against minimal and maximal size of its top object.
  **)

  PROCEDURE (w : Window) Resize*(width,height : LONGINT);

  BEGIN
    w.width:=width;
    w.height:=height;
    X11.XResizeWindow(w.display.display,w.window,width,height);
  END Resize;

  (**
    Will be called, when the Window.Open methods needs to create a window
    of the underlying OS. This maybe once before the first call to open or
    before every open when CreateAllways has been called.

    This allows delayed creation of the object-hierachie within the window.

    NOTE
    You must call the method of the baseclass if you overload this method.
  **)

  PROCEDURE (w : Window) PreInit*;

  VAR
    sHints  : xu.XSizeHintsPtr;
    wHints  : xu.XWMHintsPtr;
    cHints  : xu.XClassHintPtr;
    wName,
    iName   : xu.XTextProperty;
    tName   : POINTER TO ARRAY OF POINTER TO  ARRAY OF CHAR;
    attr    : X11.XSetWindowAttributes;


    info       : LONGINT;
    rInfo      :  ARRAY SIZE(LONGINT) OF CHAR;
(*    rInfo      : m.XmDndReceiverProp;
    rSInfo     : ARRAY SIZE(m.XmDndReceiverProp) OF CHAR;
    iInfo      : m.XmDndInitiatorProp;
    iSInfo     : ARRAY SIZE(m.XmDndInitiatorProp) OF CHAR;*)

  BEGIN
    w.inited:=TRUE;

    sHints:=xu.XAllocSizeHints();
    wHints:=xu.XAllocWMHints();
    cHints:=xu.XAllocClassHint();

    IF (sHints=NIL) OR (wHints=NIL) OR (cHints=NIL) THEN
      XFree(sHints);
      XFree(wHints);
      XFree(cHints);
    END;

    attr.background_pixel:=w.display.GetX11Color(w.background);
    attr.border_pixel:=w.display.GetX11Color(w.background);
    attr.backing_store:=X11.NotUseful; (* Give a little boost *)
    attr.save_under:=X11.False;        (* Maybe better false? *)
    attr.event_mask:= X11.KeyPressMask
                     +X11.KeyReleaseMask
                     +X11.ExposureMask
                     +X11.StructureNotifyMask
                     +X11.ButtonPressMask
                     +X11.ButtonReleaseMask
                     +X11.ButtonMotionMask
                     +X11.PointerMotionMask
                     +X11.FocusChangeMask;

    IF w.borderless THEN
      attr.override_redirect:=X11.True;
    ELSE
      attr.override_redirect:=X11.False;
    END;

    CASE w.horizontalPos OF
      centerOnParent:
        IF w.parent#NIL THEN
          w.x:=w.parent.x+(w.parent.width-w.width) DIV 2
        END;
    | centerOnScreen:
        w.x:=(w.display.scrWidth-w.width) DIV 2;
    ELSE
    END;

    CASE w.verticalPos OF
      centerOnParent:
        IF w.parent#NIL THEN
          w.y:=w.parent.y+(w.parent.height-w.height) DIV 2
        END;
    | centerOnScreen:
        w.y:=(w.display.scrHeight-w.height) DIV 2;
    ELSE
    END;

    w.window:=X11.XCreateWindow(w.display.display,
                                X11.XRootWindow(w.display.display,w.display.scrNum),
                                w.x,w.y,
                                w.width,w.height,
                                0,
                                X11.CopyFromParent,
                                X11.InputOutput,
                                w.display.visual^,
                                 X11.CWBackPixel
                                +X11.CWBorderPixel
                                +X11.CWBackingStore
                                +X11.CWOverrideRedirect
                                +X11.CWSaveUnder
                                +X11.CWEventMask,
                                attr);

    IF w.window=0 THEN
      XFree(sHints);
      XFree(wHints);
      XFree(cHints);
    END;

    sHints.flags:=(*xu.PPosition+*)(*xu.PSize+*)xu.PMinSize+xu.PMaxSize;

    sHints.min_width:=w.minWidth;
    sHints.max_width:=w.maxWidth;

    sHints.min_height:=w.minHeight;
    sHints.max_height:=w.maxHeight;

    NEW(tName,2);
    NEW(tName[0],str.Length(w.title^)+1);
    COPY(w.title^,tName[0]^);
    tName[1]:=NIL;
    IF xu.XStringListToTextProperty(s.VAL(C.charPtr2d,tName),1,wName)=0 THEN END;
    IF xu.XStringListToTextProperty(s.VAL(C.charPtr2d,tName),1,iName)=0 THEN END;

    wHints.initial_state:=xu.NormalState;
    wHints.input:=1;
    wHints.flags:=xu.StateHint+xu.InputHint;

    cHints.res_name:=s.VAL(C.charPtr1d,w.title);
    cHints.res_class:=s.VAL(C.charPtr1d,w.title);

    xu.XSetWMProperties(w.display.display,
                        w.window,
                        s.VAL(xu.XTextPropertyPtr,s.ADR(wName)),
                        s.VAL(xu.XTextPropertyPtr,s.ADR(iName)),
                        NIL,0,
                        sHints,
                        wHints,
                        cHints);

    XFree(sHints);
    XFree(wHints);
    XFree(cHints);

    IF X11.XSetWMProtocols(w.display.display,w.window,w.display.deleteProt,1)=0 THEN
      X11.XDestroyWindow(w.display.display,w.window);
      XFree(sHints);
      XFree(wHints);
      XFree(cHints);
    END;

    IF w.parent#NIL THEN
      X11.XSetTransientForHint(w.display.display,w.window,w.parent.window);
    END;

(*    IF w.display.dndRecProp#0 THEN

      (* We register the window as motif drop zone *)

      rInfo.byte_order:=GetEndianess();
      rInfo.protocol_version:=m.DND_PROTOCOL_VERSION;
      rInfo.protocol_style:=m.XmDRAG_DROP_ONLY;
      rInfo.pad1:=0;
      rInfo.proxy_window:=0;
      rInfo.num_drop_sites:=0;
      rInfo.pad2:=0;
      rInfo.total_size:=SIZE(m.XmDndReceiverProp);
      s.MOVE(s.ADR(rInfo),s.ADR(rSInfo),SIZE(m.XmDndReceiverProp));

      X11.XChangeProperty(w.display.display,w.window,w.display.dndRecProp,w.display.dndRecProp,8,
                        X11.PropModeReplace,rSInfo,SIZE(m.XmDndReceiverProp));

      (* We register the window as motif drag zone *)

      iInfo.byte_order:=GetEndianess();
      iInfo.protocol_version:=m.DND_PROTOCOL_VERSION;
      iInfo.target_index:=0;
      iInfo.selection_atom:=w.display.xSelection;
      s.MOVE(s.ADR(iInfo),s.ADR(iSInfo),SIZE(m.XmDndInitiatorProp));

      X11.XChangeProperty(w.display.display,w.window,w.display.dndIniProp,w.display.dndIniProp,8,
                        X11.PropModeReplace,iSInfo,SIZE(m.XmDndInitiatorProp));
    END;*)

    (* We mark the windows as Xdnd aware *)

    info:=XDND_VERSION;
    s.MOVE(s.ADR(info),s.ADR(rInfo),SIZE(LONGINT));

    X11.XChangeProperty(w.display.display,
                        w.window,
                        w.display.XdndAware,
                        w.display.atom,32,
                        X11.PropModeReplace,
                        rInfo,
                        1);

    NEW(w.draw);
    w.draw.Init(w.display,w.window,w);

    w.draw.InstallClip;
    w.draw.AddRegion(0,0,w.display.scrWidth,w.display.scrHeight);

    w.display.AddWindow(w);
  END PreInit;

  (**
    Grabs the mouse cursor and keyboard.
  **)

  PROCEDURE (w : Window) GrabOn;

  BEGIN
    w.display.StopContextHelp;
    IF X11.XGrabPointer(w.display.display,w.window,X11.False,
                         X11.ButtonPressMask
                        +X11.ButtonReleaseMask
                        +X11.PointerMotionMask
                        +X11.LeaveWindowMask
                        +X11.EnterWindowMask
                        +X11.StructureNotifyMask
                        +X11.VisibilityChangeMask
                        +X11.FocusChangeMask
                        +X11.ButtonMotionMask,
                        X11.GrabModeAsync,
                        X11.GrabModeAsync,
                        X11.None,
                        w.display.popCursor,
                        X11.CurrentTime)#X11.GrabSuccess THEN
      Err.String("Can't grab cursor"); Err.Ln;
    END;
    IF X11.XGrabKeyboard(w.display.display,w.window,X11.False,
                         X11.GrabModeAsync,
                         X11.GrabModeAsync,
                         X11.CurrentTime)#X11.GrabSuccess THEN
      Err.String("Can't grab cursor"); Err.Ln;
    END;
    X11.XSetInputFocus(w.display.display,w.window,X11.RevertToParent,X11.CurrentTime);
  END GrabOn;

  (**
    Releases the grab of the mouse cursor and the keyboard.
  **)

  PROCEDURE (w : Window) GrabOff;

  BEGIN
    X11.XUngrabPointer(w.display.display,X11.CurrentTime);
    X11.XUngrabKeyboard(w.display.display,X11.CurrentTime);
    w.display.RestartContextHelp;
  END GrabOff;

  (**
    Do mouse and keyboard-grabing.

    NOTE
    Changing the value does only work, when the window is not visible.
  **)

  PROCEDURE (w : Window) Grab*(grab : BOOLEAN);

  BEGIN
    IF w.grab=grab THEN
      RETURN;
    END;

    IF ~w.maped THEN
      w.grab:=grab;
    ELSE
      IF grab THEN
        w.GrabOn;
      ELSE
        w.GrabOff;
      END;
      w.grab:=grab;
    END;
  END Grab;


  (**
    Opens the window.

    NOTE
    Derived classes must call th e baseclass method.
  **)

  PROCEDURE (w : Window) Open*;

  VAR
   window : Window;

  BEGIN
    w.open:=TRUE;

    IF ~w.inited OR w.createAllways THEN
      w.display.SetNewWindow(w);
      w.PreInit;
    END;

    w.exposed:=FALSE;
    w.draws:=0;
    w.maped:=FALSE;
    w.modalCount:=0;

    IF w.modal THEN
      window:=w.display.winList;
      WHILE window#NIL DO
        IF window#w THEN
          INC(window.modalCount);
        END;
        IF window.modalCount=1 THEN
          X11.XDefineCursor(window.display.display,window.window,window.display.sleepCursor);
        END;
        window:=window.next;
      END;
    END;

    X11.XMapWindow(w.display.display,w.window);

    IF w.grab THEN
      w.GrabOn;
    END;
  END Open;

  (**
    Called, when the display want the window to reinit itself, f.e
    when the preferences of some or all of the objects has been changed.
  **)

  PROCEDURE (w : Window) ReinitWindow*;

  BEGIN
  END ReinitWindow;

  (**
    Call this method when you want the window to hide.
    The window has to call Object.Hide on each of its containing
    object to ensure, that the Object.visible has has been cleared.

    NOTE
    Hiding a window does not mean that the window will be destroyed. After
    creation you can open and hide a window as often as you like.

    Currently this message does nothing however should always call the baseclass
    method.
  **)

  PROCEDURE (w : Window) Hide*;

  BEGIN
  END Hide;

  (**
    Removes the window from the list of windows known by the
    Display.

    NOTE
    You must call this method before closing the window.
  **)

  PROCEDURE (w : Window) Close*;

  VAR
    window : Window;
    entry  : FreeEntry;

  BEGIN
    w.open:=FALSE;

    IF w.grab THEN
      w.GrabOff;
    END;

    IF w.maped & (w.window#0) THEN
      X11.XUnmapWindow(w.display.display,w.window);
    END;

    w.Hide;

    w.modalCount:=-1;
    w.parentWin:=0;
    w.maped:=FALSE;

    IF w.modal THEN
      window:=w.display.winList;
      WHILE window#NIL DO
        IF (window#w) & (window.modalCount>0) THEN
          DEC(window.modalCount);
          IF window.modalCount=0 THEN
            X11.XUndefineCursor(window.display.display,window.window);
          END;
        END;
        window:=window.next;
      END;
    END;

    IF w.createAllways THEN
      IF w.window#0 THEN
        w.draw.FreeLastClip;
        w.draw.Deinit;
        X11.XDestroyWindow(w.display.display,w.window);
        w.window:=0;
      END;

      w.maped:=FALSE;
      w.display.RemoveWindow(w);
      entry:=w.freeList;

      WHILE entry#NIL DO
        entry.object.Free;
        entry:=entry.next;
      END;
      w.freeList:=NIL;
    END;
  END Close;

  (**
    Deinitializes the window.

    NOTE
    Call this method before deleting the window.
  **)

  PROCEDURE (w : Window) Delete*;

  VAR
    entry : FreeEntry;

  BEGIN
    IF w.window#0 THEN
      w.Close;
      w.draw.FreeLastClip;
      w.draw.Deinit;
      X11.XDestroyWindow(w.display.display,w.window);
      w.window:=0;
    END;

    w.maped:=FALSE;
    w.display.RemoveWindow(w);

    entry:=w.freeList;
    WHILE entry#NIL DO
      entry.object.Free;
      entry:=entry.next;
    END;
    w.freeList:=NIL;
  END Delete;

  (**
    Convert the window relative x,y coords to absolute coords.
  **)

  PROCEDURE (w : Window) GetXY*(VAR x,y : LONGINT);

  VAR
    retAttr : X11.XWindowAttributes;
    root,
    parent,
    current : X11.Window;
    childs  : X11.WindowPtr1d;
    count   : LONGINT;

  BEGIN
    x:=0;
    y:=0;

    current:=w.window;

    IF X11.XQueryTree(w.display.display,current,root,parent,childs,count)#0 THEN
      XFree(childs);
      WHILE current#root DO
        IF X11.XGetWindowAttributes(w.display.display,current,retAttr)#0 THEN
          INC(x,retAttr.x+retAttr.border_width);
          INC(y,retAttr.y+retAttr.border_width);
        END;
        current:=parent;
        IF X11.XQueryTree(w.display.display,current,root,parent,childs,count)#0 THEN
          XFree(childs);
        END;
      END;
    END;
  END GetXY;

  (**
    Returns the mouse position in window relative and abslute coords.
  **)

  PROCEDURE (w : Window) GetMousePos*(VAR rx, ry, wx, wy : LONGINT);

  VAR
    root,
    child  : X11.Window;
    bmask  : X11.ulongmask;

  BEGIN
    IF X11.XQueryPointer(w.display.display,w.window,root,child,rx,ry,wx,wy,bmask)=X11.False THEN
      Err.String("Can't get cursorpos"); Err.Ln;
    END;
  END GetMousePos;

  PROCEDURE (d : Display) GetFontList*():Font;

  VAR
    count,
    x,y    : LONGINT;
    names  : C.charPtr2d;
    fonts  : X11.XFontStructPtr1d;
    first,
    last,
    font   : Font;
    name   : FontName;

  BEGIN
    names:=X11.XListFontsWithInfo(d.display,"-*-*-medium-r-*--*-*-*-*-*-*-*-*",
                                  MAX(LONGINT),count,fonts);

    first:=NIL;

    IF (names#NIL) & (count>0) THEN
      FOR x:=0 TO count-1 DO
         y:=0;
         WHILE names[x][y]#0X DO
           name[y]:=names[x][y];
           INC(y);
         END;
         name[y]:=0X;

         NEW(font);
         font.Init;
         font.InitFromFontInfo(name,fonts[x]);

         IF x=0 THEN
           first:=font;
           font.last:=NIL;
         ELSE
           last.next:=font;
           font.last:=last;
         END;
         last:=font;
         font.next:=NIL;
      END;
    END;

    X11.XFreeFontInfo(names,fonts[0],count);

    RETURN first;
  END GetFontList;

  (* ------------ Handlers for window-events -------------- *)

  (**
    Called when the the closing gadget of the window got pressed.
  **)

  PROCEDURE (w : Window) ClosePressed*;

  BEGIN
  END ClosePressed;

  (**
    Called, when window has been maped.
    Call baseclass method, when you inherit.
  **)

  PROCEDURE (w : Window) Maped*;

  BEGIN
    w.maped:=TRUE;
    w.GetXY(w.x,w.y);
  END Maped;

  (**
    Called, when window has been unmaped.
    Call baseclass method, when you inherit.
  **)

  PROCEDURE (w : Window) Unmaped*;

  BEGIN
    w.maped:=FALSE;
    w.draws:=0;
  END Unmaped;

  (**
    Called, when the window gets the keyboard focus.
  **)

  PROCEDURE (w : Window) FocusIn*;

  BEGIN
    IF ~w.grab THEN
      w.display.StartContextHelp;
    END;
    w.focus:=TRUE;
  END FocusIn;

  (**
    Called, when the window looses the keyboard focus.
  **)

  PROCEDURE (w : Window) FocusOut*;

  BEGIN
    IF ~w.grab THEN
      w.display.StopContextHelp;
    END;
    w.focus:=FALSE;
  END FocusOut;

  (**
    This method get calls, when the window becomes (partial) hidden, e.g.
    gets covered by another window. This is likely to not be supported for all
    platforms.
  **)

  PROCEDURE (w : Window) Hidden*;

  BEGIN
  END Hidden;

  (**
    Gets called, when the mouse (not the focus!) has left the window.
  **)

  PROCEDURE (w : Window) Left*;

  BEGIN
  END Left;

  (**
    Will be called, if you have to initial draw yourself.
    Overload it as aproximate.
  **)

  PROCEDURE (w : Window) Draw*;

  BEGIN
  END Draw;

  (**
    Will be called, if you have to redraw yourself.
    Overload it as aproximate. The window can restrict the redrawing to the given
    area.
  **)

  PROCEDURE (w : Window) Redraw*(x,y,width,height : LONGINT);

  BEGIN
    w.Draw;
  END Redraw;

  (**
    This method gets called, when the window has been resized.
    The given size is the size of the window-inner, that means,
    the size of the window without its borders.

    Note, that window#decorationwindow!
  **)

  PROCEDURE (w : Window) Resized*(width,height : LONGINT);

  BEGIN
  END Resized;

  (**
    This method gets called, when the display thinks you should
    open a tooltip help.
  **)

  PROCEDURE (w : Window) ContextHelp*;

  BEGIN
  END ContextHelp;

  (**
    This method gets called, when the display things you should
    open a context sensitiv menu.

    RESULT
    Return TRUE if you have opened a context menu, else FALSE.

    NOTE
    If this method returns FALSE, the display will propagate
    the corresponding event that started contextsensitive help
    to the window.
  **)

  PROCEDURE (w : Window) ContextMenu*():BOOLEAN;

  BEGIN
    RETURN FALSE;
  END ContextMenu;

  (**
    This one gets called, when the display thinks the window should
    change the KeyboardFocus.
  **)

  PROCEDURE (w : Window) FocusNext*;

  BEGIN

  END FocusNext;

  (**
    This one gets called by the focus object, if it hides.
  **)

  PROCEDURE (w : Window) FocusObjectHides*;

  BEGIN

  END FocusObjectHides;

  (**
    Returns the object that coveres the given point and that supports
    dragging of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.

    NOTE
    A window can be sure to be called only, if dragging or dropping makes sense.
    For example, you cannot drop data onto a window that is block due to its
    modal count. However, you can always drag data from a window.
  **)

  PROCEDURE (w : Window) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):Object;

  BEGIN
    RETURN NIL;
  END GetDnDObject;

  (**
    Returns the window currently under the mouse pointer. It also returns the
    screen relative and window relative mouse coordinates.
  **)

  PROCEDURE (d : Display) GetWindowOnScreen(VAR rX,rY,cX,cY : LONGINT):X11.Window;

  VAR
    help,rW,cW  : LONGINT;
    dragWin     : X11.Window;
    mask        : SET;

    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    data        : C.charPtr1d;

  BEGIN
    dragWin:=0;
    data:=NIL;
    itemsReturn:=0;

    IF X11.XQueryPointer(d.display,X11.XRootWindow(d.display,0),
                         rW,cW,rX,rY,cX,cY,mask)=X11.True THEN

      WHILE (rW#0) & (dragWin=0) & (cW#0) & (rW#cW) DO
        IF (X11.XGetWindowProperty(d.display,
                                   cW,d.wmState,0,MAX(LONGINT),X11.False,
                                   X11.AnyPropertyType,retType,retFormat,itemsReturn,
                                   bytesLeft,data)=X11.Success) & (retType#X11.None) THEN
          dragWin:=cW;
        END;

        IF rW#0 THEN
          help:=cW;
          IF X11.XTranslateCoordinates(d.display,
                                       rW,cW,cX,cY,cX,cY,cW)=X11.True THEN
            rW:=help;
          ELSE
            Err.String("Cannot convert coordinates!"); Err.Ln;
            rW:=0;
          END;
        END;
      END;
    END;
    RETURN dragWin;
  END GetWindowOnScreen;

  (**
    Get the VO window matching the given X11 window.
  **)

  PROCEDURE (d : Display) GetWindow*(window : X11.Window):Window;

  VAR
    help : Window;

  BEGIN
    help:=d.winList;
    WHILE help#NIL DO
      IF help.window=window THEN
        RETURN help;
      END;
      help:=help.next;
    END;
    RETURN NIL;
  END GetWindow;

  (**
    Called when some drag action should be started.
  **)

  PROCEDURE (w : Window) HandleDrag(event : E.Event):BOOLEAN;

  BEGIN
    w.display.dragStart:=FALSE;

    IF w.modalCount=0 THEN
      w.display.dragObject:=w.GetDnDObject(w.display.dragX,w.display.dragY,TRUE);
      IF w.display.dragObject#NIL THEN
        NEW(w.display.dragInfo);
        w.display.dragInfo.Init;
        w.display.dragObject.GetDragInfo(w.display.dragInfo);

        w.display.StopContextHelp;

        (* Calculation action *)
        IF event(E.MotionEvent).qualifier*E.keyMask=E.shiftMask THEN
          w.display.dndAction:=D.move;
          X11.XDefineCursor(w.display.display,w.window,w.display.moveCursor);
        ELSIF event(E.MotionEvent).qualifier*E.keyMask=E.controlMask THEN
          w.display.dndAction:=D.copy;
          X11.XDefineCursor(w.display.display,w.window,w.display.copyCursor);
        ELSIF event(E.MotionEvent).qualifier*E.keyMask=E.shiftMask+E.controlMask THEN
          w.display.dndAction:=D.link;
          X11.XDefineCursor(w.display.display,w.window,w.display.linkCursor);
        ELSE
          w.display.dndAction:=D.default;
          X11.XDefineCursor(w.display.display,w.window,w.display.dndCursor);
        END;

        w.display.dropWindow:=event.event.xmotion.window;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleDrag;

  (**
    Called when a drop action should be handled.
  **)

  PROCEDURE (w : Window) HandleDragDrop(event : E.Event):BOOLEAN;

  VAR
    dragWin    : X11.Window;
    rX,rY,
    cX,cY      : LONGINT;
    return     : BOOLEAN;
    window     : Window;
    dropObject : Object;
    dragData   : D.DnDData;
    group,type : LONGINT;

  BEGIN
    return:=FALSE;
    w.display.dragStart:=FALSE;

    IF (w.display.dropWindow#0) THEN

      w.display.RestartContextHelp;

      dragWin:=w.display.GetWindowOnScreen(rX,rY,cX,cY);
      IF dragWin#0 THEN
        window:=w.display.GetWindow(dragWin);
        IF window#NIL THEN
          dropObject:=window.GetDnDObject(cX,cY,FALSE);
          IF (window.modalCount=0) & (dropObject#NIL) & (dropObject#w.display.dragObject) THEN
            IF dropObject.GetDropDataType(w.display.dragInfo,
                                          group,type,w.display.dndAction) THEN
              dragData:=w.display.dragObject.GetDragData(group,type,w.display.dndAction);
              IF dragData#NIL THEN
                IF dropObject.HandleDrop(dragData,w.display.dndAction) THEN
                  (* nothing to do yet*)
                ELSE
                  X11.XBell(w.display.display,100);
                END;
              END;
            END;
            return:=TRUE;
          END;
        ELSE
          Err.String("End external drag -> drop"); Err.Ln;
          (*w.HandleMotifDragDrop(dragWin,rX,rY,event.event.xbutton);*)
        END;
      END;
      X11.XUndefineCursor(w.display.display,w.window);
      w.display.dropWindow:=0;
      w.display.dragObject:=NIL;
    END;

    RETURN return;
  END HandleDragDrop;

  (**
    Handle the XSelecitionNotify of X11. A SelectionNotify gets send
    when someone made a request for the selection value. The event states
    where one can get the selection value from.
  **)

  PROCEDURE (w : Window) HandleXSelectionNotify(event : X11.XSelectionEvent);

  VAR
    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    data        : C.charPtr1d;
    dndData     : D.DnDStringData;
    queryObject : Object;
    text        : U.Text;

  BEGIN
    queryObject:=w.display.querySelectObject;
    w.display.querySelectObject:=NIL;

    IF event.property=X11.None THEN
      Out.String("No property"); Out.Ln;
      RETURN;
    END;

    IF X11.XGetWindowProperty(w.display.display,
                              event.requestor,event.property,0,MAX(LONGINT),X11.True,
                              X11.AnyPropertyType,
                              retType,retFormat,itemsReturn,bytesLeft,
                              data)#X11.Success THEN
      Out.String("Cannot get property data"); Out.Ln;
      RETURN;
    END;

    IF retType=X11.None THEN
      Out.String("Illegal property data type"); Out.Ln;
      RETURN;
    END;

    IF event.property=w.display.xSelection THEN
      IF (retType=a.XA_STRING) & (retFormat=8) THEN
        NEW(dndData);
        NEW(dndData.string,itemsReturn+1);
        s.MOVE(s.VAL(LONGINT,data),
               s.VAL(LONGINT,dndData.string),itemsReturn);
        dndData.string[itemsReturn]:=0X;

        IF queryObject#NIL THEN
          IF queryObject.HandleDrop(dndData,D.insert) THEN
            (* nothing to do *)
          ELSE
            X11.XBell(w.display.display,100);
          END;
        END;
      END;
    ELSE
      Out.String("Drop result: ");
      text:=U.CStringToText(data);
      Out.String(text^);
      Out.Ln;
    END;
  END HandleXSelectionNotify;

  (**
    handle XSelectionrequest of X11. A SelectionRequest gets send when some application
    wants the selection value and your window has registered the selection. We
    ask the object which has registered the selection for the selection value
    and return a notify message to the requestor.
  **)

  PROCEDURE (w : Window) HandleXSelectionRequest(event : X11.XSelectionRequestEvent);

  VAR
    data        : D.DnDData;
    selNotify   : X11.XEvent;
    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    string      : C.charPtr1d;

  BEGIN
    selNotify.xselection.type:=X11.SelectionNotify;
    selNotify.xselection.display:=event.display;
    selNotify.xselection.requestor:=event.requestor;
    selNotify.xselection.selection:=event.selection;
    selNotify.xselection.target:=event.target;
    selNotify.xselection.property:=event.property;
    selNotify.xselection.time:=event.time;


    IF (event.target=a.XA_STRING) & (w.display.selectObject#NIL) THEN
      data:=w.display.selectObject.GetDragData(D.text,D.none,D.copy);
      IF (data#NIL) & (data IS D.DnDStringData) THEN
        string:=s.VAL(C.charPtr1d,data(D.DnDStringData).string);
        IF X11.XGetWindowProperty(event.display,
                                  event.requestor,event.property,0,MAX(LONGINT),X11.False,
                                  event.target,retType,retFormat,itemsReturn,bytesLeft,
                                  string)#X11.Success THEN
            Err.String("Cannot get property"); Err.Ln;
            selNotify.xselection.property:=X11.None;
        ELSE
          X11.XChangeProperty(event.display,event.requestor,event.property,
                              a.XA_STRING,8,
                              X11.PropModeReplace,data(D.DnDStringData).string^,
                              str.Length(data(D.DnDStringData).string^));
        END;
      ELSE
        selNotify.xselection.property:=X11.None;
      END;
    ELSE
      selNotify.xselection.property:=X11.None;
    END;

    IF X11.XSendEvent(selNotify.xselection.display,
                      selNotify.xselection.requestor,X11.True,X11.NoEventMask,
                      s.VAL(X11.XEventPtr,s.ADR(selNotify)))=0 THEN
      Err.String("Error sending selection notify"); Err.Ln;
    END;
  END HandleXSelectionRequest;

  (**
    If you derive yourself from window and want to overload the
    defaulthandler, call the baseclass first in your handler and check the
    result, if its is TRUE the defaulthandler has allready handled it.
  **)

  PROCEDURE (w : Window) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    x,y,
    width,
    height : LONGINT;
    name   : C.charPtr1d;
    text   : U.Text;
    msg    : X11.XClientMessageEvent;

    retType     : X11.Atom;
    retFormat,
    itemsReturn,
    bytesLeft   : LONGINT;
    delete      : X11.Bool;
    xData       : C.charPtr1d;
    data        : POINTER TO ARRAY OF LONGINT;

  BEGIN
(** Modified by Marco **)
(* new code *)
    w.display.lastEventIsButtonPress:=FALSE;
(* end of new code *)
    CASE event.event.type OF
      X11.ClientMessage:
        IF (event.event.xclient.data.l[0]=w.display.deleteProt[0])
        & (w.modalCount=0) THEN
          w.ClosePressed;
        ELSIF (event.event.xclient.message_type=w.display.XdndEnter) THEN
          Out.String("XDND Enter"); Out.Ln;
          Out.String("XID: "); Out.Hex(event.event.xclient.data.l[0],0); Out.Ln;
          Out.String("Protokoll: "); Out.LongInt(s.VAL(SHORTINT,event.event.xclient.data.b[7]),0); Out.Ln;

          FOR x:=2 TO 4 DO
            name:=X11.XGetAtomName(w.display.display,event.event.xclient.data.l[x]);
            IF name#NIL THEN
              Out.String("Datatype: ");
              text:=U.CStringToText(name);
              Out.String(text^); Out.Ln;
            END;
          END;

          IF event.event.xclient.data.b[4]#0X THEN (* TODO: fix *)
            IF X11.XGetWindowProperty(w.display.display,
                                      event.event.xclient.data.l[0],
                                      w.display.XdndTypeList,
                                      0,
                                      MAX(LONGINT),
                                      X11.False,
                                      w.display.atom,
                                      retType,
                                      retFormat,
                                      itemsReturn,
                                      bytesLeft,
                                      xData)#X11.Success THEN
              Err.String("Cannot get property"); Err.Ln;
            ELSE
              NEW(data,itemsReturn);
              s.MOVE(s.VAL(LONGINT,xData),s.VAL(LONGINT,data),SIZE(LONGINT)*itemsReturn);
              XFree(xData);
              FOR x:=0 TO itemsReturn-1 DO
                name:=X11.XGetAtomName(w.display.display,data[x]);
                IF name#NIL THEN
                  Out.String("Datatype: ");
                  text:=U.CStringToText(name);
                  Out.String(text^); Out.Ln;
                END;
              END;
            END;
          END;

          Out.Ln;

        ELSIF (event.event.xclient.message_type=w.display.XdndLeave) THEN
          Out.String("XDND Leave"); Out.Ln;
          Out.String("XID: "); Out.Hex(event.event.xclient.data.l[0],0); Out.Ln;
          Out.Ln;

        ELSIF (event.event.xclient.message_type=w.display.XdndPosition) THEN
          Out.String("XDND Position"); Out.Ln;
          Out.String("XID: "); Out.Hex(event.event.xclient.data.l[0],0); Out.Ln;
          Out.String("X: "); Out.LongInt(event.event.xclient.data.s[5],0); Out.Ln;
          Out.String("Y: "); Out.LongInt(event.event.xclient.data.s[4],0); Out.Ln;

          IF event.event.xclient.data.l[4]=w.display.XdndActionMove THEN
            Out.String("action move"); Out.Ln;
          ELSIF event.event.xclient.data.l[4]=w.display.XdndActionCopy THEN
            Out.String("action copy"); Out.Ln;
          ELSIF event.event.xclient.data.l[4]=w.display.XdndActionLink THEN
            Out.String("action link"); Out.Ln;
          END;
          Out.Ln;

          msg.type:=X11.ClientMessage;
          msg.format:=32;
          msg.display:=w.display.display;
          msg.window:=event.event.xclient.data.l[0];
          msg.message_type:=w.display.XdndStatus;
          msg.data.l[0]:=w.window;
          msg.data.l[1]:=3;
          msg.data.s[4]:=0; (* y *)
          msg.data.s[5]:=0; (* x *)
          msg.data.s[6]:=0; (* h *)
          msg.data.s[7]:=0; (* w *)
          msg.data.l[4]:=w.display.XdndActionCopy;

          IF X11.XSendEvent(w.display.display,
                            event.event.xclient.data.l[0],
                            X11.False,X11.NoEventMask,
                            s.VAL(X11.XEventPtr,s.ADR(msg)))=0 THEN
            Out.String("Could not send XDNDStatus message"); Out.Ln;
          END;


        ELSIF (event.event.xclient.message_type=w.display.XdndStatus) THEN
          Out.String("XDND Status"); Out.Ln;

        ELSIF (event.event.xclient.message_type=w.display.XdndDrop) THEN
          Out.String("XDND Drop"); Out.Ln;

          msg.type:=X11.ClientMessage;
          msg.format:=32;
          msg.display:=w.display.display;
          msg.window:=w.window;
          msg.message_type:=w.display.XdndFinished;
          msg.data.l[0]:=w.window;
          msg.data.l[1]:=0;

          X11.XConvertSelection(w.display.display,
                                w.display.XdndSelection,
                                297, (* text/plain *)
                                w.display.xDrop,
                                w.window,
                                event.event.xclient.data.l[2]);

(*          dropObject:=window.GetDnDObject(cX,cY,FALSE);
                IF dropObject.HandleDrop(dragData,w.display.dndAction) THEN*)

(*          IF X11.XSendEvent(w.display.display,
                            event.event.xclient.data.l[0],
                            X11.False,X11.NoEventMask,
                            s.VAL(X11.XEventPtr,s.ADR(msg)))=0 THEN
            Out.String("Could not send XDNDStatus message"); Out.Ln;
          END;*)

        ELSIF (event.event.xclient.message_type=w.display.XdndFinished) THEN
          Out.String("XDND Finished"); Out.Ln;
          Out.String("XID: "); Out.Hex(event.event.xclient.data.l[0],0); Out.Ln;
          Out.Ln;
        ELSE
          Out.String("Unknown client message: ");
          Out.LongInt(event.event.xclient.message_type,0); Out.Ln;
        END;
    | X11.ConfigureNotify:
        w.GetXY(w.x,w.y);
        IF (event.event.xconfigure.width#w.width)
        OR (event.event.xconfigure.height#w.height) THEN
          w.width:=event.event.xconfigure.width;  (* Where get they set the first time? *)
          w.height:=event.event.xconfigure.height;
          w.Resized(event.event.xconfigure.width -event.event.xconfigure.border_width,
                    event.event.xconfigure.height-event.event.xconfigure.border_width);
        END;
    | X11.Expose:
        IF event.event.xexpose.count=0 THEN
          IF ~w.exposed THEN
            w.draw.InstallClip;
            w.exposed:=TRUE;
          END;
          w.draw.AddRegion(event.event.xexpose.x,event.event.xexpose.y,
                           event.event.xexpose.width,event.event.xexpose.height);
          IF w.draws=0 THEN
            w.Draw;
          ELSE
            w.draw.GetClipRegion(x,y,width,height);
            w.Redraw(x,y,width,height);
          END;
          w.draw.FreeLastClip;
          w.exposed:=FALSE;
          INC(w.draws);
        ELSE
         IF ~w.exposed THEN
            w.exposed:=TRUE;
            w.draw.InstallClip;
          END;
          w.draw.AddRegion(event.event.xexpose.x,event.event.xexpose.y,
                           event.event.xexpose.width,event.event.xexpose.height);
        END;
    | X11.GraphicsExpose:
        IF event.event.xgraphicsexpose.count=0 THEN
          IF ~w.exposed THEN
            w.draw.InstallClip;
            w.exposed:=TRUE;
          END;
          w.draw.AddRegion(event.event.xgraphicsexpose.x,event.event.xgraphicsexpose.y,
                           event.event.xgraphicsexpose.width,event.event.xgraphicsexpose.height);
          IF w.draws=0 THEN
            w.Draw;
          ELSE
            w.draw.GetClipRegion(x,y,width,height);
            w.Redraw(x,y,width,height);
          END;
          w.draw.FreeLastClip;
          w.exposed:=FALSE;
          INC(w.draws);
        ELSE
         IF ~w.exposed THEN
            w.exposed:=TRUE;
            w.draw.InstallClip;
          END;
          w.draw.AddRegion(event.event.xgraphicsexpose.x,event.event.xgraphicsexpose.y,
                           event.event.xgraphicsexpose.width,event.event.xgraphicsexpose.height);
        END;
    | X11.CreateNotify:
        w.parentWin:=event.event.xcreatewindow.parent;
    | X11.ReparentNotify:
        w.parentWin:=event.event.xreparent.parent;
    | X11.MappingNotify:
        X11.XRefreshKeyboardMapping(s.VAL(X11.XMappingEventPtr,s.ADR(event.event)));
    | X11.UnmapNotify:
        w.Unmaped;
    | X11.MapNotify:
        w.Maped;
    | X11.FocusIn:
        w.FocusIn;
    | X11.FocusOut:
        w.FocusOut;
    | X11.EnterNotify:
        (* TODO w.Entered *)
    | X11.LeaveNotify:
        w.Left;
    | X11.VisibilityNotify:
        IF event.event.xvisibility.state>0 THEN
          w.Hidden;
        END;
    | X11.SelectionNotify:
        Out.String("SelectionNotify"); Out.Ln;
        w.HandleXSelectionNotify(event.event.xselection);
    | X11.SelectionClear:
        IF w.display.selectObject#NIL THEN
          w.display.selectObject.Deselect;
          w.display.selectObject:=NIL;
        END;
    | X11.SelectionRequest:
        w.HandleXSelectionRequest(event.event.xselectionrequest);
(** Modified by Marco **)
(* new code *)
    | X11.ButtonPress:
        IF (event.event.xbutton.button=E.dragDropButton) THEN
          w.display.dragStart:=TRUE;
          w.display.dragX:=event.event.xbutton.x;
          w.display.dragY:=event.event.xbutton.y;
        END;
        w.display.lastEventIsButtonPress:=TRUE;
        w.display.lastBut1Button:=w.display.lastButton;
        w.display.lastBut1PressTime:=w.display.lastPressTime;
        w.display.lastButton:=SHORT(SHORT(event.event.xbutton.button));
        w.display.lastPressTime:=event.event.xbutton.time;
        IF w.modalCount=0 THEN
          RETURN FALSE;
        ELSE
          RETURN TRUE;
        END;
    | X11.ButtonRelease:
        IF (event.event.xbutton.type=X11.ButtonRelease)
        & (event.event.xbutton.button=E.dragDropButton) THEN
          IF w.HandleDragDrop(event) THEN
            RETURN TRUE;
          END;
        END;
        IF w.modalCount=0 THEN
          RETURN FALSE;
        ELSE
          RETURN TRUE;
        END;
(* old code:
    | X11.ButtonPress,
      X11.ButtonRelease:
        IF (event.event.xbutton.type=X11.ButtonPress)
        & (event.event.xbutton.button=E.dragDropButton) THEN
          w.display.dragStart:=TRUE;
          w.display.dragX:=event.event.xbutton.x;
          w.display.dragY:=event.event.xbutton.y;
        ELSIF (event.event.xbutton.type=X11.ButtonRelease)
        & (event.event.xbutton.button=E.dragDropButton) THEN
          IF w.HandleDragDrop(event) THEN
            RETURN TRUE;
          END;
        END;
        IF w.modalCount=0 THEN
          RETURN FALSE;
        ELSE
          RETURN TRUE;
        END;
(* end modification *) *)
    | X11.MotionNotify:
        IF (event(E.MotionEvent).qualifier*E.buttonMask={E.dragDropButton})
         & (w.display.dragStart) THEN
          IF   (event.event.xmotion.x>w.display.dragX+w.display.spaceWidth DIV 2)
            OR (event.event.xmotion.x<w.display.dragX-w.display.spaceWidth DIV 2)
            OR (event.event.xmotion.y>w.display.dragY+w.display.spaceHeight DIV 2)
            OR (event.event.xmotion.y<w.display.dragY-w.display.spaceHeight DIV 2) THEN
            IF w.HandleDrag(event) THEN
              RETURN TRUE;
            END;
          END;
        END;
        IF w.modalCount=0 THEN
          RETURN FALSE;
        ELSE
          RETURN TRUE;
        END;
    ELSE
      IF w.modalCount=0 THEN
        RETURN FALSE;
      ELSE
        RETURN TRUE;
      END;
    END;
    RETURN TRUE;
  END HandleEvent;

  (* ------------ Display ----------------- *)

  (**
    Initialize a instance of the Display class.
  **)

  PROCEDURE (d : Display) InitDisplay*(name : ARRAY OF CHAR):BOOLEAN;

  VAR
    result     : C.charPtr1d;
    i,j        : LONGINT;
    exactColor : X11.XColor;
    error      : BOOLEAN;
    pixel      : ARRAY 1 OF LONGINT;

    PROCEDURE CreatePixmapCursor(bit1,bit2 : C.address; w,h : LONGINT):X11.Cursor;

    VAR
      pix1,pix2 : X11.Pixmap;
      cursor    : X11.Cursor;

    BEGIN
      pix1:=X11.XCreatePixmapFromBitmapData(d.display,X11.XRootWindow(d.display,d.scrNum),
                                            bit1,w,h,
                                            X11.XBlackPixel(d.display,d.scrNum),
                                            X11.XWhitePixel(d.display,d.scrNum),1);

      pix2:=X11.XCreatePixmapFromBitmapData(d.display,X11.XRootWindow(d.display,d.scrNum),
                                            bit2,w,h,
                                            X11.XWhitePixel(d.display,d.scrNum),
                                            X11.XBlackPixel(d.display,d.scrNum),1);

      IF (pix1#0) & (pix2#0) THEN
        cursor:=X11.XCreatePixmapCursor(d.display,pix1,pix2,
                                        d.color[whiteColor].color,
                                        d.color[blackColor].color,
                                        w DIV 2,h DIV 2);
      ELSE
        cursor:=0;
      END;

      IF pix1#0 THEN
        X11.XFreePixmap(d.display,pix1);
      END;
      IF pix2#0 THEN
        X11.XFreePixmap(d.display,pix2);
      END;

      RETURN cursor;
    END CreatePixmapCursor;


  BEGIN
    d.Init;

    NEW(d.appName,str.Length(name)+1);
    COPY(name,d.appName^);

    d.name:="";
    result:=X11.XDisplayName(NIL);
    IF result#NIL THEN
      COPY(result^,d.name);
    END;

    d.display:=X11.XOpenDisplay(d.name);
    IF d.display=NIL THEN
      RETURN FALSE;
    END;

<* IF LIB_HAVE_LIBIMLIB=TRUE THEN *>

    d.imlib:=Imlib.Imlib_init(d.display);

    d.display:=d.imlib.x.display;
    d.scrNum:=d.imlib.x.screen;
    d.visual:=d.imlib.x.visual;
    d.colorDepth:=d.imlib.x.depth;
    d.colorMap:=d.imlib.x.root_cmap;

<* ELSE *>

    d.scrNum:=X11.XDefaultScreen(d.display);
    d.visual:=X11.XDefaultVisual(d.display,d.scrNum);
    IF d.visual=NIL THEN
      Err.String("Cannot get visual!"); Err.Ln;
      RETURN FALSE;
    END;
    d.colorDepth:=X11.XDefaultDepth(d.display,d.scrNum);
    d.colorMap:=X11.XDefaultColormap(d.display,d.scrNum);

<* END *>

    d.scrWidth:=X11.XDisplayWidth(d.display,d.scrNum);
    d.scrHeight:=X11.XDisplayHeight(d.display,d.scrNum);

    IF (d.visual.class=X11.GrayScale)
    OR (d.visual.class=X11.StaticGray) THEN
      IF d.colorDepth=1 THEN
        d.colorMode:=monochromeMode;
        prefs:=monoPrefs;
      ELSE
        d.colorMode:=greyScaleMode;
        prefs:=greyPrefs;
      END;
    ELSIF (d.visual.class=X11.PseudoColor)
    OR (d.visual.class=X11.StaticColor)
    OR (d.visual.class=X11.DirectColor)
    OR (d.visual.class=X11.TrueColor) THEN
      d.colorMode:=colorMode;
      prefs:=colorPrefs;
    ELSE
      Err.String("Unsupported visual class!"); Err.Ln;
      RETURN FALSE;
    END;

    prefs.display:=d;

    REPEAT
      error:=FALSE;

      IF prefsCallback#NIL THEN
        prefsCallback(name,d);
      END;

      i:=0;
      WHILE (i<colorCount) & ~error DO
        IF X11.XLookupColor(d.display,
                            d.colorMap,
                            prefs.colors[i],
                            exactColor,
                            d.color[i].color)=0 THEN
          X11.XCloseDisplay(d.display);
          Err.String("Cannot parse color '"); Err.String(prefs.colors[i]); Err.String("'"); Err.Ln;
          error:=TRUE;
        END;
        IF X11.XAllocColor(d.display,d.colorMap,d.color[i].color)=0 THEN
          Err.String("Cannot allocate color '"); Err.String(prefs.colors[i]); Err.String("'"); Err.Ln;
          error:=TRUE;
        END;
        INC(i);
      END;

      IF error THEN
        FOR j:=0 TO i-2 DO
          pixel[0]:=d.color[j].color.pixel;
          X11.XFreeColors(d.display,d.colorMap,pixel,1,0);
        END;
        DEC(d.colorMode);
        IF d.colorMode=greyScaleMode THEN
          prefs:=greyPrefs;
        ELSIF d.colorMode=monochromeMode THEN
          prefs:=monoPrefs;
        END;
        prefs.display:=d;
      END;

    UNTIL ~error OR (d.colorMode<0);

    IF d.colorMode<0 THEN
      Err.String("cannot allocate enough colors, giving up"); Err.Ln;
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

    FOR i:=0 TO fontCount-1 DO
      IF prefs.fonts[i]#"" THEN
        NEW(d.font[i]);
        d.font[i].Init;
        IF ~d.font[i].LoadFont(d,prefs.fonts[i],FALSE) THEN
          Err.String("Cannot load font '"); Err.String(prefs.fonts[i]); Err.String("'"); Err.Ln;
          (* TODO: Free colors *)
          X11.XCloseDisplay(d.display);
          RETURN FALSE;
        ELSE
          d.AddFont(d.font[i]);
        END;
      END;

    END;

    d.spaceHeight:=(d.font[fixedFont].height+d.font[normalFont].height) DIV 4;
    d.spaceWidth:=d.spaceHeight;

    d.focusBorder:=2; (* !!! *)

    d.deleteProt[0]:=X11.XInternAtom(d.display,"WM_DELETE_WINDOW",X11.True); (* Check result *)
    IF d.deleteProt[0]=X11.None THEN
      Err.String("Window manager does not support WM_DELETE_WINDOW"); Err.Ln;
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

(*    d.dndMessage:=X11.XInternAtom(d.display,"_MOTIF_DRAG_AND_DROP_MESSAGE",X11.False);
    d.dndRecProp:=X11.XInternAtom(d.display,"_MOTIF_DRAG_RECEIVER_INFO",X11.False);
    d.dndIniProp:=X11.XInternAtom(d.display,"_MOTIF_DRAG_INITIATOR_INFO",X11.False);
    d.dndSuccess:=X11.XInternAtom(d.display,"XmTRANSFER_SUCCESS",X11.False);
    d.dndFailure:=X11.XInternAtom(d.display,"XmTRANSFER_FAILURE",X11.False);*)

    (* X11 *)
    d.atom:=X11.XInternAtom(d.display,"ATOM",X11.False);

    (* Xdnd *)
    d.XdndAware:=X11.XInternAtom(d.display,"XdndAware",X11.False);
    d.XdndEnter:=X11.XInternAtom(d.display,"XdndEnter",X11.False);
    d.XdndLeave:=X11.XInternAtom(d.display,"XdndLeave",X11.False);
    d.XdndPosition:=X11.XInternAtom(d.display,"XdndPosition",X11.False);
    d.XdndStatus:=X11.XInternAtom(d.display,"XdndStatus",X11.False);
    d.XdndFinished:=X11.XInternAtom(d.display,"XdndFinished",X11.False);
    d.XdndDrop:=X11.XInternAtom(d.display,"XdndDrop",X11.False);
    d.XdndActionCopy:=X11.XInternAtom(d.display,"XdndActionCopy",X11.False);
    d.XdndActionMove:=X11.XInternAtom(d.display,"XdndActionMove",X11.False);
    d.XdndActionLink:=X11.XInternAtom(d.display,"XdndActionLink",X11.False);
    d.XdndActionAsk:=X11.XInternAtom(d.display,"XdndActionAsk",X11.False);
    d.XdndActionPrivate:=X11.XInternAtom(d.display,"XdndActionPrivate",X11.False);
    d.XdndActionList:=X11.XInternAtom(d.display,"XdndActionList",X11.False);
    d.XdndSelection:=X11.XInternAtom(d.display,"XdndSelection",X11.False);
    d.XdndTypeList:=X11.XInternAtom(d.display,"XdndTypeList",X11.False);

    (* X selection mechzanism *)
    d.xSelection:=X11.XInternAtom(d.display,"_VISUALOBERON_SELECTION_DATA",X11.False);
    d.xDrop:=X11.XInternAtom(d.display,"_VISUALOBERON_DROP_DATA",X11.False);
    d.wmState:=X11.XInternAtom(d.display,"WM_STATE",X11.False);

    d.mainWin:=X11.XCreateSimpleWindow(d.display,X11.XRootWindow(d.display,d.scrNum),
                                       0,0,10,10,
                                       0,
                                       d.color[backgroundColor].color.pixel,
                                       d.color[backgroundColor].color.pixel);

    IF d.mainWin=0 THEN
      Err.String("Cannot open main applicationwindow!"); Err.Ln;
      X11.XCloseDisplay(d.display);
      RETURN FALSE;
    END;

    d.sleepCursor:=X11.XCreateFontCursor(d.display,sleepCursor);
    IF d.sleepCursor=0 THEN
      Err.String("Cannot get sleep cursor!"); Err.Ln;
      X11.XCloseDisplay(d.display);
      X11.XDestroyWindow(d.display,d.mainWin);
      RETURN FALSE;
    END;

    d.popCursor:=X11.XCreateFontCursor(d.display,popCursor);
    IF d.popCursor=0 THEN
      Err.String("Cannot get popup cursor!"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XCloseDisplay(d.display);
      X11.XDestroyWindow(d.display,d.mainWin);
      RETURN FALSE;
    END;

    d.dndCursor:=X11.XCreateFontCursor(d.display,dndCursor);
    IF d.dndCursor=0 THEN
      Err.String("Cannot get drag & drop cursor!"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XFreeCursor(d.display,d.popCursor);
      X11.XCloseDisplay(d.display);
      X11.XDestroyWindow(d.display,d.mainWin);
      RETURN FALSE;
    END;

    d.copyCursor:=CreatePixmapCursor(s.ADR(copyCursorData),s.ADR(copyMaskData),14,15);
    IF d.copyCursor=0 THEN
      Err.String("Cannot create copy cursor"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XFreeCursor(d.display,d.popCursor);
      X11.XFreeCursor(d.display,d.dndCursor);
      X11.XCloseDisplay(d.display);
      X11.XDestroyWindow(d.display,d.mainWin);
      RETURN FALSE;
    END;

    d.moveCursor:=CreatePixmapCursor(s.ADR(moveCursorData),s.ADR(moveMaskData),11,13);
    IF d.moveCursor=0 THEN
      Err.String("Cannot create move cursor"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XFreeCursor(d.display,d.popCursor);
      X11.XFreeCursor(d.display,d.dndCursor);
      X11.XFreeCursor(d.display,d.copyCursor);
      X11.XCloseDisplay(d.display);
      X11.XDestroyWindow(d.display,d.mainWin);
      RETURN FALSE;
    END;

    d.linkCursor:=CreatePixmapCursor(s.ADR(linkCursorData),s.ADR(linkMaskData),15,17);
    IF d.linkCursor=0 THEN
      Err.String("Cannot create move cursor"); Err.Ln;
      X11.XFreeCursor(d.display,d.sleepCursor);
      X11.XFreeCursor(d.display,d.popCursor);
      X11.XFreeCursor(d.display,d.dndCursor);
      X11.XFreeCursor(d.display,d.copyCursor);
      X11.XFreeCursor(d.display,d.moveCursor);
      X11.XCloseDisplay(d.display);
      X11.XDestroyWindow(d.display,d.mainWin);
      RETURN FALSE;
    END;


    d.winList:=NIL;
    d.winCount:=0;
    d.exit:=FALSE;
    d.currentWin:=NIL;

    d.fonts:=NIL;
    d.usrFontID:=-1;

    d.timeOutList:=NIL;
    d.timeOut:=NIL;
    d.contextHelp:=TRUE;

    d.selectObject:=NIL;
    d.querySelectObject:=NIL;

    d.dragging:=FALSE;
    d.dragObject:=NIL;

    d.userColor:=NIL;

(** Modified by Marco **)
(* new code *)
    d.lastEventIsButtonPress:=FALSE;
    d.lastButton:=-2;
    d.lastBut1Button:=-1;
    d.multiClickTime:=200;
(* end of new code *)

    RETURN TRUE;
  END InitDisplay;

(** Modified by Marco **)
(* new code *)
  PROCEDURE (d : Display) IsDoubleClicked * ():BOOLEAN;
  BEGIN
    IF ~d.lastEventIsButtonPress  THEN
      RETURN FALSE;
    END;
    RETURN (d.lastButton = d.lastBut1Button)
          & (d.lastPressTime - d.lastBut1PressTime <= d.multiClickTime);
  END IsDoubleClicked;

  PROCEDURE (d : Display) SetMultiClickTime * ( t : LONGINT );
  BEGIN
    IF t>0 THEN d.multiClickTime:=t END;
  END SetMultiClickTime;
(** end of new code **)

  (**
    Flushes all pending events. makes only sense for asnsycronous
    windowing systems like X11.
  **)

  PROCEDURE (d : Display) Flush*;

  BEGIN
    X11.XSync(d.display,X11.False);
  END Flush;

  (**
    using this method an object can register itself a holder of the current
    selection.
  **)

  PROCEDURE (d : Display) RegisterSelection*(object : Object; window : Window):BOOLEAN;

  BEGIN
    X11.XSetSelectionOwner(d.display,a.XA_PRIMARY,window.window,X11.CurrentTime);
    IF X11.XGetSelectionOwner(d.display,a.XA_PRIMARY)=window.window THEN
      IF (d.selectObject#NIL) & (d.selectObject#object) THEN
        d.selectObject.Deselect;
        d.selectObject:=NIL;
      END;
      d.selectObject:=object;
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END RegisterSelection;

  (**
    Using this method an object can cacel the before registered selection. This can f.e
    happen, when the object hides.
  **)

  PROCEDURE (d : Display) CancelSelection*;

  BEGIN
    IF d.selectObject#NIL THEN
      X11.XSetSelectionOwner(d.display,a.XA_PRIMARY,X11.None,X11.CurrentTime);
      d.selectObject:=NIL;
    END;
  END CancelSelection;

  (**
    Call this method if you want to get the value of the global selection. If
    VisualOberon can get the selection the handleDropData method of the
    querying object will be called with a VODragDrop.insert action. Note that there
    may be some undeterminate amount of time between the query and the drop, it is
    also possible that the HandleDrop method will never will called.
  **)

  PROCEDURE (d : Display) QuerySelection*(window: Window; object: Object; type: LONGINT):BOOLEAN;

  VAR
    propType : X11.Atom;

  BEGIN
    CASE type OF
      text: propType:=a.XA_STRING;
    ELSE
      RETURN FALSE;
    END;

    X11.XConvertSelection(d.display,
                          a.XA_PRIMARY,propType,
                          window.display.xSelection,
                          window.window,
                          X11.CurrentTime);

    d.querySelectObject:=object;
    RETURN TRUE;
  END QuerySelection;

  (**
    Call this methods, if you want all windows to reinit themself, f.e.
    when the preferences of the some of the objects have been changed.
  **)

  PROCEDURE (d : Display) ReinitWindows*;

  VAR
    win : Window;

  BEGIN
    win:=d.winList;
    WHILE win#NIL DO
      win.ReinitWindow;
      win:=win.next;
    END;
  END ReinitWindows;

  (**
    Put an event back into the message handling queue.
  **)

  PROCEDURE (d : Display) PutBackEvent*(event : E.Event; destWin : Window);

  BEGIN
    event.event.xany.window:=destWin.window;
    IF X11.XSendEvent(d.display,destWin.window,X11.True,X11.NoEventMask,
                     s.VAL(X11.XEventPtr,s.ADR(event.event)))=0 THEN
      Err.String("Cannot resend event!"); Err.Ln;
    END;
  END PutBackEvent;

  PROCEDURE (d : Display) GetEvent*;

  VAR
    e     : X11.XEvent;
    event : E.Event;

  BEGIN
    X11.XNextEvent(d.display,e);

    IF d.timeOut#NIL THEN
      d.RemoveTimeOut(d.timeOut);
    END;
    IF d.contextHelp THEN
      d.timeOut:=d.AddTimeOut(contextTimeOutSec,contextTimeOutMil,d);
    END;

    d.currentWin:=d.GetWindow(e.xany.window);

    IF d.currentWin#NIL THEN

      IF ~(
        (e.type=X11.ButtonPress) & (e.xbutton.state={})
      & (e.xbutton.button=X11.Button3) & (d.currentWin.modalCount=0)
      & d.currentWin.ContextMenu()) THEN
        event:=E.GetEvent(e);
        REPEAT
          event.reUse:=FALSE;
          IF d.currentWin.HandleEvent(event) THEN END;
        UNTIL event.reUse=FALSE;
      END;
    END;
  END GetEvent;

  (**
    Waits for events on the given filedescriptor with the given timeout.

    Returns TRUE, if the wait exceeds the given timeout, else FALSE, if Wait
    returned because the filedescriptor got available.

    You can overload this method to wait for other events (f.e. other
    filedescriptors like sockets, etc...), too.

    NOTE
    This method must be seen as lowlevel stuff. Not all windowing systems
    or OSs may use filedescriptors. It maybe that ported versions of
    VisualOberon hand other parameters to this method.
  **)

  PROCEDURE (d : Display) Wait*(x11FD : C.int; interval : t.Interval):BOOLEAN;

  VAR
    channel : Channel;
    fds     : LI.fd_set;
    timeout : LI.timeval;
    maxFD   : C.int;
    res     : C.int;

  BEGIN
    LOOP
      timeout.tv_sec:=interval.dayInt*(t.msecPerDay DIV t.msecPerSec)+interval.msecInt DIV 1000;
      timeout.tv_usec:=(interval.msecInt MOD 1000)*1000;

      IF (timeout.tv_sec<0) OR (timeout.tv_usec<0) THEN
        timeout.tv_sec:=0;
        timeout.tv_usec:=0;
      END;

      LM.FD_ZERO(fds);
      LM.FD_SET(x11FD,fds);
      maxFD:=x11FD;

      channel:=d.channelList;
      WHILE channel#NIL DO
        IF channel.channel IS p.Channel THEN
          LM.FD_SET(channel.channel(p.Channel).fd,fds);
          IF channel.channel(p.Channel).fd>maxFD THEN
            maxFD:=channel.channel(p.Channel).fd;
          END;
        END;
        channel:=channel.next;
      END;
      res:=LI.select(maxFD+1,s.VAL(LI.fd_setPtr,s.ADR(fds)),NIL,NIL,timeout);

      IF res=0 THEN
        RETURN TRUE;
      ELSIF res=-1 THEN
        Err.String("select: error!"); Err.Ln;
      ELSE
        channel:=d.channelList;
        WHILE channel#NIL DO
          IF channel.channel IS p.Channel THEN
            IF LM.FD_ISSET(channel.channel(p.Channel).fd,fds) THEN
              channel.SendNotify;
            END;
          END;
          channel:=channel.next;
        END;

        IF LM.FD_ISSET(x11FD,fds) THEN
          RETURN FALSE;
        END;
      END;
    END;
  END Wait;

  (**
    This is the main event loop of your application. Call this the get
    things started. If you want to leave the event loop and fiish the
    application, send a ExitMsg to the display.
  **)

  PROCEDURE (d : Display) Handler*;

  VAR
    interval : t.Interval;

  BEGIN
    LOOP
      IF d.sleepList#NIL THEN
        IF X11.XEventsQueued(d.display,X11.QueuedAlready)=X11.False THEN
          d.CheckSleeps;
          d.CheckTimeOuts;
        ELSE
          d.GetEvent;
        END;

      ELSE
        IF X11.XEventsQueued(d.display,X11.QueuedAlready)=X11.False THEN

          IF X11.XEventsQueued(d.display,X11.QueuedAfterFlush)=X11.False THEN
            d.CheckTimeOuts;
            d.GetNextTimeOut(interval);
            IF d.Wait(d.display.fd,interval) THEN
              d.CheckTimeOuts;
            END;
          ELSE
            d.CheckTimeOuts;
          END;
        ELSE
          d.CheckTimeOuts;
        END;
      END;

      IF X11.XEventsQueued(d.display,X11.QueuedAfterReading)#X11.False THEN

        d.GetEvent;

        d.CheckTimeOuts;

      END;

      IF d.exit THEN
        EXIT;
      END;

    END;
  END Handler;

  (**
    This implements a local event loop for a modal window. The method will
    return, when the window has been closed.
  **)

  PROCEDURE (w : Window) Go*;

  VAR
    interval : t.Interval;
    opened   : BOOLEAN;

  BEGIN
    opened:=FALSE; (* TODO: Fix this quick workaround *)

    LOOP
      IF w.display.sleepList#NIL THEN
        IF X11.XEventsQueued(w.display.display,X11.QueuedAlready)=X11.False THEN
          w.display.CheckSleeps;
          w.display.CheckTimeOuts;
        ELSE
          w.display.GetEvent;
        END;

      ELSE
        IF X11.XEventsQueued(w.display.display,X11.QueuedAlready)=X11.False THEN

          IF X11.XEventsQueued(w.display.display,X11.QueuedAfterFlush)=X11.False THEN
            w.display.CheckTimeOuts;
            w.display.GetNextTimeOut(interval);
            IF w.display.Wait(w.display.display.fd,interval) THEN
              w.display.CheckTimeOuts;
            END;
          ELSE
            w.display.CheckTimeOuts;
          END;
        ELSE
          w.display.CheckTimeOuts;
        END;
      END;

      IF X11.XEventsQueued(w.display.display,X11.QueuedAfterReading)#X11.False THEN

        w.display.GetEvent;

        w.display.CheckTimeOuts;

      END;

      IF ~w.maped THEN
        IF opened THEN
          EXIT;
        END;
      ELSE
        opened:=TRUE;
      END;

    END;
  END Go;


  (**
    This sets the exit flag to true. The handler-method will be left after
    this flag has been set.
  **)

  PROCEDURE (d : Display) Exit*;

  BEGIN
    d.exit:=TRUE;
  END Exit;

  (**
    The message receive function of the Display. Currently only ExitMsg and
    TimeOutMsgs, which get generated by Display when a contextHelp should
    occure, are supported.
  **)

  PROCEDURE (d : Display) Receive*(message : O.Message);

  BEGIN
    WITH
      message : ExitMsg DO
        d.exit:=TRUE;
    | message : TimeOutMsg DO
        IF (d.currentWin#NIL) & (message.timeOut=d.timeOut) THEN
          d.timeOut:=NIL;
          d.currentWin.ContextHelp;
        END;
    ELSE
    END;
  END Receive;

  (**
    Deinitialize the display. Call this when your application finishes.
  **)

  PROCEDURE (d : Display) Deinit*;

  VAR
    w    : Window;
    font : Font;
    x    : LONGINT;

  BEGIN
    w:=d.winList;
    WHILE w#NIL DO
(*
      Err.String("Warning: window ");
      IF w.title#NIL THEN
        Err.String(w.title^);
      END;
      Err.String(" not explicitely closed"); Err.Ln;
*)
      w.Delete;
      w:=w.next;
    END;

    (* Freeing all preferences *)
    P.Free;

    (* Freeing colors *)

    IF d.userColor#NIL THEN
      x:=0;
      FOR x:=0 TO LEN(d.userColor^)-1 DO
        (* TODO: Free colors *)
(*        IF d.userColor[x].useCount>0 THEN
          Err.String("Color ");
          Err.LongInt(d.userColor[x].color.red,0);
          Err.String(" ");
          Err.LongInt(d.userColor[x].color.green,0);
          Err.String(" ");
          Err.LongInt(d.userColor[x].color.blue,0);
          Err.String(" ");
          Err.String("not explicitely freed"); Err.Ln;
        END;*)
      END;
    END;


    (* Freeing fonts *)
    font:=d.fonts;
    WHILE font#NIL DO
      font.useCount:=0;
      font.Free(d);
      font:=font.next;
    END;

    IF d.sleepCursor#0 THEN
      X11.XFreeCursor(d.display,d.sleepCursor);
    END;

    IF d.mainWin#0 THEN
      X11.XDestroyWindow(d.display,d.mainWin);
    END;

    IF d.display#NIL THEN
      X11.XCloseDisplay(d.display);
    END;
  END Deinit;

  (**
    Converts any incomming message to a ExitMsg.
  **)

  PROCEDURE (h : Msg2Exit) Convert*(message : O.Message):O.Message;

  VAR
    new : ExitMsg;

  BEGIN
    NEW(new);
    RETURN new;
  END Convert;

  (**
    Initializes an instance.
  **)

  PROCEDURE (p : DisplayPrefs) Init*;

  BEGIN
    p.fonts[tinyFont]      :="-adobe-helvetica-medium-r-*--8-*-*-*-*-*-iso8859-1";
    p.fonts[scriptFont]    :="-adobe-helvetica-medium-r-*--8-*-*-*-*-*-iso8859-1";
    p.fonts[footnoteFont]  :="-adobe-helvetica-medium-r-*--10-*-*-*-*-*-iso8859-1";
    p.fonts[smallFont]     :="-adobe-helvetica-medium-r-*--10-*-*-*-*-*-iso8859-1";
    p.fonts[normalFont]    :="-adobe-helvetica-medium-r-*--12-*-*-*-*-*-iso8859-1";
    p.fonts[largeFont]     :="-adobe-helvetica-medium-r-*--14-*-*-*-*-*-iso8859-1";
    p.fonts[LargeFont]     :="-adobe-helvetica-medium-r-*--18-*-*-*-*-*-iso8859-1";
    p.fonts[LARGEFont]     :="-adobe-helvetica-medium-r-*--18-*-*-*-*-*-iso8859-1";
    p.fonts[hugeFont]      :="-adobe-helvetica-medium-r-*--24-*-*-*-*-*-iso8859-1";
    p.fonts[HUGEFont]      :="-adobe-helvetica-medium-r-*--25-*-*-*-*-*-iso8859-1";

    p.fonts[smallFixedFont]:="-adobe-courier-medium-r-*--9-*-*-*-*-*-iso8859-1";
    p.fonts[fixedFont]     :="-adobe-courier-medium-r-*--10-*-*-*-*-*-iso8859-1";
    p.fonts[hugeFixedFont] :="-adobe-courier-medium-r-*--12-*-*-*-*-*-iso8859-1";
  END Init;

BEGIN
  disablePattern[0]:=CHR(170);  (* 10101010 *)
  disablePattern[1]:=CHR(85);   (* 01010101 *)

  sPointLine[0]:=CHR(1);
  sPointLine[1]:=CHR(1);

  mPointLine[0]:=CHR(2);
  mPointLine[1]:=CHR(2);

  lPointLine[0]:=CHR(3);
  lPointLine[1]:=CHR(3);

  copyCursorData[ 0]:=000X;
  copyCursorData[ 1]:=000X;
  copyCursorData[ 2]:=0FEX;
  copyCursorData[ 3]:=003X;
  copyCursorData[ 4]:=002X;
  copyCursorData[ 5]:=002X;
  copyCursorData[ 6]:=002X;
  copyCursorData[ 7]:=01EX;
  copyCursorData[ 8]:=072X;
  copyCursorData[ 9]:=012X;
  copyCursorData[10]:=002X;
  copyCursorData[11]:=012X;
  copyCursorData[12]:=072X;
  copyCursorData[13]:=012X;
  copyCursorData[14]:=002X;
  copyCursorData[15]:=012X;
  copyCursorData[16]:=072X;
  copyCursorData[17]:=012X;
  copyCursorData[18]:=002X;
  copyCursorData[19]:=012X;
  copyCursorData[20]:=002X;
  copyCursorData[21]:=012X;
  copyCursorData[22]:=0FEX;
  copyCursorData[23]:=013X;
  copyCursorData[24]:=010X;
  copyCursorData[25]:=010X;
  copyCursorData[26]:=0F0X;
  copyCursorData[27]:=01FX;
  copyCursorData[28]:=000X;
  copyCursorData[29]:=000X;

  copyMaskData[ 0]:=0FFX;
  copyMaskData[ 1]:=007X;
  copyMaskData[ 2]:=0FFX;
  copyMaskData[ 3]:=007X;
  copyMaskData[ 4]:=0FFX;
  copyMaskData[ 5]:=03FX;
  copyMaskData[ 6]:=0FFX;
  copyMaskData[ 7]:=03FX;
  copyMaskData[ 8]:=0FFX;
  copyMaskData[ 9]:=03FX;
  copyMaskData[10]:=0FFX;
  copyMaskData[11]:=03FX;
  copyMaskData[12]:=0FFX;
  copyMaskData[13]:=03FX;
  copyMaskData[14]:=0FFX;
  copyMaskData[15]:=03FX;
  copyMaskData[16]:=0FFX;
  copyMaskData[17]:=03FX;
  copyMaskData[18]:=0FFX;
  copyMaskData[19]:=03FX;
  copyMaskData[20]:=0FFX;
  copyMaskData[21]:=03FX;
  copyMaskData[22]:=0FFX;
  copyMaskData[23]:=03FX;
  copyMaskData[24]:=0FFX;
  copyMaskData[25]:=03FX;
  copyMaskData[26]:=0F8X;
  copyMaskData[27]:=03FX;
  copyMaskData[28]:=0F8X;
  copyMaskData[29]:=03FX;

  moveCursorData[ 0]:=000X;
  moveCursorData[ 1]:=000X;
  moveCursorData[ 2]:=0FEX;
  moveCursorData[ 3]:=003X;
  moveCursorData[ 4]:=002X;
  moveCursorData[ 5]:=002X;
  moveCursorData[ 6]:=002X;
  moveCursorData[ 7]:=002X;
  moveCursorData[ 8]:=072X;
  moveCursorData[ 9]:=002X;
  moveCursorData[10]:=002X;
  moveCursorData[11]:=002X;
  moveCursorData[12]:=072X;
  moveCursorData[13]:=002X;
  moveCursorData[14]:=002X;
  moveCursorData[15]:=002X;
  moveCursorData[16]:=072X;
  moveCursorData[17]:=002X;
  moveCursorData[18]:=002X;
  moveCursorData[19]:=002X;
  moveCursorData[20]:=002X;
  moveCursorData[21]:=002X;
  moveCursorData[22]:=0FEX;
  moveCursorData[23]:=003X;
  moveCursorData[24]:=000X;
  moveCursorData[25]:=000X;

  moveMaskData[ 0]:=0FFX;
  moveMaskData[ 1]:=007X;
  moveMaskData[ 2]:=0FFX;
  moveMaskData[ 3]:=007X;
  moveMaskData[ 4]:=0FFX;
  moveMaskData[ 5]:=007X;
  moveMaskData[ 6]:=0FFX;
  moveMaskData[ 7]:=007X;
  moveMaskData[ 8]:=0FFX;
  moveMaskData[ 9]:=007X;
  moveMaskData[10]:=0FFX;
  moveMaskData[11]:=007X;
  moveMaskData[12]:=0FFX;
  moveMaskData[13]:=007X;
  moveMaskData[14]:=0FFX;
  moveMaskData[15]:=007X;
  moveMaskData[16]:=0FFX;
  moveMaskData[17]:=007X;
  moveMaskData[18]:=0FFX;
  moveMaskData[19]:=007X;
  moveMaskData[20]:=0FFX;
  moveMaskData[21]:=007X;
  moveMaskData[22]:=0FFX;
  moveMaskData[23]:=007X;
  moveMaskData[24]:=0FFX;
  moveMaskData[25]:=007X;

  linkCursorData[ 0]:=000X;
  linkCursorData[ 1]:=000X;
  linkCursorData[ 2]:=0FEX;
  linkCursorData[ 3]:=003X;
  linkCursorData[ 4]:=002X;
  linkCursorData[ 5]:=02AX;
  linkCursorData[ 6]:=002X;
  linkCursorData[ 7]:=002X;
  linkCursorData[ 8]:=072X;
  linkCursorData[ 9]:=022X;
  linkCursorData[10]:=002X;
  linkCursorData[11]:=002X;
  linkCursorData[12]:=072X;
  linkCursorData[13]:=022X;
  linkCursorData[14]:=002X;
  linkCursorData[15]:=002X;
  linkCursorData[16]:=072X;
  linkCursorData[17]:=03EX;
  linkCursorData[18]:=002X;
  linkCursorData[19]:=022X;
  linkCursorData[20]:=002X;
  linkCursorData[21]:=02EX;
  linkCursorData[22]:=0FEX;
  linkCursorData[23]:=023X;
  linkCursorData[24]:=000X;
  linkCursorData[25]:=021X;
  linkCursorData[26]:=000X;
  linkCursorData[27]:=02DX;
  linkCursorData[28]:=000X;
  linkCursorData[29]:=021X;
  linkCursorData[30]:=000X;
  linkCursorData[31]:=03FX;
  linkCursorData[32]:=000X;
  linkCursorData[33]:=000X;

  linkMaskData[ 0]:=0FFX;
  linkMaskData[ 1]:=007X;
  linkMaskData[ 2]:=0FFX;
  linkMaskData[ 3]:=07FX;
  linkMaskData[ 4]:=0FFX;
  linkMaskData[ 5]:=07FX;
  linkMaskData[ 6]:=0FFX;
  linkMaskData[ 7]:=07FX;
  linkMaskData[ 8]:=0FFX;
  linkMaskData[ 9]:=077X;
  linkMaskData[10]:=0FFX;
  linkMaskData[11]:=077X;
  linkMaskData[12]:=0FFX;
  linkMaskData[13]:=077X;
  linkMaskData[14]:=0FFX;
  linkMaskData[15]:=07FX;
  linkMaskData[16]:=0FFX;
  linkMaskData[17]:=07FX;
  linkMaskData[18]:=0FFX;
  linkMaskData[19]:=07FX;
  linkMaskData[20]:=0FFX;
  linkMaskData[21]:=07FX;
  linkMaskData[22]:=0FFX;
  linkMaskData[23]:=07FX;
  linkMaskData[24]:=0FFX;
  linkMaskData[25]:=07FX;
  linkMaskData[26]:=080X;
  linkMaskData[27]:=07FX;
  linkMaskData[28]:=080X;
  linkMaskData[29]:=07FX;
  linkMaskData[30]:=080X;
  linkMaskData[31]:=07FX;
  linkMaskData[32]:=080X;
  linkMaskData[33]:=07FX;

  colorNames[backgroundColor]      :="background";
  colorNames[tableBackgroundColor] :="tableBackground";
  colorNames[tableTextColor]       :="tableText";
  colorNames[textBackgroundColor]  :="textBackground";
  colorNames[buttonBackgroundColor]:="buttonBackground";
  colorNames[textColor]            :="text";
  colorNames[shineColor]           :="shine";
  colorNames[shadowColor]          :="shadow";
  colorNames[fillColor]            :="fill";
  colorNames[fillTextColor]        :="fillText";
  colorNames[halfShineColor]       :="halfShine";
  colorNames[halfShadowColor]      :="halfShadow";
  colorNames[warnColor]            :="warn";
  colorNames[disabledColor]        :="disabled";
  colorNames[focusColor]           :="focus";
  colorNames[blackColor]           :="black";
  colorNames[whiteColor]           :="white";
  colorNames[helpBackgroundColor]  :="helpBackground";

  fontNames[tinyFont]      :="tiny";
  fontNames[scriptFont]    :="script";
  fontNames[footnoteFont]  :="footnote";
  fontNames[smallFont]     :="small";
  fontNames[normalFont]    :="normal";
  fontNames[largeFont]     :="large";
  fontNames[LargeFont]     :="Large";
  fontNames[LARGEFont]     :="LARGE";
  fontNames[hugeFont]      :="huge";
  fontNames[HUGEFont]      :="HUGE";
  fontNames[smallFixedFont]:="smallFixed";
  fontNames[fixedFont]     :="fixed";
  fontNames[hugeFixedFont] :="hugeFixed";

  NEW(monoPrefs);
  monoPrefs.Init;
  NEW(greyPrefs);
  greyPrefs.Init;
  NEW(colorPrefs);
  colorPrefs.Init;

  monoPrefs.colors[backgroundColor]:="white";
  monoPrefs.colors[tableBackgroundColor]:=monoPrefs.colors[backgroundColor];
  monoPrefs.colors[textBackgroundColor]:=monoPrefs.colors[backgroundColor];
  monoPrefs.colors[buttonBackgroundColor]:=monoPrefs.colors[backgroundColor];
  monoPrefs.colors[tableTextColor]:="black";
  monoPrefs.colors[halfShineColor]:="black";
  monoPrefs.colors[halfShadowColor]:="black";
  monoPrefs.colors[textColor]:="black";
  monoPrefs.colors[shineColor]:="black";
  monoPrefs.colors[shadowColor]:="black";
  monoPrefs.colors[fillColor]:="black";
  monoPrefs.colors[fillTextColor]:="white";
  monoPrefs.colors[warnColor]:="black";
  monoPrefs.colors[disabledColor]:="black";
  monoPrefs.colors[focusColor]:="black";
  monoPrefs.colors[blackColor]:="black";
  monoPrefs.colors[whiteColor]:="white";
  monoPrefs.colors[helpBackgroundColor]:=monoPrefs.colors[backgroundColor];

  greyPrefs.colors[backgroundColor]:="grey70";
  greyPrefs.colors[tableBackgroundColor]:=greyPrefs.colors[backgroundColor];
  greyPrefs.colors[textBackgroundColor]:=greyPrefs.colors[backgroundColor];
  greyPrefs.colors[buttonBackgroundColor]:=greyPrefs.colors[backgroundColor];
  greyPrefs.colors[tableTextColor]:="black";
  greyPrefs.colors[textColor]:="black";
  greyPrefs.colors[shineColor]:="grey95";
  greyPrefs.colors[halfShineColor]:="grey82";
  greyPrefs.colors[halfShadowColor]:="grey45";
  greyPrefs.colors[shadowColor]:="grey20";
  greyPrefs.colors[fillColor]:="grey60";
  greyPrefs.colors[fillTextColor]:="black";
  greyPrefs.colors[warnColor]:="grey92";
  greyPrefs.colors[disabledColor]:="grey20";
  greyPrefs.colors[focusColor]:="grey30";
  greyPrefs.colors[blackColor]:="black";
  greyPrefs.colors[whiteColor]:="white";
  greyPrefs.colors[helpBackgroundColor]:=greyPrefs.colors[backgroundColor];

  colorPrefs.colors[backgroundColor]:=greyPrefs.colors[backgroundColor];
  colorPrefs.colors[tableBackgroundColor]:=greyPrefs.colors[tableBackgroundColor];
  colorPrefs.colors[tableTextColor]:=greyPrefs.colors[tableTextColor];
  colorPrefs.colors[textBackgroundColor]:=greyPrefs.colors[textBackgroundColor];
  colorPrefs.colors[buttonBackgroundColor]:=greyPrefs.colors[textBackgroundColor];
  colorPrefs.colors[textColor]:=greyPrefs.colors[textColor];
  colorPrefs.colors[shineColor]:=greyPrefs.colors[shineColor];
  colorPrefs.colors[halfShineColor]:=greyPrefs.colors[halfShineColor];
  colorPrefs.colors[halfShadowColor]:=greyPrefs.colors[halfShadowColor];
  colorPrefs.colors[shadowColor]:=greyPrefs.colors[shadowColor];
  colorPrefs.colors[fillColor]:="royal blue";
  colorPrefs.colors[fillTextColor]:=greyPrefs.colors[fillTextColor];
  colorPrefs.colors[warnColor]:="red";
  colorPrefs.colors[disabledColor]:=greyPrefs.colors[disabledColor];
  colorPrefs.colors[focusColor]:=greyPrefs.colors[focusColor];
  colorPrefs.colors[blackColor]:=greyPrefs.colors[blackColor];
  colorPrefs.colors[whiteColor]:=greyPrefs.colors[whiteColor];
  colorPrefs.colors[helpBackgroundColor]:="light yellow";

  prefsCallback:=NIL;

END VODisplay.
